/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Michael Nimz <michael.nimz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'globals/sheethelper'
], function (AppHelper, SheetHelper) {

    'use strict';

    // convenience shortcuts
    var a = SheetHelper.a;
    var r = SheetHelper.r;

    // class CellCollection ===================================================

    describe('Spreadsheet class CellCollection', function () {

        // private helpers ----------------------------------------------------

        // the operations to be applied by the document model
        var OPERATIONS = [
            { name: 'setDocumentAttributes', attrs: { document: { cols: 16384, rows: 1048576 } } },
            { name: 'insertAutoStyle', type: 'cell', styleId: 'a0', attrs: {}, default: true },

            { name: 'insertSheet', sheet: 0, sheetName: 'Sheet1' },
            { name: 'changeCells', sheet: 0, start: [0, 0], contents: [
                ['headline', 'headline', 'headline', 9],
                [2, 2, 2, 'hier'],
                [3, 2, 5, 12],
                [4, 2, 4, 0],
                [5, 2, 1, 'bla']
            ] },

            { name: 'insertSheet', sheet: 1, sheetName: 'Sheet2' },
            { name: 'changeCells', sheet: 1, start: [0, 0], contents: [
                { c: [{ v: 'Name'   }, { v: 'Vorname'   }, { v: 'PLZ' }, { v: 'Strasse'       }, { v: 'Haus-Nr.' }] },
                { c: [{ v: 'Dirk'   }, { v: 'Dirksen'   }, { v: 36892 }, { v: 'Hauptstr'      }, { v: 1          }] },
                { c: [{ v: 'Dirk'   }, { v: 'Auch Dirk' }, { v: 37257 }, { v: 'Nebenstrasse'  }, { v: 5          }] },
                { c: [{ v: 'Horst'  }, { v: 'Vorhanden' }, { v: 37622 }, { v: 'Dorfstr'       }, { v: 2          }] },
                { c: [{ v: 'Volker' }, { v: 'Racho'     }, { v: 37987 }, { v: 'Lindenstrasse' }, { v: '315a'     }] }
            ] }
        ];

        // initialize test document
        var docModel = null, sheetModel = null, cellCollection = null;
        AppHelper.createSpreadsheetApp('ooxml', OPERATIONS).done(function (app) {
            docModel = app.getModel();
            sheetModel = docModel.getSheetModel(0);
            cellCollection = sheetModel.getCellCollection();
        });

        // public methods -----------------------------------------------------

        describe('method "generateSortOperations"', function () {
            it('should exist', function () {
                expect(cellCollection).to.respondTo('generateSortOperations');
            });
            it('should generate correct cell operations (vertical; single rule) #1', function (done) {
                var sheet       = 0,

                    col_from    = 'A',
                    col_to      = 'D',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C1'),
                    options     = {
                        orderRules: [{ orderBy: 'toggle' }]
                        // direction: null,
                        // headers: null
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('A2'), val_before: 2,          val_after: 5            },
                        { adr: a('A3'), val_before: 3,          val_after: 2            },
                        { adr: a('A4'), val_before: 4,          val_after: 4            },
                        { adr: a('A5'), val_before: 5,          val_after: 3            },
                        { adr: a('B1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 2            },
                        { adr: a('C1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('C2'), val_before: 2,          val_after: 1            },
                        { adr: a('C3'), val_before: 5,          val_after: 2            },
                        { adr: a('C4'), val_before: 4,          val_after: 4            },
                        { adr: a('C5'), val_before: 1,          val_after: 5            },
                        { adr: a('D1'), val_before: 9,          val_after: 9            },
                        { adr: a('D2'), val_before: 'hier',     val_after: 'bla'        },
                        { adr: a('D3'), val_before: 12,         val_after: 'hier'       },
                        { adr: a('D4'), val_before: 0,          val_after: 0            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 12           }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });

                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                });
                promise.always(function () {
                    done();
                });
            });
            it('should generate correct cell operations (vertical; single rule) #2', function (done) {
                var sheet       = 0,

                    col_from    = 'A',
                    col_to      = 'D',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C1'),
                    options     = {
                        orderRules: [{ orderBy: 'toggle' }],
                        direction: 'vertical',
                        headers: false
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 5            },
                        { adr: a('A2'), val_before: 2,          val_after: 2            },
                        { adr: a('A3'), val_before: 3,          val_after: 4            },
                        { adr: a('A4'), val_before: 4,          val_after: 3            },
                        { adr: a('A5'), val_before: 5,          val_after: 'headline'   },
                        { adr: a('B1'), val_before: 'headline', val_after: 2            },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 'headline'   },
                        { adr: a('C1'), val_before: 'headline', val_after: 1            },
                        { adr: a('C2'), val_before: 2,          val_after: 2            },
                        { adr: a('C3'), val_before: 5,          val_after: 4            },
                        { adr: a('C4'), val_before: 4,          val_after: 5            },
                        { adr: a('C5'), val_before: 1,          val_after: 'headline'   },
                        { adr: a('D1'), val_before: 9,          val_after: 'bla'        },
                        { adr: a('D2'), val_before: 'hier',     val_after: 'hier'       },
                        { adr: a('D3'), val_before: 12,         val_after: 0            },
                        { adr: a('D4'), val_before: 0,          val_after: 12           },
                        { adr: a('D5'), val_before: 'bla',      val_after: 9            }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });

                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                });

                promise.always(function () {
                    done();
                });
            });
            it('should generate correct cell operations (vertical; single rule) #3', function (done) {
                var sheet       = 0,

                    col_from    = 'A',
                    col_to      = 'D',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C1'),
                    options     = {
                        orderRules: [{ orderBy: 'ascending' }],
                        direction: 'vertical',
                        headers: true
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('A2'), val_before: 2,          val_after: 5            },
                        { adr: a('A3'), val_before: 3,          val_after: 2            },
                        { adr: a('A4'), val_before: 4,          val_after: 4            },
                        { adr: a('A5'), val_before: 5,          val_after: 3            },
                        { adr: a('B1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 2            },
                        { adr: a('C1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('C2'), val_before: 2,          val_after: 1            },
                        { adr: a('C3'), val_before: 5,          val_after: 2            },
                        { adr: a('C4'), val_before: 4,          val_after: 4            },
                        { adr: a('C5'), val_before: 1,          val_after: 5            },
                        { adr: a('D1'), val_before: 9,          val_after: 9            },
                        { adr: a('D2'), val_before: 'hier',     val_after: 'bla'        },
                        { adr: a('D3'), val_before: 12,         val_after: 'hier'       },
                        { adr: a('D4'), val_before: 0,          val_after: 0            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 12           }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
            it('should generate correct cell operations (vertical; single rule) #4', function (done) {
                var sheet       = 0,

                    col_from    = 'C',
                    col_to      = 'D',

                    row_from    = '3',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('D4'),
                    options     = {
                        orderRules: [{ index: 3, orderBy: 'ascending' }],
                        direction: 'vertical',
                        headers: false
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('A2'), val_before: 2,          val_after: 2            },
                        { adr: a('A3'), val_before: 3,          val_after: 3            },
                        { adr: a('A4'), val_before: 4,          val_after: 4            },
                        { adr: a('A5'), val_before: 5,          val_after: 5            },
                        { adr: a('B1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 2            },
                        { adr: a('C1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('C2'), val_before: 2,          val_after: 2            },
                        { adr: a('C3'), val_before: 5,          val_after: 4            },
                        { adr: a('C4'), val_before: 4,          val_after: 5            },
                        { adr: a('C5'), val_before: 1,          val_after: 1            },
                        { adr: a('D1'), val_before: 9,          val_after: 9            },
                        { adr: a('D2'), val_before: 'hier',     val_after: 'hier'       },
                        { adr: a('D3'), val_before: 12,         val_after: 0            },
                        { adr: a('D4'), val_before: 0,          val_after: 12           },
                        { adr: a('D5'), val_before: 'bla',      val_after: 'bla'        }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
            it('should generate correct cell operations (vertical; single rule) #5', function (done) {
                var sheet       = 0,

                    col_from    = 'C',
                    col_to      = 'D',

                    row_from    = '3',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C4'),
                    options     = {
                        orderRules: [{ index: 2, orderBy: 'ascending' }],
                        direction: 'vertical',
                        headers: false
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('A2'), val_before: 2,          val_after: 2            },
                        { adr: a('A3'), val_before: 3,          val_after: 3            },
                        { adr: a('A4'), val_before: 4,          val_after: 4            },
                        { adr: a('A5'), val_before: 5,          val_after: 5            },
                        { adr: a('B1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 2            },
                        { adr: a('C1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('C2'), val_before: 2,          val_after: 2            },
                        { adr: a('C3'), val_before: 5,          val_after: 1            },
                        { adr: a('C4'), val_before: 4,          val_after: 4            },
                        { adr: a('C5'), val_before: 1,          val_after: 5            },
                        { adr: a('D1'), val_before: 9,          val_after: 9            },
                        { adr: a('D2'), val_before: 'hier',     val_after: 'hier'       },
                        { adr: a('D3'), val_before: 12,         val_after: 'bla'        },
                        { adr: a('D4'), val_before: 0,          val_after: 0            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 12           }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });

            it('should generate correct cell operations (horizontal; single rule) #1', function (done) {
                var sheet       = 0,

                    col_from    = 'A',
                    col_to      = 'D',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C1'),
                    options     = {
                        orderRules: [{ index: 4, orderBy: 'ascending' }],
                        direction: 'horizontal',
                        headers: false
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('A2'), val_before: 2,          val_after: 2            },
                        { adr: a('A3'), val_before: 3,          val_after: 5            },
                        { adr: a('A4'), val_before: 4,          val_after: 4            },
                        { adr: a('A5'), val_before: 5,          val_after: 1            },
                        { adr: a('B1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 2            },
                        { adr: a('C1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('C2'), val_before: 2,          val_after: 2            },
                        { adr: a('C3'), val_before: 5,          val_after: 3            },
                        { adr: a('C4'), val_before: 4,          val_after: 4            },
                        { adr: a('C5'), val_before: 1,          val_after: 5            },
                        { adr: a('D1'), val_before: 9,          val_after: 9            },
                        { adr: a('D2'), val_before: 'hier',     val_after: 'hier'       },
                        { adr: a('D3'), val_before: 12,         val_after: 12           },
                        { adr: a('D4'), val_before: 0,          val_after: 0            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 'bla'        }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });

                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
            it('should generate correct cell operations (horizontal; single rule) #2', function (done) {
                var sheet       = 0,

                    col_from    = 'A',
                    col_to      = 'D',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('A1'),
                    options     = {
                        orderRules: [{ index: 0, orderBy: 'ascending' }],
                        direction: 'horizontal',
                        headers: false
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'headline', val_after: 9            },
                        { adr: a('A2'), val_before: 2,          val_after: 'hier'       },
                        { adr: a('A3'), val_before: 3,          val_after: 12           },
                        { adr: a('A4'), val_before: 4,          val_after: 0            },
                        { adr: a('A5'), val_before: 5,          val_after: 'bla'        },
                        { adr: a('B1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('B2'), val_before: 2,          val_after: 2            },
                        { adr: a('B3'), val_before: 2,          val_after: 3            },
                        { adr: a('B4'), val_before: 2,          val_after: 4            },
                        { adr: a('B5'), val_before: 2,          val_after: 5            },
                        { adr: a('C1'), val_before: 'headline', val_after: 'headline'   },
                        { adr: a('C2'), val_before: 2,          val_after: 2            },
                        { adr: a('C3'), val_before: 5,          val_after: 2            },
                        { adr: a('C4'), val_before: 4,          val_after: 2            },
                        { adr: a('C5'), val_before: 1,          val_after: 2            },
                        { adr: a('D1'), val_before: 9,          val_after: 'headline'   },
                        { adr: a('D2'), val_before: 'hier',     val_after: 2            },
                        { adr: a('D3'), val_before: 12,         val_after: 5            },
                        { adr: a('D4'), val_before: 0,          val_after: 4            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 1            }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
            it('should generate correct cell operations (horizontal; single rule) #3', function (done) {
                var sheet       = 0,

                    col_from    = 'C',
                    col_to      = 'D',

                    row_from    = '3',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C3'),
                    options     = {
                        orderRules: [{ orderBy: 'descending' }],
                        direction: 'horizontal',
                        headers: false
                    },

                    cells = [
                        { adr: a('C3'), val_before: 5,          val_after: 12            },
                        { adr: a('C4'), val_before: 4,          val_after: 0            },
                        { adr: a('C5'), val_before: 1,          val_after: 'bla'            },
                        { adr: a('D3'), val_before: 12,         val_after: 5            },
                        { adr: a('D4'), val_before: 0,          val_after: 4            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 1            }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
            it('should generate correct cell operations (horizontal; single rule) #4', function (done) {
                var sheet       = 0,

                    col_from    = 'B',
                    col_to      = 'D',

                    row_from    = '3',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('D5'),
                    options     = {
                        orderRules: [{ index: 2, orderBy: 'descending' }],
                        direction: 'horizontal',
                        headers: true
                    },

                    cells = [
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 2            },

                        { adr: a('C3'), val_before: 5,          val_after: 12           },
                        { adr: a('C4'), val_before: 4,          val_after: 0            },
                        { adr: a('C5'), val_before: 1,          val_after: 'bla'        },

                        { adr: a('D3'), val_before: 12,         val_after: 5            },
                        { adr: a('D4'), val_before: 0,          val_after: 4            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 1            }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
            it('should generate correct cell operations (horizontal; single rule) #5', function (done) {
                var sheet       = 0,

                    col_from    = 'B',
                    col_to      = 'D',

                    row_from    = '3',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('D5'),
                    options     = {
                        orderRules: [{ index: 4, orderBy: 'descending' }],
                        direction: 'horizontal',
                        headers: true
                    },

                    cells = [
                        { adr: a('B3'), val_before: 2,          val_after: 2            },
                        { adr: a('B4'), val_before: 2,          val_after: 2            },
                        { adr: a('B5'), val_before: 2,          val_after: 2            },

                        { adr: a('C3'), val_before: 5,          val_after: 12           },
                        { adr: a('C4'), val_before: 4,          val_after: 0            },
                        { adr: a('C5'), val_before: 1,          val_after: 'bla'        },

                        { adr: a('D3'), val_before: 12,         val_after: 5            },
                        { adr: a('D4'), val_before: 0,          val_after: 4            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 1            }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
            it('should generate correct cell operations (horizontal; single rule) #6', function (done) {
                var sheet       = 0,

                    col_from    = 'B',
                    col_to      = 'D',

                    row_from    = '3',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('D5'),
                    options     = {
                        orderRules: [{ index: 4, orderBy: 'descending' }],
                        direction: 'horizontal',
                        headers: false
                    },

                    cells = [
                        { adr: a('B3'), val_before: 2,          val_after: 12           },
                        { adr: a('B4'), val_before: 2,          val_after: 0            },
                        { adr: a('B5'), val_before: 2,          val_after: 'bla'        },

                        { adr: a('C3'), val_before: 5,          val_after: 2            },
                        { adr: a('C4'), val_before: 4,          val_after: 2            },
                        { adr: a('C5'), val_before: 1,          val_after: 2            },

                        { adr: a('D3'), val_before: 12,         val_after: 5            },
                        { adr: a('D4'), val_before: 0,          val_after: 4            },
                        { adr: a('D5'), val_before: 'bla',      val_after: 1            }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });
                });

                promise.always(function () {
                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });

            it('should generate correct cell operations (vertical; multiple rules)', function (done) {
                var sheet       = 1,

                    col_from    = 'A',
                    col_to      = 'E',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C1'),
                    options     = {
                        orderRules: [
                            { index: 0, orderBy: 'descending' },    // 'Name'     absteigend
                            { index: 4, orderBy: 'ascending' }      // 'Haus-Nr.' aufsteigend
                        ],
                        direction: 'vertical',
                        headers: true
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'Name',             val_after: 'Name'           },
                        { adr: a('A2'), val_before: 'Dirk',             val_after: 'Volker'         },
                        { adr: a('A3'), val_before: 'Dirk',             val_after: 'Horst'          },
                        { adr: a('A4'), val_before: 'Horst',            val_after: 'Dirk'           },
                        { adr: a('A5'), val_before: 'Volker',           val_after: 'Dirk'           },
                        { adr: a('B1'), val_before: 'Vorname',          val_after: 'Vorname'        },
                        { adr: a('B2'), val_before: 'Dirksen',          val_after: 'Racho'          },
                        { adr: a('B3'), val_before: 'Auch Dirk',        val_after: 'Vorhanden'      },
                        { adr: a('B4'), val_before: 'Vorhanden',        val_after: 'Dirksen'        },
                        { adr: a('B5'), val_before: 'Racho',            val_after: 'Auch Dirk'      },
                        { adr: a('C1'), val_before: 'PLZ',              val_after: 'PLZ'            },
                        { adr: a('C2'), val_before: 36892,              val_after: 37987            },
                        { adr: a('C3'), val_before: 37257,              val_after: 37622            },
                        { adr: a('C4'), val_before: 37622,              val_after: 36892            },
                        { adr: a('C5'), val_before: 37987,              val_after: 37257            },
                        { adr: a('D1'), val_before: 'Strasse',          val_after: 'Strasse'        },
                        { adr: a('D2'), val_before: 'Hauptstr',         val_after: 'Lindenstrasse'  },
                        { adr: a('D3'), val_before: 'Nebenstrasse',     val_after: 'Dorfstr'        },
                        { adr: a('D4'), val_before: 'Dorfstr',          val_after: 'Hauptstr'       },
                        { adr: a('D5'), val_before: 'Lindenstrasse',    val_after: 'Nebenstrasse'   },
                        { adr: a('E1'), val_before: 'Haus-Nr.',         val_after: 'Haus-Nr.'       },
                        { adr: a('E2'), val_before: 1,                  val_after: '315a'           },
                        { adr: a('E3'), val_before: 5,                  val_after: 2                },
                        { adr: a('E4'), val_before: 2,                  val_after: 1                },
                        { adr: a('E5'), val_before: '315a',             val_after: 5                }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.then(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });

                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                });
                promise.always(function () {
                    done();
                });
            });
            it('should generate correct cell operations (horizontal; multiple rules)', function (done) {

                var sheet       = 1,

                    col_from    = 'A',
                    col_to      = 'E',

                    row_from    = '1',
                    row_to      = '5',

                    range       = r(col_from + row_from + ':' + col_to + row_to),
                    activeCell  = a('C1'),
                    options     = {
                        orderRules: [
                            { index: 0, orderBy: 'ascending' },     // 'Zeile 1'  aufsteigend
                            { index: 4, orderBy: 'ascending' }      // 'Zeile 2'  aufsteigend
                        ],
                        direction: 'horizontal',
                        headers: false
                    },

                    cells = [
                        { adr: a('A1'), val_before: 'Name',             val_after: 'Haus-Nr.'       },
                        { adr: a('A2'), val_before: 'Dirk',             val_after: 1                },
                        { adr: a('A3'), val_before: 'Dirk',             val_after: 5                },
                        { adr: a('A4'), val_before: 'Horst',            val_after: 2                },
                        { adr: a('A5'), val_before: 'Volker',           val_after: '315a'           },
                        { adr: a('B1'), val_before: 'Vorname',          val_after: 'Name'           },
                        { adr: a('B2'), val_before: 'Dirksen',          val_after: 'Dirk'           },
                        { adr: a('B3'), val_before: 'Auch Dirk',        val_after: 'Dirk'           },
                        { adr: a('B4'), val_before: 'Vorhanden',        val_after: 'Horst'          },
                        { adr: a('B5'), val_before: 'Racho',            val_after: 'Volker'         },
                        { adr: a('C1'), val_before: 'PLZ',              val_after: 'PLZ'            },
                        { adr: a('C2'), val_before: 36892,              val_after: 36892            },
                        { adr: a('C3'), val_before: 37257,              val_after: 37257            },
                        { adr: a('C4'), val_before: 37622,              val_after: 37622            },
                        { adr: a('C5'), val_before: 37987,              val_after: 37987            },
                        { adr: a('D1'), val_before: 'Strasse',          val_after: 'Strasse'        },
                        { adr: a('D2'), val_before: 'Hauptstr',         val_after: 'Hauptstr'       },
                        { adr: a('D3'), val_before: 'Nebenstrasse',     val_after: 'Nebenstrasse'   },
                        { adr: a('D4'), val_before: 'Dorfstr',          val_after: 'Dorfstr'        },
                        { adr: a('D5'), val_before: 'Lindenstrasse',    val_after: 'Lindenstrasse'  },
                        { adr: a('E1'), val_before: 'Haus-Nr.',         val_after: 'Vorname'        },
                        { adr: a('E2'), val_before: 1,                  val_after: 'Dirksen'        },
                        { adr: a('E3'), val_before: 5,                  val_after: 'Auch Dirk'      },
                        { adr: a('E4'), val_before: 2,                  val_after: 'Vorhanden'      },
                        { adr: a('E5'), val_before: '315a',             val_after: 'Racho'          }
                    ];

                sheetModel = docModel.getSheetModel(sheet);
                cellCollection = sheetModel.getCellCollection();

                cells.forEach(function (obj) {
                    expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_before);
                });

                var generator   = sheetModel.createOperationGenerator({ applyImmediately: true });
                var promise     = cellCollection.generateSortOperations(generator, range, activeCell, options);

                expect(promise).to.respondTo('promise');

                promise.always(function () {
                    expect(promise.state()).to.equal('resolved');

                    cells.forEach(function (obj) {
                        expect(cellCollection.getValue(obj.adr)).to.equal(obj.val_after);
                    });

                    expect(docModel.applyOperations(generator, { undo: true })).to.equal(true);
                    done();
                });
            });
        });
    });

    // ========================================================================
});
