/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/presentation/components/drawing/drawingresize',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/tk/utils'
], function (AppHelper, DrawingResize, DOM, Utils) {

    'use strict';

    // class DrawingResize ==============================================

    describe('Presentation class DrawingResize', function () {

        // private helpers ----------------------------------------------------

        var layoutId_1 = 'layout1',
            masterId_1 = 'master1',
            ctrTitleLeft = 1900,
            subTitleLeft = 3800,
            ctrTitleTop = 5900,
            subTitleTop = 10800,
            ctrTitleHeight = 4000,
            subTitleHeight = 5000,
            ctrTitleWidth = 21600,
            subTitleWidth = 17800,

            drawZeroLeft = 2903,
            drawZeroTop = 2573,
            drawOneLeft = 7553,
            drawOneTop = 2573,

            // the operations to be applied by the document model
            OPERATIONS = [
                {
                    name: 'setDocumentAttributes',
                    attrs: { page: { width: 33866, height: 19050, orientation: 'landscape' } }
                },
                { name: 'insertMasterSlide', id: masterId_1 },

                { name: 'insertLayoutSlide', id: layoutId_1, target: masterId_1 },
                { name: 'insertDrawing', start: [0, 0], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'ctrTitle' }, drawing: { name: 'Titel 1', left: ctrTitleLeft, top: ctrTitleTop, width: ctrTitleWidth, height: ctrTitleHeight } } },
                { name: 'insertParagraph', start: [0, 0, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 0, 0, 0], target: layoutId_1, text: 'Mastertitelformat bearbeiten' },
                { name: 'insertDrawing', start: [0, 1], target: layoutId_1, type: 'shape', attrs: { presentation: { phType: 'subTitle', phIndex: 1 }, drawing: { name: 'Untertitel 2', left: subTitleLeft, top: subTitleTop, width: subTitleWidth, height: subTitleHeight } } },
                { name: 'insertParagraph', start: [0, 1, 0], target: layoutId_1 },
                { name: 'insertText', start: [0, 1, 0, 0], target: layoutId_1, text: 'Master-Untertitelformat bearbeiten' },

                { name: 'insertSlide', start: [0], target: layoutId_1 },
                { name: 'insertDrawing', start: [0, 0], type: 'shape', attrs: { drawing: { name: 'TextBox 3', left: drawZeroLeft, top: drawZeroTop, width: 3958, height: 1026 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'none' } } },
                { name: 'insertParagraph', start: [0, 0, 0] },
                { name: 'insertText', start: [0, 0, 0, 0], text: 'test' },
                { name: 'insertDrawing', start: [0, 1], type: 'shape', attrs: { drawing: { name: 'TextBox 4', left: drawOneLeft, top: drawOneTop, width: 3958, height: 1026 }, geometry: { presetShape: 'rect', avList: {} }, fill: { type: 'none' } } },
                { name: 'insertParagraph', start: [0, 1, 0] },
                { name: 'insertText', start: [0, 1, 0, 0], text: 'test2' }
            ];

        var app = null;
        var model = null;
        var selection = null;
        AppHelper.createPresentationApp('ooxml', OPERATIONS).done(function (theApp) {
            app = theApp;
            model = app.getModel();
            selection = model.getSelection();

            selection.setDrawingResizeHandler(DrawingResize.drawDrawingSelection);
        });

        function triggerTrackingEvent(node, type, x, y, options) {
            var isCopy = Utils.getBooleanOption(options, 'copy', false);
            var positionNode = Utils.getObjectOption(options, 'drawingNode', node);
            var startX = parseFloat(positionNode.css('left'));
            var startY = parseFloat(positionNode.css('top'));
            var event = new $.Event(type, {
                which: 1,
                startX: startX,
                startY: startY,
                pageX: x + startX,
                pageY: y + startY,
                ctrlKey: isCopy
            });
            node.trigger(event);
        }

        // existence check ----------------------------------------------------

        it('should exist', function () {
            expect(DrawingResize).to.be.a('object');
        });

        // public methods -----------------------------------------------------

        describe('method "DrawingResize.drawDrawingSelection"', function () {

            it('should exist', function () {
                expect(DrawingResize).to.respondTo('drawDrawingSelection');
            });

            it('move drawing with "tracking.js" should work fine', function () {

                selection.setTextSelection([0, 0], [0, 1]);
                var activeSlide = model.getSlideById(model.getActiveSlideId());
                var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                var firstDrawing = $(drawingsOnSlide[0]);

                var oldSetLeft = Utils.convertCssLengthToHmm(firstDrawing.css('left'));
                var oldSetTop = Utils.convertCssLengthToHmm(firstDrawing.css('top'));

                expect(oldSetLeft).to.be.closeTo(drawZeroLeft, 20);
                expect(oldSetTop).to.be.closeTo(drawZeroTop, 20);

                DrawingResize.drawDrawingSelection(app, firstDrawing);

                triggerTrackingEvent(firstDrawing, 'tracking:start', 0, 0);
                triggerTrackingEvent(firstDrawing, 'tracking:move', 10, 35);
                triggerTrackingEvent(firstDrawing, 'tracking:end', 10, 35);

                var newSetLeft = Utils.convertCssLengthToHmm(firstDrawing.css('left'));
                var newSetTop = Utils.convertCssLengthToHmm(firstDrawing.css('top'));

                expect(newSetLeft).to.be.closeTo(0, 20);
                expect(newSetTop).to.be.closeTo(0, 20);

                var newAttributes = firstDrawing.data('attributes');
                expect(newAttributes.drawing.left).to.be.closeTo(0, 20);
                expect(newAttributes.drawing.top).to.be.closeTo(0, 20);

            });

            it('copy drawing with "tracking.js" should work fine', function () {

                selection.setTextSelection([0, 1], [0, 2]);
                var activeSlide = model.getSlideById(model.getActiveSlideId());
                var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);

                var secondDrawing = $(drawingsOnSlide[1]);

                var oldSetLeft = Utils.convertCssLengthToHmm(secondDrawing.css('left'));
                var oldSetTop = Utils.convertCssLengthToHmm(secondDrawing.css('top'));
                var oldAttributes = secondDrawing.data('attributes');

                expect(oldSetLeft).to.be.closeTo(drawOneLeft, 20);
                expect(oldSetTop).to.be.closeTo(drawOneTop, 20);

                DrawingResize.drawDrawingSelection(app, secondDrawing);

                triggerTrackingEvent(secondDrawing, 'tracking:start', 0, 0, { copy: true });
                triggerTrackingEvent(secondDrawing, 'tracking:move', 10, 35, { copy: true });
                triggerTrackingEvent(secondDrawing, 'tracking:end', 10, 35, { copy: true });

                drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                var newDrawing = $(drawingsOnSlide[2]);

                var newSetLeft = Utils.convertCssLengthToHmm(newDrawing.css('left'));
                var newSetTop = Utils.convertCssLengthToHmm(newDrawing.css('top'));

                expect(newSetLeft).to.be.closeTo(0, 20);
                expect(newSetTop).to.be.closeTo(0, 20);

                var newAttributes = newDrawing.data('attributes');

                expect(newAttributes.drawing.left).to.be.closeTo(0, 20);
                expect(newAttributes.drawing.top).to.be.closeTo(0, 20);

                expect(newAttributes.drawing.width).to.be.equal(oldAttributes.drawing.width);
                expect(newAttributes.drawing.height).to.be.equal(oldAttributes.drawing.height);

            });

            it('should flip drawing horizontally on resize', function () {
                selection.setTextSelection([0, 2], [0, 3]);
                var activeSlide = model.getSlideById(model.getActiveSlideId());
                var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                var lastDrawing = $(drawingsOnSlide[2]);
                var oldAttributes = lastDrawing.data('attributes');

                expect(oldAttributes.drawing.flipH).not.to.be.equal(true);
                DrawingResize.drawDrawingSelection(app, lastDrawing);

                var selectionDrawingNode = model.getNode().children('.drawingselection-overlay').children('.selectiondrawing').first();
                var leftResizer = selectionDrawingNode.find('.resizers > [data-pos="l"]');

                triggerTrackingEvent(leftResizer, 'tracking:start', 0, 0, { drawingNode: lastDrawing });
                triggerTrackingEvent(leftResizer, 'tracking:move', 500, 0, { drawingNode: lastDrawing });
                triggerTrackingEvent(leftResizer, 'tracking:end', 500, 0, { drawingNode: lastDrawing });
                var newAttributes = lastDrawing.data('attributes');

                expect(newAttributes.drawing.height).to.be.closeTo(oldAttributes.drawing.height, 20);
                expect(newAttributes.drawing.width).not.to.be.equal(oldAttributes.drawing.width);
                expect(newAttributes.drawing.width).to.be.closeTo(Utils.convertLengthToHmm(500, 'px') - oldAttributes.drawing.width, 20);
                expect(newAttributes.drawing.flipH).to.be.equal(true).and.not.to.be.equal(oldAttributes.drawing.flipH);
            });

            it('should flip drawing vertically on resize', function () {
                selection.setTextSelection([2, 0]);
                var activeSlide = model.getSlideById(model.getActiveSlideId());
                var drawingsOnSlide = activeSlide.children(DOM.ABSOLUTE_DRAWING_SELECTOR);
                var lastDrawing = $(drawingsOnSlide[2]);
                var oldAttributes = lastDrawing.data('attributes');

                expect(oldAttributes.drawing.flipV).not.to.be.equal(true);
                DrawingResize.drawDrawingSelection(app, lastDrawing);

                var selectionDrawingNode = model.getNode().children('.drawingselection-overlay').children('.selectiondrawing').first();
                var topResizer = selectionDrawingNode.find('.resizers > [data-pos="t"]');

                triggerTrackingEvent(topResizer, 'tracking:start', 0, 0, { drawingNode: lastDrawing });
                triggerTrackingEvent(topResizer, 'tracking:move', 0, 500, { drawingNode: lastDrawing });
                triggerTrackingEvent(topResizer, 'tracking:end', 0, 500, { drawingNode: lastDrawing });
                var newAttributes = lastDrawing.data('attributes');

                expect(newAttributes.drawing.width).to.be.closeTo(oldAttributes.drawing.width, 20);
                expect(newAttributes.drawing.height).not.to.be.equal(oldAttributes.drawing.height);
                expect(newAttributes.drawing.height).to.be.closeTo(Utils.convertLengthToHmm(500, 'px') - oldAttributes.drawing.height, 20);
                expect(newAttributes.drawing.flipV).to.be.equal(true).and.not.to.be.equal(oldAttributes.drawing.flipV);
            });

        });

    });

    // ========================================================================
});
