/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define([
    'globals/apphelper',
    'io.ox/office/editframework/utils/operationerror',
    'io.ox/office/editframework/model/operationcontext',
    'io.ox/office/editframework/model/autostylecollection',
    'io.ox/office/editframework/model/stylecollection'
], function (AppHelper, OperationError, OperationContext, AutoStyleCollection, StyleCollection) {

    'use strict';

    // class AutoStyleCollection ==============================================

    describe('EditFramework class AutoStyleCollection', function () {

        it('should exist', function () {
            expect(AutoStyleCollection).to.be.a('function');
        });

        // private helpers ----------------------------------------------------

        // dummy attribute definitions for three families
        var A_ATTRS = { a1: { def: 'a1' }, a2: { def: 'a2' } };
        var B_ATTRS = { b1: { def: 'b1' }, b2: { def: 'b2' } };
        var C_ATTRS = { c1: { def: 'c1' }, c2: { def: 'c2' } };

        // initialize an edit application with style sheets and auto-styles
        var docModel = null;
        var styleSheets1 = null, autoStyles1 = null;
        var styleSheets2 = null, autoStyles2 = null;
        AppHelper.createEditApp('ooxml').done(function (app) {
            docModel = app.getModel();
            docModel.registerAttributes('A', A_ATTRS);
            docModel.registerAttributes('B', B_ATTRS);
            docModel.registerAttributes('C', C_ATTRS);
            docModel.addStyleCollection(styleSheets1 = new StyleCollection(docModel, 'A', { families: 'B C' }));
            docModel.addStyleCollection(styleSheets2 = new StyleCollection(docModel, 'B', { families: 'A C' }));
            docModel.addAutoStyleCollection(autoStyles1 = new AutoStyleCollection(docModel, 'A', { standardPrefix: 'as', strictMode: true }));
            docModel.addAutoStyleCollection(autoStyles2 = new AutoStyleCollection(docModel, 'B'));
        });

        // creates an operation context with the passed operation properties
        function op(properties) {
            return new OperationContext(docModel, properties, false);
        }

        // public methods -----------------------------------------------------

        describe('method "getStyleFamily"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('getStyleFamily');
            });
            it('should return the style family', function () {
                expect(autoStyles1.getStyleFamily()).to.equal('A');
                expect(autoStyles2.getStyleFamily()).to.equal('B');
            });
        });

        describe('method "getStyleCollection"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('getStyleCollection');
            });
            it('should return the style collection', function () {
                expect(autoStyles1.getStyleCollection()).to.equal(styleSheets1);
                expect(autoStyles2.getStyleCollection()).to.equal(styleSheets2);
            });
        });

        describe('method "getEffectiveStyleId"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('getEffectiveStyleId');
            });
            it('should return the initial default auto-style identifier', function () {
                expect(autoStyles1.getEffectiveStyleId('a1')).to.equal('a1');
                expect(autoStyles1.getEffectiveStyleId('')).to.equal('');
            });
        });

        describe('method "getDefaultStyleId"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('getDefaultStyleId');
            });
            it('should return the initial default auto-style identifier', function () {
                expect(autoStyles1.getDefaultStyleId()).to.equal('');
                expect(autoStyles2.getDefaultStyleId()).to.equal('');
            });
        });

        describe('method "isDefaultStyleId"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('isDefaultStyleId');
            });
            it('should return true for the initial default auto-style identifier', function () {
                expect(autoStyles1.isDefaultStyleId('')).to.equal(true);
                expect(autoStyles1.isDefaultStyleId('as0')).to.equal(false);
                expect(autoStyles2.isDefaultStyleId('')).to.equal(true);
                expect(autoStyles2.isDefaultStyleId('a0')).to.equal(false);
            });
        });

        describe('method "areEqualStyleIds"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('areEqualStyleIds');
            });
            it('should return true for the initial default auto-style identifier', function () {
                expect(autoStyles1.areEqualStyleIds('', '')).to.equal(true);
                expect(autoStyles1.areEqualStyleIds('', 'as1')).to.equal(false);
                expect(autoStyles1.areEqualStyleIds('as1', 'as1')).to.equal(true);
                expect(autoStyles1.areEqualStyleIds('as1', 'as2')).to.equal(false);
            });
        });

        describe('method "getAutoStyle"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('getAutoStyle');
            });
            it('should return null for unknown auto-style', function () {
                expect(autoStyles1.getAutoStyle('__unknown__')).to.equal(null);
            });
        });

        describe('method "getMergedAttributeSet"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('getMergedAttributeSet');
            });
            it('should return the attribute defaults for missing auto-styles', function () {
                expect(autoStyles1.getMergedAttributeSet('as1')).to.deep.equal({ styleId: '', A: { a1: 'a1', a2: 'a2' }, B: { b1: 'b1', b2: 'b2' }, C: { c1: 'c1', c2: 'c2' } });
            });
        });

        describe('method "getExplicitAttributeSet"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('getExplicitAttributeSet');
            });
            it('should return an empty object for missing auto-styles', function () {
                expect(autoStyles1.getExplicitAttributeSet('as1')).to.deep.equal({});
            });
        });

        describe('method "getDefaultAttributeSet"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('getDefaultAttributeSet');
            });
            it('should return the attribute defaults for missing auto-styles', function () {
                expect(autoStyles1.getDefaultAttributeSet()).to.deep.equal({ styleId: '', A: { a1: 'a1', a2: 'a2' }, B: { b1: 'b1', b2: 'b2' }, C: { c1: 'c1', c2: 'c2' } });
            });
        });

        describe('method "applyInsertAutoStyleOperation"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('applyInsertAutoStyleOperation');
            });
            describe('in strict mode', function () {
                it('should insert the default auto-style', function () {
                    expect(autoStyles1.getAutoStyle('as0')).to.equal(null);
                    expect(autoStyles1.getDefaultStyleId()).to.equal('');
                    autoStyles1.applyInsertAutoStyleOperation(op({ type: 'A', styleId: 'as0', attrs: {}, default: true }));
                    expect(autoStyles1.getAutoStyle('as0')).to.be.an('object');
                    expect(autoStyles1.getEffectiveStyleId('')).to.equal('as0');
                    expect(autoStyles1.getDefaultStyleId()).to.equal('as0');
                    expect(autoStyles1.isDefaultStyleId('as0')).to.equal(true);
                    expect(autoStyles1.isDefaultStyleId('')).to.equal(true);
                    expect(autoStyles1.areEqualStyleIds('', 'as0')).to.equal(true);
                });
                it('should fail to insert another default auto-style', function () {
                    expect(autoStyles1.applyInsertAutoStyleOperation.bind(autoStyles1, op({ type: 'A', styleId: 'as1', attrs: {}, default: true }))).to.throw(OperationError);
                    expect(autoStyles1.getAutoStyle('as1')).to.equal(null);
                    expect(autoStyles1.getDefaultStyleId()).to.equal('as0');
                });
                it('should insert another auto-style', function () {
                    expect(autoStyles1.getAutoStyle('as1')).to.equal(null);
                    autoStyles1.applyInsertAutoStyleOperation(op({ type: 'A', styleId: 'as1', attrs: {} }));
                    expect(autoStyles1.getAutoStyle('as1')).to.be.an('object');
                });
                it('should fail to insert an auto-style with wrong identifier', function () {
                    expect(autoStyles1.applyInsertAutoStyleOperation.bind(autoStyles1, op({ type: 'A', styleId: 'as1', attrs: {} }))).to.throw(OperationError);
                    expect(autoStyles1.applyInsertAutoStyleOperation.bind(autoStyles1, op({ type: 'A', styleId: 'as3', attrs: {} }))).to.throw(OperationError);
                    expect(autoStyles1.getAutoStyle('as3')).to.equal(null);
                    expect(autoStyles1.applyInsertAutoStyleOperation.bind(autoStyles1, op({ type: 'A', styleId: 'a2', attrs: {} }))).to.throw(OperationError);
                    expect(autoStyles1.getAutoStyle('a2')).to.equal(null);
                    expect(autoStyles1.getAutoStyle('as2')).to.equal(null);
                    autoStyles1.applyInsertAutoStyleOperation(op({ type: 'A', styleId: 'as2', attrs: {} }));
                    expect(autoStyles1.getAutoStyle('as2')).to.be.an('object');
                });
                it('should fail to insert an auto-style with wrong family', function () {
                    expect(autoStyles1.applyInsertAutoStyleOperation.bind(autoStyles1, op({ type: 'B', styleId: 'as3', attrs: {} }))).to.throw(OperationError);
                    expect(autoStyles1.getAutoStyle('as3')).to.equal(null);
                });
                it('should insert the attributes for an auto-style', function () {
                    autoStyles1.applyInsertAutoStyleOperation(op({ type: 'A', styleId: 'as3', attrs: { styleId: 'style', A: { a1: 1 }, B: { b2: 2, wrong: 42 }, WRONG: { a1: 1 } } }));
                    expect(autoStyles1.getAutoStyle('as3')).to.be.an('object');
                    expect(autoStyles1.getMergedAttributeSet('as3')).to.deep.equal({ styleId: 'style', A: { a1: 1, a2: 'a2' }, B: { b1: 'b1', b2: 2 }, C: { c1: 'c1', c2: 'c2' } });
                    expect(autoStyles1.getExplicitAttributeSet('as3')).to.deep.equal({ styleId: 'style', A: { a1: 1 }, B: { b2: 2 } });
                });
            });
            describe('in relaxed mode', function () {
                it('should insert an auto-style', function () {
                    expect(autoStyles2.getAutoStyle('as3')).to.equal(null);
                    autoStyles2.applyInsertAutoStyleOperation(op({ type: 'B', styleId: 'as3', attrs: {} }));
                    expect(autoStyles2.getAutoStyle('as3')).to.be.an('object');
                });
                it('should insert the default auto-style', function () {
                    expect(autoStyles2.getAutoStyle('as1')).to.equal(null);
                    autoStyles2.applyInsertAutoStyleOperation(op({ type: 'B', styleId: 'as1', attrs: {}, default: true }));
                    expect(autoStyles2.getAutoStyle('as1')).to.be.an('object');
                    expect(autoStyles2.getEffectiveStyleId('')).to.equal('as1');
                    expect(autoStyles2.getDefaultStyleId()).to.equal('as1');
                });
                it('should fail to insert another default auto-style', function () {
                    expect(autoStyles2.applyInsertAutoStyleOperation.bind(autoStyles2, op({ type: 'B', styleId: 'as2', attrs: {}, default: true }))).to.throw(OperationError);
                    expect(autoStyles2.getAutoStyle('as2')).to.equal(null);
                    expect(autoStyles2.getDefaultStyleId()).to.equal('as1');
                });
                it('should fail to insert an auto-style with wrong identifier', function () {
                    expect(autoStyles2.applyInsertAutoStyleOperation.bind(autoStyles2, op({ type: 'B', styleId: 'as1', attrs: {} }))).to.throw(OperationError);
                });
                it('should fail to insert an auto-style with wrong family', function () {
                    expect(autoStyles2.applyInsertAutoStyleOperation.bind(autoStyles2, op({ type: 'A', styleId: 'as4', attrs: {} }))).to.throw(OperationError);
                    expect(autoStyles2.getAutoStyle('as4')).to.equal(null);
                });
            });
        });

        describe('method "applyChangeAutoStyleOperation"', function () {
            it('should exist', function () {
                expect(autoStyles1).to.respondTo('applyChangeAutoStyleOperation');
            });
        });
    });

    // ========================================================================
});
