/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

'use strict';

module.exports = function (grunt) {

    // imports ================================================================

    var _ = require('underscore');

    // class CollectorError ===================================================

    /**
     * A special exception used while collecting multiple errors of the task.
     *
     * @constructor
     *
     * @extends Error
     */
    function CollectorError() {
        this.name = 'CollectorError';
        this.message = 'Task errors collected.';
    } // class CollectorError

    // derive manually from built-in class Error
    CollectorError.prototype = Object.create(Error.prototype);
    CollectorError.prototype.constructor = CollectorError;

    // private helpers ========================================================

    /**
     * Returns whether the passed value is a non-empty string.
     */
    function isStringNE(value) {
        return _.isString(value) && (value.length > 0);
    }

    /**
     * Returns whether the passed value is a real object.
     */
    function isObject(value) {
        return _.isObject(value) && !_.isArray(value) && !_.isFunction(value) && !_.isRegExp(value);
    }

    /**
     * Returns whether the passed value is a non-empty array.
     */
    function isArrayNE(value) {
        return _.isArray(value) && (value.length > 0);
    }

    // recursive invocation count of task error collectors
    var collectorCount = 0;

    // whether a task error has been collected
    var errorFlag = false;

    /**
     * Error collector callback function. May be invoked recursively.
     */
    function collector(callback, context) {

        if (collectorCount === 0) {
            errorFlag = false;
        }
        collectorCount += 1;

        try {
            callback.call(context);
        } catch (error) {
            if (error instanceof CollectorError) {
                errorFlag = true;
            } else {
                throw error;
            }
        }

        collectorCount -= 1;
        if ((collectorCount === 0) && errorFlag) {
            grunt.fatal('Task failed.');
        }
    }

    /**
     * Collects errors for all elements of the passed array, or properties of
     * the passed object.
     */
    collector.forEach = function (list, callback, context) {
        _.each(list, function (element, key) {
            collector(callback.bind(context, element, key, list));
        });
    };

    /**
     * Implementation helper for the GruntUtils.check...() methods. Returns the
     * resulting boolean value of the check.
     */
    function checkHelper(value, validator, args, index) {
        if (validator(value)) { return true; }

        args = _.toArray(args).slice(index);
        GruntUtils.printWarn.apply(GruntUtils, args);
        return false;
    }

    /**
     * Implementation helper for the GruntUtils.ensure...() methods. Exits the
     * current grunt task if the performed check fails.
     */
    function ensureHelper(value, validator, args, index) {
        if (validator(value)) { return; }

        args = _.toArray(args).slice(index);
        if (collectorCount === 0) {
            grunt.fatal.apply(grunt, args);
        }

        GruntUtils.printError.apply(GruntUtils, args);
        throw new CollectorError();
    }

    // static class GruntUtils ================================================

    var GruntUtils = {};

    // generic helpers --------------------------------------------------------

    /**
     * Returns a passed array unmodified, converts null and undefined to an
     * empty array, and converts all other values to an array with one element.
     *
     * @param {Any} value
     *  Any value to be converted to an array.
     *
     * @returns {Array<Any>}
     *  An array containing the value, unless the value is already an array.
     */
    GruntUtils.makeArray = function (value) {
        return _.isArray(value) ? value : (_.isUndefined(value) || _.isNull(value)) ? [] : [value];
    };

    // logging helpers --------------------------------------------------------

    /**
     * Prints a warning message, but does not abort the current grunt task.
     *
     * @param {Any} [...]
     *  All arguments will be printed with a leading 'Warning:' label.
     */
    GruntUtils.printWarn = function () {
        grunt.log.writeln.apply(grunt.log, ['Warning:'.yellow].concat(_.toArray(arguments)));
    };

    GruntUtils.printPathMessage = function (path, message) {
        grunt.log.writeln(message + ' at ' + path.green + ' :');
    };

    /**
     * Writes the specified code excerpt to the logger.
     *
     * @param {Array<String>} lines
     *  A complete array of source code lines.
     *
     * @param {Number} line
     *  Zero-based index of the first line in 'lines' to be logged.
     *
     * @param {Number} count
     *  Number of lines to be logged.
     */
    GruntUtils.dumpLines = function (lines, line, count) {
        if (line < 0) { count += line; line = 0; }
        lines.slice(line, line + count).forEach(function (l, i) {
            grunt.log.writeln(('     ' + (line + i + 1)).substr(-6) + ' |' + l);
        });
    };

    /**
     * Writes a code excerpt with specific error position to the logger.
     *
     * @param {Array<String>} lines
     *  A complete array of source code lines.
     *
     * @param {Number} line
     *  Zero-based index of the line in 'lines' containing the error.
     *
     * @param {Number} column
     *  Zero-based index of the character containing the error.
     */
    GruntUtils.dumpErrorInLines = function (lines, line, column) {
        GruntUtils.dumpLines(lines, line - 2, 3);
        grunt.log.writeln(new Array(column + 9).join('-') + '^');
        GruntUtils.dumpLines(lines, line + 1, 2);
    };

    /**
     * Checks that the passed value is truthy. Prints a warning otherwise, but
     * does not abort the current task.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the passed value is falsy.
     *
     * @returns {Boolean}
     *  Whether the passed value is truthy.
     */
    GruntUtils.check = function (value) {
        return checkHelper(value, _.identity, arguments, 1);
    };

    /**
     * Checks that the passed value is a string. Prints a warning otherwise,
     * but does not abort the current task.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the value is not a string.
     *
     * @returns {Boolean}
     *  Whether the passed value is a string.
     */
    GruntUtils.checkString = function (value) {
        return checkHelper(value, _.isString, arguments, 1);
    };

    /**
     * Checks that the passed value is a non-empty string. Prints a warning
     * otherwise, but does not abort the current task.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the value is not a non-empty string.
     *
     * @returns {Boolean}
     *  Whether the passed value is a non-empty string.
     */
    GruntUtils.checkStringNE = function (value) {
        return checkHelper(value, isStringNE, arguments, 1);
    };

    /**
     * Checks that the passed value is an object (and neither an array, nor a
     * function, nor a regular expression). Prints a warning otherwise, but
     * does not abort the current task.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the value is not an object.
     *
     * @returns {Boolean}
     *  Whether the passed value is an object.
     */
    GruntUtils.checkObject = function (value) {
        return checkHelper(value, isObject, arguments, 1);
    };

    /**
     * Checks that the passed value is an array. Prints a warning otherwise,
     * but does not abort the current task.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the value is not an array.
     *
     * @returns {Boolean}
     *  Whether the passed value is an array.
     */
    GruntUtils.checkArray = function (value) {
        return checkHelper(value, _.isArray, arguments, 1);
    };

    /**
     * Checks that the passed value is a non-empty array. Prints a warning
     * otherwise, but does not abort the current task.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the value is not a non-empty array.
     *
     * @returns {Boolean}
     *  Whether the passed value is a non-empty array.
     */
    GruntUtils.checkArrayNE = function (value) {
        return checkHelper(value, isArrayNE, arguments, 1);
    };

    /**
     * Prints an error message, but does not abort the current grunt task.
     *
     * @param {Any} [...]
     *  All arguments will be printed with a leading 'Error:' label.
     */
    GruntUtils.printError = function () {
        grunt.log.writeln.apply(grunt.log, ['Error:'.red].concat(_.toArray(arguments)));
    };

    /**
     * Prints an error message, and aborts the task. If the task is in error
     * collector mode (see method GruntUtils.collectErrors() for more details),
     * it will resume with the next available subtask.
     *
     * @param {Any} [...]
     *  All arguments will be printed.
     */
    GruntUtils.fail = function () {
        ensureHelper(false, _.identity, arguments, 0);
    };

    /**
     * Ensures that the passed value is truthy, aborts the task otherwise.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the passed value is falsy.
     */
    GruntUtils.ensure = function (value) {
        ensureHelper(value, _.identity, arguments, 1);
    };

    /**
     * Ensures that the passed value is a string, aborts the task otherwise.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the value is not a string.
     */
    GruntUtils.ensureString = function (value) {
        ensureHelper(value, _.isString, arguments, 1);
    };

    /**
     * Ensures that the passed value is a non-empty string, aborts the task
     * otherwise.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the value is not a non-empty string.
     */
    GruntUtils.ensureStringNE = function (value) {
        ensureHelper(value, isStringNE, arguments, 1);
    };

    /**
     * Ensures that the passed value is an object (and neither an array, nor a
     * function, nor a regular expression), aborts the task otherwise.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the value is not an object.
     */
    GruntUtils.ensureObject = function (value) {
        ensureHelper(value, isObject, arguments, 1);
    };

    /**
     * Ensures that the passed value is an array, aborts the task otherwise.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the value is not an array.
     */
    GruntUtils.ensureArray = function (value) {
        ensureHelper(value, _.isArray, arguments, 1);
    };

    /**
     * Ensures that the passed value is a non-empty array, aborts the task
     * otherwise.
     *
     * @param {Any} value
     *  The value to be checked.
     *
     * @param {Any} [...]
     *  Other arguments will be printed if the value is not a non-empty array.
     */
    GruntUtils.ensureArrayNE = function (value) {
        ensureHelper(value, isArrayNE, arguments, 1);
    };

    GruntUtils.collectErrors = function (callback, context) {
        return callback.call(context, collector);
    };

    // file helpers -----------------------------------------------------------

    /**
     * Writes a log message for creation of a new file.
     *
     * @param {String} path
     *  The file path to be logged.
     */
    GruntUtils.logFileCreated = function (path) {
        grunt.log.writeln('File ' + path.cyan + ' created.');
    };

    /**
     * Creates a new text file with JSON data.
     *
     * @param {String} path
     *  The full name of the file to be written.
     *
     * @param {Any} value
     *  The value to be written as JSON data to the text file.
     */
    GruntUtils.writeJSON = function (path, value) {
        grunt.file.write(path, JSON.stringify(value));
        GruntUtils.logFileCreated(path);
    };

    /**
     * Creates a new text file from an array of strings.
     *
     * @param {String} path
     *  The full name of the file to be written.
     *
     * @param {Array<String>} contents
     *  The text contents to be written to the text file, as array of text
     *  lines.
     */
    GruntUtils.writeLines = function (path, lines) {
        grunt.file.write(path, lines.join('\n') + '\n');
        GruntUtils.logFileCreated(path);
    };

    // exports ================================================================

    return GruntUtils;
};
