/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/popup/basemenu', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/popup/basepopup'
], function (Config, Utils, KeyCodes, Forms, BasePopup) {

    'use strict';

    // events registered at the document while the menu is open
    var DOCUMENT_EVENTS = 'mousedown touchstart';

    // whether pop-up nodes remain open when losing focus (for debugging)
    var DEBUG_STICKY_POPUPS = Config.getDebugUrlFlag('office:sticky-popups');

    // class BaseMenu =========================================================

    /**
     * Wrapper class for a DOM node used as interactive pop-up menu element,
     * shown on top of the application window, and relative to an arbitrary DOM
     * node. The class adds basic browser focus handling. On opening, the
     * browser focus will be moved invisibly into the menu element to enable
     * keyboard events (focus navigation) inside the menu. When the browser
     * focus leaves the menu node (regardless how that happened, e.g. clicks or
     * taps anywhere outside the menu, or special key shortcuts to move the
     * focus away), the menu will close itself automatically.
     *
     * @constructor
     *
     * @extends BasePopup
     *
     * @param {String|Object} windowId
     *  The identifier of the root window of the context application owning the
     *  pop-up menu object, or an object with a method 'getWindowId' that
     *  returns such a window identifier. Used for debugging and logging of
     *  running timers in automated test environments.
     *
     * @param {Object} [initOptions]
     *  Initial options to control the properties of the pop-up menu element.
     *  Supports all options also supported by the base class BasePopup.
     *  Additionally, the following options are supported:
     *  @param {Boolean} [initOptions.autoFocus=true]
     *      If set to true (or omitted), the browser focus will be moved into
     *      the pop-up menu after it has been opened. If set to false, the
     *      browser focus will not be changed. In this case, the node currently
     *      focused must be contained in the option 'focusableNodes' to prevent
     *      that the menu will be closed immediately afterwards due to missing
     *      menu focus.
     *  @param {Boolean} [initOptions.autoClose=true]
     *      If set to true (or omitted), and the browser focus leaves the
     *      pop-up menu node (due to outside clicks, TAB key, etc.), the menu
     *      will be closed automatically.
     *  @param {HTMLElement|jQuery|Function} [initOptions.focusableNodes]
     *      Additional DOM nodes located outside the root node of this pop-up
     *      menu that do not cause to close the menu automatically while they
     *      are focused. More focusable nodes can be registered at runtime with
     *      the method BaseMenu.registerFocusableNodes(). Can be a function
     *      that will be called on every focus change, and that must return
     *      a jQuery object or a plain DOM element.
     *  @param {String|Function} [initOptions.preferFocusFilter]
     *      A jQuery selector that filters the available focusable control
     *      elements to a list of preferred controls, that will be used when
     *      moving the browser focus into this pop-up menu using the method
     *      BaseMenu.grabFocus(). The selector will be passed to the jQuery
     *      method jQuery.filter(). If this selector is a function, it will be
     *      called with the DOM node bound to the symbol 'this'. See the jQuery
     *      API documentation at http://api.jquery.com/filter for details.
     */
    function BaseMenu(windowId, initOptions) {

        // self reference
        var self = this;

        // whether to focus the menu after it has been opened
        var autoFocus = Utils.getBooleanOption(initOptions, 'autoFocus', true);

        // whether to close the menu automatically after losing focus
        var autoClose = Utils.getBooleanOption(initOptions, 'autoClose', true);

        // static collection of DOM elements preventing auto-close of the menu
        var focusableNodes = $();

        // dynamic callbacks for additional focusable DOM elements preventing auto-close
        var focusableNodeResolvers = [];

        // filter selector for preferred focusable elements
        var preferFocusFilter = Utils.getOption(initOptions, 'preferFocusFilter');

        // base constructor ---------------------------------------------------

        BasePopup.call(this, windowId, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Returns whether the passed DOM node is contained in the list of
         * focusable nodes which prevent auto-closing the menu if it loses the
         * browser focus.
         */
        function isFocusableNode(node) {

            // bug 33548: treat IE's focus helper node as focusable node (see Utils.clearBrowserSelection() for details)
            if ($(node).is('.focus-helper')) { return true; }

            // all focusable nodes (start with static nodes)
            var allFocusableNodes = focusableNodes;

            // add dynamic focusable nodes
            focusableNodeResolvers.forEach(function (resolver) {
                allFocusableNodes = allFocusableNodes.add(resolver.call(self));
            });

            // check whether the passed node is contained in the focusable nodes (directly, or as descendant)
            return (allFocusableNodes.filter(node).length > 0) || (allFocusableNodes.has(node).length > 0);
        }

        /**
         * Closes this menu if the passed node is not included in the
         * registered focusable nodes.
         */
        function hideMenuOnFocusNode(node) {
            if (autoClose && !(node && isFocusableNode(node))) {
                // do not close the menu, if the debug URL flag 'office:sticky-popups' is set (but still return true)
                if (!DEBUG_STICKY_POPUPS) { self.hide(null, node); }
                return true;
            }
            return false;
        }

        /**
         * Moves the browser focus to the content node, if automatic focus
         * handling is enabled. When the virtual keyboard is open this focus change will close it (e.g. virtual
         * keyboard is open and you open a popup). So check if an open virtual keyboard should be hidden or not
         * ( e.g. it should be hidden on small devices and tablet landscape to get more screen space).
         */
        function autoFocusContentNode() {

            if (autoFocus && (!Utils.isSoftKeyboardOpen() || Utils.shouldHideSoftKeyboard())) {
                Utils.setFocus(self.getContentNode());
            }
        }

        /**
         * Hides the menu node automatically when the browser focus leaves.
         */
        function changeFocusHandler(event, focusNode) {

            // close the menu unless the focus has been moved to the registered nodes
            if (hideMenuOnFocusNode(focusNode)) {
                return;
            }

            // scroll the pop-up menu to make the focused node visible
            if (focusNode && Utils.containsNode(self.getContentNode(), focusNode)) {
                self.scrollToChildNode(focusNode);
            }
        }

        /**
         * Handles mouse clicks everywhere on the page, and closes the context
         * menu automatically.
         */
        function globalClickHandler(event) {
            hideMenuOnFocusNode(event.target);
        }

        /**
         * Initializes the pop-up menu after it has been opened.
         */
        function popupShowHandler() {

            // move focus into the menu node, for keyboard handling
            autoFocusContentNode();

            // listen to focus events, hide the menu when focus leaves the menu
            self.listenTo(Utils, 'change:focus', changeFocusHandler);

            // Add global click handler that closes the menu automatically when
            // clicking somewhere in the page without changing the browser focus
            // (elements without tab-index). The event handler must be attached
            // deferred, otherwise it would close the context menu immediately
            // after it has been opened with a mouse click while the event bubbles
            // up to the document root.
            self.executeDelayed(function () {
                if (self.isVisible()) {
                    self.listenTo($(document), DOCUMENT_EVENTS, globalClickHandler);
                }
            }, 'BaseMenu.popupShowHandler');
        }

        /**
         * Deinitializes the pop-up menu before it will be closed.
         */
        function popupBeforeHideHandler() {
            // unregister the focus handler
            self.stopListeningTo(Utils, 'change:focus', changeFocusHandler);
            // unregister the global click/touch handler
            self.stopListeningTo($(document), DOCUMENT_EVENTS, globalClickHandler);
        }

        /**
         * Initializes the pop-up menu after it has left the busy mode.
         */
        function popupIdleHandler() {
            // move focus into the menu node, for keyboard handling
            autoFocusContentNode();
        }

        /**
         * Hides the pop-up menu, if the ESCAPE key has been pressed inside.
         */
        function keyDownHandler(event) {

            if (event.keyCode === KeyCodes.ESCAPE) {

                // Bugfix 42347: in case the basemenu is destroyed on hide(), other key listeners
                // should not be called on that destroyed object anymore
                event.stopImmediatePropagation();

                self.hide(event);
                return false;
            }

            // we need to call preventDefault() because of focus traveling back to the document when this menu has no input fields
            if (!$(event.target).is('input,textarea') && !KeyCodes.hasFunctionKeys(event)) {
                event.preventDefault();
            }
        }

        // methods ------------------------------------------------------------

        /**
         * Returns whether this pop-up menu contains the browser focus.
         *
         * @returns {Boolean}
         *  Whether this pop-up menu contains the browser focus.
         */
        this.hasFocus = function () {
            return Forms.containsFocus(this.getNode());
        };

        /**
         * Sets the browser focus into the first focusable control element in
         * this pop-up menu element.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.bottom=false]
         *      If set to true, the last available entry of the pop-up menu
         *      will be focused instead of the first.
         *
         * @returns {BaseMenu}
         *  A reference to this instance.
         */
        this.grabFocus = function (options) {

            // do nothing if a descendant of the content node is already focused
            if (Forms.containsFocus(this.getContentNode())) { return this; }

            // all focusable control elements
            var focusNodes = Forms.findFocusableNodes(this.getNode());
            // the preferred controls returned by the callback function
            var preferredNodes = preferFocusFilter ? focusNodes.filter(preferFocusFilter) : focusNodes;
            // whether to select the last control
            var bottom = Utils.getBooleanOption(options, 'bottom', false);

            // fall back to all focusable controls if no preferred controls are available
            if (preferredNodes.length === 0) {
                preferredNodes = (focusNodes.length > 0) ? focusNodes : this.getContentNode();
            }
            Utils.setFocus((bottom ? preferredNodes.last() : preferredNodes.first()));
            return this;
        };

        /**
         * Registers one or more DOM nodes located outside the root node of
         * this pop-up menu that do not cause to close the menu automatically
         * while they are focused.
         *
         * @param {HTMLElement|jQuery|Function} nodes
         *  The DOM node(s) to be added to the list of focusable elements. Can
         *  be a function that will be called on every focus change, and that
         *  must return a jQuery object or a plain DOM element.
         *
         * @returns {BaseMenu}
         *  A reference to this instance.
         */
        this.registerFocusableNodes = function (nodes) {
            if ((nodes instanceof HTMLElement) || (nodes instanceof $)) {
                focusableNodes = focusableNodes.add(nodes);
            } else if (_.isFunction(nodes)) {
                focusableNodeResolvers.push(nodes);
            }
            return this;
        };

        /**
         * Returns whether the pop-up menu will be focused after it has been
         * opened. See constructor option 'autoFocus' for details.
         *
         * @returns {Boolean}
         *  The current state of the auto-focus mode.
         */
        this.getAutoFocusMode = function () {
            return autoFocus;
        };

        /**
         * Specifies whether the pop-up menu will be focused after it has been
         * opened. See constructor option 'autoFocus' for details.
         *
         * @param {Boolean} newAutoFocus
         *  The new state of the auto-focus mode.
         *
         * @returns {BaseMenu}
         *  A reference to this instance.
         */
        this.setAutoFocusMode = function (newAutoFocus) {
            autoFocus = newAutoFocus;
            return this;
        };

        /**
         * Returns whether the menu will be closed automatically, when the
         * browser focus leaves the pop-up menu node. See constructor option
         * 'autoClose' for details.
         *
         * @returns {Boolean}
         *  The current state of the auto-close mode.
         */
        this.getAutoCloseMode = function () {
            return autoClose;
        };

        /**
         * Specifies whether the menu will be closed automatically, when the
         * browser focus leaves the pop-up menu node. See constructor option
         * 'autoClose' for details.
         *
         * @param {Boolean} newAutoClose
         *  The new state of the auto-close mode.
         *
         * @returns {BaseMenu}
         *  A reference to this instance.
         */
        this.setAutoCloseMode = function (newAutoClose) {
            autoClose = newAutoClose;
            return this;
        };

        // initialization -----------------------------------------------------

        // add basic pop-up menu styling
        this.getNode().addClass('popup-menu');

        // initialize static focusable nodes and dynamic callbacks for focusable nodes
        this.registerFocusableNodes(Utils.getOption(initOptions, 'focusableNodes'));

        // make the content node focusable to be able to provide keyboard support
        // without actually focusing a visible node inside the menu
        this.getContentNode().attr('tabindex', -1);
        this.registerFocusableNodes(this.getNode());

        // register event handlers
        this.on({ 'popup:show': popupShowHandler, 'popup:beforehide': popupBeforeHideHandler, 'popup:idle': popupIdleHandler });
        this.getNode().on('keydown', keyDownHandler);

        // Bug 35239: Windows/Chrome only: When clicking a button in a pop-up menu while
        // the browser tool-tip of the button is visible, the browser will not generate
        // a 'click' event (every second time only). Workaround is to prevent the default
        // action of the 'mousedown' events on button elements.
        if (_.browser.Windows && _.browser.Chrome) {
            this.getNode().on('mousedown', Forms.BUTTON_SELECTOR, function (evt) { evt.preventDefault(); });
        }

        // destroy all class members on destruction
        this.registerDestructor(function () {
            initOptions = self = focusableNodes = focusableNodeResolvers = preferFocusFilter = null;
        });

    } // class BaseMenu

    // exports ================================================================

    // derive this class from class BasePopup
    return BasePopup.extend({ constructor: BaseMenu });

});
