/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/object/baseobject', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/scheduler'
], function (Utils, Scheduler) {

    'use strict';

    // a predicate function that matches everything
    var ALL_MATCHER = _.constant(true);

    // private global functions ===============================================

    /**
     * Returns a matcher predicate function for the passed event source object.
     */
    function createEventSourceMatcher(source) {

        // accept plain DOM elements, convert to jQuery collection
        if (source instanceof HTMLElement) { source = $(source); }

        // return the predicate function
        return function sourceMatcher(s) {
            // jQuery objects must be tested with is() method instead of equality operator
            return (s instanceof $) ? s.is(source) : (s === source);
        };
    }

    /**
     * Returns a matcher predicate function for the passed event types (null to
     * match all, or string list).
     */
    function createEventTypesMatcher(types) {

        // all types will match for non-string value
        if (typeof types !== 'string') { return ALL_MATCHER; }

        // convert space-separated string to a flag set of single event types
        var typeSet = Utils.makeSet(types.split(/\s+/).filter(_.identity));

        // return the predicate function
        return function (t) { return t in typeSet; };
    }

    /**
     * Returns a matcher predicate function for the passed event listener (null
     * to match all, or function).
     */
    function createEventListenerMatcher(listener) {
        return listener ? function (l) { return l === listener; } : ALL_MATCHER;
    }

    /**
     * Unbinds the passed event type and/or listener (either parameter can be
     * null).
     */
    function unbindEventListener(listeners, source, type, listener) {

        // the matcher predicate for the passed event source object
        var sourceMatcher = createEventSourceMatcher(source);
        // a matcher predicate for an event type
        var typeMatcher = createEventTypesMatcher(type);
        // a matcher predicate for a listener callback function
        var listenerMatcher = createEventListenerMatcher(listener);

        // detach all matching event listeners for the passed object
        listeners.forEach(function (data) {
            if (sourceMatcher(data.source) && typeMatcher(data.type) && listenerMatcher(data.listener)) {
                source.off(data.type, data.listener);
            }
        });
    }

    /**
     * Registers and binds the passed event type and listener.
     */
    function bindEventListener(listeners, source, type, listener, once) {

        // once mode: create a version of the listener that unbinds itself on first invocation
        var listenerFunc = once ? function () {
            unbindEventListener(listeners, source, type, listenerFunc);
            return listener.apply(this, arguments);
        } : listener;

        // register and bind the listener
        listeners.push({ source: source, type: type, listener: listenerFunc });
        source.on(type, listenerFunc);
    }

    /**
     * Registers and binds the passed event types and listeners.
     */
    function bindEventListeners(listeners, source, type, listener, once) {

        // accept plain DOM elements, convert to jQuery collection
        if (source instanceof HTMLElement) { source = $(source); }

        // bug 36850: handle event maps and string/function parameters
        var eventMap = _.isObject(type) ? type : Utils.makeSimpleObject(type, listener);
        _.each(eventMap, function (evtListener, evtTypes) {
            _.each(evtTypes.split(/\s+/), function (evtType) {
                if (evtType.length > 0) {
                    bindEventListener(listeners, source, evtType, evtListener, once);
                }
            });
        });
    }

    /**
     * Returns whether the passed value is a pending abortable promise.
     *
     * @param {Any} value
     *  The value to be inspected.
     *
     * @returns {Boolean}
     *  Whether the passed value is a pending abortable promise.
     */
    function isPendingAbortablePromise(value) {
        return Utils.isPromise(value) && (value.state() === 'pending') && _.isFunction(value.abort);
    }

    /**
     * Adds an abort() method to the passed promise which invokes the specified
     * callback functions. The abort() method will accept a single optional
     * parameter 'cause', that if omitted defaults to the string 'abort'.
     *
     * @param {jQuery.Promise}
     *  A promise that will be extended with an abort() method.
     *
     * @param {Function} abort
     *  The implementation of the generated abort() method. Will be called with
     *  undefined context, and forwards the parameter 'cause' passed to the
     *  generated abort() method.
     *
     * @param {Function} [custom]
     *  An additional custom user-defined callback function that will be called
     *  before executing the specified 'abort' callback function. Will be
     *  called in the context of the promise, and forwards the parameter
     *  'cause' passed to the generated abort() method.
     */
    function createAbortMethod(promise, abort, custom) {

        // shortcut for resolved/rejected promises
        if (promise.state() !== 'pending') {
            promise.abort = Utils.NOOP;
            return;
        }

        // create the new abort() method
        promise.abort = function (cause) {
            // prevent recursive calls from callback functions
            if (promise.state() === 'pending') {
                if (_.isUndefined(cause)) { cause = 'abort'; }
                if (_.isFunction(custom)) { custom.call(this, cause); }
                abort(cause);
            }
            return this;
        };

        // replace the abort() method in resolved promise
        promise.always(function () { promise.abort = Utils.NOOP; });
    }

    /**
     * Overrides the then() method of the passed abortable promise with a
     * version that creates and returns a piped promise with an abort() method
     * which aborts the original abortable promise.
     *
     * @param {jQuery.Promise} abortablePromise
     *  The abortable promise whose then() method will be overridden.
     */
    function overrideThenMethod(abortablePromise) {

        // the original then() method of the passed promise
        var thenMethod = abortablePromise.then.bind(abortablePromise);
        // the original abort() method of the passed promise
        var abortMethod = abortablePromise.abort.bind(abortablePromise);

        // create the new then() method which returns an abortable promise
        abortablePromise.then = function (done, fail, notify, abort) {

            // the promise returned by the done or fail handler
            var resultPromise = null;

            // Returns a wrapper function for the passed callback function (the 'done' or 'fail' handlers passed
            // to the then() method of the source promise). Upon invocation, the result value of the callback will
            // be inspected, and if it is a pending abortable promise, it will be stored internally. If the
            // resulting chained promise will be aborted, the cached result promise will be aborted too.
            function callbackWrapper(callback) {
                return _.isFunction(callback) ? function () {
                    var result = callback.apply(this, arguments);
                    if (isPendingAbortablePromise(result)) { resultPromise = result; }
                    return result;
                } : callback;
            }

            // the new piped promise returned by the original then() method
            var pipedPromise = thenMethod(callbackWrapper(done), callbackWrapper(fail), notify);

            // add a custom abort() method, that aborts the passed promise
            createAbortMethod(pipedPromise, abortMethod, function (cause) {
                if (_.isFunction(abort)) { abort(cause); }
                if (resultPromise) { resultPromise.abort(cause); }
            });

            // override the then() method of the promise with an abortable version
            overrideThenMethod(pipedPromise);

            return pipedPromise;
        };

        // prevent using the deprecated pipe() method
        delete abortablePromise.pipe;
    }

    /**
     * Returns a new function wrapping the passed callback function, intended
     * to be used by an instance of BaseObject waiting for a promise.
     *
     * @param {BaseObject} object
     *  The instance of BaseObject that wants to wait for the passed promise,
     *  and invoke the passed callback function, unless it destroys itself.
     *
     * @param {Function} callback
     *  The callback function to be invoked by the passed object instance after
     *  waiting for a promise.
     *
     * @returns {Function}
     *  A new function that will invoke the passed callback function, unless
     *  the object instance is currently destroying itself, or has already been
     *  destroyed.
     */
    function wrapPromiseCallback(object, callback) {
        return function () {
            if (!object.destroying && !object.destroyed) {
                return callback.apply(object, arguments);
            }
        };
    }

    // class BaseObject =======================================================

    /**
     * A generic base class for all classes that want to register destructor
     * code that will be executed when the public method BaseObject.destoy()
     * has been invoked. Provides helper methods for listening to events and
     * promises, regarding the lifetime of the instances.
     *
     * @constructor
     *
     * @param {String|BaseObject} windowId
     *  The identifier of the root window of the context application owning the
     *  instances of this class, or another instance of this class whose window
     *  identifier will be used. Used for debugging and logging of running
     *  timers and pending promises in automated test environments. The
     *  constant BaseObject.SINGLETON can be passed if an instance of this
     *  class will be used as global static singleton that is not bound to any
     *  context application.
     */
    var BaseObject = _.makeExtendable(function (windowId) {

        // the window identifier used in automated tests
        this.__windowId = windowId.getWindowId ? windowId.getWindowId() : windowId;

        // internal unique object identifier
        this.__uid = _.uniqueId('obj');

        // all event source objects and listeners
        this.__listeners = null;

        // all pending abortable promises
        this.__promises = null;

        // all destructor callbacks
        this.__dtors = null;

    }); // class BaseObject

    // constants --------------------------------------------------------------

    /**
     * A pseudo window identifier that can be passed to the constructor of the
     * class BaseObject if an instance of that class will be used as global
     * static singleton that is not bound to any context application.
     *
     * @constant
     * @type String
     */
    BaseObject.SINGLETON = 'obj-global-singleton';

    // protected methods ------------------------------------------------------

    /**
     * Registers a destructor callback function that will be invoked when the
     * method BaseObject.destroy() has been called on this instance.
     *
     * @param {Function} destructor
     *  A destructor callback function. The method BaseObject.destroy() will
     *  invoke all registered destructor callbacks in reverse order of their
     *  registration. The callback functions will be invoked in the context of
     *  this instance.
     *
     * @returns {BaseObject}
     *  A reference to this instance.
     */
    BaseObject.prototype.registerDestructor = function (destructor) {
        (this.__dtors || (this.__dtors = [])).unshift(destructor);
        return this;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns the window identifier for this instance.
     *
     * @returns {String|BaseObject}
     *  The window identifier for this instance, as passed to the constructor.
     */
    BaseObject.prototype.getWindowId = function () {
        return this.__windowId;
    };

    /**
     * Returns the globally unique identifier for this instance.
     *
     * @returns {String}
     *  The globally unique identifier for this instance.
     */
    BaseObject.prototype.getUid = function () {
        return this.__uid;
    };

    /**
     * Destroys this object. Invokes all destructor callback functions that
     * have been registered for this instance in reverse order. Afterwards,
     * all public properties and methods of this instance will be deleted,
     * and a single property 'destroyed' will be inserted and set to the
     * value true.
     */
    BaseObject.prototype.destroy = function () {

        // set a flag that specifies that the destructor is currently running
        this.destroying = true;

        // Abort all asynchronous code to prevent JS errors from callbacks when the
        // promises resolve/reject normally. Each aborted promise removes itself from
        // the array, but a single abort() call inside the while-loop MAY abort other
        // dependent promises as well, therefore a while loop will be used that checks
        // the array length intentionally in each iteration.
        var promises = this.__promises;
        if (promises) {
            while (promises.length > 0) { promises[0].abort('destroy'); }
        }

        // unregister all event listeners
        var listeners = this.__listeners;
        if (listeners) {
            listeners.forEach(function (data) {
                // source object may have been destroyed already
                if (data.source.off) { data.source.off(data.type, data.listener); }
            });
        }

        // invoke all destructor callbacks
        var destructors = this.__dtors;
        if (destructors) {
            destructors.forEach(function (destructor) { destructor.call(this); }, this);
        }

        // delete all public members, to detect any misuse after destruction
        var uid = this.__uid;
        _.each(this, function (value, key) { delete this[key]; }, this);
        this.destroyed = true;
        this.uid = uid;
    };

    // promises ---------------------------------------------------------------

    /**
     * Creates a promise for the passed deferred object representing code
     * running asynchronously. The promise will contain an additional
     * method abort() that (when called before the deferred object has been
     * resolved or rejected) invokes the specified callback function, and
     * rejects the passed deferred object. Additionally, the promise will
     * be stored internally as long as it is in pending state. When this
     * instance will be destroyed, all pending promises will be aborted
     * automatically with the value 'destroy'.
     *
     * @param {jQuery.Deferred} deferred
     *  The deferred object to create an abortable promise for.
     *
     * @param {Function} [callback]
     *  An optional callback function that will be invoked when the promise
     *  has been aborted, and the deferred object is still pending. Will be
     *  called in the context of this instance.
     *
     * @param {Number} [timeout]
     *  If specified and a positive number, the delay time in milliseconds
     *  after the promise returned by this method will be rejected with the
     *  value 'timeout', if the passed original deferred object is still
     *  pending.
     *
     * @returns {jQuery.Promise}
     *  A promise for the passed deferred object, with an additional method
     *  abort().
     */
    BaseObject.prototype.createAbortablePromise = function (deferred, callback, timeout) {

        // the promise of the passed deferred object
        var promise = deferred.promise();

        // add a custom abort() method to the promise, rejecting the passed deferred object
        createAbortMethod(promise, deferred.reject.bind(deferred), callback ? callback.bind(this) : null);

        // override the then() method of the promise with an abortable version
        overrideThenMethod(promise);

        // do not process a deferred object that is not pending anymore, but create the abort() method
        if (promise.state() === 'pending') {

            // create the array of pending promises on demand, add the new promise
            var promises = this.__promises || (this.__promises = []);
            promises.push(promise);

            // remove the promise from the array when it resolves
            promise.always(function () {
                promises.splice(promises.indexOf(promise), 1);
            });

            // abort automatically after the specified timeout
            if (_.isNumber(timeout) && (timeout > 0)) {
                var timer = window.setTimeout(function () {
                    if (promise.state() === 'pending') { promise.abort('timeout'); }
                }, timeout);
                promise.always(function () {
                    if (timer) { window.clearTimeout(timer); timer = null; }
                });
            }
        }

        callback = null;
        return promise;
    };

    /**
     * Creates an abortable promise that has been resolved with the passed
     * result value.
     *
     * @param {Any} [result]
     *  The result value to be provided by the returned promise. If omitted,
     *  the promise will be resolved without a value.
     *
     * @returns {jQuery.Promise}
     *  An abortable promise that has been resolved with the passed result
     *  value. The method abort() of the promise will do nothing. The method
     *  then() of the promise will return abortable promises too.
     */
    BaseObject.prototype.createResolvedPromise = function (result) {
        return this.createAbortablePromise(new $.Deferred().resolve(result));
    };

    /**
     * Creates an abortable promise that has been rejected with the passed
     * result value.
     *
     * @param {Any} [result]
     *  The result value to be provided by the returned promise. If omitted,
     *  the promise will be rejected without a value.
     *
     * @returns {jQuery.Promise}
     *  An abortable promise that has been rejected with the passed result
     *  value. The method abort() of the promise will do nothing. The method
     *  then() of the promise will return abortable promises too.
     */
    BaseObject.prototype.createRejectedPromise = function (result) {
        return this.createAbortablePromise(new $.Deferred().reject(result));
    };

    /**
     * Converts the passed arbitrary value to a promise.
     *
     * @param {Any} result
     *  The result value to be converted to a promise. If this value is already
     *  a promise, it will be returned immediately. If the result value matches
     *  the 'rejectOn' parameter, a promise will be created that has been
     *  rejected with that value. Otherwise, a promise will be created that has
     *  been resolved with the passed value.
     *
     * @param {Any} [rejectOn]
     *  If specified, a predicate callback function or a constant value that
     *  will be compared with the passed result value in order to decide
     *  whether to reject the promise created by this method. The callback
     *  function will receive the result value as first parameter, and a truthy
     *  return value will cause to reject the promise. Any other value will be
     *  compared directly to the result value. On strict equality, the promise
     *  will be rejected.
     *
     * @returns {jQuery.Promise}
     *  The promise that has been resolved or rejected with the passed result
     *  value.
     */
    BaseObject.prototype.convertToPromise = function (result, rejectOn) {

        // directly return a passed promise
        if (Utils.isPromise(result)) { return result; }

        // return a resolved promise if nothing has been passed
        if (arguments.length === 1) { return this.createResolvedPromise(); }

        // invoke a predicate function, or compare with any other value
        var rejected = (typeof rejectOn === 'function') ? rejectOn.call(this, result) : (result === rejectOn);
        return rejected ? this.createRejectedPromise(result) : this.createResolvedPromise(result);
    };

    /**
     * Creates a new pending deferred object that is bound to the lifetime of
     * this instance.
     *
     * @param {String} logMessage
     *  A description for the pending promise that will be inserted into the
     *  debug log used in automated test environments.
     *
     * @param {Object} [logOptions]
     *  Optional parameters for debug logging:
     *  @param {Boolean} [logOptions.background=false]
     *      Whether the pending promise represents a long-running background
     *      task with low priority. These background tasks will be ignored when
     *      waiting for pending tasks in automated test environments.
     *
     * @returns {jQuery.Deferred}
     *  A new pending deferred object.
     */
    BaseObject.prototype.createDeferred = function (logMessage, logOptions) {

        // the new deferred object (will be registered for logging in automated test environments)
        var deferred = Scheduler.createDeferred(this.__windowId, logMessage, logOptions);
        // reject the deferred object when this instance will be destroyed
        this.createAbortablePromise(deferred);

        return deferred;
    };

    /**
     * Creates an abortable promise that will be rejected automatically
     * after a specific time, and will invoke a callback function. The
     * promise will contain an additional method abort() that can be used
     * to shorten the timeout. When this instance will be destroyed, all
     * pending timeouts will be aborted automatically with the value
     * 'destroy'.
     *
     * @param {Number} timeout
     *  The delay time in milliseconds after the promise returned by this
     *  method will be rejected with the value 'timeout'.
     *
     * @param {Function} [callback]
     *  An optional callback function that will be invoked when the promise
     *  has been aborted, or the specified time has elapsed. Will be called
     *  in the context of this instance. This parameter can be omitted
     *  completely (parameter 'logMessage' becomes the second parameter).
     *
     * @param {String} logMessage
     *  A description for the pending promise that will be inserted into
     *  the debug log used in automated test environments.
     *
     * @param {Object} [logOptions]
     *  Optional parameters for debug logging:
     *  @param {Boolean} [logOptions.background=false]
     *      Whether the pending promise represents a long-running
     *      background task with low priority. These background tasks will
     *      be ignored when waiting for pending tasks in automated test
     *      environments.
     *
     * @returns {jQuery.Promise}
     *  A promise with an additional method abort().
     */
    BaseObject.prototype.createTimeout = function (timeout, callback, logMessage, logOptions) {

        // insert missing 'callback' parameter
        if (typeof callback === 'string') {
            logOptions = logMessage;
            logMessage = callback;
            callback = null;
        }

        // a deferred object (will be registered for logging in automated test environments)
        var deferred = Scheduler.createDeferred(this.__windowId, logMessage, logOptions);
        // reject the deferred object after the timeout, or when this instance will be destroyed
        return this.createAbortablePromise(deferred, callback, timeout);
    };

    /**
     * Registers a callback function at the specified promise that waits for it
     * to be resolved. When this instance will be destroyed before the promise
     * resolves, the callback function will not be invoked.
     *
     * @param {jQuery.Promise} promise
     *  The promise.
     *
     * @param {Function} callback
     *  The callback function to be invoked when the promise resolves.
     *
     * @returns {BaseObject}
     *  A reference to this instance.
     */
    BaseObject.prototype.waitForSuccess = function (promise, callback) {
        // register a callback function that checks the destroy flags of this instance
        promise.done(wrapPromiseCallback(this, callback));
        return this;
    };

    /**
     * Registers a callback function at the specified promise that waits for it
     * to be rejected. When this instance will be destroyed before the promise
     * rejects, the callback function will not be invoked.
     *
     * @param {jQuery.Promise} promise
     *  The promise.
     *
     * @param {Function} callback
     *  The callback function to be invoked when the promise rejects.
     *
     * @returns {BaseObject}
     *  A reference to this instance.
     */
    BaseObject.prototype.waitForFailure = function (promise, callback) {
        // register a callback function that checks the destroy flags of this instance
        promise.fail(wrapPromiseCallback(this, callback));
        return this;
    };

    /**
     * Registers a callback function at the specified promise that waits for it
     * to be resolved or rejected. When this instance will be destroyed before
     * the promise resolves or rejects, the callback function will not be
     * invoked.
     *
     * @param {jQuery.Promise} promise
     *  The promise.
     *
     * @param {Function} callback
     *  The callback function to be invoked when the promise resolves or
     *  rejects.
     *
     * @returns {BaseObject}
     *  A reference to this instance.
     */
    BaseObject.prototype.waitForAny = function (promise, callback) {
        // register a callback function that checks the destroy flags of this instance
        promise.always(wrapPromiseCallback(this, callback));
        return this;
    };

    // events -----------------------------------------------------------------

    /**
     * Registers an event listener at the specified event source object. On
     * destruction of this instance, all event listeners registered with this
     * method will be removed from the source object.
     *
     * @param {Events|jQuery|HTMLElement} source
     *  The event source object. Can be any object that provides a method on()
     *  to register an event listener (for example, jQuery objects, or any
     *  object extended with the core Events mix-in class). Additionally, a
     *  plain DOM element instance can be passed, which will be converted to a
     *  jQuery collection internally.
     *
     * @param {String|Object} type
     *  The type of the event to listen to. Can be a space-separated list of
     *  event type names. Alternatively, can be an object mapping event types
     *  to listener callback functions.
     *
     * @param {Function} [listener]
     *  The event listener that will be invoked for the events triggered by the
     *  event source object. This parameter will be ignored, if the parameter
     *  'type' is an object mapping event types to listener callback functions.
     *
     * @returns {BaseObject}
     *  A reference to this instance.
     */
    BaseObject.prototype.listenTo = function (source, type, listener) {

        // create the array of known listeners on demand
        this.__listeners = this.__listeners || [];

        // bind the listener to the event source
        bindEventListeners(this.__listeners, source, type, listener, false);

        return this;
    };

    /**
     * Registers an event listener at the specified event source object, that
     * will be triggered exactly once, and will unregister itself afterwards.
     * On destruction of this instance, all event listeners registered with
     * this method will be removed from the source object.
     *
     * @param {Events|jQuery|HTMLElement} source
     *  The event source object. See method BaseObject.listenTo() for more
     *  details about supported object types.
     *
     * @param {String|Object} type
     *  The type of the event to listen to. Can be a space-separated list of
     *  event type names. Alternatively, can be an object mapping event types
     *  to listener callback functions.
     *
     * @param {Function} [listener]
     *  The event listener that will be invoked for the events triggered by the
     *  event source object. This parameter will be ignored, if the parameter
     *  'type' is an object mapping event types to listener callback functions.
     *
     * @returns {BaseObject}
     *  A reference to this instance.
     */
    BaseObject.prototype.listenOnceTo = function (source, type, listener) {

        // create the array of known listeners on demand
        this.__listeners = this.__listeners || [];

        // bind the listener to the event source
        bindEventListeners(this.__listeners, source, type, listener, true);

        return this;
    };

    /**
     * Removes event listeners that have been registered for the passed event
     * source object using the method BaseObject.listenTo().
     *
     * @param {Events|jQuery|HTMLElement} source
     *  The event source object, as has been passed to the public method
     *  BaseObject.listenTo() before.
     *
     * @param {String|Object} [type]
     *  If specified, the type of the event to stop listening to. Can be a
     *  space-separated list of event type names. Alternatively, can be an
     *  object mapping event types to listener callback functions. If omitted,
     *  the event listeners for all registered event types will be removed
     *  (optionally filtered by a specific event listener function, see
     *  parameter 'listener').
     *
     * @param {Function} [listener]
     *  If specified, the event listener that will be removed for the event
     *  source object. If omitted, all event listeners of the event source will
     *  be removed (optionally filtered by specific event types, see parameter
     *  'type'). This parameter will be ignored, if the parameter 'type' is an
     *  object mapping event types to listener callback functions.
     *
     * @returns {BaseObject}
     *  A reference to this instance.
     */
    BaseObject.prototype.stopListeningTo = function (source, type, listener) {

        // nothing to do without registered listeners
        var listeners = this.__listeners;
        if (!listeners) { return this; }

        if (_.isFunction(type) && _.isUndefined(listener)) {
            // 'type' may be omitted completely (instead of being set to null)
            unbindEventListener(listeners, source, null, type);
        } else if (_.isObject(type)) {
            // bug 36850: unbind all passed event types of an event map
            _.each(type, function (listener2, type2) {
                unbindEventListener(listeners, source, type2, listener2);
            });
        } else {
            unbindEventListener(listeners, source, type, listener);
        }

        return this;
    };

    /**
     * Registers an event listener at the specified event source object, that
     * will remain active as long as the passed promise is in pending state. On
     * destruction of this instance, all event listeners registered with this
     * method will be removed from the source object.
     *
     * @param {jQuery.Promise} promise
     *  A promise that will restrict the life time of the event listener. When
     *  this promise changes to resolved or rejected state, the event listener
     *  will be removed automatically.
     *
     * @param {Events|jQuery|HTMLElement} source
     *  The event source object. See method BaseObject.listenTo() for more
     *  details about supported object types.
     *
     * @param {String|Object} type
     *  The type of the event to listen to. Can be a space-separated list of
     *  event type names. Alternatively, can be an object mapping event types
     *  to listener callback functions.
     *
     * @param {Function} [listener]
     *  The event listener that will be invoked for the events triggered by the
     *  event source object. This parameter will be ignored, if the parameter
     *  'type' is an object mapping event types to listener callback functions.
     *
     * @returns {BaseObject}
     *  A reference to this instance.
     */
    BaseObject.prototype.listenToWhile = function (promise, source, type, listener) {
        this.listenTo(source, type, listener);
        promise.always(function () { this.stopListeningTo(source, type, listener); }.bind(this));
        return this;
    };

    /**
     * Registers an event listener at the specified event source object, that
     * will remain active as long as the passed promise is in pending state,
     * that will be triggered exactly once, and will unregister itself
     * afterwards. On destruction of this instance, all event listeners
     * registered with this method will be removed from the source object.
     *
     * @param {jQuery.Promise} promise
     *  A promise that will restrict the life time of the event listener. When
     *  this promise changes to resolved or rejected state, the event listener
     *  will be removed automatically.
     *
     * @param {Events|jQuery|HTMLElement} source
     *  The event source object. See method BaseObject.listenTo() for more
     *  details about supported object types.
     *
     * @param {String|Object} type
     *  The type of the event to listen to. Can be a space-separated list of
     *  event type names. Alternatively, can be an object mapping event types
     *  to listener callback functions.
     *
     * @param {Function} [listener]
     *  The event listener that will be invoked for the events triggered by the
     *  event source object. This parameter will be ignored, if the parameter
     *  'type' is an object mapping event types to listener callback functions.
     *
     * @returns {BaseObject}
     *  A reference to this instance.
     */
    BaseObject.prototype.listenOnceToWhile = function (promise, source, type, listener) {
        this.listenOnceTo(source, type, listener);
        promise.always(function () { this.stopListeningTo(source, type, listener); }.bind(this));
        return this;
    };

    /**
     * Returns a promise that will be resolved when the passed event source
     * object triggers the specified event.
     *
     * @param {Events|jQuery|HTMLElement} source
     *  The event source object. See method BaseObject.listenTo() for more
     *  details about supported object types.
     *
     * @param {String} type
     *  The type of the event to wait for. Can be a space-separated list of
     *  event type names. In the latter case, the first event matching one of
     *  the specified types will resolve the promise, subsequent events with
     *  matching type will do nothing.
     *
     * @param {Number} [timeout]
     *  If specified, a delay time in milliseconds. If the source object does
     *  not trigger the expected event before the time has elapsed, the promise
     *  will be rejected with the string 'timeout'.
     *
     * @returns {jQuery.Promise}
     *  An abortable promise that will be resolved with all parameters passed
     *  with the event (including the leading event object), or that will be
     *  rejected with the string 'timeout' after the specified delay time. The
     *  promise contains the additional method abort() that allows to stop
     *  listening to the pending event immediately.
     */
    BaseObject.prototype.waitForEvent = function (source, type, timeout) {

        // the deferred object representing the event state
        var deferred = Scheduler.createDeferred(this.__windowId, 'BaseObject.waitForEvent');
        // the abortable promise for the deferred object, with timeout
        var promise = this.createAbortablePromise(deferred, null, timeout);
        // the event handler
        var handler = deferred.resolve.bind(deferred);

        // listen to the specified event(s), and resolve the deferred object
        this.listenToWhile(promise, source, type, handler);

        // unregister the listener (in case multiple event types have been passed)
        return promise.always(this.stopListeningTo.bind(this, source, type, handler));
    };

    // timers -----------------------------------------------------------------

    /**
     * Invokes the passed callback function once in a browser timeout. If this
     * instance will be destroyed before the callback function has been
     * started, it will not be called anymore.
     *
     * @param {Function} callback
     *  The callback function that will be executed in a browser timeout after
     *  the delay time. Does not receive any parameters. Will be called in the
     *  context of this instance.
     *
     * @param {String} logMessage
     *  A description for the timer that will be inserted into the debug log
     *  used in automated test environments.
     *
     * @param {Object|Number} [options]
     *  Optional parameters:
     *  @param {Number} [options.delay=0]
     *      The time (in milliseconds) the execution of the passed callback
     *      function will be delayed at least.
     *  @param {Boolean} [options.background=false]
     *      Whether the timer is part of a long-running background task with
     *      low priority. These background tasks will be ignored when waiting
     *      for pending tasks in automated test environments.
     *  This parameter can also be a number, allowing to pass the value of the
     *  option 'delay' directly.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after the callback function has been
     *  executed. If the callback function returns a simple value or object,
     *  the promise will be resolved with that value. If the callback function
     *  returns a promise by itself, its state and result value will be
     *  forwarded to the promise returned by this method. The promise contains
     *  an additional method abort() that can be called before the timeout has
     *  been fired to cancel the pending execution of the callback function. In
     *  that case, the promise will be rejected. If this instance will be
     *  destroyed before invoking the callback function, the timer will be
     *  aborted automatically, and the promise will be rejected.
     */
    BaseObject.prototype.executeDelayed = function (callback, logMessage, options) {

        // the delay time for the next execution of the callback
        var delay = _.isNumber(options) ? Math.max(0, options) : Utils.getIntegerOption(options, 'delay', 0, 0);
        // a pending abortable result promise returned by the callback function
        var abortablePromise = null;
        // the deferred object representing the timer state
        var deferred = Scheduler.createDeferred(this.__windowId, logMessage + ' [executeDelayed]', options);

        // bind callback to this instance
        callback = callback.bind(this);

        // create a new synchronized timer
        var timerId = Scheduler.setTimeout(function () {

            // timer fired successfully: immediately clear the timer identifier
            timerId = null;

            // the result of the callback function
            var result = callback();

            // forward the result to the promise returned by this method
            if (Utils.isPromise(result)) {
                if ((result.state() === 'pending') && _.isFunction(result.abort)) { abortablePromise = result; }
                result.done(deferred.resolve.bind(deferred));
                result.fail(deferred.reject.bind(deferred));
            } else {
                deferred.resolve(result);
            }
        }, delay, this.__windowId, logMessage, options);

        // the abortable promise for the deferred object
        return this.createAbortablePromise(deferred, function () {
            if (timerId) { Scheduler.clearTimeout(timerId); timerId = null; }
            if (abortablePromise) { abortablePromise.abort(); }
        });
    };

    /**
     * Invokes the passed callback function repeatedly in a browser timeout
     * loop. If this instance will be destroyed before the callback function
     * has been started, or while the callback function will be repeated, it
     * will not be called anymore.
     *
     * @param {Function} callback
     *  The callback function that will be invoked repeatedly in a browser
     *  timeout loop after the initial delay time. Receives the zero-based
     *  index of the execution cycle as first parameter. Will be called in the
     *  context of this instance. The return value of the function will be used
     *  to decide whether to continue the repeated execution. If the function
     *  returns Utils.BREAK, execution will be stopped. If the function returns
     *  a promise, looping will be deferred until the promise is resolved or
     *  rejected. After resolving the promise, the delay time will start, and
     *  the callback will be invoked again. Otherwise, after the promise has
     *  been rejected, execution will be stopped, and the promise returned by
     *  this method will be rejected too. All other return values will be
     *  ignored, and the callback loop will continue.
     *
     * @param {String} logMessage
     *  A description for the timer loop that will be inserted into the debug
     *  log used in automated test environments.
     *
     * @param {Object|Number} [options]
     *  Optional parameters:
     *  @param {Number} [options.delay]
     *      The time (in milliseconds) the initial invocation of the passed
     *      callback function will be delayed. If this option will be omitted,
     *      the callback function will be executed immediately (synchronously)
     *      once with the invocation of this method.
     *  @param {Number} [options.repeatDelay]
     *      The time (in milliseconds) the repeated invocation of the passed
     *      callback function will be delayed. This value must not be less than
     *      10 milliseconds. If omitted, the specified initial delay time that
     *      has been passed with the option 'delay' will be used, unless it has
     *      been omitted, or is less than 10 milliseconds (in this case, the
     *      minimum value of 10 milliseconds will be used).
     *  @param {Number} [options.cycles]
     *      If specified, the maximum number of cycles to be executed.
     *  @param {Boolean} [options.fastAsync=false]
     *      If set to true, and the callback function returns a pending
     *      promise, the delay time for the next invocation of the callback
     *      function will be reduced by the time the promise is pending. By
     *      default, the delay time always starts after the promise has been
     *      resolved.
     *  @param {Boolean} [options.background=false]
     *      Whether the timer loop represents a long-running background task
     *      with low priority. These background tasks will be ignored when
     *      waiting for pending tasks in automated test environments.
     *  This parameter can also be a number, allowing to pass the value of the
     *  option 'delay' (and also 'repeatDelay' which defaults to the value of
     *  the option 'delay') directly.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved or rejected after the callback function
     *  has been invoked the last time. The promise will be resolved, if the
     *  callback function has returned the Utils.BREAK object in the last
     *  iteration, or if the maximum number of iterations (see option 'cycles')
     *  has been reached. It will be rejected, if the callback function has
     *  returned a rejected promise. The returned promise contains an
     *  additional method abort() that can be called before or while the
     *  callback loop is executed to stop the loop immediately. In that case,
     *  the promise will be rejected. If this instance will be destroyed before
     *  or during the callback loop, the loop timer will be aborted
     *  automatically, and the promise will be rejected.
     */
    BaseObject.prototype.repeatDelayed = function (callback, logMessage, options) {

        // self reference for local functions
        var self = this;
        // the delay time for the first invocation of the callback
        var initDelay = Utils.isFiniteNumber(options) ? Math.max(0, options) : Utils.getIntegerOption(options, 'delay', null, 0);
        // the delay time for the next invocations of the callback
        var repeatDelay = Math.max(10, Utils.getIntegerOption(options, 'repeatDelay', initDelay));
        // the number of cycles to be executed
        var cycles = Utils.getIntegerOption(options, 'cycles');
        // whether to reduce delay time after asynchronous callbacks
        var fastAsync = Utils.getBooleanOption(options, 'fastAsync', false);
        // whether this timer loop is a background task
        var background = Utils.getBooleanOption(options, 'background', false);
        // the index of the next cycle
        var index = 0;
        // the current timer promise that defers invoking the callback
        var timer = null;
        // current pending abortable result promise returned by the callback function
        var abortablePromise = null;
        // the resulting deferred object
        var deferred = Scheduler.createDeferred(this.__windowId, logMessage + ' [repeatDelayed]', options);

        // bind callback to this instance
        callback = callback.bind(this);

        // invokes the callback function
        function invokeCallback() {

            // the result of the callback, immediately break the loop if callback returns Utils.BREAK
            var result = callback(index);
            if (result === Utils.BREAK) {
                deferred.resolve(Utils.BREAK);
                return;
            }

            // wait for the result
            if (Utils.isPromise(result)) {
                if ((result.state() === 'pending') && _.isFunction(result.abort)) { abortablePromise = result; }
                var t0 = (result.state() === 'pending') ? _.now() : -1;
                result.done(function () { startNextTimer(t0); });
                result.fail(deferred.reject.bind(deferred));
            } else {
                startNextTimer(-1);
            }
        }

        // creates and registers a browser timeout that executes the callback
        function createTimer(delay) {
            abortablePromise = null;
            timer = self.executeDelayed(invokeCallback, logMessage, background ? { delay: delay, background: true } : delay);
        }

        function startNextTimer(t0) {
            // do not create a new timer if execution has been aborted while the asynchronous code was running
            if (deferred.state() !== 'pending') { return; }
            // decide whether to start the next iteration
            index += 1;
            if (!_.isNumber(cycles) || (index < cycles)) {
                createTimer((fastAsync && (t0 > 0)) ? Math.max(0, repeatDelay - (_.now() - t0)) : repeatDelay);
            } else {
                deferred.resolve(null);
            }
        }

        // immediately invoke the callback function, if no delay time has been specified
        if (initDelay === null) {
            invokeCallback();
        } else {
            createTimer(initDelay);
        }

        // the abortable promise for the deferred object
        return this.createAbortablePromise(deferred, function () {
            if (timer) { timer.abort(); timer = null; }
            if (abortablePromise) { abortablePromise.abort(); }
        });
    };

    /**
     * Invokes the passed callback function repeatedly until it returns a
     * specific value. To prevent performance problems (frozen user interface
     * of the browser), the callback function may be invoked from several time
     * slices of a browser timeout loop. In difference to the method
     * BaseObject.repeatDelayed() that uses an own browser timeout for each
     * invocation of the callback function, this method tries to pack multiple
     * invocations into a single time slice until the total execution time of
     * the time slice is exceeded, and continues with a new time slice in
     * another browser timeout afterwards. If this instance will be destroyed
     * while the loop is running, it will be aborted immediately.
     *
     * @param {Function} callback
     *  The callback function that will be invoked repeatedly. Receives the
     *  zero-based index of the execution cycle as first parameter. Will be
     *  called in the context of this instance. The return value of the
     *  function will be used to decide whether to continue the repeated
     *  execution (see parameter 'callback' of the method
     *  BaseObject.repeatDelayed() for details).
     *
     * @param {String} logMessage
     *  A description for the timer loop that will be inserted into the debug
     *  log used in automated test environments.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.delay]
     *      The initial delay time (in milliseconds) before invoking the
     *      callback function the first time. If omitted, the callback function
     *      will be executed immediately (synchronously) once with the
     *      invocation of this method.
     *  @param {Number} [options.slice=200]
     *      The time (in milliseconds) for a single synchronous time slice. The
     *      callback function will be invoked repeatedly until the time has
     *      elapsed; or until the callback function returns Utils.BREAK, a
     *      pending promise, or a rejected promise (see parameter 'callback'
     *      above for more details).
     *  @param {Number} [options.interval=10]
     *      The delay time (in milliseconds) between two time slices. If the
     *      callback function returns a pending promise, the implementation
     *      will wait for that promise to resolve, and will add a shorter delay
     *      (shortened by the time the promise has needed to resolve) before
     *      continuing the loop.
     *  @param {Number} [options.cycles]
     *      If specified, the maximum number of cycles to be executed.
     *  @param {Boolean} [options.background=false]
     *      Whether the timer loop represents a long-running background task
     *      with low priority. These background tasks will be ignored when
     *      waiting for pending tasks in automated test environments.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after the callback function returns
     *  Utils.BREAK, or if the maximum number of iterations (see option
     *  'cycles') has been reached; or that will be rejected, if the callback
     *  function returns a rejected promise. The promise will be notified each
     *  time a new time slice starts (with the zero-based index of the next
     *  iteration cycle). The promise contains an additional method abort()
     *  that can be called to abort the loop immediately. In that case, the
     *  promise will be rejected. If this instance will be destroyed before or
     *  during the callback loop, the loop timer will be aborted automatically,
     *  and the promise will be rejected.
     */
    BaseObject.prototype.repeatSliced = function (callback, logMessage, options) {

        // slice time for synchronous processing
        var slice = Utils.getIntegerOption(options, 'slice', 200, 10);
        // the number of cycles to be executed
        var cycles = Utils.getIntegerOption(options, 'cycles');
        // index of the next cycle
        var index = 0;
        // the resulting deferred object
        var deferred = Scheduler.createDeferred(this.__windowId, logMessage + ' [repeatSliced]', options);

        // bind callback to this instance
        callback = callback.bind(this);

        // create the interval timer with the passed delay options
        var timer = this.repeatDelayed(function () {

            // start time of this iteration
            var start = _.now();
            // current return value of the callback function
            var result = null;

            // notify the own deferred object about the progress (once per time slice)
            deferred.notify(index);

            // time slice implementation: invoke callback repeatedly (loop body returns on any exit conditions)
            while (true) {

                // exit loop, if the specified number of cycles has been reached
                if (_.isNumber(cycles) && (index >= cycles)) {
                    deferred.resolve();
                    return Utils.BREAK;
                }

                // invoke the callback function
                result = callback(index);
                index += 1;

                // callback has returned Utils.BREAK: resolve the own deferred
                // object immediately, and break the background loop
                if (result === Utils.BREAK) {
                    deferred.resolve(Utils.BREAK);
                    return Utils.BREAK;
                }

                // callback has returned a pending or rejected promise: return it to defer
                // the loop (ignore resolved promises and continue with next invocation)
                if (Utils.isPromise(result) && (result.state() !== 'resolved')) {
                    return result;
                }

                // check elapsed time at the end of the loop body (this ensures
                // that the callback will be executed at least once)
                if (_.now() - start >= slice) {
                    return;
                }
            }
        }, logMessage, {
            delay: Utils.getIntegerOption(options, 'delay', null),
            repeatDelay: Utils.getIntegerOption(options, 'interval', 10, 10),
            fastAsync: true,
            background: Utils.getBooleanOption(options, 'background', false)
        });

        // resolve/reject the own deferred object when the timer has finished
        timer.done(deferred.resolve.bind(deferred));
        timer.fail(deferred.reject.bind(deferred));

        // the abortable promise for the resulting deferred object
        return this.createAbortablePromise(deferred, function () {
            if (timer) { timer.abort(); timer = null; }
        });
    };

    /**
     * Invokes the passed callback function for all steps of the passed
     * iterator object. To prevent performance problems (frozen user user
     * interface of the browser), the callback function will be invoked from
     * several time slices of a browser timeout loop. If this instance will be
     * destroyed while the loop is running, it will be aborted immediately.
     *
     * @param {Object} iterator
     *  An object implementing the standard iterator protocol (must provide a
     *  method next() that returns an object with the properties 'done' and
     *  'value').
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each valid result of the
     *  passed iterator. Receives the following parameters:
     *  (1) {Any} value
     *      The 'value' property from the iterator result.
     *  (2) {Object} result
     *      The complete iterator result, that may contain additional
     *      non-standard value properties.
     *  Will be called in the context of this instance. The return value of the
     *  callback function will be used to decide whether to continue to process
     *  the next iterator result (see parameter 'callback' of the method
     *  BaseObject.repeatDelayed() for details).
     *
     * @param {String} logMessage
     *  A description for the timer loop that will be inserted into the debug
     *  log used in automated test environments.
     *
     * @param {Object} [options]
     *  Optional parameters. See method BaseObject.repeatSliced() for details
     *  about all supported options.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after all iterator steps have been
     *  processed successfully, or the callback function returns Utils.BREAK;
     *  or that will be rejected, if the callback function returns a rejected
     *  promise. The promise will NOT be notified about the progress (iterators
     *  do not provide their size). The promise contains an additional method
     *  abort() that can be called to cancel the loop immediately while
     *  processing the iterator. In that case, the promise will be rejected. If
     *  this instance will be destroyed before or during the callback loop, the
     *  loop timer will be aborted automatically, and the promise will be
     *  rejected.
     */
    BaseObject.prototype.iterateSliced = function (iterator, callback, logMessage, options) {
        return this.repeatSliced(function () {
            var result = iterator.next();
            return result.done ? Utils.BREAK : callback.call(this, result.value, result);
        }, logMessage, options);
    };

    /**
     * Invokes the passed callback function for all steps of the passed
     * iterator object to reduce the iterator values to a single result value.
     * To prevent performance problems (frozen user user interface of the
     * browser), the callback function will be invoked from several time slices
     * of a browser timeout loop. If this instance will be destroyed while the
     * loop is running, it will be aborted immediately.
     *
     * @param {Any} initial
     *  The initial result value passed to the first invocation of the callback
     *  function (if the iterator is empty, this value becomes the final result
     *  value immediately).
     *
     * @param {Object} iterator
     *  An object implementing the standard iterator protocol (must provide a
     *  method next() that returns an object with the properties 'done' and
     *  'value').
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each result of the
     *  passed iterator. Receives the following parameters:
     *  (1) {Any} prevValue
     *      The result value of the previous invocation of this callback
     *      function (or the initial value on first step of the iterator).
     *  (2) {Any} value
     *      The 'value' property from the iterator result.
     *  (3) {Object} result
     *      The complete iterator result, that may contain additional
     *      non-standard value properties.
     *  Will be called in the context of this instance. The return value of the
     *  callback function will become the new result value that will be passed
     *  as 'prevValue' with the next iterator step. If the return value is a
     *  promise, it will be waited for, and its resolved value will become the
     *  next result value.
     *
     * @param {String} logMessage
     *  A description for the timer loop that will be inserted into the debug
     *  log used in automated test environments.
     *
     * @param {Object} [options]
     *  Optional parameters. See method BaseObject.repeatSliced() for details
     *  about all supported options.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved with the final reduced result value
     *  after all iterator steps have been processed successfully; or that will
     *  be rejected, if the callback function returns a rejected promise. The
     *  promise will NOT be notified about the progress (iterators do not
     *  provide their size). The promise contains an additional method abort()
     *  that can be called to cancel the loop immediately while processing the
     *  iterator. In that case, the promise will be rejected. If this instance
     *  will be destroyed before or during the callback loop, the loop timer
     *  will be aborted automatically, and the promise will be rejected.
     */
    BaseObject.prototype.reduceSliced = function (initial, iterator, callback, logMessage, options) {

        // reduce the passed iterator to the final result
        var promise = this.iterateSliced(iterator, function (value, result) {
            var nextValue = callback.call(this, initial, value, result);
            if (Utils.isPromise(nextValue)) {
                return nextValue.done(function (next) { initial = next; });
            }
            initial = nextValue;
        }, logMessage, options);

        // return the final result through the promise
        return promise.then(function () { return initial; });
    };

    /**
     * Invokes the passed callback function for all elements contained in the
     * passed data array. To prevent performance problems (frozen user user
     * interface of the browser), the callback function will be invoked from
     * several time slices of a browser timeout loop. If this instance will be
     * destroyed while the loop is running, it will be aborted immediately.
     *
     * @param {Array|jQuery|String} array
     *  A JavaScript array, or another array-like object that provides a
     *  'length' property, and element access via bracket notation.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each array element.
     *  Receives the following parameters:
     *  (1) {Any} element
     *      The current array element.
     *  (2) {Number} index
     *      The array index of the current element.
     *  (3) {Array|jQuery} array
     *      The entire array as passed in the 'array' parameter.
     *  Will be called in the context of this instance. The return value of the
     *  callback function will be used to decide whether to continue to process
     *  the next array elements (see parameter 'callback' of the method
     *  BaseObject.repeatDelayed() for details).
     *
     * @param {String} logMessage
     *  A description for the timer loop that will be inserted into the debug
     *  log used in automated test environments.
     *
     * @param {Object} [options]
     *  Optional parameters. See method BaseObject.repeatSliced() for details
     *  about all supported options. Additionally, the following options are
     *  supported:
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the array elements will be visited in reversed
     *      order. In reverse mode, the callback function may remove the array
     *      element currently visited, or elements that are following the
     *      visited element, from the array in-place.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after all array elements have been
     *  processed successfully, or the callback function returns Utils.BREAK;
     *  or that will be rejected, if the callback function returns a rejected
     *  promise. The promise will be notified about the progress (as a
     *  floating-point value between 0.0 and 1.0). The promise contains an
     *  additional method abort() that can be called to cancel the loop
     *  immediately while processing the array. In that case, the promise will
     *  be rejected. If this instance will be destroyed before or during the
     *  callback loop, the loop timer will be aborted automatically, and the
     *  promise will be rejected.
     */
    BaseObject.prototype.iterateArraySliced = function (array, callback, logMessage, options) {

        // whether to iterate in reversed order
        var reverse = Utils.getBooleanOption(options, 'reverse', false);
        // current array index
        var arrayIndex = reverse ? (array.length - 1) : 0;
        // the resulting deferred object
        var deferred = Scheduler.createDeferred(this.__windowId, logMessage + ' [iterateArraySliced]', options);

        // do nothing if passed array is empty
        if (array.length === 0) {
            deferred.notify(1).resolve();
            return this.createAbortablePromise(deferred);
        }

        // bind callback to this instance
        callback = callback.bind(this);

        // start the sliced loop
        var timer = this.repeatSliced(function () {

            // end of array reached
            if ((arrayIndex < 0) || (arrayIndex >= array.length)) {
                deferred.notify(1).resolve();
                return Utils.BREAK;
            }

            // invoke the callback function for a single array element
            var result = callback(array[arrayIndex], arrayIndex, array);

            // next array index
            arrayIndex = reverse ? (arrayIndex - 1) : (arrayIndex + 1);

            // immediately resolve after last element unless the result is a pending or rejected promise
            if (((arrayIndex < 0) || (arrayIndex >= array.length)) && !(Utils.isPromise(result) && (result.state() !== 'resolved'))) {
                deferred.notify(1).resolve();
                return Utils.BREAK;
            }

            return result;
        }, logMessage, options);

        // resolve/reject the own deferred object according to the timer result
        timer.done(deferred.resolve.bind(deferred));
        timer.fail(deferred.reject.bind(deferred));

        // notify the own deferred object about the progress (once per time slice)
        timer.progress(function (index) { deferred.notify(index / array.length); });

        // the abortable promise of the resulting deferred object
        return this.createAbortablePromise(deferred, function () {
            if (timer) { timer.abort(); timer = null; }
        });
    };

    /**
     * Invokes the passed callback function for all properties contained in the
     * passed object. To prevent performance problems (frozen user user
     * interface of the browser), the callback function will be invoked from
     * several time slices of a browser timeout loop. If this instance will be
     * destroyed while the loop is running, it will be aborted immediately.
     *
     * @param {Object} object
     *  A JavaScript object whose properties will be iterated.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each object property (in
     *  no specific order). Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current property.
     *  (2) {String} key
     *      The name of the current property.
     *  (3) {Object} object
     *      The entire object as passed in the 'object' parameter.
     *  Will be called in the context of this instance. The return value of the
     *  callback function will be used to decide whether to continue to process
     *  the next object properties (see parameter 'callback' of the method
     *  BaseObject.repeatDelayed() for details).
     *
     * @param {String} logMessage
     *  A description for the timer loop that will be inserted into the debug
     *  log used in automated test environments.
     *
     * @param {Object} [options]
     *  Optional parameters. See method BaseObject.repeatSliced() for details
     *  about all supported options.
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved after all object properties have been
     *  processed successfully, or the callback function returns Utils.BREAK;
     *  or that will be rejected, if the callback function returns a rejected
     *  promise. The promise will be notified about the progress (as a
     *  floating-point value between 0.0 and 1.0). The promise contains an
     *  additional method abort() that can be called to cancel the loop
     *  immediately while processing the properties. In that case, the promise
     *  will be rejected. If this instance will be destroyed before or during
     *  the callback loop, the loop timer will be aborted automatically, and
     *  the promise will be rejected.
     */
    BaseObject.prototype.iterateObjectSliced = function (object, callback, logMessage, options) {

        return this.iterateArraySliced(_.keys(object), function (key) {
            if (key in object) { return callback.call(this, object[key], key, object); }
        }, logMessage, options);
    };

    // method generators ------------------------------------------------------

    /**
     * Creates a synchronized method wrapping a callback function that executes
     * asynchronous code. The synchronized method buffers multiple fast
     * invocations and executes the callback function successively, always
     * waiting for the asynchronous code. In difference to debounced methods,
     * invocations of a synchronized method will never be skipped, and each
     * call of the asynchronous callback function receives its original
     * arguments passed to the synchronized method.
     *
     * @param {String} methodName
     *  A descriptive name for the synchronized method that will be used to
     *  generate debug log messages in automated test environments.
     *
     * @param {Function} callback
     *  A function that will be called every time the synchronized method has
     *  been called. Will be called in the context the synchronized method has
     *  been called with, and receives all parameters that have been passed to
     *  the synchronized method. If this function returns a pending promise,
     *  subsequent invocations of the synchronized method will be postponed
     *  until the promise will be resolved or rejected. All other return values
     *  will be interpreted as synchronous invocations of the callback
     *  function.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.background=false]
     *      Whether the synchronized method represents a long-running
     *      background task with low priority. These background tasks will be
     *      ignored when waiting for pending tasks in automated test
     *      environments.
     *
     * @returns {Function}
     *  The synchronized method that can be called multiple times, and that
     *  executes the asynchronous callback function sequentially. Returns a
     *  promise that will be resolved or rejected after the callback function
     *  has been invoked. If the callback function returns a promise, the
     *  synchronized method will wait for it, and will forward its state and
     *  response to its promise. Otherwise, the promise will be resolved with
     *  the return value of the callback function. If this instance will be
     *  destroyed, processing the chain of pending method invocations will be
     *  aborted.
     */
    BaseObject.prototype.createSynchronizedMethod = function (methodName, callback, options) {

        // self reference
        var self = this;
        // arguments and returned promise of pending calls of the method
        var pendingInvocations = [];
        // the background loop processing all pending invocations
        var timer = null;
        // whether processing the invocations has started
        var running = false;
        // the message string inserted into the automated test log
        var logMessage = methodName + ' [synchronized method]';
        // whether this timer loop is a background task
        var background = Utils.getBooleanOption(options, 'background', false);

        // invokes the first pending callback function (used as callback for repeatSliced())
        function invokeCallback() {

            // stop background loop if array is empty (all callbacks invoked)
            if (pendingInvocations.length === 0) { return Utils.BREAK; }

            // the first pending invocation (keep in array until it resolves)
            var invocationData = pendingInvocations[0];
            // create the promise representing the result of the callback function
            var promise = $.when(callback.apply(invocationData.ctxt, invocationData.args));
            // the deferred object bound to the result of the callback
            var deferred = invocationData.def;

            // remove invocation data from array when the promise has been resolved/rejected
            promise.always(function () { pendingInvocations.shift(); });

            // forward result of the promise to the deferred object
            promise.done(deferred.resolve.bind(deferred));
            promise.fail(deferred.reject.bind(deferred));

            // return the promise (unless last callback was synchronous and is already resolved)
            return (pendingInvocations.length === 0) ? Utils.BREAK : promise;
        }

        // rejects the deferred objects of all cached invocations
        function rejectInvocations(cause) {
            pendingInvocations.forEach(function (invocationData) {
                invocationData.def.reject(cause);
            });
        }

        // create and return the synchronized method
        return function () {

            // the deferred object representing this function call
            var deferred = Scheduler.createDeferred(self.__windowId, logMessage, options);
            // all data about the current invocation (arguments and returned deferred object)
            var invocationData = { ctxt: this, args: arguments, def: deferred };

            // push new invocation to the end of the array
            pendingInvocations.push(invocationData);

            // if missing, start a new timer that processes the array
            if (!running) {
                // immediately set the running flag before starting the first invocation (prevent synchronous recursive calls)
                running = true;
                // set up the background loop that processes all pending invocations
                timer = self.repeatSliced(invokeCallback, logMessage, { background: background });
                // reject all pending promises on abort/destruction
                timer.fail(rejectInvocations);
                // forget the timer after the last callback invocation
                timer.always(function () { running = false; timer = null; });
            }

            // return a promise that will be resolved/rejected after invocation
            return deferred.promise();
        };
    };

    /**
     * Creates a debounced method that can be called multiple times during the
     * current script execution. The deferred callback will be executed later
     * once in a browser timeout.
     *
     * @param {String} methodName
     *  A descriptive name for the debounced method that will be used to
     *  generate debug log messages in automated test environments.
     *
     * @param {Function|Null} directCallback
     *  A function that will be called every time the debounced method has been
     *  called. Will be called in the context the debounced method has been
     *  called with, and receives all parameters that have been passed to the
     *  created debounced method. Can be set to null, if nothing has to be done
     *  when calling the debounced method directly.
     *
     * @param {Function} deferredCallback
     *  A function that will be called in a browser timeout after the debounced
     *  method has been called at least once during the execution of the
     *  current script. Will be called in the context the debounced method has
     *  been called with. For convenience, this callback function receives the
     *  return value of the LAST invocation of the direct callback function as
     *  first parameter.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Number} [options.delay=0]
     *      The delay time (in milliseconds) for the deferred callback
     *      function. The delay time will restart after each call of the
     *      debounced method, causing the execution of the deferred callback to
     *      be postponed until the debounced method has not been called again
     *      during the delay (this is the behavior of the method _.debounce()).
     *  @param {Number} [options.maxDelay]
     *      If specified, a delay time used as a hard limit to execute the
     *      deferred callback after the first call of the debounced method,
     *      even if it has been called repeatedly afterwards and the normal
     *      delay time is still running.
     *
     * @returns {Function}
     *  The debounced method that can be called multiple times, and that
     *  executes the deferred callback function once after execution of the
     *  current script ends. Passes all arguments to the direct callback
     *  function, and returns its result (returns the calling context, if no
     *  direct callback has been specified). If this instance will be
     *  destroyed, pending debounced methods will not be invoked anymore.
     */
    BaseObject.prototype.createDebouncedMethod = function (methodName, directCallback, deferredCallback, options) {

        // self reference
        var self = this;
        // delay time per invocation
        var delay = Utils.getIntegerOption(options, 'delay', 0, 0);
        // maximum accumulated delay time
        var maxDelay = Utils.getIntegerOption(options, 'maxDelay', 0, 0);
        // the current timer used to execute the callback
        var minTimer = null;
        // timer used for the maxDelay option
        var maxTimer = null;
        // whether to request an animation frame that will invoke the deferred callback
        var animFrame = Utils.getBooleanOption(options, 'animFrame', false);
        // the identifier of a requested animation frame
        var frameId = null;
        // the last cached result of the direct callback
        var lastResult = null;
        // the message string inserted into the automated test log
        var logMessage = methodName + ' [debounced method]';

        // aborts and clears all timers
        function clearTimers() {
            if (minTimer) { minTimer.abort(); }
            if (maxTimer) { maxTimer.abort(); }
            minTimer = maxTimer = frameId = null;
        }

        // creates the timers that execute the deferred callback
        function createTimers(context) {

            // timer callback invoking the deferred callback, bound to the current context (but ignoring the
            // return value, especially promises returned by the callback will not be forwarded to the timer)
            function invokeDeferredCallback() {
                clearTimers();
                deferredCallback.call(context, lastResult); // ignore the return value of the callback
            }

            // request an animation frame instead of using timers, if specified
            if (animFrame) {
                if (!frameId) {
                    frameId = window.requestAnimationFrame(function () {
                        if (!self.destroyed) {
                            invokeDeferredCallback();
                        }
                    });
                }
                return;
            }

            // abort running timer on first call
            if (minTimer) {
                minTimer.abort();
                minTimer = null;
            }

            // create a new timeout executing the callback function
            if (!minTimer) {
                minTimer = self.executeDelayed(invokeDeferredCallback, logMessage, delay);
            }

            // on first call, create a timer for the maximum delay
            if (!maxTimer && (maxDelay > 0)) {
                maxTimer = self.executeDelayed(invokeDeferredCallback, logMessage, maxDelay);
            }
        }

        // create and return the debounced method
        return function () {

            // first, call the direct callback with the passed arguments (especially, before requesting an animation frame)
            lastResult = directCallback ? directCallback.apply(this, arguments) : this;

            // create a new timeout executing the callback function
            createTimers(this);

            // return the result of the direct callback
            return lastResult;
        };
    };

    // exports ================================================================

    return BaseObject;

});
