/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/container/typedarray', [
    'io.ox/office/tk/utils/iteratorutils'
], function (IteratorUtils) {

    'use strict';

    // static class TypedArray ================================================

    var TypedArray = {};

    // static methods ---------------------------------------------------------

    /**
     * Creates a subclass of the JavaScript class Array, intended to store
     * array elements of a specific type.
     *
     * Instances of the new array class behave like regular JavaScript arrays
     * (instanceof operator with class Array, length property, element access
     * via brackets, most instance methods), with a few differences that will
     * be described in the following text. The typed array class will be called
     * 'ArrayClass' from now on.
     *
     * The class constructor accepts an arbitrary number of parameters that
     * will be inserted into the new instance. Each constructor parameter can
     * be another instance of ArrayClass, a plain JavaScript array, or a single
     * instance of ElementType. Array parameters will be concatenated (no
     * nested arrays as elements).
     *
     * The new array class provides a few new static methods by itself for
     * convenience:
     *
     * ArrayClass.get(value)
     *  If the passed value is an instance of ArrayClass, it will be returned
     *  directly. Otherwise, a new array will be created by passing the value
     *  to the ArrayClass constructor.
     *
     * ArrayClass.forEach(value, callback, context)
     *  Visits all elements in the passed array. Works similar to the instance
     *  method forEach() of a JavaScript array. The parameter 'value' can be an
     *  instance of ArrayClass, a plain JavaScript array, or a single instance
     *  of ElementType. In the latter case, the callback function will be
     *  invoked once for that value (as if it was a one-element array).
     *
     * ArrayClass.forEachReverse(value, callback, context)
     *  Visits all elements in the passed array in reversed order. See method
     *  ArrayClass.forEach() for details.
     *
     * ArrayClass.some(value, predicate, context)
     *  Works similar to the instance method some() of a JavaScript array. The
     *  parameter 'value' can be an instance of ArrayClass, a plain JavaScript
     *  array, or a single instance of ElementType. In the latter case, the
     *  predicate function will be invoked once for that value (as if it was a
     *  one-element array).
     *
     * ArrayClass.someReverse(value, predicate, context)
     *  Works similar to ArrayClass.some() but visits the elements in an array
     *  in reversed order.
     *
     * ArrayClass.every(value, predicate, context)
     *  Works similar to the instance method every() of a JavaScript array. The
     *  parameter 'value' can be an instance of ArrayClass, a plain JavaScript
     *  array, or a single instance of ElementType. In the latter case, the
     *  predicate function will be invoked once for that value (as if it was a
     *  one-element array).
     *
     * ArrayClass.everyReverse(value, predicate, context)
     *  Works similar to ArrayClass.every() but visits the elements in an array
     *  in reversed order.
     *
     * ArrayClass.filter(value, predicate, context)
     *  Creates a new instance of ArrayClass with all elements that have passed
     *  a truth test. The parameter 'value' can be an instance of ArrayClass, a
     *  plain JavaScript array, or a single instance of ElementType. In the
     *  latter case, the predicate function will be invoked once for that value
     *  (as if it was a one-element array).
     *
     * ArrayClass.reject(value, predicate, context)
     *  Creates a new instance of ArrayClass with all elements that have failed
     *  a truth test. This method is the exact opposite of ArrayClass.filter().
     *  The parameter 'value' can be an instance of ArrayClass, a plain
     *  JavaScript array, or a single instance of ElementType. In the latter
     *  case, the predicate function will be invoked once for that value (as if
     *  it was a one-element array).
     *
     * ArrayClass.map(value, callback, context)
     *  Creates a new instance of ArrayClass with the return values of the
     *  callback function. The callback function MUST return an object of
     *  ElementType, an array of elements (either a plain JavaScript array, or
     *  an instance of ArrayClass), or a falsy value that will be filtered from
     *  the resulting array. The parameter 'value' can be an instance of
     *  ArrayClass, a plain JavaScript array, or a single instance of
     *  ElementType. In the latter case, the callback function will be invoked
     *  once for that value (as if it was a one-element array).
     *
     * ArrayClass.invoke(value, method)
     *  Creates a new instance of ArrayClass with the return values of the
     *  specified element method (as string). The method MUST return an object
     *  of ElementType, an array of elements (either a plain JavaScript array,
     *  or an instance of ArrayClass), or a falsy value that will be filtered
     *  from the resulting array. The parameter 'value' can be an instance of
     *  ArrayClass, a plain JavaScript array, or a single instance of
     *  ElementType. In the latter case, the method will be invoked directly at
     *  that value (as if it was a one-element array).
     *
     * ArrayClass.group(value, callback, context)
     *  Distributes the elements of the source array into a map (a plain
     *  JavaScript object) of ArrayClass instances that are keyed by the return
     *  value of the callback function. The callback function will be invoked
     *  once for all elements of the source array. The parameter 'value' can be
     *  an instance of ArrayClass, a plain JavaScript array, or a single
     *  instance of ElementType. In the latter case, the callback function will
     *  be invoked directly at that value (as if it was a one-element array).
     *
     * ArrayClass.iterator(value, options)
     *  Creates an array iterator that can be used to visit the elements of the
     *  passed value. The iterator will implement the standard EcmaScript
     *  iterator protocol, i.e. it provides the method next() that returns a
     *  result object with the properties 'done' and 'value'. Additionally, the
     *  result object contains a property 'index' with the array index of the
     *  visited element (see IteratorUtils.createArrayIterator() for details).
     *  The parameter 'value' passed to this method can be an instance of
     *  ArrayClass, a plain JavaScript array, or a single instance of
     *  ElementType. In the latter case, the iterator will visit this value as
     *  if it was an array with one element.
     *
     * The following instance methods are new, or behave differently than the
     * instance methods of the class Array:
     *
     * ArrayClass.prototype.empty()
     *  New method. Returns whether the array is empty.
     *
     * ArrayClass.prototype.first()
     *  New method. Returns the first element of the array (null, if the array
     *  is empty).
     *
     * ArrayClass.prototype.last()
     *  New method. Returns the last element of the array (null, if the array
     *  is empty).
     *
     * ArrayClass.prototype.clear()
     *  New method. Removes all elements from the array.
     *
     * ArrayClass.prototype.clone([deep])
     *  New method. Returns a shallow clone (parameter 'deep' omitted or
     *  false), or deep clone (parameter 'deep' set to true) of the array. To
     *  create deep clones of this array, the elements MUST provide a clone()
     *  instance method. The parameter 'deep' will be passed to the clone()
     *  method of the elements in order to achieve recursive deep cloning.
     *
     * ArrayClass.prototype.equals(array, [deep])
     *  New method. Returns whether this array is equal to the passed other
     *  array. The passed array MUSt be of the same type as this array. Returns
     *  whether the arrays are equal shallowly (parameter 'deep' omitted or
     *  false, compares elements with JS strict equality operator), or deeply
     *  (parameter 'deep' set to true, elements will be compared deeply). To
     *  compare array elements deeply, the elements MUST provide an equals()
     *  instance method. The parameter 'deep' will be passed after the array
     *  element as second parameter to the equals() method of the array element
     *  in order to achieve recursive deep comparison.
     *
     * ArrayClass.prototype.append(...)
     *  New method. Appends all parameters to the array. Each parameter can be
     *  another instance of ArrayClass, a plain JavaScript array, or a single
     *  instance of ElementType. All elements from array parameters will be
     *  appended directly to this array (no nested arrays). Invalid values (for
     *  example null or undefined) will be skipped silently.
     *
     * ArrayClass.prototype.concat(...)
     *  Works similar to the native Array method. Creates a shallow clone of
     *  the array, and appends all parameters to the new array. Each parameter
     *  can be another instance of ArrayClass, a plain JavaScript array, or a
     *  single instance of ElementType.
     *
     * ArrayClass.prototype.slice([begin[, end]])
     *  Works similar to the native Array method, but returns a new instance of
     *  ArrayClass instead of a plain JavaScript array.
     *
     * ArrayClass.prototype.forEachReverse(callback, context)
     *  Works similar to the native array method Array.prototype.forEach(), but
     *  visits all elements in the array in reversed order.
     *
     * ArrayClass.prototype.someReverse(predicate, context)
     *  Works similar to the native array method Array.prototype.some(), but
     *  visits all elements in the array in reversed order.
     *
     * ArrayClass.prototype.everyReverse(predicate, context)
     *  Works similar to the native array method Array.prototype.every(), but
     *  visits all elements in the array in reversed order.
     *
     * ArrayClass.prototype.filter(predicate, context)
     *  Works similar to the native Array method, but returns a new instance of
     *  ArrayClass instead of a plain JavaScript array.
     *
     * ArrayClass.prototype.reject(predicate, context)
     *  New method. Returns an array with all elements that have failed a truth
     *  test (the exact opposite of ArrayClass.prototype.filter()). Returns a
     *  new instance of ArrayClass instead of a plain JavaScript array.
     *
     * ArrayClass.prototype.find(predicate, context)
     *  New method. Returns the first array element that passes the truth test;
     *  or null, if no element passes the test.
     *
     * ArrayClass.prototype.sort([comparator])
     *  Works similar to the native Array method. If no comparator callback
     *  function has been passed to the method, and the static method
     *  ElementType.compare() exists, it will be used as default comparator.
     *
     * ArrayClass.prototype.sortBy(callback, context)
     *  Works similar to the Underscore method _.sortBy(). The elements will be
     *  sorted according to the return values of the callback function, or to
     *  the property value of the elements specified as string parameter. But
     *  in difference to _.sortBy(), this method will sort the array in-place.
     *
     * ArrayClass.prototype.iterator(options)
     *  Creates an iterator that can be used to visit the elements of this
     *  array. The iterator will implement the standard EcmaScript iterator
     *  protocol, i.e. it provides a method next() that returns a result object
     *  with 'done' and 'value' properties. Additionally, the result object
     *  contains a property 'index' with the array index of the visited
     *  element. See static method ArrayClass.iterator() for details.
     *
     * ArrayClass.prototype.toString([separator])
     *  Works similar to the native Array method. If available, inserts the
     *  specified separator between the array elements in the resulting string.
     *
     * ArrayClass.prototype.toJSON()
     *  Converts the array to a plain JavaScript array. Uses the instance
     *  method ElementType.toJSON() of the array elements if available.
     *
     * @param {Function} ElementType
     *  The type (the constructor function) of the array elements. Must provide
     *  the instance method clone(). May provide a custom implementation of the
     *  instance methods toString() and toJSON(). May provide the static method
     *  ElementType.compare() which will be used as natural element order for
     *  sorting the array.
     *
     * @returns {Function}
     *  The constructor function of the typed array class.
     */
    TypedArray.create = function (ElementType) {

        // the new typed array class to be returned
        function ArrayClass() {
            Array.call(this);
            this.append.apply(this, arguments);
        }

        // derive from class Array
        function ArrayClassPrototype() {}
        ArrayClassPrototype.prototype = Array.prototype;
        ArrayClass.prototype = new ArrayClassPrototype();
        ArrayClass.prototype.constructor = ArrayClass;

        // store type names at class constructor
        ArrayClass.ArrayType = ArrayClass;
        ArrayClass.ElementType = ElementType;
        ArrayClass.__super__ = Array.prototype;

        // returns an instance of ArrayClass as-is, otherwise creates an instance
        ArrayClass.get = function (arg) {
            return (arg instanceof ArrayClass) ? arg : new ArrayClass(arg);
        };

        // iterates all elements in the passed array, or invokes callback once for element type
        ArrayClass.forEach = function (arg, callback, context) {
            if (arg instanceof Array) { arg.forEach(callback, context); } else { callback.call(context, arg, 0, arg); }
        };

        // iterates all elements in the passed array in reversed order, or invokes callback once for element type
        ArrayClass.forEachReverse = function (arg, callback, context) {
            if (arg instanceof Array) {
                var arrayIt = IteratorUtils.createArrayIterator(arg, { reverse: true });
                IteratorUtils.forEach(arrayIt, function (element, result) { callback.call(context, element, result.index, arg); });
            } else {
                callback.call(context, arg, 0, arg);
            }
        };

        // iterates elements in the passed array until truth test passes, or invokes callback once for element type
        ArrayClass.some = function (arg, predicate, context) {
            return (arg instanceof Array) ? arg.some(predicate, context) : !!predicate.call(context, arg, 0, arg);
        };

        // iterates elements in the passed array in reversed order until truth test passes, or invokes callback once for element type
        ArrayClass.someReverse = function (arg, predicate, context) {
            if (arg instanceof Array) {
                var arrayIt = IteratorUtils.createArrayIterator(arg, { reverse: true });
                return IteratorUtils.some(arrayIt, function (element, result) { return predicate.call(context, element, result.index, arg); });
            }
            return !!predicate.call(context, arg, 0, arg);
        };

        // iterates elements in the passed array until truth test fails, or invokes callback once for element type
        ArrayClass.every = function (arg, predicate, context) {
            return (arg instanceof Array) ? arg.every(predicate, context) : !!predicate.call(context, arg, 0, arg);
        };

        // iterates elements in the passed array in reversed order until truth test fails, or invokes callback once for element type
        ArrayClass.everyReverse = function (arg, predicate, context) {
            if (arg instanceof Array) {
                var arrayIt = IteratorUtils.createArrayIterator(arg, { reverse: true });
                return IteratorUtils.every(arrayIt, function (element, result) { return predicate.call(context, element, result.index, arg); });
            }
            return !!predicate.call(context, arg, 0, arg);
        };

        // creates a new filtered array with matching elements
        ArrayClass.filter = function (arg, predicate, context) {
            var array = new ArrayClass();
            ArrayClass.forEach(arg, function (element) {
                if (predicate.apply(context, arguments)) { array.append(element); }
            });
            return array;
        };

        // creates a new filtered array with non-matching elements
        ArrayClass.reject = function (arg, predicate, context) {
            var array = new ArrayClass();
            ArrayClass.forEach(arg, function (element) {
                if (!predicate.apply(context, arguments)) { array.append(element); }
            });
            return array;
        };

        // creates a new array from the passed array and callback
        ArrayClass.map = function (arg, callback, context) {
            var array = new ArrayClass();
            ArrayClass.forEach(arg, function () {
                array.append(callback.apply(context, arguments));
            });
            return array;
        };

        // creates a new array from the return values of an element method
        ArrayClass.invoke = function (arg, method) {
            var methodArgs = Array.prototype.slice.call(arguments, 2);
            return ArrayClass.map(arg, function (element) {
                return element[method].apply(element, methodArgs);
            });
        };

        // creates a map of arrays, keyed by the return value of the callback function
        ArrayClass.group = function (arg, callback, context) {
            var map = {};
            ArrayClass.forEach(arg, function (element) {
                var key = callback.apply(context, arguments).toString();
                (map[key] || (map[key] = new ArrayClass())).push(element);
            });
            return map;
        };

        // creates an iterator for the passed array or value
        ArrayClass.iterator = function (arg, options) {
            return IteratorUtils.createArrayIterator((arg instanceof Array) ? arg : [arg], options);
        };

        // returns whether the array is empty
        ArrayClass.prototype.empty = function () {
            return this.length === 0;
        };

        // returns the first array element
        ArrayClass.prototype.first = function () {
            return this.empty() ? null : this[0];
        };

        // returns the last array element
        ArrayClass.prototype.last = function () {
            return this.empty() ? null : this[this.length - 1];
        };

        // removes all elements from the array
        ArrayClass.prototype.clear = function () {
            this.splice(0);
            return this;
        };

        // creates a (shallow or deep) clone of this array
        ArrayClass.prototype.clone = function (deep) {
            return deep ? ArrayClass.invoke(this, 'clone', true) : new ArrayClass(this);
        };

        // compares this array with another array (shallow or deep)
        ArrayClass.prototype.equals = function (array, deep) {
            return (this.length === array.length) && this.every(function (element, index) {
                return deep ? element.equals(array[index], true) : (element === array[index]);
            });
        };

        // appends all parameters to this array (flattens array parameters automatically)
        ArrayClass.prototype.append = function () {
            var push = function (element) { this.push(element); }.bind(this);
            for (var i = 0; i < arguments.length; i += 1) {
                // work-around for a problem in PhantomJS (and probably other JS engines) where
                // using Array.push.apply() with an instance of ArrayClass as parameter fails
                var arg = arguments[i];
                if (arg instanceof ArrayClass) {
                    arg.forEach(push);
                } else if (arg instanceof Array) {
                    this.push.apply(this, arg);
                } else if (arg instanceof ElementType) {
                    this.push(arg);
                }
            }
            return this;
        };

        // creates a clone, appends all parameters (flattens array parameters automatically)
        ArrayClass.prototype.concat = function () {
            var array = this.clone();
            array.append.apply(array, arguments);
            return array;
        };

        // creates a sub-array
        ArrayClass.prototype.slice = function () {
            return new ArrayClass(Array.prototype.slice.apply(this, arguments));
        };

        // iterates all elements in reversed order
        ArrayClass.prototype.forEachReverse = function (callback, context) {
            return ArrayClass.forEachReverse(this, callback, context);
        };

        // tests all elements in reversed order
        ArrayClass.prototype.someReverse = function (predicate, context) {
            return ArrayClass.someReverse(this, predicate, context);
        };

        // tests all elements in reversed order
        ArrayClass.prototype.everyReverse = function (predicate, context) {
            return ArrayClass.everyReverse(this, predicate, context);
        };

        // creates a new filtered array with matching elements
        ArrayClass.prototype.filter = function (predicate, context) {
            return ArrayClass.filter(this, predicate, context);
        };

        // creates a new filtered array with non-matching elements
        ArrayClass.prototype.reject = function (predicate, context) {
            return ArrayClass.reject(this, predicate, context);
        };

        // creates a map of arrays, keyed by the return value of the callback function
        ArrayClass.prototype.group = function (callback, context) {
            return ArrayClass.group(this, callback, context);
        };

        // returns the first matching array element
        ArrayClass.prototype.find = function (predicate, context) {
            return _.find(this, predicate, context) || null;
        };

        // default sort() to natural order of array elements
        if (typeof ElementType.compare === 'function') {
            ArrayClass.prototype.sort = function (order) {
                return Array.prototype.sort.call(this, order || ElementType.compare);
            };
        }

        // sort the array by a numeric or string key
        ArrayClass.prototype.sortBy = function (callback, context) {
            var key = (typeof callback === 'string') ? function (elem) { return elem[callback]; } : callback.bind(context);
            return this.sort(function (elem1, elem2) {
                var key1 = key(elem1), key2 = key(elem2);
                return (key1 < key2) ? -1 : (key1 > key2) ? 1 : 0;
            });
        };

        // creates an iterator for this array
        ArrayClass.prototype.iterator = function (options) {
            return IteratorUtils.createArrayIterator(this, options);
        };

        // stringifies all array elements with the passed separator
        ArrayClass.prototype.toString = function (separator) {
            return this.join(separator);
        };

        // converts this array to a plain JSON array with JSON elements
        ArrayClass.prototype.toJSON = (typeof ElementType.prototype.toJSON === 'function') ?
            function () { return _.invoke(this, 'toJSON'); } :
            function () { return this.map(_.identity); };

        return ArrayClass;
    };

    // exports ================================================================

    return TypedArray;

});
