/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/container/container', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils'
], function (Utils, IteratorUtils) {

    'use strict';

    // private global functions ===============================================

    function getCallback(args) {

        var callback = args[0];

        // bind callback function to context
        if (typeof callback === 'function') {
            return callback.bind(args[1]);
        }

        // performance shortcut for few number of arguments
        switch (args.length) {
            case 1: return function (obj) { return obj[callback](); };
            case 2: return function (obj) { return obj[callback](args[1]); };
            case 3: return function (obj) { return obj[callback](args[1], args[2]); };
            case 4: return function (obj) { return obj[callback](args[1], args[2], args[3]); };
        }

        // generic solution for more arguments
        args = Array.prototype.slice.call(args, 1);
        return function (obj) { return obj[callback].apply(obj, args); };
    }

    // class Container ========================================================

    /**
     * An internal base class used as implementation helper for prototype based
     * classes implementing an associative container. This class provides the
     * protected property '_cont' (plain JavaScript object) and various
     * iterator methods.
     *
     * @constructor
     */
    var Container = _.makeExtendable(function () {

        // the internal storage of this container
        this._cont = {};

    }); // class Container

    // static methods ---------------------------------------------------------

    /**
     * Invokes the passed callback function for each element in the list.
     *
     * @param {Object} list
     *  A generic data source. Can be any class instance that provides a public
     *  method 'forEach()' (e.g. instances of Array), or any other object that
     *  will be accepted by the method '_.each()' (e.g. plain JS objects).
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each list element.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Object} list
     *      A reference to the passed list.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    Container.forEach = function (list, callback, context) {
        if (typeof list.forEach === 'function') {
            list.forEach(callback, context);
        } else {
            _.each(list, callback, context);
        }
    };

    /**
     * Invokes the passed callback function for each element in the list, and
     * passes the element keys as first parameter.
     *
     * @param {Object} list
     *  A generic data source. Can be any class instance that provides a public
     *  method 'forEach()' (e.g. instances of Array), or any other object that
     *  will be accepted by the method '_.each()' (e.g. plain JS objects).
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each list element.
     *  Receives the following parameters:
     *  (1) {String} key
     *      The key of the current element.
     *  (2) {Any} value
     *      The value of the current element.
     *  (3) {Object} list
     *      A reference to the passed list.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     */
    Container.forEachKey = function (list, callback, context) {
        Container.forEach(list, function (value, key) {
            callback.call(context, key, value, list);
        });
    };

    // protected methods ------------------------------------------------------

    /**
     * Returns whether this container contains an element with the specified
     * key.
     *
     * @param {String} key
     *  The key to be checked.
     *
     * @returns {Boolean}
     *  Whether this container contains an element with the specified key.
     */
    Container.prototype._has = function (key) {
        return key in this._cont;
    };

    /**
     * Returns the value of the element with the specified key.
     *
     * @param {String} key
     *  The key of an element.
     *
     * @returns {Any}
     *  The value of the element with the specified key; or undefined, if the
     *  container does not contain such an element.
     */
    Container.prototype._get = function (key) {
        return this._cont[key];
    };

    /**
     * Inserts an element into this container. An old existing element will be
     * overwritten.
     *
     * @param {String} key
     *  The key for the element to be inserted.
     *
     * @param {Any} value
     *  The value to be inserted at the specified key.
     *
     * @returns {Any}
     *  The value passed to this method, for convenience.
     */
    Container.prototype._insert = function (key, value) {
        return (this._cont[key] = value);
    };

    /**
     * Removes an element from this container.
     *
     * @param {String} key
     *  The key of the element to be removed.
     *
     * @returns {Boolean}
     *  Whether the element was actually part of this container.
     */
    Container.prototype._remove = function (key) {
        var has = this._has(key);
        delete this._cont[key];
        return has;
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns whether this container is empty.
     *
     * @returns {Boolean}
     *  Whether this container is empty.
     */
    Container.prototype.empty = function () {
        // fastest way to detect an empty object
        for (var key in this._cont) { return false; } // eslint-disable-line no-unused-vars
        return true;
    };

    /**
     * Returns whether this container contains exactly one element.
     *
     * @returns {Boolean}
     *  Whether this container contains exactly one element.
     */
    Container.prototype.single = function () {
        return Utils.hasSingleProperty(this._cont);
    };

    /**
     * Removes all elements from this container.
     *
     * @returns {Container}
     *  A reference to this instance.
     */
    Container.prototype.clear = function () {
        this._cont = {};
        return this;
    };

    /**
     * Returns the value of the element with the specified key.
     *
     * @param {String} key
     *  The key of the element to be returned.
     *
     * @param {Any} [def]
     *  The default value that will be returned if this container does not
     *  contain an element with the specified key. If omitted, undefined will
     *  be returned in this case.
     *
     * @returns {Any}
     *  The value of the element with the specified key; or the specified
     *  default value, if the container does not contain such an element.
     */
    Container.prototype.get = function (key, def) {
        return this._has(key) ? this._get(key) : def;
    };

    /**
     * Invokes the passed callback function, if this container contains an
     * element with the specified key.
     *
     * @param {String} key
     *  The key of an element.
     *
     * @param {Function} callback
     *  The callback function that will be invoked if this container contains
     *  an element with the sepcified key. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the element with the specified key.
     *
     * @param {Object} context
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The return value of the callback function, if the element exists in
     *  this container; otherwise undefined.
     */
    Container.prototype.with = function (key, callback, context) {
        if (this._has(key)) { return callback.call(context, this._get(key)); }
    };

    /**
     * Returns a single unspecified element from this instance.
     *
     * @returns {Any}
     *  An unspecified element from this container; or undefined, if the
     *  container is empty. Repeated invocations of this method will always
     *  return the same element unless the container has been modified.
     */
    Container.prototype.getAny = function () {
        for (var key in this._cont) { return this._cont[key]; }
    };

    /**
     * Returns the keys of all elements in this container as array.
     *
     * @returns {Array<String>}
     *  The keys of all elements in this container, in no specific order.
     */
    Container.prototype.keys = function () {
        return Object.keys(this._cont);
    };

    /**
     * Returns the values of all elements in this container as array.
     *
     * @returns {Array<Any>}
     *  The values of all elements in this container, in no specific order.
     */
    Container.prototype.values = function () {
        return _.values(this._cont);
    };

    /**
     * Returns the contents of this container as simple JavaScript object.
     *
     * @returns {Object<String,Any>}
     *  The contents of this container as simple JavaScript object (the values
     *  of all elements in this container, mapped by their keys).
     */
    Container.prototype.toObject = function () {
        return this._cont;
    };

    /**
     * Returns an iterator that visits all elements in this container. The
     * elements will be visited in no specific order.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the container has been visited completely. No more
     *      elements are available; this result object does not contain any
     *      other properties!
     *  - {Any} value
     *      The value of the element currently visited. This property will be
     *      omitted, if the iterator is done (see property 'done').
     *  - {String} key
     *      The key of the element currently visited. This property will be
     *      omitted, if the iterator is done (see property 'done').
     */
    Container.prototype.iterator = function () {
        return IteratorUtils.createObjectIterator(this._cont);
    };

    /**
     * Returns an iterator that visits the keys of all elements in this
     * container. The keys will be visited in no specific order.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the container has been visited completely. No more
     *      elements are available; this result object does not contain any
     *      other properties!
     *  - {String} value
     *      The key of the element currently visited. This property will be
     *      omitted, if the iterator is done (see property 'done').
     *  - {Any} element
     *      The value of the element currently visited. This property will be
     *      omitted, if the iterator is done (see property 'done').
     */
    Container.prototype.keyIterator = function () {
        var iterator = IteratorUtils.createObjectIterator(this._cont);
        return IteratorUtils.createTransformIterator(iterator, function (value, iterResult) {
            return { value: iterResult.key, element: value };
        });
    };

    /**
     * Invokes the passed callback function for all elements in this container.
     * The elements will be visited in no specific order.
     *
     * @param {Function|String} callback
     *  The callback function that will be invoked for each element. Receives
     *  the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Container}
     *  A reference to this instance.
     */
    Container.prototype.forEach = function () {
        var cont = this._cont;
        var callback = getCallback(arguments);
        for (var key in cont) {
            callback(cont[key], key, this);
        }
        return this;
    };

    /**
     * Invokes the passed callback function for all elements in this container,
     * and passes the element keys as first parameter. The elements will be
     * visited in no specific order. This additional method next to the method
     * Container.forEach() may be useful when a callback function wants to
     * process the element keys only, and thus can be passed directly as
     * parameter to this method without having to create an anonymous function
     * with an unused first parameter.
     *
     * @param {Function} callback
     *  The callback function that will be invoked for each element. Receives
     *  the following parameters:
     *  (1) {String} key
     *      The key of the current element.
     *  (2) {Any} value
     *      The value of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Conatienr}
     *  A reference to this instance.
     */
    Container.prototype.forEachKey = function (callback, context) {
        var cont = this._cont;
        for (var key in cont) {
            callback.call(context, key, cont[key], this);
        }
        return this;
    };

    /**
     * Returns whether the predicate callback function returns a truthy value
     * for at least one element in this container. The elements will be visited
     * in no specific order.
     *
     * @param {Function|String} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this container until it returns a truthy value. After that,
     *  iteration will be stopped immediately, and the remaining elements will
     *  not be visited anymore. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Boolean}
     *  Whether the predicate callback function returns a truthy value for at
     *  least one element in this container.
     */
    Container.prototype.some = function () {
        var cont = this._cont;
        var predicate = getCallback(arguments);
        for (var key in cont) {
            if (predicate(cont[key], key, this)) {
                return true;
            }
        }
        return false;
    };

    /**
     * Returns whether the predicate callback function returns a truthy value
     * for all elements in this container. The elements will be visited in no
     * specific order.
     *
     * @param {Function|String} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this container until it returns a falsy value. After that, iteration
     *  will be stopped immediately, the remaining elements will not be visited
     *  anymore. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Boolean}
     *  Whether the predicate callback function returns a truthy value for all
     *  elements in this container.
     */
    Container.prototype.every = function () {
        var cont = this._cont;
        var predicate = getCallback(arguments);
        for (var key in cont) {
            if (!predicate(cont[key], key, this)) {
                return false;
            }
        }
        return true;
    };

    /**
     * Invokes the passed callback function for all elements in this container,
     * and returns the return values as plain array. The elements will be
     * visited in no specific order.
     *
     * @param {Function|String} generator
     *  The generator callback function that will be invoked for each element
     *  in this container. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method. The return value of the
     *  callback function will be inserted into the resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Array<Any>}
     *  The return values of the callback function, in no specific order.
     */
    Container.prototype.map = function () {
        var cont = this._cont;
        var generator = getCallback(arguments);
        var result = [];
        for (var key in cont) {
            result.push(generator(cont[key], key, this));
        }
        return result;
    };

    /**
     * Invokes the passed predicate callback function for all elements in this
     * container, and returns a plain array with all element values resulting
     * in a truthy return value. The elements will be visited in no specific
     * order.
     *
     * @param {Function|String} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this container. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method. If the return value is
     *  truthy, the element value will be inserted into the resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Array<Any>}
     *  The element values that passed the truth test, in no specific order.
     */
    Container.prototype.filter = function () {
        var cont = this._cont;
        var predicate = getCallback(arguments);
        var result = [];
        for (var key in cont) {
            var value = cont[key];
            if (predicate(value, key, this)) {
                result.push(value);
            }
        }
        return result;
    };

    /**
     * Invokes the passed predicate callback function for all elements in this
     * container, and returns a plain array with all element values resulting
     * in a falsy return value. The elements will be visited in no specific
     * order.
     *
     * @param {Function|String} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this container. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method. If the return value is
     *  falsy, the element value will be inserted into the resulting array.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Array<Any>}
     *  The element values that do not pass the truth test, in no specific
     *  order.
     */
    Container.prototype.reject = function () {
        var cont = this._cont;
        var predicate = getCallback(arguments);
        var result = [];
        for (var key in cont) {
            var value = cont[key];
            if (!predicate(value, key, this)) {
                result.push(value);
            }
        }
        return result;
    };

    /**
     * Invokes the passed callback function for all elements in this container,
     * and reduces them to a single result value. The elements will be visited
     * in no specific order.
     *
     * @param {Any} initial
     *  The initial result value passed to the first invocation of the callback
     *  function (if the container is empty, this value becomes the final
     *  result value immediately).
     *
     * @param {Function} aggregate
     *  The aggregator callback function that will be invoked for each element
     *  in this container. Receives the following parameters:
     *  (1) {Any} prevValue
     *      The return value of the previous invocation of this callback
     *      function (or the initial value on first invocation).
     *  (2) {Any} value
     *      The value of the current element from this container.
     *  (3) {String} key
     *      The key of the current element from this container.
     *  (4) {Container} cont
     *      A reference to this instance.
     *  The return value of the callback function will become the new result
     *  value that will be passed as 'prevValue' with the next element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Any}
     *  The final result (the last return value of the callback function).
     */
    Container.prototype.reduce = function (initial, aggregate, context) {
        var cont = this._cont;
        for (var key in cont) {
            initial = aggregate.call(context, initial, cont[key], key, this);
        }
        return initial;
    };

    /**
     * Returns the values of the elements in this container as sorted array.
     *
     * @param {Function|String} sorter
     *  The callback function that will be invoked for each element in this
     *  container to receive a sort index relative to the other elements, or
     *  the name of a property of each object element that will be used for
     *  sorting.
     *  Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  The return value of the  callback function will be used as sort index
     *  for the respective element.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Array<Any>}
     *  The values of the elements in this container sorted in ascending order.
     */
    Container.prototype.sortBy = function (sorter, context) {
        return _.sortBy(this._cont, (typeof sorter === 'function') ? function (value, key) {
            return sorter.call(context, value, key, this);
        } : sorter, this);
    };

    /**
     * Returns the values of a specific property from all object elements in
     * this container. The elements will be visited in no specific order.
     *
     * @param {String} name
     *  The name of an object property.
     *
     * @returns {Array<Any>}
     *  The values of the specified property from all object elements in this
     *  container, in no specific order.
     */
    Container.prototype.pluck = function (name) {
        var cont = this._cont;
        var result = [];
        for (var key in cont) {
            result.push(cont[key][name]);
        }
        return result;
    };

    /**
     * Invokes the passed callback function for all elements in this container,
     * and returns the return values as plain JavaScript object mapped by the
     * same keys. The elements will be visited in no specific order.
     *
     * @param {Function|String} generator
     *  The generator callback function that will be invoked for each element
     *  in this container. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method. The return value of the
     *  callback function will be inserted into the resulting object.
     *
     * @param {Object} [context]
     *  The calling context for the callback function.
     *
     * @returns {Object}
     *  The return values of the callback function, as JavaScript object.
     */
    Container.prototype.mapObject = function () {
        var cont = this._cont;
        var generator = getCallback(arguments);
        var result = {};
        for (var key in cont) {
            result[key] = generator(cont[key], key, this);
        }
        return result;
    };

    /**
     * Invokes the passed predicate callback function for all elements in this
     * container, and returns a plain JavaScript object with all element values
     * resulting in a truthy return value, mapped by the same keys. The
     * elements will be visited in no specific order.
     *
     * @param {Function|String} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this container. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method. If the return value is
     *  truthy, the element value will be inserted into the resulting object.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Object}
     *  The element values that passed the truth test, in no specific order.
     */
    Container.prototype.pick = function () {
        var cont = this._cont;
        var predicate = getCallback(arguments);
        var result = {};
        for (var key in cont) {
            var value = cont[key];
            if (predicate(value, key, this)) {
                result[key] = value;
            }
        }
        return result;
    };

    /**
     * Invokes the passed predicate callback function for all elements in this
     * container, and returns a plain JavaScript object with all element values
     * resulting in a falsy return value, mapped by the same keys. The elements
     * will be visited in no specific order.
     *
     * @param {Function|String} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this container. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method. If the return value is
     *  falsy, the element value will be inserted into the resulting object.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Object}
     *  The element values that do not pass the truth test, in no specific
     *  order.
     */
    Container.prototype.omit = function () {
        var cont = this._cont;
        var predicate = getCallback(arguments);
        var result = {};
        for (var key in cont) {
            var value = cont[key];
            if (!predicate(value, key, this)) {
                result[key] = value;
            }
        }
        return result;
    };

    /**
     * Returns the value of the first element in this container that passes a
     * truth test according to the specified predicate. The elements will be
     * visited in no specific order.
     *
     * @param {Function|String} predicate
     *  The predicate callback function that will be invoked for each element
     *  in this container until it returns a truthy value. After that, the
     *  element value will be returned immediately. Receives the following
     *  parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Any}
     *  The value of the first element that passes the truth test; or the value
     *  undefined, if no element passes the truth test.
     */
    Container.prototype.find = function () {
        var cont = this._cont;
        var predicate = getCallback(arguments);
        for (var key in cont) {
            var value = cont[key];
            if (predicate(value, key, this)) {
                return value;
            }
        }
    };

    /**
     * Transforms all elements in this container in-place by replacing them
     * with the return value of the specified callback function. The elements
     * will be visited in no specific order.
     *
     * @param {Function|String} generator
     *  The generator callback function that will be invoked for each element
     *  in this container. The element will be replaced with the return value
     *  of this function. Receives the following parameters:
     *  (1) {Any} value
     *      The value of the current element.
     *  (2) {String} key
     *      The key of the current element.
     *  (3) {Container} cont
     *      A reference to this instance.
     *  Alternatively, the name of a method that will be invoked on the object
     *  elements in this container. All following parameters passed to this
     *  method will be forwarded to the object method.
     *
     * @param {Object} [context]
     *  The calling context for the predicate callback function.
     *
     * @returns {Container}
     *  A reference to this instance.
     */
    Container.prototype.transform = function () {
        var cont = this._cont;
        var generator = getCallback(arguments);
        for (var key in cont) {
            cont[key] = generator(cont[key], key, this);
        }
        return this;
    };

    // exports ================================================================

    return Container;

});
