/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/tk/config', [
    'settings!io.ox/office'
], function (settings) {

    'use strict';

    // static class Config ====================================================

    /**
     * Wraps the configuration of the OX Documents module.
     */
    var Config = {};

    // methods ----------------------------------------------------------------

    /**
     * Returns the value of a specific configuration property, or the complete
     * set of all available configuration properties.
     *
     * @param {String} key
     *  The key of a specific configuration property.
     *
     * @param {Any} [def]
     *  A default value that will be returned if the configuration property
     *  specified with the parameter 'key'does not exist.
     *
     * @returns {Any}
     *  The value of the specified configuration property.
     */
    Config.get = function (key, def) {
        return settings.get(key, def);
    };

    /**
     * Returns the state of a specific boolean configuration property.
     *
     * @param {String} key
     *  The key of a specific boolean configuration property.
     *
     * @param {Boolean} [def=false]
     *  The default state, if the configuration property specified with the
     *  parameter 'key' does not exist.
     *
     * @returns {Boolean}
     *  The state of the specified configuration property. If the property does
     *  not exist, this method will return the default value.
     */
    Config.getFlag = function (key, def) {
        return Config.get(key, def === true) === true;
    };

    /**
     * Returns the state of a specific integer configuration property.
     *
     * @param {String} key
     *  The key of a specific integer configuration property.
     *
     * @param {Number} [def=0]
     *  The default value, if the configuration property specified with the
     *  parameter 'key' does not exist.
     *
     * @param {Number} [min]
     *  The lower bound for the value. If omitted, all integer values will be
     *  allowed.
     *
     * @returns {Number}
     *  The state of the specified configuration property. If the property does
     *  not exist, this method will return the default value.
     */
    Config.getInt = function (key, def, min) {
        var int = Config.get(key, def);
        if (typeof int !== 'number') { int = 0; }
        if (typeof min === 'number') { int = Math.max(int, min); }
        return Math.floor(int);
    };

    /**
     * Sets a value of a specific configuration property. Be careful
     * to only set single properties.
     *
     * @param {String} key
     *  The key of a specific configuration property.
     *
     * @param {Any} value
     *  The value to be set for the specified configuration key.
     *
     *  if value is an [Array] you can't delete single entries in that array,
     *  better call set(key, null) and then save the correct array again
     *
     * @returns {jQuery.Promise}
     *  A promise that will be resolved when the server request has been sent
     *  successfully.
     */
    Config.set = function (key, value) {

        // create a object tree with the value as property
        var updateTree = key.split('/').reduceRight(function (val, dir) {
            var obj = {};
            obj[dir] = val;
            return obj;
        }, value);

        // Call merge with update tree to prevent that the whole
        // tree is sent to the backend
        return settings.set(key, value).merge(updateTree);
    };

    /**
     * Returns the state of the specified Boolean flag in the anchor part of
     * the browser URL.
     *
     * @param {String} flag
     *  The name of the URL anchor flag.
     *
     * @param {Boolean} [def=false]
     *  The default state, if the URL does not contain the specified option.
     *
     * @returns {Boolean}
     *  The state of the specified flag in the anchor part of the URL.
     */
    Config.getUrlFlag = function (flag, def) {
        var value = _.url.hash(flag);
        return (/^(1|true)$/i).test(value) || ((/^(0|false)$/i).test(value) ? false : (def === true));
    };

    /**
     * Returns the state of the specified Boolean flag in the anchor part of
     * the browser URL, but only if the application runs in debug mode (see
     * constant Config.DEBUG for details). Without debug mode, this method
     * simply returns the passed default value.
     *
     * @param {String} flag
     *  The name of the URL anchor flag.
     *
     * @param {Boolean} [def=false]
     *  The default state, if the URL does not contain the specified option, or
     *  if the application is not in debug mode.
     *
     * @returns {Boolean}
     *  The state of the specified flag in the anchor part of the URL.
     */
    Config.getDebugUrlFlag = function (flag, def) {
        return Config.DEBUG ? Config.getUrlFlag(flag, def) : (def === true);
    };

    /**
     * Returns whether a specific debug mode is enabled in all OX Documents
     * applications. The debug mode will be enabled, if the passed server-side
     * configuration property is set to true, and the passed URL hash option is
     * NOT explicitly set to false; or if the server-side configuration
     * property 'debugondemand' is set to true, and the passed URL hash option
     * is set to true.
     *
     * @param {String} configKey
     *  The name of the server-side configuration item.
     *
     * @param {String} urlFlag
     *  The name of the URL hash option.
     *
     * @returns {Boolean}
     *  Whether the specified debug mode is enabled.
     */
    Config.getDebugState = function (configKey, urlFlag) {
        return Config.getFlag(configKey) ? Config.getUrlFlag(urlFlag, true) : (Config.getFlag('module/debugondemand') && Config.getUrlFlag(urlFlag));
    };

    // constants --------------------------------------------------------------

    /**
     * The full identifier of the current locale, with leading lower-case
     * language identifier, and trailing upper-case country identifier,
     * separated by an underscore character, e.g. 'en_US'.
     *
     * @constant
     *
     * @type {String}
     */
    Config.LOCALE = ox.language;

    /**
     * The lower-case language identifier of the current locale, e.g. 'en'.
     *
     * @constant
     *
     * @type {String}
     */
    Config.LANGUAGE = ox.language.split('_')[0];

    /**
     * The upper-case country identifier of the current locale, e.g. 'US'.
     *
     * @constant
     *
     * @type {String}
     */
    Config.COUNTRY = ox.language.split('_')[1];

    /**
     * A Boolean flag specifying whether the debug mode is enabled in all OX
     * Documents applications. Debug mode will be enabled, if the server-side
     * configuration property 'debugavailable' is set to true, and the URL hash
     * option 'office:enable-debug' is NOT explicitly set to false; or if the
     * server-side configuration property 'debugondemand' is set to true, and
     * the URL hash option 'office:enable-debug' is set to true.
     *
     * @constant
     *
     * @type {Boolean}
     */
    Config.DEBUG = Config.getDebugState('module/debugavailable', 'office:enable-debug');

    /**
     * Specifies whether the current platform is an automated test environment
     * (e.g. Selenium).
     *
     * @constant
     *
     * @type {Boolean}
     */
    Config.AUTOTEST = Config.getDebugUrlFlag('office:testautomation');

    /**
     * Specifies whether the current platform is a unit test environment.
     *
     * @constant
     *
     * @type {Boolean}
     */
    Config.UNITTEST = ox._UNITTEST === true;

    // exports ================================================================

    return Config;

});
