/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/format/tablestyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/gradient',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/tablestylecollection',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/components/table/table'
], function (Utils, Color, Gradient, Border, LineHeight, AttributeUtils, TableStyleCollection, DOM, Table) {

    'use strict';

    // definitions for table attributes
    var DEFINITIONS = {

        /**
         * Width of the table, as number in 1/100 of millimeters.
         */
        width: {
            def: 'auto'
        },

        /**
         * Fill color of the table.
         */
        fillColor: {
            def: Color.AUTO
        },

        /**
         * Grid width of columns in relative units. It is an array of numbers
         */
        tableGrid: {
            def: [],
            scope: 'element'
        },

        /**
         * Array containing information, if conditional attributes will be
         * used. As default value, all styles will be used, so that this
         * array can be empty.
         */
        exclude: {
            def: [],
            scope: 'element'
        },

        /**
         * Left border of the table (will be set in the table cells).
         */
        borderLeft: { def: Border.NONE },

        /**
         * Top border of the table (will be set in the table cells).
         */
        borderTop: { def: Border.NONE },

        /**
         * Right border of the table (will be set in the table cells).
         */
        borderRight: { def: Border.NONE },

        /**
         * Bottom border of the table (will be set in the table cells).
         */
        borderBottom: { def: Border.NONE },

        /**
         * Inner horizontal borders inside the table (will be set in the
         * table cells).
         */
        borderInsideHor: { def: Border.NONE },

        /**
         * Inner vertical borders inside the table (will be set in the
         * table cells).
         */
        borderInsideVert: { def: Border.NONE },

        /**
         * Top padding of the whole table
         */
        paddingTop: { def: 0 },

        /**
         * Bottom padding of the whole table
         */
        paddingBottom: { def: 0 },

        /**
         * Left padding of the whole table
         */
        paddingLeft: { def: 0 },

        /**
         * Right padding of the whole table
         */
        paddingRight: { def: 0 }
    };

    // the conditional table attributes (only 'wholeTable' has no geometric dependencies)
    var GEOMETRIC_TABLE_ATTRIBUTES = ['firstRow', 'lastRow', 'firstCol', 'lastCol', 'band1Vert', 'band2Vert', 'band1Hor', 'band2Hor', 'northEastCell', 'northWestCell', 'southEastCell', 'southWestCell'],

        // the border properties
        BORDER_PROPS = ['borderTop', 'borderBottom', 'borderLeft', 'borderRight', 'borderInsideHor', 'borderInsideVert'];

    // class TableStyles ======================================================

    /**
     * Contains the style sheets for table formatting attributes. The CSS
     * formatting will be written to table elements and their rows and cells.
     *
     * @constructor
     *
     * @extends TableStyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    var TableStyles = TableStyleCollection.extend({ constructor: function (docModel) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        TableStyleCollection.call(this, docModel, {
            families: 'cell paragraph character changes',
            styleAttributesResolver: resolveTableStyleAttributes,
            elementAttributesResolver: resolveTableElementAttributes
        });

        // private methods ----------------------------------------------------

        // in presentation app the border attributes are assigned to 'cell' family, not 'table' family. But for correct
        // resolving of table border properties it is necessary, that they are assigned to the 'table' family, too (48647).
        function transferWholeTableBordersToTable(tableStyleAttributes, attributes) {
            if (tableStyleAttributes.wholeTable.cell) {
                _.each(BORDER_PROPS, function (prop) {
                    if (tableStyleAttributes.wholeTable.cell[prop] && (!tableStyleAttributes.wholeTable.table || !tableStyleAttributes.wholeTable.table[prop])) {
                        if (!attributes.table) { attributes.table = {}; }
                        attributes.table[prop] = _.copy(tableStyleAttributes.wholeTable.cell[prop], true);
                    }
                });
            }
        }

        /**
         * Returns the attributes of the specified attribute family contained
         * in table style sheets. Resolves the conditional attributes that
         * match the position of the passed source element.
         *
         * @param {Object} styleAttributes
         *  The complete 'attributes' object of a table style sheet.
         *
         * @param {jQuery} tableNode
         *  The DOM table node referring to a table style sheet, as jQuery
         *  object.
         *
         * @param {jQuery} cellNode
         *  The DOM cell node that has initially requested the formatting
         *  attributes of a table style sheet, as jQuery object (may be empty).
         *
         * @returns {Object}
         *  The formatting attributes extracted from the passed style sheet
         *  attributes object, as map of attribute value maps (name/value
         *  pairs), keyed by attribute family.
         */
        function resolveTableStyleAttributes(styleAttributes, tableNode, cellNode) {

            // the column interval covered by the cell
            var colInterval = (cellNode.length > 0) ? Table.getGridColumnRangeOfCell(cellNode) : null;
            // the row interval covered by the cell
            var rowInterval = (cellNode.length > 0) ? Table.getGridRowRangeOfCell(cellNode) : null;
            // the width of the table (number of columns)
            var colCount = Table.getColumnCount(tableNode);
            // the height of the table (number of rows)
            var rowCount = Table.getRowCount(tableNode);
            // the resulting cell attribute set
            var attrSet = {};

            // the options needed to resolve the correct cell attributes
            var options = { firstRow: true, lastRow: true, firstCol: true, lastCol: true, bandsHor: true, bandsVert: true };
            _.each(AttributeUtils.getExplicitAttributes(tableNode, { family: 'table', direct: true }).exclude, function (key) {
                delete options[key];
            });

            // in presentation app the border attributes are set at the 'cell' family
            if (docModel.useSlideMode() && styleAttributes.wholeTable) {
                transferWholeTableBordersToTable(styleAttributes, attrSet);
            }

            return self.extendAttributeSet(attrSet, self.resolveCellAttributeSet(styleAttributes, colInterval, rowInterval, colCount, rowCount, options));
        }

        /**
         * Returns the attributes of the specified attribute family contained
         * in table style sheets. Resolves the conditional attributes that
         * match the position of the passed source element.
         *
         * @param {Object} elementAttributes
         *  The explicit attributes of the table element, as map of attribute
         *  value maps (name/value pairs), keyed by attribute family.
         *
         * @param {jQuery} tableNode
         *  The DOM table node, as jQuery object.
         *
         * @param {jQuery} cellNode
         *  The DOM cell node that has initially requested the formatting
         *  attributes, as jQuery object (may be empty).
         *
         * @returns {Object}
         *  The resolved explicit formatting attributes, as map of attribute
         *  value maps (name/value pairs), keyed by attribute family.
         */
        function resolveTableElementAttributes(elementAttributes, tableNode, cellNode) {
            return resolveTableStyleAttributes({ wholeTable: elementAttributes }, tableNode, cellNode);
        }

        /**
         * Setting the table background. This can currently only be done using a style.
         * Only supported in presentation app.
         *
         * @param {jQuery} table
         *  The DOM table node, as jQuery object.
         *
         * @param {Object} styleSheetAttributeMap
         *  The style sheet attributes of the table
         */
        function handleTableBackground(table, styleSheetAttributeMap) {

            if (styleSheetAttributeMap && styleSheetAttributeMap.backgroundStyle && styleSheetAttributeMap.backgroundStyle.fill) {

                var // the fill attributes of the slide
                    fillAttrs = styleSheetAttributeMap.backgroundStyle.fill,
                    // the CSS properties for the slide
                    cssProps = {};

                // calculate the CSS fill attributes
                switch (fillAttrs.type) {
                    case 'none':
                        // clear everything: color and bitmaps
                        cssProps.background = '';
                        break;

                    case 'solid':
                        cssProps.background = docModel.getCssColor(fillAttrs.color, 'fill', docModel.getThemeTargets(table));
                        break;

                    case 'gradient':
                        var
                            elmSlide  = table[0],
                            targets   = docModel.getThemeTargets(table),
                            gradient  = Gradient.create(Gradient.getDescriptorFromAttributes(fillAttrs)),
                            dataUrl   = Gradient.parseImageDataUrl(gradient, docModel, { width: elmSlide.offsetWidth, height: elmSlide.offsetHeight }, targets);

                        cssProps.background = ['url("', dataUrl, '")'].join('');
                        cssProps.backgroundSize = '100%'; // #49272
                        break;

                    default:
                        Utils.warn('TableStyles.handleTableBackground(): unknown fill type "' + fillAttrs.type + '"');
                }

                // apply the fill attributes
                table.css(cssProps);

            } else {
                table.css({ background: '' }); // removing background
            }

        }

        /**
         * Will be called for every table element whose attributes have been
         * changed. Repositions and reformats the table according to the passed
         * attributes.
         *
         * @param {jQuery} table
         *  The <table> element whose table attributes have been changed, as
         *  jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute value maps (name/value pairs), keyed by
         *  attribute family, containing the effective attribute values merged
         *  from style sheets and explicit attributes.
         *
         * @param {Boolean} async
         *  If set to true, the table formatting will be updated asynchronously
         *  in a browser timeout loop.
         *
         * @returns {jQuery.Promise}
         *  The promise of a Deferred object that will be resolved when the
         *  the table is completely formatted.
         */
        function updateTableFormatting(table, mergedAttributes, async) {

            var // the table styles/formatter
                tableStyles = docModel.getStyleCollection('table'),
                // the table row styles/formatter
                tableRowStyles = docModel.getStyleCollection('row'),
                // the table cell styles/formatter
                tableCellStyles = docModel.getStyleCollection('cell'),
                // the paragraph styles/formatter
                paragraphStyles = docModel.getStyleCollection('paragraph'),
                // the column widths
                tableGrid = mergedAttributes.table.tableGrid,
                // the explicit table attributes
                tableAttributes = AttributeUtils.getExplicitAttributes(table, { family: 'table' }),
                // all row nodes in the table
                rowNodes = DOM.getTableRows(table),
                // a collector for all cells in the first row
                allCellsInFirstRow = rowNodes.first().children('td').children('div.cell'),
                // the upper left cell of the table (the div.cell element)
                firstCell = allCellsInFirstRow.first(),
                // a counter for the cells (performance)
                cellCounter = allCellsInFirstRow.length * rowNodes.length,
                // whether this update was triggered by a resize of table (column width)
                isTableResize = false,
                // whether the table has a 'simple' style
                isSimpleTableStyle = false,
                // whether cells were removed from the table
                isGUIRemoveOperation = false,
                // the number of the first row, that requires a formatting update
                minUpdateRow = null,
                // the style sheet attributes for the table
                styleSheetAttributeMap = null;

            // checking if the attributes have table geometric dependencies
            // -> then a complete reformatting of the table is required.
            // Otherwise only the new cells (insert operations) or no cells
            // (delete operations) need to be formatted.
            function checkSimpleTableStyle() {

                var isSimple = true;

                _.each(GEOMETRIC_TABLE_ATTRIBUTES, function (attribute) {
                    if ((attribute in styleSheetAttributeMap) && !_.contains(tableAttributes.exclude, attribute)) {
                        isSimple = false;
                    }
                });
                return isSimple;
            }

            // helper function to update the top and the left resizers in the table cell
            function updateTopLeftResizers() {

                // the old upper left cell is no longer the current upper left cell
                table.find('.upperLeftCell').removeClass('upperLeftCell').removeData('cellcount');

                // removing old top and left resizer nodes
                table.find('.resize.top, .resize.left').remove();

                // adding new top and left resizers to first row and first column
                allCellsInFirstRow.prepend($('<div>').addClass('resize top'));

                rowNodes.children('td:first-child').children('div.cell').prepend($('<div>').addClass('resize left'));

                // marking the new upper left cell (saving cell count for performance reasons)
                firstCell.addClass('upperLeftCell').data('cellcount', cellCounter);
            }

            // updates the formatting of the passed table row
            function updateRowFormatting(rowNode) {

                // update the table row itself
                tableRowStyles.updateElementFormatting(rowNode);

                // update table cells (do NOT pass merged attributes passed
                // to the updateTableFormatting() method as base attributes,
                // attributes must be recalculated for each single cell!)
                $(rowNode).children('td').each(function () {

                    var cellFormatting = true;

                    // Performance: Updating only new cells, if this update was triggered from a GUI event
                    // and if the table has a simple table style -> otherwise all cells need to be updated
                    // Performance: Simplified cell formatting for deleteColumn and deleteRow GUI operations.
                    // For the insert operations (insertRow or insertColumn) cell (and paragraph) formatting
                    // is always required.
                    if (isGUIRemoveOperation) {
                        if (isSimpleTableStyle) {
                            cellFormatting = false;  // using simplified cell formatting update (only taking care of tab stops)
                            // TODO: For not simple table styles the full cell formatting is only required for those cells
                            // that are behind the new/removed cell in the row for insertColumn and deleteColumn
                        }
                    }

                    // Performance: Resizing the table (column width) requires only tab stop recalculation
                    if (isTableResize) { cellFormatting = false; }

                    if (cellFormatting) {
                        // complete formatting update of the cell -> this is necessary for complex table styles, but also for simple
                        // table styles, because paragraph formatting is also necessary. For example determines the paragraph height the
                        // height of the row, if the table row height is set to 'auto'.
                        tableCellStyles.updateElementFormatting(this);
                    } else {
                        // checking if a cell contains tab stops, at least these need to be updated, if cell width was changed
                        if ($(this).find(DOM.TAB_NODE_SELECTOR).length > 0) {
                            // updating the tabs is always required, because the width of a cell might have changed -> updating all paragraphs in the table cell
                            Utils.iterateSelectedDescendantNodes(DOM.getCellContentNode(this), DOM.PARAGRAPH_NODE_SELECTOR, function (paragraph) {
                                paragraphStyles.updateTabStops(paragraph);
                            }, undefined, { children: true });
                        }
                    }
                });
            }

            // no formatting of exceeded size table (52505)
            if (DOM.isExceededSizeTableNode(table)) {
                docModel.getChangeTrack().updateChangeTrackAttributes(table, mergedAttributes); // change track attribute handling
                return $.when();
            }

            // checking if the attributes have table geometric dependencies or table background
            if (mergedAttributes.styleId) {
                styleSheetAttributeMap = tableStyles.getStyleSheetAttributeMap(mergedAttributes.styleId);
                isSimpleTableStyle = checkSimpleTableStyle();
            }

            var width = tableAttributes.width;
            // TODO: Adding support for width in percent
            // -> better for changes of page size or page orientation
            if (width === 'auto') {
                table.css('width', '100%');
            } else {
                // adapt size of table to its parent (36132,36163)
                width = Utils.convertHmmToLength(width, 'px', 1);
                var parentWidth = table.parent().width();
                table.css('width', (parentWidth < width) ? '100%' : (width + 'px'));
            }

            table.css('background-color', docModel.getCssColor(tableAttributes.fillColor, 'fill'));

            // handling of table background
            // if (docModel.useSlideMode()) { handleTableBackground(table, styleSheetAttributeMap); }

            if (table.data('gui')) {
                if (table.data('gui') === 'remove') {
                    isGUIRemoveOperation = true;
                }
                table.removeData('gui');
            }

            if (table.data('tableResize')) {
                isTableResize = true;
                table.removeData('tableResize');
            }

            // reformatting table colgroup in Firefox during pasting internal clip board, task 29401, task 30477
            if ((table.data('internalClipboard') || table.data('undoRedoRunning')) && _.browser.Firefox) {
                table.removeData('internalClipboard');
                table.removeData('undoRedoRunning');
            }

            // update column widths (also required for bug fixes 29212 and 29675)
            Table.updateColGroup(table, tableGrid);

            // Fix for 30821 (on webkit browsers), max-width of 100% is ignored, if width is defined
            if ((_.browser.WebKit) && (table.closest('div.cellcontent').length > 0) && (table.css('max-width') === '100%') && (table.width() > table.closest('div.cellcontent').width())) {
                table.css('width', '');
            }

            // Performance: Immediately return, if no update required for simple tables (for example after deleteRows).
            // Even update of tab stops is not required for insertRow and deleteRow. But insertRow requires update of
            // paragraph attributes.
            if (_.isNumber(table.data('reducedTableFormatting'))) {
                minUpdateRow = table.data('reducedTableFormatting');
                table.removeData('reducedTableFormatting');
                if ((isSimpleTableStyle) && (isGUIRemoveOperation)) {
                    return $.when();
                }
            }

            // Performance check for very slow devices. In this case an asynchronous update is
            // required, so that the user gets faster feed back
            async = async || (Utils.IOS && Utils.SLOW_DEVICE);

            // updating the top and left resize markers in the table. This is necessary, if the
            // upper left cell is no longer the upper left cell of the table or if the number of
            // cells in the table has changed (insertRow, ...)
            if (docModel.useSlideMode() && firstCell.length > 0 && (!firstCell.hasClass('upperLeftCell') || firstCell.data('cellcount') !== cellCounter)) { updateTopLeftResizers(); }

            // Performance: Reducing the number of rows, because upper rows require no update
            if (_.isNumber(minUpdateRow)) {
                rowNodes = rowNodes.slice(minUpdateRow);
            }

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(table, mergedAttributes);

            // update rows, cells, ... , but only if required
            var promise = null;
            if (async) {
                // process the table asynchronously
                promise = self.iterateArraySliced(rowNodes, updateRowFormatting, 'TableStyles.updateTableFormatting');
                promise.always(function () {
                    if (docModel.useSlideMode()) { handleTableBackground(table, styleSheetAttributeMap); } // (48370)
                    docModel.trigger('table:formatting:done');
                });

            } else {
                // update all rows at once (but asynchronous, so that for example a new column is visible immediately)
                promise = self.executeDelayed(function () {
                    rowNodes.each(function () { updateRowFormatting(this); });
                    if (docModel.useSlideMode()) { handleTableBackground(table, styleSheetAttributeMap); } // (48370)
                    docModel.trigger('table:formatting:done');
                }, 'TableStyles.updateTableFormatting');
            }

            return promise;
        }

        // inserts the table style sheet described by the passed parameters, if it does not exist yet, and marks it as dirty
        function insertMissingTableStyle(styleId, styleName, uiPriority, category, attrs) {
            // check by style sheet name (they are always loaded in English from Word, while ste style IDs are localized)
            if (!self.containsStyleSheetByName(styleName)) {
                self.createDirtyStyleSheet(styleId, styleName, attrs, { priority: uiPriority, category: category });
            } else {
                self.setStyleOptions(styleId, { category: category, priority: uiPriority });
            }
        }

        // inserts a group of table style definitions for the primary text color and all six accent colors
        function insertMissingTableStyleGroup(baseStyleId, baseStyleName, uiPriority, generateAttributesFunc, category) {
            insertMissingTableStyle(baseStyleId, baseStyleName, uiPriority, category ? category : baseStyleName, generateAttributesFunc('text1'));
            for (var index = 1; index <= 6; index += 1) {
                insertMissingTableStyle(baseStyleId + '-Accent' + index, baseStyleName + ' Accent ' + index, uiPriority, category ? category : baseStyleName, generateAttributesFunc('accent' + index));
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Performance: Helper function to update an optional table background.
         * This is only supported in the presentation app.
         *
         * @param {Node|jQuery} drawing
         *  The table drawing node.
         */
        this.updateTableDrawingBackground = function (drawing) {

            // the explicit drawing attributes
            var expAttrs = AttributeUtils.getExplicitAttributes(drawing);
            // the table node inside the drawing
            var tableNode = $(drawing).find(DOM.TABLE_NODE_SELECTOR);

            if (tableNode.length > 0 && expAttrs && expAttrs.styleId) {
                var styleSheetAttributeMap = self.getStyleSheetAttributeMap(expAttrs.styleId);
                handleTableBackground(tableNode, styleSheetAttributeMap);
            }
        };

        /**
         * Check the stored table styles of a document and adds a 'missing'
         * default table style. This ensures that we can insert tables that
         * are based on a reasonable default style.
         */
        this.insertMissingTableStyles = function () {

            var styleNames = self.getStyleSheetNames(),
                parentId = self.getDefaultStyleId(),
                hasDefaultStyle = _.isString(parentId) && (parentId.length > 0),
                defTableDef = docModel.getDefaultLateralTableDefinition(),
                defTableAttr = docModel.getDefaultLateralTableAttributes(),
                defTableOdfDef = null,
                defTableOdfAttr = null,
                isODF = docModel.getApp().isODF();

            if (!hasDefaultStyle) {
                // Add a missing default table style
                self.createDirtyStyleSheet(defTableDef.styleId, defTableDef.styleName, defTableAttr, { priority: 59, defStyle: defTableDef.default });
            } else {
                // Search for a style defined in the document that can be used for tables
                // If we cannot find it we have to add it.
                var lowestUIPriority = 99,
                    tableStyleId = null;
                _(styleNames).each(function (name, id) {
                    var uiPriority = self.getUIPriority(id);

                    if (_.isNumber(uiPriority) && (uiPriority < lowestUIPriority)) {
                        tableStyleId = id;
                        lowestUIPriority = uiPriority;
                    }
                });

                if ((!tableStyleId) || ((tableStyleId === self.getDefaultStyleId()) && (lowestUIPriority === 99))) {
                    // OOXML uses a default table style which contains no border
                    // definitions. Therfore we add our own default table style
                    // if we only find the default style with uiPriority 99
                    self.createDirtyStyleSheet(defTableDef.styleId, defTableDef.styleName, defTableAttr, { parent: parentId, priority: 59 });
                }

                if (docModel.getApp().isODF() && tableStyleId === self.getDefaultStyleId() && tableStyleId === '_default') {
                    // ODF uses a default table style which contains no border definitions.
                    // Therefore we add our own default table style, see #49674
                    defTableOdfDef = docModel.getDefaultLateralTableODFDefinition();
                    defTableOdfAttr = docModel.getDefaultLateralTableODFAttributes();
                    self.createDirtyStyleSheet(defTableOdfDef.styleId, defTableOdfDef.styleName, defTableOdfAttr, { parent: parentId, priority: 59 });
                }
            }

            // adding further table styles, if not already available
            if (docModel.useSlideMode()) {

                insertMissingTableStyle('{2D5ABB26-0587-4C30-8999-92F81FD0307C}', 'No Style, No Grid', 80, '', TableStyles.getTableNoGridNoStylePresentation('text1'));
                insertMissingTableStyle('{5940675A-B579-460E-94D1-54222C63F5DA}', 'No Style, Table Grid', 82, '', TableStyles.getTableGridPresentation('text1'));

                insertMissingTableStyle('{616DA210-FB5B-4158-B5E0-FEB733F419BA}', 'Light Style 3', 85, 'Light', TableStyles.getLightStyle3TableStyleAttributesPresentation('text1', 'text1', isODF));
                insertMissingTableStyle('{BC89EF96-8CEA-46FF-86C4-4CE0E7609802}', 'Light Style 3 - Accent 1', 85, 'Light', TableStyles.getLightStyle3TableStyleAttributesPresentation('accent1', 'text1', isODF));
                insertMissingTableStyle('{5DA37D80-6434-44D0-A028-1B22A696006F}', 'Light Style 3 - Accent 2', 85, 'Light', TableStyles.getLightStyle3TableStyleAttributesPresentation('accent2', 'text1', isODF));
                insertMissingTableStyle('{8799B23B-EC83-4686-B30A-512413B5E67A}', 'Light Style 3 - Accent 3', 85, 'Light', TableStyles.getLightStyle3TableStyleAttributesPresentation('accent3', 'text1', isODF));
                insertMissingTableStyle('{ED083AE6-46FA-4A59-8FB0-9F97EB10719F}', 'Light Style 3 - Accent 4', 85, 'Light', TableStyles.getLightStyle3TableStyleAttributesPresentation('accent4', 'text1', isODF));
                insertMissingTableStyle('{BDBED569-4797-4DF1-A0F4-6AAB3CD982D8}', 'Light Style 3 - Accent 5', 85, 'Light', TableStyles.getLightStyle3TableStyleAttributesPresentation('accent5', 'text1', isODF));
                insertMissingTableStyle('{E8B1032C-EA38-4F05-BA0D-38AFFFC7BED3}', 'Light Style 3 - Accent 6', 85, 'Light', TableStyles.getLightStyle3TableStyleAttributesPresentation('accent6', 'text1', isODF));

                insertMissingTableStyle('{073A0DAA-6AF3-43AB-8588-CEC1D06C72B9}', 'Medium Style 2', 59, 'Medium', TableStyles.getMediumStyle2TableStyleAttributesPresentation('dark1', 'light1', 'dark1'));
                insertMissingTableStyle('{5C22544A-7EE6-4342-B048-85BDC9FD1C3A}', 'Medium Style 2 - Accent 1', 59, 'Medium', TableStyles.getMediumStyle2TableStyleAttributesPresentation('accent1', 'light1', 'dark1'));
                insertMissingTableStyle('{21E4AEA4-8DFA-4A89-87EB-49C32662AFE0}', 'Medium Style 2 - Accent 2', 59, 'Medium', TableStyles.getMediumStyle2TableStyleAttributesPresentation('accent2', 'light1', 'dark1'));
                insertMissingTableStyle('{F5AB1C69-6EDB-4FF4-983F-18BD219EF322}', 'Medium Style 2 - Accent 3', 59, 'Medium', TableStyles.getMediumStyle2TableStyleAttributesPresentation('accent3', 'light1', 'dark1'));
                insertMissingTableStyle('{00A15C55-8517-42AA-B614-E9B94910E393}', 'Medium Style 2 - Accent 4', 59, 'Medium', TableStyles.getMediumStyle2TableStyleAttributesPresentation('accent4', 'light1', 'dark1'));
                insertMissingTableStyle('{7DF18680-E054-41AD-8BC1-D1AEF772440D}', 'Medium Style 2 - Accent 5', 59, 'Medium', TableStyles.getMediumStyle2TableStyleAttributesPresentation('accent5', 'light1', 'dark1'));
                insertMissingTableStyle('{93296810-A885-4BE3-A3E7-6D5BEEA58F35}', 'Medium Style 2 - Accent 6', 59, 'Medium', TableStyles.getMediumStyle2TableStyleAttributesPresentation('accent6', 'light1', 'dark1'));

                insertMissingTableStyle('{8EC20E35-A176-4012-BC5E-935CFFF8708E}', 'Medium Style 3', 59, 'Medium', TableStyles.getMediumStyle3TableStyleAttributesPresentation('dark1', 'light1', 'dark1'));
                insertMissingTableStyle('{6E25E649-3F16-4E02-A733-19D2CDBF48F0}', 'Medium Style 3 - Accent 1', 59, 'Medium', TableStyles.getMediumStyle3TableStyleAttributesPresentation('accent1', 'light1', 'dark1'));
                insertMissingTableStyle('{85BE263C-DBD7-4A20-BB59-AAB30ACAA65A}', 'Medium Style 3 - Accent 2', 59, 'Medium', TableStyles.getMediumStyle3TableStyleAttributesPresentation('accent2', 'light1', 'dark1'));
                insertMissingTableStyle('{EB344D84-9AFB-497E-A393-DC336BA19D2E}', 'Medium Style 3 - Accent 3', 59, 'Medium', TableStyles.getMediumStyle3TableStyleAttributesPresentation('accent3', 'light1', 'dark1'));
                insertMissingTableStyle('{EB9631B5-78F2-41C9-869B-9F39066F8104}', 'Medium Style 3 - Accent 4', 59, 'Medium', TableStyles.getMediumStyle3TableStyleAttributesPresentation('accent4', 'light1', 'dark1'));
                insertMissingTableStyle('{74C1A8A3-306A-4EB7-A6B1-4F7E0EB9C5D6}', 'Medium Style 3 - Accent 5', 59, 'Medium', TableStyles.getMediumStyle3TableStyleAttributesPresentation('accent5', 'light1', 'dark1'));
                insertMissingTableStyle('{2A488322-F2BA-4B5B-9748-0D474271808F}', 'Medium Style 3 - Accent 6', 59, 'Medium', TableStyles.getMediumStyle3TableStyleAttributesPresentation('accent6', 'light1', 'dark1'));

                insertMissingTableStyle('{D7AC3CCA-C797-4891-BE02-D94E43425B78}', 'Medium Style 4', 59, 'Medium', TableStyles.getMediumStyle4TableStyleAttributesPresentation('dark1', 'dark1'));
                insertMissingTableStyle('{69CF1AB2-1976-4502-BF36-3FF5EA218861}', 'Medium Style 4 - Accent 1', 59, 'Medium', TableStyles.getMediumStyle4TableStyleAttributesPresentation('accent1', 'dark1'));
                insertMissingTableStyle('{8A107856-5554-42FB-B03E-39F5DBC370BA}', 'Medium Style 4 - Accent 2', 59, 'Medium', TableStyles.getMediumStyle4TableStyleAttributesPresentation('accent2', 'dark1'));
                insertMissingTableStyle('{0505E3EF-67EA-436B-97B2-0124C06EBD24}', 'Medium Style 4 - Accent 3', 59, 'Medium', TableStyles.getMediumStyle4TableStyleAttributesPresentation('accent3', 'dark1'));
                insertMissingTableStyle('{C4B1156A-380E-4F78-BDF5-A606A8083BF9}', 'Medium Style 4 - Accent 4', 59, 'Medium', TableStyles.getMediumStyle4TableStyleAttributesPresentation('accent4', 'dark1'));
                insertMissingTableStyle('{22838BEF-8BB2-4498-84A7-C5851F593DF1}', 'Medium Style 4 - Accent 5', 59, 'Medium', TableStyles.getMediumStyle4TableStyleAttributesPresentation('accent5', 'dark1'));
                insertMissingTableStyle('{16D9F66E-5EB9-4882-86FB-DCBF35E3C3E4}', 'Medium Style 4 - Accent 6', 59, 'Medium', TableStyles.getMediumStyle4TableStyleAttributesPresentation('accent6', 'dark1'));

            } else {

                insertMissingTableStyle('TableGrid', 'Table Grid', 1, '', TableStyles.getTableGrid());
                insertMissingTableStyle('TableNormal', 'Normal Table', 2, '', TableStyles.getNormalTable());

                insertMissingTableStyleGroup('LightShading',   'Light Shading',    60, TableStyles.getLightShadingTableStyleAttributes, 'Light');
                insertMissingTableStyleGroup('MediumShading1', 'Medium Shading 1', 63, TableStyles.getMediumShading1TableStyleAttributes, 'Medium');
                insertMissingTableStyleGroup('MediumShading2', 'Medium Shading 2', 64, TableStyles.getMediumShading2TableStyleAttributes, 'Medium');
                insertMissingTableStyleGroup('MediumGrid1',    'Medium Grid 1',    67, TableStyles.getMediumGrid1TableStyleAttributes, 'Medium');
            }
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributes('table', DEFINITIONS);

        // register the formatting handlers for DOM elements
        this.registerFormatHandler(updateTableFormatting);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = self = null;
        });

    } }); // class TableStyles

    // static methods ---------------------------------------------------------

    TableStyles.getNormalTable = function () {
        return {
            wholeTable: {
                table: {
                    paddingLeft: 190,
                    paddingTop: 0,
                    paddingRight: 190,
                    paddingBottom: 0
                }
            }
        };
    };

    TableStyles.getTableGrid = function () {
        return {
            wholeTable: {
                table: {
                    borderLeft: { style: 'single', color: { type: 'scheme', value: 'text1' }, width: 18 },
                    borderTop: { style: 'single', color: { type: 'scheme', value: 'text1' }, width: 18 },
                    borderRight: { style: 'single', color: { type: 'scheme', value: 'text1' }, width: 18 },
                    borderBottom: { style: 'single', color: { type: 'scheme', value: 'text1' }, width: 18 },
                    borderInsideHor: { style: 'single', color: { type: 'scheme', value: 'text1' }, width: 18 },
                    borderInsideVert: { style: 'single', color: { type: 'scheme', value: 'text1' }, width: 18 },
                    paddingLeft: 190,
                    paddingTop: 0,
                    paddingRight: 190,
                    paddingBottom: 0
                },
                paragraph: {
                    lineHeight: { type: 'percent', value: 100 },
                    marginBottom: 0
                }
            }
        };
    };

    TableStyles.getBorderStyleFromAttributes = function (attributes) {

        var allWidths = [];
        var width = 'none';

        if (attributes) {
            if (Border.isVisibleBorder(attributes.borderLeft)) { allWidths.push(attributes.borderLeft.width); }
            if (Border.isVisibleBorder(attributes.borderRight)) { allWidths.push(attributes.borderRight.width); }
            if (Border.isVisibleBorder(attributes.borderTop)) { allWidths.push(attributes.borderTop.width); }
            if (Border.isVisibleBorder(attributes.borderBottom)) { allWidths.push(attributes.borderBottom.width); }
            if (Border.isVisibleBorder(attributes.borderInsideHor)) { allWidths.push(attributes.borderInsideHor.width); }
            if (Border.isVisibleBorder(attributes.borderInsideVert)) { allWidths.push(attributes.borderInsideVert.width); }
        }

        allWidths = _.uniq(allWidths);

        if (allWidths.length === 1) {
            width = allWidths[0];
            width = Utils.convertHmmToLength(width, 'pt', 0.1);  // converting from 1/100 mm to pt
        }

        return width;
    };

    TableStyles.getAttributesFromBorderStyle = function (borderWidth, attributes) {

        var borderLeft = _.clone(attributes.borderLeft),
            borderRight = _.clone(attributes.borderRight),
            borderTop = _.clone(attributes.borderTop),
            borderBottom = _.clone(attributes.borderBottom),
            borderInsideHor = _.clone(attributes.borderInsideHor),
            borderInsideVert = _.clone(attributes.borderInsideVert);

        // converting from pt to 1/100 mm
        borderWidth = Utils.convertLengthToHmm(borderWidth, 'pt');

        borderLeft.width = borderWidth;
        borderRight.width = borderWidth;
        borderTop.width = borderWidth;
        borderBottom.width = borderWidth;
        borderInsideHor.width = borderWidth;
        borderInsideVert.width = borderWidth;

        return {
            borderLeft:       borderLeft,
            borderRight:      borderRight,
            borderTop:        borderTop,
            borderBottom:     borderBottom,
            borderInsideHor:  borderInsideHor,
            borderInsideVert: borderInsideVert
        };
    };

    // Light Shading table style attributes

    TableStyles.getLightShadingTableStyleAttributes = function (colorValue) {

        var SINGLE_BORDER_LINE = { style: 'single', color: { type: 'scheme', value: colorValue }, width: 35 },
            BAND_COLOR = { type: 'scheme', value: colorValue, transformations: [{ type: 'tint', value: 24706 }] },
            ROW_CELL_ATTRIBUTES = { borderLeft: Border.NONE, borderTop: SINGLE_BORDER_LINE, borderRight: Border.NONE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE },
            BAND_CELL_ATTRIBUTES = { borderLeft: Border.NONE, borderRight: Border.NONE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: BAND_COLOR };

        return {
            wholeTable: {
                table: { borderTop: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE },
                paragraph: { lineHeight: LineHeight.SINGLE, marginBottom: 0 },
                character: { color: { type: 'scheme', value: colorValue, transformations: [{ type: 'shade', value: 74902 }] } }
            },
            firstRow: { cell: ROW_CELL_ATTRIBUTES, paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 }, character: { bold: true } },
            lastRow: { cell: ROW_CELL_ATTRIBUTES, paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 }, character: { bold: true } },
            firstCol: { character: { bold: true } },
            lastCol: { character: { bold: true } },
            band1Vert: { cell: BAND_CELL_ATTRIBUTES },
            band1Hor: { cell: BAND_CELL_ATTRIBUTES }
        };
    };

    // Medium Shading 1 table style attributes

    TableStyles.getMediumShading1TableStyleAttributes = function (colorValue) {

        var SCHEME_COLOR = { type: 'scheme', value: colorValue },
            BAND_COLOR = { type: 'scheme', value: colorValue, transformations: [{ type: 'tint', value: 24706 }] },
            SINGLE_BORDER_LINE = { style: 'single', color: SCHEME_COLOR, width: 35 },
            DOUBLE_BORDER_LINE = { style: 'double', color: SCHEME_COLOR, width: 26 };

        return {
            wholeTable: {
                table: { borderLeft: SINGLE_BORDER_LINE, borderTop: SINGLE_BORDER_LINE, borderRight: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: SINGLE_BORDER_LINE },
                paragraph: { lineHeight: LineHeight.SINGLE, marginBottom: 0 }
            },
            firstRow: {
                cell: { borderLeft: SINGLE_BORDER_LINE, borderTop: SINGLE_BORDER_LINE, borderRight: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: SCHEME_COLOR },
                paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 },
                character: { bold: true, color: { type: 'scheme', value: 'background1' } }
            },
            lastRow: {
                cell: { borderLeft: SINGLE_BORDER_LINE, borderTop: DOUBLE_BORDER_LINE, borderRight: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE },
                paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 },
                character: { bold: true }
            },
            firstCol: { character: { bold: true } },
            lastCol: { character: { bold: true } },
            band1Vert: { cell: { fillColor: BAND_COLOR } },
            band1Hor: { cell: { borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: BAND_COLOR } },
            band2Hor: { cell: { borderInsideHor: Border.NONE, borderInsideVert: Border.NONE } }
        };
    };

    // Medium Shading 2 table style attributes

    TableStyles.getMediumShading2TableStyleAttributes = function (colorValue) {

        var SCHEME_COLOR = { type: 'scheme', value: colorValue },
            BACK_COLOR = { type: 'scheme', value: 'background1' },
            BAND_COLOR = { type: 'scheme', value: 'background1', transformations: [{ type: 'shade', value: 84706 }] },
            SINGLE_BORDER_LINE = { style: 'single', color: Color.AUTO, width: 79 },
            DOUBLE_BORDER_LINE = { style: 'double', color: Color.AUTO, width: 26 },
            NORTH_CELL_ATTRIBUTES = { borderLeft: Border.NONE, borderTop: SINGLE_BORDER_LINE, borderRight: Border.NONE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE };

        return {
            wholeTable: {
                table: { borderTop: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE },
                paragraph: { lineHeight: LineHeight.SINGLE, marginBottom: 0 }
            },
            firstRow: {
                cell: { borderLeft: Border.NONE, borderTop: SINGLE_BORDER_LINE, borderRight: Border.NONE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: SCHEME_COLOR },
                paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 },
                character: { bold: true, color: BACK_COLOR }
            },
            lastRow: {
                cell: { borderLeft: Border.NONE, borderTop: DOUBLE_BORDER_LINE, borderRight: Border.NONE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: BACK_COLOR },
                paragraph: { lineHeight: LineHeight.SINGLE, marginTop: 0, marginBottom: 0 },
                character: { color: Color.AUTO }
            },
            firstCol: {
                cell: { borderLeft: Border.NONE, borderTop: Border.NONE, borderRight: Border.NONE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: SCHEME_COLOR },
                character: { bold: true, color: BACK_COLOR }
            },
            lastCol: {
                cell: { borderLeft: Border.NONE, borderRight: Border.NONE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: SCHEME_COLOR },
                character: { bold: true, color: BACK_COLOR }
            },
            band1Vert: { cell: { borderLeft: Border.NONE, borderRight: Border.NONE, borderInsideHor: Border.NONE, borderInsideVert: Border.NONE, fillColor: BAND_COLOR } },
            band1Hor: { cell: { fillColor: BAND_COLOR } },
            northEastCell: { cell: NORTH_CELL_ATTRIBUTES },
            northWestCell: { cell: NORTH_CELL_ATTRIBUTES, character: { color: BACK_COLOR } }
        };
    };

    // Medium Grid 1 table style attributes

    TableStyles.getMediumGrid1TableStyleAttributes = function (colorValue) {

        var BAND_COLOR = { type: 'scheme', value: colorValue, transformations: [{ type: 'tint', value: 49804 }] },
            SINGLE_BORDER_LINE = { style: 'single', color: { type: 'scheme', value: colorValue }, width: 35 },
            THICK_BORDER_LINE = { style: 'single', color: { type: 'scheme', value: colorValue }, width: 79 };

        return {
            wholeTable: {
                table: { borderLeft: SINGLE_BORDER_LINE, borderTop: SINGLE_BORDER_LINE, borderRight: SINGLE_BORDER_LINE, borderBottom: SINGLE_BORDER_LINE, borderInsideHor: SINGLE_BORDER_LINE, borderInsideVert: SINGLE_BORDER_LINE },
                cell: { fillColor: { type: 'scheme', value: colorValue, transformations: [{ type: 'tint', value: 24706 }] } },
                paragraph: { lineHeight: LineHeight.SINGLE, marginBottom: 0 }
            },
            firstRow: { character: { bold: true } },
            lastRow: { cell: { borderTop: THICK_BORDER_LINE }, character: { bold: true } },
            firstCol: { character: { bold: true } },
            lastCol: { character: { bold: true } },
            band1Vert: { cell: { fillColor: BAND_COLOR } },
            band1Hor: { cell: { fillColor: BAND_COLOR } }
        };
    };

    // Light Style 3 table style attributes (Presentation)
    TableStyles.getLightStyle3TableStyleAttributesPresentation = function (colorValue1, colorValue2, isODF) {
        return {
            wholeTable: {
                cell: {
                    fillType: 'none',
                    borderLeft: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } },
                    borderRight: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } },
                    borderBottom: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } },
                    borderTop: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } },
                    borderInsideHor: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } },
                    borderInsideVert: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } }
                },
                character: { color: { type: 'scheme', value: colorValue2 } }
            },
            band1Hor: { cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue1, transformations: [{ type: (isODF ? 'tint' : 'alpha'), value: 20000 }] } } }, // 52757, ODP, using 'tint' instead of 'alpha' for fallback value
            band1Vert: { cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue1, transformations: [{ type: (isODF ? 'tint' : 'alpha'), value: 20000 }] } } },
            firstRow: {
                cell: { fillType: 'none', borderBottom: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } } },
                character: { bold: true }
            },
            lastRow: {
                cell: { fillType: 'none', borderTop: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } } },
                character: { bold: true }
            }
        };
    };

    // Medium Style 2 table style attributes (Presentation)
    TableStyles.getMediumStyle2TableStyleAttributesPresentation = function (colorValue1, colorValue2, colorValue3) { // 'accent1', 'light1', 'dark1'
        return {
            wholeTable: {
                cell: {
                    fillType: 'solid',
                    fillColor: { type: 'scheme', value: colorValue1, transformations: [{ type: 'tint', value: 20000 }] },
                    borderLeft: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue2 } },
                    borderRight: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue2 } },
                    borderBottom: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue2 } },
                    borderTop: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue2 } },
                    borderInsideHor: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue2 } },
                    borderInsideVert: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue2 } }
                },
                character: { color: { type: 'scheme', value: colorValue3 } }
            },
            band1Hor: { cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue1, transformations: [{ type: 'tint', value: 40000 }] } } },
            band1Vert: { cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue1, transformations: [{ type: 'tint', value: 40000 }] } } },
            firstCol: {
                cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue1 } },
                character: { color: { type: 'scheme', value: colorValue2 }, bold: true }
            },
            firstRow: {
                cell: {
                    fillType: 'solid',
                    fillColor: { type: 'scheme', value: colorValue1 },
                    borderBottom: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue2 } }
                },
                character: {
                    color: { type: 'scheme', value: colorValue2 },
                    bold: true
                }
            },
            lastCol: {
                cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue1 } },
                character: { color: { type: 'scheme', value: colorValue2 }, bold: true }
            },
            lastRow: {
                cell: {
                    fillType: 'solid',
                    fillColor: { type: 'scheme', value: colorValue1 },
                    borderTop: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue2 } }
                },
                character: {
                    color: { type: 'scheme', value: colorValue2 },
                    bold: true
                }
            }
        };
    };

    // Medium Style 3 table style attributes (Presentation)
    TableStyles.getMediumStyle3TableStyleAttributesPresentation = function (colorValue1, colorValue2, colorValue3) { // 'accent1', 'light1', 'dark1'
        return {
            wholeTable: {
                cell: {
                    fillType: 'solid',
                    fillColor: { type: 'scheme', value: colorValue2 },
                    borderBottom: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue3 } },
                    borderTop: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue3 } }
                },
                character: { color: { type: 'scheme', value: colorValue3 } }
            },
            band1Hor: { cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue3, transformations: [{ type: 'tint', value: 20000 }] } } },
            band1Vert: { cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue3, transformations: [{ type: 'tint', value: 20000 }] } } },
            firstCol: {
                cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue1 } },
                character: { color: { type: 'scheme', value: colorValue2 }, bold: true }
            },
            firstRow: {
                cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue1 }, borderBottom: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue3 } } },
                character: { color: { type: 'scheme', value: colorValue2 }, bold: true }
            },
            lastCol: {
                cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue1 } },
                character: { color: { type: 'scheme', value: colorValue2 }, bold: true }
            },
            lastRow: {
                cell: { fillType: 'solid', fillColor: { type: 'scheme', value: colorValue2 }, borderTop: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue3 } } }
            }
            // southEastCell:{
            //     character: { color: { type: 'scheme', value: colorValue3 } }
            // },
            // southWestCell:{
            //     character: { color: { type: 'scheme', value: colorValue3 } }
            // }
        };
    };

    // Medium Style 4 table style attributes (Presentation)
    TableStyles.getMediumStyle4TableStyleAttributesPresentation = function (colorValue1, colorValue2) {
        return {
            wholeTable: {
                cell: {
                    fillType: 'solid',
                    fillColor: { type: 'scheme', value: colorValue1, transformations: [{ type: 'tint', value: 20000 }] },
                    borderLeft: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } },
                    borderRight: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } },
                    borderBottom: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } },
                    borderTop: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } },
                    borderInsideHor: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } },
                    borderInsideVert: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } }
                },
                character: { color: { type: 'scheme', value: colorValue2 } }
            },
            band1Hor: { cell: { fillColor: { type: 'scheme', value: colorValue1, transformations: [{ type: 'tint', value: 40000 }] } } },
            band1Vert: { cell: { fillColor: { type: 'scheme', value: colorValue1, transformations: [{ type: 'tint', value: 40000 }] } } },
            firstRow: {
                cell: { fillColor: { type: 'scheme', value: colorValue1, transformations: [{ type: 'tint', value: 20000 }] } },
                character: { bold: true }
            },
            lastRow: {
                cell: { fillColor: { type: 'scheme', value: colorValue1, transformations: [{ type: 'tint', value: 20000 }] }, borderTop: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue1 } } },
                character: { bold: true }
            }
        };
    };

    // Table grid, no style
    TableStyles.getTableGridPresentation = function (colorValue) {
        return {
            wholeTable: {
                cell: {
                    fillType: 'none',
                    borderLeft: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue } },
                    borderRight: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue } },
                    borderBottom: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue } },
                    borderTop: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue } },
                    borderInsideHor: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue } },
                    borderInsideVert: { style: 'single', width: 35, color: { type: 'scheme', value: colorValue } }
                }
            }
        };
    };

    // No grid, no style
    TableStyles.getTableNoGridNoStylePresentation = function (colorValue) {
        return {
            wholeTable: {
                cell: {
                    fillType: 'none',
                    borderLeft: { style: 'none' },
                    borderRight: { style: 'none' },
                    borderBottom: { style: 'none' },
                    borderTop: { style: 'none' },
                    borderInsideHor: { style: 'none' },
                    borderInsideVert: { style: 'none' }
                },
                character: { fontName: 'Calibri Light', color: { type: 'scheme', value: colorValue }, bold: false }
            }
        };
    };

    // exports ================================================================

    return TableStyles;

});
