/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/format/tablerowstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/textframework/utils/dom'
], function (Utils, StyleCollection, DOM) {

    'use strict';

    var // definitions for table row attributes
        DEFINITIONS = {

            /**
             * The height of the table row. If 'auto' then the row is auto sized.
             */
            height: { def: 'auto' },
            /**
             * Denotes row to be repeated across pages of split table (repeated rows).
             */
            headerRow: { def: false }

        };

    // class TableRowStyles ===================================================

    /**
     * Contains the style sheets for table row formatting attributes. The CSS
     * formatting will be read from and written to <tr> elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     */
    var TableRowStyles = StyleCollection.extend({ constructor: function (docModel) {

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'row', {
            families: 'changes',
            styleSheetSupport: false
        });

        // private methods ----------------------------------------------------

        /**
         * Will be called for every table row element whose attributes have been
         * changed. Repositions and reformats the table row according to the passed
         * attributes.
         *
         * @param {jQuery} row
         *  The <tr> element whose table row attributes have been changed,
         *  as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family,
         *  containing the effective attribute values merged from style sheets and
         *  explicit attributes.
         */
        function updateTableRowFormatting(row, mergedAttributes) {

            var // the row attributes of the passed attribute map
                rowAttributes = mergedAttributes.row,
                rowHeight = null,
                cellHeight = null,
                maxHeight = 0,
                paddingHeight = 0;

            if (rowAttributes.height !== 'auto') {
                if (_.browser.WebKit || _.browser.IE) {

                    // Chrome requires row height at the cells, setting height at <tr> is ignored.
                    rowHeight = Utils.convertHmmToLength(rowAttributes.height, 'px', 1);
                    row.children('td').each(function () {

                        if (docModel.useSlideMode()) {
                            paddingHeight = 0; // this is necessary for precision of vertical resize of table drawings
                        } else {
                            paddingHeight =  Utils.getElementCssLength(this, 'padding-top') +
                                             Utils.getElementCssLength(this, 'padding-bottom') +
                                             Utils.getElementCssLength(this, 'border-bottom-width');
                        }

                        cellHeight = (rowHeight - paddingHeight) + 'px';
                        if (_.browser.IE) {
                            maxHeight = Math.max(rowHeight - paddingHeight, maxHeight);
                            $(this).css('min-height', cellHeight);
                        } else {
                            $(this).css('height', cellHeight);
                        }
                    });
                } else {
                    // FireFox requires row height at the rows. Setting height at cells, makes
                    // height of cells unchanged, even if text leaves the cell.
                    row.css('height', Utils.convertHmmToCssLength(rowAttributes.height, 'px', 1));
                }
            } else {
                // required for undo, if the table cell was not set before
                if (_.browser.WebKit) {
                    // Chrome requires row height at the cells.
                    row.children('td').css('height', 0);
                } else if (_.browser.IE) {
                    row.children('td').css('min-height', '');
                } else {
                    // FireFox requires row height at the rows.
                    row.css('height', '');
                }
            }
            if (rowAttributes.headerRow) { row.addClass(DOM.TABLE_REPEAT_ROWNODE_CLASSNAME); }

            // change track attribute handling
            docModel.getChangeTrack().updateChangeTrackAttributes(row, mergedAttributes);

        }

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributes('row', DEFINITIONS);

        // register the formatting handler for DOM elements
        this.registerFormatHandler(updateTableRowFormatting);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } }); // class TableRowStyles

    // exports ================================================================

    return TableRowStyles;

});
