/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/textframework/format/paragraphstyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/lineheight',
    'io.ox/office/editframework/model/stylecollection',
    'io.ox/office/textframework/utils/dom'
], function (Utils, ValueMap, Color, LineHeight, StyleCollection, DOM) {

    'use strict';

    // definitions for paragraph attributes
    var DEFINITIONS = {

        alignment: {
            def: 'left'
        },

        fillColor: {
            def: Color.AUTO // auto for paragraph fill resolves to 'transparent'
        },

        /**
         * Line height relative to font settings. The CSS attribute 'line-height'
         * must be set separately at every descendant text span because a relative
         * line height (e.g. 200%) would not be derived from the paragraph relatively
         * to the spans, but absolutely according to the paragraph's font size.
         *
         * Example: The paragraph has a font size of 12pt and a line-height of 200%,
         * resulting in 24pt. This value will be derived absolutely to a span with a
         * font size of 6pt, resulting in a relative line height of 24pt/6pt = 400%
         * instead of the expected 200%.
         */
        lineHeight: { def: LineHeight.SINGLE },

        tabStops: {
            def: [],
            merge: function (tabStops1, tabStops2) {
                // Merge tabStops2 into array tabStops1

                // process as map for faster access
                var tabStopMap = ValueMap.rekey(tabStops1, 'pos');

                // process all tab stops in the second array
                tabStops2.forEach(function (tabStop2) {
                    if (tabStop2.value === 'clear') {
                        tabStopMap.remove(tabStop2.pos);
                    } else {
                        tabStopMap.insert(tabStop2.pos, tabStop2);
                    }
                });

                // create and return a sorted array
                return tabStopMap.sortBy('pos');
            }
        },

        indentFirstLine: { def: 0 },

        indentLeft: { def: 0 },

        indentRight: { def: 0 }
    };

    // non-breaking space character
    var NBSP = '\xa0';

    // maps fill character attribute values to fill characters
    var TAB_FILL_CHARS = { dot: '.', hyphen: '-', underscore: '_' };

    // class ParagraphStyles ==================================================

    /**
     * Contains the style sheets for paragraph formatting attributes. The CSS
     * formatting will be read from and written to DOM paragraph elements. This
     * class is intended to be used as a base class for paragraph style
     * collections of the different applications
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     */
    var ParagraphStyles = StyleCollection.extend({ constructor: function (docModel, initOptions) {

        // the attribute pool used by this instance
        var attributePool = null;

        // the collection of character style sheets
        var charStyles = null;

        // whether default tab stop size is specified by paragraph (false), or globally at document (true)
        var globalDefTabStop = false;

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'paragraph', initOptions);

        // private methods ----------------------------------------------------

        /**
         * Will be called for every paragraph whose character attributes have
         * been changed.
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose attributes have been changed, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        function updateParagraphFormatting(paragraph, mergedAttributes) {

            // the merged paragraph attributes
            var paraAttrs = mergedAttributes.paragraph;

            paragraph.css({
                textAlign: paraAttrs.alignment,
                backgroundColor: docModel.getCssColor(paraAttrs.fillColor, 'fill')
            });
        }

        /**
         * Will be called for paragraphs used as preview elements in the GUI.
         *
         * @param {jQuery} paragraph
         *  The preview paragraph node, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A complete attribute set, containing the effective attribute values
         *  merged from style sheets and explicit attributes.
         */
        function updatePreviewFormatting(paragraph, mergedAttributes) {

            // the merged paragraph attributes
            var paraAttrs = mergedAttributes.paragraph;

            // update formatting of the passed paragraph node
            paragraph.css('text-align', paraAttrs.alignment);

            // format the text spans contained in the paragraph element
            paragraph.children().each(function () {
                charStyles.updateElementFormatting(this, { baseAttributes: mergedAttributes, preview: true });
            });
        }

        /**
         * Calculates the width of a right aligned tab element
         * depending on the position and the remaining size of
         * the following nodes.
         *
         * @param {Object} tabStop
         *  The right aligned tab stop.
         *
         * @param {jQuery} tabNode
         *  The right aligned tab node as jQuery.
         *
         * @param {Number} tabNodePosHMM
         *  The current position of the tab node in 1/100th mm.
         *
         * @returns {Number}
         *  The width of the right aligned tab stop. May be zero if the
         *  tab stop is not active.
         */
        function calcRightAlignTabstop(tabStop, tabNode, tabNodePosHMM) {
            var nextNodes = tabNode.nextAll(),
                node,
                rightPos = tabNodePosHMM,
                width = 0, i = 0, len = nextNodes.length;

            // Loop through all following nodes and sum up their width. Exceeding
            // the tab stop position or a new tab stops the iteration.
            for (i = 0; i < len && rightPos < tabStop.pos; i++) {
                node = $(nextNodes.get(i));
                if (DOM.isTabNode(node)) {
                    break;
                }
                rightPos += Utils.convertLengthToHmm(node.width(), 'px');
            }

            if (rightPos < tabStop.pos) {
                // Active right aligned tab stop. Subtract at least 1/10mm
                // rounding error regarding position values from the browser.
                width = tabStop.pos - rightPos - 10;
                width = Math.max(0, width);
            }

            return width;
        }

        function getMixedTab(paraTabStops) {
            var leftTab = _.find(paraTabStops, function (tab) {
                return !tab.value || tab.value === 'left';
            });
            var rightTab = _.find(paraTabStops, function (tab) {
                return tab.value === 'center' || tab.value === 'right';
            });
            if (leftTab && rightTab) {
                return rightTab;
            }
        }

        /**
         * right tab can lay right above from the border, this cannot be triggered by MS-WORD interface,
         * but it is a typical behavior for MS-"table of content"
         * so we calculate if right tab is outside of the margin, and if true,
         * we reduce the margin-right of the paragraph
         */
        function calculateMarginForMixedTab(paragraph, paraAttributes, mixedTab) {
            paragraph.css('margin-right', null);

            if (!mixedTab) {
                return;
            }

            var pageAttrs = attributePool.getDefaultValues('page');
            var maxParaWidth = pageAttrs.width - pageAttrs.marginLeft - pageAttrs.marginRight;
            var indentRight = paraAttributes.paragraph.indentRight;
            var currentParaWidth = mixedTab.pos + indentRight;

            if (currentParaWidth > maxParaWidth) {
                var newIndentRight = indentRight - (currentParaWidth - maxParaWidth);
                paragraph.css('margin-right', (newIndentRight / 100) + 'mm');
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Updates all tabulators in the passed paragraph node. This method
         * uses both the default tab size and existing paragraph tab stop
         * definitions. Fill characters are also supported.
         *
         * @attention
         *  Currently, only left&right aligned tabs are supported!
         *
         * @param {jQuery} paragraph
         *  The paragraph node whose tabulator nodes will be updated, as jQuery
         *  object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute
         *  family, containing the effective attribute values merged from style
         *  sheets and explicit attributes.
         */
        this.implUpdateTabStops = function (paragraph, mergedAttributes) {

            // performance: check existence of tab elements first
            if (paragraph.find(DOM.TAB_NODE_SELECTOR).length === 0) { return; }

            var // default tab stop width from document settings or paragraph settings
                defaultTabStop = globalDefTabStop ? docModel.getDocumentAttribute('defaultTabStop') : mergedAttributes.paragraph.defaultTabSize,
                // paragraph tab stop definitions
                paraTabStops = mergedAttributes.paragraph.tabStops,
                // effective left margin of the paragraph
                marginLeft = Utils.convertCssLengthToHmm(paragraph.css('margin-left'), 'px'),
                // end position of previous tabulator
                prevTabEndPos = null,
                // first node of paragraph
                firstNode = null,
                // top position of first node
                topFirst = 0,
                // height of first node
                heightFirst = 0,
                // leftmost position used a base to calculate tab raster
                zeroLeft = 0,
                // first line  indent
                indentFirstLine = mergedAttributes.paragraph.indentFirstLine,
                // intersect function
                intersect = function (xs1, xe1, xs2, xe2) {
                    return ((xs1 <= xs2) && (xe1 >= xs2)) || ((xs2 <= xs1) && (xe2 >= xs1));
                },
                // active paragraph tab stops
                activeParaTabStops = null,
                // if paraTabStops has left and right oriented tabs, mixedTabs is set by the right-tab. if it is set it overwrites behavior of text-aligment and para-margin
                mixedTab = getMixedTab(paraTabStops);

            // zoom factor in floating point notation
            var zoomFactor = docModel.getApp().getView().getZoomFactor();

            if (paragraph.length) {
                // Calculate top position and height of first paragraph child. Our paragraph
                // have always at least one child element.
                firstNode = $(paragraph.get(0).firstChild);
                topFirst = firstNode.length ? Utils.convertLengthToHmm(firstNode.position().top, 'px') : 0;
                heightFirst = firstNode.length ? Utils.convertLengthToHmm(firstNode.height(), 'px') : 0;

                //reset all tab sizes, for a much cleaner new calculating
                //(for example changing text-alignment from justify to left)
                paragraph.children(DOM.TAB_NODE_SELECTOR).each(function () {
                    $(this).css('width', 0);
                });

                paragraph.children(DOM.TAB_NODE_SELECTOR).each(function () {

                    var // the current tab node of the paragraph
                        tabNode = $(this),
                        // the span within the tab node containing the fill characters
                        tabSpan = $(this.firstChild),
                        // the position of the tab node
                        pos = tabNode.position(),
                        // the left bound position of the tab node in 1/100th mm
                        leftHMM = 0,
                        // the top bound position of the tab node in 1/100th mm
                        topHMM = Utils.convertLengthToHmm(pos.top, 'px'),
                        // the top bound position of the first child node of the paragraph in 1/100th mm
                        topSpanHMM = 0,
                        // the height of the first child node of the paragraph in 1/100th mm
                        heightSpanHMM = 0,
                        // the calculated width of the tab stop
                        width = 0,
                        // the fill character of the tab stop
                        fillChar = null,
                        // the tab stop values
                        tabStop,
                        // ignore the tab position and continue with the next one
                        ignore = false,
                        // tab stop type
                        tabStopType = 'left',
                        // indent first line tab stop position
                        indentFirstLineTabStopValue = 0,
                        // first line tab stop
                        indentFirstLineTabStop = null,
                        // insert index for virtual first line indent tab stop
                        indentFirstLineTabStopIndex = 0;

                    function calcLeftTab() { return Math.max(0, tabStop.pos - (leftHMM % tabStop.pos)); }
                    function calcRightTab() { return calcRightAlignTabstop(tabStop, tabNode, leftHMM); }
                    function calcMiddleTab() { return (calcLeftTab() + calcRightTab()) / 2; }

                    if (mergedAttributes.paragraph.alignment === 'center') {
                        // for a centered paragraph tab stops are calculated where the first character is treated as leftmost (0) position
                        zeroLeft = firstNode.length ? -(marginLeft + Utils.convertLengthToHmm(firstNode.position().left, 'px')) : -marginLeft;
                    } else {
                        zeroLeft = marginLeft;
                    }

                    // calculate left bound position of the tab node in 1/100th mm, including zoom factor for browsers using transf.scale
                    leftHMM = zeroLeft + Utils.convertLengthToHmm(pos.left / zoomFactor, 'px');

                    if (prevTabEndPos) {
                        // Check to see if we are in a chain of tabs. Force to use the previous tab end position
                        // as start position. Some browsers provide imprecise positions therefore we use the correct
                        // previous position from the last calculation.

                        // Check previous node and that text is empty - this is a precondition for the chaining
                        // code.
                        if (tabNode.prev().length && tabNode.prev().text().length === 0) {
                            var checkPrevPrevNode = tabNode.prev().length ? tabNode.prev().prev() : null;
                            if (checkPrevPrevNode.length && DOM.isTabNode(checkPrevPrevNode) && prevTabEndPos.top === topHMM) {
                                leftHMM = prevTabEndPos.center || prevTabEndPos.right;
                            }
                        }
                    }

                    // Special case for first line indent. For the first line the negative indent
                    // must be used as a tab stop position using the absolute value. We need to set
                    // at least one character for the tabSpan to have a valid width otherwise we
                    // won't get a correct height which is needed for the check.
                    if (indentFirstLine < 0) {
                        indentFirstLineTabStopValue = Math.abs(indentFirstLine);
                        if (marginLeft > 0) {
                            // In case of a margin left, we need to add the value to the indentFirstLineTabStopValue! Bug 45030
                            indentFirstLineTabStopValue += marginLeft;
                        }
                        // Fix for 29265 and 30847: Removing empty text node in span with '.contents().remove().end()'.
                        // Otherwise there are two text nodes in span after '.text('abc')' in IE.
                        tabSpan.contents().remove().end().text(NBSP);
                        heightSpanHMM = Utils.convertLengthToHmm(tabSpan.height(), 'px');
                        topSpanHMM = Utils.convertLengthToHmm(tabSpan.position().top, 'px');
                        // checkout if the first line indent is active for this special tab stop
                        if (intersect(topFirst, topFirst + heightFirst, topSpanHMM, topSpanHMM + heightSpanHMM) &&
                            (leftHMM + 10) < indentFirstLineTabStopValue) {
                            width = Math.max(0, indentFirstLineTabStopValue - leftHMM);
                        }

                        if (width > 1) {
                            // find index for the first line indent position within the paragraph tab stop array
                            indentFirstLineTabStopIndex = Utils.findLastIndex(paraTabStops, function (tab) {
                                return (indentFirstLineTabStopValue > tab.pos);
                            }) + 1;
                            // create a copy of the paragraph tab stops and add the first indent tab stop at
                            // the correct position to the active paragraph tab stop array
                            indentFirstLineTabStop = { value: 'left', pos: indentFirstLineTabStopValue, fillChar: NBSP, processed: false };
                            activeParaTabStops = paraTabStops.slice();
                            activeParaTabStops.splice(indentFirstLineTabStopIndex, 0, indentFirstLineTabStop);
                            // reset width - the tab stop position must be calculated using all tab stop positions
                            width = 0;
                        } else {
                            // reset text within span
                            tabSpan.contents().remove();
                            tabSpan[0].appendChild(document.createTextNode(''));
                            // use only paragraph tab stops
                            activeParaTabStops = paraTabStops;
                        }
                    } else {
                        // reset text within span
                        tabSpan.contents().remove();
                        tabSpan[0].appendChild(document.createTextNode(''));
                        // use only paragraph tab stops
                        activeParaTabStops = paraTabStops;
                    }

                    // Paragraph tab stops. Only paragraph tab stop can have a fill character and
                    // define a new alignment
                    if (width <= 1) {
                        tabStop = _(activeParaTabStops).find(function (tab) {
                            return ((leftHMM + 10) < tab.pos) && !tab.processed;
                        });
                        if (tabStop) {

                            tabStopType = tabStop.value || 'left';

                            // calculate tab stop size based on tab stop properties
                            if (tabStopType === 'left') {
                                // left bound tab stop
                                width = calcLeftTab();
                            } else if (tabStopType === 'right') {
                                // right bound tab stop
                                width = calcRightTab();
                                // Ignore this tab stop if width is zero. Don't use the default
                                // tab stop which is only active to left bound tab stops!
                                ignore = (width === 0);

                                if (!ignore && _.indexOf(activeParaTabStops, tabStop) === activeParaTabStops.length - 1) {
                                    if (!_.browser.IE) {
                                        paragraph.css('width', 'calc(100% + 5mm)');
                                    }
                                }

                            } else if (tabStopType === 'center') {
                                // right bound tab stop combinded with left bound tab stop
                                width = calcMiddleTab();
                                // Ignore this tab stop if width is zero. Don't use the default
                                // tab stop which is only active to left bound tab stops!
                                ignore = (width === 0);
                            }

                            // insert fill character
                            if (width > 1) {
                                fillChar = _.isString(tabStop.fillChar) ? TAB_FILL_CHARS[tabStop.fillChar] : NBSP;
                                if (fillChar) {
                                    ParagraphStyles.insertTabFillChar(tabSpan, fillChar, width);
                                }
                                // Set processed flag to prevent using the same tab again due to rounding errors
                                // resulting by the browser calculation.
                                tabStop.processed = true;
                            }
                        }
                    }

                    if (!ignore) {
                        // only process default tab stop if tab stop is not set to ignore
                        if (width <= 1) {
                            // tab size calculation based on default tab stop
                            width = Math.max(0, defaultTabStop - (leftHMM % defaultTabStop));
                            width = (width <= 10) ? defaultTabStop : width; // no 0 tab size allowed, check for <= 10 to prevent rounding errors
                            // reset possible fill character
                            ParagraphStyles.insertTabFillChar(tabSpan, NBSP, width);
                        }
                        prevTabEndPos = { right: leftHMM + width, top: topHMM };
                    }
                    // always set a width to the tab div (even an ignored tab div needs to set the size to zero)
                    tabNode.css('width', (width / 100) + 'mm');
                });

                calculateMarginForMixedTab(paragraph, mergedAttributes, mixedTab);

                // reset processed flag for tab stops again
                paraTabStops.forEach(function (tab) {
                    tab.processed = false;
                });
            }
        };

        // public methods -----------------------------------------------------

        /**
         * Updates all tabulators in the passed paragraph node. This method
         * uses both the default tab size and existing paragraph tab stop
         * definitions. Fill characters are also supported.
         *
         * @param {HTMLElement|jQuery} paragraph
         *  The paragraph node whose tabulator nodes will be updated. If this
         *  object is a jQuery collection, uses the first DOM node it contains.
         *
         * @returns {ParagraphStyles}
         *  A reference to this instance.
         */
        this.updateTabStops = function (paragraph) {
            this.implUpdateTabStops($(paragraph), this.getElementAttributes(paragraph));
            return this;
        };

        // initialization -----------------------------------------------------

        // deferred initialization
        docModel.getApp().onInit(function () {
            attributePool = this.getAttributePool();
            charStyles = docModel.getCharacterStyles();
            globalDefTabStop = !attributePool.isSupportedAttribute('paragraph', 'defaultTabSize');
        }, this);

        // register the attribute definitions for the style family
        docModel.registerAttributes('paragraph', DEFINITIONS);

        // register the formatting handlers for DOM elements
        this.registerFormatHandler(updateParagraphFormatting);
        this.registerPreviewHandler(updatePreviewFormatting);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = attributePool = charStyles = null;
        });

    } }); // class ParagraphStyles

    // static methods ---------------------------------------------------------

    /**
     * Fills the passed text span with a sufficient number of the specified
     * fill character.
     *
     * @param {jQuery} spanNode
     *  The span node to be filled. The current CSS formatting of this node is
     *  used to calculate the number of fill characters needed.
     *
     * @param {String} fillChar
     *  The fill character.
     *
     * @param {Number} width
     *  The target width of the tabulator, in 1/100 of millimeters.
     */
    ParagraphStyles.insertTabFillChar = function (spanNode, fillChar, width) {

        // multiplier for a better average calculation
        var multiplier = 15;
        // multiplier fill characters, used to calculate average character width
        var checkString = Utils.repeatString(fillChar, multiplier);
        // average character width, in 1/100 mm
        var charWidth = Utils.convertLengthToHmm(spanNode.contents().remove().end().text(checkString).width(), 'px') / multiplier;
        // number of characters needed to fill the specified width
        var charCount = Math.floor(width / charWidth);
        // the fill character, repeated by the calculated number
        var fillString = (charCount > 0) ? Utils.repeatString(fillChar, charCount) : NBSP;
        // a shortened string, if element is too wide
        var shortFillString = null;

        // insert the fill string into the element
        spanNode.contents().remove().end().text(fillString);
        if (!fillString) { DOM.ensureExistingTextNode(spanNode); }

        // shorten fill string by one character, if element is too wide (e.g. due to rounding errors)
        if ((fillString.length > 1) && (Utils.convertLengthToHmm(spanNode.width(), 'px') >= width)) {
            shortFillString = fillString.slice(0, -1);
            spanNode.contents().remove().end().text(shortFillString);
            if (!shortFillString) { DOM.ensureExistingTextNode(spanNode); }
        }

        // workaround for iOS & Safari mobile - needs a text node directly within
        // the tab div to enable an external keyboard to jump through a tab chains
        // with empty spans
        if (_.device('ios') && _.device('safari')) {
            if (!spanNode[0].nextSibling) {
                spanNode.after(document.createTextNode(NBSP));
            }
        }
    };

    // exports ================================================================

    return ParagraphStyles;

});
