/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/textframework/components/rangemarker/rangemarker', [
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/baseframework/app/appobjectmixin',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/textutils',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position'
], function (TriggerObject, AppObjectMixin, Operations, Utils, DOM, Position) {

    'use strict';

    // class RangeMarker ======================================================

    /**
     * An instance of this class represents the model for all markers in the
     * edited document.
     *
     * @constructor
     *
     * @extends TriggerObject
     * @extends AppObjectMixin
     *
     * @param {TextApplication} app
     *  The application instance.
     */
    function RangeMarker(app) {

        var // self reference
            self = this,
            // an object with all range start markers (jQuery)
            allStartMarkers = {},
            // an object with all range end markers (jQuery)
            allEndMarkers = {},
            // the text model object
            model = null,
            // the overlay node that is used to visualize the ranges
            rangeOverlayNode = null,
            // a marker class to identify highlight elements introduced by range marker nodes
            rangeMarkerClass = 'isRangeMarker',
            // a selector to identify highlight elements introduced by range marker nodes
            rangeMarkerSelector = '.' + rangeMarkerClass,
            // the minimum width of a highlighted range in pixel
            minHighlightRangeWidth = 2,
            // whether the range marker model was refreshed after the loading phase
            rangeMarkerModelRefreshed = false;

        // base constructors --------------------------------------------------

        TriggerObject.call(this, app);
        AppObjectMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Registering one range marker in the range marker collection.
         *
         * @param {HTMLElement|jQuery} marker
         *  One range marker node.
         *
         * @param {String} id
         *  The unique id of the range marker node.
         *
         * @param {String} [target]
         *  The target, where the range marker is located.
         */
        function addIntoRangeMarkerCollection(marker, id, target) {

            var // the comment node in jQuery format
                $marker = $(marker),
                // whether the target is inside header or footer
                // -> in this case it is useless to save the node itself, but saving
                //    the target allows to find the range with specific id fast
                isMarginalId = target && model.getPageLayout().isIdOfMarginalNode(target);

            // adding the node also into the type specific collector
            if (DOM.isRangeMarkerStartNode(marker)) {
                allStartMarkers[id] = isMarginalId ? target : $marker;
            } else if (DOM.isRangeMarkerEndNode(marker)) {
                allEndMarkers[id] = isMarginalId ? target : $marker;
            }
        }

        /**
         * Removing one range marker with specified id from range marker collections (the model).
         *
         * @param {HTMLElement|jQuery} marker
         *  One range marker node.
         *
         * @param {String} id
         *  The unique id of the range marker node.
         */
        function removeFromRangeMarkerCollection(marker, id) {

            // adding the node also into the type specific collector
            if (DOM.isRangeMarkerStartNode(marker)) {
                delete allStartMarkers[id];
            } else if (DOM.isRangeMarkerEndNode(marker)) {
                delete allEndMarkers[id];
            }
        }

        /**
         * Generating or simply returning the overlay node, that is used to visualize the ranges.
         *
         * @returns {Boolean}
         *  The overlay node, that contains the nodes to visualize the ranges.
         */
        function getOrCreateRangeOverlayNode() {

            var // the page content node
                pageContentNode = null;

            // is there is already a comment overlay node, return it
            if (rangeOverlayNode && rangeOverlayNode.length > 0) { return rangeOverlayNode; }

            // create a new comment layer node
            pageContentNode = DOM.getPageContentNode(model.getNode());

            // Adding the text drawing layer behind an optional footer wrapper
            if (DOM.isFooterWrapper(pageContentNode.next())) { pageContentNode = pageContentNode.next(); }

            // creating the (global) overlay node
            rangeOverlayNode = $('<div>').addClass(DOM.RANGEMARKEROVERLAYNODE_CLASS).attr('contenteditable', false);

            // inserting the range layer node in the page
            pageContentNode.after(rangeOverlayNode);

            return rangeOverlayNode;
        }

        /**
         * Finding in a header or footer specified by the target the range marker
         * with the specified id of type 'start' or 'end'.
         *
         * @param {String} id
         *  The id string.
         *
         * @param {String} target
         *  The target string to identify the header or footer node
         *
         * @param {String} selector
         *  The selector string to identify range start marker or range end marker
         *
         * @param {Number} [index]
         *  Cardinal number of target node in the document from top to bottom. If passed,
         *  forces to return node with that index in document, from top to bottom.
         *
         * @returns {jQuery|null}
         *  The range marker node with the specified id and type, or null, if no such
         *  range marker exists.
         */
        function getMarginalRangeMarker(id, target, selector, index) {

            var // the specified header or footer node
                marginalNode = null,
                // the searched range marker with the specified id
                marker = null;

            marginalNode = model.getRootNode(target, index, { allowMarginalUpdate: false }); // option required because of 40199 (TODO: Needs to be removed again)

            if (marginalNode) {
                marker = _.find(marginalNode.find(selector), function (marker) { return DOM.getRangeMarkerId(marker) === id; });
            }

            return marker ? $(marker) : null;
        }

        /**
         * After load from local storage or fast load the collectors for the range markers need to be filled.
         */
        function refreshRangeMarker() {

            var // the page content node of the document
                pageContentNode = DOM.getPageContentNode(model.getNode()),
                // all range marker nodes in the document
                allRangeMarkerNodes = pageContentNode.find(DOM.RANGEMARKERNODE_SELECTOR),
                // a collector for all marginal template nodes
                allMargins = model.getPageLayout().getHeaderFooterPlaceHolder().children().add(DOM.getCommentLayerNode(model.getNode())),
                // the target string node
                target = '';

            // helper function to add a collection of range marker nodes into the model
            function addMarginalNodesIntoCollection(collection) {
                _.each(collection, function (oneRangeMarker) {
                    addIntoRangeMarkerCollection(oneRangeMarker, DOM.getRangeMarkerId(oneRangeMarker), target);
                });
            }

            // reset model
            allStartMarkers = {};
            allEndMarkers = {};

            // 1. Search in page content node
            // 2. Search in header/footer template node (-> overwriting previous nodes by target strings)

            // adding all range marker nodes into the collector (not those from header or footer)
            _.each(allRangeMarkerNodes, function (oneRangeMarker) {
                if (!DOM.isMarginalNode($(oneRangeMarker).parent())) {
                    addIntoRangeMarkerCollection(oneRangeMarker, DOM.getRangeMarkerId(oneRangeMarker));
                }
            });

            // adding the range markers that are located inside header or footer (using the template node)
            _.each(allMargins, function (margin) {

                var // a collector for all range marker nodes inside one margin
                    allMarginRangeMarkerNodes = $(margin).find(DOM.RANGEMARKERNODE_SELECTOR);

                if (allMarginRangeMarkerNodes.length > 0) {
                    target = DOM.getTargetContainerId(margin);
                    addMarginalNodesIntoCollection(allMarginRangeMarkerNodes);
                }
            });

            // during loading, it is necessary, that other components (like comments) know,
            // the the range marker model is update-to-date
            rangeMarkerModelRefreshed = true;
        }

        /**
         * Returns, whether the range marker model was refreshed at least once. This
         * is necessary after loading from local storage or fast load.
         *
         * @returns {Boolean}
         *  Whether the range marker model was refreshed at least once.
         */
        function isRangeMarkerModelRefreshed() {
            return rangeMarkerModelRefreshed;
        }

        // methods ------------------------------------------------------------

        /**
         * Whether the document contains range markers in the main document.
         *
         * @returns {Boolean}
         *  Whether the document contains at least one range marker.
         */
        this.isEmpty = function () {
            return _.isEmpty(allStartMarkers) && _.isEmpty(allEndMarkers);
        };

        /**
         * Check, whether the specified id is a valid id of a range start marker.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id of a range start marker node.
         */
        this.isStartMarkerId = function (id) {
            return self.getStartMarker(id) !== null;
        };

        /**
         * Check, whether the specified id is a valid id of a range end marker.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id string is a valid id of a range end marker node.
         */
        this.isEndMarkerId = function (id) {
            return self.getEndMarker(id) !== null;
        };

        /**
         * Check, whether the specified id is a valid id for a range marker.
         *
         * @param {String} id
         *  The id string.
         *
         * @returns {Boolean}
         *  Whether the specified id is a valid id of a range node.
         */
        this.isRangeMarkerId = function (id) {
            return self.isStartMarkerId(id) || self.isEndMarkerId(id);
        };

        /**
         * Whether there are highlighted ranges.
         *
         * @returns {Boolean}
         *  Whether there is at least one highlighted range.
         */
        this.isRangeHighlighted = function () {
            return rangeOverlayNode && rangeOverlayNode.children.length > 0;
        };

        /**
         * Provides a jQuerified range start marker node.
         *
         * @returns {jQuery|null}
         *  The start marker with the specified id, or null, if no such start marker exists.
         */
        this.getStartMarker = function (id) {

            var // the value saved under the specified id
                marker = allStartMarkers[id] || null;

            return _.isString(marker) ? getMarginalRangeMarker(id, marker, DOM.RANGEMARKER_STARTTYPE_SELECTOR) : marker;
        };

        /**
         * Provides a jQuerified range end marker node.
         *
         * @returns {jQuery|null}
         *  The end marker with the specified id, or null, if no such end marker exists.
         */
        this.getEndMarker = function (id) {

            var // the value saved under the specified id
                marker = allEndMarkers[id] || null;

            return _.isString(marker) ? getMarginalRangeMarker(id, marker, DOM.RANGEMARKER_ENDTYPE_SELECTOR) : marker;
        };

        /**
         * Whether the specified id is already used by a range marker node.
         *
         * @returns {Boolean}
         *  Returns whether the specified id is already used by a range marker node.
         */
        this.isUsedRangeMarkerId = function (id) {
            return (self.getStartMarker(id) !== null) || (self.getEndMarker(id) !== null);
        };

        /**
         * Provides the range overlay node, that contains the nodes, that are used to
         * visualize the ranges.
         *
         * @returns {jQuery|null}
         *  The range overlay node, that can be used to visualize the ranges.
         */
        this.getRangeOverlayNode = function () {
            return rangeOverlayNode;
        };

        /**
         * In the loading phase, other components like comments or complex fields,
         * need to rely on an up-to-date model for the range markers. Therefore
         * these other components can trigger an update of the model by calling
         * this function.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.onlyOnce=false]
         *      If set to true, the model will only be refreshed, if it was not
         *      refreshed before.
         */
        this.refreshModel = function (options) {

            var // whether the model needs to be refreshed only once (if it was not refreshed before)
                onlyOnce = Utils.getBooleanOption(options, 'onlyOnce', false);

            if (!(onlyOnce && isRangeMarkerModelRefreshed())) {
                refreshRangeMarker();
            }
        };

        /**
         * Removing one range marker with specified id from range marker collections.
         *
         * @param {String} id
         *  The unique id of the range marker node.
         */
        this.removeRangeMarker = function (marker) {

            var // the unique id of the range node
                rangeId = DOM.getRangeMarkerId(marker);

            if (_.isString(rangeId)) {
                removeFromRangeMarkerCollection(marker, rangeId);
            }
        };

        /**
         * Handler for inserting a range marker node into the document DOM. These ranges can be
         * used for several features like comments, complex fields, ...
         *
         * @param {Number[]} start
         *  The logical start position for the new range marker.
         *
         * @param {String} id
         *  The unique id of the range marker. This can be used to identify the
         *  corresponding node, that requires this range. This can be a comment
         *  node for example.
         *
         * @param {String} type
         *  The type of the range marker.
         *
         * @param {String} position
         *  The position of the range marker. Currently supported are 'start'
         *  and 'end'.
         *
         * @param {String} [target]
         *  The target corresponding to the specified logical start position.
         *
         * @returns {Boolean}
         *  Whether the range marker has been inserted successfully.
         */
        this.insertRangeHandler = function (start, id, type, position, target) {

            var // whether this is a range marker for the start or the end of a range
                rangeMarkerTypeClass = (position === 'end') ? DOM.RANGEMARKER_ENDTYPE_CLASS : DOM.RANGEMARKER_STARTTYPE_CLASS,
                // the range marker element
                rangeMarkerNode = $('<div>', { contenteditable: false }).addClass('inline ' + DOM.RANGEMARKERNODE_CLASS + ' ' + rangeMarkerTypeClass),
                // the text span needed for inserting the range marker node
                textSpan = model.prepareTextSpanForInsertion(start, null, target),
                // if range belongs to complex field
                field = null,
                // if range is at position start or end
                rangeStart, rangeEnd,
                // field instruction
                instruction;

            if (!textSpan) { return false; }

            // if is range for type field, get id from stack
            if (type === 'field') {
                rangeEnd = position === 'end';
                rangeStart = !rangeEnd;
                id = model.getFieldManager().getComplexFieldIdFromStack(rangeStart, rangeEnd);
            }

            // No inheritance of change track to following empty text span node (this would not be resolved during change track resolving)
            if (DOM.isChangeTrackNode(textSpan) && textSpan.nextSibling && DOM.isEmptySpan(textSpan.nextSibling)) {
                model.getCharacterStyles().setElementAttributes(textSpan.nextSibling, { changes: { inserted: null, removed: null, modified: null } });
            }

            // saving also type and id at the range marker node
            rangeMarkerNode.attr('data-range-type', type);
            rangeMarkerNode.attr('data-range-id', id);

            // splitting text span to insert range marker node
            rangeMarkerNode.insertAfter(textSpan);

            // adding the range marker node in the collection of all range markers
            // -> this also needs to be done after loading document from fast load or using local storage
            addIntoRangeMarkerCollection(rangeMarkerNode, id, target);

            // complex fields highlight
            if (position === 'end' && type === 'field') {
                model.getFieldManager().markFieldForHighlighting(id);
                // converts page fields in header&footer to special fields
                if (target && model.getPageLayout().isIdOfMarginalNode(target)) {
                    field = model.getFieldManager().getComplexField(id);
                    instruction = DOM.getComplexFieldInstruction(field);
                    if ((/PAGE/i).test(instruction) && !DOM.isSpecialField(field)) {
                        model.getFieldManager().convertToSpecialField(id, target, instruction);
                    }
                }
            }

            return true;
        };

        /**
         * After splitting a paragraph, it is necessary, that all range markers in the cloned
         * 'new' paragraph are updated in the collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.updateRangeMarkerCollector = function (para) {

            var // whether each single node needs to be checked
                checkMarginal = false,
                // all range markers inside the paragraph
                allMarkers = null;

            // not necessary for paragraphs in header or footer -> only the target are stored, not the nodes
            if (DOM.isMarginalNode(para)) { return; }

            // if this is not a paragraph, each single node need to be checked
            checkMarginal = !DOM.isParagraphNode(para);

            // finding all markers inside the specified node
            allMarkers = $(para).find(DOM.RANGEMARKERNODE_SELECTOR);

            // update  the marker nodes in the collection objects
            _.each(allMarkers, function (oneMarker) {
                if (!checkMarginal || !DOM.isMarginalNode(oneMarker)) {
                    // simply overwriting the old markers with the new values
                    addIntoRangeMarkerCollection(oneMarker, DOM.getRangeMarkerId(oneMarker));
                }
            });
        };

        /**
         * After deleting a complete paragraph, it is necessary, that all range markers
         * in the deleted paragraph are also removed from the model collectors.
         *
         * @param {Node|jQuery} paragraph
         *  The paragraph node.
         */
        this.removeAllInsertedRangeMarker = function (para) {

            var // all range markers inside the paragraph
                allMarkers = $(para).find(DOM.RANGEMARKERNODE_SELECTOR);

            // update  the marker nodes in the collection objects
            _.each(allMarkers, function (marker) {

                var // the id of the marker node
                    rangeId = DOM.getRangeMarkerId(marker);

                if (_.isString(rangeId)) {
                    removeFromRangeMarkerCollection(marker, rangeId);
                }
            });
        };

        /**
         * Highlighting the range with the specified unique id. It is required, that there is
         * a start marker and an end marker for this range. Otherwise a highlighting is not
         * possible and this function returns 'false'.
         * The type of highlighting can be specified optionally by assigning specified
         * class names to the generated 'div'-elements.
         *
         * @param {String} id
         *  The unique id of the range, that will be highlighted. To be highlighted it is
         *  necessary that there is an existing start range marker and end range marker
         *  corresponding to the specified id.
         *
         * @param {String} [highlightClassNames]
         *  One or more space-separated classes to be added to the class attribute of the
         *  generated 'div'-elements, that are used to highlight the range.
         *
         * @returns {Boolean}
         *  Whether the range with the specified id was successfully highlighted.
         */
        this.highLightRange = function (id, highlightClassNames) {

            var // the range marker start node
                startNode = self.getStartMarker(id),
                // the range marker end node
                endNode = self.getEndMarker(id),
                // the current zoom factor
                zoomFactor = app.getView().getZoomFactor() * 100,
                // the parent nodes of the start and end node
                startParentNode = null, endParentNode = null,
                // the pixel positions of the range marker nodes and its parents
                startPos = null, endPos = null, startParentPos = null, endParentPos = null,
                // whether this is a multi line range
                isMultiLine = false,
                // the page node
                pageNode = model.getNode(),
                // the overlay node for visualizing the range
                overlayNode = null,
                // the transparent nodes for visualizing the commented range
                headOverlay = null, bodyOverlay = null, tailOverlay = null,
                // the height of start line and end line in pixel
                startLineHeight = 0, endLineHeight = 0,
                // the nodes before start node and end node
                startPrevNode = null, endPrevNode = null,
                // the widths of the nodes before the start node and the end node
                startPrevNodeWidth = 0, endPrevNodeWidth = 0,
                // the width and height of the main overlay node in pixel
                bodyWidth = 0, bodyHeight = 0,
                // the svg name space string, used only for IE 10
                svgNamespace = null,
                // whether the range starts or ends in a list
                isListRange = false,
                // the position of the upper left corner of the highlighted range
                upperLeftCorner = null,
                // the classes for the new highlight elements (adding marker for ranges)
                classNames = highlightClassNames ? highlightClassNames + ' ' + rangeMarkerClass : rangeMarkerClass,
                // the height of the start node and the end node in pixel
                startNodeHeight = 0, endNodeHeight = 0,
                // the span following the start node
                textSpan = null,
                // a correction value that handles the font-size
                correction = null;

            // 1. in ODF the comment place holder node can be used as start node
            // 2. if no start node and no end node could be found, use the comment node itself (if possible)
            if (!startNode && (app.isODF() || !endNode) && model.getCommentLayer().getCommentRootNode(id)) {
                startNode = DOM.getCommentPlaceHolderNode(model.getCommentLayer().getCommentRootNode(id));
                if (startNode) { startNode = $(startNode); }
            }

            // drawing the comment range, if start and end range markers are valid
            if (startNode && endNode) {

                overlayNode = getOrCreateRangeOverlayNode();

                startPos = Position.getPixelPositionToRootNodeOffset(pageNode, startNode, zoomFactor);
                endPos = Position.getPixelPositionToRootNodeOffset(pageNode, endNode, zoomFactor);

                startNodeHeight = startNode.height();
                endNodeHeight = endNode.height();

                // comparing the bottom positions, to check, if this is a multi line range
                isMultiLine = ((startPos.y + startNodeHeight) !== (endPos.y + endNodeHeight));

                // adding simple correction factors for spans with different font sizes
                textSpan = Utils.findNextSiblingNode(startNode, 'span');
                correction = textSpan ? Utils.convertCssLength($(textSpan).css('font-size'), 'px', 1) - Utils.convertCssLength($(startNode).css('font-size'), 'px', 1) : 0;

                startPos.y -= correction;
                endPos.y -= correction;

                // setting the line height values for start and end node
                startLineHeight = startNodeHeight + correction;
                endLineHeight = endNodeHeight + correction;

                if (isMultiLine) {

                    // creating three transparent overlay nodes
                    if (_.browser.IE === 10) {
                        svgNamespace = 'http://www.w3.org/2000/svg';
                        headOverlay = $(document.createElementNS(svgNamespace, 'svg'));
                        bodyOverlay = $(document.createElementNS(svgNamespace, 'svg'));
                        tailOverlay = $(document.createElementNS(svgNamespace, 'svg'));
                    } else {
                        headOverlay = $('<div>');
                        bodyOverlay = $('<div>');
                        tailOverlay = $('<div>');
                    }

                    // TODO: This needs to be tested for IE 10
                    headOverlay.addClass(classNames).attr('data-range-id', id);
                    bodyOverlay.addClass(classNames).attr('data-range-id', id);
                    tailOverlay.addClass(classNames).attr('data-range-id', id);

                    // the nodes directly before the range nodes
                    startPrevNode = startNode.prev();
                    endPrevNode = endNode.prev();

                    // checking if the range marker are inside a list item
                    isListRange = DOM.isListLabelNode(startPrevNode) || DOM.isListLabelNode(endPrevNode);

                    if (isListRange) {
                        // the widths of the nodes before the range nodes
                        startPrevNodeWidth = startPrevNode.width();
                        endPrevNodeWidth = endPrevNode.width();
                    }

                    // the parent nodes of the range nodes
                    startParentNode = startNode.parent();
                    endParentNode = endNode.parent();

                    // the width of the body overlay node is the maximum of the parents widths
                    // -> using outerWidth() instead of width() to include the padding (51588)
                    bodyWidth = Math.max(startParentNode.outerWidth(), endParentNode.outerWidth());

                    // start and end node are empty lines (paragraphs)
                    if (!bodyWidth) { bodyWidth = pageNode.width(); }

                    // setting the parent positions (the paragraphs)
                    startParentPos = Position.getPixelPositionToRootNodeOffset(pageNode, startParentNode, zoomFactor);
                    endParentPos = Position.getPixelPositionToRootNodeOffset(pageNode, endParentNode, zoomFactor);

                    headOverlay.css({
                        top: startPos.y,
                        left: startPos.x,
                        width: isListRange ? startParentPos.x + startParentNode.width() - startPos.x + startPrevNodeWidth : startParentPos.x  + startParentNode.outerWidth() - startPos.x,
                        height: startLineHeight
                    });

                    bodyOverlay.css({
                        top: startPos.y + startLineHeight,
                        left: (isListRange ? DOM.getPageContentNode(pageNode).offset().left - pageNode.offset().left : Math.min(startParentPos.x, endParentPos.x)),
                        width: isListRange ? pageNode.width() : bodyWidth,
                        height: endPos.y - startPos.y - startLineHeight
                    });

                    tailOverlay.css({
                        top: endPos.y,
                        left: isListRange ? endPos.x - endPrevNodeWidth : endParentPos.x,
                        width: endPos.x - endParentPos.x,
                        height: endLineHeight
                    });

                    overlayNode.append(headOverlay, bodyOverlay, tailOverlay);
                    upperLeftCorner = startPos;

                } else {

                    // adding one transparent overlay node is sufficient
                    bodyWidth = Math.max(endPos.x - startPos.x, minHighlightRangeWidth); // comparing with minimum width in pixel
                    bodyHeight = startLineHeight > 0 ? startLineHeight : Utils.convertCssLength(startNode.next().css('line-height'), 'px', 1); // using line-height of following span

                    bodyOverlay = $('<div>').addClass(classNames).attr('data-range-id', id).css({ top: startPos.y, left: startPos.x, width: bodyWidth, height: bodyHeight });
                    overlayNode.append(bodyOverlay);
                    upperLeftCorner = startPos;
                }
            } else if (startNode || endNode) {

                overlayNode = getOrCreateRangeOverlayNode();

                // only one range marker (or comment node could be found)
                startNode = startNode || endNode;

                startPos = Position.getPixelPositionToRootNodeOffset(pageNode, startNode, zoomFactor);

                // adding one transparent overlay node is sufficient
                bodyOverlay = $('<div>').addClass(classNames).attr('data-range-id', id).css({ top: startPos.y, left: startPos.x, width: minHighlightRangeWidth, height: startNode.height() });
                overlayNode.append(bodyOverlay);
                upperLeftCorner = startPos;
            }

            return upperLeftCorner;
        };

        /**
         * Removing the highlighting of one or all ranges.
         *
         * @param {String} [id]
         *  The optional unique id of the range, whose highlighting shall be removed. If not
         *  specified, all highlighting of ranges is removed.
         */
        this.removeHighLightRange = function (id) {

            var // the overlay node
                overlayNode = getOrCreateRangeOverlayNode(),
                // a selector to remove only specified visualized ranges
                idSelector = null;

            if (id) {
                idSelector = '[data-range-id=' + id + ']';
                overlayNode.children(idSelector).remove();
            } else {
                overlayNode.empty();
            }

        };

        /**
         * Removing the highlight nodes specified by a class name.
         *
         * @param {String} className
         *  The class name that is used to specify the nodes to be removed.
         */
        this.removeHighLightRangeByClass = function (className) {

            var // the overlay node
                overlayNode = getOrCreateRangeOverlayNode(),
                // the selector
                classSelector = '.' + className;

            overlayNode.children(classSelector).remove();
        };

        /**
         * Updating the high lighting of a currently highlighted range. This can
         * happen, if an external operation modifies the document in that way, that
         * a repaint of the range highlighting is necessary (or after undo/redo).
         */
        this.updateHighlightedRanges = function () {

            // the document was modified (by an external operation)
            // -> highlighted ranges need to be updated.

            // the document was modified while a comment range is visible (caused by hover)
            // -> highlighted ranges need to be updated.

            var // a collector for the children in the range overlay node
                allOverlayChildren = rangeOverlayNode && rangeOverlayNode.children(),
                // all IDs
                allIDs = null,
                // the node, whose change triggered this update
                currentNode = null,
                // the vertical offset of the current node (if it is not defined, update all range hightlights
                currentNodePos = 0,
                // the jQuery object received from offset function
                offset = null;

            if (allOverlayChildren && allOverlayChildren.length > 0) {

                allIDs = {};
                currentNode = model.getCurrentProcessingNode();

                if (currentNode) {
                    offset = $(currentNode).offset();
                    if (offset) { currentNodePos = offset.top; }
                }

                _.each(allOverlayChildren, function (child) {

                    var id = 0;

                    if ($(child).is(rangeMarkerSelector)) { // only collecting
                        id = DOM.getRangeMarkerId(child);
                        allIDs[id] = child; // collecting all IDs
                    }
                });

                // iterating over all keys
                _.each(_.keys(allIDs), function (id) {

                    var // the end marker node with the specific id
                        endMarkerNode = self.getEndMarker(id),
                        // the vertical pixel position of the end marker node
                        endMarkerPos = (endMarkerNode && endMarkerNode.offset().top) || -1,
                        // the classes at the element
                        allClassString = '',
                        // the start position of the connection line
                        startPos = null;

                    // comparing the vertical pixel positions of end marker node and current node
                    if (endMarkerPos < 0 || endMarkerPos >= currentNodePos) {
                        allClassString = $(allIDs[id]).attr('class');  // creating string with all classes
                        self.removeHighLightRange(id);  // removing all elements with corresponding id
                        startPos = self.highLightRange(id, allClassString); // reusing the classes

                        // checking if this is a comment id
                        // -> then the connection line needs to be redrawn
                        model.getCommentLayer().updateCommentLineConnection(id, startPos, allOverlayChildren);
                    }
                });
            }

        };

        /**
         * Handling delete operations, that contain only a part of an existing range. In this case additional
         * operations need to be generated. These operations can be used to insert or delete the range nodes.
         * This can be used, to reduce the width of the range, or to clean up nodes, if, for example, a comment
         * node is removed. The behavior is dependent from the file type (OOXML <-> ODT).
         * This function is called from the textModel from deleteSelected, deleteRows or deleteColumns.
         * In these functions, all range marker nodes from the components, that will be deleted, must be
         * collected. Then affected start marker without end marker or affected end marker without start
         * marker can be detected.
         * In the OOXML case a deleted start marker can be inserted at a following logical position, so that
         * the width of the range is reduced. A deleted end marker requires, that the start marker is also
         * deleted, so that there are no superfluous range markers inside the document.
         * TODO: Handling for ODT.
         *
         * @param {jQuery} allRangeMarkerNodes
         *  The sorted jQuery list of all collected range marker nodes, that are in the range, that will be
         *  deleted.
         *
         * @param {Object} generator
         *  The operations generator, that already contains the delete operations for the selected area. These
         *  operations must already be in the correct order. This means, that the opertions with the 'largest'
         *  logical position is the first operation saved in the generator.
         *
         * @param {Number[]} [insertPosition]
         *  A logical position at which the start ranges will be inserted. If not defined, the corresponding
         *  end range markers will be removed to avoid superfluous range markers in the code.
         */
        this.handlePartlyDeletedRanges = function (allRangeMarkerNodes, generator, insertPosition) {

            var // the ids of all start range markers
                allLocalStartMarkers = {},
                // the ids of all end range markers
                allLocalEndMarkers = {},
                // the ids of all comment nodes
                allLocalComments = {},
                // all affected IDs
                allIds = [],
                // the ids of the range start markers, that are inside the selection and whose end markers are not inside the selection
                allStartPartIds = [],
                // the ids of the range end markers, that are inside the selection and whose start markers are not inside the selection
                allEndPartIds = [],
                // ODF comments start nodes (the comment nodes itself)
                allODFCommentStartIds = [],
                // ODF comments end nodes (range end nodes, without start node)
                allODFCommentEndIds = [],
                // a temporary helper for logical positions
                localPos = null,
                // the position of the start markers, that need to be deleted additionally (before the selected range)
                allPrePositions = [],
                // the position of the end markers, that need to be deleted additionally (behind the selected range)
                allPostPositions = [],
                // the new start position that need to be used after deleteSelected. The old position can be invalid, because of
                // additional delete or insert operations
                newStartPosition = _.clone(insertPosition);

            // helper function, that adds additionally required operations to the existing generator.
            // In this case a simple process can be used, that creates only delete operations from
            // the specified logical positions. In the future this might become more complicated, if
            // there will be another type of operation, that is added to the existing generator.
            function addAdditionalOperations(allPositions, options) {

                var // whether the parameter 'localGenerator' is the generator for pre or post operations
                    postPositions = Utils.getBooleanOption(options, 'postPositions', true);

                // an ordering of operations is necessary, if the array contains more than one value
                if (allPositions.length > 1) {
                    allPositions.sort(Utils.compareNumberArrays);
                    if (!postPositions) { allPositions.reverse(); } // positions before the selection, can be added directly, but in reverted order
                }

                // delete operations need to be executed before the already defined delete operations,
                // because the nodes are behind the selected range
                if (postPositions) { generator.reverseOperations(); }  // positions behind the selection must be added to the beginning of the generated operations
                _.each(allPositions, function (position) {
                    generator.generateOperation(Operations.DELETE, { start: position });
                });
                if (postPositions) { generator.reverseOperations(); }

            }

            if (!allRangeMarkerNodes || allRangeMarkerNodes.length === 0 || !generator) { return; }

            // searching for start marker, end marker and comments -> comparing the id
            _.each(allRangeMarkerNodes, function (oneMarker) {

                var currentId = null;

                if (DOM.isRangeMarkerStartNode(oneMarker)) {
                    currentId = DOM.getRangeMarkerId(oneMarker);
                    allLocalStartMarkers[currentId] = oneMarker;
                } else if (DOM.isRangeMarkerEndNode(oneMarker)) {
                    currentId = DOM.getRangeMarkerId(oneMarker);
                    allLocalEndMarkers[currentId] = oneMarker;
                } else if (app.isODF() && DOM.isCommentPlaceHolderNode(oneMarker)) {
                    currentId = DOM.getTargetContainerId(oneMarker);
                    allLocalComments[currentId] = oneMarker;  // only filled for ODF documents
                }

                if (currentId && !_.contains(allIds, currentId)) { allIds.push(currentId);  } // using array, to keep correct order
            });

            // Finding ranges, that are only partly affected (requires special handling for comments in ODF)
            _.each(allIds, function (id) {

                if (app.isODF()) {

                    if (allLocalStartMarkers[id] && !allLocalEndMarkers[id]) {  // classical range start marker (no comment)
                        allStartPartIds.push(id);
                    } else if (allLocalEndMarkers[id]) {
                        // is this an end marker for a comment or for a 'classic' range
                        if (model.getCommentLayer().getCommentRootNode(id)) {
                            if (!allLocalComments[id]) {  // the comment itself is not in the current selection
                                allODFCommentEndIds.push(id);
                            }
                        } else if (!allLocalStartMarkers[id]) {
                            allEndPartIds.push(id);  // classical range end marker (no comment)
                        }
                    } else if (allLocalComments[id] && !allLocalEndMarkers[id]) {
                        allODFCommentStartIds.push(id); // a comment node without the corresponding end range marker
                    }

                } else {

                    if (allLocalStartMarkers[id] && !allLocalEndMarkers[id]) {
                        allStartPartIds.push(id);
                    } else if (allLocalEndMarkers[id] && !allLocalStartMarkers[id]) {
                        allEndPartIds.push(id);
                    }
                }

            });

            // Finally generating new operations.
            // special handling for comments in ODF, where the start range is marked by the comment node itself
            // -> this is handled by the two specific collectors 'allODFCommentStartIds' and 'allODFCommentEndIds'.
            if (app.isODF()) {

                // Comments in ODF:
                // Deleting only the start comment node -> additionally delete the end range node (comment is removed completely)
                // Deleting only the end range node -> insert the end range node before the delete selection

                // only the start node was removed (the comment node itself) -> deleting the comment
                // -> in case of existing child comments, these need to be removed, too (only comment nodes without range)
                if (allODFCommentStartIds.length > 0) {
                    _.each(allODFCommentStartIds, function (id) {
                        var // the range end marker for the comment range
                            endMarker = self.getEndMarker(id),
                            // the place holder node that will be deleted
                            placeHolderNode = allLocalComments[id],
                            // the comment node in the comment layer
                            commentNode = placeHolderNode && DOM.getCommentPlaceHolderNode(placeHolderNode),
                            // the comment thread node in the comment layer
                            commentThread = null;

                        // deleting the end marker
                        if (endMarker) {
                            localPos = Position.getOxoPosition(model.getCurrentRootNode(), endMarker);
                            allPostPositions.push(_.clone(localPos));
                        }

                        // deleting the comment children in the same thread
                        if (commentNode && DOM.isParentCommentNode(commentNode)) {
                            commentThread = $(commentNode).parent();
                            _.each(commentThread.children(), function (oneNode) {
                                var onePlaceHolderNode = null;
                                if (DOM.isChildCommentNode(oneNode)) {
                                    onePlaceHolderNode = DOM.getCommentPlaceHolderNode(oneNode);
                                    if (onePlaceHolderNode) { allPostPositions.push(Position.getOxoPosition(model.getCurrentRootNode(), onePlaceHolderNode)); }
                                }
                            });
                        }
                    });
                }

                // only the end node of a comment range was removed -> reducing the width of the comment, if the
                // new position at insertPosition was defined
                if (allODFCommentEndIds.length > 0) {

                    if (insertPosition) {
                        allODFCommentEndIds.reverse(); // -> simplifies the insertion at the start position
                        _.each(allODFCommentEndIds, function (id) {
                            var endMarker = allLocalEndMarkers[id],
                                type = DOM.getRangeMarkerType(endMarker);

                            // this operations can simply be added to the already filled operations generator, because the affected positions are
                            // directly behind the delete range and the insertions happens directly after the delete operations. So it is not
                            // possible, that the logical positions become invalid.
                            generator.generateOperation(Operations.RANGE_INSERT, { start: insertPosition, id: id, type: type, position: 'end' });
                        });
                    }
                }
            }

            // Classical behavior for range marker nodes (not valid for comments in ODF)
            // Deleting only the end range node -> additionally delete the start range node
            // Deleting only the start range node -> insert the start range node behind the delete selection
            // -> the comment node itself can be ignored

            // if only the start marker is removed, add operations to insert the start marker at a new position
            if (allStartPartIds.length > 0) {
                if (insertPosition) {
                    allStartPartIds.reverse(); // -> simplifies the insertion at the start position
                    _.each(allStartPartIds, function (id) {
                        var startMarker = allLocalStartMarkers[id],
                            type = DOM.getRangeMarkerType(startMarker);

                        // this operations can simply be added to the already filled operations generator, because the affected positions are
                        // directly behind the delete range and the insertions happens directly after the delete operations. So it is not
                        // possible, that the logical positions become invalid.
                        generator.generateOperation(Operations.RANGE_INSERT, { start: insertPosition, id: id, type: type, position: 'start' });
                    });
                } else {
                    // failed to determine new position for start range markers (for example after deleteColumn)
                    // -> deleting also the end range marker to avoid superfluous range markers in the code
                    // -> this is also the behavior of ODF comments
                    _.each(allStartPartIds, function (id) {
                        var endMarker = self.getEndMarker(id);
                        if (endMarker) {
                            localPos = Position.getOxoPosition(model.getCurrentRootNode(), endMarker);
                            allPostPositions.push(_.clone(localPos));
                        }
                    });
                }
            }

            // if only the end marker is removed, add operations to delete the start marker, too
            // -> in this case the 'allPrePositions' needs to be filled, because the logical positions are before the selected and deleted area
            if (allEndPartIds.length > 0) {
                _.each(allEndPartIds, function (id) {
                    var startMarker = self.getStartMarker(id);
                    if (startMarker) {
                        localPos = Position.getOxoPosition(model.getCurrentRootNode(), startMarker);
                        allPrePositions.push(_.clone(localPos));
                        // if the position is inside the same paragraph as the current start position,
                        // this position need to be reduced. So after a call of deleteSelected(), there is a
                        // valid position for a directly following input.
                        if (Position.hasSameParentComponent(localPos, newStartPosition)) { newStartPosition = Position.decreaseLastIndex(newStartPosition); }
                    }
                });
            }

            // checking, if additional operations from 'preGenerator' or 'postGenerator' need to be added to the existing operation generator.
            // In this case, it is necessary to sort the logical positions and to add the operations to the end of the operations list
            // (from allPrePositions) and to add them to the beginning of the operations list (from allPostPositions).
            if (allPrePositions.length > 0) { addAdditionalOperations(allPrePositions, { postPositions: false }); }
            if (allPostPositions.length > 0) { addAdditionalOperations(allPostPositions, { postPositions: true }); }

            return newStartPosition;
        };

        // initialization -----------------------------------------------------

        app.onInit(function () {

            model = app.getModel();

            self.waitForImportSuccess(function () {
                if ((!rangeMarkerModelRefreshed) && (model.isLocalStorageImport() || model.isFastLoadImport())) { refreshRangeMarker(); }
            });

            model.on('update:absoluteElements', self.updateHighlightedRanges.bind(self));

            model.on('document:reloaded', refreshRangeMarker);
        });

    } // class RangeMarker

    // export =================================================================

    // derive this class from class TriggerObject
    return TriggerObject.extend({ constructor: RangeMarker });

});
