/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Carsten Driesner <carsten.driesner@open-xchange.com>
 */

define('io.ox/office/textframework/components/hyperlink/hyperlink', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/dialogs',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/hyperlinkutils',
    'io.ox/office/textframework/utils/operations',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/utils/position'
], function (Utils, KeyCodes, Dialogs, AttributeUtils, HyperlinkUtils, Operations, DOM, Position) {

    'use strict';

    // static private functions ===============================================

    function createResultFromHyperlinkSelection(position, hyperlinkSelection) {

        var result = null,
            start = _.clone(position),
            end = _.clone(position);

        // create result with correct Position objects
        start[start.length - 1] = hyperlinkSelection.start;
        end[end.length - 1] = hyperlinkSelection.end;
        result = { start: start, end: end, text: hyperlinkSelection.text, url: null };

        return result;
    }

    // static class Hyperlink =================================================

    /**
     * Provides static helper methods for manipulation and calculation
     * of a hyperlink.
     */
    var Hyperlink = {
        Separators: ['!', '?', '.', ' ', '-', ':', ',', '\xa0'],
        Protocols: ['http://', 'https://', 'ftp://', 'mailto:'],

        // templates for web/ftp identifications
        TextTemplates: ['www.', 'ftp.'],

        /**
         * Predefined character attributes needed to remove a hyperlink style
         * from a piece of text.
         */
        CLEAR_ATTRIBUTES: { styleId: null, character: { url: null } }
    };

    // methods ----------------------------------------------------------------

    /**
     * Provides the URL of a selection without range.
     *
     * @param {Editor} editor
     *  The editor object.
     *
     * @param {Selection} selection
     *  A selection object which describes the current selection.
     *
     * @param {Object} [options]
     *  Optional options, introduced for performance reasons.
     *
     * @returns {Object}
     *  - {String|Null} url
     *      The URL or null if no URL character attribute is set at the
     *      selection.
     *  - {Boolean} beforeHyperlink
     *      True if we provide the URL for the hyperlink which is located at
     *      the next position.
     *  - {Boolean} behindHyperlink
     *      True if the current selection is exactly behind a hyperlink
     *  - {Boolean} clearAttributes
     *      True if the preselected attributes of the text model have to be
     *      extended by character attributes that remove the current URL style.
     */
    Hyperlink.getURLFromPosition = function (editor, selection, options) {

        var result = { url: null, beforeHyperlink: false, clearAttributes: false },
            splitOperation = Utils.getBooleanOption(options, 'splitOperation', false);

        if (!selection.hasRange()) {
            // find out a possible URL set for the current position
            var obj = null;

            if (splitOperation && selection.getInsertTextPoint()) {
                obj = selection.getInsertTextPoint();  // Performance: Using previously cached node point
            } else {
                obj = Position.getDOMPosition(editor.getCurrentRootNode(), selection.getStartPosition());
            }

            if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {
                var attributes = AttributeUtils.getExplicitAttributes(obj.node.parentNode, { family: 'character', direct: true });
                if (attributes.url) {
                    // Now we have to check some edge cases to prevent to show
                    // the popup for a paragraph which contains only an empty span
                    // having set the url attribute.
                    var span = $(obj.node.parentNode);
                    if ((span.text().length > 0) || (span.prev().length > 0) || DOM.isTextSpan(span.next())) {
                        result.url = attributes.url;
                    } else {
                        result.clearAttributes = true;
                    }
                }

                // Check next special case: Before a hyperlink we always want to show the popup
                if (splitOperation && selection.getParagraphCache()) {
                    // Performance: Using previously cached paragraph and text position
                    obj = Position.getDOMPosition(selection.getParagraphCache().node, [selection.getParagraphCache().offset + 1]);
                } else {
                    var nextPosition = selection.getStartPosition();
                    nextPosition[nextPosition.length - 1] += 1;
                    obj = Position.getDOMPosition(editor.getCurrentRootNode(), nextPosition);
                }

                if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {
                    var nextAttributes = AttributeUtils.getExplicitAttributes(obj.node.parentNode, { family: 'character', direct: true });
                    if (nextAttributes.url && (attributes.url !== nextAttributes.url)) {
                        result.url = nextAttributes.url;
                        result.beforeHyperlink = true;
                    }
                }
            }
        }

        return result;
    };

    /**
     * Tries to find a selection range based on the current text cursor
     * position. The URL character style is used to find a consecutive
     * range.
     *
     * @param {Editor} editor
     *  The editor object.
     *
     * @param {Selection} selection
     *  The current selection.
     *
     * @returns {Object}
     *  The new selection properties which includes a range with the same URL
     *  character style or null if no URL character style could be found.
     */
    Hyperlink.findSelectionRange = function (editor, selection) {

        var // the return object
            newSelection = null,
            // the paragraph node
            paragraph = null,
            // the logical start position
            startPosition = null,
            // the position inside the paragraph
            pos = null,
            // the url
            url = null,
            // the result object from 'getURLFromPosition'
            result = null;

        if (!selection.hasRange() && selection.getEnclosingParagraph()) {

            paragraph = selection.getEnclosingParagraph();
            startPosition = selection.getStartPosition();
            pos = startPosition[startPosition.length - 1];

            if (!selection.hasRange()) {
                result = Hyperlink.getURLFromPosition(editor, selection);
                url = result.url;
            }

            if (url) {
                if (result.beforeHyperlink) {
                    pos += 1;
                }
                startPosition[startPosition.length - 1] = pos;
                newSelection = Hyperlink.findURLSelection(editor, startPosition, url);
            } else {
                newSelection = Position.getWordSelection(paragraph, pos, Hyperlink.Separators);
            }
        }

        return newSelection;
    };

    /**
     * Tries to find a selection based on the provided position which includes
     * the provided url as character style.
     *
     * @param {Editor} editor
     *  The editor instance.
     *
     * @param {Array<Number>} startPosition
     *  The startPosition in the paragraph
     *
     * @param {String} url
     *  The hyperlink URL which is set as character style at pos
     *
     * @returns {Object}
     *  Contains start and end position of the selection where both could be
     *  null which means that there is no selection but the hyperlink should be
     *  inserted at the position.
     */
    Hyperlink.findURLSelection = function (editor, startPosition, url) {

        var startPos,
            endPos,
            startNode = null,
            endNode = null,
            attributes = null,
            obj = null,
            characterStyles = editor.getStyleCollection('character'),
            result = { start: null, end: null };

        obj = Position.getDOMPosition(editor.getCurrentRootNode(), startPosition);
        if (obj && obj.node && DOM.isTextSpan(obj.node.parentNode)) {

            startNode = obj.node.parentNode;
            endNode = obj.node.parentNode;

            while (endNode && endNode.nextSibling && (DOM.isTextSpan(endNode.nextSibling) || DOM.isValidNodeInsideWord(endNode.nextSibling))) {

                if (DOM.isTextSpan(endNode.nextSibling)) {
                    attributes = characterStyles.getElementAttributes(endNode.nextSibling).character;
                    if (attributes.url !== url) {
                        break;
                    }
                }

                endNode = endNode.nextSibling;
            }

            while (startNode && startNode.previousSibling && (DOM.isTextSpan(startNode.previousSibling) || DOM.isValidNodeInsideWord(startNode.previousSibling))) {

                if (DOM.isTextSpan(startNode.previousSibling)) {

                    attributes = characterStyles.getElementAttributes(startNode.previousSibling).character;
                    if (attributes.url !== url) {
                        break;
                    }
                }

                startNode = startNode.previousSibling;
            }

            startPos = Position.getPositionRangeForNode(editor.getCurrentRootNode(), startNode, true);
            if (startNode !== endNode) {
                endPos = Position.getPositionRangeForNode(editor.getCurrentRootNode(), endNode, true);
            } else {
                endPos = startPos;
            }

            result = { start: startPos.start[startPos.start.length - 1], end: endPos.end[endPos.end.length - 1] };
        }

        return result;
    };

    /**
     * Checks for a text in a paragraph which defines a hyperlink
     * e.g. http://www.open-xchange.com
     *
     * @param {Editor} editor
     *  The editor instance.
     *
     * @param {HTMLElement|jQuery} paragraph
     *  The paragraph node to check
     *
     * @param {Position} position
     *  The rightmost position to check the text to the left for
     *  a hyperlink text.
     *
     * @returns {Object}
     *  Returns an object containing the start and end position
     *  to set the hyperlink style/url or null if no hyperlink
     *  has been found.
     */
    Hyperlink.checkForHyperlinkText = function (editor, paragraph, position) {

        var // the return object of this function
            result = null,
            // the position inside the paragraph
            pos = 0,
            // an optional existing url
            url = null,
            // the hyper link selection object
            hyperlinkSelection = null,
            // the selection range object for an URL
            urlSelection = null,
            // whether the position is at the start of a paragraph (first text position)
            isStartOfParagraph = false;

        // helper function to receive url attribute from specified text span node
        function getUrlFromTextNode(node) {

            var characterStyles = editor.getStyleCollection('character'),
                attributes = null;

            if (DOM.isTextSpan(node)) { attributes = characterStyles.getElementAttributes(node).character; }

            return attributes && attributes.url ? attributes.url : null;
        }

        if (position !== null) {

            pos = position[position.length - 1];
            url = null;
            hyperlinkSelection = Position.getWordSelection(paragraph, pos, [' ', '\xa0'], { onlyLeft: true, returnTextSpan: true });

            // check, if the node already contains a valid url (at the position, where 'space' or 'enter' was pressed)
            if (hyperlinkSelection && hyperlinkSelection.node) {
                url = getUrlFromTextNode(hyperlinkSelection.node);
            }

            // if the space was inserted into an existing link, it shall not split the link
            if (url) {

                urlSelection = Hyperlink.findURLSelection(editor, position, url); // finding the complete hyper link range

                if (urlSelection) {

                    // special case for spaces at the beginning of paragraphs before hyper link (43830)
                    if (pos === hyperlinkSelection.start && pos === hyperlinkSelection.end && _.isNumber(urlSelection.start) && pos === urlSelection.start) { isStartOfParagraph = true; }

                    // space in a hyper link
                    if (!isStartOfParagraph && _.isNumber(urlSelection.end) && urlSelection.end > pos) { return null; }
                }
            }

            // if no url was found, try to evaluate it from the specified text
            if (!url && hyperlinkSelection && hyperlinkSelection.text) {
                url = HyperlinkUtils.checkForHyperlink(hyperlinkSelection.text);
            }

            if (url) {
                // for the setAttributes operation, it is necessary to reduce the end position by '1'
                // -> otherwise following text spans behind 'space' or 'enter' receive also the url.
                if (hyperlinkSelection && hyperlinkSelection.end) { hyperlinkSelection.end--; }
                // generating the return object from the found selection
                result = createResultFromHyperlinkSelection(position, hyperlinkSelection);
                result.url = url;
                result.isStartOfParagraph = isStartOfParagraph;
            }
        }

        return result;
    };

    /**
     * 'Inserts' a hyperlink at the provided selection using the
     * provided url.
     *
     * @param {Editor} editor
     *  The editor instance to use.
     *
     * @param {Position} start
     *  The start position of the selection
     *
     * @param {Position} end
     *  The end position of the selection.
     *
     * @param {String} url
     *  The url of the hyperlink to set at the selection
     *
     * @param {String} [target]
     *  The Id referencing to header or footer root node.
     *
     * @param {Object} [options]
     *  Additional options:
     *  Optional parameters:
     *  @param {Boolean} [options.removeHyperLinkAttributes=false]
     *      If set to true, an additional operation for clearing the URL attributes
     *      will be generated.
     */
    Hyperlink.insertHyperlink = function (editor, start, end, url, target, options) {

        var hyperlinkStyleId = editor.getDefaultUIHyperlinkStylesheet(),
            characterStyles = editor.getStyleCollection('character'),
            generator = editor.createOperationGenerator(),
            operationProperties = {},
            removeHyperLinkAttributes = Utils.getBooleanOption(options, 'removeHyperLinkAttributes', false);

        // never generation hyperlink operation on empty paragraph (42353)
        if (_.last(start) === 0 && _.last(end) === 0 && Position.getParagraphLength(editor.getRootNode(target), _.initial(start)) === 0) {
            Utils.warn('Ignoring invalid hyperlink detection in empty paragraph. URL: ' + url);
            return;
        }

        if (characterStyles.isDirty(hyperlinkStyleId) && !editor.useSlideMode()) {

            operationProperties = {
                attrs: characterStyles.getStyleSheetAttributeMap(hyperlinkStyleId),
                type: 'character',
                styleId: hyperlinkStyleId,
                styleName: characterStyles.getName(hyperlinkStyleId),
                parent: characterStyles.getParentId(hyperlinkStyleId),
                uiPriority: characterStyles.getUIPriority(hyperlinkStyleId)
            };

            // parent is an optional value, should not be send as 'null'
            if (operationProperties.parent === null || operationProperties.parent === '') { delete operationProperties.parent; }

            // insert hyperlink style to document
            generator.generateOperation(Operations.INSERT_STYLESHEET, operationProperties);
            characterStyles.setDirty(hyperlinkStyleId, false);
        }

        operationProperties = {
            attrs: { character: { url: url } },
            start: _.clone(start),
            end: removeHyperLinkAttributes ? Position.increaseLastIndex(end) : _.clone(end)
        };
        // ppt doesnt use slideId
        if (!editor.useSlideMode()) {
            operationProperties.attrs.styleId = hyperlinkStyleId;
        }

        if (target) {
            operationProperties.target = target;
        }
        generator.generateOperation(Operations.SET_ATTRIBUTES, operationProperties);

        if (removeHyperLinkAttributes) {
            generator.generateOperation(Operations.SET_ATTRIBUTES, { start: Position.increaseLastIndex(end), attrs: Hyperlink.CLEAR_ATTRIBUTES });
        }

        editor.applyOperations(generator);
    };

    /**
     * 'Removes' the hyperlink at the provided selection.
     *
     * @param {Editor} editor
     *  The editor instance to use.
     *
     * @param {Position} start
     *  The start position of the selection
     *
     * @param {Position} end
     *  The end position of the selection.
     */
    Hyperlink.removeHyperlink = function (editor, start, end) {

        var generator = editor.createOperationGenerator();

        generator.generateOperation(Operations.SET_ATTRIBUTES, {
            attrs: Hyperlink.CLEAR_ATTRIBUTES,
            start: _.clone(start),
            end: _.clone(end)
        });

        editor.applyOperations(generator);
    };

    /**
     * Checks if element is a clickable part of the hyperlink popup
     * return true if yes otherwise false.
     *
     * @param {HTMLElement|jQuery} element
     */
    Hyperlink.isClickableNode = function (element) {

        var node = $(element);

        if (Hyperlink.isPopupNode(node.parent())) {
            return node.is('a');
        }

        return false;
    };

    /**
     * Checks if element is the topmost div of a popup returns true if yes
     * otherwise false.
     *
     * @param {HTMLElement|jQuery} element
     *
     * @returns {Boolean}
     */
    Hyperlink.isPopupNode = function (element) {
        return $(element).is('.inline-popup');
    };

    /**
     * Check if element is the hyperlink popup or a child within the popup.
     *
     * @param {HTMLElement|jQuery} element
     *
     * @returns {Boolean}
     */
    Hyperlink.isPopupOrChildNode = function (element) {
        var node = $(element);

        return (Hyperlink.isClickableNode(node, false) ||
                 Hyperlink.isPopupNode(node));
    };

    /**
     * Provides the text from a selection.
     *
     * @param {Selection} selection
     *  A selection object which is used to extract the text from.
     */
    Hyperlink.getSelectedText = function (selection) {
        var text = '';

        selection.iterateNodes(function (node, pos, start, length) {
            if ((start >= 0) && (length >= 0) && DOM.isTextSpan(node)) {
                var nodeText = $(node).text();
                if (nodeText) {
                    text = text.concat(nodeText.slice(start, start + length));
                }
            }
        });

        return text;
    };

    /**
     * Returns the hyperlink pop-up menu, as jQuery object.
     *
     * @param {Editor} editor
     *  An editor instance.
     *
     * @returns {jQuery}
     *  The hyperlink pop-up menu node, as jQuery object.
     */
    Hyperlink.getPopupNode = function (editor) {
        return editor.getNode().children('.inline-popup.hyperlink');
    };

    /**
     * Returns whether the hyperlink pop-up menu is visible.
     *
     * @param {Editor} editor
     *  An editor instance.
     *
     * @returns {Boolean}
     *  Whether the hyperlink pop-up menu is currently open.
     */
    Hyperlink.hasPopup = function (editor) {
        return Hyperlink.getPopupNode(editor).css('display') !== 'none';
    };

    // exports ================================================================

    return Hyperlink;

});
