/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/statuspane', [
    'io.ox/office/tk/utils',
    'io.ox/office/baseframework/view/toolpane',
    'io.ox/office/baseframework/view/toolbar',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/controls',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, ToolPane, ToolBar, Labels, Controls, gt) {

    'use strict';

    // convenience shortcuts
    var Button = Controls.Button;
    var CompoundButton = Controls.CompoundButton;

    // class StatusPane =======================================================

    /**
     * The status pane in spreadsheet documents containing the sheet tabs and
     * the subtotal results of the current selection.
     *
     * @constructor
     *
     * @extends ToolPane
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    var StatusPane = ToolPane.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // the application instance
        var app = docView.getApp();

        // the tool bar for all sheet controls
        var sheetToolBar = null;

        // the sheet selector controls
        var sheetGroup = null;
        var largeSheetList = null;
        var smallSheetList = null;
        var moreMenuGroup = null;

        // base constructor ---------------------------------------------------

        ToolPane.call(this, docView, { position: 'bottom', classes: 'status-pane standard-design' });

        // private methods ----------------------------------------------------

        /**
         * Refreshes the layout of the sheet selectors according to the current
         * width of the sheet tool bar.
         */
        function refreshSheetToolBar() {

            // the node of the sheet tool bar
            var sheetNode = sheetToolBar.getNode();
            // the width of the center node
            var avaSheetWidth = Utils.getFloorNodeSize(sheetNode.closest('.float-wrapper')).width;

            // max width of the info node in pixels
            var minSheetWidth = 240;

            function getToolBarWidth() {
                return Utils.getCeilNodeSize(sheetNode).width;
            }

            // calculate and set remaining space for the specified sheet selector
            function setRemainingWidth(group) {
                var remainingWidth = group.getNode().outerWidth() + avaSheetWidth - getToolBarWidth();
                group.setWidth(remainingWidth);
            }

            // hide the sheet list selectors, show the sheet tab selector with automatic width
            largeSheetList.hide();
            smallSheetList.hide();
            sheetGroup.show();
            sheetGroup.setFullWidth();

            // add leading padding with the width of the row header panes
            sheetNode.css({ width: 'auto', paddingLeft: docView.getHeaderWidth() });

            if (avaSheetWidth < getToolBarWidth()) {
                sheetNode.css({ paddingLeft: '' });

                if (avaSheetWidth < 380) {
                    // check min width for sheets
                    if (avaSheetWidth < minSheetWidth) {
                        // set fix min width for sheets
                        avaSheetWidth = minSheetWidth;
                    }
                    // show large sheet drop-down list for small devices
                    sheetGroup.hide();
                    largeSheetList.show();
                    // calculate and set remaining space for the sheet drop-down button
                    largeSheetList.setWidth(avaSheetWidth);
                    setRemainingWidth(largeSheetList);
                } else {
                    // show additional sheet selector controls
                    smallSheetList.show();
                    // calculate and set remaining space for the sheet tabs
                    setRemainingWidth(sheetGroup);
                }
            }

            // reset sheet component to its current width
            sheetNode.css({ width: avaSheetWidth });
        }

        // initialization -----------------------------------------------------

        // create the 'More' drop-down menu with all standard actions for sheets
        moreMenuGroup = new CompoundButton(docView, { classes: 'moreBtn', label: Labels.MORE_LABEL, tooltip: gt('More sheet operations') });
        docView.addSheetActionButtons(moreMenuGroup);

        this.addViewComponent(sheetToolBar = new ToolBar(docView)
            .addGroup('view/sheet/active', largeSheetList = new Controls.ActiveSheetList(docView, { showNames: true }))
            .addGap()
            .addGroup('view/sheet/active', smallSheetList = new Controls.ActiveSheetList(docView))
            .addGap()
            .addGroup('view/sheet/active', sheetGroup = new Controls.ActiveSheetGroup(docView))
            .addGap()
            .addGroup('document/insertsheet', new Button(docView, { icon: 'fa-plus', tooltip: gt('Insert sheet') }))
            .addGroup('document/editable', moreMenuGroup)
        );

        this.addViewComponent(new ToolBar(docView, { autoHideGroups: true })
            .addGroup('view/status/label', new Controls.DynamicLabel(docView))
            .addGroup('view/status/subtotals', new Controls.SubtotalList(docView))
        , { targetArea: 'trailing' });

        // refresh sheet tool bar after sheet tabs have been inserted, deleted, or changed
        this.listenTo(sheetGroup, 'change:sheets', refreshSheetToolBar);

        // handle external refresh (e.g. browser resize), but wait for an active sheet
        this.listenOnceTo(docView, 'change:activesheet', function () {
            refreshSheetToolBar();
            self.on('pane:layout', refreshSheetToolBar);
        });

        // initialize visibility
        this.toggle(app.getUserSettingsValue('showStatusBar', true));

        // always hide the status pane when an internal application error occurs
        this.listenTo(app, 'docs:state:error', function () { self.hide(); });

        if (_.browser.Android) {
            //".window-container" is oriented in screen with "top left right bottom" set to zero,
            //when the soft keyboard on Android devices pops up, the status bar stays over the keyboard
            //this makes no sense and looks ugly, so we detect the open keyboard and hide the status pane
            app.registerWindowResizeHandler(function () {
                // old visible boolean for hiding this bar when soft keyboard is open on Android devices
                var visibleOverwrite = null;

                return function () {
                    if (Utils.isSoftKeyboardOpen() && (Utils.getInnerWindowHeight() / Utils.getScreenHeight()) < 0.7) {
                        if (visibleOverwrite === null && self.isVisible()) {
                            visibleOverwrite = true;
                            self.hide();
                        }
                    } else {
                        if (visibleOverwrite !== null) {
                            self.toggle(visibleOverwrite);
                        }
                        visibleOverwrite = null;
                    }
                };
            }());
        }

        if (_.browser.iOS) {
            this.getNode().on('taphold', function (e) {
                $(e.target).trigger('contextmenu');

                self.getNode().one('touchend', function () {
                    return false;
                });

                return false;
            });
        }

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = app = docView = sheetToolBar = null;
            sheetGroup = largeSheetList = smallSheetList = moreMenuGroup = null;
        });

    } }); // class StatusPane

    // exports ================================================================

    return StatusPane;

});
