/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/rendercache', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/object/triggerobject',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/autostylecache'
], function (Utils, IteratorUtils, ValueSet, ValueMap, TriggerObject, SheetUtils, PaneUtils, RenderUtils, AutoStyleCache) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var IntervalArray = SheetUtils.IntervalArray;
    var RangeArray = SheetUtils.RangeArray;
    var RangeSet = SheetUtils.RangeSet;

    // line width for grid colors in the canvas
    var GRID_LINE_WIDTH = RenderUtils.GRID_LINE_WIDTH;

    // class RenderCache ======================================================

    /**
     * A cache that stores settings for the visible areas in a specific sheet.
     *
     * Instances of this class trigger the following events:
     *  - 'update:matrix'
     *      After updating the existing cell elements in the rendering matrix
     *      due to cell changes from document operations or from view update
     *      notifications received from the server. Event listeners receive the
     *      following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {RangeArray} dirtyRanges
     *          The addresses of the changed cell ranges to be rendered.
     *
     * @constructor
     *
     * @extends TriggerObject
     *
     * @param {SpreadsheetView} docView
     *  The document view that contains this cache instance.
     */
    var RenderCache = TriggerObject.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // the application instance
        var app = docView.getApp();

        // the spreadsheet model and its containers
        var docModel = docView.getDocModel();

        // the collection of table style sheets
        var tableStyles = docModel.getStyleCollection('table');

        // the cache for all auto style descriptors
        var styleCache = new AutoStyleCache(docView);

        // all cell elements of the rendering matrix, mapped by cell key
        var cellMatrix = new ValueSet('key');

        // column headers in the rendering matrix (as sorted array, and mapped by column index)
        var colHeader = new RenderUtils.MatrixHeaderCache(true);

        // row headers in the rendering matrix (as sorted array, and mapped by column index)
        var rowHeader = new RenderUtils.MatrixHeaderCache(false);

        // the current bounding ranges and rectangles, per pane position (missing entry: hidden)
        var boundaryMap = new ValueMap();

        // the unified bounding ranges covered by this cache
        var boundRanges = new RangeArray();

        // the total range covered by all bounding ranges
        var totalRange = null;

        // all merged ranges overlapping with the bounding ranges
        var mergedRangeSet = new RangeSet();

        // all pending cell ranges to be updated in the rendering matrix
        var dirtyRanges = new RangeArray();

        // all pending cell ranges currently updated in an asynchronous loop
        var pendingRanges = new RangeArray();

        // timer for the current asynchronous update cycle
        var updateTimer = null;

        // special behavior for OOXML and ODF files
        var ooxml = app.isOOXML();
        var odf = app.isODF();

        // base constructor ---------------------------------------------------

        TriggerObject.call(this, docView);

        // private methods ----------------------------------------------------

        /**
         * Deletes the sorted arrays of merge descriptors from all row header
         * elements covering the passed ranges.
         *
         * @param {RangeArray} changedRanges
         *  The ranges whose merge descriptors will be deleted from the row
         *  header elements of the rendering matrix.
         */
        function resetMergeDescsInRows(changedRanges) {
            // visit all header elements in the merged intervals
            changedRanges.rowIntervals().merge().forEach(function (interval) {
                rowHeader.iterateElements(interval.first, interval.last, function (element) { element.mergeDescs = null; });
            });
        }

        /**
         * Returns the descriptors of all merged ranges covering the specified
         * row in the rendering matrix.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @returns {Array<MergeDescriptor>}
         *  A sorted array with descriptors of all merged ranges covering the
         *  specified row.
         */
        function getMergeDescsInRow(row) {

            // the matrix header element of the passed row
            var headerElement = rowHeader.get(row);
            if (headerElement.mergeDescs) { return headerElement.mergeDescs; }

            // find all merged ranges covering the specified row
            headerElement.mergeDescs = mergedRangeSet.findRanges(docModel.makeRowRange(row)).map(function (mergedRange) {
                return mergedRange.mergeDesc;
            });

            // sort the merge descriptors by start column
            headerElement.mergeDescs = _.sortBy(headerElement.mergeDescs, function (mergeDesc) {
                return mergeDesc.mergedRange.start[0];
            });

            return headerElement.mergeDescs;
        }

        /**
         * Updates the cached merged range settings. Inserts new merged ranges
         * not yet cached, and removes old merged ranges not visible anymore.
         *
         * @returns {RangeArray}
         *  The addresses of all merged ranges that have been added, removed,
         *  or that contain a changed visible area.
         */
        var updateMergedRanges = RenderUtils.profileMethod('RenderCache.updateMergedRanges()', function () {

            // the model of the active sheet
            var sheetModel = docView.getSheetModel();
            // addresses of entire visible rows (needed to support overlapping text)
            var rowRanges = docModel.makeRowRanges(rowHeader.intervals);
            // all merged ranges inside the bounding row intervals
            var mergedRanges = sheetModel.getMergeCollection().getMergedRanges(rowRanges);
            // all changed merged ranges (inserted, deleted, changed visible area)
            var changedRanges = new RangeArray();
            // copy of the range set containing the merged ranges currently cached (used to find ranges to be removed)
            var oldRangeSet = mergedRangeSet.clone();

            // create new descriptors, remove descriptors of existing merged ranges from 'oldRangeSet'
            // (remaining merged ranges will be deleted from the cache)
            mergedRanges.forEach(function (mergedRange) {

                // get an existing merged range in the cache
                var oldMergedRange = oldRangeSet.get(mergedRange);
                // shrink to visible columns/rows in the sheet
                var shrunkenRange = sheetModel.shrinkRangeToVisible(mergedRange);

                // nothing to do, if the merged range was not known, but is invisible
                if (!oldMergedRange && !shrunkenRange) { return; }

                // create a new entry, if the merged range is not known yet
                if (!oldMergedRange) {
                    RenderUtils.withLogging(function () { RenderUtils.log('added merged range ' + mergedRange); });

                    // register the entire merged range for rendering
                    changedRanges.push(mergedRange);

                    // store a new descriptor of the merged range in the range set
                    var mergeDesc = new RenderUtils.MergeDescriptor(mergedRange, shrunkenRange);
                    mergedRange = mergedRange.clone();
                    mergedRange.mergeDesc = mergeDesc;
                    mergedRangeSet.insert(mergedRange);

                    // set or create the matrix cell element for the visible origin
                    var origAddress = shrunkenRange.start;
                    mergeDesc.element = cellMatrix.get(origAddress.key()) || new RenderUtils.MatrixCellElement(origAddress);
                    mergeDesc.element.merged = mergeDesc;
                    return;
                }

                // remove the range from 'oldRangeSet' so that it will not be removed from the cache (see below)
                oldRangeSet.remove(oldMergedRange);

                // update the visible area of the merged range
                if (shrunkenRange && oldMergedRange.mergeDesc.shrunkenRange.differs(shrunkenRange)) {
                    oldMergedRange.mergeDesc.shrunkenRange = shrunkenRange;
                    changedRanges.push(mergedRange);
                }
            });

            // delete remaining merged ranges from the cache
            oldRangeSet.forEach(function (mergedRange) {
                RenderUtils.withLogging(function () { RenderUtils.log('removed merged range ' + mergedRange); });
                changedRanges.push(mergedRange);
                mergedRangeSet.remove(mergedRange);
            });

            // delete all sorted arrays of merged ranges per row (will be updated on demand)
            if (!changedRanges.empty()) { resetMergeDescsInRows(changedRanges); }

            return changedRanges;
        });

        /**
         * Returns a descriptor for the nearest merged range in the same row,
         * but left of or right of the specified cell address.
         *
         * @param {Address} address
         *  The address of a cell.
         *
         * @param {Boolean} forward
         *  Specifies where to search for a merged range: If set to true, a
         *  merged range will be searched right of the cell, otherwise left of
         *  the cell.
         *
         * @returns {MergeDescriptor|Null}
         *  The descriptor of the nearest merged range next to the specified
         *  cell address, if available; otherwise null.
         */
        function findNextMergeDesc(address, forward) {

            // the merged ranges covering the row of the passed address, sorted by column
            var mergeDescsInRow = getMergeDescsInRow(address[1]);

            // binary search in the array of merged ranges according to direction
            return forward ?
                Utils.findFirst(mergeDescsInRow, function (mergeDesc) { return address[0] < mergeDesc.mergedRange.start[0]; }, { sorted: true }) :
                Utils.findLast(mergeDescsInRow, function (mergeDesc) { return mergeDesc.mergedRange.end[0] < address[0]; }, { sorted: true });
        }

        /**
         * Updates the matrix header elements according to the current bounding
         * intervals. Removes all header and cell elements that are not visible
         * anymore, and inserts new header and cell elements that have become
         * visible.
         *
         * @returns {RangeArray}
         *  The addresses of all cell ranges that have been added to the
         *  rendering matrix.
         */
        function buildMatrixElements(columns) {

            // the model of the active sheet
            var sheetModel = docView.getSheetModel();
            // the column/row collection
            var collection = columns ? sheetModel.getColCollection() : sheetModel.getRowCollection();
            // the matrix header to be updated in this pass
            var header1 = columns ? colHeader : rowHeader;
            // the matrix header in the opposite direction
            var header2 = columns ? rowHeader : colHeader;
            // all header elements from header1, as plain array
            var elements1 = header1.getElements();
            // array index of the next header element not processed yet
            var nextIndex = 0;
            // the property name for the next linked cell element
            var nextProp = columns ? 'b' : 'r';
            // column/row indexes of all new header elements
            var newIndexes = [];

            // unlinks and releases the passed header element, and all attached cell elements
            function releaseHeaderElement(headerElement) {

                // the current matrix cell element
                var cellElement = headerElement.f;

                // delete all matrix cell elements attached to the header element
                while (cellElement) {
                    var nextElement = cellElement[nextProp];
                    cellMatrix.remove(cellElement);
                    cellElement.unlink();
                    cellElement = nextElement;
                }

                // unlink and delete the header element
                header1.map.remove(headerElement.index);
                headerElement.unlink();
            }

            // returns a cell element from the rendering matrix
            function getCellElement(colElement, rowElement) {
                return (colElement && rowElement) ? cellMatrix.get(Address.key(colElement.index, rowElement.index), null) : null;
            }

            // process all visible columns/rows
            var iterator = collection.createIterator(header1.intervals, { visible: true });
            IteratorUtils.forEach(iterator, function (entryDesc, result) {

                // an existing or new header element in header1
                var element1 = null;

                // release all preceding existing header elements that are not visible anymore
                while ((element1 = elements1[nextIndex]) && (element1.index < entryDesc.index)) {
                    elements1.splice(nextIndex, 1);
                    releaseHeaderElement(element1);
                }

                // insert a new header element if needed
                element1 = elements1[nextIndex];
                if (!element1 || (entryDesc.index < element1.index)) {

                    // create and store the new header element
                    element1 = new RenderUtils.MatrixHeaderElement(entryDesc.index, columns, elements1[nextIndex - 1], element1);
                    header1.map.insert(element1.index, element1);
                    elements1.splice(nextIndex, 0, element1);
                    newIndexes.push(element1.index);

                    // create all cell elements in the matrix
                    header2.getElements().forEach(function (element2) {

                        // the column and row header element, according to direction
                        var colElement = columns ? element1 : element2;
                        var rowElement = columns ? element2 : element1;
                        // the address of the new cell
                        var address = new Address(colElement.index, rowElement.index);
                        // the adjacent cell elements
                        var l = getCellElement(colElement.p, rowElement);
                        var r = getCellElement(colElement.n, rowElement);
                        var t = getCellElement(colElement, rowElement.p);
                        var b = getCellElement(colElement, rowElement.n);
                        // the new matrix cell element
                        var cellElement = new RenderUtils.MatrixCellElement(address, colElement, rowElement, l, r, t, b);

                        // insert the new cell element into the rendering matrix
                        cellMatrix.insert(cellElement);
                    });
                }
                nextIndex += 1;

                // update contents and formatting of the header element
                element1.update(styleCache, entryDesc);

                // update the leading/trailing flags specifying whether the header
                // element is located at the beginning or end of a bounding interval
                element1.leading = element1.index === result.orig.first;
                element1.trailing = element1.index === result.orig.last;
            });

            // release all trailing existing header elements that are not visible anymore
            elements1.splice(nextIndex).forEach(releaseHeaderElement);

            // create and return the range addresses for all new columns/rows in the matrix
            return docModel.makeFullRanges(IntervalArray.mergeIndexes(newIndexes), columns);
        }

        /**
         * Visits all cells in the passed ranges, and returns the descriptors
         * of all covered merged ranges starting outside.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.columns=false]
         *      If set to true, the cells in the passed ranges will be visited
         *      in vertical orientation (all cells top-down in first column,
         *      then all cells top-down in second column, etc.). By default,
         *      the cells will be visited in horizontal orientation (all cells
         *      left-to-right in first row, then all cells left-to-right in
         *      second row, etc.).
         *  - {Boolean} [options.first=false]
         *      If set to true, only the matrix elements of the first visible
         *      row (or the first visible column in vertical mode, see option
         *      'columns') of each range will be visited.
         *
         * @returns {Array<MergeDescriptor>}
         *  The descriptors of all merged ranges covering but not starting in
         *  the passed ranges.
         */
        function iterateCellsInRanges(ranges, callback, options) {

            // whether to visit the first column/row only
            var firstOnly = Utils.getBooleanOption(options, 'first', false);
            // iteration direction (true for columns instead of rows)
            var columns = Utils.getBooleanOption(options, 'columns', false);
            // descriptors of all visited merged ranges
            var visitedMergedSet = new ValueSet('originKey');
            // descriptors of all visited merged ranges whose visible origin has been visited
            var visitedOriginsSet = new ValueSet('originKey');

            function invokeCallback(element, range) {

                // first, invoke the callback function (internal update process adds merged range information)
                callback.call(self, element, range);

                // collect origin cells of all merged ranges covering the visited cells (value false
                // for cells covered by the merged range, value true for top-left visible cells)
                if (element.isMergedOrigin()) {
                    visitedOriginsSet.insert(element.merged);
                } else if (element.isMerged()) {
                    visitedMergedSet.insert(element.merged);
                }
            }

            // process all ranges separately
            RangeArray.forEach(ranges, function (range) {

                // the first visible column header element
                var colElement = colHeader.findFirst(range.start[0], range.end[0]);
                // the first visible row header element
                var rowElement = colElement ? rowHeader.findFirst(range.start[1], range.end[1]) : null;
                // the first cell element in the current row
                var firstElement = rowElement ? cellMatrix.get(Address.key(colElement.index, rowElement.index), null) : null;

                // do nothing, if the range is not visible at all
                if (!firstElement) { return; }

                // last column/row index to be visited
                var lastCol = range.end[0];
                var lastRow = range.end[1];
                // the current cell element in the current row
                var currElement = null;

                // visit all cells in the passed range (performance: implemented separate loops
                // for horizontal and vertical iteration orientation, to prevent fancy indirect
                // property access which takes a little more time)
                if (columns) {

                    // vertical orientation: column-by-column
                    while (firstElement && (firstElement.col.index <= lastCol)) {
                        currElement = firstElement;
                        while (currElement && (currElement.row.index <= lastRow)) {
                            invokeCallback(currElement, range);
                            currElement = currElement.b;
                        }
                        firstElement = firstOnly ? null : firstElement.r;
                    }

                } else {

                    // horizontal orientation: row-by-row
                    while (firstElement && (firstElement.row.index <= lastRow)) {
                        currElement = firstElement;
                        while (currElement && (currElement.col.index <= lastCol)) {
                            invokeCallback(currElement, range);
                            currElement = currElement.r;
                        }
                        firstElement = firstOnly ? null : firstElement.b;
                    }
                }
            });

            // collect and return descriptors of all merged ranges whose top-left cell has NOT been visited
            return visitedMergedSet.reject(visitedOriginsSet.has, visitedOriginsSet);
        }

        function abortUpdateCycle() {
            if (updateTimer) {
                updateTimer.abort();
                updateTimer = null;
            }
        }

        /**
         * Registers the specified cell ranges as 'dirty', which will include
         * them in the next update cycle.
         *
         * @param {RangeArray|Range} ranges
         *  An array of cell range addresses, or a single cell range address.
         */
        var registerDirtyRanges = (function () {

            // direct callback: register passed ranges
            function registerRanges(ranges) {
                RenderUtils.log('RenderCache.registerDirtyRanges(): (1) store dirty ranges');
                dirtyRanges.append(ranges);
            }

            // deferred callback: updates all dirty cell elements in the rendering matrix
            var updateMatrix = RenderUtils.profileAsyncMethod('RenderCache.registerDirtyRanges(): (2) update rendering matrix', function () {

                // the model of the active sheet
                var sheetModel = docView.getSheetModel();
                // the cell collection of the active sheet
                var cellCollection = sheetModel.getCellCollection();
                // the hyperlink collection of the passed sheet
                var hyperlinkCollection = sheetModel.getHyperlinkCollection();
                // the conditional formatting collection of the passed sheet
                var condFormatCollection = sheetModel.getCondFormatCollection();
                // the table collection of the active sheet
                var tableCollection = sheetModel.getTableCollection();

                // the cell attributes of active conditional formatting rules (cached by address for repeated usage in merged ranges)
                var condFormatResultMap = new ValueMap();
                // the settings needed to render the cells of a table range (cached by table model UID for repeated usage)
                var tableSettingsMap = new ValueMap();
                // the cell attributes of a table style sheet covered by the cell (cached by address for repeated usage in merged ranges)
                var tableAttrSetMap = new ValueMap();

                // the ranges to be updated and rendered
                var updateRanges = new RangeArray();
                // all matrix cell elements with overflowing text, mapped by cell key, for second-step-update
                var overflowElementMap = new ValueMap();
                // all matrix cell elements whose overflowing neighbors need to be updated
                var neighborElementSet = new ValueSet('key');

                // returns the cell attributes of active conditional formatting rules
                function getCondFormatResult(address) {
                    return condFormatResultMap.getOrCreate(address.key(), function () {
                        return condFormatCollection.resolveRules(address);
                    });
                }

                // returns the settings needed to render the cells of a table range (cached for repeated usage)
                function getTableSettings(tableModel) {
                    return tableSettingsMap.getOrCreate(tableModel.getUid(), function () {
                        var styleId = tableModel.getStyleId();
                        var rawAttrSet = styleId ? tableStyles.getStyleSheetAttributeMap(styleId, true) : null;
                        return _.isEmpty(rawAttrSet) ? null : {
                            range: tableModel.getRange(),
                            attrs: rawAttrSet,
                            flags: tableModel.getStyleFlags()
                        };
                    });
                }

                // returns the cell attributes of a table style sheet covered by the cell
                function getTableAttrSet(address) {
                    return tableAttrSetMap.getOrCreate(address.key(), function () {
                        // resolve the table style settings
                        var tableModel = tableCollection.findTable(address);
                        var settings = tableModel ? getTableSettings(tableModel) : null;
                        if (!settings) { return null; }
                        // resolve the formatting attributes for the current cell address
                        var range = settings.range;
                        var colInterval = { start: address[0] - range.start[0], end: address[0] - range.start[0] };
                        var rowInterval = { start: address[1] - range.start[1], end: address[1] - range.start[1] };
                        var attrSet = tableStyles.resolveCellAttributeSet(settings.attrs, colInterval, rowInterval, range.cols(), range.rows(), settings.flags);
                        // remove the table attributes (prevent to use them in auto-style cache)
                        delete attrSet.table;
                        return attrSet;
                    });
                }

                // updates cell formatting and all text settings of a single matrix cell element
                function updateCellElement(element) {

                    // address of the source cell (if the cell is covered by a merged range, use its top-left cell)
                    var address = element.merged ? element.merged.mergedRange.start : element.address;
                    // the cell model at the source address
                    var cellModel = cellCollection.getCellModel(address);
                    // the identifier of the initial auto-style (bug 40429: always set a style descriptor to (blank) merged ranges)
                    var baseStyleId = cellModel ? cellModel.s : element.isMergedOrigin() ? cellCollection.getDefaultStyleId(address) : null;

                    // OOXML: use the auto-style of current cell (not merged origin) for outer borders in merged ranges
                    // ODF: expand the borders of the top-left auto-style in merged ranges
                    var borderStyleId = (ooxml && element.isMergedBorder()) ? cellCollection.getStyleId(element.address) : null;

                    // result descriptor for conditional formatting
                    var condFormatResult = getCondFormatResult(address);
                    // explicit attributes of conditional formatting to be rendered over the auto-style of the cell
                    var condAttrSet = condFormatResult ? condFormatResult.attributeSet : null;
                    // additional rendering properties returned by conditional formattings
                    var renderProps = condFormatResult ? condFormatResult.renderProps : null;

                    // explicit attributes of a table style to be rendered over the auto-style of the cell
                    var tableAttrSet = getTableAttrSet(address);

                    // the auto-style identifier of the source cell (resolve default style for undefined cells with conditional formatting)
                    var styleId = (baseStyleId !== null) ? baseStyleId : ((borderStyleId !== null) || condAttrSet || tableAttrSet) ? cellCollection.getDefaultStyleId(address) : null;
                    // the rendering style descriptor; or null for undefined cells without any special settings
                    var styleDesc = (styleId !== null) ? styleCache.getStyle(styleId, { borderStyleId: borderStyleId, condAttrSet: condAttrSet, tableAttrSet: tableAttrSet }) : null;

                    // hyperlinks in ODF files will be drawn with blue text color by default
                    if (odf && hyperlinkCollection.getCellURL(address)) {
                        (renderProps || (renderProps = {})).hyperlink = true;
                    }

                    element.update(sheetModel, cellModel, styleDesc, renderProps);
                }

                // returns the pixel position of the next available adjacent merged range
                function findNextMergedRectangle(element, forward) {
                    var mergeDesc = findNextMergeDesc(element.address, forward);
                    return mergeDesc ? docView.getRangeRectangle(mergeDesc.mergedRange) : null;
                }

                // finds the content cell next to the passed cell element
                function findNextContentElement(element, forward) {
                    var prop = forward ? 'r' : 'l';
                    for (element = element[prop]; element && element.blank; element = element[prop]) {}
                    return element ? element : null;
                }

                // finds the column header of a content cell next to the passed cell element
                function findNextContentCol(element, forward) {
                    element = findNextContentElement(element, forward);
                    return element ? element.col : null;
                }

                // returns the address of the next adjacent text cell with overflowing text
                function findNextOverflowTextCell(element, forward) {

                    // find the next visible value cell in teh specified direction
                    var address = cellCollection.findFirstCellLinear(element.address, forward ? 'right' : 'left', { type: 'value', visible: true });
                    if (!address) { return null; }

                    // skip non-text cells
                    var value = cellCollection.getValue(address);
                    if (typeof value !== 'string') { return null; }

                    // skip existing cells without overflowing text
                    var attrSet = cellCollection.getAttributeSet(address);
                    return SheetUtils.hasWrappingAttributes(attrSet) ? null : address;
                }

                // finds and registers a matrix element with overflowing text next to the passed element
                function registerAdjacentOverflowCell(element, forward) {

                    // find next available content cell element
                    var nextElement = findNextContentElement(element, forward);

                    // no content cell in the matrix: search in cell collection, create a dummy cell element
                    if (!nextElement) {
                        var address = findNextOverflowTextCell(element, forward);
                        if (!address) { return; }
                        // create a dummy matrix element
                        nextElement = new RenderUtils.MatrixCellElement(address);
                        updateCellElement(nextElement);
                    }

                    // skip existing elements without overflowing text
                    if (!nextElement.hasOverflowText(!forward)) { return; }

                    // check that no merged range is between overflow cell and current position
                    var mergeDesc = findNextMergeDesc(element.address, forward);
                    if (mergeDesc && (forward ? (mergeDesc.mergedRange.start[0] <= nextElement.address[0]) : (nextElement.address[0] <= mergeDesc.mergedRange.end[0]))) { return; }

                    // register matrix cell element for clipping update
                    var overflowEntry = overflowElementMap.getOrCreate(nextElement.key, function () { return { element: nextElement }; });

                    // performance: store the original matrix element causing text clipping in the map,
                    // to prevent searching for that element later (used as clipping boundary)
                    if (!element.blank) {
                        if (forward) { overflowEntry.l = element; } else { overflowEntry.r = element; }
                    }
                }

                // adds a range address to the update ranges covering the surrounding cells of an overflowing text cell
                function addOverflowRange(address, overflowL, overflowR) {
                    var overflowRange = docModel.makeRowRange(address[1]);
                    if (!overflowL) { overflowRange.start[0] = address[0]; }
                    if (!overflowR) { overflowRange.end[0] = address[0]; }
                    updateRanges.push(overflowRange);
                }

                // abort the current asynchronous update cycle (this restores 'dirtyRanges' and 'pendingRanges')
                abortUpdateCycle();

                // copy of the dirty ranges, for asynchronous registration of new dirty ranges, timer abort, etc.
                pendingRanges = dirtyRanges.merge();
                dirtyRanges.clear();

                // reduce update ranges to bounding row intervals (but do not reduce to bounding
                // column intervals, original ranges are needed for updating overflowing text cells)
                rowHeader.intervals.forEach(function (interval) {
                    updateRanges.append(pendingRanges.intersect(docModel.makeRowRange(interval)));
                });

                // merge the ranges to be updated, and shrink them to visible areas
                updateRanges = sheetModel.mergeAndShrinkRanges(updateRanges);
                RenderUtils.withLogging(function () { RenderUtils.log('update ranges: ' + updateRanges); });

                // collect all existing matrix elements
                var updateElements = [];
                var outerMergeDescs = RenderUtils.takeTime('collect matrix elements', function () {
                    return iterateCellsInRanges(updateRanges, function (element) {
                        updateElements.push(element);
                    });
                });

                // update all existing matrix elements
                updateTimer = RenderUtils.takeAsyncTime('update matrix elements', function () {
                    return self.iterateArraySliced(updateElements, function (element) {

                        // whether the cell element was blank before
                        var oldBlank = element.blank;
                        // whether the cell element has contained overflowing text before
                        var oldOverflowL = element.hasOverflowText(false);
                        var oldOverflowR = element.hasOverflowText(true);

                        // find and store descriptor of a merged range covering the cell element
                        var mergedRanges = mergedRangeSet.findByAddress(element.address);
                        element.merged = mergedRanges.empty() ? null : mergedRanges.first().mergeDesc;

                        // update cell formatting and all text settings
                        updateCellElement(element);

                        // whether the cell element is blank now
                        var newBlank = element.blank;
                        // whether the cell element contains overflowing text now
                        var newOverflowL = element.hasOverflowText(false);
                        var newOverflowR = element.hasOverflowText(true);

                        // register overflowing cell elements for update of clipping area
                        // in next step (after all matrix cell elements have been updated)
                        if (newOverflowL || newOverflowR) {
                            overflowElementMap.insert(element.key, { element: element });
                        }

                        // ensure to invalidate cell ranges that contained overflowing text before
                        if (oldOverflowL || oldOverflowR) {
                            addOverflowRange(element.address, oldOverflowL, oldOverflowR);
                        }

                        // update adjacent cell elements with text floating towards the updated ranges, if blank state changes
                        if (oldBlank !== newBlank) {
                            neighborElementSet.insert(element);
                        }
                    }, 'Rendercache.registerDirtyRanges.updateMatrix');
                });

                // update origin cells of outer merged ranges overlapping the dirty ranges
                updateTimer = updateTimer.then(RenderUtils.profileAsyncMethod('update origins of merged range', function () {
                    return self.iterateArraySliced(outerMergeDescs, function (mergeDesc) {
                        updateCellElement(mergeDesc.element);
                    }, 'Rendercache.registerDirtyRanges.updateMatrix');
                }));

                // update clipping rectangles of cells with overflowing text (restrict to content cells and merged ranges)
                updateTimer = updateTimer.then(RenderUtils.profileAsyncMethod('update text clipping rectangles', function () {

                    // find adjacent cells with overflowing text (left/right of changed cells)
                    neighborElementSet.forEach(function (element) {
                        registerAdjacentOverflowCell(element, false);
                        registerAdjacentOverflowCell(element, true);
                    });

                    // update clipping rectangles of all registered cells
                    return self.iterateSliced(overflowElementMap.iterator(), function (overflowEntry) {

                        // the matrix cell element to be updated
                        var element = overflowEntry.element;
                        // the inner cell rectangle
                        var innerRect = element.text.inner;
                        // the new clipping positions
                        var clipL = innerRect.left - element.text.overflowL;
                        var clipR = innerRect.right() + element.text.overflowR;
                        // whether left or right text overflows
                        var overflowL = element.hasOverflowText(false);
                        var overflowR = element.hasOverflowText(true);
                        // position of merged range next to the current cell element
                        var mergedRect = null;
                        // column header of adjacent cell element with content value
                        var contentCol = null;

                        // reduce left clipping position according to existing merged ranges and content cell
                        if (overflowL) {
                            if (overflowEntry.l) {
                                contentCol = overflowEntry.l.col;
                            } else {
                                mergedRect = findNextMergedRectangle(element, false);
                                contentCol = findNextContentCol(element, false);
                            }
                            if (mergedRect) { clipL = Math.max(clipL, mergedRect.right()); }
                            if (contentCol) { clipL = Math.max(clipL, contentCol.offset + contentCol.size); }
                        }

                        // reduce right clipping position according to existing merged ranges and content cell
                        if (overflowR) {
                            if (overflowEntry.r) {
                                contentCol = overflowEntry.r.col;
                            } else {
                                mergedRect = findNextMergedRectangle(element, true);
                                contentCol = findNextContentCol(element, true);
                            }
                            if (mergedRect) { clipR = Math.min(clipR, mergedRect.left - GRID_LINE_WIDTH); }
                            if (contentCol) { clipR = Math.min(clipR, contentCol.offset - GRID_LINE_WIDTH); }
                        }

                        // set the new clip position
                        element.text.clip.left = clipL;
                        element.text.clip.width = clipR - clipL;

                        // add overflow range to update ranges for rendering
                        if (overflowL || overflowR) {
                            addOverflowRange(element.address, overflowL, overflowR);
                        }
                    }, 'Rendercache.registerDirtyRanges.updateMatrix');
                }));

                // notify rendering listeners
                updateTimer.done(function () {
                    pendingRanges.clear();
                    if (!updateRanges.empty()) {
                        self.trigger('update:matrix', updateRanges.merge());
                    }
                });

                // restore dirty ranges, if the update cycle will be aborted
                updateTimer.fail(function (cause) {
                    if (cause === 'abort') {
                        dirtyRanges.append(pendingRanges);
                        pendingRanges.clear();
                    }
                });

                return updateTimer; // needed by Logger.profileAsyncMethod()
            });

            // the actual method RenderCache.registerDirtyRanges() to be returned from the local scope
            // (debounced version of RenderCache.registerDirtyRanges(), that waits for document operations currently applied)
            return app.createDebouncedActionsMethodFor(self, 'RenderCache.registerDirtyRanges', registerRanges, updateMatrix);
        }());

        // public methods -----------------------------------------------------

        /**
         * Invokes the passed callback function for all visible columns or rows
         * covered by the passed interval.
         *
         * @param {Range} range
         *  The cell range whose visible columns or rows will be visited.
         *
         * @param {Boolean} columns
         *  Whether to visit the visible columns in the passed range (true), or
         *  the visible rows (false).
         *
         * @param {Function} callback
         *  The callback function invoked for every visible column or row.
         *  Receives the following parameters:
         *  (1) {MatrixHeaderElement} element
         *      The header element of the rendering matrix currently visited.
         *  Will be called in the context of this rendering cache instance.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateInterval = function (range, columns, callback) {
            var header = columns ? colHeader : rowHeader;
            var interval = range.interval(columns);
            header.iterateElements(interval.first, interval.last, callback, this);
            return this;
        };

        /**
         * Visits all merged ranges overlapping with the rendering matrix.
         *
         * @param {Function} callback
         *  The callback function invoked for every merged range overlapping
         *  with the rendering matrix. Receives the following parameters:
         *  (1) {MergeDescriptor} mergeDesc
         *      The merge descriptor, containing the original address of the
         *      merged range, and the address of the merged range shrunken to
         *      the visible area (without leading and trailing hidden columns
         *      and rows).
         *  Will be called in the context of this rendering cache instance.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateMergedRanges = function (callback) {
            mergedRangeSet.forEach(function (mergedRange) {
                callback.call(this, mergedRange.mergeDesc);
            }, this);
            return this;
        };

        /**
         * Invokes the passed callback function for all existing cell elements
         * in the rendering matrix covered by the passed cell range.
         *
         * @param {Range} range
         *  The address of a cell range whose matrix elements will be visited.
         *
         * @param {Function} callback
         *  The callback function invoked for every matrix cell element.
         *  Receives the following parameters:
         *  (1) {MatrixCellElement} element
         *      The cell element from the rendering matrix.
         *  Will be called in the context of this rendering cache instance.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.columns=false]
         *      If set to true, the cells in the passed range will be visited
         *      in vertical orientation (all cells top-down in first column,
         *      then all cells top-down in second column, etc.). By default,
         *      the cells will be visited in horizontal orientation (all cells
         *      left-to-right in first row, then all cells left-to-right in
         *      second row, etc.).
         *  - {Boolean} [options.first=false]
         *      If set to true, only the matrix elements of the first visible
         *      row (or the first visible column in vertical mode, see option
         *      'columns') will be visited.
         *  - {Boolean} [options.origins=false]
         *      If set to true, the origin cells of all merged ranges covering
         *      the passed range will be visited, regardless if they are inside
         *      the passed range; and the cells in the passed range that are
         *      covered by these merged ranges will not be visited.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateCells = function (range, callback, options) {

            // whether to visit the origin cells of merged range sonly
            var origins = Utils.getBooleanOption(options, 'origins', false);

            // invoke passed callback function for the cells in all ranges
            var outerMergeDescs = iterateCellsInRanges(range, function (element) {
                if (!origins || !element.isMergedCovered()) {
                    callback.call(self, element);
                }
            }, options);

            // visit origins of all merged ranges starting outside
            if (origins) {
                outerMergeDescs.forEach(function (mergeDesc) {
                    callback.call(self, mergeDesc.element);
                });
            }

            return this;
        };

        /**
         * Invokes the passed callback function for all existing cell elements
         * in the specified column of the rendering matrix.
         *
         * @param {Number} col
         *  The zero-based column index.
         *
         * @param {Function} callback
         *  The callback function invoked for every matrix cell element in the
         *  column. Receives the following parameters:
         *  (1) {MatrixCellElement} element
         *      The cell element from the rendering matrix.
         *  Will be called in the context of this rendering cache instance.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateCellsInCol = function (col, callback) {
            return this.iterateCells(docModel.makeColRange(col), callback);
        };

        /**
         * Invokes the passed callback function for all existing cell elements
         * in the specified row of the rendering matrix.
         *
         * @param {Number} row
         *  The zero-based row index.
         *
         * @param {Function} callback
         *  The callback function invoked for every matrix cell element in the
         *  column. Receives the following parameters:
         *  (1) {MatrixCellElement} element
         *      The cell element from the rendering matrix.
         *  Will be called in the context of this rendering cache instance.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateCellsInRow = function (row, callback) {
            return this.iterateCells(docModel.makeRowRange(row), callback);
        };

        /**
         * Invokes the passed callback function for all existing cell elements
         * with text contents in the rendering matrix covered by the passed
         * cell range.
         *
         * @param {Range} range
         *  The address of a cell range whose matrix elements will be visited.
         *
         * @param {Function} callback
         *  The callback function invoked for every matrix cell element.
         *  Receives the following parameters:
         *  (1) {MatrixCellElement} element
         *      The cell element from the rendering matrix. Each visited matrix
         *      element will contain a property 'text' referring to a valid
         *      text descriptor.
         *  Will be called in the context of this rendering cache instance.
         *
         * @returns {RenderCache}
         *  A reference to this instance.
         */
        this.iterateTextCells = function (range, callback) {

            function findNextOverflowElement(element, forward) {
                var prop = forward ? 'r' : 'l';
                for (element = element[prop]; element && element.blank && !element.isMerged(); element = element[prop]) {}
                return (element && !element.isMerged() && element.hasOverflowText(!forward)) ? element : null;
            }

            // visit all matrix cell elements in the range
            this.iterateCells(range, function (element) {

                // invoke the callback function for text elements
                if (element.text) {
                    callback.call(self, element);
                    return;
                }

                // do nothing else for non-blank cells without visible display text
                if (!element.blank) { return; }

                // blank cell in first column: invoke callback function for text floating from left
                var prevElement = null;
                if (!element.l || (element.l.address[0] < range.start[0])) {
                    if ((prevElement = findNextOverflowElement(element, false))) {
                        callback.call(self, prevElement);
                    }
                }

                // blank cell in last column: invoke callback function for text floating from right
                var nextElement = null;
                if (!element.r || (element.r.address[0] > range.end[0])) {
                    if ((nextElement = findNextOverflowElement(element, true))) {
                        callback.call(self, nextElement);
                    }
                }
            }, { origins: true });

            return this;
        };

        /**
         * Returns whether this cache currently contains pending dirty ranges
         * overlapping with the passed cell range.
         *
         * @param {Range} range
         *  The range to be compared with the pending dirty ranges.
         *
         * @returns {Boolean}
         *  Whether this cache currently contains pending dirty ranges
         *  overlapping with the passed cell range.
         */
        this.hasDirtyRanges = function (range) {
            return dirtyRanges.overlaps(range) || pendingRanges.overlaps(range);
        };

        /**
         * Updates the layer ranges of this rendering cache according to the
         * passed bounding intervals, and returns descriptors for all changed
         * layer ranges.
         *
         * @param {ValueMap<HeaderBoundary>} headerBoundaryMap
         *  All changed bounding intervals, mapped by pane side identifiers.
         *  The boundary intervals of pane sides missing in this map will not
         *  be changed. If a boundary in this map exists and is invalid (all of
         *  its properties are null), the respective layer ranges will be
         *  removed from the rendering cache.
         *
         * @param {RangeArray} invalidRanges
         *  An array with the addresses of all dirty cell ranges in the sheet
         *  that must be recalculated and repainted (e.g. after one or more
         *  document operations such as inserting or deleting columns or rows).
         *
         * @returns {ValueMap<RangeBoundary>}
         *  A map containing descriptors for all changed layer ranges, mapped
         *  by pane position identifier. The boundary ranges of pane positions
         *  missing in this map have not been changed. If a descriptor in this
         *  map exists and is invalid (all of its properties are null), the
         *  layer range has been removed from the rendering cache.
         */
        this.updateLayerRanges = RenderUtils.profileMethod('RenderCache.updateLayerRanges()', function (headerBoundaryMap, invalidRanges) {

            // the model of the active sheet
            var sheetModel = docView.getSheetModel();
            // the column and row collections of the active sheet
            var colCollection = sheetModel.getColCollection();
            var rowCollection = sheetModel.getRowCollection();
            // flags for changed intervals (per pane side), and changed columns/rows
            var changedSides = {};
            var changedCols = false;
            var changedRows = false;
            // the pane positions of all changed bounding ranges
            var changedPanes = [];
            // all cell ranges to be refreshed in the rendering matrix
            var refreshRanges = new RangeArray();
            // the changed range boundaries returned to caller
            var resultBoundaryMap = new ValueMap();

            headerBoundaryMap.forEach(function (boundary, paneSide) {

                // whether to change a column or row interval
                var columns = PaneUtils.isColumnSide(paneSide);
                // the column/row collection
                var collection = columns ? colCollection : rowCollection;
                // the header elements of the rendering matrix
                var header = columns ? colHeader : rowHeader;
                // the passed interval, shrunken to visible columns/rows
                var shrunkenInterval = boundary.isValid() ? collection.shrinkIntervalToVisible(boundary.interval) : null;

                // store the new bounding intervals
                if (shrunkenInterval) {
                    boundary.interval = shrunkenInterval;
                    header.boundaries.insert(paneSide, boundary);
                } else if (!header.boundaries.remove(paneSide)) {
                    return;
                }

                // set the correct change marker flags
                changedSides[paneSide] = true;
                if (columns) { changedCols = true; } else { changedRows = true; }
            });

            // nothing more to do, if no bounding intervals have changed at all
            if (!changedCols && !changedRows) { return resultBoundaryMap; }

            // calculate the unified bounding intervals (shrunken to visible parts)
            colHeader.intervals = colCollection.mergeAndShrinkIntervals(colHeader.boundaries.pluck('interval'));
            rowHeader.intervals = rowCollection.mergeAndShrinkIntervals(rowHeader.boundaries.pluck('interval'));

            // calculate the new bounding ranges for all combinations of column/row intervals
            PaneUtils.ALL_PANE_POSITIONS.forEach(function (panePos) {

                // column/row side identifier
                var colPaneSide = PaneUtils.getColPaneSide(panePos);
                var rowPaneSide = PaneUtils.getRowPaneSide(panePos);

                // nothing to do, if neither column nor row interval has changed
                if (!changedSides[colPaneSide] && !changedSides[rowPaneSide]) { return; }

                // current column/row interval and position
                var colBoundary = colHeader.boundaries.get(colPaneSide, null);
                var rowBoundary = rowHeader.boundaries.get(rowPaneSide, null);

                // update bounding range
                if (colBoundary && rowBoundary) {
                    boundaryMap.insert(panePos, new PaneUtils.RangeBoundary(colBoundary, rowBoundary));
                    changedPanes.push(panePos);
                } else {
                    boundaryMap.remove(panePos);
                    changedPanes.push(panePos);
                }
            });

            // calculate unified bounding ranges, get total cell range
            boundRanges = new RangeArray(boundaryMap.pluck('range')).merge();
            totalRange = boundRanges.boundary();
            RenderUtils.withLogging(function () { RenderUtils.log('bounding ranges: ' + boundRanges); });

            // update settings in auto style cache
            styleCache.updateStyles();

            // update column and row headers in the rendering matrix (delete unused cell elements, insert new cell elements)
            RenderUtils.takeTime('create/delete matrix elements', function () {
                if (changedCols) { refreshRanges.append(buildMatrixElements(true)); }
                if (changedRows) { refreshRanges.append(buildMatrixElements(false)); }
            });

            // nothing more to update without existing bounding ranges
            if (totalRange) {

                // add dirty ranges passed to this method
                refreshRanges.append(invalidRanges);

                // Refresh the cached merged ranges, if either the bounding row intervals have changed,
                // or if columns or rows have been changed in the document (especially the visibility
                // of columns or rows). There is no need to update them when the column intervals have
                // changed without any changes in the document (e.g. while scrolling horizontally),
                // because all merged ranges from the entire bounding rows will be cached.
                if (!invalidRanges.empty() || changedRows) {
                    refreshRanges.append(updateMergedRanges());
                }

                // register the ranges to be refreshed (there may already be other pending dirty
                // ranges received from 'change:cells' events).
                registerDirtyRanges(refreshRanges);

            } else {
                // no more bounding ranges available: reset all pending dirty ranges immediately
                abortUpdateCycle();
                mergedRangeSet.clear();
                dirtyRanges.clear();
            }

            // return a map with the changed range boundaries to caller
            changedPanes.forEach(function (panePos) {
                var boundary = boundaryMap.get(panePos);
                resultBoundaryMap.insert(panePos, boundary ? boundary.clone() : new PaneUtils.RangeBoundary());
            });
            return resultBoundaryMap;
        });

        // initialization -----------------------------------------------------

        // update all cells that have been changed in the active sheet
        this.listenTo(docView, 'change:cells', RenderUtils.profileMethod('RenderCache.listenTo(change:cells)', function (event, changeDesc) {
            // the changed cell ranges
            var changedRanges = changeDesc.getAllRanges();
            registerDirtyRanges(changedRanges);
            // add all merged ranges starting at the changed cells
            registerDirtyRanges(docView.getMergeCollection().getMergedRanges(changedRanges, 'reference'));
        }));

        // update all cell ranges that have been invalidated in the active sheet
        this.listenTo(docView, 'refresh:ranges', RenderUtils.profileMethod('RenderCache.listenTo(refresh:ranges)', function (event, ranges) {
            registerDirtyRanges(ranges);
        }));

        // update the cached merged ranges after new merged ranges have been inserted into the
        // active sheet, or after existing merged ranges have been deleted from the active sheet
        this.listenTo(docView, 'insert:merged delete:merged', RenderUtils.profileMethod('RenderCache.listenTo(insert:merged delete:merged)', function () {
            registerDirtyRanges(updateMergedRanges());
        }));

        // render cells of inserted and deleted table ranges (table styling)
        this.listenTo(docView, 'insert:table delete:table', function (event, tableModel) {
            registerDirtyRanges(tableModel.getRange());
        });

        // render table ranges with changed attributes (table styling)
        this.listenTo(docView, 'change:table', function (event, tableModel, type) {
            if (type === 'change:attributes') {
                registerDirtyRanges(tableModel.getRange());
            }
        });

        // refresh all table ranges using a changed table style sheet
        this.listenOnceTo(docModel, 'change:activesheet', function () {
            self.listenTo(docModel.getTableStyles(), 'triggered', function (event, type, styleId) {
                IteratorUtils.forEach(docView.getTableCollection().createModelIterator(), function (tableModel) {
                    if (tableModel.getStyleId() === styleId) {
                        registerDirtyRanges(tableModel.getRange());
                    }
                });
            });
        });

        // update the rendering cache, after the grid color of the active sheet has been changed
        // (cell borders with automatic line color are using the current sheet grid color)
        this.listenTo(docView, 'change:sheet:attributes', function (event, newAttributes, oldAttributes) {
            if (!_.isEqual(newAttributes.sheet.gridColor, oldAttributes.sheet.gridColor)) {
                styleCache.updateStyles();
                if (totalRange) { registerDirtyRanges(totalRange); }
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            styleCache.destroy();
            self = docView = docModel = app = null;
            styleCache = cellMatrix = colHeader = rowHeader = null;
            boundaryMap = boundRanges = totalRange = null;
            mergedRangeSet = dirtyRanges = pendingRanges = null;
        });

    } }); // class RenderCache

    // exports ================================================================

    return RenderCache;

});
