/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/highlightrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/rendererbase'
], function (Utils, SheetUtils, PaneUtils, RenderUtils, RendererBase) {

    'use strict';

    // convenience shortcuts
    var RangeArray = SheetUtils.RangeArray;

    // class HighlightRenderer ================================================

    /**
     * Renders the highlighted cell ranges (shown e.g. during editing formulas
     * with references into the active sheet) into the DOM layers of a single
     * grid pane.
     *
     * @constructor
     *
     * @extends RendererBase
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this highlighting layer renderer.
     */
    var HighlightRenderer = RendererBase.extend({ constructor: function (gridPane) {

        // self reference
        var self = this;

        // the spreadsheet view
        var docView = gridPane.getDocView();

        // the range highlight layer (container for the highlighted ranges of formulas)
        var highlightLayerNode = gridPane.createLayerNode('highlight-layer');

        // the active range layer (container for the active range selected in cell edit mode)
        var activeRangeLayerNode = gridPane.createLayerNode('active-layer');

        // the cross-hair frame selection layer
        var frameSelectLayerNode = gridPane.createLayerNode('frame-select-layer');
        var frameLineLNode = $('<div class="frame-line v">').appendTo(frameSelectLayerNode);
        var frameLineRNode = $('<div class="frame-line v">').appendTo(frameSelectLayerNode);
        var frameLineTNode = $('<div class="frame-line h">').appendTo(frameSelectLayerNode);
        var frameLineBNode = $('<div class="frame-line h">').appendTo(frameSelectLayerNode);
        var frameRectNode = $('<div class="frame-rect">').appendTo(frameSelectLayerNode);

        // animation timer for marching-ants effect
        var animationTimer = null;

        // base constructor ---------------------------------------------------

        RendererBase.call(this, gridPane);

        // private methods ----------------------------------------------------

        /**
         * Renders all visible highlighted cell ranges.
         */
        function renderHighlightedRanges() {

            // the HTML mark-up for all highlight ranges
            var markup = '';

            // render all ranges that are visible in this grid pane
            if (gridPane.isVisible()) {

                // current document edit mode
                var editMode = docView.isEditable();
                // the index of the active sheet
                var sheet = docView.getActiveSheet();
                // all range descriptors of the active sheet
                var highlightRanges = [];
                // the cell range addresses of all ranges to be rendered
                var ranges = new RangeArray();
                // maps range keys to zero-based color indexes
                var rangeColors = {};
                // next free zero-based color index
                var nextColor = 0;

                // collect all range descriptors of the active sheet
                docView.iterateHighlightedRanges(function (rangeDesc, rangeIndex) {
                    if (rangeDesc.range.containsSheet(sheet)) {
                        highlightRanges.push({ rangeDesc: rangeDesc, rangeIndex: rangeIndex });
                        ranges.push(rangeDesc.range.toRange()); // push simple range without sheet indexes
                    }
                });

                // render the highlighted ranges
                gridPane.iterateRangesForRendering(ranges, function (range, rectangle, index) {

                    // the current range descriptor
                    var rangeDesc = highlightRanges[index].rangeDesc;
                    // the original array index of the current range descriptor
                    var rangeIndex = highlightRanges[index].rangeIndex;
                    // the unique key of the formula token containing the range (will be resolved to a scheme color)
                    var tokenKey = rangeDesc.tokenKey;
                    // whether the range will be rendered with tracking handle elements
                    var draggable = editMode && rangeDesc.draggable;
                    // whether the range is currently tracked (rendered with special glow effect)
                    var tracking = draggable && rangeDesc.tracking;

                    // register a new range key in the 'rangeColors' map
                    if (!(tokenKey in rangeColors)) {
                        rangeColors[tokenKey] = nextColor;
                        nextColor += 1;
                    }

                    // one-based index of the scheme color to be used for the range
                    var styleId = (rangeColors[tokenKey] % Utils.SCHEME_COLOR_COUNT) + 1;

                    // adjust position of static ranges
                    if (!draggable) { rectangle.expandSelf(1, 1, 0, 0); }

                    // generate the HTML mark-up for the highlighted range
                    markup += '<div class="range' + (tracking ? ' tracking-active' : '') + '" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"';
                    markup += ' data-style="' + styleId + '" data-index="' + rangeIndex + '"><div class="fill"></div>';
                    if (draggable) {
                        markup += '<div class="borders">';
                        markup += _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div><div class="resizers">';
                        markup += _.map(['tl', 'tr', 'bl', 'br'], function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                        markup += '</div>';
                    } else {
                        markup += '<div class="static-border"></div>';
                    }
                    markup += '</div>';
                });
            }

            // insert entire HTML mark-up into the container node
            highlightLayerNode[0].innerHTML = markup;
        }

        /**
         * Renders the activated cell ranges. These ranges are used for example
         * while selecting new ranges for a formula while in cell edit mode.
         */
        function renderActiveSelection() {

            // the active ranges
            var selection = docView.getSheetModel().getViewAttribute('activeSelection');
            // the HTML mark-up for the range
            var markup = '';

            // generate the HTML mark-up for the ranges
            if (gridPane.isVisible() && selection) {
                gridPane.iterateRangesForRendering(selection.ranges, function (range, rectangle) {
                    markup += '<div class="range" style="' + PaneUtils.getRectangleStyleMarkup(rectangle) + '"><div class="borders">';
                    markup += _.map('lrtb', function (pos) { return '<div data-pos="' + pos + '"></div>'; }).join('');
                    markup += '</div></div>';
                });
                if (!animationTimer) {
                    animationTimer = self.repeatDelayed(function (index) {
                        activeRangeLayerNode.attr('data-frame', index % 6);
                    }, 'HighlightRenderer.renderActiveSelection', { delay: 100, background: true });
                }
            } else if (animationTimer) {
                animationTimer.abort();
                animationTimer = null;
            }

            // insert entire HTML mark-up into the container node
            activeRangeLayerNode[0].innerHTML = markup;
        }

        /**
         * Renders the cross-hair and rectangle for activated frame selection
         * mode.
         *
         * @param {Rectangle} frameRect
         *  The rectange (absolute sheet location) to be rendered, in pixels.
         */
        function renderFrameSelection(frameRect) {

            // hide DOM layer and exit, if frame selection mode is not active
            frameSelectLayerNode.toggleClass('active', !!frameRect);
            if (!frameRect) { frameRectNode.empty(); return; }

            // the location of the frame rectangle, relative to the layer nodes
            var layerRect = gridPane.convertToLayerRectangle(frameRect);
            var isVisible = layerRect.area() > 0;

            // set the effective coordinates to the frame nodes
            frameLineLNode.css('left', layerRect.left);
            frameLineRNode.toggle(layerRect.width > 0).css('left', layerRect.right() - 1);
            frameLineTNode.css('top', layerRect.top);
            frameLineBNode.toggle(layerRect.height > 0).css('top', layerRect.bottom() - 1);
            frameRectNode.toggle(isVisible).css(layerRect.toCSS());

            // notify render listeners
            layerRect.reverseX = frameRect.reverseX;
            layerRect.reverseY = frameRect.reverseY;
            if (isVisible) { self.trigger('render:frameselection', frameRectNode, layerRect); }
        }

        // protected methods --------------------------------------------------

        /**
         * Changes the layers according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('HighlightRenderer.setLayerRange()', function () {
            renderHighlightedRanges();
            renderActiveSelection();
        });

        /**
         * Resets this renderer, clears the DOM layer nodes.
         */
        this.hideLayerRange = function () {
            highlightLayerNode.empty();
            activeRangeLayerNode.empty();
        };

        // initialization -----------------------------------------------------

        // update highlighting when document edit mode or view attributes have changed
        this.listenToWhenVisible(docView.getApp(), 'docs:editmode', renderHighlightedRanges);
        this.listenToWhenVisible(docView, 'change:highlight:ranges', renderHighlightedRanges);

        // render the cross-hair or the frame rectangle while frame selection mode is active
        this.listenToWhenVisible(docView, 'change:frameselect', function (event, frameRect) {
            renderFrameSelection(frameRect);
        });

        // render the active selection of the sheet after it has been changed
        this.listenToWhenVisible(docView, 'change:sheet:viewattributes', function (event, attributes) {
            if ('activeSelection' in attributes) {
                renderActiveSelection();
            }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = gridPane = docView = null;
            highlightLayerNode = activeRangeLayerNode = frameSelectLayerNode = null;
            animationTimer = null;
        });

    } }); // class HighlightRenderer

    // exports ================================================================

    return HighlightRenderer;

});
