/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/render/formrenderer', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/rendererbase',
    'io.ox/office/spreadsheet/view/popup/tablecolumnmenu',
    'io.ox/office/spreadsheet/view/popup/validationlistmenu'
], function (Utils, KeyCodes, Forms, IteratorUtils, ValueMap, Rectangle, SheetUtils, RenderUtils, RendererBase, TableColumnMenu, ValidationListMenu) {

    'use strict';

    // convenience shortcuts
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;

    // class FormRenderer =====================================================

    /**
     * Renders the form controls (especially cell drop-down buttons) into the
     * DOM layers of a single grid pane.
     *
     * @constructor
     *
     * @extends RendererBase
     *
     * @param {GridPane} gridPane
     *  The grid pane instance that owns this form layer renderer.
     */
    var FormRenderer = RendererBase.extend({ constructor: function (gridPane) {

        // self reference
        var self = this;

        // the spreadsheet view
        var docView = gridPane.getDocView();

        // the form control layer (container for drop-down buttons and other controls)
        var layerNode = gridPane.createLayerNode('form-layer');

        // the cell range covered by the layer nodes in the sheet area
        var layerRange = null;

        // all registered embedded cell pop-up menus, mapped by identifier
        var menuRegistry = new ValueMap();

        // the menu identifiers in insertion order
        var orderedMenuIds = [];

        // the cell button element caused to open the active cell menu
        var activeCellButton = null;

        // the embedded cell menu instance currently activated (visible)
        var activeCellMenu = null;

        // base constructor ---------------------------------------------------

        RendererBase.call(this, gridPane);

        // private methods ----------------------------------------------------

        /**
         * Registers a cell menu instance to be shown for specific drop-down
         * buttons embedded in the active sheet.
         *
         * @param {Function} MenuClass
         *  The constructor of the pop-up menu class that will be shown when
         *  clicking the specified cell drop-down buttons. MUST be a subclass
         *  of BaseMenu. The menu may provide a public method initialize() that
         *  will be invoked before opening the menu. That method will receive
         *  the following parameters:
         *  (1) {Address} address
         *      The address of the cell containing the button node that caused
         *      opening the menu.
         *  (2) {Any} userData
         *      Additional user data that have been passed to the method
         *      FormRenderer.createCellButton().
         *  The method may return a promise indicating that initialization of
         *  the drop-down menu has been finished. The promise returned by the
         *  method may be abortable; in this case, when opening the drop-down
         *  menu repeatedly, or opening another cell drop-down menu quickly,
         *  old initialization requests will be aborted, before calling the
         *  initialization callback function again.
         */
        function createCellMenu(menuId, MenuClass) {

            // create the menu, and add it to the registry map
            var cellMenu = new MenuClass(docView);
            menuRegistry.insert(menuId, cellMenu);
            orderedMenuIds.push(menuId);

            // only the active button must be included in the focus handling of the menu
            cellMenu.registerFocusableNodes(function () { return activeCellButton; });

            // dynamic anchor position according to the clicked drop-down button
            cellMenu.setAnchor(function () {

                // current page position of the button node
                var buttonRect = Utils.getNodePositionInPage(activeCellButton);
                // the width of cell and button rectangle, used for anchor of the drop-down menu node
                var anchorWidth = activeCellButton.data('anchorWidth');

                // create the anchor rectangle according to the current position of the button node
                return new Rectangle(
                    buttonRect.left + buttonRect.width - anchorWidth,
                    buttonRect.top,
                    anchorWidth,
                    buttonRect.height
                );
            });

            // hide the menu automatically, if the anchor leaves the visible area of the grid pane
            cellMenu.setAnchorBox(gridPane.getScrollNode());

            // remember active menu for usage in other methods
            self.listenTo(cellMenu, 'popup:show', function () {
                activeCellButton.addClass('dropdown-open');
                activeCellMenu = cellMenu;
            });

            // abort initialization if pop-up menu will be closed quickly
            self.listenTo(cellMenu, 'popup:hide', function () {
                activeCellButton.removeClass('dropdown-open');
                activeCellMenu = null;
            });
        }

        /**
         * Opens or closes the embedded cell pop-up menu associated to the
         * passed cell drop-down button.
         *
         * @param {jQuery} buttonNode
         *  The cell button node whose menu will be opened or closed.
         */
        var toggleCellMenu = (function () {

            // current asynchronous initialization of the cell menu contents
            var initPromise = null;

            // aborts current menu initialization
            function abortInitialization() {
                if (initPromise && initPromise.abort) {
                    initPromise.abort();
                }
            }

            // the actual toggleCellMenu() method to be returned from local scope
            function toggleCellMenu(buttonNode) {

                // check existence of menu registration
                var cellMenu = buttonNode.data('menu');
                if (!cellMenu) {
                    Utils.warn('FormRenderer.toggleCellMenu(): missing menu registration for clicked cell drop-down button');
                    return;
                }

                // abort another initialization of a cell menu currently running
                abortInitialization();

                // always leave text edit mode (do not open the menu, if formula validation fails)
                docView.leaveTextEditMode().done(function () {

                    // if the menu is currently visible, close it and return
                    if (cellMenu.isVisible()) {
                        cellMenu.hide();
                        docView.grabFocus();
                        return;
                    }

                    // remember current button node, will be returned as focusable node to prevent
                    // closing the menu too fast (the menu may be associated to different buttons)
                    activeCellButton = buttonNode;

                    // invoke initialization callback handler
                    var initResult = null;
                    if (typeof cellMenu.initialize === 'function') {
                        initResult = cellMenu.initialize(buttonNode.data('address'), buttonNode.data('userData'));
                    }

                    // show the menu (unless the promise has already been rejected synchronously)
                    initPromise = self.convertToPromise(initResult);
                    if (initPromise.state() !== 'rejected') { cellMenu.busy().show(); }

                    // show warning alerts if available
                    docView.yellOnFailure(initPromise);

                    // further initialization for the open menu
                    if (cellMenu.isVisible()) {
                        self.waitForFailure(initPromise, function () { cellMenu.hide(); });
                        cellMenu.one('popup:hide', abortInitialization);
                    }

                    // final clean-up
                    self.waitForAny(initPromise, function () {
                        cellMenu.idle();
                        initPromise = null;
                    });
                });
            }

            return toggleCellMenu;
        }());

        /**
         * Tries to activate an embedded cell pop-up menu attached to the
         * specified cell. If the cell contains multiple drop-down menus, the
         * method searches for a cell menu currently activated for that cell,
         * and activates the next registered menu. The order of the menus is
         * equal to the registration order via the method registerCellMenu().
         *
         * @param {Address} address
         *  The address of a cell to open an embedded pop-up menu for.
         *
         * @returns {Boolean}
         *  Whether a drop-down menu has been found and activated for the cell.
         */
        function activateCellMenu(address) {

            // all cell button nodes attached to the specified cell (nothing to do, if no cell drop-down button found)
            var cellButtons = getCellButtons(address);
            if (cellButtons.length === 0) { return false; }

            // always scroll to the cell containing the pop-up menu
            gridPane.scrollToCell(address);

            // filter the menus of the existing cell buttons in registration order
            var menuIds = orderedMenuIds.filter(function (menuId) {
                return cellButtons.filter('[data-menu="' + menuId + '"]').length > 0;
            });

            // get type of the pop-up menu currently opened and attached to the specified cell
            var isMatchingAddress = activeCellButton && activeCellMenu && activeCellButton.data('address').equals(address);
            var activeMenuId = isMatchingAddress ? activeCellButton.attr('data-menu') : null;

            // do nothing if the active menu is the only menu for the cell
            if (activeMenuId && (menuIds.length === 1)) { return true; }

            // always hide the active cell menu before opening the next menu
            hideActiveCellMenu();

            // pick the next menu type from the list of available menu types
            activeMenuId = menuIds[(menuIds.indexOf(activeMenuId) + 1) % menuIds.length];

            // open the menu
            toggleCellMenu(cellButtons.filter('[data-menu="' + activeMenuId + '"]'));
            return true;
        }

        /**
         * Hides the cell pop-up menu currently shown.
         *
         * @returns {Boolean}
         *  Whether a cell list menu was actually open and has been hidden.
         */
        function hideActiveCellMenu() {
            if (activeCellMenu && activeCellMenu.isVisible()) {
                activeCellMenu.hide();
                return true;
            }
            return false;
        }

        /**
         * Returns all cell drop-down buttons attached to the specified cell.
         *
         * @param {Address} address
         *  The address of the cell whose drop-down menu buttons will be
         *  returned.
         *
         * @returns {jQuery}
         *  A jQuery collection with all cell drop-down menu buttons at the
         *  passed cell address.
         */
        function getCellButtons(address) {
            return layerNode.find('>.cell-dropdown-button[data-address="' + address.key() + '"]');
        }

        /**
         * Removes all cell drop-down button elements of the specified menu.
         *
         * @param {String} menuId
         *  The identifier of the pop-up menu whose existing drop-down buttons
         *  will be removed.
         */
        function removeCellButtons(menuId) {
            layerNode.find('>.cell-dropdown-button[data-menu="' + menuId + '"]').remove();
        }

        /**
         * Creates a drop-down button element intended to be attached to the
         * trailing border of the specified cell, and inserts it into the form
         * control layer.
         *
         * @param {BaseMenu} cellMenu
         *  A registered pop-up cell menu. If the created button will be
         *  clicked, the menu will be shown.
         *
         * @param {Address} address
         *  The address of the cell the button will be associated to.
         *
         * @param {Rectangle} rectangle
         *  The position of the cell range the button is attached to (pixels).
         *  This rectangle may be larger than the area covered by the specified
         *  cell, e.g. if the cell is part of a merged range, and the drop-down
         *  button has to be displayed in the bottom-right corner of the range.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.icon]
         *      A specific icon to be shown in the button, instead of the
         *      default drop-down caret icon.
         *  @param {String} [options.tooltip]
         *      A tool tip text to be shown when the mouse hovers the drop-down
         *      button.
         *  @param {Boolean} [options.outside=false]
         *      If set to true, the button will be positioned outside the cell
         *      rectangle. By default, the button will be positioned inside.
         *  @param {Number} [options.distance=0]
         *      Distance of the button to the border of the cell rectangle.
         *  @param {Any} [options.userData]
         *      Any additional data that will be added to the jQuery data map
         *      of the button element, and that can be used during the
         *      initialization phase or event handling of pop-up menus
         *      associated to the drop-down button.
         *
         * @returns {jQuery}
         *  The created and inserted button element, as jQuery object. The
         *  jQuery data map of the button element will contain the following
         *  properties:
         *  - {Address} address
         *      The passed cell address.
         *  - {Number} anchorWidth
         *      The effective total width of the passed rectangle and the
         *      button rectangle.
         *  - {Object} [userData]
         *      The user data passed with the option 'userData'.
         */
        function createCellButton(menuId, address, rectangle, options) {

            // whether to place the button element outside the cell rectangle
            var outside = Utils.getBooleanOption(options, 'outside', false);
            // distance of the button element to the cell rectangle
            var distance = Utils.getIntegerOption(options, 'distance', 0, 0);
            // additional properties for the jQuery data map
            var userData = Utils.getOption(options, 'userData');
            // outer size of the button element (according to cell height)
            var size = Utils.minMax(rectangle.height - 1, 18, 30);
            // size of the sheet area hidden by frozen split
            var hiddenSize = gridPane.getHiddenSize();
            // position of the button in the sheet area
            var buttonRect = new Rectangle(
                Math.max(rectangle.right() + (outside ? distance : -(size + 1)), hiddenSize.width),
                Math.max(rectangle.bottom() - size - 1, hiddenSize.height),
                size,
                size
            );
            // the anchor rectangle to be used for attached drop-down menu
            var anchorRect = rectangle.boundary(buttonRect);

            // create button mark-up
            var markup = '<div class="cell-dropdown-button skip-tracking" tabindex="-1" data-menu="' + menuId + '" data-address="' + address.key() + '"';
            markup += ' style="' + gridPane.getLayerRectangleStyleMarkup(buttonRect) + 'line-height:' + (buttonRect.height - 2) + 'px;">';
            markup += Forms.createIconMarkup(Utils.getStringOption(options, 'icon', 'fa-caret-down')) + '</div>';

            // create the button element
            var buttonNode = $(markup).data({ menu: menuRegistry.get(menuId), address: address, anchorWidth: anchorRect.width, userData: userData });
            Forms.setToolTip(buttonNode, options);
            layerNode.append(buttonNode);

            // handle button clicks, suppress double-clicks
            buttonNode.on('click', function () { toggleCellMenu(buttonNode); });
            buttonNode.on('dblclick', false);
            return buttonNode;
        }

        /**
         * Creates the form controls (cell drop-down buttons) currently visible
         * in this grid pane.
         */
        var renderFormControlsDebounced = this.createDebouncedMethod('FormRenderer.renderFormControlsDebounced', null, function () {

            // nothing to do, if all columns/rows are hidden (method may be called debounced)
            if (!gridPane.isVisible()) {
                layerNode.empty();
                return;
            }

            // remove all filter buttons, do not render them in locked sheets
            removeCellButtons('table');
            if (docView.isSheetLocked()) { return; }

            // the collections of the active sheet
            var colCollection = docView.getColCollection();
            var rowCollection = docView.getRowCollection();

            // create drop-down buttons for all visible table ranges with activated filter/sorting
            docView.getTableCollection().findTables(layerRange).forEach(function (tableModel) {

                // bug 51782, bug 51456: do not render buttons without header row, or if buttons are disabled explicitly
                if (!tableModel.areButtonsVisible()) { return; }

                // the cell range covered by the table
                var tableRange = tableModel.getRange();
                // the settings of the header row
                var rowDesc = rowCollection.getEntry(tableRange.start[1]);

                // do nothing if the header row is outside the layer range, or hidden
                if ((tableRange.start[1] < layerRange.start[1]) || (rowDesc.size === 0)) { return; }

                // visit all visible columns of the table in the layer range
                var colInterval = tableRange.colInterval().intersect(layerRange.colInterval());
                var colIt = colCollection.createIterator(colInterval, { visible: true });
                IteratorUtils.forEach(colIt, function (colDesc) {

                    // relative table column index
                    var tableCol = colDesc.index - tableRange.start[0];
                    var columnModel = tableModel.getColumnModel(tableCol);

                    // bug 36152: do not render buttons covered by merged ranges
                    if (!columnModel.isButtonVisible()) { return; }

                    // bug 36152: visible drop-down button in a merged range is placed in the last column of the
                    // merged range, but is associated to (shows cell data of) the first column in the merged range
                    while ((tableCol > 0) && !tableModel.getColumnModel(tableCol - 1).isButtonVisible()) {
                        tableCol -= 1;
                    }

                    var address = new Address(colDesc.index, rowDesc.index);
                    var rectangle = new Rectangle(colDesc.offset, rowDesc.offset, colDesc.size, rowDesc.size);

                    // determine the custom filter/sort icon for the drop-down button
                    var isFiltered = columnModel.isFiltered();
                    var isSorted = columnModel.isSorted();
                    var direction = columnModel.isDescending() ? 'down' : 'up';
                    var icon = (isFiltered && isSorted) ? ('docs-sort-' + direction + '-filter') : isFiltered ? 'docs-filter' : isSorted ? ('docs-sort-' + direction) : null;

                    createCellButton('table', address, rectangle, {
                        tooltip: TableColumnMenu.BUTTON_TOOLTIP,
                        icon: icon,
                        userData: { tableModel: tableModel, tableCol: tableCol }
                    });
                });
            });
        });

        /**
         * Creates or updates the drop-down button for list validation, after
         * the selection layer has been rendered.
         */
        function renderValidationButton() {

            // remove the old button element
            removeCellButtons('validation');

            // model of the active sheet
            var sheetModel = docView.getSheetModel();
            // do not show in read-only mode, or while using auto-fill feature, or in locked cells
            if (!docView.isEditable() || sheetModel.getViewAttribute('autoFillData')) { return; }

            // do not show the drop-down button in locked cells
            var msgCode = docView.ensureUnlockedActiveCell({ lockMatrixes: 'full', sync: true });
            if (msgCode !== null) { return; }

            // the current selection in the document view
            var selection = docView.getSelection();
            // active cell in the current selection
            var activeCell = selection.address;
            // active cell, expanded to a merged range
            var activeRange = sheetModel.getMergeCollection().expandRangeToMergedRanges(new Range(activeCell));

            // check that the active range is inside the row interval covered by this grid pane (needed to prevent
            // painting the upper part of the button, if cell is located in the top row of the lower part of a frozen split)
            var availableInterval = gridPane.getRowHeaderPane().getAvailableInterval();
            if (!availableInterval || (activeRange.end[1] < availableInterval.first)) { return; }

            // validation settings of the active cell in the selection
            var validationSettings = sheetModel.getValidationCollection().getValidationSettings(activeCell);

            // check that the active cell provides a validation drop-down list
            if (!validationSettings || !/^(source|list)$/.test(validationSettings.attributes.type) || !validationSettings.attributes.showDropDown) { return; }

            // whether the right border of the active cell touches the right border of any selection range
            var rightBorder = selection.ranges.some(function (range) { return activeRange.end[0] === range.end[0]; });

            // create and insert the button node
            createCellButton('validation', activeCell, docView.getRangeRectangle(activeRange), {
                tooltip: ValidationListMenu.BUTTON_TOOLTIP,
                outside: true,
                distance: rightBorder ? 2 : 0
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Changes the layers according to the passed layer range.
         */
        this.setLayerRange = RenderUtils.profileMethod('FormRenderer.setLayerRange()', function (layerSettings) {
            layerRange = layerSettings.range;
            renderFormControlsDebounced();
        });

        /**
         * Resets this renderer, clears the DOM layer nodes.
         */
        this.hideLayerRange = function () {
            layerRange = null;
            layerNode.empty();
        };

        /**
         * Handler for 'keydown' events received in the grid pane containing
         * this instance.
         *
         * @param {jQuery.Event} event
         *  The 'keydown' event to be processed.
         *
         * @returns {Boolean}
         *  Whether the key event represents a supported key and has been
         *  processed successfully.
         */
        this.handleKeyDownEvent = function (event) {

            // close cell pop-up menu currently open
            if ((event.keyCode === KeyCodes.ESCAPE) && hideActiveCellMenu()) {
                return true;
            }

            // show cell drop-down menu of active cell on ALT+DOWN
            if (KeyCodes.matchKeyCode(event, 'DOWN_ARROW', { alt: true })) {
                return activateCellMenu(docView.getActiveCell());
            }

            // forward keyboard shortcuts to the cell pop-up menu currently visible
            if (activeCellMenu && activeCellMenu.handleKeyDownEvent) {
                return activeCellMenu.handleKeyDownEvent(event);
            }

            // unsupported key
            return false;
        };

        // initialization -----------------------------------------------------

        // drop-down menu for table columns (sorting and filtering)
        createCellMenu('table', TableColumnMenu);
        // drop-down menu for data validation of active cell
        createCellMenu('validation', ValidationListMenu);

        // hide the active pop-up menu, if the document changes to read-only mode
        this.listenToWhenVisible(docView.getApp(), 'docs:editmode:leave', hideActiveCellMenu);

        // update layers according to changed contents (only, if the grid pane is visible)
        this.listenToWhenVisible(docView, 'change:sheet:attributes insert:table change:table delete:table', renderFormControlsDebounced);

        // hide the active cell pop-up menu after specific attributes have been changed
        this.listenToWhenVisible(docView, 'change:sheet:viewattributes', function (event, attributes) {
            if (Utils.hasProperty(attributes, /^(selection$|split|activePane|autoFillData$)/)) {
                hideActiveCellMenu();
            }
        });

        // render validation drop-down button after selection changes
        this.listenToWhenVisible(gridPane, 'render:cellselection', renderValidationButton);

        // immediately hide the active pop-up menu on selection tracking
        this.listenToWhenVisible(gridPane, 'select:start', hideActiveCellMenu);

        // always hide the active cell pop-up menu when edit mode starts
        this.listenToWhenVisible(docView, 'textedit:enter', hideActiveCellMenu);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            menuRegistry.forEach(function (cellMenu) { cellMenu.destroy(); });
            self = gridPane = docView = layerNode = null;
            menuRegistry = activeCellButton = activeCellMenu = null;
        });

    } }); // class FormRenderer

    // exports ================================================================

    return FormRenderer;

});
