/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/tablecolumnmenu', [
    'io.ox/office/tk/control/radiogroup',
    'io.ox/office/baseframework/view/popup/compoundmenu',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/control/discretefiltergroup',
    'gettext!io.ox/office/spreadsheet/main'
], function (RadioGroup, CompoundMenu, Labels, DiscreteFilterGroup, gt) {

    'use strict';

    // class TableColumnMenu ==================================================

    /**
     * A drop-down menu for a header cell in a table range, containing options
     * for sorting and filtering.
     *
     * @constructor
     *
     * @extends CompoundMenu
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this instance.
     */
    var TableColumnMenu = CompoundMenu.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // the model of the table range represented by this menu
        var tableModel = null;
        // the table column model represented by this menu
        var columnModel = null;
        // the column index in the table range
        var tableCol = 0;

        // the radio group for sorting in ascending/descending order
        var sortGroup = null;
        // the check list control for discrete filters
        var filterGroup = null;

        // base constructor ---------------------------------------------------

        CompoundMenu.call(this, docView, {
            anchorAlign: 'trailing',
            actionKey: 'table/column/attributes',
            actionValue: getActionValue,
            enableActionHandler: isActionEnabled
        });

        // private methods ----------------------------------------------------

        /**
         * Returns a result object with the current filter settings as expected
         * by the controller item bound to the OK button of this menu.
         *
         * @returns {Object}
         *  The action value containing the current filter settings, with the
         *  following properties:
         *  - {String} tableName
         *      The name of the table range represented by this menu.
         *  - {Number} tableCol
         *      The column index in the table range.
         *  - {Object} attributes
         *      The attribute set with filter and sorting attributes to be set
         *      for the table column.
         */
        function getActionValue() {

            // the attribute set to be sent as result value
            var attrSet = {};

            // create discrete filter, or clear filter (for undefined entries)
            var filterEntries = filterGroup.getValue();
            if (!filterEntries || (filterEntries.length === filterGroup.getOptionButtons().length)) {
                attrSet.filter = { type: 'none', entries: [] };
            } else {
                attrSet.filter = { type: 'discrete', entries: filterEntries };
            }

            // create the sorting attributes
            switch (sortGroup.getValue()) {
                case 'ascending':
                    attrSet.sort = { type: 'value', descending: false };
                    break;
                case 'descending':
                    attrSet.sort = { type: 'value', descending: true };
                    break;
                default:
                    attrSet.sort = { type: 'none',  descending: false };
            }

            // return the entire action value
            return { tableName: tableModel.getName(), tableCol: tableCol, attributes: attrSet };
        }

        /**
         * Returns whether the OK button of this drop-down menu is currently
         * enabled (it is not allowed to apply a discrete filter without any
         * visible entries).
         */
        function isActionEnabled() {
            var entries = filterGroup.getValue();
            return _.isArray(entries) && (entries.length > 0);
        }

        // public methods -----------------------------------------------------

        /**
         * Collects and inserts the cell entries of a table column into this
         * menu, to be used to define discrete filters. Called by grid pane
         * instances from the click handler of cell drop-down buttons.
         *
         * @param {Address} address
         *  The address of the header cell of the table column.
         *
         * @param {Object} userData
         *  The user data stored during creation of the drop-down button
         *  associated to this menu. Contains the properties 'tableName' (name
         *  of the table model in the active sheet to be filtered), and
         *  'tableCol' (zero-based column index relative to the table range).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the menu has been initialized.
         */
        this.initialize = function (address, userData) {

            // store the table model and table column model for usage in private methods
            tableModel = userData.tableModel;
            tableCol = userData.tableCol;
            columnModel = tableModel.getColumnModel(tableCol);

            // create the secton wth the sorting controls
            this.addSectionLabel(Labels.SORT_HEADER_LABEL);
            this.addGroup(null, sortGroup = new RadioGroup(docView, { toggleValue: 'none' })
                .createOptionButton('ascending',  Labels.SORT_ASC_BUTTON_OPTIONS)
                .createOptionButton('descending', Labels.SORT_DESC_BUTTON_OPTIONS)
            );
            sortGroup.setValue(!columnModel.isSorted() ? 'none' : columnModel.isDescending() ? 'descending' : 'ascending');

            // create the secton with the filtering controls
            this.addSectionLabel(Labels.FILTER_HEADER_LABEL);
            this.addGroup(null, filterGroup = new DiscreteFilterGroup(docView));

            // intitialize the filter entries in a background loop (shows a busy spinner for large lists)
            return filterGroup.initialize(tableModel, tableCol);
        };

        // initialization -----------------------------------------------------

        this.on('popup:beforeshow', function () {
            self.registerFocusableNodes($('.clipboard'));
        });

        this.on('popup:hide', function () {
            self.destroyAllGroups();
            filterGroup = sortGroup = null;
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = tableModel = columnModel = null;
            filterGroup = sortGroup = null;
        });

    } }); // class TableColumnMenu

    // constants --------------------------------------------------------------

    /**
     * Tool tip for cell drop-down buttons attached to an instance of the class
     * TableColumnMenu.
     *
     * @constant
     */
    TableColumnMenu.BUTTON_TOOLTIP = gt.pgettext('filter', 'Select values for filtering or sorting');

    // exports ================================================================

    return TableColumnMenu;

});
