/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/popup/sheetcontextmenu', [
    'io.ox/office/tk/forms',
    'io.ox/office/baseframework/view/popup/contextmenu'
], function (Forms, ContextMenu) {

    'use strict';

    // class SheetContextMenu =================================================

    /**
     * A context menu for sheet tab buttons. Provides menu actions to
     * manipulate the active sheet, or the sheet collection of the document.
     *
     * @constructor
     *
     * @extends ContextMenu
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    function SheetContextMenu(docView, rootNode) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        ContextMenu.call(this, docView, rootNode, {
            selector: Forms.OPTION_BUTTON_SELECTOR,
            enableKey: 'document/editable', // do not show context menu in read-only mode
            delay: 200,
            lazyInitHandler: initMenu,
            autoHideGroups: false
        });

        // private methods ----------------------------------------------------
        /**
         * Activate the clicked sheet, before the context menu will be shown.
         */
        function contextMenuPrepareHandler(event) {

            // the clicked button element
            var sheetButton = $(event.sourceEvent.target);

            // click the button if it is not selected (activate the sheet)
            if (!Forms.isCheckedButtonNode(sheetButton)) {
                sheetButton.click();
            }
        }

        function initMenu() {
            docView.addSheetActionButtons(self, { insertSheet: true });
        }

        // public methods -----------------------------------------------------

        this.getXY = function (event) {
            return {
                pageX: event.sourceEvent.pageX,
                pageY: event.sourceEvent.pageY
            };
        };

        // initialization -----------------------------------------------------

        // preprocessing before the context menu will be shown
        this.on('contextmenu:prepare', contextMenuPrepareHandler);

        // hide contextmenu if the user start scrolling
        this.listenTo(docView, 'change:scrollpos', function () { self.hide(); });

        // destroy all class members
        this.registerDestructor(function () {
            self = docView = rootNode = null;
        });

    } // class SheetContextMenu

    // exports ================================================================

    // derive this class from class ContextMenu
    return ContextMenu.extend({ constructor: SheetContextMenu });

});
