/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/gridpane', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/drawinglayer/view/imageutil',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/utils/clipboard',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/trackingpane',
    'io.ox/office/spreadsheet/view/popup/cellcontextmenu',
    'io.ox/office/spreadsheet/view/popup/hyperlinkcontextmenu',
    'io.ox/office/spreadsheet/view/render/renderutils',
    'io.ox/office/spreadsheet/view/render/cellrenderer',
    'io.ox/office/spreadsheet/view/render/selectionrenderer',
    'io.ox/office/spreadsheet/view/render/drawingrenderer',
    'io.ox/office/spreadsheet/view/render/formrenderer',
    'io.ox/office/spreadsheet/view/render/highlightrenderer',
    'io.ox/office/spreadsheet/view/mixin/gridtrackingmixin',
    'io.ox/office/spreadsheet/view/chartcreator',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, KeyCodes, Forms, ValueSet, Rectangle, DrawingFrame, ImageUtil, SheetUtils, PaneUtils, Clipboard, Labels, TrackingPane, CellContextMenu, HyperlinkContextMenu, RenderUtils, CellRenderer, SelectionRenderer, DrawingRenderer, FormRenderer, HighlightRenderer, GridTrackingMixin, ChartCreator, gt) {

    'use strict';

    // convenience shortcuts
    var MergeMode = SheetUtils.MergeMode;
    var Address = SheetUtils.Address;
    var Range = SheetUtils.Range;
    var RangeArray = SheetUtils.RangeArray;

    // class GridPane =========================================================

    /**
     * Represents a single scrollable grid pane in the spreadsheet view. If the
     * view has been split or frozen at a specific cell position, the view will
     * consist of up to four panes (top-left, top-right, bottom-left, and
     * bottom-right pane).
     *
     * Instances of this class trigger the following events:
     * - 'render:cellselection'
     *      After the current cell selection has actually been rendered into
     *      the DOM selection layer node (rendering may happen debounced after
     *      several 'change:selection' view events).
     * - 'render:drawingselection'
     *      After the current drawing selection has actually been rendered into
     *      the DOM drawing layer (rendering may happen debounced after several
     *      'change:selection' view events). Event handlers receive the DOM
     *      drawing frames currently selected in this grid pane, as jQuery
     *      collection.
     * - 'render:frameselection'
     *      After the current frame selection has actually been rendered into
     *      the DOM highlighting layer. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {jQuery} frameNode
     *          The DOM node representing the frame rectangle, as jQuery
     *          collection.
     *      (3) {Rectangle} frameRect
     *          The location of the selected frame, in pixels. The rectangle
     *          object will contain the additional boolean properties 'reverseX'
     *          and 'reverseY'.
     * - 'change:scrollpos'
     *      After the scroll position of the grid pane has been changed in any
     *      direction. Event listeners may want to update the position of
     *      additional DOM content inserted into the grid pane.
     * - 'select:start'
     *      After selection tracking has been started. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Address} address
     *          The address of the start cell.
     *      (3) {String} mode
     *          The selection mode according to the keyboard modifier keys:
     *          - 'select': Standard selection without modifier keys.
     *          - 'append': Append new range to current selection (CTRL/META).
     *          - 'extend': Extend current active range (SHIFT).
     * - 'select:move'
     *      After the address of the tracked cell has changed while selection
     *      tracking is active. Event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Address} address
     *          The address of the current cell.
     * - 'select:end'
     *      After selection tracking has been finished (either successfully or
     *      not). Event handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Address} address
     *          The address of the last cell.
     *      (3) {Boolean} success
     *          True if selecting has finished successfully, false if selecting
     *          has been canceled.
     * - 'select:change'
     *      While selection tracking on touch devices is active, and a selected
     *      range has been modified by tracking the resizer handles. Event
     *      handlers receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Number} rangeIndex
     *          The array index of the modified range in the current selection.
     *      (3) {Range} range
     *          The new address of the specified cell range.
     *      (4) {Address} [activeCell]
     *          The address of the active cell. If omitted, the current active
     *          cell will not be changed, unless the new active range does not
     *          contain it anymore.
     *
     * @constructor
     *
     * @extends TrackingPane
     * @extends GridTrackingMixin
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view that contains this header pane.
     *
     * @param {String} panePos
     *  The position of this grid pane.
     */
    var GridPane = TrackingPane.extend({ constructor: function (docView, panePos) {

        // self reference
        var self = this;

        // column and row pane side identifiers
        var colPaneSide = PaneUtils.getColPaneSide(panePos);
        var rowPaneSide = PaneUtils.getRowPaneSide(panePos);

        // the application instance
        var app = docView.getApp();

        // the spreadsheet model
        var docModel = docView.getDocModel();

        // the spreadsheet view, and the header panes associated to the grid pane
        var colHeaderPane = docView.getColHeaderPane(panePos);
        var rowHeaderPane = docView.getRowHeaderPane(panePos);

        // the container node of this grid pane
        var rootNode = $('<div class="abs grid-pane f6-target" data-pos="' + panePos + '">');

        // the scrollable node of this pane (embed into rootNode to allow to hide scroll bars if needed)
        // (attribute unselectable='on' is needed for IE to prevent focusing the node when clicking on a scroll bar)
        var scrollNode = $('<div class="grid-scroll" unselectable="on">').appendTo(rootNode);

        // helper node controlling the size and position of the scroll bars
        var scrollSizeNode = $('<div class="grid-scroll-size">').appendTo(scrollNode);

        // the root node for all layer nodes, positioned dynamically in the scrollable area
        var layerRootNode = $('<div class="grid-layer-root noI18n">').appendTo(scrollSizeNode);

        // the clipboard node that carries the browser focus for copy/paste events
        var clipboardNode = docView.createClipboardNode().attr('tabindex', 0).text('\xa0').appendTo(rootNode);

        // the static node of this pane overlapping the client area of the scroll node, but not the scroll bars
        var staticRootNode = $('<div class="grid-static-root">').appendTo(rootNode);

        // the ratio between (public) absolute scroll position, and internal scroll bars
        var scrollLeftRatio = 1;
        var scrollTopRatio = 1;

        // distance between cell A1 and the top-left visible cell of this pane
        var hiddenWidth = 0;
        var hiddenHeight = 0;

        // the registered layer renderers, mapped by object UIDs
        var rendererRegistry = new ValueSet('getUid()');

        // shortcuts to specific renderers
        var formRenderer = null;
        var drawingRenderer = null;

        // the cell range and absolute position covered by the layer nodes in the sheet area
        var layerRange = null;
        var layerRectangle = null;

        // current sheet zoom factor (changed zoom factors will be passed to renderes for fast updates)
        var layerZoom = 0;

        // context menu for cells
        var cellContextMenu = null;

        // the context menu for read only cell
        var hyperlinkContextMenu = null;

        // base constructors --------------------------------------------------

        TrackingPane.call(this, docView, layerRootNode);

        // private methods ----------------------------------------------------

        /**
         * Registers a new layer renderer.
         *
         * @param {Function} RendererClass
         *  The class constructor for the new renderer instance.
         *
         * @returns {BaseObject}
         *  The new renderer instance.
         */
        function createLayerRenderer(RendererClass) {

            // create and store the new layer renderer
            var layerRenderer = rendererRegistry.insert(new RendererClass(self));

            // forward layer renderer events to own listeners
            if (_.isFunction(layerRenderer.on)) {
                self.forwardEvents(layerRenderer);
            }

            return layerRenderer;
        }

        /**
         * Refreshes the appearance of this grid pane according to the position
         * of the grid pane that is currently active.
         */
        function updateFocusDisplay() {

            // model of the active sheet
            var sheetModel = docView.getSheetModel();
            // which header pane is currently focused (while selection tracking is active)
            var focusPaneSide = sheetModel.getViewAttribute('activePaneSide');
            // which grid pane is currently focused
            var focusPanePos = sheetModel.getViewAttribute('activePane');
            // whether this grid pane appears focused
            var focused = false;

            if (docView.isTextEditMode('cell')) {
                // never highlight selection in cell edit mode
                focused = false;
            } else if (sheetModel.hasFrozenSplit()) {
                // always highlight all panes in frozen split mode
                focused = true;
            } else if (_.isString(focusPaneSide)) {
                // a header pane is active (selection tracking): focus this
                // grid pane if it is related to the active header pane
                focused = (colPaneSide === focusPaneSide) || (rowPaneSide === focusPaneSide);
            } else {
                // a single grid pane is focused
                focused = focusPanePos === panePos;
            }

            rootNode.toggleClass(Forms.FOCUSED_CLASS, focused);
        }

        /**
         * Handles changed size of the scroll area received from the column or
         * row header pane.
         */
        function updateScrollAreaSize() {

            // model of the active sheet
            var sheetModel = docView.getSheetModel();

            // the available size according to existence of scroll bars
            var clientWidth = scrollNode[0].clientWidth;
            var clientHeight = scrollNode[0].clientHeight;
            // difference of scroll area size between header pane and this grid pane
            var widthCorrection = rootNode.width() - clientWidth;
            var heightCorrection = rootNode.height() - clientHeight;

            // the new absolute width and height of the scrollable area
            var scrollWidth = 0;
            var scrollHeight = 0;

            // special case for chart sheets: enlarge scroll area size to include the entire chart
            if (sheetModel.isChartsheet()) {
                var chartModel = sheetModel.getDrawingCollection().getModel([0], { type: 'chart' });
                var rectangle = chartModel ? chartModel.getRectangle() : { width: 0, height: 0 };
                scrollWidth = Math.max(rectangle.width, rootNode.width() - widthCorrection);
                scrollHeight = Math.max(rectangle.height, rootNode.height() - heightCorrection);
            } else {
                scrollWidth = Math.max(0, colHeaderPane.getScrollSize() - widthCorrection);
                scrollHeight = Math.max(0, rowHeaderPane.getScrollSize() - heightCorrection);
            }

            // restrict to maximum allowed node size according to current browser
            var domWidth = Math.min(scrollWidth, Utils.MAX_NODE_SIZE);
            var domHeight = Math.min(scrollHeight, Utils.MAX_NODE_SIZE);
            scrollSizeNode.css({ width: domWidth, height: domHeight });

            // recalculate ratio between absolute and effective scroll area size
            scrollLeftRatio = (domWidth > clientWidth) ? ((scrollWidth - clientWidth) / (domWidth - clientWidth)) : 0;
            scrollTopRatio = (domHeight > clientHeight) ? ((scrollHeight - clientHeight) / (domHeight - clientHeight)) : 0;
            RenderUtils.withLogging(function () {
                RenderUtils.log('scroll area size: sheet-w=' + scrollWidth + ' sheet-h=' + scrollHeight + ' dom-w=' + domWidth + ' dom-h=' + domHeight + ' ratio-w=' + scrollLeftRatio + ' ratio-h=' + scrollTopRatio);
            });
        }

        /**
         * Handles changed scroll position received from the column or row
         * header pane.
         */
        function updateScrollPosition() {

            // the new DOM scroll position
            var scrollLeft = (scrollLeftRatio > 0) ? Math.round(colHeaderPane.getScrollPos() / scrollLeftRatio) : 0;
            var scrollTop = (scrollTopRatio > 0) ? Math.round(rowHeaderPane.getScrollPos() / scrollTopRatio) : 0;

            // restrict size of layer root node to size of sheet
            var layerRootWidth = Math.min(layerRectangle.width, docView.getSheetWidth() - layerRectangle.left);
            var layerRootHeight = Math.min(layerRectangle.height, docView.getSheetHeight() - layerRectangle.top);

            // calculate the effective offsets according to current scroll position
            var offsetLeft = Math.max(-2 * layerRootWidth, layerRectangle.left - colHeaderPane.getScrollPos() + scrollLeft - hiddenWidth);
            var offsetTop = Math.max(-2 * layerRootHeight, layerRectangle.top - rowHeaderPane.getScrollPos() + scrollTop - hiddenHeight);
            layerRootNode.css({ left: offsetLeft, top: offsetTop });

            // Only set new DOM scroll position if it differs from current position. This test
            // MUST be done, otherwise Firefox scrolls extremely slow when using the mouse wheel!
            if (scrollLeft !== scrollNode[0].scrollLeft) { scrollNode[0].scrollLeft = scrollLeft; }
            if (scrollTop !== scrollNode[0].scrollTop) { scrollNode[0].scrollTop = scrollTop; }

            // set size of the layer root node (should always be inside the node size limits)
            layerRootNode.css({ width: layerRootWidth, height: layerRootHeight });

            // notify listeners
            self.trigger('change:scrollpos');
        }

        /**
         * Handles 'mousedown' events from the root node.
         */
        function mouseDownHandler(event) {

            // the target node of the event, as jQuery object
            var targetNode = $(event.target);

            // bug 41094: user wants to enter text, keyboard shows then hides immediately
            if (Utils.IOS && targetNode.is('.cell-layer')) {
                event.preventDefault();
            }

            // special handling for right mouse button
            if (event.button === 2) {

                // select a drawing object if available, otherwise select the clicked cell
                var drawingFrame = self.getDrawingFrameForEvent(event);
                if (drawingFrame.length > 0) {
                    if (!DrawingFrame.isSelected(drawingFrame)) {
                        var drawingPos = DrawingFrame.getModel(drawingFrame).getPosition();
                        docView.selectDrawing(drawingPos);
                        docView.scrollToDrawingFrame(drawingPos);
                    }
                } else {
                    var address = self.getCellDataForEvent(event).address;
                    if (!docView.getSelectedRanges().containsAddress(address)) {
                        docView.selectCell(address);
                        docView.scrollToCell(address);
                    }
                }
            }
        }

        /**
         * Handles 'keydown' events from the root node.
         */
        function keyDownHandler(event) {

            // do not evaluate any key event during drawing text edit mode
            if (docView.isTextEditMode('drawing')) {
                return;
            }

            // ESCAPE key (only one action per key event)
            if (event.keyCode === KeyCodes.ESCAPE) {

                // close an embedded drop-down menu
                if (formRenderer.handleKeyDownEvent(event)) {
                    return false;
                }

                // cancel various selection modes (before deselecting drawings)
                if (docView.cancelFrameSelectionMode() || docView.cancelCustomSelectionMode()) {
                    return false;
                }

                // drawing selection: back to cell selection
                if (docView.hasDrawingSelection()) {
                    docView.removeDrawingSelection();
                    return false;
                }

                // otherwise: let ESCAPE key bubble up
                return;
            }

            // let the form control layer process the key event (embedded drop-down menus)
            if (formRenderer.handleKeyDownEvent(event)) {
                return false;
            }
        }

        /**
         * Handles 'drop' events from the root node.
         */
        function dropHandler(event) {

            // nothing to do in locked sheets
            // TODO: the error message needs to be adjusted when dropping cell data will be supported
            var promise = docView.ensureUnlockedDrawings({ errorCode: 'drawing:insert:locked' }).then(function () {

                // promises for the image descriptors (URL and name)
                var imagePromises = [];
                // whether any of the dropped data is not an image
                var unsupported = false;

                if (event.originalEvent.dataTransfer.files.length > 0) {
                    // data originates from local file system
                    _.each(event.originalEvent.dataTransfer.files, function (eventFile) {
                        if (ImageUtil.hasFileImageExtension(eventFile.name)) {
                            imagePromises.push(ImageUtil.insertFile(app, eventFile));
                        } else {
                            unsupported = true;
                        }
                    });
                } else if (event.originalEvent.dataTransfer.getData('Url').length > 0) {
                    // data originates from another browser window
                    var url = event.originalEvent.dataTransfer.getData('Url');
                    if (ImageUtil.hasUrlImageExtension(url)) {
                        imagePromises.push(ImageUtil.insertURL(url));
                    } else {
                        unsupported = true;
                    }
                }

                // show a warning if nothing of the dropped data may be an image;
                // nothing more to do without pending images
                if (imagePromises.length === 0) {
                    return unsupported ? SheetUtils.makeRejected('drop:unsupported') : null;
                }

                // fail with empty resolved promise to be able to use $.when() which would
                // otherwise abort immediately if ANY of the promises fails
                imagePromises = imagePromises.map(function (imagePromise) {
                    return imagePromise.then(null, _.constant($.when()));
                });

                // immediately go into busy mode, wait for all image descriptors, insert the images afterwards
                docView.enterBusy();
                $.when.apply($, imagePromises).then(function () {
                    // $.when() passes the results as function arguments, filter out invalid images
                    var imageDescs = _.filter(arguments, _.identity);
                    return app.getController().execInsertImages(imageDescs);
                }).always(function () {
                    docView.leaveBusy();
                });
            });

            // show warning alert if necessary (this is an event handler, do not return the promise)
            docView.yellOnFailure(promise);
        }

        /**
         * Handles 'cut' events of the clipboard node.
         * Invokes the copyHandler, then deletes the values and clears
         * the attributes of the selected cells.
         */
        function cutHandler(event) {

            // the clipboard event data
            var clipboardData = Utils.getClipboardData(event);

            // check for cell protection
            if (docView.ensureUnlockedSelection({ lockTables: 'header', lockMatrixes: 'full', sync: true })) {
                event.preventDefault();
                return false;
            }

            // invoke clipboard copy handler first
            if (copyHandler(event) === false) { return false; }

            if (clipboardData) {
                // prevent default cut handling for desktop browsers, but not for touch devices
                if (!Utils.TOUCHDEVICE) {
                    event.preventDefault();
                }

                docView.executeControllerItem(docView.hasDrawingSelection() ? 'drawing/delete' : 'cell/clear/all');
            } else {
                // delete values and clear attributes of the selected cells (bug 39533: via controller for busy screen etc.)
                self.executeDelayed(function () {
                    return docView.executeControllerItem(docView.hasDrawingSelection() ? 'drawing/delete' : 'cell/clear/all');
                }, 'GridPane.cutHandler');
            }
        }

        /**
         * Handles 'copy' events of the clipboard node.
         * Creates a HTML table of the active range of the current selection
         * and generates the client clipboard id. Adds both to the system clipboard.
         * And invokes the Calcengine copy operation.
         */
        var copyHandler = Utils.profileMethod('GridPane.copyHandler()', function (event) {

            // model of the active sheet
            var sheetModel = docView.getSheetModel();
            // current selection in the active sheet
            var selection = docView.getSelection();
            // the active cell range in the selection
            var activeRange = selection.activeRange();
            // the clipboard event data
            var clipboardData = Utils.getClipboardData(event);
            // the client clipboard id to identify the copy data when pasting
            var clientClipboardId = null;
            // get drawing selection, if available, first
            var selectedDrawingModels = docView.getSelectedDrawingModels();
            // the HTML mark-up promise, whose resolved value is to be to copied
            var htmlMarkup = null;
            // the plain text Promise, whose resolved value is to be to copied
            var plainText = null;

            if (selectedDrawingModels.length > 0) {

                // generate HTML representation of drawing(s), no plain-text can be generated
                Utils.log('generating HTML mark-up for ' + selectedDrawingModels.length + ' drawing objects');
                htmlMarkup = Clipboard.convertDrawingModels(app, selectedDrawingModels);

            } else {

                // bug 48170: shrink active range only, when a complete col/row is selected.
                if (docModel.isColRange(activeRange) || docModel.isRowRange(activeRange)) {
                    // bug 40391: shrink active range to the used area of the sheet (do not copy anything from the empty space)
                    var usedRange = sheetModel.getUsedRange();
                    activeRange = usedRange ? activeRange.intersect(usedRange) : null;
                    if (!activeRange) {
                        event.preventDefault();
                        return false;
                    }
                }

                // restrict the number of cells to copy at the same time
                if (activeRange.cells() > SheetUtils.MAX_FILL_CELL_COUNT) {
                    docView.yell({ type: 'info', message: gt('It is not possible to copy more than %1$d cells at the same time.', _.noI18n(SheetUtils.MAX_FILL_CELL_COUNT)) });
                    event.preventDefault();
                    return false;
                }

                // generate HTML table representation
                Utils.log('generating HTML mark-up for selected cell range');
                htmlMarkup = Clipboard.getHtmlMarkupFromRange(sheetModel, activeRange, clientClipboardId, { cut: (event.type === 'cut') });

                // generate plain text representation
                Utils.log('generating plain-text representation of selected cell range');
                plainText = Clipboard.getPlainTextFromRange(sheetModel, activeRange);
            }

            // set clipboard debug pane content
            self.trigger('debug:clipboard', htmlMarkup);

            // if browser supports the clipboard API, add the generated data to the event
            if (clipboardData) {

                // clear clipboard and add the generated plain-text contents and and/or the HTML mark-up
                Utils.log('adding data to system clipboard');
                clipboardData.clearData();
                if (plainText) { clipboardData.setData('text/plain', plainText); }
                if (htmlMarkup) { clipboardData.setData('text/html', htmlMarkup); }

                // prevent default copy handling for desktop browsers, but not for touch devices
                if (!Utils.TOUCHDEVICE) {
                    event.preventDefault();
                }

            } else if (htmlMarkup) {

                // add Excel XML name spaces for custom tag support
                $('html').attr({
                    xmlns: 'http://www.w3.org/TR/REC-html40',
                    'xmlns:o': 'urn:schemas-microsoft-com:office:office',
                    'xmlns:x': 'urn:schemas-microsoft-com:office:excel'
                });

                // insert contents into the internal clipboard node, and focus it
                Utils.log('adding copied data to internal clipboard DOM node');
                app.destroyImageNodes(clipboardNode);
                clipboardNode.empty().append(htmlMarkup).focus();

                // remove the XML namespaces after the browser has copied the data from the clipboard node
                _.defer(function () { $('html').removeAttr('xmlns xmlns:o xmlns:x'); });
            }
        });

        /**
         * Handles 'paste' events of the clipboard node.
         */
        var pasteHandler = Utils.profileMethod('GridPane.pasteHandler()', function (event) {

            // model of the active sheet
            var sheetModel = docView.getSheetModel();
            // the collection with the merged ranges of the active sheet
            var mergeCollection = sheetModel.getMergeCollection();
            // the collection with all cells of the active sheet
            var cellCollection = sheetModel.getCellCollection();
            // the collection with the conditional formattings of the active sheet
            var condFormatCollection = sheetModel.getCondFormatCollection();
            // the drawing collection of the active sheet
            var drawingCollection = sheetModel.getDrawingCollection();
            // current selection in the active sheet
            var selection = docView.getSelection();
            // address of the active cell
            var activeCell = docView.getActiveCell();
            // the clipboard event data
            var clipboardData = event && event.originalEvent && event.originalEvent.clipboardData;
            // the cleaned up HTML data
            var htmlData;
            // the HTML data attached to the event
            var htmlRawData;
            // the plain text data attached to the event
            var textData;

            /**
             * Tries to add the 'imageData' or 'imageUrl' attribute to the
             * passed image attribute set, if possible.
             */
            function addImageSourceToAttributes(imgNode, attributes) {

                // ignore all images without valid source URL
                var imageUrl = imgNode.attr('src');
                if (!imageUrl) { return false; }

                // add sub-object for image attributes
                if (!attributes.image) { attributes.image = {}; }

                // special handling for data URL, ignore all images without valid file extension
                if (/^data:image\//.test(imageUrl)) {
                    attributes.image.imageData = imageUrl;
                    return true;
                }

                // URL must contain valid file extension
                if (ImageUtil.hasFileImageExtension(imageUrl)) {
                    attributes.image.imageUrl = imageUrl;
                    return true;
                }

                return false;
            }

            /**
             * Inserts all drawing objects described in the passed drawing data
             * array with blocked user interface.
             *
             * @param {Array<Object>} drawingDescs
             *  The type and formatting attributes of the drawing objects to be
             *  inserted. See method DrawingMixin.execInsertDrawings() for more
             *  details.
             *
             * @returns {jQuery.Promise}
             *  A promise that will be resolved after all drawing objects have
             *  been inserted into the document, or that will be rejected on
             *  any error.
             */
            function insertDrawings(drawingDescs) {

                // nothing to do without any valid drawing objects
                if (drawingDescs.length === 0) { return $.when(); }

                // index to be inserted into the names of the pasted objects
                var nameIndex = drawingCollection.getModelCount({ deep: true });

                // generate a name for each drawing (with effective type that may have changed e.g. from chart to image)
                drawingDescs.forEach(function (drawingDesc) {
                    nameIndex += 1;
                    if (!drawingDesc.attrs.drawing) { drawingDesc.attrs.drawing = {}; }
                    drawingDesc.attrs.drawing.name = _.noI18n(Labels.getDrawingTypeLabel(drawingDesc.type)) + ' ' + nameIndex;
                });

                // insert the new drawing objects into the document
                var promise = app.getController().execInsertDrawings(drawingDescs, function (generator, sheet, position, drawingDesc) {
                    if (drawingDesc.type === 'chart') {
                        ChartCreator.generateOperationsFromModelData(generator, position, drawingDesc.modelData);
                    }
                });

                // show warning alert if necessary
                return docView.yellOnFailure(promise);
            }

            /**
             * tries to paste all drawing objects contained in the passed
             * drawing container node.
             */
            var pasteDrawingClipboard = Utils.profileAsyncMethod('GridPane.pasteDrawingClipboard()', function (containerNode, otherAppGuid) {

                // pixel position of the active cell
                var activeCellRect = sheetModel.getCellRectangle(activeCell, { pixel: true });
                // the drawing attributes to be passed to insertDrawings()
                var drawingDescs = [];
                // the attribute pool for drawing attributes
                var attributePool = docModel.getDrawingAttributePool();

                // generate the formatting attributes to be passed to insertDrawings()
                containerNode.children().each(function () {

                    // current drawing data node, as jQuery object
                    var dataNode = $(this);
                    // the drawing type stored in the data node
                    var drawingType = dataNode.attr('data-type');
                    // the location of the drawing object (relative to the bounding rectangle)
                    var rectangle = Clipboard.getRectangleFromNode(dataNode, activeCellRect.left, activeCellRect.top);
                    // the explicit formatting attributes of the object
                    var attributes = Clipboard.getAttributesFromNode(dataNode);
                    // additional JSON data from the DOM node
                    var data = Clipboard.getJSONDataFromNode(dataNode);
                    // child image node (for image objects, or as fall-back replacement image)
                    var imgNode = dataNode.find('>img');
                    // whether the data node has been processed successfully (ready to be pasted)
                    var success = false;

                    // merge the formatting attributes for the target rectangle
                    attributes = attributePool.extendAttributeSet(attributes, drawingCollection.getAttributeSetForRectangle(rectangle));

                    // TODO: other drawing types

                    // paste charts only if they have been copied from this document
                    if (drawingType === 'chart') {
                        success = _.isObject(data) && _.isObject(data.modelData) && (app.getGlobalUid() === otherAppGuid);
                        if (!success) { delete attributes.chart; }
                    }

                    // process image objects, or try to get a replacement image node as last fall-back
                    if (!success && (imgNode.length > 0)) {
                        drawingType = 'image';
                        success = addImageSourceToAttributes(imgNode, attributes);
                    }

                    // collect the attributes for all valid objects for asynchronous insertion
                    if (success) {
                        drawingDescs.push(_.extend({ type: drawingType, attrs: attributes }, data));
                    }
                });

                // lock the GUI, and insert the new drawing objects into the document
                return insertDrawings(drawingDescs);
            });

            var pasteImageClipboard = Utils.profileAsyncMethod('GridPane.pasteImageClipboard()', function (htmlData, jqImgElements) {

                // pixel position of the active cell
                var activeCellRect = sheetModel.getCellRectangle(activeCell, { pixel: true });
                // the drawing attributes to be passed to insertDrawings()
                var drawingDescs = [];
                var promise = self.createResolvedPromise(null);

                // generate the formatting attributes to be passed to insertDrawings()
                jqImgElements.each(function (i, img) {
                    promise = promise.then(function () {
                        return app.createImageNode($(img).attr('src')).then(function (imgNode) {
                            var img = imgNode[0];
                            // create the formatting attributes for the target rectangle
                            var attributeSet = drawingCollection.getAttributeSetForRectangle(new Rectangle(
                                activeCellRect.left,
                                activeCellRect.top,
                                img.width || img.naturalWidth || 250,
                                img.height || img.naturalHeight || 250
                            ));

                            // add generic drawing attributes
                            attributeSet.drawing.anchorType = 'twoCellAsOneCell';

                            // add image source link
                            if (addImageSourceToAttributes($(img), attributeSet)) {
                                drawingDescs.push({ type: 'image', attrs: attributeSet });
                            }
                            return $.when();
                        });
                    });
                });

                return promise.then(function () {
                    // insert the new images into the document
                    return insertDrawings(drawingDescs);
                });

            });

            function handleParserResult(parserResult) {
                if (!parserResult.contents) {
                    Utils.warn('no content');
                    return $.when();
                }

                // creates a range from a startAddress and a contents-object
                function createRangeFromContentsObject(address, contents) {
                    var range = new Range(address),
                        endCol = range.end[0],
                        endRow = range.end[1];

                    // calculate rows
                    _.each(contents, function (row) {
                        // get 'repeat'-attribute, or count one up
                        endRow += (row.r) ? row.r : 1;
                    });

                    // calculate columns (on the basis of cells)
                    _.each(contents[0].c, function (cell) {
                        // get 'repeat'-attribute, or count one up
                        endCol += (cell.r) ? cell.r : 1;
                    });

                    // count one down to prevent counting the start twice
                    range.end[0] = (endCol - 1);
                    range.end[1] = (endRow - 1);

                    return range;
                }

                var paste       = {},
                    promise     = null,
                    cellCount   = 0;

                // reject paste-promise when the clipboard-range at least does not fit in one of the target ranges
                if ((selection.ranges.length > 1) && parserResult.sourceRange && !parserResult.sourceRange.fitsIn(selection.ranges)) {
                    promise = SheetUtils.makeRejected('paste:ranges:unfit');

                } else {
                    // the parsed result
                    paste.ranges = cellCollection.getRepeatedCellContents(selection.ranges, parserResult.contents, {
                        intern:         parserResult.intern,
                        cut:            parserResult.cut,
                        selection:      selection,
                        sourceRange:    parserResult.sourceRange
                    });

                    // if there were condFormats, add them to the paste-object
                    if (_.isArray(parserResult.condFormats)) {
                        paste.condFormats = parserResult.condFormats;
                        paste.condFormats.forEach(function (cf) {
                            cf.ranges = new RangeArray();
                        });
                    }

                    // iterate over all paste ranges
                    paste.ranges.forEach(function (range) {
                        // the target range of the paste
                        var currentRange = createRangeFromContentsObject(range.address, range.contents);
                        // count up changing cells
                        cellCount += currentRange.cells();
                        // set merged ranges for all paste ranges
                        if (parserResult.mergedRanges) {
                            range.mergedRanges = parserResult.mergedRanges;
                        }
                        // add condFormats for all paste ranges
                        if (_.isArray(parserResult.condFormats)) {
                            paste.condFormats.forEach(function (format) {
                                format.ranges.push(currentRange);
                            });
                        }

                        // check that the target range is inside the sheet;
                        if ((currentRange.end[0] > docModel.getMaxCol()) || (currentRange.end[1] > docModel.getMaxRow())) {
                            promise = SheetUtils.makeRejected('paste:outside');

                        // check for locked cells in the target range
                        } else {
                            promise = docView.ensureUnlockedRanges(currentRange, { lockTables: 'header', lockMatrixes: 'partial', errorCode: 'paste:locked' });
                        }
                    });

                    // restrict the number of cells to paste at the same time;
                    if (cellCount > SheetUtils.MAX_FILL_CELL_COUNT) {
                        promise = SheetUtils.makeRejected('paste:overflow');
                    }
                }

                // paste the contents contained in the passed result object
                promise = promise.then(function () {

                    function processMergedRanges(ranges, address, callback) {

                        var // column offset for pasting contents that are available relative to column A
                            colOff = address[0],
                            // row offset for pasting contents that are available relative to row 1
                            rowOff = address[1];

                        // transform the ranges to the paste position
                        ranges = RangeArray.map(ranges, function (range) {
                            return docModel.getCroppedMovedRange(range, colOff, rowOff);
                        });

                        // invoke callback handler, if any valid ranges are left
                        return ranges.empty() ? null : callback(ranges);
                    }

                    return sheetModel.createAndApplyOperations(function (generator) {
                        var promise2 = $.when();

                        paste.ranges.forEach(function (range) {

                            // the current range of the paste
                            var currentRange = createRangeFromContentsObject(range.address, range.contents);

                            // unmerge cells if necessary
                            if ((currentRange.cells() > 1) && mergeCollection.rangesOverlapAnyMergedRange(currentRange)) {
                                promise2 = promise2.then(function () {
                                    return mergeCollection.generateMergeCellsOperations(generator, currentRange, MergeMode.UNMERGE);
                                });
                            }

                            // merge cells
                            if (range.mergedRanges) {
                                processMergedRanges(range.mergedRanges, range.address, function (mergedRanges) {
                                    promise2 = promise2.then(function () {
                                        return mergeCollection.generateMergeCellsOperations(generator, mergedRanges, MergeMode.MERGE);
                                    });
                                });
                            }

                            // reduce the target ranges of the conditional formatting rules for an internal copy/paste
                            if (parserResult.intern === true) {
                                promise2 = promise2.then(function () {
                                    return condFormatCollection.generateReduceRangesOperations(generator, currentRange);
                                });
                            }

                            // set cell contents and hyperlinks
                            promise2 = promise2.then(function () {
                                return cellCollection.generateCellContentOperations(generator, range.address, range.contents);
                            });

                            // recalculate references, when paste on
                            // - the same document
                            // - the same sheets (different sheets are dull)
                            // - with contents from "cut"-event
                            if (parserResult.sameDoc && parserResult.cut && parserResult.sourceSheet === docModel.getActiveSheet()) {
                                promise2 = promise2.then(function () {
                                    return cellCollection.generateCutPasteOperations(generator, parserResult.sourceSheet, parserResult.sourceRange, currentRange);
                                });
                            }

                            // recalculate the optimal row height
                            promise2 = promise2.then(function () {
                                return docView.setOptimalRowHeight(null, { generator: generator, update: true, changed: new RangeArray(currentRange) });
                            });
                        });

                        // conditional formatting
                        if (_.isArray(paste.condFormats) && (paste.condFormats.length > 0)) {
                            promise2 = promise2.then(function () {
                                return condFormatCollection.generatePasteOperations(generator, paste.condFormats);
                            });
                        }

                        return promise2;

                    }, { storeSelection: true });
                });

                // show alert messages on error
                return docView.yellOnFailure(promise);
            }

            function pasteHtmlClipboard(html) {
                Utils.log('pasting HTML clipboard contents');
                return handleParserResult(Clipboard.parseHTMLData(docView, html));
            }

            function pasteTextClipboard(text) {
                // set clipboard debug pane content
                self.trigger('debug:clipboard', text);
                // parse text
                Utils.log('pasting plain-text clipboard contents');
                return handleParserResult(Clipboard.parseTextData(text));
            }

            // check if we have edit rights before pasting
            if (!docView.requireEditMode()) {
                event.preventDefault();
                return false;
            }

            htmlRawData = clipboardData && clipboardData.getData('text/html');
            htmlData = _.isString(htmlRawData) ? Utils.parseAndSanitizeHTML(htmlRawData) : $();

            var drawingContainerNode = Clipboard.getDrawingContainerNode(htmlData);
            var jqTableElements = Clipboard.getContainedHtmlElements(htmlData, 'table');
            var jqImgElements = !jqTableElements ? Clipboard.getContainedHtmlElements(htmlData, 'img') : null;

            if (drawingContainerNode || jqTableElements || jqImgElements) {
                // check if clipboard event data contains HTML

                // prevent default paste handling for desktop browsers, but not for touch devices
                if (!Utils.TOUCHDEVICE) {
                    event.preventDefault();
                }

                // set clipboard debug pane content
                self.trigger('debug:clipboard', htmlRawData);
                // render the HTML to apply CSS styles provided by Excel or Calc
                app.destroyImageNodes(clipboardNode);
                clipboardNode.empty().append(htmlData);

                // enable busy blocker screen
                docView.enterBusy();

                // first, try to paste drawing objects that have been copied by ourselves
                var promise = null;
                if (drawingContainerNode) {
                    promise = pasteDrawingClipboard(drawingContainerNode, Clipboard.getApplicationGuid(htmlData));
                } else if (jqImgElements) {
                    promise = pasteImageClipboard(htmlData, jqImgElements);
                } else if (docView.hasSingleDrawingSelection()) {
                    // if exactly one drawing is selected paste into this drawing
                    promise = docModel.parseHTMLClipboardAndCreateOperations(clipboardNode);
                } else if (docView.hasDrawingSelection()) {
                    // if more than one drawing is selected do nothing
                    promise = $.when();
                } else {
                    promise = pasteHtmlClipboard(clipboardNode);
                }

                // leave busy blocker screen
                promise.always(function () { docView.leaveBusy().grabFocus(); });

                // clear clipboard
                clipboardNode.text('\xa0');
                return false;
            }

            // focus and select the clipboard node
            clipboardNode.focus();

            // check if clipboard event data contains plain text
            // to be used in case the clipboard node contains no HTML table to parse
            textData = (_.browser.IE < 12) ? window.clipboardData && window.clipboardData.getData('text') : clipboardData && clipboardData.getData('text/plain');

            // read and parse pasted data
            self.executeDelayed(function () {
                self.trigger('debug:clipboard', clipboardNode);

                drawingContainerNode = Clipboard.getDrawingContainerNode(clipboardNode);
                jqTableElements = Clipboard.getContainedHtmlElements(clipboardNode, 'table');
                jqImgElements = !jqTableElements ? Clipboard.getContainedHtmlElements(clipboardNode, 'img') : null;

                // enable busy blocker screen
                docView.enterBusy();

                // first, try to paste drawing objects that have been copied by ourselves
                var promise = null;
                if (drawingContainerNode) {
                    promise = pasteDrawingClipboard(drawingContainerNode, Clipboard.getApplicationGuid(clipboardNode));
                } else if (jqImgElements) {
                    promise = pasteImageClipboard(clipboardNode, jqImgElements);
                } else if (docView.hasSingleDrawingSelection()) {
                    // if exactly one drawing is selected paste into this drawing
                    promise = docModel.parseHTMLClipboardAndCreateOperations(clipboardNode);
                } else if (docView.hasDrawingSelection()) {
                    // if more than one drawing is selected do nothing
                    promise = $.when();
                } else if (jqTableElements) {
                    promise = pasteHtmlClipboard(clipboardNode);
                } else if (textData) {
                    promise = pasteTextClipboard(textData);
                } else {
                    promise = this.createResolvedPromise();
                }

                // leave busy blocker screen
                promise.always(function () { docView.leaveBusy().grabFocus(); });

                // clear clipboard
                clipboardNode.text('\xa0');
            }, 'GridPane.pasteHandler');
        });

        /**
         * Handles 'wheel' events of the column and row header nodes.
         */
        var wheelHandler = function (event) {
            event.preventDefault();
            self.scrollRelative(event.originalEvent.deltaX * 2, event.originalEvent.deltaY / 2);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the root DOM node of this grid pane.
         *
         * @returns {jQuery}
         *  The root node of this grid pane, as jQuery object.
         */
        this.getNode = function () {
            return rootNode;
        };

        /**
         * Returns the scrollable container node of all content layers in this
         * grid pane.
         *
         * @returns {jQuery}
         *  The scrollable container node of this grid pane, as jQuery object.
         */
        this.getScrollNode = function () {
            return scrollNode;
        };

        /**
         * Returns the root container node of all rendering layers in this grid
         * pane.
         *
         * @returns {jQuery}
         *  The root container node of all rendering layers in this grid pane,
         *  as jQuery object.
         */
        this.getLayerRootNode = function () {
            return layerRootNode;
        };

        /**
         * Returns the root container node of statically positioned contents in
         * this grid pane.
         *
         * @returns {jQuery}
         *  The root container node of statically positioned contents in this
         *  grid pane, as jQuery object.
         */
        this.getStaticRootNode = function () {
            return staticRootNode;
        };

        /**
         * Returns the column header pane associated to this grid pane.
         *
         * @returns {HeaderPane}
         *  The column header pane associated to this grid pane.
         */
        this.getColHeaderPane = function () {
            return colHeaderPane;
        };

        /**
         * Returns the row header pane associated to this grid pane.
         *
         * @returns {HeaderPane}
         *  The row header pane associated to this grid pane.
         */
        this.getRowHeaderPane = function () {
            return rowHeaderPane;
        };

        /**
         * Returns the specified header pane associated to this grid pane.
         *
         * @param {Boolean} columns
         *  Whether to return the column header pane (true), or the row header
         *  pane (false).
         *
         * @returns {HeaderPane}
         *  The column or row header pane associated to this grid pane.
         */
        this.getHeaderPane = function (columns) {
            return columns ? colHeaderPane : rowHeaderPane;
        };

        /**
         * Creates a new DOM rendering layer node, and inserts it into the
         * layer root node of this grid pane.
         *
         * @param {String} className
         *  Name of a CSS class that will be added to the created layer node.
         *
         * @returns {jQuery}
         *  The new DOM rendering layer node.
         */
        this.createLayerNode = function (className) {
            return $('<div class="grid-layer ' + className + '">').appendTo(layerRootNode);
        };

        /**
         * Sets the browser focus into this grid pane, unless text edit mode is
         * currently active in another grid pane.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.grabFocus = function () {
            Utils.setFocus(docView.getTextEditFocusNode() || clipboardNode);
            return this;
        };

        /**
         * Initializes the settings and layout of this grid pane.
         *
         * @param {Object} colSettings
         *  The view settings of the horizontal pane side (left or right).
         *  Supports the following properties:
         *  - {Number} colSettings.offset
         *      The absolute horizontal offset of the grid pane in the view
         *      root node, in pixels.
         *  - {Number} colSettings.size
         *      The total outer width of the grid pane, in pixels. The value
         *      zero will have the effect to hide the entire grid pane.
         *  - {Boolean} colSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) horizontally.
         *  - {Boolean} colSettings.showOppositeScroll
         *      Whether the vertical scroll bar will be visible or hidden
         *      outside the pane root node.
         *  - {Number} colSettings.hiddenSize
         *      The total width of the hidden columns in front of the sheet in
         *      frozen view mode.
         *
         * @param {Object} rowSettings
         *  The view settings of the vertical pane side (top or bottom).
         *  Supports the following properties:
         *  - {Number} rowSettings.offset
         *      The absolute vertical offset of the grid pane in the view root
         *      node, in pixels.
         *  - {Number} rowSettings.size
         *      The total outer height of the grid pane, in pixels. The value
         *      zero will have the effect to hide the entire grid pane.
         *  - {Boolean} rowSettings.frozen
         *      Whether the grid pane is frozen (not scrollable) vertically.
         *  - {Boolean} rowSettings.showOppositeScroll
         *      Whether the horizontal scroll bar will be visible or hidden
         *      outside the pane root node.
         *  - {Number} rowSettings.hiddenSize
         *      The total height of the hidden rows in front of the sheet in
         *      frozen view mode.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.initializePaneLayout = function (colSettings, rowSettings) {

            // initialize according to visibility
            if ((colSettings.size > 0) && (rowSettings.size > 0)) {

                // show pane root node and initialize auto-scrolling
                rootNode.show();

                this.enableTracking({
                    autoScroll: colSettings.frozen ? (rowSettings.frozen ? false : 'vertical') : (rowSettings.frozen ? 'horizontal' : true),
                    borderNode: rootNode,
                    dblClick: 'mouse',
                    dblClickDelay: _.browser.IE ? 1800 : null // TODO: IE needs too much time to process tracking events
                });

                // position and size
                rootNode.css({ left: colSettings.offset, top: rowSettings.offset, width: colSettings.size, height: rowSettings.size });

                // initialize the scroll node containing the scroll bars
                scrollNode.css({
                    overflowX: colSettings.frozen ? 'hidden' : '',
                    overflowY: rowSettings.frozen ? 'hidden' : '',
                    // ensure to hide scroll bars overlaying the DOM node, e.g. MacOS
                    right: (rowSettings.frozen || colSettings.showOppositeScroll) ? 0 : -Math.max(10, Utils.SCROLLBAR_WIDTH),
                    bottom: (colSettings.frozen || rowSettings.showOppositeScroll) ? 0 : -Math.max(10, Utils.SCROLLBAR_HEIGHT)
                });

                // reduce size of static root node by the visible size of the scroll bars
                staticRootNode.css({
                    right: rowSettings.showOppositeScroll ? Utils.SCROLLBAR_WIDTH : 0,
                    bottom: colSettings.showOppositeScroll ? Utils.SCROLLBAR_HEIGHT : 0
                });

                // additional layout information for frozen splits
                hiddenWidth = colSettings.hiddenSize;
                hiddenHeight = rowSettings.hiddenSize;

                // update scroll area size and CSS classes at root node for focus display
                updateScrollAreaSize();
                updateFocusDisplay();

            } else {

                // hide pane root node and deinitialize auto-scrolling
                rootNode.hide();
                this.disableTracking();
            }

            return this;
        };

        /**
         * Updates the DOM layer nodes according to the passed layer range.
         *
         * @param {RangeBoundary} layerBoundary
         *  The address of the new cell range covered by this grid pane, and
         *  the exact rectangle to be covered by the DOM layers. If this object
         *  is invalid (all properties are null), all rendering layers will be
         *  hidden.
         *
         * @param {Array} dirtyRanges
         *  An array with the addresses of all dirty cell ranges in the sheet
         *  that must be recalculated and repainted (e.g. after one or more
         *  document operations such as inserting or deleting columns or rows).
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.setLayerRange = function (layerBoundary, dirtyRanges) {

            if (layerBoundary.isValid()) {

                // update visible sheet contents according to the new layer range
                layerRange = layerBoundary.range;
                layerRectangle = layerBoundary.rectangle;

                // update the scroll position (exact position of the layer root node in the DOM)
                scrollNode.show();
                updateScrollAreaSize();
                updateScrollPosition();

                // detect changed zoom factor
                var newLayerZoom = docView.getZoomFactor();
                var zoomScale = layerZoom ? (newLayerZoom / layerZoom) : 1;
                layerZoom = newLayerZoom;

                // invoke all layer renderers
                RenderUtils.takeTime('GridPane.setLayerRange(): update rendering layers, pos=' + panePos, function () {
                    RenderUtils.log('range=' + layerRange + ' rectangle=' + layerRectangle);
                    rendererRegistry.forEach('setLayerRange', {
                        range: layerRange,
                        rectangle: layerRectangle,
                        dirtyRanges: dirtyRanges,
                        zoomScale: zoomScale
                    });
                });

            } else if (layerRange) {

                // clear the layer rectangle
                layerRange = layerRectangle = null;
                layerZoom = 0;

                // invoke all layer renderers
                rendererRegistry.forEach('hideLayerRange');

                // hide the scroll node to prevent any left-overs if all columns/rows are hidden in a visible grid pane
                scrollNode.hide();
            }

            return this;
        };

        /**
         * Returns whether a layer range is currently available (and the DOM
         * rendering layers are visible).
         *
         * @returns {Boolean}
         *  Whether a layer range is currently available.
         */
        this.isVisible = function () {
            return _.isObject(layerRange);
        };

        /**
         * Registers an event handler at the specified event source object that
         * will only be invoked when this grid pane is visible. See method
         * BaseObject.listenTo() for details about the parameters.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.listenToWhenVisible = function (source, type, handler) {
            return this.listenTo(source, type, function () {
                if (layerRange) { return handler.apply(self, arguments); }
            });
        };

        /**
         * Registers an event handler at the specified event source object that
         * will only be invoked when this grid pane is activated. See method
         * BaseObject.listenTo() for details about the parameters.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.listenToWhenActive = function (source, type, handler) {
            return this.listenToWhenVisible(source, type, function () {
                if (self === docView.getActiveGridPane()) { return handler.apply(self, arguments); }
            });
        };

        /**
         * Returns the address of the cell range currently covered by the
         * rendering layer nodes, including the ranges around the visible area.
         *
         * @returns {Range|Null}
         *  The address of the cell range covered by the layer nodes, if the
         *  layers are visible; otherwise null.
         */
        this.getLayerRange = function () {
            return layerRange ? layerRange.clone() : null;
        };

        /**
         * Returns the location of the entire sheet rectangle covered by the
         * rendering layer nodes, including the ranges around the visible area.
         *
         * @returns {Rectangle|Null}
         *  The location of the sheet area covered by the layer nodes (in
         *  pixels), if the layers are visible; otherwise null.
         */
        this.getLayerRectangle = function () {
            return layerRectangle ? layerRectangle.clone() : null;
        };

        /**
         * Converts the passed absolute sheet rectangle to a rectangle relative
         * to the current rendering layer nodes.
         *
         * @param {Rectangle|Object} rectangle
         *  An absolute sheet rectangle, in pixels.
         *
         * @returns {Rectangle}
         *  The resulting layer rectangle, relative to the current layer nodes.
         */
        this.convertToLayerRectangle = function (rectangle) {
            return new Rectangle(
                rectangle.left - (layerRectangle ? layerRectangle.left : 0),
                rectangle.top - (layerRectangle ? layerRectangle.top : 0),
                layerRectangle ? rectangle.width : 0,
                layerRectangle ? rectangle.height : 0
            );
        };

        /**
         * Returns the CSS position and size attributes for the passed absolute
         * sheet rectangle in pixels, as HTML mark-up value for the 'style'
         * element attribute, adjusted to the current position of the layer
         * nodes in this grid pane.
         *
         * @param {Rectangle|Object} rectangle
         *  The absolute sheet rectangle, in pixels.
         *
         * @returns {String}
         *  The value for the 'style' element attribute in HTML mark-up text.
         */
        this.getLayerRectangleStyleMarkup = function (rectangle) {
            return PaneUtils.getRectangleStyleMarkup(this.convertToLayerRectangle(rectangle));
        };

        /**
         * Invokes the passed iterator function for all cell ranges that are
         * located inside the current layer range. The iterator function will
         * receive additional data needed for rendering those ranges in some
         * way.
         *
         * @param {RangeArray|Range}
         *  An array of cell range addresses, or a single cell range address.
         *
         * @param {Function} iterator
         *  The iterator function that will be invoked for each range that is
         *  inside the visible area of this grid pane. Receives
         *  the following parameters:
         *  (1) {Range} range
         *      The address of the cell range to be rendered.
         *  (2) {Rectangle} rectangle
         *      The location of the current range, relative to the layer root
         *      node of this grid pane.
         *  (3) {Number} index
         *      The array index of the current cell range.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.alignToGrid=false]
         *      If set to true, the rectangles passed to the iterator callback
         *      function will be expanded by one pixel to the left and to the
         *      top, so that all its outer pixels are located on the grid lines
         *      surrounding the range.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.iterateRangesForRendering = function (ranges, callback, options) {

            // do nothing, if all columns or rows are hidden
            if (!this.isVisible()) { return this; }

            // the bounding rectangle for range nodes (bug 41353: prevent oversized DOM nodes)
            var boundRect = new Rectangle(0, 0, Utils.MAX_NODE_SIZE, Utils.MAX_NODE_SIZE);
            // whether to enlarge rectangle by one pixel to the left and top, to place it on grid lines
            var alignToGrid = Utils.getBooleanOption(options, 'alignToGrid', false);

            // get start position of the bounding rectangle for range nodes, to prevent oversized DOM nodes
            boundRect.left = Math.max(0, layerRectangle.left - Math.floor((boundRect.width - layerRectangle.width) / 2));
            boundRect.top = Math.max(0, layerRectangle.top - Math.floor((boundRect.height - layerRectangle.height) / 2));

            // restrict to ranges in the current cell area to prevent oversized DOM nodes
            // that would collapse to zero size, e.g. when entire columns or rows are selected
            // (for details, see the comments for the Utils.MAX_NODE_SIZE constant)
            RangeArray.forEach(ranges, function (range, index) {

                // restrict location of the range to the bounding rectangle,
                // skip ranges located completely outside the bounding rectangle
                var rectangle = docView.getRangeRectangle(range).intersect(boundRect);
                if (!rectangle) { return; }

                // enlarge rectangle to match the grid lines between the cells
                if (alignToGrid) { rectangle.expandSelf(1, 1, 0, 0); }

                callback.call(this, range, this.convertToLayerRectangle(rectangle), index);
            }, this);
            return this;
        };

        /**
         * Returns the size of the leading hidden area that cannot be shown in
         * this grid pane in a frozen split view.
         *
         * @returns {Object}
         *  The size of the of the leading hidden area that cannot be shown in
         *  this grid pane, in the following properties:
         *  - {Number} width
         *      The width of the hidden area left of this grid pane. Will be 0,
         *      if the view does not contain a frozen split.
         *  - {Number} height
         *      The height of the hidden area left of this grid pane. Will be
         *      0, if the view does not contain a frozen split.
         */
        this.getHiddenSize = function () {
            return { width: hiddenWidth, height: hiddenHeight };
        };

        /**
         * Returns the effective offset and size of the sheet area that is
         * really visible in this grid pane.
         *
         * @returns {Rectangle}
         *  The location of the sheet area that is visible in this grid pane,
         *  in pixels.
         */
        this.getVisibleRectangle = function () {
            return new Rectangle(
                colHeaderPane.getVisiblePosition().offset,
                rowHeaderPane.getVisiblePosition().offset,
                scrollNode[0].clientWidth,
                scrollNode[0].clientHeight
            );
        };

        /**
         * Returns the address of the top-left cell visible in this grid pane.
         * The cell is considered visible, it its column and row are at least
         * half visible.
         *
         * @returns {Address}
         *  The address of the top-left cell in this grid pane.
         */
        this.getTopLeftAddress = function () {
            return new Address(colHeaderPane.getFirstVisibleIndex(), rowHeaderPane.getFirstVisibleIndex());
        };

        /**
         * Changes the scroll position of this grid pane relative to the
         * current scroll position.
         *
         * @param {Number} leftDiff
         *  The difference for the horizontal scroll position, in pixels.
         *
         * @param {Number} topDiff
         *  The difference for the vertical scroll position, in pixels.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollRelative = function (leftDiff, topDiff) {
            colHeaderPane.scrollRelative(leftDiff);
            rowHeaderPane.scrollRelative(topDiff);
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified rectangle in the sheet
         * visible.
         *
         * @param {Rectangle|Object} rectangle
         *  The sheet rectangle to be made visible, in pixels.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.forceTop=false]
         *      If set to true, the rectangle will always be scrolled to the
         *      upper border of the grid pane, also if it is already visible in
         *      the bottom area.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToRectangle = function (rectangle, options) {

            // whether to force the rectangle to the top border
            var forceTop = Utils.getBooleanOption(options, 'forceTop', false);

            colHeaderPane.scrollToPosition(rectangle.left, rectangle.width);
            rowHeaderPane.scrollToPosition(rectangle.top, rectangle.height, { forceLeading: forceTop });
            return this;
        };

        /**
         * Scrolls this grid pane to make the specified cell visible.
         *
         * @param {Address} address
         *  The cell position to be made visible.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.forceTop=false]
         *      If set to true, the cell will always be scrolled into the upper
         *      border of the grid pane, also if it is already visible in the
         *      bottom area.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToCell = function (address, options) {

            // the location of the specified cell
            var rectangle = docView.getCellRectangle(address, { expandMerged: true });

            return this.scrollToRectangle(rectangle, options);
        };

        // event page coordinates ---------------------------------------------

        /**
         * Returns whether the page coordinates in the passed browser event are
         * visible inside the root node of this grid pane.
         *
         * @param {Event|jQuery.Event} event
         *  A browser event, expected to contain the numeric properties 'pageX'
         *  and 'pageY'.
         *
         * @returns {Boolean}
         *  Whether the page coordinates in the passed browser event are
         *  visible inside the root node of this grid pane.
         */
        this.containsEventOffset = function (event) {

            // invisible grid panes cannot contain a page offset
            if (!this.isVisible()) { return false; }

            // the page position of the root node and scroll node
            var rootRect = Rectangle.from(Utils.getClientPositionInPage(rootNode));
            var scrollRect = Rectangle.from(Utils.getClientPositionInPage(scrollNode));
            // the area that is really visible
            var visibleRect = rootRect.intersect(scrollRect);

            // return whether the event pixel is contained
            return visibleRect.containsPixel(event.pageX, event.pageY);
        };

        /**
         * Converts the page coordinates in the passed browser event to the
         * absolute position in the active sheet (in pixels), according to the
         * scroll position of this grid pane, and the current zoom factor of
         * the active sheet. It does not matter whether the page position is
         * visible in the scroll node.
         *
         * @param {Event|jQuery.Event} event
         *  A browser event, expected to contain the numeric properties 'pageX'
         *  and 'pageY'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.restricted=false]
         *      If set to true, this method will NOT return sheet coordinates
         *      that are outside the actual area of the active sheet.
         *
         * @returns {Object|Null}
         *  The value null, if this grid pane is currently hidden, or if the
         *  option 'restricted' has been set and the page coordinates are not
         *  covering the sheet area; otherwise the absolute sheet coordinates
         *  in the properties 'left' and 'top'.
         */
        this.getSheetOffsetForEvent = function (event, options) {

            // nothing to return in hidden grid panes
            if (!layerRectangle) { return null; }

            // the absolute position of the layer root node
            var layerOffset = layerRootNode.offset();
            // whether to restrict sheet coordinates to the sheet area
            var restricted = Utils.getBooleanOption(options, 'restricted', false);

            // calculate horizontal sheet offset in pixels, test for overflow
            var left = event.pageX - layerOffset.left + layerRectangle.left;
            if (restricted && ((left < 0) || (left >= docView.getSheetWidth()))) { return null; }

            // calculate vertival sheet offset in pixels, test for overflow
            var top = event.pageY - layerOffset.top + layerRectangle.top;
            if (restricted && ((top < 0) || (top >= docView.getSheetHeight()))) { return null; }

            return { left: left, top: top };
        };

        /**
         * Returns all data for the cell covered by the page coordinates in the
         * passed browser event. It does not matter whether the page position
         * is visible in the scroll node.
         *
         * @param {Event|jQuery.Event} event
         *  A browser event, expected to contain the numeric properties 'pageX'
         *  and 'pageY'.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  method ColRowCollection.getEntryByOffset(), and the following
         *  additional options:
         *  - {Boolean} [options.overflow=false]
         *      If set to true, this method will return sheet coordinates that
         *      are outside the actual area of the active sheet.
         *
         * @returns {Object|Null}
         *  The value null, if this grid pane is currently hidden; otherwise a
         *  result descriptor with the following properties:
         *  - {Number} left
         *      The horizontal absolute sheet position, in pixels.
         *  - {Number} top
         *      The vertical absolute sheet position, in pixels.
         *  - {Object} colDesc
         *      The column descriptor received from the column collection.
         *  - {Object} rowDesc
         *      The row descriptor received from the row collection.
         *  - {Address} address
         *      The address of the cell covering the specified page position.
         */
        this.getCellDataForEvent = function (event, options) {

            // get absolute sheet coordinates (nothing to return in hidden grid panes)
            var result = this.getSheetOffsetForEvent(event, options);
            if (!result) { return null; }

            // add the column and row collection entry
            options = _.extend({}, options, { pixel: true });
            result.colDesc = docView.getColCollection().getEntryByOffset(result.left, options);
            result.rowDesc = docView.getRowCollection().getEntryByOffset(result.top, options);

            // add the cell address
            result.address = new Address(result.colDesc.index, result.rowDesc.index);

            return result;
        };

        // drawing layer ------------------------------------------------------

        /**
         * Returns the DOM drawing frame node at the passed document position.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing model. May specify a position
         *  inside another drawing object.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame, as jQuery object; or null, if the drawing
         *  frame does not exist.
         */
        this.getDrawingFrame = function (position) {
            return drawingRenderer.getDrawingFrame(position);
        };

        /**
         * Returns the DOM drawing frame nodes for all specified document
         * positions.
         *
         * @param {Array<Array<Number>>} positions
         *  The document positions of the drawing models.
         *
         * @returns {jQuery}
         *  The DOM nodes of the drawing frames, as jQuery object.
         */
        this.getDrawingFrames = function (positions) {
            return drawingRenderer.getDrawingFrames(positions);
        };

        /**
         * Returns the DOM drawing frame that represents the passed drawing
         * model.
         *
         * @param {DrawingModel} drawingModel
         *  The drawing model instance to return the DOM drawing frame for.
         *
         * @returns {jQuery|Null}
         *  The DOM drawing frame for the passed drawing model; or null, if no
         *  drawing frame could be found.
         */
        this.getDrawingFrameForModel = function (drawingModel) {
            return drawingRenderer.getDrawingFrameForModel(drawingModel);
        };

        /**
         * Returns the drawing frame to be selected or manipulated by the
         * passed tracking event.
         *
         * @param {jQuery.Event} event
         *  The tracking event to be evaluated.
         *
         * @returns {jQuery}
         *  The drawing frame to be selected or manipulated by the passed
         *  tracking event if existing; otherwise an empty jQuery collection.
         */
        this.getDrawingFrameForEvent = function (event) {
            return drawingRenderer.getDrawingFrameForEvent(event);
        };

        /**
         * Scrolls this grid pane to make the specified drawing frame visible.
         *
         * @param {Array<Number>} position
         *  The document position of the drawing frame.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.forceTop=false]
         *      If set to true, the cell will always be scrolled into the upper
         *      border of the grid pane, also if it is already visible in the
         *      bottom area.
         *
         * @returns {GridPane}
         *  A reference to this instance.
         */
        this.scrollToDrawingFrame = function (position, options) {

            // the drawing model at the specified document position
            var drawingModel = docView.getDrawingCollection().getModel(position);

            // the position of the drawing model (will be null for hidden drawing objects)
            var rectangle = drawingModel ? drawingModel.getRectangle() : null;
            if (rectangle) { this.scrollToRectangle(rectangle, options); }

            return this;
        };

        // initialization -----------------------------------------------------

        // tracking mix-in expects fully initialized grid pane
        GridTrackingMixin.call(this);

        // marker for touch devices and browser types
        Forms.addDeviceMarkers(rootNode);

        // on touch devices, disable the content-editable mode of the clipboard node,
        // to prevent showing the virtual keyboard all the time
        if (Utils.TOUCHDEVICE) { clipboardNode.removeAttr('contenteditable'); }

        // intercept DOM focus requests at the clipboard node (select clipboard contents)
        clipboardNode[0].focus = _.wrap(clipboardNode[0].focus, function (nativeFocusMethod) {

            // bug 29948: prevent grabbing to clipboard node while in text edit mode
            if (docView.isTextEditMode()) {
                Utils.warn('GridPane.clipboardNode.focus(): clipboard will not be focused in text edit mode');
                return;
            }

            // first, move the browser focus to the clipboard node
            nativeFocusMethod.call(this);

            // On touch devices, selecting text nodes in a content-editable node will always show
            // the virtual keyboard. TODO: need to find a way to provide clipboard support.
            if (Utils.TOUCHDEVICE) { return; }

            // safely clear the old browser selection
            Utils.clearBrowserSelection();

            // set the browser selection
            try {
                var docRange = window.document.createRange();
                docRange.setStart(clipboardNode[0], 0);
                docRange.setEnd(clipboardNode[0], clipboardNode[0].childNodes.length);
                var selection = window.getSelection();
                selection.removeAllRanges();
                selection.addRange(docRange);
            } catch (ex) {
                Utils.error('GridPane.clipboardNode.focus(): failed to select clipboard node: ' + ex);
            }
        });

        // create the DOM layer nodes and layer renderers
        createLayerRenderer(CellRenderer);
        createLayerRenderer(SelectionRenderer);
        formRenderer = createLayerRenderer(FormRenderer);
        drawingRenderer = createLayerRenderer(DrawingRenderer);
        createLayerRenderer(HighlightRenderer);

        // bug 53205: restore scroll position when showing the entire application
        this.listenToWhenVisible(app.getWindow(), 'show', updateScrollPosition);

        // prepare this grid pane for entering the text edit mode
        this.listenTo(docView, 'textedit:enter', function () {
            // hide clipboard node to make it unavailable for code looking up focusable nodes
            clipboardNode.hide();
            // disable global F6 focus traveling into this grid pane while edit mode is active in another grid pane
            rootNode.toggleClass('f6-target', self === docView.getActiveGridPane());
            // update colored highlighting of selection frames etc.
            updateFocusDisplay();
        });

        // prepare this grid pane for leaving the text edit mode
        this.listenTo(docView, 'textedit:leave', function () {
            // show clipboard node to make it available for code looking up focusable nodes
            clipboardNode.show();
            // enable global F6 focus traveling into this grid pane
            rootNode.addClass('f6-target');
            // update colored highlighting of selection frames etc.
            updateFocusDisplay();
        });

        // update layers according to changed view attributes (only, if this grid pane is visible)
        this.listenToWhenVisible(docView, 'change:sheet:viewattributes', function (event, attributes) {
            // refresh focus display ('activePane', and 'activePaneSide')
            if (Utils.hasProperty(attributes, /^activePane/)) {
                updateFocusDisplay();
            }
        });

        // listen to changed scroll position/size from header panes
        this.listenToWhenVisible(colHeaderPane, 'change:scrollsize', updateScrollAreaSize);
        this.listenToWhenVisible(colHeaderPane, 'change:scrollpos', updateScrollPosition);
        this.listenToWhenVisible(rowHeaderPane, 'change:scrollsize', updateScrollAreaSize);
        this.listenToWhenVisible(rowHeaderPane, 'change:scrollpos', updateScrollPosition);

        // listen to DOM scroll events, and forward the scroll position to the header panes
        // for synchronous scrolling in all visible header panes and grid panes
        scrollNode.on('scroll', function () {
            colHeaderPane.scrollTo(Math.round(scrollNode[0].scrollLeft * scrollLeftRatio));
            rowHeaderPane.scrollTo(Math.round(scrollNode[0].scrollTop * scrollTopRatio));
        });

        // activate this pane on focus change
        rootNode.on('focusin', function () { docView.updateActiveGridPane(panePos); });

        // handle key and drop events
        rootNode.on({ mousedown: mouseDownHandler, keydown: keyDownHandler, drop: dropHandler });

        // handle system clipboard events
        clipboardNode.on({ cut: cutHandler, copy: copyHandler, paste: pasteHandler });

        // create the context menus for cells and drawing objects
        cellContextMenu = new CellContextMenu(this);
        hyperlinkContextMenu = new HyperlinkContextMenu(this);

        // Listen on 'wheel' events to scroll on the column/row headers
        colHeaderPane.getNode().on('wheel', wheelHandler);
        rowHeaderPane.getNode().on('wheel', wheelHandler);

        // destroy all class members on destruction
        this.registerDestructor(function () {

            rendererRegistry.forEach('destroy').clear();
            cellContextMenu.destroy();
            hyperlinkContextMenu.destroy();
            app.destroyImageNodes(clipboardNode);

            self = app = docModel = docView = colHeaderPane = rowHeaderPane = null;
            rootNode = scrollNode = scrollSizeNode = layerRootNode = clipboardNode = staticRootNode = null;
            rendererRegistry = formRenderer = drawingRenderer = null;
            cellContextMenu = hyperlinkContextMenu = null;
        });

    } }); // class GridPane

    // exports ================================================================

    return GridPane;

});
