/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/edit/drawingtexteditor', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/keycodes',
    'io.ox/office/drawinglayer/view/drawingframe',
    'io.ox/office/textframework/utils/position',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/drawing/text/textframeutils',
    'io.ox/office/spreadsheet/view/edit/texteditorbase'
], function (Utils, KeyCodes, DrawingFrame, Position, SheetUtils, TextFrameUtils, TextEditorBase) {

    'use strict';

    // class DrawingTextEditor ================================================

    /**
     * Implementation of the text edit mode in drawing objects.
     *
     * @constructor
     *
     * @extends TextEditorBase
     *
     * @param {SpreadsheetView} docView
     *  The document view that has created this instance.
     */
    var DrawingTextEditor = TextEditorBase.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // the spreadsheet application, model, and other model objects
        var docModel = docView.getDocModel();
        // the root container for DOM models of all drawing objects in all sheets
        var textRootNode = docModel.getNode();
        // the text selection engine of the text framework
        var textSelection = docModel.getSelection();

        // the model of the drawing frame whose text is edited in drawing edit mode
        var editDrawingModel = null;
        // the DOM drawing frame currently edited in drawing edit mode (directly manipulated by the text framework)
        var editDrawingFrame = null;
        // the DOM text frame currently edited in drawing edit mode (directly manipulated by the text framework)
        var editTextFrame = null;

        // the DOM drawing frame that has been rendered by the active grid pane (used as source for position and size)
        var renderDrawingFrame = null;

        // base constructor ---------------------------------------------------

        TextEditorBase.call(this, docView, 'drawing', {
            enterHandler: enterHandler,
            cancelHandler: cancelHandler,
            formatResolver: formatResolver,
            formatHandler: formatHandler,
            focusResolver: function () { return editTextFrame[0]; }
        });

        // private methods ----------------------------------------------------

        /**
         * Returns the complete document position before the first character in
         * the edited drawing frame.
         *
         * @returns {Array<Number>|Null}
         *  The complete document position before the first character in the
         *  edited drawing frame.
         */
        function getFirstCursorPosition() {
            return Position.getFirstTextPositionInTextFrame(textRootNode, editDrawingFrame);
        }

        /**
         * Returns the complete document position behind the last character in
         * the edited drawing frame.
         *
         * @returns {Array<Number>|Null}
         *  The complete document position behind the last character in the
         *  edited drawing frame.
         */
        function getLastCursorPosition() {
            return Position.getLastTextPositionInTextFrame(textRootNode, editDrawingFrame);
        }

        /**
         * Updates the position and size of the drawing frame that is currently
         * edited, and its text frame.
         */
        function updateDrawingFramePosition() {

            // grid pane visible rectangle, used to fetch grid scroll positions
            var visibleGridRect = self.getEditGridPane().getVisibleRectangle();
            // drawing rectangle values
            var drawingRectangle = editDrawingModel.getRectangle();
            // CSS inline styles of the source text frame
            var drawingFrameStyle = renderDrawingFrame[0].style;
            // the current zoom factor of the active sheet
            var zoom = docView.getZoomFactor();

            // update position and size of the edited drawing frame
            editDrawingFrame.css({
                left: Utils.minMax(drawingRectangle.left - visibleGridRect.left, -Utils.MAX_NODE_SIZE, Utils.MAX_NODE_SIZE),
                top: Utils.minMax(drawingRectangle.top - visibleGridRect.top, -Utils.MAX_NODE_SIZE, Utils.MAX_NODE_SIZE),
                width: drawingFrameStyle.width,
                height: drawingFrameStyle.height,
                transform: drawingFrameStyle.transform,
                zIndex: drawingFrameStyle.transform.length ? '1' : ''
            });

            // dynamically resolve the text frame (no caching, it may be recreated when updating the drawing frame)
            TextFrameUtils.withTextFrame(renderDrawingFrame, function (renderTextFrame) {

                // CSS inline styles of the source text frame
                var textFrameStyle = renderTextFrame[0].style;

                // update position and size of the edited text frame
                editTextFrame.css({
                    left: textFrameStyle.left,
                    top: textFrameStyle.top,
                    width: textFrameStyle.width,
                    height: textFrameStyle.height,
                    transform: textFrameStyle.transform,
                    fontSize: Math.round(16 * zoom) + 'px'
                });
            });
        }

        /**
         * Event handler for 'keydown' events received from the text frame
         * currently edited.
         */
        function textFrameKeyDownHandler(event) {

            // quit drawing edit mode on ESCAPE key regardless of any modifier keys
            if (event.keyCode === KeyCodes.ESCAPE) {
                self.cancelEditMode();
                return false;
            }

            // Ctrl+A selects the entire text in the drawing
            if (KeyCodes.matchKeyCode(event, 'A', { ctrlOrMeta: true })) {
                var firstPos = getFirstCursorPosition();
                var lastPos = getLastCursorPosition();
                if (firstPos && lastPos) {
                    textSelection.setTextSelection(firstPos, lastPos);
                }
                return false;
            }
        }

        /**
         * Callback handler for starting the drawing edit mode.
         *
         * @param {GridPane} gridPane
         *  The active grid pane where text editing will take place.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.text]
         *      The text to be appended to the existing text contents.
         */
        function enterHandler(gridPane, options) {

            // reject editing in locked sheets
            var errorCode = docView.ensureUnlockedDrawings({ sync: true });
            if (errorCode) { return SheetUtils.makeRejected(errorCode); }

            // single drawing selection required for text edit mode
            var drawingSelection = docView.getSelectedDrawings();
            if (drawingSelection.length !== 1) { return false; }

            // the model of the selected drawing object to be edited
            var drawingCollection = docView.getDrawingCollection();
            editDrawingModel = drawingCollection.getModel(drawingSelection[0]);
            if (!editDrawingModel) { return false; }

            // the text frame of the selected drawing object (early exit for drawing objects without text)
            editDrawingFrame = drawingCollection.getDrawingFrameForModel(editDrawingModel);
            editTextFrame = TextFrameUtils.getTextFrame(editDrawingFrame);
            if (!editTextFrame) { return false; }

            // Creating Firefox specific drawing structure to enable text selection inside text frame (53502)
            if (_.browser.Firefox) {
                editDrawingFrame.removeAttr('contenteditable');
                DrawingFrame.getContentNode(editDrawingFrame).attr('contenteditable', '');
            }

            // the drawing frame rendered by the active grid pane (used as source for position and size of text frame)
            renderDrawingFrame = gridPane.getDrawingFrameForModel(editDrawingModel);
            if (!renderDrawingFrame) { return false; }

            // position after the last character in the text contents
            var cursorPos = getLastCursorPosition();
            if (!cursorPos) { return false; }

            // insert the entire DOM tree with all drawing objects into the active grid pane
            gridPane.getStaticRootNode().append(textRootNode);

            // update scroll position of the edit text frame (keep in sync with drawing frame)
            self.listenTo(gridPane, 'change:scrollpos', updateDrawingFramePosition);

            // update CSS attributes of the edit text frame, after the drawing frame has been (re-)rendered
            self.listenTo(gridPane, 'render:drawingframe', function (event, drawingFrame, drawingModel) {
                if (drawingModel === editDrawingModel) { updateDrawingFramePosition(); }
            });

            // update zoom settings during edit mode
            self.listenTo(docView, 'change:sheet:viewattributes', function (event, attributes) {
                if ('zoom' in attributes) { updateDrawingFramePosition(); }
            });

            // initialize the source and target text frames
            editDrawingFrame.addClass('edit-active');
            renderDrawingFrame.addClass('edit-active');
            updateDrawingFramePosition();

            // restore the internal browser text selection after focusing
            self.listenTo(editTextFrame, 'focusin', function () {
                textSelection.restoreBrowserSelection();
            });

            // process keyboard events of the text feame
            self.listenTo(editTextFrame, 'keydown', textFrameKeyDownHandler);

            // update paragraph/character formatting of the text frame
            drawingCollection.updateTextFormatting(editDrawingModel);

            // the text to be appended to the existing text contents
            var initialText = Utils.getStringOption(options, 'text', null);
            if (initialText) {
                docModel.insertText(initialText, cursorPos);
                cursorPos = getLastCursorPosition();
            }

            // set initial text selection
            textSelection.setTextSelection(cursorPos);

            if (_.browser.Firefox) { textSelection.restoreBrowserSelection(); } // 53635, cursor visibility in FF

            // indicate successfully started edit mode
            return true;
        }

        /**
         * Cleanup when canceling the drawing edit mode.
         */
        function cancelHandler() {

            self.stopListeningTo(docView);
            self.stopListeningTo(editTextFrame);

            editDrawingFrame.removeClass('edit-active');
            editDrawingFrame.css('transform', ''); // reset previously set transform to fetch original width/height/positions
            editTextFrame.css({ transform: '' });
            editDrawingModel = editDrawingFrame = editTextFrame = null;

            renderDrawingFrame.removeClass('edit-active');
            renderDrawingFrame = null;
        }

        /**
         * Returns the attribute set with the merged character and paragraph
         * attributes of the current text selection, as provided by the text
         * framework.
         *
         * @returns {Object}
         *  The merged attribute set of the text selection.
         */
        function formatResolver() {
            return docView.getTextAttributeSet();
        }

        /**
         * Applies the passed attributes to the current text selection.
         *
         * @param {Object} attributeSet
         *  The attributes to be applied to the text selection.
         */
        function formatHandler(attributeSet) {
            return docView.setTextAttributeSet(attributeSet);
        }

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docView = docModel = textRootNode = textSelection = null;
            editDrawingModel = editDrawingFrame = editTextFrame = null;
            renderDrawingFrame = null;
        });

    } }); // class DrawingTextEditor

    // exports ================================================================

    return DrawingTextEditor;

});
