/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author York Richter <york.richter@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/dialog/unhidesheetsdialog', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/dialogs',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/control/checkgroup',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, Dialogs, IteratorUtils, CheckGroup, gt) {

    'use strict';

    // class UnhideSheetsDialog ===============================================

    /**
     * A dialog that allows to make some or all hidden sheets visible in the
     * spreadsheet document.
     *
     * @constructor
     *
     * @extends ModalDialog
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view instance that has created this dialog.
     */
    var UnhideSheetsDialog = Dialogs.ModalDialog.extend({ constructor: function (docView) {

        // self reference
        var self = this;

        // list of checkboxes for all hidden sheets
        var checkGroup = new CheckGroup(docView, { boxed: true, checkAll: true });

        // base constructor ---------------------------------------------------

        Dialogs.ModalDialog.call(this, docView, { width: 400, title: gt('Unhide Sheets') });

        // public methods -----------------------------------------------------

        /**
         * Shows the dialog.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved after the
         *  action handler callback function has been invoked successfully; or
         *  will be rejected, if the dialog has been canceled, or if the
         *  document has switched to read-only mode.
         */
        this.show = _.wrap(this.show, function (show) {
            return show.call(this).then(function () {
                return checkGroup.getValue();
            }).always(function () {
                checkGroup.destroy();
                self = docView = checkGroup = null;
            });
        });

        // initialization -----------------------------------------------------

        // close dialog automatically after losing edit rights
        docView.closeDialogOnReadOnlyMode(this);

        this.on('show', function () {
            Utils.setFocus(self.getCancelButton());
        });

        // create the list of check boxes
        this.append(checkGroup.getNode().addClass('dialog-group'));
        var iterator = docView.getDocModel().createSheetIterator({ supported: true, visible: false });
        IteratorUtils.forEach(iterator, function (sheetModel, iterResult) {
            var sheetLabel = _.noI18n(iterResult.name);
            checkGroup.createOptionButton(iterResult.sheet, { label: sheetLabel });
        });

        // update enabled state of the OK button
        this.enableOkButton(false);
        checkGroup.on('group:change', function (event, sheets) {
            self.enableOkButton(sheets.length > 0);
        });

    } }); // class UnhideSheetsDialog

    // exports ================================================================

    return UnhideSheetsDialog;

});
