/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/view/controls', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/view/editcontrols',
    'io.ox/office/spreadsheet/view/labels',
    'io.ox/office/spreadsheet/view/control/alignmentpicker',
    'io.ox/office/spreadsheet/view/control/formatcategorygroup',
    'io.ox/office/spreadsheet/view/control/formatcategorypicker',
    'io.ox/office/spreadsheet/view/control/formatcodepicker',
    'io.ox/office/spreadsheet/view/control/cellbordercolorpicker',
    'io.ox/office/spreadsheet/view/control/cellstylepicker',
    'io.ox/office/spreadsheet/view/control/mergecellspicker',
    'io.ox/office/spreadsheet/view/control/namedrangesmenubutton',
    'io.ox/office/spreadsheet/view/control/sortmenubutton',
    'io.ox/office/spreadsheet/view/control/activesheetgroup',
    'io.ox/office/spreadsheet/view/control/activesheetlist',
    'io.ox/office/spreadsheet/view/control/subtotallist',
    'gettext!io.ox/office/spreadsheet/main'
], function (Utils, EditControls, Labels, AlignmentPicker, FormatCategoryGroup, FormatCategoryPicker, FormatCodePicker, CellBorderColorPicker, CellStylePicker, MergeCellsPicker, NamedRangesMenuButton, SortMenuButton, ActiveSheetGroup, ActiveSheetList, SubtotalList, gt) {

    'use strict';

    // convenience shortcuts
    var Button = EditControls.Button;
    var CheckBox = EditControls.CheckBox;
    var Label = EditControls.Label;
    var UnitField = EditControls.UnitField;
    var CompoundButton = EditControls.CompoundButton;
    var CompoundCheckBox = EditControls.CompoundCheckBox;

    // static class SpreadsheetControls =======================================

    /**
     * Provides different classes for GUI form controls. Collects all standard
     * control group classes defined in the toolkit and the base framework, and
     * adds more controls in a single map for convenience.
     *
     * @extends EditControls
     */
    var SpreadsheetControls = _.extend({}, EditControls, {
        AlignmentPicker: AlignmentPicker,
        FormatCategoryGroup: FormatCategoryGroup,
        FormatCategoryPicker: FormatCategoryPicker,
        FormatCodePicker: FormatCodePicker,
        CellBorderColorPicker: CellBorderColorPicker,
        CellStylePicker: CellStylePicker,
        MergeCellsPicker: MergeCellsPicker,
        NamedRangesMenuButton: NamedRangesMenuButton,
        SortMenuButton: SortMenuButton,
        ActiveSheetGroup: ActiveSheetGroup,
        ActiveSheetList: ActiveSheetList,
        SubtotalList: SubtotalList
    });

    // class ColRowOpIconButton ===============================================

    // icons and tooltips for all column/row operations
    var COL_ROW_ICON_OPTIONS = [
        /* 0 = 0b00 = delete row */    { icon: 'docs-table-delete-row',    tooltip: gt('Delete selected rows') },
        /* 1 = 0b01 = delete column */ { icon: 'docs-table-delete-column', tooltip: gt('Delete selected columns') },
        /* 2 = 0b10 = insert row */    { icon: 'docs-table-insert-row',    tooltip: gt('Insert row') },
        /* 3 = 0b10 = insert column */ { icon: 'docs-table-insert-column', tooltip: gt('Insert column') }
    ];

    /**
     * A button control consisting of a single icon used to insert or delete
     * columns or rows.
     *
     * @constructor
     *
     * @extends Button
     */
    SpreadsheetControls.ColRowOpIconButton = Button.extend({ constructor: function (docView, insert, columns) {

        // build the array index into COL_ROW_ICON_OPTIONS according to the passed boolean flags
        Button.call(this, docView, COL_ROW_ICON_OPTIONS[2 * insert + columns]);

    } }); // class ColRowOpIconButton

    // class ColRowOpLabelButton ==============================================

    // labels for all column/row operations
    var COL_ROW_LABELS = [
        /* 0 = 0b000 = delete one row */    gt('Delete row'),
        /* 1 = 0b001 = delete rows */       gt('Delete rows'),
        /* 2 = 0b010 = delete one column */ gt('Delete column'),
        /* 3 = 0b011 = delete columns */    gt('Delete columns'),
        /* 4 = 0b100 = insert one row */    gt('Insert row'),
        /* 5 = 0b101 = insert rows */       gt('Insert rows'),
        /* 6 = 0b110 = insert one column */ gt('Insert column'),
        /* 7 = 0b111 = insert columns */    gt('Insert columns')
    ];

    /**
     * A button control consisting of a label used to insert or delete columns
     * or rows.
     *
     * @constructor
     *
     * @extends Button
     */
    SpreadsheetControls.ColRowOpLabelButton = Button.extend({ constructor: function (docView, insert, columns, plural) {

        // build the array index into COL_ROW_LABELS according to the passed boolean flags
        Button.call(this, docView, { label: COL_ROW_LABELS[4 * insert + 2 * columns + plural] });

    } }); // class ColRowOpLabelButton

    // class ColRowSizeLabel ==================================================

    /**
     * A label control showing an icon in front of a unit field which is used
     * to manipulate the width of a column, or the height of a row.
     *
     * @constructor
     *
     * @extends Label
     */
    SpreadsheetControls.ColRowSizeLabel = Label.extend({ constructor: function (docView, columns) {

        Label.call(this, docView, {
            icon: columns ? 'docs-table-resize-column' : 'docs-table-resize-row',
            tooltip: columns ? gt('Column width') : gt('Row height'),
            smallerVersion: { hide: true }
        });

    } }); // class ColRowSizeLabel

    // class ColRowSizeField ==================================================

    /**
     * A unit field control used to manipulate the width of a column, or the
     * height of a row.
     *
     * @constructor
     *
     * @extends UnitField
     */
    SpreadsheetControls.ColRowSizeField = UnitField.extend({ constructor: function (docView, columns) {

        UnitField.call(this, docView, {
            width: 70,
            tooltip: columns ? gt('Column width') : gt('Row height'),
            min: 0,
            max: docView.getDocModel().getMaxColRowSizeHmm(columns),
            smallerVersion: { hide: true }
        });

    } }); // class ColRowSizeField

    // class CellFillColorPicker ==============================================

    /**
     * A picker control for the cell background color.
     *
     * @constructor
     *
     * @extends FillColorPicker
     */
    SpreadsheetControls.CellFillColorPicker = EditControls.FillColorPicker.extend({ constructor: function (docView) {

        EditControls.FillColorPicker.call(this, docView, { icon: 'docs-cell-fill-color' });

    } }); // class CellFillColorPicker

    // class CellBorderModePicker =============================================

    /**
     * The selector for cell borders. Hides specific items in the drop-down
     * menu, according to the current cell selection.
     *
     * @constructor
     *
     * @extends BorderModePicker
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    SpreadsheetControls.CellBorderModePicker = EditControls.BorderModePicker.extend({ constructor: function (docView) {

        EditControls.BorderModePicker.call(this, docView, {
            tooltip: Labels.CELL_BORDERS_LABEL,
            showInsideHor: function () { return docView.hasMultipleRowsSelected(); },
            showInsideVert: function () { return docView.hasMultipleColumnsSelected(); },
            dropDownVersion: { label: Labels.CELL_BORDERS_LABEL }
        });

    } }); // class CellBorderModePicker

    // class CellBorderStylePicker ============================================

    /**
     * A picker control for cell border styles. The available list entries will
     * depend on the current file format.
     *
     * @constructor
     *
     * @extends BorderStylePicker
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    SpreadsheetControls.CellBorderStylePicker = EditControls.BorderStylePicker.extend({ constructor: function (docView) {

        var PRESET_STYLES = docView.getApp().isOOXML() ? Labels.BORDER_OOXML_PRESET_STYLES : Labels.BORDER_ODF_LINE_STYLES;

        EditControls.BorderStylePicker.call(this, docView, PRESET_STYLES, {
            dropDownVersion: { label: Labels.BORDER_STYLE_LABEL }
        });

    } }); // class CellBorderStylePicker

    // class CellProtectionMenuButton =========================================

    /**
     * A compound button with options to change the cell protection settings.
     *
     * @constructor
     *
     * @extends CompoundButton
     *
     * @param {SpreadsheetView} docView
     *  The spreadsheet view containing this instance.
     */
    SpreadsheetControls.CellProtectionMenuButton = CompoundButton.extend({ constructor: function (docView, initOptions) {

        CompoundButton.call(this, docView, Utils.extendOptions({
            icon: 'fa-lock',
            label: gt('Cell protection'),
            tooltip: gt('Change cell protection for the selected cells'),
            autoHideGroups: true,
            smallerVersion: { hideLabel: true }
        }, initOptions));

        this.addGroup('cell/unlocked',      new Button(docView, { label: gt('Unlock cell'),    tooltip: gt('Unlock the cell in protected sheets') }));
        this.addGroup('cell/locked',        new Button(docView, { label: gt('Lock cell'),      tooltip: gt('Lock the cell in protected sheets') }));
        this.addGroup('cell/unhideformula', new Button(docView, { label: gt('Unhide formula'), tooltip: gt('Unhide the formula in protected sheets') }));
        this.addGroup('cell/hideformula',   new Button(docView, { label: gt('Hide formula'),   tooltip: gt('Hide the formula in protected sheets') }));

    } }); // class CellProtectionMenuButton

    // class DynamicSplitCheckBox =============================================

    /**
     * A check box control for a dynamic (movable) column/row split in a sheet.
     *
     * @constructor
     *
     * @extends CheckBox
     */
    SpreadsheetControls.DynamicSplitCheckBox = CheckBox.extend({ constructor: function (docView, initOptions) {

        CheckBox.call(this, docView, Utils.extendOptions({
            label: /*#. check box: split a spreadsheet into 2 or 4 different parts that can be scrolled independently */ gt.pgettext('sheet-split', 'Split sheet'),
            tooltip: gt.pgettext('sheet-split', 'Split the sheet above and left of the current cursor position')
        }, initOptions));

    } }); // class DynamicSplitCheckBox

    // class FrozenSplitCheckBox ==============================================

    /**
     * A check box control with a combined drop-down menu with different
     * settings for frozen columns and rows in a sheet.
     *
     * @constructor
     *
     * @extends CompoundCheckBox
     */
    SpreadsheetControls.FrozenSplitCheckBox = CompoundCheckBox.extend({ constructor: function (docView, initOptions) {

        CompoundCheckBox.call(this, docView, Utils.extendOptions({
            label: /*#. check box: split a spreadsheet into 2 or 4 different parts, the leading (left/upper) parts are frozen and cannot be scrolled */ gt.pgettext('sheet-split', 'Freeze sheet'),
            tooltip: gt.pgettext('sheet-split', 'Freeze the rows above and the columns left of the cursor')
        }, initOptions));

        this.addGroup('view/split/frozen/fixed', new Button(docView, { label: gt('Freeze first row'), tooltip: gt('Freeze the first visible row'), value: { cols: 0, rows: 1 } }));
        this.addGroup('view/split/frozen/fixed', new Button(docView, { label: gt('Freeze first column'), tooltip: gt('Freeze the first visible column'), value: { cols: 1, rows: 0 } }));
        this.addGroup('view/split/frozen/fixed', new Button(docView, { label: gt('Freeze first row and column'), tooltip: gt('Freeze the first visible row and the first visible column'), value: { cols: 1, rows: 1 } }));

    } }); // class FrozenSplitCheckBox

    // class ShowSheetButton ==================================================

    /**
     * A toggle button that allows to show or hide a sheet. The label of the
     * button will change automatically according to its current state.
     *
     * @constructor
     *
     * @extends Button
     */
    SpreadsheetControls.ShowSheetButton = Button.extend({ constructor: function (docView) {

        Button.call(this, docView, { toggle: true, highlight: _.constant(false) });

        this.registerUpdateHandler(function (state) {
            this.setLabel(state ? Labels.HIDE_SHEET_LABEL : Labels.SHOW_SHEET_LABEL);
        });

    } }); // class ShowSheetButton

    // class LockSheetButton ==================================================

    /**
     * A toggle button that allows to lock or unlock a sheet. The label of the
     * button will change automatically according to its current state.
     *
     * @constructor
     *
     * @extends Button
     */
    SpreadsheetControls.LockSheetButton = Button.extend({ constructor: function (docView) {

        Button.call(this, docView, { toggle: true, highlight: _.constant(false) });

        this.registerUpdateHandler(function (state) {
            this.setLabel(state ? Labels.UNPROTECT_SHEET_LABEL : Labels.PROTECT_SHEET_LABEL);
        });

    } }); // class LockSheetButton

    // class InsertAutoSumButton ==============================================

    /**
     * A button that allows to insert SUM formulas into the selected cells.
     *
     * @constructor
     *
     * @extends Button
     */
    SpreadsheetControls.InsertAutoSumButton = Button.extend({ constructor: function (docView, initOptions) {

        Button.call(this, docView, Utils.extendOptions({
            icon: 'docs-auto-sum',
            label: gt('Sum'),
            //#. automatically create a SUM function for selected cells
            tooltip: gt('Sum automatically'),
            value: 'SUM',
            smallerVersion: { css: { width: 35 }, hideLabel: true }
        }, initOptions));

    } }); // class InsertAutoSumButton

    // class InsertFunctionButton =============================================

    /**
     * A button that allows to insert a new function into the selected cell.
     *
     * @constructor
     *
     * @extends Button
     */
    SpreadsheetControls.InsertFunctionButton = Button.extend({ constructor: function (docView, initOptions) {

        Button.call(this, docView, Utils.extendOptions({
            icon: 'docs-functions',
            label: gt('Function'),
            //#. Insert a function at the current cursor location
            tooltip: gt('Insert function at cursor location'),
            smallerVersion: { css: { width: 35 }, hideLabel: true }
        }, initOptions));

    } }); // class InsertFunctionButton

    // class FilterButton =====================================================

    /**
     * A button that allows to toggle filtering drop-downs on a cell range.
     *
     * @constructor
     *
     * @extends Button
     */
    SpreadsheetControls.FilterButton = Button.extend({ constructor: function (docView, initOptions) {

        Button.call(this, docView, Utils.extendOptions({
            icon: 'docs-filter',
            //#. Button label: filter a cell range
            label: gt.pgettext('filter', 'Filter'),
            tooltip: gt('Filter the selected cells'),
            toggle: true,
            smallerVersion: { hideLabel: true }
        }, initOptions));

    } }); // class FilterButton

    // class RefreshButton ====================================================

    /**
     * A button that allows to refresh filtering and sorting in a table range.
     *
     * @constructor
     *
     * @extends Button
     */
    SpreadsheetControls.RefreshButton = Button.extend({ constructor: function (docView, initOptions) {

        Button.call(this, docView, Utils.extendOptions({
            icon: 'fa-refresh',
            //#. Button label: refresh a filtered cell range
            label: gt.pgettext('filter', 'Reapply'),
            tooltip: gt('Reapply the filter in the selected cells'),
            smallerVersion: { hideLabel: true }
        }, initOptions));

    } }); // class RefreshButton

    // class SheetTableStylePicker ============================================

    // an entry for the table style picker for a bare tyble style
    var NO_TABLE_STYLE_ITEM = {
        id: '',
        category: '',
        //#. name of a table stylesheet without formatting
        name: gt('No Style, No Grid'),
        priority: -1
    };

    /**
     * A style sheet picker for table ranges in spreadsheets.
     *
     * @constructor
     *
     * @extends TableStylePicker
     */
    SpreadsheetControls.SheetTableStylePicker = EditControls.TableStylePicker.extend({ constructor: function (docView) {

        EditControls.TableStylePicker.call(this, docView, {
            firstRow: true,
            bandsHor: true,
            additionalItems: [NO_TABLE_STYLE_ITEM]
        });

    } }); // class SheetTableStylePicker

    // exports ================================================================

    return SpreadsheetControls;

});
