/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/utils/interval', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/spreadsheet/utils/address'
], function (Utils, IteratorUtils, Address) {

    'use strict';

    // regular expression object matching a column interval
    var RE_COL_INTERVAL_NAME = new RegExp('^(' + Address.COL_PATTERN + '):(' + Address.COL_PATTERN + ')$', 'i');

    // regular expression object matching a row interval
    var RE_ROW_INTERVAL_NAME = new RegExp('^(' + Address.ROW_PATTERN + '):(' + Address.ROW_PATTERN + ')$');

    // class Interval =========================================================

    /**
     * Represents the position of an interval of columns or rows in a sheet of
     * a spreadsheet document (an 'index interval').
     *
     * @constructor
     *
     * @property {Number} first
     *  The zero-based index of the first column/row in the interval.
     *
     * @property {Number} last
     *  The zero-based index of the last column/row in the interval.
     *
     * @param {Number} first
     *  The zero-based index of the first column/row in the interval.
     *
     * @param {Number} [last]
     *  The zero-based index of the last column/row in the interval. Can be
     *  omitted to construct an index interval covering a single column or row.
     */
    function Interval(first, last) {

        this.first = first;
        this.last = (typeof last === 'number') ? last : first;

    } // class Interval

    // static methods ---------------------------------------------------------

    /**
     * Creates an index interval from the passed column/row indexes. The first
     * and last index in the resulting interval will be ordered automatically.
     *
     * @param {Number} index1
     *  The index of one of the borders in the index interval.
     *
     * @param {Number} index2
     *  The index of the other border in the index interval.
     *
     * @returns {Interval}
     *  The index interval with adjusted indexes.
     */
    Interval.create = function (index1, index2) {
        return new Interval(Math.min(index1, index2), Math.max(index1, index2));
    };

    /**
     * Returns the index interval for the passed string representation of a
     * column interval. The first and last index in the resulting interval will
     * be ordered automatically.
     *
     * @param {String} str
     *  The string representation of a column interval, e.g. 'A:D'.
     *
     * @param {Number} [max]
     *  If specified, the zero-based index of the largest allowed column index.
     *  If any of the parsed column indexes is greater, this method will return
     *  null.
     *
     * @returns {Interval|Null}
     *  The index interval, if the passed string is valid, otherwise null.
     */
    Interval.parseAsCols = function (str, max) {
        var matches = RE_COL_INTERVAL_NAME.exec(str);
        if (!matches) { return null; }
        var index1 = Address.parseCol(matches[1], max);
        var index2 = Address.parseCol(matches[2], max);
        return (Math.min(index1, index2) >= 0) ? Interval.create(index1, index2) : null;
    };

    /**
     * Returns the index interval for the passed string representation of a row
     * interval. The first and last index in the resulting interval will be
     * ordered automatically.
     *
     * @param {String} str
     *  The string representation of a row interval, e.g. '1:4'.
     *
     * @param {Number} [max]
     *  If specified, the zero-based index of the largest allowed row index. If
     *  any of the parsed row indexes is greater, this method will return null.
     *
     * @returns {Interval|Null}
     *  The index interval, if the passed string is valid, otherwise null.
     */
    Interval.parseAsRows = function (str, max) {
        var matches = RE_ROW_INTERVAL_NAME.exec(str);
        if (!matches) { return null; }
        var index1 = Address.parseRow(matches[1], max);
        var index2 = Address.parseRow(matches[2], max);
        return (Math.min(index1, index2) >= 0) ? Interval.create(index1, index2) : null;
    };

    /**
     * Returns the index interval for the passed string representation of a
     * column or row interval. The first and last index in the resulting
     * interval will be ordered automatically.
     *
     * @param {String} str
     *  The string representation of a column or row interval.
     *
     * @param {Boolean} columns
     *  Whether the passed string represents a column interval (true), or a row
     *  interval (false).
     *
     * @param {Number} [max]
     *  If specified, the zero-based index of the largest allowed column/row
     *  index. If any of the parsed indexes is greater, this method will return
     *  null.
     *
     * @returns {Interval|Null}
     *  The index interval, if the passed string is valid, otherwise null.
     */
    Interval.parseAs = function (str, columns, max) {
        return columns ? Interval.parseAsCols(str, max) : Interval.parseAsRows(str, max);
    };

    /**
     * Compares the passed index intervals. Defines a natural order for sorting
     * with the following behavior (let A and B be index intervals): If the
     * first index of A is less than the first index of B; or if both first
     * indexes are equal, and the last index of A is less than the last index
     * of B, then A is considered less than B.
     *
     * @param {Interval} interval1
     *  The first index interval to be compared to the second index interval.
     *
     * @param {Address} address2
     *  The second index interval to be compared to the first index interval.
     *
     * @returns {Number}
     *  - A negative number, if interval1 is less than interval2.
     *  - A positive number, if interval1 is greater than interval2.
     *  - Zero, if both index intervals are equal.
     */
    Interval.compare = function (interval1, interval2) {
        return Utils.comparePairs(interval1.first, interval1.last, interval2.first, interval2.last);
    };

    // public methods ---------------------------------------------------------

    /**
     * Returns a unique string key for this index interval that can be used for
     * example as key in an associative map. This method is faster than the
     * method Interval.toString().
     *
     * @returns {String}
     *  A unique string key for this index interval.
     */
    Interval.prototype.key = function () {
        return this.first + ':' + this.last;
    };

    /**
     * Creates a clone of this index interval.
     *
     * @returns {Interval}
     *  A clone of this index interval.
     */
    Interval.prototype.clone = function () {
        return new Interval(this.first, this.last);
    };

    /**
     * Returns whether the passed index intervals is equal to this interval.
     *
     * @param {Interval} interval
     *  The other index interval to be compared to this interval.
     *
     * @returns {Boolean}
     *  Whether both intervals contain the same column/row indexes.
     */
    Interval.prototype.equals = function (interval) {
        return (this.first === interval.first) && (this.last === interval.last);
    };

    /**
     * Returns whether the passed index intervals differs from this interval.
     *
     * @param {Interval} interval
     *  The other index interval to be compared to this interval.
     *
     * @returns {Boolean}
     *  Whether the passed index intervals differs from this interval.
     */
    Interval.prototype.differs = function (interval) {
        return (this.first !== interval.first) || (this.last !== interval.last);
    };

    /**
     * Compares this index interval with the passed index interval. See static
     * method Interval.compare() for details.
     *
     * @param {Interval} interval
     *  The other index interval to be compared to this index interval.
     *
     * @returns {Number}
     *  - A negative number, if this index interval is less than the passed
     *      index interval.
     *  - A positive number, if this index interval is greater than the passed
     *      index interval.
     *  - Zero, if both index intervals are equal.
     */
    Interval.prototype.compareTo = function (interval) {
        return Interval.compare(this, interval);
    };

    /**
     * Returns the number of columns/rows covered by this index interval.
     *
     * @returns {Number}
     *  The number of columns/rows covered by this index interval.
     */
    Interval.prototype.size = function () {
        return this.last - this.first + 1;
    };

    /**
     * Returns whether this index interval covers a single index (first and
     * last index are equal).
     *
     * @returns {Boolean}
     *  Whether this index interval covers a single index.
     */
    Interval.prototype.single = function () {
        return this.first === this.last;
    };

    /**
     * Returns whether this index interval contains the passed column/row
     * index.
     *
     * @param {Number} index
     *  The zero-based column/row index to be checked.
     *
     * @returns {Boolean}
     *  Whether this index interval contains the passed column/row index.
     */
    Interval.prototype.containsIndex = function (index) {
        return (this.first <= index) && (index <= this.last);
    };

    /**
     * Returns whether this index interval completely contains the passed index
     * interval.
     *
     * @param {Interval} interval
     *  The index interval to be checked.
     *
     * @returns {Boolean}
     *  Whether this index interval completely contains the passed index
     *  interval.
     */
    Interval.prototype.contains = function (interval) {
        return (this.first <= interval.first) && (interval.last <= this.last);
    };

    /**
     * Returns whether this index interval overlaps with the passed index
     * interval by at least one column/row.
     *
     * @param {Interval} interval
     *  The index interval to be checked.
     *
     * @returns {Boolean}
     *  Whether this index interval overlaps with the passed index interval, or
     *  single index.
     */
    Interval.prototype.overlaps = function (interval) {
        return (this.first <= interval.last) && (interval.first <= this.last);
    };

    /**
     * Returns the bounding interval of this index interval, and the passed
     * index interval (the smallest index interval that contains both
     * intervals).
     *
     * @param {Interval} interval
     *  The other index interval to create the bounding interval from.
     *
     * @returns {Interval}
     *  The bounding interval containing this and the passed index interval.
     */
    Interval.prototype.boundary = function (interval) {
        return new Interval(Math.min(this.first, interval.first), Math.max(this.last, interval.last));
    };

    /**
     * Returns the index interval covered by this index interval, and the
     * passed index interval.
     *
     * @param {Interval} interval
     *  The other index interval to be intersected with this index interval.
     *
     * @returns {Interval|Null}
     *  The index interval covered by this and the passed index interval, if
     *  existing; otherwise null.
     */
    Interval.prototype.intersect = function (interval) {
        var first = Math.max(this.first, interval.first),
            last = Math.min(this.last, interval.last);
        return (first <= last) ? new Interval(first, last) : null;
    };

    /**
     * Modifies both indexes of this index interval relatively.
     *
     * @param {Number} distance
     *  The distance to move the first and last index of this index interval.
     *  Negative values will decrease the indexes.
     *
     * @returns {Interval}
     *  A reference to this instance.
     */
    Interval.prototype.move = function (distance) {
        this.first += distance;
        this.last += distance;
        return this;
    };

    /**
     * Creates an iterator that visits the single indexes of this interval.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.reverse=false]
     *      If set to true, the indexes in this interval will be visited in
     *      reversed order.
     *
     * @returns {Object}
     *  An iterator object that implements the standard EcmaScript iterator
     *  protocol, i.e. it provides the method next() that returns a result
     *  object with the following properties:
     *  - {Boolean} done
     *      If set to true, the interval has been visited completely. No more
     *      indexes are available; this result object does not contain any
     *      other properties!
     *  - {Number} value
     *      The current index. This property will be omitted, if the iterator
     *      is done (see property 'done').
     */
    Interval.prototype.iterator = function (options) {
        return IteratorUtils.createIntervalIterator(this.first, this.last, options);
    };

    /**
     * Returns the string representation of this index interval as columns.
     *
     * @returns {String}
     *  The upper-case string representation of this index interval as columns.
     */
    Interval.prototype.stringifyAsCols = function () {
        return Address.stringifyCol(this.first) + ':' + Address.stringifyCol(this.last);
    };

    /**
     * Returns the string representation of this index interval as rows.
     *
     * @returns {String}
     *  The one-based string representation of this index interval as rows.
     */
    Interval.prototype.stringifyAsRows = function () {
        return Address.stringifyRow(this.first) + ':' + Address.stringifyRow(this.last);
    };

    /**
     * Returns the string representation of this index interval as columns, or
     * as rows.
     *
     * @param {Boolean} columns
     *  Whether to return the column representation (true), or the row
     *  representation (false) of the index interval.
     *
     * @returns {String}
     *  The string representation of this index interval.
     */
    Interval.prototype.stringifyAs = function (columns) {
        return columns ? this.stringifyAsCols() : this.stringifyAsRows();
    };

    /**
     * Returns the string representation of this index interval as rows.
     *
     * @returns {String}
     *  The one-based string representation of this index interval as rows.
     */
    Interval.prototype.toString = Interval.prototype.stringifyAsRows;

    /**
     * Returns the JSON representation of this index interval.
     *
     * @returns {Object}
     *  The JSON representation of this index interval (a plain JavaScript
     *  object with the properties 'first' and 'last').
     */
    Interval.prototype.toJSON = function () {
        return { first: this.first, last: this.last };
    };

    // exports ================================================================

    return _.makeExtendable(Interval);

});
