/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author York Richter <york.richter@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/tablestylecollection', [
    'io.ox/office/editframework/model/tablestylecollection',
    'io.ox/office/spreadsheet/utils/sheetutils'
], function (TableStyleCollection, SheetUtils) {

    'use strict';

    var WIDTH_THIN = 26;
    var WIDTH_MEDIUM = 53;
    var WIDTH_THICK = 79;

    var CHARACTER_LIGHT = { color: { type: 'scheme', value: 'light1' } };
    var CHARACTER_DARK = { color: { type: 'scheme', value: 'dark1' } };
    var CHARACTER_BOLD_LIGHT = { color: { type: 'scheme', value: 'light1' }, bold: true };
    var CHARACTER_BOLD_DARK = { color: { type: 'scheme', value: 'dark1' }, bold: true };

    var BORDER_THIN_LIGHT = { style: 'single', width: WIDTH_THIN, color: { type: 'scheme', value: 'light1' } };
    var BORDER_THIN_DARK = { style: 'single', width: WIDTH_THIN, color: { type: 'scheme', value: 'dark1' } };
    var BORDER_MEDIUM_LIGHT = { style: 'single', width: WIDTH_MEDIUM, color: { type: 'scheme', value: 'light1' } };
    var BORDER_MEDIUM_DARK = { style: 'single', width: WIDTH_MEDIUM, color: { type: 'scheme', value: 'dark1' } };
    var BORDER_THICK_LIGHT = { style: 'single', width: WIDTH_THICK, color: { type: 'scheme', value: 'light1' } };

    // class SheetTableStyleCollection ========================================

    /**
     * Contains the style sheets for table formatting attributes. The CSS
     * formatting will be written to table elements and their rows and cells.
     *
     * @constructor
     *
     * @extends TableStyleCollection
     *
     * @param {SpreadsheetModel} docModel
     *  The text document model containing instance.
     */
    var SheetTableStyleCollection = TableStyleCollection.extend({ constructor: function (docModel) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        TableStyleCollection.call(this, docModel, {
            poolId: SheetUtils.CELL_POOL_ID,
            families: 'cell paragraph character',
            rowBandsOverColBands: true, // row band attributes will be rendered over column band attributes
            expandColBandsToOuterCols: true, // merged column band attributes over active first/last columns
            restrictColBandsToInnerRows: true // column bands will not be drawn into header/footer rows
        });

        // private methods ----------------------------------------------------

        // TableStyleMedium23 - TableStyleMedium28 Zahl - 20 + 1 = colorValue
        // TableStyleMedium22 colorValue = 1
        function getTableStyleMedium23_28(colorSchema) {

            var BORDER = { style: 'single', width: WIDTH_THIN, color: { type: 'scheme', value: colorSchema, transformations: [{ type: 'tint', value: 60000 }] } },
                BAND_CELL = { fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'tint', value: 40000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderLeft: BORDER, borderRight: BORDER, borderTop: BORDER, borderBottom: BORDER, borderInsideHor: BORDER, borderInsideVert: BORDER },
                    cell: { fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'tint', value: 20000 }] } }
                },
                firstRow: { character: CHARACTER_BOLD_DARK },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: { style: 'single', width: WIDTH_MEDIUM, color: { type: 'scheme', value: colorSchema } } } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleMedium22() {

            var BAND_CELL = { fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 65000 }] } };

            var theme = {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderLeft: BORDER_THIN_DARK, borderRight: BORDER_THIN_DARK, borderTop: BORDER_THIN_DARK, borderBottom: BORDER_THIN_DARK, borderInsideHor: BORDER_THIN_DARK, borderInsideVert: BORDER_THIN_DARK },
                    cell: { fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 85000 }] } }
                },
                firstRow: { character: CHARACTER_BOLD_DARK },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: BORDER_MEDIUM_DARK } },
                firstCol: { character: CHARACTER_BOLD_LIGHT },
                lastCol: { character: CHARACTER_BOLD_LIGHT },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
            return theme;
        }

        // fillSchema = StyleZahl - 12
        function getTableStyleMedium21_16(colorSchema) {

            var BAND_CELL = { fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 85000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderTop: BORDER_MEDIUM_DARK, borderBottom: BORDER_MEDIUM_DARK }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: colorSchema }, borderBottom: BORDER_MEDIUM_DARK }
                },
                lastRow: {
                    character:  { color: { type: 'scheme', value: 'dark1' } },
                    cell: { borderTop: { style: 'double', width: WIDTH_THIN, color: { type: 'scheme', value: 'dark1' } } }
                },

                firstCol: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { fillColor: { type: 'scheme', value: colorSchema } }
                },
                lastCol: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { fillColor: { type: 'scheme', value: colorSchema } }
                },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleMedium15() {

            var BAND_CELL = { fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 85000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderLeft: BORDER_THIN_DARK, borderRight: BORDER_THIN_DARK, borderTop: BORDER_MEDIUM_DARK, borderBottom: BORDER_MEDIUM_DARK, borderInsideHor: BORDER_THIN_DARK, borderInsideVert: BORDER_THIN_DARK }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1' },
                        borderBottom: BORDER_THIN_DARK
                    }
                },
                lastRow: {
                    cell: { borderTop: { style: 'double', width: WIDTH_THIN, color: { type: 'scheme', value: 'dark1' } } }
                },
                firstCol: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1' }
                    }
                },
                lastCol: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1' }
                    }
                },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleMedium14_9(colorSchema) {

            var BAND_CELL = { fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'tint', value: 40000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    cell: {
                        fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'tint', value: 20000 }] }
                    },
                    table: { borderInsideHor: BORDER_THIN_LIGHT, borderInsideVert: BORDER_THIN_LIGHT
                    }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: colorSchema },
                        borderBottom: BORDER_THICK_LIGHT
                    }
                },
                lastRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: colorSchema },
                        borderTop: BORDER_THICK_LIGHT
                    }
                },

                firstCol: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { fillColor: { type: 'scheme', value: colorSchema } }
                },
                lastCol: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { fillColor: { type: 'scheme', value: colorSchema } }
                },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleMedium8() {

            var BAND_CELL = { fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 65000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    cell: {
                        fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 85000 }] }
                    },
                    table: { borderInsideHor: BORDER_THIN_LIGHT, borderInsideVert: BORDER_THIN_LIGHT }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1' },
                        borderBottom: BORDER_THICK_LIGHT
                    }
                },
                lastRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1' },
                        borderTop: BORDER_THICK_LIGHT
                    }
                },
                firstCol: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { fillColor: { type: 'scheme', value: 'light1' } }
                },
                lastCol: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { fillColor: { type: 'scheme', value: 'light1' } }
                },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleMedium7_2(colorSchema) {

            var BORDER = { style: 'single', width: WIDTH_THIN, color: { type: 'scheme', value: colorSchema, transformations: [{ type: 'tint', value: 60000 }] } },
                BAND_CELL = { fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'tint', value: 20000 }] }, borderTop: BORDER, borderBottom: BORDER };

            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderLeft: BORDER, borderRight: BORDER, borderTop: BORDER, borderBottom: BORDER, borderInsideHor: BORDER }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: colorSchema } }
                },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: { style: 'double', width: WIDTH_THIN, color: { type: 'scheme', value: colorSchema } } } },
                firstCol: { character: CHARACTER_BOLD_LIGHT },
                lastCol: { character: CHARACTER_BOLD_LIGHT },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleMedium1() {

            var BAND_CELL = { fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 85000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderLeft: BORDER_THIN_DARK, borderRight: BORDER_THIN_DARK, borderTop: BORDER_THIN_DARK, borderBottom: BORDER_THIN_DARK, borderInsideHor: BORDER_THIN_DARK }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: 'dark1' } }
                },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: { style: 'double', width: WIDTH_THIN, color: { type: 'scheme', value: 'dark1' } } } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleLight21_16(colorSchema) {

            var BORDER = { style: 'single', width: WIDTH_THIN, color: { type: 'scheme', value: colorSchema } },
                BAND_CELL = { fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'tint', value: 20000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderLeft: BORDER, borderRight: BORDER, borderTop: BORDER, borderBottom: BORDER, borderInsideHor: BORDER, borderInsideVert: BORDER }
                },
                firstRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderBottom: { style: 'single', type: 'scheme', color: { type: 'scheme', value: colorSchema }, width: WIDTH_MEDIUM } }
                },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: { style: 'double', width: WIDTH_THIN, color: { type: 'scheme', value: colorSchema } } } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleLight15() {

            var BAND_CELL = { fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 85000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderLeft: BORDER_THIN_DARK, borderRight: BORDER_THIN_DARK, borderTop: BORDER_THIN_DARK, borderBottom: BORDER_THIN_DARK, borderInsideHor: BORDER_THIN_DARK, borderInsideVert: BORDER_THIN_DARK }
                },
                firstRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderBottom: { style: 'single', type: 'scheme', color: { type: 'scheme', value: 'dark1' }, width: WIDTH_MEDIUM } }
                },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: { style: 'double', width: WIDTH_THIN, color: { type: 'scheme', value: 'dark1' } } } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleLight14_9(colorSchema) {

            var BORDER = { style: 'single', width: WIDTH_THIN, color: { type: 'scheme', value: colorSchema } },
                BAND = { cell: { borderTop: BORDER, borderBottom: BORDER } };
            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderLeft: BORDER, borderRight: BORDER, borderTop: BORDER, borderBottom: BORDER }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: colorSchema } }
                },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: { style: 'double', width: WIDTH_THIN, color: { type: 'scheme', value: colorSchema } } } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: BAND,
                band1Hor: BAND
            };
        }

        function getTableStyleLight8() {
            var BAND = { cell: { borderTop: BORDER_THIN_DARK, borderBottom: BORDER_THIN_DARK } };
            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderLeft: BORDER_THIN_DARK, borderRight: BORDER_THIN_DARK, borderTop: BORDER_THIN_DARK, borderBottom: BORDER_THIN_DARK }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: 'dark1' } }
                },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: { style: 'double', width: WIDTH_THIN, color: { type: 'scheme', value: 'dark1' } } } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: BAND,
                band1Hor: BAND
            };
        }

        function getTableStyleLight7_2(colorSchema) {

            var BORDER = { style: 'single', width: WIDTH_THIN, color: { type: 'scheme', value: colorSchema } },
                CHAR_BOLD = { color: { type: 'scheme', value: colorSchema, transformations: [{ type: 'shade', value: 75000 }] }, bold: true },
                BAND_CELL = { fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'tint', value: 20000 }] } };

            return {
                wholeTable: {
                    character: { color: { type: 'scheme', value: colorSchema, transformations: [{ type: 'shade', value: 75000 }] } },
                    table: { borderTop: BORDER, borderBottom: BORDER }
                },
                firstRow: {
                    character: CHAR_BOLD,
                    cell: { borderBottom: BORDER }
                },
                lastRow: {
                    character: CHAR_BOLD,
                    cell: { borderTop: { style: 'single', width: WIDTH_THIN, color: { type: 'scheme', value: colorSchema } } } },
                firstCol: { character: CHAR_BOLD },
                lastCol: { character: CHAR_BOLD },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleLight1() {

            var BAND_CELL =  { fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 85000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_DARK,
                    table: { borderTop: BORDER_THIN_DARK, borderBottom: BORDER_THIN_DARK }
                },
                firstRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderBottom: BORDER_THIN_DARK }
                },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: BORDER_THIN_DARK }
                },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleDark11_9(colorSchema1, colorSchema2) {

            var BAND_CELL = { fillColor: { type: 'scheme', value: colorSchema1, transformations: [{ type: 'tint', value: 40000 }] } };

            return {
                wholeTable: {
                    cell: { fillColor: { type: 'scheme', value: colorSchema1, transformations: [{ type: 'tint', value: 20000 }] } }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: colorSchema2 } }
                },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: { style: 'double', width: WIDTH_THIN, color: { type: 'scheme', value: 'dark1' } } } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleDark8() {

            var BAND_CELL = { fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 65000 }] } };

            return {
                wholeTable: {
                    cell: { fillColor: { type: 'scheme', value: 'light1', transformations: [{ type: 'shade', value: 85000 }] } }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: 'dark1' } }
                },
                lastRow: {
                    character: CHARACTER_BOLD_DARK,
                    cell: { borderTop: { style: 'double', width: WIDTH_THIN, color: { type: 'scheme', value: 'dark1' } } } },
                firstCol: { character: CHARACTER_BOLD_DARK },
                lastCol: { character: CHARACTER_BOLD_DARK },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleDark7_5(colorSchema) {

            var BAND_CELL = { fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'shade', value: 75000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: colorSchema } }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1' },
                        borderBottom: BORDER_MEDIUM_LIGHT
                    }
                },
                lastRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'shade', value: 50000 }] },
                        borderTop: BORDER_MEDIUM_LIGHT
                    }
                },
                firstCol: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'shade', value: 75000 }] },
                        borderRight: BORDER_MEDIUM_LIGHT
                    }
                },
                lastCol: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'shade', value: 75000 }] },
                        borderLeft: BORDER_MEDIUM_LIGHT
                    }
                },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleDark4() {

            var BAND_CELL = { fillColor: { type: 'scheme', value: 'accent3', transformations: [{ type: 'shade', value: 75000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: 'accent3' } }
                },
                firstRow: {
                    character: CHARACTER_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'accent6' },
                        borderBottom: BORDER_MEDIUM_LIGHT
                    }
                },
                lastRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        cell: { fillColor: { type: 'scheme', value: 'dark1' } },
                        borderTop: { style: 'single', width: WIDTH_MEDIUM, color: { type: 'scheme', value: 'accent3', transformations: [{ type: 'shade', value: 50000 }] } }
                    }
                },
                firstCol: {
                    character: CHARACTER_BOLD_DARK,
                    cell: {
                        cell: { fillColor: { type: 'scheme', value: 'accent3', transformations: [{ type: 'shade', value: 75000 }] } },
                        borderRight: BORDER_MEDIUM_LIGHT
                    }
                },
                lastCol: {
                    character: CHARACTER_BOLD_DARK,
                    cell: {
                        cell: { fillColor: { type: 'scheme', value: 'accent3', transformations: [{ type: 'shade', value: 75000 }] } },
                        borderLeft: BORDER_MEDIUM_LIGHT
                    }
                },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleDark3_2(colorSchema) {

            var BAND_CELL = { fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'shade', value: 75000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: colorSchema } }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1' },
                        borderBottom: BORDER_MEDIUM_LIGHT
                    }
                },
                lastRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'shade', value: 50000 }] },
                        borderTop: BORDER_MEDIUM_LIGHT
                    }
                },
                firstCol: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'shade', value: 75000 }] },
                        borderRight: BORDER_MEDIUM_LIGHT
                    }
                },
                lastCol: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: colorSchema, transformations: [{ type: 'shade', value: 75000 }] },
                        borderLeft: BORDER_MEDIUM_LIGHT
                    }
                },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function getTableStyleDark1() {

            var BAND_CELL = { fillColor: { type: 'scheme', value: 'dark1', transformations: [{ type: 'tint', value: 75000 }] } };

            return {
                wholeTable: {
                    character: CHARACTER_LIGHT,
                    cell: { fillColor: { type: 'scheme', value: 'dark1', transformations: [{ type: 'tint', value: 55000 }] } }
                },
                firstRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1' },
                        borderBottom: BORDER_MEDIUM_LIGHT
                    }
                },
                lastRow: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1', transformations: [{ type: 'tint', value: 85000 }] },
                        borderTop: BORDER_MEDIUM_LIGHT
                    }
                },
                firstCol: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1', transformations: [{ type: 'tint', value: 75000 }] },
                        borderRight: BORDER_MEDIUM_LIGHT
                    }
                },
                lastCol: {
                    character: CHARACTER_BOLD_LIGHT,
                    cell: {
                        fillColor: { type: 'scheme', value: 'dark1', transformations: [{ type: 'tint', value: 75000 }] },
                        borderLeft: BORDER_MEDIUM_LIGHT
                    }
                },
                band1Vert: { cell: BAND_CELL },
                band1Hor: { cell: BAND_CELL }
            };
        }

        function insertMissingTableStyle(styleId, styleName, uiPriority, category, attrs) {
            // check by style sheet name (they are always loaded in English from Excel, while ste style IDs are localized)
            if (!self.containsStyleSheetByName(styleName)) {
                self.createDirtyStyleSheet(styleId, styleName, attrs, { priority: uiPriority, category: category });
            } else {
                self.setStyleOptions(styleId, { category: category, priority: uiPriority });
            }
        }

        function insertMissingTableStyleGroup(baseStyleId, baseStyleName, uiPriority, startIndex, category, generateAttributesFunc) {

            for (var index = 1; index <= 6; index += 1) {
                insertMissingTableStyle(baseStyleId + startIndex, baseStyleName + ' ' + startIndex, uiPriority, category, generateAttributesFunc('accent' + index));
                startIndex++;
            }
        }

        function insertMissingTableStyleGroupDark1(baseStyleId, baseStyleName, uiPriority, category, generateAttributesFunc) {
            var nameIndex = 9;
            for (var index = 1; index <= 6; index += 2) {
                insertMissingTableStyle(baseStyleId + nameIndex, baseStyleName + ' ' + nameIndex, uiPriority, category, generateAttributesFunc('accent' + index, 'accent' + (index + 1)));
                nameIndex++;
            }
        }

        function insertMissingTableStyleGroupDark2(baseStyleId, baseStyleName, uiPriority, category, generateAttributesFunc) {
            for (var index = 4; index <= 6; index += 1) {
                insertMissingTableStyle(baseStyleId + (index + 1), baseStyleName + ' ' + (index + 1), uiPriority, category, generateAttributesFunc('accent' + index));
            }
        }

        function insertMissingTableStyleGroupDark3(baseStyleId, baseStyleName, uiPriority, category, generateAttributesFunc) {
            for (var index = 1; index <= 2; index += 1) {
                insertMissingTableStyle(baseStyleId + (index + 1), baseStyleName + ' ' + (index + 1), uiPriority, category, generateAttributesFunc('accent' + index));
            }
        }

        // public methods -----------------------------------------------------

        this.insertMissingTableStyles = function () {
            insertMissingTableStyle('TableStyleLight15', 'Table Style Light 15', 72, 'Light', getTableStyleLight15());
            insertMissingTableStyleGroup('TableStyleLight', 'Table Style Light', 72, 16, 'Light', getTableStyleLight21_16);
            insertMissingTableStyle('TableStyleLight8', 'Table Style Light 8', 72, 'Light', getTableStyleLight8());
            insertMissingTableStyleGroup('TableStyleLight', 'Table Style Light', 72, 9, 'Light', getTableStyleLight14_9);
            insertMissingTableStyle('TableStyleLight1', 'Table Style Light 1', 72, 'Light', getTableStyleLight1());
            insertMissingTableStyleGroup('TableStyleLight', 'Table Style Light', 72, 2, 'Light', getTableStyleLight7_2);

            insertMissingTableStyle('TableStyleMedium1', 'Table Style Medium 1', 71, 'Medium', getTableStyleMedium1());
            insertMissingTableStyleGroup('TableStyleMedium', 'Table Style Medium', 71, 2, 'Medium', getTableStyleMedium7_2);
            insertMissingTableStyle('TableStyleMedium8', 'Table Style Medium 8', 71, 'Medium', getTableStyleMedium8());
            insertMissingTableStyleGroup('TableStyleMedium', 'Table Style Medium', 71, 9, 'Medium', getTableStyleMedium14_9);
            insertMissingTableStyle('TableStyleMedium15', 'Table Style Medium 15', 71, 'Medium', getTableStyleMedium15());
            insertMissingTableStyleGroup('TableStyleMedium', 'Table Style Medium', 71, 16, 'Medium', getTableStyleMedium21_16);
            insertMissingTableStyle('TableStyleMedium22', 'Table Style Medium 22', 71, 'Medium', getTableStyleMedium22());
            insertMissingTableStyleGroup('TableStyleMedium', 'Table Style Medium', 71, 23, 'Medium', getTableStyleMedium23_28);

            insertMissingTableStyle('TableStyleDark1', 'Table Style Dark 1', 73, 'Dark', getTableStyleDark1());
            insertMissingTableStyleGroupDark3('TableStyleDark', 'Table Style Dark', 73, 'Dark', getTableStyleDark3_2);
            insertMissingTableStyle('TableStyleDark4', 'Table Style Dark 4', 73, 'Dark', getTableStyleDark4());
            insertMissingTableStyleGroupDark2('TableStyleDark', 'Table Style Dark', 73, 'Dark', getTableStyleDark7_5);
            insertMissingTableStyle('TableStyleDark8', 'Table Style Dark 8', 73, 'Dark', getTableStyleDark8());
            insertMissingTableStyleGroupDark1('TableStyleDark', 'Table Style Dark', 73, 'Dark', getTableStyleDark11_9);
        };

        // initialization -----------------------------------------------------

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = self = null;
        });

    } }); // class SheetTableStyleCollection

    // exports ================================================================

    return SheetTableStyleCollection;

});
