/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/model', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/tk/container/valueset',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/editframework/model/viewattributesmixin',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/textframework/model/editor',
    'io.ox/office/textframework/model/listhandlermixin',
    'io.ox/office/textframework/model/updatelistsmixin',
    'io.ox/office/spreadsheet/utils/config',
    'io.ox/office/spreadsheet/utils/operations',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/operationcontext',
    'io.ox/office/spreadsheet/model/operationgenerator',
    'io.ox/office/spreadsheet/model/numberformatter',
    'io.ox/office/spreadsheet/model/namecollection',
    'io.ox/office/spreadsheet/model/listcollection',
    'io.ox/office/spreadsheet/model/sheetmodel',
    'io.ox/office/spreadsheet/model/modelattributesmixin',
    'io.ox/office/spreadsheet/model/formula/parser/formularesource',
    'io.ox/office/spreadsheet/model/formula/parser/formulagrammar',
    'io.ox/office/spreadsheet/model/formula/parser/formulaparser',
    'io.ox/office/spreadsheet/model/formula/interpret/formulacompiler',
    'io.ox/office/spreadsheet/model/formula/interpret/formulainterpreter',
    'io.ox/office/spreadsheet/model/formula/deps/dependencymanager'
], function (Utils, IteratorUtils, ValueSet, ValueMap, LocaleData, ViewAttributesMixin, DOMUtils, TextBaseModel, ListHandlerMixin, UpdateListsMixin, Config, Operations, SheetUtils, SheetOperationContext, SheetOperationGenerator, NumberFormatter, NameCollection, ListCollection, SheetModel, ModelAttributesMixin, FormulaResource, FormulaGrammar, FormulaParser, FormulaCompiler, FormulaInterpreter, DependencyManager) {

    'use strict';

    // convenience shortcuts
    var SheetType = SheetUtils.SheetType;
    var Address = SheetUtils.Address;
    var Interval = SheetUtils.Interval;
    var Range = SheetUtils.Range;
    var RangeArray = SheetUtils.RangeArray;
    var Range3DArray = SheetUtils.Range3DArray;

    // definitions for document view attributes
    var VIEW_ATTRIBUTE_DEFINITIONS = {

        /**
         * Index of the active (visible) sheet in the document.
         */
        activeSheet: {
            def: -1,
            validate: function (sheet) {
                return ((sheet >= -1) && (sheet < this.getSheetCount())) ? sheet : Utils.BREAK;
            }
        },

        /**
         * Whether to show cell references in formulas in R1C1 notation.
         */
        rcStyle: {
            def: false
        }
    };

    // maps sheet event names to different document event names
    var SHEET_TO_DOCUMENT_EVENTS = {
        'change:attributes': 'change:sheet:attributes',
        'change:viewattributes': 'change:sheet:viewattributes'
    };

    // the names of document operations that modify text contents in drawing objects
    var DRAWING_TEXT_OPERATIONS = Utils.makeSet([
        Operations.SET_ATTRIBUTES,
        Operations.DELETE,
        Operations.MOVE,
        Operations.TEXT_INSERT,
        Operations.TAB_INSERT,
        Operations.HARDBREAK_INSERT,
        Operations.PARA_INSERT,
        Operations.PARA_SPLIT,
        Operations.PARA_MERGE,
        Operations.INSERT_LIST,
        Operations.DELETE_LIST
    ]);

    // whether to let each 'insertSheet' operation fail in debug mode
    var DEBUG_FAILING_IMPORT = Config.getDebugUrlFlag('spreadsheet:failing-import');

    // class SpreadsheetModel =================================================

    /**
     * Represents a 'workbook', the spreadsheet document model. Implements
     * execution of all supported operations.
     *
     * Triggers the events supported by the base class EditModel, and the
     * following additional events:
     *  - 'insert:sheet:before':
     *      Before a new sheet will be inserted into the document. The event
     *      handler receives the zero-based index of the new sheet it will have
     *      in the collection of all sheets.
     *  - 'insert:sheet:after':
     *      After a new sheet has been inserted into the document. The event
     *      handler receives the zero-based index of the new sheet in the
     *      collection of all sheets, and the new sheet model instance.
     *  - 'delete:sheet:before':
     *      Before a sheet will be removed from the document. The event handler
     *      receives the zero-based index of the sheet in the collection of all
     *      sheets, and the model instance of the sheet to be removed.
     *  - 'delete:sheet:after':
     *      After a sheet has been removed from the document. The event handler
     *      receives the zero-based index of the sheet in the collection of all
     *      sheets before it has been removed.
     *  - 'move:sheet:before':
     *      Before a sheet will be moved to a new position. The event handler
     *      receives the zero-based index of the old position in the collection
     *      of all sheets, the zero-based index of the new position, and the
     *      model instance of the moved sheet.
     *  - 'move:sheet:after':
     *      After a sheet has been moved to a new position. The event handler
     *      receives the zero-based index of the old position in the collection
     *      of all sheets, the zero-based index of the new position, and the
     *      model instance of the moved sheet.
     *  - 'transform:sheet':
     *      After one of the events 'insert:sheet:after', 'delete:sheet:after',
     *      or 'move:sheet:after'. The event handler receives the zero-based
     *      index of the old position (or null for inserted sheets), and the
     *      zero-based index of the new sheet (or null for deleted sheets).
     *  - 'rename:sheet':
     *      After a sheet has been renamed. The event handler receives the
     *      zero-based index of the renamed sheet in the collection of all
     *      sheets, the new name of the sheet, and the sheet model instance.
     * - 'change:viewattributes'
     *      After at least one view attribute of the document has been changed.
     *      Event handlers receive an incomplete attribute map containing all
     *      changed view attributes with their new values.
     *
     * Additionally, events of all sheets will be forwarded to the own model
     * listeners. The sheet event 'change:attributes' will be triggered as
     * 'change:sheet:attributes', and the sheet event 'change:viewattributes'
     * will be triggered as 'change:sheet:viewattributes'. The second parameter
     * passed to all event listeners will be the sheet index (following the
     * jQuery event object). The actual sheet event parameters will be appended
     * to the parameters of the own model listeners. Example: The sheet event
     * 'insert:columns' with the parameters (event, interval) will be triggered
     * as document model event 'insert:columns' with the parameters (event,
     * sheet, interval).
     *
     * The events of the own collection of defined names will be triggered as
     * well as the same events of the name collections of all sheets. The
     * second parameter passed to the event listeners will be either null for
     * globally defined names, or will be the sheet index for sheet-locally
     * defined names.
     *
     * Events will always be triggered, also during importing the document.
     *
     * @constructor
     *
     * @extends TextBaseModel
     * @extends ModelAttributesMixin
     * @extends ViewAttributesMixin
     *
     * @param {SpreadsheetApplication} app
     *  The application containing this document model.
     */
    var SpreadsheetModel = TextBaseModel.extend({ constructor: function (app) {

        // self reference
        var self = this;

        // the file format identifier of the spreadsheet document
        var fileFormat = app.getFileFormat();

        // all existing sheets and their names, by zero-based index
        var sheetsCollection = [];

        // sheet indexes, mapped by sheet UID
        var sheetIndexMap = new ValueMap();

        // models of all table ranges in all sheets, mapped by upper-case table names
        var tableModelSet = new ValueSet('getKey()');

        // the formula grammars used by this document (no ownership)
        var formulaGrammarMap = new ValueMap();

        // the number formatter for this document
        var numberFormatter = null;

        // dependencies of all formulas in the document
        var dependencyManager = null;

        // the formula parser for this document
        var formulaParser = null;

        // the formula compiler for this document
        var formulaCompiler = null;

        // the formula interpreter for this document
        var formulaInterpreter = null;

        // the collection of globally defined names
        var nameCollection = null;

        // the collection of globally defined (user-)lists
        var listCollection = null;

        // the maximum address in a sheet
        var maxAddress = Address.A1.clone();

        // cached operations that will be sent to the server when generating operations the next time
        var sendOperationsCache = [];

        // the root container for DOM models of all drawing objects in all sheets
        var pageRootNode = null;

        // the content node of the page node (child of page node, direct parent of the sheet container nodes)
        var pageContentNode = null;

        // base constructors --------------------------------------------------

        TextBaseModel.call(this, app, {
            contextClass: SheetOperationContext,
            selectionStateHandler: selectionStateHandler,
            slideMode: true
        });

        ViewAttributesMixin.call(this, VIEW_ATTRIBUTE_DEFINITIONS);
        ListHandlerMixin.call(this, app);
        UpdateListsMixin.call(this, app);

        // private methods ----------------------------------------------------

        /**
         * Updates the maximum available column and row index.
         */
        function updateMaxColRow(cols, rows) {
            maxAddress[0] = cols - 1;
            maxAddress[1] = rows - 1;
        }

        /**
         * Returns the sheet info structure at the specified index.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Object|Null}
         *  The info structure of the sheet at the passed index or with the
         *  passed name, containing the sheet name and the reference to the
         *  sheet model instance, or null if the passed index or name is
         *  invalid.
         */
        function getSheetInfo(sheet) {
            sheet = _.isNumber(sheet) ? sheet : _.isString(sheet) ? self.getSheetIndex(sheet) : -1;
            return ((sheet >= 0) && (sheet < sheetsCollection.length)) ? sheetsCollection[sheet] : null;
        }

        /**
         * Returns an error code if the passed sheet name is not valid (empty,
         * or with invalid characters, or already existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet (case-insensitive).
         *
         * @param {Number} [exceptSheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {String|Null}
         *  The value null, if the sheet name is valid; otherwise one of the
         *  following error codes:
         *  - 'sheet:name:empty': The passed sheet name is empty.
         *  - 'sheet:name:invalid': The passed sheet name contains invalid
         *      characters.
         *  - 'sheet:name:used': The passed sheet name is already used in the
         *      document.
         */
        function validateSheetName(sheetName, exceptSheet) {

            // an existing sheet with the passed name
            var sheetModelByName = self.getSheetModel(sheetName);
            // the model of the specified sheet
            var sheetModelByIndex = _.isNumber(exceptSheet) ? self.getSheetModel(exceptSheet) : null;

            // name must not be empty
            if (sheetName.length === 0) {
                return 'sheet:name:empty';
            }

            // name must not start nor end with an apostrophe (embedded apostrophes
            // are allowed) and must not contain any NPCs or other invalid characters
            if (/^'|[\x00-\x1f\x80-\x9f[\]*?:/\\]|'$/.test(sheetName)) {
                return 'sheet:name:invalid';
            }

            // name must not be used already for another sheet
            if (_.isObject(sheetModelByName) && (!_.isObject(sheetModelByIndex) || (sheetModelByName !== sheetModelByIndex))) {
                return 'sheet:name:used';
            }

            return null;
        }

        /**
         * Returns whether the passed sheet name is valid (non-empty, no
         * invalid characters, and not existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet (case-insensitive).
         *
         * @param {Number} [exceptSheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {Boolean}
         *  Whether the sheet name is valid and not used.
         */
        function isValidSheetName(sheetName, exceptSheet) {
            return validateSheetName(sheetName, exceptSheet) === null;
        }

        /**
         * Returns an error code if the passed sheet name is not valid (empty,
         * or with invalid characters, or already existing in the document).
         *
         * @param {String} sheetName
         *  The desired name of the sheet (case-insensitive).
         *
         * @param {Number} [exceptSheet]
         *  If specified, the zero-based index of the sheet that is allowed to
         *  contain the passed name (used while renaming an existing sheet).
         *
         * @returns {jQuery.Promise}
         *  A resolved promise, if the passed sheet name is valid and not used
         *  in the document; or a rejected promise, if the sheet name is not
         *  valid. The promise will be rejected with a result object with a
         *  'cause' property set to an error code as returned by the method
         *  validateSheetName().
         */
        function ensureValidSheetName(sheetName, exceptSheet) {
            var result = validateSheetName(sheetName, exceptSheet);
            return (result === null) ? $.when() : SheetUtils.makeRejected(result);
        }

        /**
         * Creates and inserts a new empty sheet into this document, and
         * triggers the appropriate document events.
         *
         * @param {Number} sheet
         *  The insertion index of the new sheet. MUST be valid (not negative,
         *  and not greater than the number of existing sheets).
         *
         * @param {SheetType} sheetType
         *  The type identifier of the new sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. MUST NOT be used yet in this document.
         *
         * @param {Object} [attributes]
         *  The initial sheet attributes.
         *
         * @returns {SheetModel|Null}
         *  The model instance of the new sheet; or null, if no more sheets can
         *  be inserted into this document.
         */
        function createSheetAndTrigger(sheet, sheetType, sheetName, attributes) {

            // first, check maximum sheet count
            if (sheetsCollection.length === Config.MAX_SHEET_COUNT) { return null; }

            // notify all listeners before insertion
            self.trigger('insert:sheet:before', sheet);

            // calculate the new active sheet index (-1 if it will not change)
            var activeSheet = self.getActiveSheet();
            activeSheet = (sheet <= activeSheet) ? (activeSheet + 1) : -1;

            // invalidate active sheet index while inserting a sheet before it
            if (activeSheet >= 0) {
                self.setActiveSheet(-1);
            }

            // create the new sheet model
            var sheetModel = new SheetModel(self, sheetType, attributes);
            var sheetUid = sheetModel.getUid();
            // the DOM container node for drawing frames
            var containerNode = sheetModel.getDrawingCollection().getContainerNode();

            // insert the sheet model descriptor into the collection
            sheetsCollection.splice(sheet, 0, { model: sheetModel, uid: sheetUid, name: sheetName, node: containerNode });
            // insert the DOM container node for drawing frames into the page node
            pageContentNode.insertBefore(containerNode, pageContentNode.childNodes.item(sheet));

            // clear the sheet index cache unless the sheet has been appended
            if (sheet === sheetsCollection.length) {
                sheetIndexMap.insert(sheetUid, sheet);
            } else {
                sheetIndexMap.clear();
            }

            // update active sheet index, if the sheet has been inserted before
            if (activeSheet >= 0) {
                self.setActiveSheet(activeSheet);
            }

            // notify all listeners after insertion
            self.trigger('insert:sheet:after', sheet, sheetModel);
            self.trigger('transform:sheet', null, sheet);

            // forward all events of the new sheet to the listeners of this document model
            self.listenTo(sheetModel, 'triggered', function (event, type) {
                var triggerType = SHEET_TO_DOCUMENT_EVENTS[type] || type;
                self.trigger.apply(self, [triggerType, sheetModel.getIndex()].concat(_.toArray(arguments).slice(2)));
            });

            return sheetModel;
        }

        /**
         * Callback handler for the document operation 'insertSheet'. Creates
         * and inserts a new sheet in this spreadsheet document.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'insertSheet' document operation.
         */
        function applyInsertSheetOperation(context) {

            // the sheet index passed in the operation
            var sheet = context.getInt('sheet');
            // the sheet name passed in the operation
            var sheetName = context.getStr('sheetName');
            // the sheet type passed in the operation
            var sheetType = context.getOptEnum('type', SheetType, SheetType.WORKSHEET);
            // the sheet attributes passed in the operation (optional)
            var attributes = context.getOptObj('attrs');

            // let the 'insertSheet' operation fail on purpose if specified in URL options
            context.ensure(!DEBUG_FAILING_IMPORT, 'debug mode: insert sheet fails');

            // check the sheet index, name (must not exist yet), and type
            context.ensure((sheet >= 0) && (sheet <= sheetsCollection.length), 'invalid sheet index');
            context.ensure(isValidSheetName(sheetName), 'invalid sheet name');

            // create and insert the new sheet
            var sheetModel = createSheetAndTrigger(sheet, sheetType, sheetName, attributes);
            context.ensure(sheetModel, 'cannot create sheet');
        }

        /**
         * Callback handler for the document operation 'deleteSheet'. Deletes
         * an existing sheet from this spreadsheet document.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'deleteSheet' document operation.
         *
         * @param {SheetModel} sheetModel
         *  The sheet model to be deleted.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet model.
         */
        function applyDeleteSheetOperation(context, sheetModel, sheet) {

            // there must remain at least one sheet in the document
            context.ensure(sheetsCollection.length > 1, 'cannot delete last sheet');

            // notify all listeners before deletion
            self.trigger('delete:sheet:before', sheet, sheetModel);

            // calculate the new active sheet index (-1 if it will not change)
            var activeSheet = self.getActiveSheet();
            if ((sheet < activeSheet) || ((sheet === activeSheet) && (activeSheet + 1 === sheetsCollection.length))) {
                activeSheet -= 1; // sheet before active sheet will be deleted (or deleted active sheet is the last sheet)
            } else if (activeSheet < sheet) {
                activeSheet = -1; // sheet after active sheet will be deleted
            }

            // invalidate active sheet index while deleting a sheet before it
            if (activeSheet >= 0) {
                self.setActiveSheet(-1);
            }

            // clear the sheet index cache unless the sheet was the last
            if (sheet + 1 === sheetsCollection.length) {
                sheetIndexMap.remove(sheetModel.getUid());
            } else {
                sheetIndexMap.clear();
            }

            // remove the DOM container node for drawing frames from the page node
            pageContentNode.removeChild(sheetsCollection[sheet].node);

            // remove the sheet from the collection and destroy it
            sheetsCollection.splice(sheet, 1);
            sheetModel.destroy();

            // update active sheet index, if a sheet before the active sheet has been deleted
            // (find a visible sheet, if the sheet at the specified position is hidden)
            if (activeSheet >= 0) {
                self.setActiveSheet(self.findVisibleSheet(activeSheet));
            }

            // notify all listeners after deletion
            self.trigger('delete:sheet:after', sheet);
            self.trigger('transform:sheet', sheet, null);
        }

        /**
         * Callback handler for the document operation 'setSheetName'. Renames
         * an existing sheet in this spreadsheet document.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'setSheetName' document operation.
         *
         * @param {SheetModel} sheetModel
         *  The sheet model to be renamed.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet model.
         */
        function applyRenameSheetOperation(context, sheetModel, sheet) {

            // the sheet object
            var sheetInfo = getSheetInfo(sheet);

            // the sheet name passed in the operation (no other sheet must contain this name)
            var sheetName = context.getStr('sheetName');
            context.ensure(isValidSheetName(sheetName, sheet), 'invalid sheet name');

            // rename the sheet (no event, if passed name matches old name)
            if (sheetInfo.name !== sheetName) {
                sheetInfo.name = sheetName;
                self.trigger('rename:sheet', sheet, sheetName, sheetModel);
            }
        }

        /**
         * Callback handler for the document operation 'moveSheet'. Moves an
         * existing sheet in this document to a new position.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'moveSheet' document operation.
         *
         * @param {SheetModel} sheetModel
         *  The source sheet model to be moved.
         *
         * @param {Number} fromSheet
         *  The zero-based index of the source sheet model.
         */
        function applyMoveSheetOperation(context, sheetModel, fromSheet) {

            // the target position of the sheet
            var toSheet = context.getInt('to');
            context.ensure((toSheet >= 0) && (toSheet <= sheetsCollection.length), 'invalid target index');

            // notify all listeners before moving
            self.trigger('move:sheet:before', fromSheet, toSheet, sheetModel);

            // calculate the new active sheet index
            var oldSheet = self.getActiveSheet();
            var newSheet = Utils.transformArrayIndex(oldSheet, fromSheet, toSheet);

            // invalidate active sheet index while moving a sheet
            if (oldSheet !== newSheet) {
                self.setActiveSheet(-1);
            }

            // move the sheet in the collection
            var sheetInfo = sheetsCollection[fromSheet];
            sheetsCollection.splice(fromSheet, 1);
            sheetsCollection.splice(toSheet, 0, sheetInfo);

            // move the DOM container node for drawing frames to its new position
            pageContentNode.removeChild(sheetInfo.node);
            pageContentNode.insertBefore(sheetInfo.node, pageContentNode.childNodes.item(toSheet));

            // clear the sheet index cache
            sheetIndexMap.clear();

            // update the active sheet index
            if (oldSheet !== newSheet) {
                self.setActiveSheet(newSheet);
            }

            // notify all listeners after moving
            self.trigger('move:sheet:after', fromSheet, toSheet, sheetModel);
            self.trigger('transform:sheet', fromSheet, toSheet);
        }

        /**
         * Callback handler for the document operation 'copySheet'. Creates a
         * copy of an existing sheet in this spreadsheet document, and inserts
         * that sheet to a new position.
         *
         * @param {SheetOperationContext} context
         *  A wrapper representing the 'copySheet' document operation.
         *
         * @param {SheetModel} sheetModel
         *  The source sheet model to be copied.
         */
        function applyCopySheetOperation(context, sheetModel) {

            // the target position for the new sheet
            var toSheet = context.getInt('to');
            context.ensure((toSheet >= 0) && (toSheet <= sheetsCollection.length), 'invalid target index');

            // the new sheet name passed in the operation (no existing sheet must contain this name)
            var sheetName = context.getStr('sheetName');
            context.ensure(isValidSheetName(sheetName), 'invalid sheet name');

            // insert a new sheet model into the collection, clone all contents from the existing sheet
            var cloneModel = createSheetAndTrigger(toSheet, sheetModel.getType(), sheetName);
            context.ensure(cloneModel, 'cannot create sheet');
            cloneModel.applyCopySheetOperation(context, sheetModel);
        }

        /**
         * Callback function to save and restore the document selection during
         * undo and redo operations.
         */
        function selectionStateHandler(selectionState) {

            // no argument passed: return current selection state
            if (arguments.length === 0) {
                var sheet = self.getActiveSheet();
                var activeSheetModel = self.getSheetModel(sheet);
                return activeSheetModel ? { sheet: sheet, selection: activeSheetModel.getViewAttribute('selection') } : null;
            }

            // otherwise: restore the passed selection state
            var sheetModel = self.getSheetModel(selectionState.sheet);
            if (sheetModel) {
                self.setActiveSheet(selectionState.sheet);
                sheetModel.setViewAttribute('selection', selectionState.selection);
            }
        }

        /**
         * Sends the own selection settings to remote clients.
         */
        var sendActiveSelectionDebounced = this.createDebouncedMethod('SpreadsheetModel.sendActiveSelectionDebounced', null, function () {

            // do not send anything while settings are invalid
            var selectionState = selectionStateHandler();
            if (!selectionState) { return; }

            // convert selected ranges to strings
            app.updateUserData({
                sheet: selectionState.sheet,
                ranges: formulaParser.formatRangeList('op', selectionState.selection.ranges),
                drawings: selectionState.selection.drawings
            });

        }, { delay: 250 });

        // operation handler registration -------------------------------------

        /**
         * Registers a public method of a model collection as callback function
         * for the specified document operation.
         *
         * @param {String} name
         *  The name of a document operation to be handled by the specified
         *  collection method.
         *
         * @param {BaseObject} collection
         *  An arbitrary model collection of this document model.
         *
         * @param {String} method
         *  The name of a public method of the specified collection. The method
         *  will receive the operation context (instance of OperationContext)
         *  as first parameter.
         */
        function registerCollectionMethod(name, collection, method) {
            self.registerContextOperationHandler(name, function (context) {
                collection[method](context);
            });
        }

        /**
         * Registers a callback function for the specified document operation
         * addressing an existing sheet in this document. The operation must
         * contain a property 'sheet' with the index of an existing sheet,
         * unless the operation can be used to address this document model as
         * well as any sheet model (see option 'allowDoc' below).
         *
         * @param {String} name
         *  The name of a document operation to be handled by the passed
         *  callback function.
         *
         * @param {Function} handler
         *  The callback function that will be invoked for every operation with
         *  the specified name, and a property 'sheet' referring to an existing
         *  sheet. Receives the following parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *  (2) {SheetModel|SpreadsheetModel} targetModel
         *      The model of the sheet addressed by the operation. May be this
         *      document model instead of a sheet model, if the operation
         *      property 'sheet' is missing, and the operation is allowed to
         *      address the document (option 'allowDoc' has been set to true).
         *  (3) {Number|Null} sheet
         *      The zero-based index of the sheet addressed by the operation,
         *      or null, if the operation addresses this document model.
         *  Will be called in the context of this document model instance.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.allowDoc=false]
         *      If set to true, the operation property 'sheet' can be omitted
         *      to allow to address the entire document with the operation.
         */
        function registerSheetOperationHandler(name, handler, options) {
            var allowDocModel = Utils.getBooleanOption(options, 'allowDoc', false);
            return self.registerContextOperationHandler(name, function (context) {
                var isDocModel = allowDocModel && !context.has('sheet');
                var sheet = isDocModel ? null : context.getInt('sheet');
                var targetModel = isDocModel ? self : self.getSheetModel(sheet);
                context.ensure(targetModel, 'missing sheet');
                return handler.call(self, context, targetModel, sheet);
            });
        }

        /**
         * Registers a public method of a sheet model as callback function for
         * the specified document operation. The operation must contain a
         * property 'sheet' with the index of an existing sheet, unless the
         * operation can be used to address this document model as well as any
         * sheet model (see option 'allowDoc' below).
         *
         * @param {String} name
         *  The name of a document operation to be handled by a sheet model.
         *
         * @param {String} method
         *  The name of a public instance method of the sheet model addressed
         *  by the operation. Receives the following parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.allowDoc=false]
         *      If set to true, the operation property 'sheet' can be omitted
         *      to allow to address the entire document with the operation.
         */
        function registerSheetModelMethod(name, method, options) {
            registerSheetOperationHandler(name, function (context, targetModel) {
                targetModel[method](context);
            }, options);
        }

        /**
         * Registers a public method of a collection contained by a sheet model
         * as callback function for the specified document operation. The
         * operation must contain a property 'sheet' with the index of an
         * existing sheet, unless the operation can be used to address this
         * document model as well as any sheet model (see option 'allowDoc'
         * below).
         *
         * @param {String} name
         *  The name of a document operation to be handled by a collection of a
         *  sheet model.
         *
         * @param {String} getter
         *  The name of a public instance method of the sheet model returning
         *  one of its collection instances, e.g. 'getCellCollection'.
         *
         * @param {String} method
         *  The name of a public instance method of the collection returned by
         *  the sheet model addressed by the operation. Receives the following
         *  parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.allowDoc=false]
         *      If set to true, the operation property 'sheet' can be omitted
         *      to allow to address the entire document with the operation. The
         *      collection getter method will be invoked on this document model
         *      instance.
         */
        function registerSheetCollectionMethod(name, getter, method, options) {
            registerSheetOperationHandler(name, function (context, targetModel) {
                targetModel[getter]()[method](context);
            }, options);
        }

        /**
         * Registers a callback function for the specified document operation
         * addressing a drawing model in a specific sheet of this spreadsheet
         * document. The operation MUST contain a property 'start' (non-empty
         * array of integers) with the position of the drawing object in the
         * document. The position may point to a non-existing drawing (e.g. to
         * insert a new drawing), but it must point to an existing sheet (the
         * first array element is the sheet index).
         *
         * @param {String} name
         *  The name of a document operation for drawing objects to be handled
         *  by the passed callback function.
         *
         * @param {Function} handler
         *  The callback function that will be invoked for every operation with
         *  the specified name, and a property 'start' referring to an existing
         *  sheet. Receives the following parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *  (2) {SheetDrawingCollection} drawingCollection
         *      The drawing collection of the sheet addressed by the operation.
         *  (3) {Array<Number>} position
         *      The position of the drawing object. This is the value of the
         *      property 'start' from the operation without the first array
         *      element (which was the sheet index).
         *  Will be called in the context of this document model instance.
         */
        function registerDrawingOperationHandler(name, handler) {
            return self.registerContextOperationHandler(name, function (context) {

                // the entire start position of the drawing object, and its (optional) embedded target
                var position = context.getPos('start');
                context.ensure(position.length >= 2, 'invalid drawing start position');

                // resolve the sheet model (first array element of the position)
                var sheetModel = self.getSheetModel(position[0]);
                context.ensure(sheetModel, 'invalid sheet index');

                // invoke the operation handler with shortened start position (no leading sheet index)
                return handler.call(self, context, sheetModel.getDrawingCollection(), position.slice(1));
            });
        }

        /**
         * Registers a public method of the drawing collection of a sheet as
         * callback function for the specified drawing operation. See the
         * description of the method registerDrawingOperationHandler() for more
         * details about drawing collections.
         *
         * @param {String} name
         *  The name of a document operation for drawing objects to be handled
         *  by a drawing collection.
         *
         * @param {String} method
         *  The name of a public instance method of the drawing collection of
         *  the sheet model addressed by the operation. Receives the following
         *  parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *  (2) {Array<Number>} position
         *      The drawing position (the property 'start' from the operation
         *      without the first array element which was the sheet index).
         */
        function registerDrawingCollectionMethod(name, method) {
            registerDrawingOperationHandler(name, function (context, collection, position) {
                collection[method](context, position);
            });
        }

        /**
         * Registers a public method of a drawing model as callback function
         * for the specified drawing operation. See the description of the
         * method registerDrawingOperationHandler() for more details about
         * drawing collections.
         *
         * @param {String} name
         *  The name of a document operation for drawing objects to be handled
         *  by the addressed drawing model.
         *
         * @param {String} method
         *  The name of a public instance method of the drawing model addressed
         *  by the operation. Receives the following parameters:
         *  (1) {SheetOperationContext} context
         *      The operation context wrapping the JSON operation object.
         *  (2) {Array<Number>|Null} startPos
         *      The start position of the embedded child component inside the
         *      drawing object. Used e.g. for manipulating text contents of
         *      shape objects and text frames. This parameter will be null, if
         *      the value of the option 'childPos' is false.
         *  (3) {Array<Number>|Null} endPos
         *      The end position of the embedded child component inside the
         *      drawing object. Used e.g. for manipulating text contents of
         *      shape objects and text frames. This parameter will be null, if
         *      the value of the option 'childPos' is false, or if the document
         *      operation does not contain the property 'end'.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.type]
         *      If specified, the drawing model addressed by the operation must
         *      be of the specified type. If a drawing model with another type
         *      exists at the specified document position, the operation will
         *      fail.
         */
        function registerDrawingModelMethod(name, method, options) {

            // the expected type of the drawing object addressed by the operation
            var type = Utils.getStringOption(options, 'type', null);

            registerDrawingOperationHandler(name, function (context, collection, position) {

                // find the drawing model in the drawing collection
                var modelDesc = collection.getModelDescriptor(position);
                context.ensure(modelDesc, 'invalid drawing position');

                // check the type of the drawing object if specified
                context.ensure(!type || (type === modelDesc.model.getType()), 'invalid drawing type');

                // check that the operation does not address embedded text contents
                context.ensure(modelDesc.remaining.length === 0, 'invalid drawing start position');

                // call the operation handler (instance method of the drawing model)
                modelDesc.model[method](context);
            });
        }

        // protected methods --------------------------------------------------

        /**
         * Post-processing of the document, after all import operations have
         * been applied successfully.
         *
         * @internal
         *  Called from the application import process. MUST NOT be called from
         *  external code.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the document has been
         *  post-processed successfully; or rejected when the document is
         *  invalid, or any other error has occurred.
         */
        this.postProcessImport = function () {

            // document must contain at least one sheet (returning a rejected Deferred
            // object will cause invocation of the prepareInvalidDocument() method)
            if (sheetsCollection.length === 0) { return $.Deferred().reject(); }

            // create all built-in cell styles not imported from the file
            this.getCellStyles().createMissingStyles();

            // bug 46465: parse all cell formulas in a timer loop to prevent script warnings
            return this.iterateArraySliced(sheetsCollection, function (sheetInfo) {
                return sheetInfo.model.postProcessImport();
            }, 'SpreadsheetModel.postProcessImport');
        };

        /**
         * Inserts a new table model into the global map of all table models.
         *
         * @internal
         *  Will be called from the constructor of the class TableModel only!
         *
         * @param {TableModel} tableModel
         *  The model instance of a table range inserted into a sheet of this
         *  document.
         */
        this._registerTableModel = function (tableModel) {
            if (!tableModel.isAutoFilter()) {
                tableModelSet.insert(tableModel);
            }
        };

        /**
         * Removes a deleted table model from the global map of all table
         * models.
         *
         * @internal
         *  Will be called from the destructor of the class TableModel only!
         *
         * @param {TableModel} tableModel
         *  The model instance of a table range deleted from a sheet of this
         *  document.
         */
        this._unregisterTableModel = function (tableModel) {
            if (!tableModel.isAutoFilter()) {
                tableModelSet.remove(tableModel);
            }
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the number formatter of this document.
         *
         * @returns {NumberFormatter}
         *  The number formatter of this document.
         */
        this.getNumberFormatter = function () {
            return numberFormatter;
        };

        /**
         * Returns the localized formula resource data of this document.
         *
         * @returns {FormulaResource}
         *  The localized formula resource data of this document.
         */
        this.getFormulaResource = function () {
            // FormulaResource.create() caches and reuses existing instances
            return FormulaResource.create(fileFormat, LocaleData.LOCALE);
        };

        /**
         * Returns the formula grammar of this document for the specified
         * grammar identifier.
         *
         * @param {String} grammarId
         *  The identifier of the formula grammar, in one of the following
         *  forms:
         *      - '<GRAMMAR>',
         *      - '<GRAMMAR>:<FILEFORMAT>',
         *      - '<GRAMMAR>!<REFSTYLE>',
         *      - '<GRAMMAR>:<FILEFORMAT>!<REFSTYLE>'.
         *  The token <GRAMMAR> must be one of the formula grammar identifiers
         *  accepted by the method FormulaGrammar.create(), e.g. the grammar
         *  'op' used in document operations, or the grammar 'ui' used in the
         *  user interface.
         *  The optional token <FILEFORMAT> specifies the file format of the
         *  formula grammar, e.g. 'ooxml' for the grammars used by OOXML
         *  documents, or 'odf' used by ODF documents. If omitted, the current
         *  file format of the document will be used.
         *  The optional token <REFSTYLE> specifies the style of cell range
         *  references. It must be set to either 'a1' for A1 style, or to 'rc'
         *  for R1C1 style. If omitted, the current reference style of the
         *  document (as imported from the document file) will be used.
         *
         * @returns {FormulaGrammar}
         *  The formula grammar of this document for the passed identifier.
         */
        this.getFormulaGrammar = function (grammarId) {
            return formulaGrammarMap.getOrCreate(grammarId, function () {

                // try to extract an explicit reference style, e.g. 'ui!rc'
                var parts1 = grammarId.split('!');
                var rcStyle = parts1[1] ? (parts1[1] === 'rc') : this.getViewAttribute('rcStyle');

                // try to extract an explicit file format identifier, e.g. 'op:ooxml'
                var parts2 = parts1[0].split(':');
                var format = parts2[1] || fileFormat;

                // FormulaGrammar.create() caches and reuses existing instances
                return FormulaGrammar.create(parts2[0], format, rcStyle);
            }, this);
        };

        /**
         * Returns the formula parser of this document.
         *
         * @returns {FormulaParser}
         *  The formula parser of this document.
         */
        this.getFormulaParser = function () {
            return formulaParser;
        };

        /**
         * Returns the formula compiler of this document.
         *
         * @returns {FormulaCompiler}
         *  The formula compiler of this document.
         */
        this.getFormulaCompiler = function () {
            return formulaCompiler;
        };

        /**
         * Returns the formula interpreter of this document.
         *
         * @returns {FormulaInterpreter}
         *  The formula interpreter of this document.
         */
        this.getFormulaInterpreter = function () {
            return formulaInterpreter;
        };

        /**
         * Returns the collection of all global defined names contained in this
         * document.
         *
         * @returns {NameCollection}
         *  The collection of all defined names in this document.
         */
        this.getNameCollection = function () {
            return nameCollection;
        };

        /**
         * Returns the collection of all global defined (user-)lists contained in this
         * document.
         *
         * @returns {ListCollection}
         *  The collection of all defined (user-)lists in this document.
         */
        this.getListCollection = function () {
            return listCollection;
        };

        /**
         * Returns the formula dependency manager of this document.
         *
         * @returns {DependencyManager}
         *  The formula dependency manager of this document.
         */
        this.getDependencyManager = function () {
            return dependencyManager;
        };

        // sheet range --------------------------------------------------------

        /**
         * Returns the column index of the rightmost cells in a sheet.
         *
         * @returns {Number}
         *  The column index of the rightmost cells in a sheet.
         */
        this.getMaxCol = function () {
            return maxAddress[0];
        };

        /**
         * Returns the row index of the bottom cells in a sheet.
         *
         * @returns {Number}
         *  The row index of the bottom cells in a sheet.
         */
        this.getMaxRow = function () {
            return maxAddress[1];
        };

        /**
         * Returns the maximum column or row index in a sheet.
         *
         * @param {Boolean} columns
         *  Whether to return the maximum column index (true), or the maximum
         *  row index (false).
         *
         * @returns {Number}
         *  The maximum column or row index in a sheet.
         */
        this.getMaxIndex = function (columns) {
            return maxAddress.get(columns);
        };

        /**
         * Returns the maximum cell address in a sheet.
         *
         * @returns {Address}
         *  The maximum cell address in a sheet.
         */
        this.getMaxAddress = function () {
            return maxAddress.clone();
        };

        /**
         * Returns the range address of the entire area in a sheet, from the
         * top-left cell to the bottom-right cell.
         *
         * @returns {Range}
         *  The range address of an entire sheet.
         */
        this.getSheetRange = function () {
            return Range.createFromAddresses(Address.A1, maxAddress);
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * columns in the active sheet.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire columns.
         */
        this.isColRange = function (range) {
            return (range.start[1] === 0) && (range.end[1] === maxAddress[1]);
        };

        /**
         * Returns the address of the cell range covering the specified column
         * interval.
         *
         * @param {Interval|Number} interval
         *  The column interval, or a single zero-based column index.
         *
         * @returns {Range}
         *  The address of the column range.
         */
        this.makeColRange = function (interval) {
            var isIndex = typeof interval === 'number';
            return Range.create(isIndex ? interval : interval.first, 0, isIndex ? interval : interval.last, maxAddress[1]);
        };

        /**
         * Returns the addresses of the cell ranges covering the specified
         * column intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of column intervals, or a single column interval.
         *
         * @returns {RangeArray}
         *  The addresses of the cell ranges specified by the column intrevals.
         */
        this.makeColRanges = function (intervals) {
            return RangeArray.map(intervals, this.makeColRange.bind(this));
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * rows in the active sheet.
         *
         * @param {Range} range
         *  A cell range address.
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire rows.
         */
        this.isRowRange = function (range) {
            return (range.start[0] === 0) && (range.end[0] === maxAddress[0]);
        };

        /**
         * Returns the address of the cell range covering the specified row
         * interval.
         *
         * @param {Interval|Number} interval
         *  The row interval, or a single zero-based row index.
         *
         * @returns {Range}
         *  The address of the row range.
         */
        this.makeRowRange = function (interval) {
            var isIndex = typeof interval === 'number';
            return Range.create(0, isIndex ? interval : interval.first, maxAddress[0], isIndex ? interval : interval.last);
        };

        /**
         * Returns the addresses of the cell ranges covering the specified row
         * intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of row intervals, or a single row interval.
         *
         * @returns {RangeArray}
         *  The addresses of the cell ranges specified by the row intrevals.
         */
        this.makeRowRanges = function (intervals) {
            return RangeArray.map(intervals, this.makeRowRange.bind(this));
        };

        /**
         * Returns whether the passed range address covers one or more entire
         * columns or rows in the active sheet.
         *
         * @param {Range} range
         *  A range address.
         *
         * @param {Boolean} columns
         *  Whether to check for a full column range (true), or for a full row
         *  range (false).
         *
         * @returns {Boolean}
         *  Whether the passed range address covers one or more entire columns
         *  or rows.
         */
        this.isFullRange = function (range, columns) {
            return columns ? this.isColRange(range) : this.isRowRange(range);
        };

        /**
         * Returns the address of the cell range covering the specified column
         * or row interval.
         *
         * @param {Interval|Number} interval
         *  The column or row interval, or a single column or row index.
         *
         * @param {Boolean} columns
         *  Whether to create a full column range (true), or a full row range
         *  (false).
         *
         * @returns {Range}
         *  The address of the full column or row range.
         */
        this.makeFullRange = function (interval, columns) {
            return columns ? this.makeColRange(interval) : this.makeRowRange(interval);
        };

        /**
         * Returns the addresses of the cell ranges covering the specified
         * column or row intervals.
         *
         * @param {IntervalArray|Interval} intervals
         *  An array of index intervals, or a single index interval.
         *
         * @param {Boolean} columns
         *  Whether to create column ranges (true), or row ranges (false).
         *
         * @returns {RangeArray}
         *  The addresses of the cell ranges specified by the row intrevals.
         */
        this.makeFullRanges = function (intervals, columns) {
            return columns ? this.makeColRanges(intervals) : this.makeRowRanges(intervals);
        };

        /**
         * Returns whether the passed cell address contains valid column and
         * row indexes (inside the limits of the sheets in this document).
         *
         * @param {Address} address
         *  The cell address to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed cell address contains valid column and row
         *  indexes.
         */
        this.isValidAddress = function (address) {
            return (address[0] >= 0) && (address[0] <= maxAddress[0]) && (address[1] >= 0) && (address[1] <= maxAddress[1]);
        };

        /**
         * Returns whether the passed cell range address contains valid start
         * and end addresses (inside the limits of the sheets in this document)
         * that are in correct order (start before end).
         *
         * @param {Range} range
         *  The cell range address to be checked.
         *
         * @returns {Boolean}
         *  Whether the passed cell range address contains valid start and end
         *  addresses in correct order.
         */
        this.isValidRange = function (range) {
            return (range.start[0] >= 0) && (range.start[0] <= range.end[0]) && (range.end[0] <= maxAddress[0]) &&
                (range.start[1] >= 0) && (range.start[1] <= range.end[1]) && (range.end[1] <= maxAddress[1]);
        };

        /**
         * Converts the passed JSON data to a column/row interval (an instance
         * of the class Interval) if possible.
         *
         * @param {Any} json
         *  A value that will be converted to a column/row interval. To create
         *  an interval object, the data MUST be an object with the property
         *  'first' containing a valid column/row index (non-negative, less
         *  than or equal to the maximum index of this document). Optionally,
         *  the object may contain a property 'last' containing a valid index
         *  (non-negative, equal to or greater than 'first', less than or equal
         *  to the maximum index of this document).
         *
         * @param {Boolean} columns
         *  Whether to convert the passed JSON data to a column interval
         *  (true), or to a row interval (false).
         *
         * @param {Boolean} [operationMode=false]
         *  If set to true, the JSON interval is expected to contain the
         *  properties 'start' and 'end' (as used in document operations),
         *  instead of 'first' and 'last'.
         *
         * @returns {Interval|Null}
         *  A column/row interval object, if the passed JSON data can be
         *  converted to an interval; otherwise null.
         */
        this.createInterval = function (json, columns, operationMode) {

            // data must be an object
            if (!_.isObject(json)) { return null; }

            var firstName = operationMode ? 'start' : 'first';
            var lastName = operationMode ? 'end' : 'last';
            var maxIndex = this.getMaxIndex(columns);

            // 'first' property must be a valid index
            var first = json[firstName];
            if (!Utils.isFiniteNumber(first) || (first < 0) || (first > maxIndex)) { return null; }

            // 'last' property may be missing
            if (!(lastName in json)) { return new Interval(first); }

            // 'last' property must be a valid index following 'first'
            var last = json[lastName];
            if (!Utils.isFiniteNumber(last) || (last < first) || (last > maxIndex)) { return null; }

            return new Interval(first, last);
        };

        /**
         * Converts the passed JSON data to a cell address object (an instance
         * of the class Address) if possible.
         *
         * @param {Any} json
         *  A value that will be converted to a cell address. To create a cell
         *  address object, the data MUST be an array with two elements. Each
         *  element must be a finite number. The first array element MUST be a
         *  valid column index (non-negative, less than or equal to the maximum
         *  column index of this document). The second array element MUST be a
         *  valid row index (non-negative, less than or equal to the maximum
         *  row index of this document).
         *
         * @returns {Address|Null}
         *  A cell address object, if the passed JSON data can be converted to
         *  a cell address; otherwise null.
         */
        this.createAddress = function (json) {

            // data must be an array with exactly two elements
            if (!_.isArray(json) || (json.length !== 2)) { return null; }

            // elements must valid column/row indexes
            if (!Utils.isFiniteNumber(json[0]) || (json[0] < 0) || (json[0] > maxAddress[0])) { return null; }
            if (!Utils.isFiniteNumber(json[1]) || (json[1] < 0) || (json[1] > maxAddress[1])) { return null; }

            return new Address(json[0], json[1]);
        };

        /**
         * Converts the passed JSON data to a cell range address object (an
         * instance of the class Range) if possible.
         *
         * @param {Any} json
         *  A value that will be converted to a cell range address. To create a
         *  cell range address, the data MUST be an object with the property
         *  'start', containing a valid JSON representation of the start cell
         *  address of the range (see method SpreadsheetModel.createAddress()
         *  for details), and an optional property 'end' with another JSON cell
         *  address for the end of the range (defaults to 'start').
         *
         * @returns {Range|Null}
         *  A cell range address object, if the passed JSON data can be
         *  converted to a cell range address; otherwise null.
         */
        this.createRange = function (json) {

            // data must be an object
            if (!_.isObject(json)) { return null; }

            // 'start' property must be a valid address
            var start = this.createAddress(json.start);
            if (!start) { return null; }

            // 'end' property may be missing
            if (!('end' in json)) { return new Range(start); }

            // 'end' property must be a valid address following the start address
            var end = this.createAddress(json.end);
            if (!end || (end[0] < start[0]) || (end[1] < start[1])) { return null; }

            return new Range(start, end);
        };

        /**
         * Converts the passed JSON data to an array of cell range addresses
         * (an instance of the class RangeArray) if possible.
         *
         * @param {Any} json
         *  A value that will be converted to an array of cell range addresses.
         *  To create an array, the data MUST be a non-empty array with valid
         *  JSON representations of cell range addresses (see method
         *  SpreadsheetModel.createRange() for details).
         *
         * @returns {RangeArray|Null}
         *  An array of cell range addresses, if the passed JSON data can be
         *  converted to such an array; otherwise null.
         */
        this.createRangeArray = function (json) {

            // data must be a non-empty array
            if (!_.isArray(json) || (json.length === 0)) { return null; }

            // process array elements (must be valid ranges), early exit on first invalid range
            var ranges = new RangeArray();
            json.every(function (range) {
                range = this.createRange(range);
                if (range) { ranges.push(range); return true; }
            }, this);

            // ensure that all ranges have been converted successfully
            return (json.length === ranges.length) ? ranges : null;
        };

        /**
         * Returns a new range that has been moved by the specified distance,
         * but always keeping the range inside the sheet limits (see method
         * SpreadsheetModel.getCroppedMovedRange() for an alternative keeping
         * the specified move distance intact).
         *
         * @param {Range} range
         *  The address of the range to be moved.
         *
         * @param {Number} cols
         *  The number of columns the range will be moved. Negative values will
         *  move the range towards the leading border of the sheet, positive
         *  values will move the range towards the trailing border.
         *
         * @param {Number} rows
         *  The number of rows the range will be moved. Negative values will
         *  move the range towards the top border of the sheet, positive values
         *  will move the range towards the bottom border.
         *
         * @returns {Range}
         *  The address of the moved range. If the move distances passed to
         *  this method are too large, the range will only be moved to the
         *  sheet borders.
         */
        this.getBoundedMovedRange = function (range, cols, rows) {

            // reduce move distance to keep the range address valid
            cols = Utils.minMax(cols, -range.start[0], maxAddress[0] - range.end[0]);
            rows = Utils.minMax(rows, -range.start[1], maxAddress[1] - range.end[1]);

            // return the new range
            return Range.create(range.start[0] + cols, range.start[1] + rows, range.end[0] + cols, range.end[1] + rows);
        };

        /**
         * Returns a new range that has been moved by the specified distance.
         * If the moved range leaves the sheet area, it will be cropped at the
         * sheet borders (see method SpreadsheetModel.getBoundedMovedRange()
         * for an alternative keeping the original size of the range intact).
         *
         * @param {Range} range
         *  The address of the range to be moved.
         *
         * @param {Number} cols
         *  The number of columns the range will be moved. Negative values will
         *  move the range towards the leading border of the sheet, positive
         *  values will move the range towards the trailing border.
         *
         * @param {Number} rows
         *  The number of rows the range will be moved. Negative values will
         *  move the range towards the top border of the sheet, positive values
         *  will move the range towards the bottom border.
         *
         * @returns {Range|Null}
         *  The address of the moved range. If the specified move distances are
         *  too large, the range will be cropped to the sheet area. If the
         *  range has been moved completely outside the sheet, this method will
         *  return null.
         */
        this.getCroppedMovedRange = function (range, cols, rows) {

            // the address components of the moved range
            var col1 = Math.max(0, range.start[0] + cols);
            var row1 = Math.max(0, range.start[1] + rows);
            var col2 = Math.min(maxAddress[0], range.end[0] + cols);
            var row2 = Math.min(maxAddress[1], range.end[1] + rows);

            // return null, if the resulting range becomes invalid
            return ((col1 <= col2) && (row1 <= row2)) ? Range.create(col1, row1, col2, row2) : null;
        };

        // sheets -------------------------------------------------------------

        /**
         * Returns the maximum length of a sheet name according to the current
         * file format.
         *
         * @returns {Number}
         *  The maximum length of a sheet name.
         */
        this.getMaxSheetNameLength = function () {
            return app.isOOXML() ? 31 : 65535;
        };

        /**
         * Returns whether a sheet at the specified index or with the specified
         * case-insensitive name exists in the spreadsheet document.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {Boolean}
         *  Whether a sheet exists at the passed index or with the passed name.
         */
        this.hasSheet = function (sheet) {
            return getSheetInfo(sheet) !== null;
        };

        /**
         * Returns the exact name of the sheet at the specified index or with
         * the specified case-insensitive name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {String|Null}
         *  The name of the sheet at the passed index or with the passed name,
         *  or null if the passed index or name is invalid.
         */
        this.getSheetName = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return sheetInfo ? sheetInfo.name : null;
        };

        /**
         * Returns the current index of the sheet with the specified name.
         *
         * @param {String} sheetName
         *  The case-insensitive sheet name.
         *
         * @returns {Number}
         *  The current zero-based index of the sheet with the specified name,
         *  or -1 if a sheet with the passed name does not exist.
         */
        this.getSheetIndex = function (sheetName) {
            sheetName = sheetName.toLowerCase();
            return Utils.findFirstIndex(sheetsCollection, function (sheetInfo) {
                return sheetInfo.name.toLowerCase() === sheetName;
            });
        };

        /**
         * Returns the current sheet index of the passed sheet model instance.
         *
         * @param {SheetModel|String} sheetModel
         *  The sheet model whose index will be returned, or the UID of a sheet
         *  model.
         *
         * @returns {Number}
         *  The current zero-based sheet index of the passed sheet model, or -1
         *  if the passed value does not refer to a sheet model existing in
         *  this document.
         */
        this.getSheetIndexOfModel = function (sheetModel) {
            var sheetUid = sheetModel.getUid ? sheetModel.getUid() : sheetModel;
            return sheetIndexMap.getOrCreate(sheetUid, function () {
                return Utils.findFirstIndex(sheetsCollection, function (sheetInfo) {
                    return sheetInfo.uid === sheetUid;
                });
            });
        };

        /**
         * Returns the model instance of the sheet at the specified index or
         * with the specified name.
         *
         * @param {Number|String} sheet
         *  The zero-based index of the sheet, or the case-insensitive sheet
         *  name.
         *
         * @returns {SheetModel|Null}
         *  The model instance for the sheet at the passed index or with the
         *  specified name, or null if the passed index or name is invalid.
         */
        this.getSheetModel = function (sheet) {
            var sheetInfo = getSheetInfo(sheet);
            return sheetInfo ? sheetInfo.model : null;
        };

        /**
         * Returns the number of sheets in the spreadsheet document.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.supported=false]
         *      If set to true, only the sheets with a type supported by the
         *      spreadsheet application will be counted. See description of the
         *      method SheetModel.isSupportedType() for details.
         *  @param {Boolean} [options.cells=false]
         *      If set to true, only sheets that contain cells will be counted
         *      (i.e. chart sheets and dialog sheets will be skipped). By
         *      default, all sheet models will be counted regardless of their
         *      type.
         *  @param {Boolean|Null} [options.visible=null]
         *      If set to a boolean value, only the visible sheets (if set to
         *      true), or only the hidden sheets (if set to false) will be
         *      counted. By default, all sheet models in this document will be
         *      counted, regardless of their visibility.
         *
         * @returns {Number}
         *  The number of sheets in the document.
         */
        this.getSheetCount = function (options) {

            // whether to filter for supported sheet types
            var supported = Utils.getBooleanOption(options, 'supported', false);
            // whether to filter for sheets with cells
            var cells = Utils.getBooleanOption(options, 'cells', false);
            // whether to filter for visible/hidden sheets
            var visible = Utils.getBooleanOption(options, 'visible', null);

            // fast path for the number of all sheets
            if (!supported && !cells && (visible === null)) {
                return sheetsCollection.length;
            }

            // count the matching sheets
            var iterator = this.createSheetIterator({ supported: supported, cells: cells, visible: visible });
            return IteratorUtils.reduce(0, iterator, function (count) { return count + 1; });
        };

        /**
         * Creates an iterator that visits specific sheet models in this
         * document.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Number} [options.first]
         *      The zero-based index of the first sheet model to be visited. If
         *      omitted, uses the first sheet in the document.
         *  @param {Number} [options.last]
         *      The zero-based index of the last sheet model to be visited. If
         *      omitted, uses the last sheet in the document.
         *  @param {Boolean} [options.reverse=false]
         *      If set to true, the sheets in the specified interval will be
         *      visited in reversed order, from 'last' to 'first'.
         *  @param {Boolean} [options.supported=false]
         *      If set to true, only the sheets with a type supported by the
         *      spreadsheet application will be visited. See description of the
         *      method SheetModel.isSupportedType() for details.
         *  @param {Boolean} [options.cells=false]
         *      If set to true, only sheets that contain cells will be visited
         *      (i.e. chart sheets and dialog sheets will be skipped). By
         *      default, all sheet models will be visited regardless of their
         *      type.
         *  @param {Boolean|Null} [options.visible=null]
         *      If set to a boolean value, only the visible sheets (if set to
         *      true), or only the hidden sheets (if set to false) will be
         *      visited. By default, all sheet models in this document will be
         *      visited, regardless of their visibility.
         *
         * @returns {Object}
         *  An iterator object that implements the standard EcmaScript iterator
         *  protocol, i.e. it provides the method next() that returns a result
         *  object with the following properties:
         *  - {Boolean} done
         *      If set to true, all sheet models have been visited. No more
         *      sheet models are available; this result object does not contain
         *      any other properties!
         *  - {SheetModel} value
         *      The current sheet model instance.
         *  - {Number} sheet
         *      The zero-based index of the sheet.
         *  - {String} name
         *      The name of the sheet.
         *  - {Number} index
         *      The index of the sheet in the array of the sheets visited by
         *      the iterator according to the passed options.
         */
        this.createSheetIterator = function (options) {

            // the index of the first sheet to be visited
            var firstSheet = Utils.getIntegerOption(options, 'first', 0);
            // the index of the last sheet to be visited
            var lastSheet = Utils.getIntegerOption(options, 'last', sheetsCollection.length - 1);
            // whether to iterate in reversed order
            var reverse = Utils.getBooleanOption(options, 'reverse', false);

            // the effective half-open interval indexes
            var begin = reverse ? lastSheet : firstSheet;
            var end = reverse ? (firstSheet - 1) : (lastSheet + 1);
            // create an array iterator for the specified sheet interval (array iterator expects half-open range)
            var arrayIt = IteratorUtils.createArrayIterator(sheetsCollection, { begin: begin, end: end, reverse: reverse });

            // whether to filter for supported sheet types
            var supported = Utils.getBooleanOption(options, 'supported', false);
            // whether to filter for sheets with cells
            var cells = Utils.getBooleanOption(options, 'cells', false);
            // whether to filter for visible/hidden sheets
            var visible = Utils.getBooleanOption(options, 'visible', null);
            // array index of all visited sheets
            var arrayIndex = 0;

            // transform the iterator result (sheet collection entries to sheet models), filter for visible or hidden sheets if specified
            return IteratorUtils.createTransformIterator(arrayIt, function (sheetInfo, iterResult) {
                var sheetModel = sheetInfo.model;
                // skip unsupported sheet types if specified
                if (supported && !sheetModel.isSupportedType()) { return null; }
                // skip sheets without cells if specified
                if (cells && !sheetModel.isCellType()) { return null; }
                // skip visible/hidden sheets if specified
                if ((visible !== null) && (visible !== sheetModel.isVisible())) { return null; }
                // let the iterator visit the current sheet
                iterResult = { value: sheetModel, name: sheetInfo.name, sheet: iterResult.index, index: arrayIndex };
                arrayIndex += 1;
                return iterResult;
            });
        };

        /**
         * Returns the index of the nearest visible sheet next to the specified
         * sheet.
         *
         * @param {Number} sheet
         *  The zero-based index of a sheet.
         *
         * @param {String} [method='nearest']
         *  The method how to find a visible sheet:
         *  - 'next': If the specified sheet is hidden, searches for the
         *      nearest following visible sheet. All sheets preceding the
         *      specified sheet will be ignored.
         *  - 'prev': If the specified sheet is hidden, searches for the
         *      nearest preceding visible sheet. All sheets following the
         *      specified sheet will be ignored.
         *  - 'nearest' (default value): If the specified sheet is hidden,
         *      first searches with the method 'next', and if this fails, with
         *      the method 'prev'.
         *
         * @returns {Number}
         *  The zero-based index of a visible sheet; or -1, if no visible sheet
         *  has been found.
         */
        this.findVisibleSheet = function (sheet, method) {

            var iteratorOptions = (function () {
                switch (method) {
                    case 'next':
                        return { first: sheet, supported: true, visible: true };
                    case 'prev':
                        return { last: sheet, reverse: true, supported: true, visible: true };
                }
                return null;
            }());

            // existing iterator ('next' or 'prev' method): find the nearest visible sheet
            if (iteratorOptions) {
                var iterResult = this.createSheetIterator(iteratorOptions).next();
                return iterResult.done ? -1 : iterResult.sheet;
            }

            // 'nearest' method: first try to find a following sheet, then a preceding
            var visSheet = this.findVisibleSheet(sheet, 'next');
            return (visSheet < 0) ? this.findVisibleSheet(sheet, 'prev') : visSheet;
        };

        /**
         * Returns the zero-based index of the active (visible) sheet in this
         * document (the value of the 'activeSheet' view attribute).
         *
         * @returns {Number}
         *  The zero-based index of the active sheet.
         */
        this.getActiveSheet = function () {
            return this.getViewAttribute('activeSheet');
        };

        /**
         * Changes the active (visible) sheet in this document (the value of
         * the 'activeSheet' view attribute).
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be activated.
         *
         * @returns {SpreadsheetModel}
         *  A reference to this instance.
         */
        this.setActiveSheet = function (sheet) {
            return this.setViewAttribute('activeSheet', sheet);
        };

        /**
         * Creates a localized sheet name that is not yet used in the document.
         *
         * @returns {String}
         *  A sheet name not yet used in this document.
         */
        this.generateUnusedSheetName = function () {

            // the new sheet name
            var sheetName = '';
            // one-based index for the new sheet name
            var index = sheetsCollection.length;

            // generate a valid name
            while ((sheetName.length === 0) || this.hasSheet(sheetName)) {
                sheetName = SheetUtils.getSheetName(index);
                index += 1;
            }

            return sheetName;
        };

        // defined names and table ranges -------------------------------------

        /**
         * Returns whether the specified defined name exists globally, or in
         * any sheet of this document.
         *
         * @param {String} label
         *  The label of a defined name to be checked.
         *
         * @returns {Boolean}
         *  Whether the specified defined name exists in the document.
         */
        this.hasName = function (label) {
            return nameCollection.hasName(label) || sheetsCollection.some(function (sheetDesc) {
                return sheetDesc.model.getNameCollection().hasName(label);
            });
        };

        /**
         * Returns all defined names in this document as plain array.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.skipHidden=false]
         *      If set to true, hidden defined names will not be included into
         *      the result.
         *  @param {Boolean} [options.skipLocal=false]
         *      If set to true, sheet-locally defined names will not be
         *      included into the result.
         *
         * @returns {Array<NameModel>}
         *  All defined names in this document as plain array.
         */
        this.getAllNames = function (options) {

            // collect the global names
            var nameModels = nameCollection.getAllNames(options);

            // add the sheet-local names if specified
            if (!Utils.getBooleanOption(options, 'skipLocal', false)) {
                sheetsCollection.forEach(function (sheetDesc) {
                    nameModels = nameModels.concat(sheetDesc.model.getNameCollection().getAllNames(options));
                });
            }

            return nameModels;
        };

        /**
         * Returns whether the specified table range exists in any sheet of
         * this document.
         *
         * @param {String} tableName
         *  The name of the table to be checked. MUST NOT be the empty string
         *  (addresses the anonymous table range used to store filter settings
         *  for the standard auto filter of each sheet).
         *
         * @returns {Boolean}
         *  Whether the specified table exists in the document.
         */
        this.hasTable = function (tableName) {
            return tableModelSet.hasKey(SheetUtils.getTableKey(tableName));
        };

        /**
         * Returns the model of the table with the specified name.
         *
         * @param {String} tableName
         *  The name of the table. MUST NOT be the empty string (addresses the
         *  anonymous table range used to store filter settings for the
         *  standard auto filter of each sheet).
         *
         * @returns {TableModel|Null}
         *  The model of the table with the specified name; or null, if no
         *  table exists with that name.
         */
        this.getTable = function (tableName) {
            return tableModelSet.get(SheetUtils.getTableKey(tableName), null);
        };

        /**
         * Returns all table models in this document as plain array.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.autoFilter=false]
         *      If set to true, the table models for the auto-filters in all
         *      sheets will be included into the result.
         *
         * @returns {Array<TableModel>}
         *  All table models in this document as plain array.
         */
        this.getAllTables = function (options) {
            return sheetsCollection.reduce(function (tableModels, sheetDesc) {
                return tableModels.concat(sheetDesc.model.getTableCollection().getAllTables(options));
            }, []);
        };

        // range contents -----------------------------------------------------

        /**
         * Returns the contents of all cells in the passed cell ranges.
         *
         * @param {Range3DArray|Range3D} ranges
         *  An array of range addresses, or a single cell range address, with
         *  sheet indexes, whose cell contents will be returned. Each range
         *  MUST refer to a single sheet only (no sheet intervals!), but
         *  different ranges in an array may refer to different sheets.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.blanks=false]
         *      If set to true, all blank cells will be included in the result.
         *      By default, only non-blank cells will be collected.
         *  @param {Boolean} [options.visible=false]
         *      If set to true, only visible cells (cells in visible columns,
         *      AND in visible rows) will be included in the result. Otherwise,
         *      all visible and hidden cells will be returned.
         *  @param {Boolean} [options.attributes=false]
         *      If set to true, the result will contain the identifier of the
         *      cell auto-style, the merged formatting attributes, and the
         *      parsed number format of the cells.
         *  @param {Boolean} [options.display=false]
         *      If set to true, the result will contain the formatted display
         *      strings of the cells in the property 'display'.
         *  @param {Boolean} [options.compressed=false]
         *      If set to true, the result array will be optimized: Consecutive
         *      cells with equal contents and formatting will be represented by
         *      a single array element with an additional property 'count'.
         *      Especially useful, if large ranges located in the unused area
         *      of the sheets are queried.
         *  @param {Number} [options.maxCount]
         *      If specified, the maximum number of cells that will be returned
         *      in the result, regardless how large the passed ranges are.
         *
         * @returns {Array<Object>}
         *  The contents of the cells in the passed ranges. The result will be
         *  an array of cell content objects, with the additional property
         *  'count' representing the total number of cells contained in the
         *  result (this value will be different to the length of the array in
         *  compressed mode). See method CellCollection.getRangeContents() for
         *  details.
         */
        this.getRangeContents = function (ranges, options) {

            // the maximum number of cells in the result
            var maxCount = Utils.getIntegerOption(options, 'maxCount', null, 1);
            // the result array returned by this method
            var contents = [];
            // the number of cells already inserted into the result (differs to array length in compressed mode)
            contents.count = 0;

            // check sheet indexes contained in the ranges
            ranges = Range3DArray.filter(ranges, function (range) {
                if (this.hasSheet(range.sheet1) && range.singleSheet()) { return true; }
                Utils.warn('SpreadsheetModel.getRangeContents(): invalid sheet index in range address');
                return false;
            }, this);

            // create a clone of the options to be able to decrease the maximum count per cell range
            options = _.clone(options);

            // collect cell contents of all ranges from the respective cell collections
            Range3DArray.forEach(ranges, function (range3d) {

                // the cell collection to fetch the cell data from
                var cellCollection = this.getSheetModel(range3d.sheet1).getCellCollection();

                // 3D ranges must be converted to simple 2D ranges without sheet indexes, otherwise cell collection
                // uses the wrong instance methods of class Range3D (no better solution for that in JavaScript...)
                var range = range3d.toRange();

                // fetch and concatenate the new cell data
                var newContents = cellCollection.getRangeContents(range, options);
                contents = contents.concat(newContents);
                contents.count += newContents.count;

                // early exit, if the specified limit has been reached; otherwise decrease the maximum
                // count in the options for the next iteration
                if (maxCount === contents.count) { return Utils.BREAK; }
                if (maxCount) { options.maxCount -= newContents.count; }
            }, this);

            return contents;
        };

        // operation generators -----------------------------------------------

        /**
         * Sends the passed operations to the server without applying them
         * locally. Overwrites the same method of the base class EditModel, and
         * adds more specific behavior (see options).
         *
         * @param {Object|Array|OperationGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations to be sent to the
         *  server. Note that an operations generator passed to this method
         *  will NOT be cleared after its operations have been sent.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options also supported by the
         *  base class method EditModel.sendOperations(), and the following
         *  additional options:
         *  @param {Boolean} [options.defer=false]
         *      If set to true, the operations will not be sent to the server
         *      yet, but will be cached internally, and will be sent by calling
         *      the method SpreadsheetModel.createAndApplySheetOperations() the
         *      next time, before the operation generators will be invoked.
         *
         * @returns {Boolean}
         *  Whether registering all operations for sending was successful.
         */
        this.sendOperations = _.wrap(this.sendOperations, function (baseMethod, operations, options) {

            // cache the operations internally instead of sending them if specified
            if (Utils.getBooleanOption(options, 'defer', false)) {
                sendOperationsCache = sendOperationsCache.concat(SheetOperationGenerator.getArray(operations, options));
                return true;
            }

            // base class method will send the operations regularly
            return baseMethod.call(this, operations, options);
        });

        /**
         * Creates a new operations generator for spreadsheet document
         * operations and undo operations, invokes the callback function,
         * applies all operations contained in the generator, sends them to the
         * server, and creates an undo action with the undo operations that
         * have been generated by the callback function.
         *
         * @param {Function} callback
         *  The callback function. Receives the following parameters:
         *  (1) {SheetOperationGenerator} generator
         *      The operations generator to be filled with the spreadsheet
         *      document operations, and undo operations.
         *  MUST return a promise to defer applying and sending the operations
         *  until the promise has been resolved. Will be called in the context
         *  of this model instance.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  EditModel.createAndApplyOperations(), except for the option
         *  'generator' which will be set to a new instance of the class
         *  SheetOperationGenerator.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operations have been
         *  applied and sent successfully (with the result of the callback
         *  function); or rejected, if the callback has returned a rejected
         *  promise (with the result of that promise), or if applying the
         *  operations has failed (with the object {cause:'operation'}).
         */
        this.createAndApplyOperations = _.wrap(this.createAndApplyOperations, function (baseMethod, callback, options) {

            // if there are cached operation registered by using the option 'defer' of the method SpreadsheetModel.sendOperations(),
            // they need to be sent now, before generating any other operations that will be applied immediately
            if (sendOperationsCache.length > 0) {
                this.sendOperations(sendOperationsCache);
                sendOperationsCache = [];
            }

            // call the base class method to generate and send the operations
            return baseMethod.call(this, callback, options);
        });

        /**
         * Creates a new operation generator for spreadsheet operations and
         * undo operations.
         *
         * @param {Object} [options]
         *  Optional parameters passed to the constructor of the operation
         *  generator.
         *
         * @returns {SheetoperationsGenerator}
         *  The new operations generator for spreadsheet operations.
         */
        this.createSheetOperationGenerator = function (options) {
            return new SheetOperationGenerator(this, options);
        };

        /**
         * Creates a new operation generator for spreadsheet operations and
         * undo operations, invokes the callback function, applies all
         * operations contained in the generator, sends them to the server, and
         * creates an undo action with the undo operations that have been
         * generated by the callback function.
         *
         * @param {Function} callback
         *  The callback function. Receives the following parameters:
         *  (1) {SheetOperationGenerator} generator
         *      The operations generator to be filled with the spreadsheet
         *      document operations, and undo operations.
         *  MUST return a promise to defer applying and sending the operations
         *  until the promise has been resolved. Will be called in the context
         *  of this model instance.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options supported by the method
         *  EditModel.createAndApplyOperations(), except for the option
         *  'generator' which will be set to a new instance of the class
         *  SheetOperationGenerator.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operations have been
         *  applied and sent successfully (with the result of the callback
         *  function); or rejected, if the callback has returned a rejected
         *  promise (with the result of that promise), or if applying the
         *  operations has failed (with the object {cause:'operation'}).
         */
        this.createAndApplySheetOperations = SheetUtils.profileAsyncMethod('SpreadsheetModel.createAndApplySheetOperations()', function (callback, options) {

            // It is required to stop the text framework from fiddling around with the browser selection
            // when immediately applying spreadsheet operations.
            //
            // When applying undo/redo operations, the event handlers for 'undo:after' and 'redo:after'
            // of the text framework want to set the browser selection. This can be suppressed by
            // passing the option preventSelectionChange:true to the generated undo/redo operations.
            //
            // When creating empty paragraphs in a new shape drawing object, text frameworks wants to
            // update the paragraph formatting and set the browser selection afterwards. This can be
            // prevented by using the 'BlockKeyboardEvent' mode of the text framework.

            // the effective options passed to the method of the base class EditModel
            options = _.extend({
                applyImmediately: true,
                undoOptions: { preventSelectionChange: true } // suppress browser selection in text framework
            }, options, {
                generatorClass: SheetOperationGenerator
            });

            // create and apply the operations by invoking the callback function with additional
            // preparation and post-processing specific for spreadsheet documents
            return this.createAndApplyOperations(function (generator) {

                // suppress browser selection in text framework
                this.setBlockKeyboardEvent(true);

                // invoke the generator callback function
                var promise = callback.call(this, generator);

                // final cleanup
                return promise.always(function () {
                    self.setBlockKeyboardEvent(false);
                    generator.logCacheUsage();
                });
            }, options);
        });

        /**
         * Invokes the passed callback function for all specified sheet models,
         * and initializes the sheet index of the passed operations generator
         * before visiting the sheets.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator. Its sheet index will be updated before
         *  invoking the callback function for a sheet model.
         *
         * @param {Array<Number>} sheets
         *  The zero-based indexes of all sheets to be visited.
         *
         * @param {Function} callback
         *  The callback function to be invoked for each sheet model.
         *  Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The model instance of a sheet.
         *  (2) {Number} sheet
         *      The zero-based index of the visited sheet.
         *  Will be called in the context of this document model. May return a
         *  promise to defer visiting the next sheet model.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the sheet models have been
         *  visited, or that will be rejected, if the callback function returns
         *  a rejected promise for at least one sheet model.
         */
        this.generateOperationsForSheets = function (generator, sheets, callback) {

            // the sheet index of the generator, will be restored later
            var oldSheetIndex = generator.getSheetIndex();

            // visit all sheet models in a sliced loop, set the sheet index at the generator
            var promise = this.iterateArraySliced(sheets, function (sheet) {
                generator.setSheetIndex(sheet);
                return callback.call(self, self.getSheetModel(sheet), sheet);
            }, 'SpreadsheetModel.generateOperationsForSheets');

            // restore the old sheet index at the generator afterwards
            return promise.always(function () {
                generator.setSheetIndex(oldSheetIndex);
            });
        };

        /**
         * Invokes the passed callback function for all sheet models in this
         * spreadsheet document, and initializes the sheet index of the passed
         * operations generator before visiting the sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator. Its sheet index will be updated before
         *  invoking the callback function for a sheet model.
         *
         * @param {Function} callback
         *  The callback function to be invoked for each matching sheet model.
         *  Receives the following parameters:
         *  (1) {SheetModel} sheetModel
         *      The model instance of a sheet.
         *  (2) {Number} sheet
         *      The zero-based index of the visited sheet.
         *  Will be called in the context of this document model. May return a
         *  promise to defer visiting the next sheet model.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  method SpreadsheetModel.createSheetIterator().
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all sheet models have been
         *  visited, or that will be rejected, if the callback function returns
         *  a rejected promise for at least one sheet model.
         */
        this.generateOperationsForAllSheets = function (generator, callback, options) {

            // the sheet index of the generator, will be restored later
            var oldSheetIndex = generator.getSheetIndex();
            // the index of a sheet to be skipped completely
            var skipSheet = Utils.getIntegerOption(options, 'skipSheet', null);

            // visit all sheet models in a sliced loop, set the sheet index at the generator
            var promise = this.iterateSliced(this.createSheetIterator(options), function (sheetModel, result) {
                if (result.sheet !== skipSheet) {
                    generator.setSheetIndex(result.sheet);
                    return callback.call(self, sheetModel, result.sheet);
                }
            }, 'SpreadsheetModel.generateOperationsForAllSheets');

            // restore the old sheet index at the generator afterwards
            return promise.always(function () {
                generator.setSheetIndex(oldSheetIndex);
            });
        };

        /**
         * Generates the operations and undo operations to update or restore
         * the formulas of all cells, defined names, data validations,
         * formatting rules, and drawing objects in this sheet.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  method SpreadsheetModel.createSheetIterator().
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateUpdateFormulaOperations = function (generator, changeDesc, options) {

            // update the globally defined names
            var promise = nameCollection.generateUpdateFormulaOperations(generator, changeDesc);

            // update the formula expressions for all sheets in the document
            promise = promise.then(function () {
                return self.generateOperationsForAllSheets(generator, function (sheetModel) {
                    return sheetModel.generateUpdateFormulaOperations(generator, changeDesc);
                }, options);
            });

            return promise;
        };

        /**
         * Generates the operations, and the undo operations, to insert a new
         * sheet into this document model.
         *
         * @param {OperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Number} sheet
         *  The zero-based insertion index of the new sheet.
         *
         * @param {SheetType} sheetType
         *  The type identifier of the new sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. Must not be empty. Must not be equal to
         *  the name of any existing sheet.
         *
         * @param {Object} [attrs]
         *  An attribute set containing initial formatting attributes for the
         *  new sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateInsertSheetOperations = function (generator, sheet, sheetType, sheetName, attrs) {

            // first check the passed sheet name
            return ensureValidSheetName(sheetName).then(function () {

                // the properties for the 'insertSheet' operation (add optional attributes)
                var properties = { sheetName: sheetName };
                if (sheetType !== SheetType.WORKSHEET) { properties.type = sheetType.toJSON(); }
                if (_.isObject(attrs)) { properties.attrs = attrs; }

                // generate the operations
                generator.setSheetIndex(sheet);
                generator.generateSheetOperation(Operations.DELETE_SHEET, null, { undo: true });
                generator.generateSheetOperation(Operations.INSERT_SHEET, properties);
            });
        };

        /**
         * Generates the operations, and the undo operations, to delete an
         * existing sheet from this document model.
         *
         * @param {OperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateDeleteSheetOperations = function (generator, sheet) {

            // the change descriptor to be passed to the generator methods of the collections
            var changeDesc = { type: 'deleteSheet', sheet: sheet };

            // update the formula expressions for all sheets (except the sheet to be deleted)
            var promise = this.generateUpdateFormulaOperations(generator, changeDesc, { skipSheet: sheet });

            // generate the undo operations for the deleted sheet
            promise = promise.then(function () {
                generator.setSheetIndex(sheet);
                return self.getSheetModel(sheet).generateRestoreOperations(generator);
            });

            // finally, generate and apply the operation to delete the sheet
            return promise.done(function () {
                generator.setSheetIndex(sheet);
                generator.generateSheetOperation(Operations.DELETE_SHEET);
            });
        };

        /**
         * Generates the operations, and the undo operations, to move a sheet
         * in this document model to a new position.
         *
         * @param {OperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Number} from
         *  The zero-based index of the sheet to be moved.
         *
         * @param {Number} to
         *  The zero-based index of the new position of the sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateMoveSheetOperations = function (generator, from, to) {

            // generate the operations
            generator.setSheetIndex(to).generateSheetOperation(Operations.MOVE_SHEET, { to: from }, { undo: true, prepend: true });
            generator.setSheetIndex(from).generateSheetOperation(Operations.MOVE_SHEET, { to: to });

            // always successful
            return $.when();
        };

        /**
         * Generates the operations, and the undo operations, to create and
         * insert a complete clone of a sheet in this document model.
         *
         * @param {OperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Number} fromSheet
         *  The zero-based index of the sheet to be copied.
         *
         * @param {Number} toSheet
         *  The zero-based insertion index of the new sheet.
         *
         * @param {String} sheetName
         *  The name of the new sheet. Must not be empty. Must not be equal to
         *  the name of any existing sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateCopySheetOperations = function (generator, fromSheet, toSheet, sheetName) {

            // first check the passed sheet name
            return ensureValidSheetName(sheetName).then(function () {

                // collect the names of all existing tables in a flag set
                var oldTableNameSet = {};
                self.getAllTables().forEach(function (tableModel) {
                    oldTableNameSet[tableModel.getKey()] = true;
                });

                // generate unused names for all new table ranges in the cloned sheet
                var oldSheetModel = self.getSheetModel(fromSheet);
                var newTableNameMap = {};
                oldSheetModel.getTableCollection().getAllTables().forEach(function (tableModel) {

                    // try to split the table name into a base name and a trailing integer (fall-back to translated 'Table')
                    var oldTableName = tableModel.getName();
                    var matches = /^(.*?)(\d+)$/.exec(oldTableName);
                    var baseName = matches ? matches[1] : SheetUtils.getTableName();
                    var nameIndex = matches ? (parseInt(matches[2], 10) + 1) : 1;

                    // generate an unused table name by increasing the trailing index repeatedly
                    while (SheetUtils.getTableKey(baseName + nameIndex) in oldTableNameSet) { nameIndex += 1; }
                    var newTableName = baseName + nameIndex;
                    oldTableNameSet[SheetUtils.getTableKey(newTableName)] = true;
                    newTableNameMap[oldTableName] = newTableName;
                });

                // first, generate and apply the 'copySheet' operation
                var properties = { to: toSheet, sheetName: sheetName };
                if (!_.isEmpty(newTableNameMap)) { properties.tableNames = newTableNameMap; }
                generator.setSheetIndex(fromSheet).generateSheetOperation(Operations.COPY_SHEET, properties);

                // the model of the new sheet that has just been created
                var newSheetModel = self.getSheetModel(toSheet);
                // the change descriptor to be passed to the generator methods of the collections
                var changeDesc = { type: 'renameSheet', sheet: fromSheet, sheetName: sheetName, tableNames: newTableNameMap };

                // update the formula expressions in the new sheet only
                generator.setSheetIndex(toSheet);

                // delete all unsupported drawings in new sheet (workaround for Bug 52800)
                var drawingCollection = oldSheetModel.getDrawingCollection();
                for (var i = drawingCollection.getModelCount() - 1; i >= 0; i--) {
                    var drawModel = drawingCollection.getModel([i]);
                    if (!drawModel.isSupported()) { generator.generateDrawingOperation(Operations.DELETE_DRAWING, [i]); }
                }

                var promise = newSheetModel.generateUpdateFormulaOperations(generator, changeDesc);

                // delete all undo operations generated above while updating the formulas,
                // and create the undo operation that simply deletes the copied sheet
                return promise.done(function () {
                    generator.clearOperations({ undo: true });
                    generator.generateSheetOperation(Operations.DELETE_SHEET, null, { undo: true });
                });
            });
        };

        /**
         * Generates the operations, and the undo operations, to rename a sheet
         * in this document model.
         *
         * @param {OperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {Number} sheet
         *  The zero-based index of the sheet to be renamed.
         *
         * @param {String} sheetName
         *  The new name for the sheet. Must not be empty. Must not be equal to
         *  the name of any existing sheet.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when all operations have been
         *  generated.
         */
        this.generateRenameSheetOperations = function (generator, sheet, sheetName) {

            // do not generate an operation, if passed name matches the old name
            var oldSheetName = this.getSheetName(sheet);
            if (oldSheetName === sheetName) { return $.when(); }

            // first check the passed sheet name
            return ensureValidSheetName(sheetName, sheet).then(function () {

                // use a private generator to collect the operations that need to be created before
                // renaming the sheet, but must be applied afterwards
                var generator2 = new SheetOperationGenerator(self);
                // the change descriptor to be passed to the generator methods of the collections
                var changeDesc = { type: 'renameSheet', sheet: sheet, sheetName: sheetName };

                // update the formula expressions for all sheets in the document
                var promise = self.generateUpdateFormulaOperations(generator2, changeDesc);

                // finally, generate and apply the operations to rename the sheet
                return promise.done(function () {
                    generator.setSheetIndex(sheet);
                    generator.generateSheetOperation(Operations.RENAME_SHEET, { sheetName: oldSheetName }, { undo: true });
                    generator.generateSheetOperation(Operations.RENAME_SHEET, { sheetName: sheetName });
                    generator.appendOperations(generator2, { undo: true });
                    generator.appendOperations(generator2);
                });
            });
        };

        // initialization -----------------------------------------------------

        // create instance members
        numberFormatter = new NumberFormatter(this);
        dependencyManager = new DependencyManager(this);
        formulaParser = new FormulaParser(this);
        formulaCompiler = new FormulaCompiler(this);
        formulaInterpreter = new FormulaInterpreter(this);
        nameCollection = new NameCollection(this);
        listCollection = new ListCollection(this);

        // create style attribute containers
        ModelAttributesMixin.call(this);

        // prepare the DOM container for page nodes (not removing contenteditable from the page (53493))
        pageRootNode = this.getNode()[0];
        pageContentNode = DOMUtils.getPageContentNode(pageRootNode)[0];
        pageContentNode.innerHTML = '';

        // OOXML: register operations for number format codes
        if (fileFormat === 'ooxml') {
            registerCollectionMethod(Operations.INSERT_NUMBER_FORMAT, numberFormatter, 'applyInsertOperation');
            registerCollectionMethod(Operations.DELETE_NUMBER_FORMAT, numberFormatter, 'applyDeleteOperation');
        }

        // register operation handlers for manipulating the sheet collection
        this.registerContextOperationHandler(Operations.INSERT_SHEET, applyInsertSheetOperation);
        registerSheetOperationHandler(Operations.DELETE_SHEET, applyDeleteSheetOperation);
        registerSheetOperationHandler(Operations.RENAME_SHEET, applyRenameSheetOperation);
        registerSheetOperationHandler(Operations.MOVE_SHEET, applyMoveSheetOperation);
        registerSheetOperationHandler(Operations.COPY_SHEET, applyCopySheetOperation);

        // register operation handlers for a sheet in the spreadsheet document
        registerSheetModelMethod(Operations.CHANGE_SHEET, 'applyChangeSheetOperation');

        // register operation handlers for cell collections
        registerSheetCollectionMethod(Operations.CHANGE_CELLS, 'getCellCollection', 'applyChangeCellsOperation');
        registerSheetCollectionMethod(Operations.INSERT_CELLS, 'getCellCollection', 'applyInsertCellsOperation');
        registerSheetCollectionMethod(Operations.DELETE_CELLS, 'getCellCollection', 'applyDeleteCellsOperation');

        // register operation handlers for hyperlink collections
        registerSheetCollectionMethod(Operations.INSERT_HYPERLINK, 'getHyperlinkCollection', 'applyInsertHyperlinkOperation');
        registerSheetCollectionMethod(Operations.DELETE_HYPERLINK, 'getHyperlinkCollection', 'applyDeleteHyperlinkOperation');

        // register operation handlers for merged range collections
        registerSheetCollectionMethod(Operations.MERGE_CELLS, 'getMergeCollection', 'applyMergeCellsOperation');

        // register operation handlers for column collections
        registerSheetCollectionMethod(Operations.INSERT_COLUMNS, 'getColCollection', 'applyInsertOperation');
        registerSheetCollectionMethod(Operations.DELETE_COLUMNS, 'getColCollection', 'applyDeleteOperation');
        registerSheetCollectionMethod(Operations.CHANGE_COLUMNS, 'getColCollection', 'applyChangeOperation');

        // register operation handlers for row collections
        registerSheetCollectionMethod(Operations.INSERT_ROWS, 'getRowCollection', 'applyInsertOperation');
        registerSheetCollectionMethod(Operations.DELETE_ROWS, 'getRowCollection', 'applyDeleteOperation');
        registerSheetCollectionMethod(Operations.CHANGE_ROWS, 'getRowCollection', 'applyChangeOperation');

        // register operation handlers for name collections (allow operations to address global names as well as sheet names)
        registerSheetCollectionMethod(Operations.INSERT_NAME, 'getNameCollection', 'applyInsertOperation', { allowDoc: true });
        registerSheetCollectionMethod(Operations.DELETE_NAME, 'getNameCollection', 'applyDeleteOperation', { allowDoc: true });
        registerSheetCollectionMethod(Operations.CHANGE_NAME, 'getNameCollection', 'applyChangeOperation', { allowDoc: true });

        // register operation handlers for table collections
        registerSheetCollectionMethod(Operations.INSERT_TABLE, 'getTableCollection', 'applyInsertOperation');
        registerSheetCollectionMethod(Operations.DELETE_TABLE, 'getTableCollection', 'applyDeleteOperation');
        registerSheetCollectionMethod(Operations.CHANGE_TABLE, 'getTableCollection', 'applyChangeOperation');
        registerSheetCollectionMethod(Operations.CHANGE_TABLE_COLUMN, 'getTableCollection', 'applyChangeColumnOperation');

        // register operation handlers for validation collections
        registerSheetCollectionMethod(Operations.INSERT_VALIDATION, 'getValidationCollection', 'applyInsertOperation');
        registerSheetCollectionMethod(Operations.DELETE_VALIDATION, 'getValidationCollection', 'applyDeleteOperation');
        registerSheetCollectionMethod(Operations.CHANGE_VALIDATION, 'getValidationCollection', 'applyChangeOperation');

        // register operation handlers for conditional formatting collections
        registerSheetCollectionMethod(Operations.INSERT_CFRULE, 'getCondFormatCollection', 'applyInsertOperation');
        registerSheetCollectionMethod(Operations.DELETE_CFRULE, 'getCondFormatCollection', 'applyDeleteOperation');
        registerSheetCollectionMethod(Operations.CHANGE_CFRULE, 'getCondFormatCollection', 'applyChangeOperation');

        // register operation handlers for drawing collections
        registerDrawingCollectionMethod(Operations.INSERT_DRAWING, 'applyInsertDrawingOperation');
        registerDrawingCollectionMethod(Operations.DELETE_DRAWING, 'applyDeleteDrawingOperation');
        registerDrawingCollectionMethod(Operations.MOVE_DRAWING, 'applyMoveDrawingOperation');

        // register generic operation handlers for drawing models
        registerDrawingModelMethod(Operations.CHANGE_DRAWING, 'applyChangeOperation');

        // register operation handlers for chart models
        registerDrawingModelMethod(Operations.INSERT_CHART_DATASERIES, 'applyInsertSeriesOperation', { type: 'chart' });
        registerDrawingModelMethod(Operations.DELETE_CHART_DATASERIES, 'applyDeleteSeriesOperation', { type: 'chart' });
        registerDrawingModelMethod(Operations.SET_CHART_DATASERIES_ATTRIBUTES, 'applyChangeSeriesOperation', { type: 'chart' });
        registerDrawingModelMethod(Operations.SET_CHART_AXIS_ATTRIBUTES, 'applyChangeAxisOperation', { type: 'chart' });
        registerDrawingModelMethod(Operations.SET_CHART_GRIDLINE_ATTRIBUTES, 'applyChangeGridOperation', { type: 'chart' });
        registerDrawingModelMethod(Operations.SET_CHART_TITLE_ATTRIBUTES, 'applyChangeTitleOperation', { type: 'chart' });
        registerDrawingModelMethod(Operations.SET_CHART_LEGEND_ATTRIBUTES, 'applyChangeLegendOperation', { type: 'chart' });

        // further initialization after import (also if import has failed)
        this.waitForImport(function () {

            // nothing to do without any sheet models
            if (sheetsCollection.length === 0) { return; }

            // initialize sheet view settings from the imported sheet attributes
            sheetsCollection.forEach(function (sheetInfo) {
                sheetInfo.model.initializeViewAttributes();
            });

            // activate the nearest visible sheet
            var activeSheet = Utils.minMax(this.getDocumentAttribute('activeSheet'), 0, sheetsCollection.length - 1);
            var visibleSheet = this.findVisibleSheet(activeSheet);
            if (visibleSheet >= 0) { this.setActiveSheet(visibleSheet); }
        }, this);

        // forward events of the global name collection (insert null as second parameter to notify global names)
        this.listenTo(nameCollection, 'triggered', function (event, type) {
            self.trigger.apply(self, [type, null].concat(_.toArray(arguments).slice(2)));
        });

        // set the column/row count, listen to document attribute changes
        updateMaxColRow(this.getDocumentAttribute('cols'), this.getDocumentAttribute('rows'));
        this.on('change:attributes', function (event, attributes) {
            updateMaxColRow(attributes.cols, attributes.rows);
        });

        // activate another sheet, if the active sheet has been hidden
        this.on('change:sheet:attributes', function (event, sheet) {
            var activeSheet = self.getActiveSheet();
            if ((sheet === activeSheet) && !self.getSheetModel(sheet).isVisible()) {
                var visSheet = self.findVisibleSheet(activeSheet);
                if (visSheet >= 0) { self.setActiveSheet(visSheet); }
            }
        });

        // send changed selection to remote clients after activating another sheet
        this.on('change:viewattributes', function (event, attributes) {
            if ('activeSheet' in attributes) {
                sendActiveSelectionDebounced();
            }
        });

        // send changed selection in the active sheet to remote clients
        this.on('change:sheet:viewattributes', function (event, sheet, attributes) {
            if (('selection' in attributes) && (sheet === self.getActiveSheet())) {
                sendActiveSelectionDebounced();
            }
        });

        // Notify listeners about changed text contents in drawing objects. For performance,
        // this document model will evaluate the operations array once, and will trigger the
        // events at the correct drawing collections containing the changed drawing objects.
        this.on('operations:success', function (event, operations) {
            var drawingModels = new ValueSet('getUid()');
            operations.forEach(function (operation) {
                var start = operation.start;
                if ((operation.name in DRAWING_TEXT_OPERATIONS) && _.isArray(start) && (start.length > 2)) {
                    var modelDesc = self.getSheetModel(start[0]).getDrawingCollection().getModelDescriptor(start.slice(1));
                    if (modelDesc && (modelDesc.remaining.length > 0)) {
                        drawingModels.insert(modelDesc.model);
                    }
                }
            });
            drawingModels.forEach(function (drawingModel) {
                drawingModel.getParentCollection().trigger('change:drawing:text', drawingModel);
            });
        });

        // registering the mousedown handler on the page (required for textshapes)
        this.listenTo(self.getNode(), 'mousedown touchstart', function () { self.setActiveMouseDownEvent(true); });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            // bug 52790: explicitly disconnect the dependency manager before starting to destroy the document (performance/safety)
            dependencyManager.disconnect();
            sheetsCollection.forEach(function (sheetInfo) { sheetInfo.model.destroy(); });
            nameCollection.destroy();
            listCollection.destroy();
            formulaInterpreter.destroy();
            formulaCompiler.destroy();
            formulaParser.destroy();
            dependencyManager.destroy();
            numberFormatter.destroy();
            tableModelSet.clear();
            formulaGrammarMap.clear();
            app = self = sheetsCollection = sheetIndexMap = tableModelSet = formulaGrammarMap = null;
            formulaParser = formulaCompiler = formulaInterpreter = dependencyManager = null;
            numberFormatter = nameCollection = listCollection = null;
            pageRootNode = pageContentNode = null;
        });

    } }); // class SpreadsheetModel

    // exports ================================================================

    return SpreadsheetModel;

});
