/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/interpret/operand', [
    'io.ox/office/tk/utils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, SheetUtils, FormulaUtils) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var Range3D = SheetUtils.Range3D;
    var Range3DArray = SheetUtils.Range3DArray;
    var Scalar = FormulaUtils.Scalar;
    var Matrix = FormulaUtils.Matrix;
    var Dimension = FormulaUtils.Dimension;

    // class Operand ==========================================================

    /**
     * Represents a single operand stored in the operand stack of a formula
     * token interpreter.
     *
     * @constructor
     *
     * @param {FormulaContext} context
     *  The context object providing a connection to the document model.
     *
     * @param {Any} value
     *  The value to be stored as operand. Can be a scalar value (numbers,
     *  strings, boolean values, null), a Date object (will be converted to a
     *  floating-point number immediately), an error code (instance of the
     *  class ErrorCode), a matrix (instance of the class Matrix), a single
     *  cell range address (instance of the class Range3D), or an array of cell
     *  range addresses (instance of the class Range3DArray), or another
     *  instance of this class for copy construction.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {ParsedFormat} [options.format]
     *      A parsed number format associated with the value contained in this
     *      operand. If omitted, this operand does not have a special number
     *      format associated to it (the original number format will not
     *      change).
     *  @param {String} [options.url]
     *      The URL of a hyperlink that will be part of the operand value. The
     *      URL will be carried through the formula result, and can be used by
     *      the user interface to provide a clickable link.
     */
    function Operand(context, value, options) {

        this._context = context;
        this._type = null;
        this._value = null;
        this._format = Utils.getObjectOption(options, 'format', null);
        this._url = Utils.getStringOption(options, 'url', null);

        // initialization -----------------------------------------------------

        // copy construction (options for format and URL passed to constructor win over the operand's settings)
        if (value instanceof Operand) {
            this._type = value._type;
            this._value = value._value;
            if (this._format === null) { this._format = value._format; }
            if (this._url === null) { this._url = value._url; }
            return;
        }

        // cell range addresses passed: validate size
        if ((value instanceof Range3D) || (value instanceof Range3DArray)) {
            value = Range3DArray.get(value);
            if (value.empty()) {
                value = ErrorCode.NULL;
                // continue below, operand becomes an error code (a scalar value)
            } else if (value.length > FormulaUtils.MAX_REF_LIST_SIZE) {
                value = FormulaUtils.UNSUPPORTED_ERROR;
                // continue below, operand becomes an error code (a scalar value)
            } else {
                //ReferenceOperandMixin.call(this, context, value);
                this._type = 'ref';
                this._value = value;
                return;
            }
        }

        // matrix passed: validate matrix size, and all values in the matrix
        if (value instanceof Matrix) {
            FormulaUtils.ensureMatrixDim(value.dim());
            value.transform(context.validateValue, context);
            this._type = 'mat';
            this._value = value;
            return;
        }

        // otherwise: scalar value (as last check, preceding code may have changed the value)
        this._type = 'val';
        this._value = context.validateValue(value);

    } // class Operand

    // public methods ---------------------------------------------------------

    /**
     * Returns the effective data type of this operand.
     *
     * @returns {String}
     *  The effective data type of this operand. Will be one of the following
     *  strings:
     *  - 'val': This operand contains a scalar value (number, string, etc.).
     *  - 'mat': This operand contains a matrix.
     *  - 'ref': This operand contains an array of cell range addresses.
     */
    Operand.prototype.getType = function () {
        return this._type;
    };

    /**
     * Returns whether this operand contains a scalar value (type 'val').
     *
     * @returns {Boolean}
     *  Whether this operand contains a scalar value (type 'val').
     */
    Operand.prototype.isScalar = function () {
        return this._type === 'val';
    };

    /**
     * Returns whether this operand contains a matrix (type 'mat').
     *
     * @returns {Boolean}
     *  Whether this operand contains a matrix (type 'mat').
     */
    Operand.prototype.isMatrix = function () {
        return this._type === 'mat';
    };

    /**
     * Returns whether this operand contains a cell reference (type 'ref').
     *
     * @returns {Boolean}
     *  Whether this operand contains a cell reference (type 'ref').
     */
    Operand.prototype.isReference = function () {
        return this._type === 'ref';
    };

    /**
     * Returns whether this operand represents the empty value (e.g. the second
     * operand in the formula =SUM(1,,2) is empty, and the value resolved from
     * a blank cell is empty).
     *
     * @returns {Boolean}
     *  Whether this operand represents the empty value.
     */
    Operand.prototype.isEmpty = function () {
        return this._value === null;
    };

    /**
     * Returns the dimension (width and height) of this operand.
     *
     * @param {Object} [options]
     *  Optional parameters. Supports all options that are supported by the
     *  method FormulaContext.convertToRange() used to resolve a reference-type
     *  operand to a single cell range address.
     *
     * @returns {Dimension}
     *  The dimension of this operand, in the properties 'rows' and 'cols'. The
     *  dimension of a scalar value is 1x1.
     *
     * @throws {ErrorCode}
     *  The #REF! error code (or another error code according to the passed
     *  options), if this is a reference operand that cannot be resolved to a
     *  single cell range address.
     */
    Operand.prototype.getDimension = function (options) {

        // return dimensions of scalar and matrix
        switch (this._type) {
            case 'val': return new Dimension(1, 1);
            case 'mat': return this._value.dim();
            // otherwise, this operand is a reference
        }

        // extract single cell range address (throws on error)
        var range = this._context.convertToRange(this._value, options);
        return Dimension.createFromRange(range);
    };

    /**
     * Returns the unconverted value of this operand.
     *
     * @returns {Any}
     *  The unconverted value of this operand.
     *
     * @throws {ErrorCode}
     *  All internal error codes carried by this instance.
     */
    Operand.prototype.getRawValue = function () {

        // immediately throw internal error codes
        if (FormulaUtils.isInternalError(this._value)) { throw this._value; }

        return this._value;
    };

    /**
     * Converts this operand to a scalar value.
     *
     * @param {Number|Null} matRow
     *  The relative row index, used to resolve a scalar value in a matrix
     *  context. Specifies the row of a matrix, or the relative row index in a
     *  reference operand. If set to null, the scalar will be resolved in value
     *  context (top-left cell of a matrix, cell projection from a reference
     *  operand).
     *
     * @param {Number|Null} matCol
     *  The relative column index, used to resolve a scalar value in a matrix
     *  context. Specifies the column of a matrix, or the relative column index
     *  in a reference operand. If set to null, the scalar will be resolved in
     *  value context (top-left cell of a matrix, cell projection from a
     *  reference operand).
     *
     * @returns {Any}
     *  The scalar value represented by this operand. Returns an existing
     *  scalar value, or the specified element of a matrix, or the contents of
     *  the specified cell from a cell reference (only if it does not cover the
     *  reference cell of the formula context passed to the constructor).
     *
     * @throws {ErrorCode}
     *  - All internal error codes carried by this instance.
     *  - CIRCULAR_ERROR, if the operand results in a circular reference.
     *  - The #VALUE! error code, if the reference contains more than one
     *      cell range, or if the cell range cannot be resolved to a single
     *      value according to the reference cell.
     */
    Operand.prototype.getScalar = function (matRow, matCol) {

        // immediately throw internal error codes
        if (FormulaUtils.isInternalError(this._value)) { throw this._value; }

        // return scalar value, and an element of a matrix
        switch (this._type) {
            case 'val': return this._value;
            case 'mat': return this._value.get(matRow || 0, matCol || 0);
            // otherwise, this operand is a reference
        }

        // extract single cell range address (throws on error)
        var range = this._context.convertToValueRange(this._value);

        // pick value from a single cell
        if (range.single()) {
            return this._context.getCellValue(range.sheet1, range.start);
        }

        // in matrix context, return the cell relative to the position of the matrix element
        if ((matRow !== null) && (matCol !== null)) {
            var address = Matrix.getAddress(range, matRow, matCol);
            return this._context.getCellValue(range.sheet1, address);
        }

        // the target reference cell of the formula context
        var refAddress = this._context.getTargetAddress();

        // pick matching cell from a column range (row interval must cover the reference cell)
        if (range.singleCol()) {
            if (!range.containsRow(refAddress[1])) { throw ErrorCode.VALUE; }
            return this._context.getCellValue(range.sheet1, new Address(range.start[0], refAddress[1]));
        }

        // pick matching cell from a row range (column interval must cover the reference cell)
        if (range.singleRow()) {
            if (!range.containsCol(refAddress[0])) { throw ErrorCode.VALUE; }
            return this._context.getCellValue(range.sheet1, new Address(refAddress[0], range.start[1]));
        }

        // range is not left of, right of, above, or below the reference cell
        throw ErrorCode.VALUE;
    };

    /**
     * Converts this operand to a matrix.
     *
     * @returns {Matrix}
     *  The matrix represented by this operand. Returns an existing matrix, or
     *  a scalar value packed into a 1x1 matrix, or the contents of the
     *  referenced cells packed into a matrix. The reference must consist of a
     *  single cell range address that does not cover the reference cell, and
     *  the range must not exceed specific limits in order to prevent
     *  performance problems.
     *
     * @throws {ErrorCode}
     *  - All internal error codes carried by this instance.
     *  - UNSUPPORTED_ERROR, if the resulting matrix would be too large.
     *  - CIRCULAR_ERROR, if the operand results in a circular reference.
     *  - The #VALUE! error code, if the operand contains a reference with more
     *      than one cell range address.
     */
    Operand.prototype.getMatrix = function () {

        // immediately throw internal error codes
        if (FormulaUtils.isInternalError(this._value)) { throw this._value; }

        // pack scalar value into a 1x1 matrix, or return an existing matrix
        switch (this._type) {
            case 'val': return new Matrix([[this._value]]);
            case 'mat': return this._value;
            // otherwise, this operand is a reference
        }

        // extract single cell range address (throws on error), and return the matrix
        var range3d = this._context.convertToValueRange(this._value);
        return this._context.getCellMatrix(range3d.sheet1, range3d.toRange());
    };

    /**
     * Converts this operand to an unresolved array of cell range addresses.
     *
     * @returns {Range3DArray}
     *  The unresolved cell range addresses contained in this operand.
     *
     * @throws {ErrorCode}
     *  All literal error codes carried by this operand (regular error codes,
     *  and internal error codes); or the #VALUE! error, if this operand is not
     *  reference type.
     */
    Operand.prototype.getRanges = function () {

        // throw #VALUE! error for scalar values and matrixes (but throw original error codes)
        switch (this._type) {
            case 'val':
                throw (this._value instanceof ErrorCode) ? this._value : ErrorCode.VALUE;
            case 'mat':
                throw ErrorCode.VALUE;
            // otherwise, this operand is a reference
        }

        // return the value (always an instance of Range3DArray)
        return this._value;
    };

    /**
     * Returns the parsed number format associated to this operand, as passed
     * to its constructor.
     *
     * @returns {ParsedFormat|Null}
     *  The parsed number format associated to this operand; or null, if no
     *  number format is available.
     */
    Operand.prototype.getFormat = function () {
        return this._format;
    };

    /**
     * Returns the URL of a hyperlink associated to this operand, as passed to
     * its constructor.
     *
     * @returns {String|Null}
     *  The URL of a hyperlink associated to this operand; or null, if no URL
     *  is available.
     */
    Operand.prototype.getURL = function () {
        return this._url;
    };

    /**
     * Returns the string representation of this operand for debug logging.
     */
    Operand.prototype.toString = function () {
        var str = (this._type === 'val') ? Scalar.toString(this._value) : this._value.toString();
        if (this._format) { str += ' [' + this._format.category + ']'; }
        return str;
    };

    // exports ================================================================

    return Operand;

});
