/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/referencefuncs', [
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (IteratorUtils, SheetUtils, FormulaUtils, TokenArray) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions dealing with cell range
     * references used as values (cell address arithmetics), and functions that
     * look up specific values in a range of cells.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     * In spreadsheet documents, cell range references can be given literally,
     * or can be the result of functions or operators. The client-side formula
     * engine provides the function signature type 'ref', and the class
     * Reference whose instances represent a single cell range address, or a
     * list of several cell range addresses (e.g. as result of the reference
     * list operator).
     *
     *************************************************************************/

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;
    var Range3D = SheetUtils.Range3D;
    var Range3DArray = SheetUtils.Range3DArray;
    var Matrix = FormulaUtils.Matrix;
    var CellRef = FormulaUtils.CellRef;
    var SheetRef = FormulaUtils.SheetRef;

    // private global variables ===============================================

    var SEARCHOPERANDOPTIONS = {
        emptyParam: true,
        emptyCells: false,
        acceptErrors: true
    };

    // private global functions ===============================================

    function splitOperand(context, operand, subsets) {
        var value = operand.getRawValue();
        if (operand.isReference()) {
            value = context.convertToValueRange(value);
        }
        var res = [];
        _.each(subsets, function (subset) {
            var nv = null;
            if (operand.isMatrix()) {
                nv = Matrix.generate(subset.rows, subset.cols, function (row, col) {
                    return value.get(subset.row + row, subset.col + col);
                });
            } else if (operand.isReference()) {
                nv = Range3D.create(value.start[0] + subset.col, value.start[1] + subset.row, value.start[0] + subset.col + subset.cols - 1, value.start[1] + subset.row + subset.rows - 1, value.sheet1, value.sheet2);
            } else {
                nv = value;
            }
            res.push(context.createOperand(nv));
        });

        return res;
    }

    function getTypeId(value) {
        return _.isNumber(value) ? 1 : _.isString(value) ? 2 : _.isBoolean(value) ? 3 : (value instanceof ErrorCode) ? 4 : 0;
    }

    function searchInOperand(context, lookup, inexact, searchOperand, resultOperand) {

        if (resultOperand.isScalar()) { throw ErrorCode.NA; }

        var index = searchIndexInOperand(context, lookup, inexact, searchOperand);
        var res = null;
        var iterator = context.createScalarIterator(resultOperand, SEARCHOPERANDOPTIONS);

        IteratorUtils.some(iterator, function (value, result) {
            if (result.offset === index) {
                res = value;
                return true;
            }
        });
        return res;

    }

    function searchIndexInOperand(context, lookup, inexact, operand) {

        if (_.isUndefined(inexact)) { inexact = true; }
        if (_.isUndefined(lookup) || _.isNull(lookup)) { throw ErrorCode.VALUE; }

        var regex = null;
        if (!inexact && _.isString(lookup)) { regex = context.convertPatternToRegExp(lookup, true); }

        var lookupType = getTypeId(lookup);
        var last = null;
        var lastIndex = null;
        var resultIndex = Number.NaN;
        var iterator = context.createScalarIterator(operand, SEARCHOPERANDOPTIONS);

        IteratorUtils.some(iterator, function (value, result) {
            if (value === null) { return; }

            var index = result.offset;

            last = value;
            lastIndex = index;

            var currentType = getTypeId(value);
            if (currentType !== lookupType) { return; }

            var compare = context.compareScalars(value, lookup);
            if (compare === 0) {
                resultIndex = index;
                return true;
            }
            if (inexact && compare > 0) {
                if (index < 1) { throw ErrorCode.NA; }
                resultIndex = index - 1;
                return true;
            }
            if (regex && regex.test(value)) {
                resultIndex = index;
                return true;
            }
        });

        if (!_.isNaN(resultIndex)) { return resultIndex; }

        if (!_.isNaN(lastIndex) && inexact && getTypeId(last) === lookupType && context.compareScalars(last, lookup) < 0) {
            return lastIndex;
        }

        throw ErrorCode.NA;
    }

    // exports ================================================================

    return {

        ADDRESS: {
            category: 'reference',
            minParams: 2,
            maxParams: 5,
            type: 'val:str',
            signature: 'val:int val:int val:int val:bool val:str',
            resolve: function (row, col, absMode, a1Style, sheetName) {

                // the resulting address string
                var result = '';

                // default for missing or empty absolute mode (always defaults to 1)
                if (this.isMissingOrEmptyOperand(2)) {
                    absMode = 1;
                } else if ((absMode < 1) || (absMode > 4)) {
                    throw ErrorCode.VALUE;
                }

                // default for missing or empty A1 style (default value is TRUE)
                if (this.isMissingOrEmptyOperand(3)) {
                    a1Style = true;
                }

                // initialize the cell reference structure
                var cellRef = new CellRef(0, 0, (absMode % 2) > 0, absMode <= 2);
                var refAddress = new Address(0, 0);

                // get the effective column index (convert column offset in R1C1 notation to real column index)
                if (a1Style || cellRef.absCol) {
                    // absolute column index, or any column index in A1 notation (one-based)
                    cellRef.col = col - 1;
                } else if (col < 0) {
                    // negative column offset in R1C1 notation, e.g. R1C[-16383]
                    // (adjust reference address to be able to create large negative offsets)
                    var maxCol = this.docModel.getMaxCol();
                    cellRef.col = maxCol + col;
                    refAddress[0] = maxCol;
                } else {
                    // positive column offset in R1C1 notation, e.g. R1C[16383]
                    cellRef.col = col;
                    refAddress[0] = 0;
                }

                // get the effective row index (convert row offset in R1C1 notation to real row index)
                if (a1Style || cellRef.absRow) {
                    // absolute row index, or any row index in A1 notation (one-based)
                    cellRef.row = row - 1;
                } else if (row < 0) {
                    // negative row offset in R1C1 notation, e.g. R[-1048575]C1
                    // (adjust reference address to be able to create large negative offsets)
                    var maxRow = this.docModel.getMaxRow();
                    cellRef.row = maxRow + row;
                    refAddress[1] = maxRow;
                } else {
                    // positive row offset in R1C1 notation, e.g. R[1048575]C1
                    cellRef.row = row;
                    refAddress[1] = 0;
                }

                // validate the effective cell reference structure
                this.checkCellRef(cellRef);

                // start with encoded sheet name (enclosed in apostrophes if needed)
                // (not for missing or empty parameter, but for explicit empty string)
                if (_.isString(sheetName)) {

                    // enclose complex names in apostrophes
                    if ((sheetName.length > 0) && !this.formulaGrammar.isSimpleSheetName(this.docModel, sheetName, { range: true, external: true })) {
                        sheetName = SheetRef.encodeComplexSheetName(sheetName);
                    }

                    // use sheet name with an exclamation mark as separator
                    result = sheetName + '!';
                }

                // append the range address in A1 or R1C1 style
                var formulaGrammar = this.docModel.getFormulaGrammar('ui!' + (a1Style ? 'a1' : 'rc'));
                return result + formulaGrammar.formatReference(this.docModel, refAddress, null, null, cellRef, null);
            }
        },

        AREAS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'ref|deps:skip',
            resolve: function (ranges) { return ranges.length; }
        },

        CHOOSE: {
            category: 'reference',
            minParams: 2,
            type: 'any',
            signature: 'val:int any:lazy|deps:pass',
            resolve: function (index) {
                // CHOOSE uses one-based index
                if ((index < 1) || (index >= this.getOperandCount())) { throw ErrorCode.VALUE; }
                return this.getOperand(index);
            }
        },

        COLUMN: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val:num',
            signature: 'ref:single|deps:skip',
            resolve: function (range) {
                // create a matrix with all column indexes in matrix context with explicit range
                if (range && (this.getContextType() === 'mat')) {
                    FormulaUtils.ensureMatrixDim(1, range.cols());
                    return Matrix.generate(1, range.cols(), function (row, col) {
                        return range.start[0] + col + 1;
                    });
                }
                // use target reference address, if parameter is missing
                var address = range ? this.getMatrixAddress(range) : this.getTargetAddress();
                // function returns one-based column index
                return address[0] + 1;
            },
            // result depends on reference address (behaves like a relative reference), if parameter is missing
            relColRef: function (count) { return count === 0; }
        },

        COLUMNS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'any|deps:skip',
            resolve: function (operand) {
                var value = operand.getRawValue();
                switch (operand.getType()) {
                    case 'val':
                        return (value instanceof ErrorCode) ? value : 1;
                    case 'mat':
                        return value.cols();
                    case 'ref':
                        return this.convertToRange(value).cols();
                    default:
                        FormulaUtils.throwInternal('ReferenceFuncs.COLUMNS.resolve(): unknown operand type');
                }
            }
        },

        FORMULATEXT: {
            category: 'reference',
            name: { ooxml: '_xlfn.FORMULATEXT', odf: 'FORMULA' },
            minParams: 1,
            maxParams: 1,
            type: 'val:str',
            recalc: 'always', // changing a formula may not change the result, thus no recalculation would be triggered
            signature: 'ref:single|deps:skip',
            resolve: function (range) {

                // always use the first cell in the range (TODO: matrix context)
                var address = range.start;

                // TODO: resolve self-reference while entering a cell or shared formula

                // getCellFormula() returns null for blank cells, and simple value cells
                var result = this.getCellFormula(range.sheet1, address);
                if (!result) { throw ErrorCode.NA; }

                // special handling for matrix formulas
                var formula = '=' + result.formula;
                return result.matrix ? ('{' + formula + '}') : formula;
            }
        },

        GETPIVOTDATA: {
            category: 'reference',
            hidden: true,
            minParams: 2,
            repeatParams: 2,
            type: 'val:any'
        },

        HLOOKUP: {
            category: 'reference',
            minParams: 3,
            maxParams: 4,
            type: 'any',
            signature: 'val any val:int val:bool',
            resolve: function (lookup, searchOperand, rowIndexParam, inexact) {
                var rowIndex = rowIndexParam - 1;
                var dim = searchOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references

                if (rowIndex < 0) { throw ErrorCode.VALUE; }
                if (rowIndex >= dim.rows) { throw ErrorCode.REF; }

                var ops = splitOperand(this, searchOperand, [{ col: 0, cols: dim.cols, row: 0, rows: 1 }, { col: 0, cols: dim.cols, row: rowIndex, rows: 1 }]);

                return searchInOperand(this, lookup, inexact, ops[0], ops[1]);
            }
        },

        INDEX: {
            category: 'reference',
            minParams: 2,
            maxParams: 4,
            type: 'any',
            recalc: 'always',
            signature: 'any|deps:skip val:int|mat:pass val:int|mat:pass val:int|mat:pass',
            resolve: function (source, row, col, area) {
                if (source.isMatrix() && !_.isUndefined(area)) { throw ErrorCode.REF; }

                if (_.isUndefined(row)) { row = 1; }
                if (_.isUndefined(area)) { area = 1; }
                if (_.isUndefined(col)) { col = 1; }

                row--; col--; area--;

                if (source.isScalar()) { throw ErrorCode.VALUE; }

                var value = source.getRawValue();
                if (value instanceof Range3DArray) {
                    if (area < 0 || area >= value.length)  { throw ErrorCode.REF; }
                    value = value[area];
                    source = this.createOperand(value);
                }
                var dim = source.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references

                var cols = 1;
                if (col === -1) {
                    cols = dim.cols;
                    col = 0;
                } else {
                    if (col < 0 || col >= dim.cols)  { throw ErrorCode.REF; }
                }

                var rows = 1;
                if (row === -1) {
                    rows = dim.rows;
                    row = 0;
                } else {
                    if (row < 0 || row >= dim.rows)  { throw ErrorCode.REF; }
                }

                var result = splitOperand(this, source, [{ col: col, cols: cols, row: row, rows: rows }]);
                result = result[0].getRawValue();
                if (result instanceof Range3DArray && result.length === 1) {
                    result = result[0];
                }
                return result;
            }
        },

        INDIRECT: {
            category: 'reference',
            minParams: 1,
            maxParams: 2,
            type: 'ref',
            recalc: 'always',
            signature: 'val:str val:bool',
            resolve: function (refText, a1Style) {

                // default for missing or empty A1 style (default value is TRUE)
                if (this.isMissingOrEmptyOperand(1)) {
                    a1Style = true;
                }

                var sheetModel = this.getRefSheetModel();
                var tokenArray = new TokenArray(sheetModel, 'custom');
                var grammarId = 'ui!' + (a1Style ? 'a1' : 'rc');
                var refAddress = this.getTargetAddress();

                tokenArray.parseFormula(grammarId, refText, { refAddress: refAddress });

                var ranges = tokenArray.resolveRangeList({ resolveNames: 'simple', refAddress: refAddress, targetAddress: refAddress });
                if (ranges.length !== 1) { throw ErrorCode.REF; }

                return ranges.first();
            }
        },

        LOOKUP: {
            category: 'reference',
            minParams: 2,
            maxParams: 3,
            type: 'any',
            signature: 'val any any',
            resolve: function (lookup, searchOperand, resultOperand) {

                if (this.isMissingOrEmptyOperand(2)) {
                    var dim = searchOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references
                    var ops = [];
                    if (dim.rows < dim.cols) {
                        ops = splitOperand(this, searchOperand, [{ col: 0, cols: dim.cols, row: 0, rows: 1 }, { col: 0, cols: dim.cols, row: dim.rows - 1, rows: 1 }]);
                    } else {
                        ops = splitOperand(this, searchOperand, [{ col: 0, cols: 1, row: 0, rows: dim.rows }, { col: dim.cols - 1, cols: 1, row: 0, rows: dim.rows }]);
                    }

                    return searchInOperand(this, lookup, undefined, ops[0], ops[1]);
                } else {
                    if (resultOperand.isReference()) {
                        var sDim = searchOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references
                        var rDim = resultOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references
                        var resultRange = null;

                        if (rDim.cols < sDim.cols) {
                            resultRange = this.convertToValueRange(resultOperand.getRawValue());
                            resultRange.end[0] = resultRange.start[0] + sDim.cols;
                            resultOperand = this.createOperand(resultRange);
                        } else if (rDim.rows < sDim.rows) {
                            resultRange = this.convertToValueRange(resultOperand.getRawValue());
                            resultRange.end[1] = resultRange.start[1] + sDim.rows;
                            resultOperand = this.createOperand(resultRange);
                        }
                    }
                    return searchInOperand(this, lookup, undefined, searchOperand, resultOperand);
                }
            }
        },

        MATCH: {
            category: 'reference',
            minParams: 2,
            maxParams: 3,
            type: 'val:num',
            signature: 'val any|mat:pass val:int',
            resolve: function (lookupValue, searchOperand, matchType) {
                // https://github.com/sutoiku/formula.js/blob/b1174eef1c977e464f2ef15220049a2fdbcb2184/lib/lookup-reference.js

                var dim = searchOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references

                if (Math.min(dim.rows, dim.cols) !== 1) { throw ErrorCode.NA; }

                if (_.isUndefined(matchType)) { matchType = 1; }

                if (matchType !== -1 && matchType !== 0 && matchType !== 1) { throw ErrorCode.NA; }

                var iterator = this.createScalarIterator(searchOperand, SEARCHOPERANDOPTIONS);
                var index;
                var indexValue;

                IteratorUtils.some(iterator, function (value, result) {
                    var idx = result.offset;

                    if (matchType === 1) {
                        if (value === lookupValue) {
                            index = idx + 1;
                            return true;
                        } else if (value < lookupValue) {
                            if (!indexValue) {
                                index = idx + 1;
                                indexValue = value;
                            } else if (value > indexValue) {
                                index = idx + 1;
                                indexValue = value;
                            }
                        }
                    } else if (matchType === 0) {
                        if (getTypeId(lookupValue) === 2) {
                            lookupValue = lookupValue.replace(/\?/g, '.');
                            if (value.toLowerCase().match(lookupValue.toLowerCase())) {
                                index = idx + 1;
                                return true;
                            }
                        } else {
                            if (value === lookupValue) {
                                index = idx + 1;
                                return true;
                            }
                        }
                    } else if (matchType === -1) {
                        if (value === lookupValue) {
                            index = idx + 1;
                            return true;
                        } else if (value > lookupValue) {
                            if (!indexValue) {
                                index = idx + 1;
                                indexValue = value;
                            } else if (value < indexValue) {
                                index = idx + 1;
                                indexValue = value;
                            }
                        }
                    }
                });

                if (index) {
                    return index;
                } else {
                    throw ErrorCode.NA;
                }
            }
        },

        OFFSET: {
            category: 'reference',
            minParams: 3,
            maxParams: 5,
            type: 'ref',
            recalc: 'always',
            signature: 'ref|deps:skip val:int val:int val:int val:int',
            resolve: function (ranges, rows, cols, height, width) {

                // reference must contain a single range address
                var range = this.convertToRange(ranges, { errorCode: ErrorCode.VALUE });

                // apply offset
                range.start[0] += cols;
                range.start[1] += rows;
                range.end[0] += cols;
                range.end[1] += rows;

                // modify size (but not if 'height' or 'width' parameters exist but are empty)
                if (!this.isMissingOrEmptyOperand(4)) { range.end[0] = range.start[0] + width - 1; }
                if (!this.isMissingOrEmptyOperand(3)) { range.end[1] = range.start[1] + height - 1; }

                // check that the range does not left the sheet boundaries
                if (this.docModel.isValidRange(range)) { return range; }
                throw ErrorCode.REF;
            }
        },

        ROW: {
            category: 'reference',
            minParams: 0,
            maxParams: 1,
            type: 'val:num',
            signature: 'ref:single|deps:skip',
            resolve: function (range) {
                // create a matrix with all row indexes in matrix context with explicit range
                if (range && (this.getContextType() === 'mat')) {
                    FormulaUtils.ensureMatrixDim(range.rows(), 1);
                    return Matrix.generate(range.rows(), 1, function (row) {
                        return range.start[1] + row + 1;
                    });
                }
                // use target reference address, if parameter is missing
                var address = range ? this.getMatrixAddress(range) : this.getTargetAddress();
                // function returns one-based row index
                return address[1] + 1;
            },
            // result depends on reference address (behaves like a relative reference), if parameter is missing
            relRowRef: function (count) { return count === 0; }
        },

        ROWS: {
            category: 'reference',
            minParams: 1,
            maxParams: 1,
            type: 'val:num',
            signature: 'any|deps:skip',
            resolve: function (operand) {
                var value = operand.getRawValue();
                switch (operand.getType()) {
                    case 'val':
                        return (value instanceof ErrorCode) ? value : 1;
                    case 'mat':
                        return value.rows();
                    case 'ref':
                        return this.convertToRange(value).rows();
                    default:
                        FormulaUtils.throwInternal('ReferenceFuncs.ROWS.resolve(): unknown operand type');
                }
            }
        },

        SHEET: {
            category: 'reference',
            name: { ooxml: '_xlfn.SHEET' },
            minParams: 0,
            maxParams: 1,
            type: 'val:num',
            recalc: 'always', // function result depends on index of a sheet (refresh on moving sheet)
            signature: 'any|deps:skip', // string (sheet name) or reference
            resolve: function (operand) {

                // use reference sheet, if parameter is missing
                if (!operand) { return this.getRefSheet() + 1; }

                var value = operand.getRawValue();

                // use sheet index of a reference
                if (operand.isReference()) {
                    return this.convertToRange(value).sheet1 + 1;
                }

                // convert values to strings (sheet name may look like a number or boolean)
                if (operand.isScalar()) {
                    var sheet = this.docModel.getSheetIndex(this.convertToString(value));
                    if (sheet < 0) { throw ErrorCode.NA; }
                    return sheet + 1;
                }

                // always fail for matrix operands, regardless of context
                throw ErrorCode.NA;
            }
        },

        SHEETS: {
            category: 'reference',
            name: { ooxml: '_xlfn.SHEETS' },
            minParams: 0,
            maxParams: 1,
            type: 'val:num',
            recalc: 'always', // function result depends on indexes of sheets (refresh on moving sheet)
            signature: 'ref:multi|deps:skip',
            resolve: function (range) {
                // return total number of sheets in document, if parameter is missing
                return range ? range.sheets() : this.docModel.getSheetCount();
            }
        },

        SWITCH: {
            category: 'reference',
            name: { ooxml: '_xlfn.SWITCH', odf: null },
            minParams: 3,
            // no repetition pattern, SWITCH supports a 'default value' (single parameter after a pair of parameters)!
            type: 'any',
            signature: 'val any:lazy|deps:pass any:lazy|deps:pass',
            resolve: function (selector) {

                // total number of operands
                var length = this.getOperandCount();
                // number of parameter pairs following the leading selector parameter
                var pairs = Math.floor((length - 1) / 2);

                // try to find a parameter pair matching the selector
                for (var index = 0; index < pairs; index += 1) {
                    // immediately throw on error code
                    var value = this.getOperand(index * 2 + 1, 'val');
                    // do not convert data types while comparing (number 1 as selector DOES NOT select string '1')
                    if (this.compareScalars(selector, value) === 0) {
                        return this.getOperand(index * 2 + 2);
                    }
                }

                // return default value if provided, otherwise #N/A error code (thrown automatically)
                return this.getOperand(pairs * 2 + 1);
            }
        },

        VLOOKUP: {
            category: 'reference',
            minParams: 3,
            maxParams: 4,
            type: 'any',
            signature: 'val any val:int val:bool',
            resolve: function (lookup, searchOperand, colIndexParam, inexact) {
                var colIndex = colIndexParam - 1;
                var dim = searchOperand.getDimension({ errorCode: ErrorCode.VALUE }); // #VALUE! thrown for complex references

                if (colIndex < 0) { throw ErrorCode.VALUE; }
                if (colIndex >= dim.cols) { throw ErrorCode.REF; }

                var ops = splitOperand(this, searchOperand, [{ col: 0, cols: 1, row: 0, rows: dim.rows }, { col: colIndex, cols: 1, row: 0, rows: dim.rows }]);

                return searchInOperand(this, lookup, inexact, ops[0], ops[1]);
            }
        }
    };

});
