/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/funcs/databasefuncs', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/iteratorutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/formulautils'
], function (Utils, IteratorUtils, SheetUtils, FormulaUtils) {

    'use strict';

    /**************************************************************************
     *
     * This module implements all spreadsheet functions treating cell ranges
     * like minimal databases.
     *
     * See the README file in this directory for a detailed documentation about
     * the format of function descriptor objects.
     *
     *************************************************************************/

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Range3D = SheetUtils.Range3D;
    var MathUtils = FormulaUtils.Math;

    // shortcuts to mathematical functions
    var div = MathUtils.div;

    // class Database =========================================================

    /**
     * Database created from the range of cells.
     * The first row of the range will be used for the database column labels.
     *
     * @param {Range3D} range to create the database.
     * @param {String | Number} field indicate the column wich are used for the databasefuncs.
     *  It's the name or the index of the column.
     * @param {FormulaContext} context the formulacontext.
     * @param {Boolean} [emptyFieldIsNotValid=true] true if the field value can be null, otherwise
     *  the field must be a String or Number.
     */
    function Database(range, field, context, emptyFieldIsNotValid) {

        var self = this;

        // private methods ----------------------------------------------------

        /**
         * Set the column index with the field value.
         */
        function setColumnIndex() {
            var index = -1;
            if (_.isString(field)) {
                index = self.getColumnIndexByName(field);
            } else if (_.isNumber(field) && field >= 1 && range.cells() >= field) {
                index = field - 1;
            }
            self.columnIndex = index;
        }

        // public methods ----------------------------------------------------

        /**
         * Return the database column index by the name of the label.
         * @param {String} name to the column inde.
         * @returns {Number} the column index or -1 if the column label with the given name not exists.
         */
        this.getColumnIndexByName = function (name) {
            var index = -1;

            if (name) {
                name = name.toLowerCase();
                var iterator = context.createScalarIterator(getRowRange(range, range.start[1]));
                IteratorUtils.some(iterator, function (columnName, result) {
                    if (name === context.convertToString(columnName).toLowerCase()) {
                        index = result.offset;
                        return true;
                    }
                });
            }

            return index;
        };

        this.getRange = function () {
            return range;
        };

        this.columnIndex = -1;

        // initialization -----------------------------------------------------

        if (range.rows() < 2) {
            throw ErrorCode.VALUE;
        }

        // Todo Check max row size
        if ((_.isBoolean(emptyFieldIsNotValid) ? emptyFieldIsNotValid : true) && field === null) {
            throw ErrorCode.NAME;
        }

        if (field !== null) {
            setColumnIndex();

            if (self.columnIndex === -1) {
                throw ErrorCode.VALUE;
            }
        }
    }

    /**
     * Criteria is a range wich specifie the criteria to filter the database values.
     * The minimum of rows are two rows, the first is the name to identifie the database
     * columns and the second is for the criterias.
     *
     * @param {Range3D} range the range to create the criteria.
     * @param {Database} db to check if the columns of the criteria exists.
     * @param {FormulaContext} context the formulacontext.
     */
    function Criteria(range, db, context) {

        var filters = [],
            criteriaRowSize,
            withoutCriteria;

        function getFilterForRow(rowIndex) {
            var filter = _.find(filters, function (rowFilter) {
                return rowFilter.rowIndex === rowIndex;
            });
            if (!filter) {
                filter = [];
                filter.rowIndex = rowIndex;
                filters.push(filter);
            }
            return filter;
        }

        /**
         * Create the criteria filters and check if the column exists in the databse.
         */
        function createFilters() {
            var dbColumnIndex,
                filterMatcher,
                columnFilters,
                rowFilters;

            // iterate through all columns
            context.iterateScalars(getRowRange(range, range.start[1]), function (colName, index1, offset1, col1) {

                dbColumnIndex = db.getColumnIndexByName(context.convertToString(colName));
                if (dbColumnIndex < 0) { throw ErrorCode.DIV0; }

                // Iterate through all rows. If a criteria is reached save it.
                context.iterateScalars(getColumnValueRange(col1), function (value, index2, offset2, col2, row2) {

                    rowFilters = getFilterForRow(row2);

                    filterMatcher = context.createFilterMatcher(value);

                    columnFilters = rowFilters[dbColumnIndex];

                    if (!_.isArray(columnFilters)) {
                        columnFilters = [];
                        columnFilters.dbColumnIndex = dbColumnIndex;
                        rowFilters.push(columnFilters);
                    }

                    columnFilters.push(filterMatcher);

                });
            });

            if (context.fileFormat === 'ooxml') {
                while (criteriaRowSize > filters.length) {
                    filters.push(null);
                }
            }
            withoutCriteria = filters.length === 0;
        }

        /**
         * Return a range of criteria values from a column without the label.
         * @param {Number} columnIndex the column index to values.
         * @returns {Range3D} the created range.
         */
        function getColumnValueRange(columnIndex) {
            return Range3D.create(range.start[0] + columnIndex, range.start[1] + 1, range.start[0] + columnIndex, range.end[1], range.sheet1, range.sheet2);
        }

        /**
         * Check if values match to the criteria filters.
         * @param {Object[]} values the values to check.
         * @returns {Boolean} true if the filter matches, otherwise false.
         */
        this.matchValues = function (values) {
            var rowValue;

            return withoutCriteria || _.some(filters, function (rowFilters) {

                return _.all(rowFilters, function (columnFilters) {

                    rowValue = values[columnFilters.dbColumnIndex];

                    return _.all(columnFilters, function (filterMatcher) {
                        return filterMatcher(rowValue);
                    });
                });
            });
        };

        // initialization -----------------------------------------------------

        if (range.rows() < 2) {
            throw ErrorCode.VALUE;
        }

        criteriaRowSize = range.rows() - 1;

        createFilters();
    }

    /**
     * Create a range which includes a singel row with all columns.
     * @param {Range3D} range the range to extract the new range
     * @param {type} rowIndex the row index for create the range
     * @returns {Range3D} the created range.
     */
    function getRowRange(range, rowIndex) {
        return Range3D.create(range.start[0], rowIndex, range.end[0], rowIndex, range.sheet1, range.sheet2);
    }

    /**
     * Get the function result.
     *
     * @param {Range3D} range
     *  to create the database.
     *
     * @param {String|Number} field
     *  indicate the column wich are used for the databasefuncs. It's the name
     *  or the index of the column.
     *
     * @param {Range3D} criteriaRange
     *  the range to create the criteria.
     *
     * @param {FormulaContext} context
     *  the formulacontext.
     *
     * @param {Function} resultCallback
     *  a function wich get the result array as the first parameter.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.resultType='array']
     *      If 'array' add the number values to an result array, if 'count'
     *      only count the the matched rows.
     *  @param {Boolean} [options.onlyNumbers=true]
     *      if true the value for the result must be a Number, otherwise all
     *      values will be added to the result
     *  @param {Number} [options.stopOnMatchCount=0]
     *      if <=0 all values will be added to the result, otherwise stops on
     *      the x match of the value.
     *  @param {Boolean} [options.emptyFieldIsNotValid=true]
     *      true if the field value can be null, otherwise the field must be a
     *      String or Number.
     *
     * @returns {Any}
     *  the return value of the resultCallback.
     */
    function getResult(dbRange, field, criteriaRange, context, resultCallback, options) {

        var resultIsArray = Utils.getStringOption(options, 'resultType', 'array') === 'array',
            result = resultIsArray ? [] : 0,
            onlyNumbers = Utils.getBooleanOption(options, 'onlyNumbers', true),
            stopOnMatchCount = Utils.getNumberOption(options, 'stopOnMatchCount', 0),
            sum = 0,
            emptyFieldIsNotValid = Utils.getBooleanOption(options, 'emptyFieldIsNotValid', true),
            db = new Database(dbRange, field, context, emptyFieldIsNotValid),
            criteria = new Criteria(criteriaRange, db, context),
            rowIndex = db.getRange().start[1] + 1,
            endRowIndex = db.getRange().end[1];

        for (rowIndex; rowIndex <= endRowIndex; rowIndex++) {
            var rowValues = context.getScalarsAsArray(getRowRange(db.getRange(), rowIndex), { emptyCells: true, acceptErrors: true });

            if (criteria.matchValues(rowValues)) {
                var cellValue = rowValues[db.columnIndex];
                if (resultIsArray && (cellValue instanceof ErrorCode)) { throw cellValue; }
                var isNumber = typeof cellValue === 'number';
                if ((!onlyNumbers && cellValue !== null) || isNumber) {

                    if (isNumber) { sum += cellValue; }

                    if (resultIsArray) {
                        result.push(cellValue);
                        if (stopOnMatchCount > 0 && result.length >= stopOnMatchCount) {
                            break;
                        }
                    } else {
                        result++;
                        if (stopOnMatchCount > 0 && result >= stopOnMatchCount) {
                            break;
                        }
                    }
                }
            }
        }

        return resultCallback(result, sum);
    }

    /**
     * Return the count of the cells which are matched to the criteria
     * @param {Range3D} dbRange @see getResult()
     * @param {String | Number} field @see getResult()
     * @param {Range3D} criteriaRange @see getResult()
     * @param {FormulaContext} context @see getResult()
     * @param {Boolean} nonblankCells if true or if the field is null all field cells will count
     * @returns {Number} the count of cells.
     */
    function getCountResult(dbRange, field, criteriaRange, context, nonblankCells) {
        return getResult(dbRange, field, criteriaRange, context, _.identity, {
            resultType: 'count',
            onlyNumbers: (field !== null) && !nonblankCells,
            emptyFieldIsNotValid: false
        });
    }

    // exports ================================================================

    return {

        DAVERAGE: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getResult(dbRange, field, criteriaRange, this, function (numbers, sum) {
                    return div(sum, numbers.length);
                });
            }
        },

        DCOUNT: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getCountResult(dbRange, field, criteriaRange, this, false);
            }
        },

        DCOUNTA: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getCountResult(dbRange, field, criteriaRange, this, true);
            }
        },

        DGET: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:any',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getResult(dbRange, field, criteriaRange, this, function (values) {
                    if (values.length === 0) { throw ErrorCode.VALUE; }
                    if (values.length > 1) { throw ErrorCode.NUM; }
                    return values[0];
                }, { onlyNumbers: false });
            }
        },

        DMAX: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getResult(dbRange, field, criteriaRange, this, function (numbers) {
                    return (numbers.length === 0) ? 0 : _.max(numbers);
                });
            }
        },

        DMIN: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getResult(dbRange, field, criteriaRange, this, function (numbers) {
                    return (numbers.length === 0) ? 0 : _.min(numbers);
                });
            }
        },

        DPRODUCT: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getResult(dbRange, field, criteriaRange, this, function (numbers) {
                    return (numbers.length === 0) ? 0 : MathUtils.product(numbers);
                });
            }
        },

        DSTDEV: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getResult(dbRange, field, criteriaRange, this, function (numbers, sum) {
                    return MathUtils.variance(numbers, sum, 'dev', false);
                });
            }
        },

        DSTDEVP: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getResult(dbRange, field, criteriaRange, this, function (numbers, sum) {
                    return MathUtils.variance(numbers, sum, 'dev', true);
                });
            }
        },

        DSUM: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getResult(dbRange, field, criteriaRange, this, function (numbers, sum) {
                    return sum;
                });
            }
        },

        DVAR: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getResult(dbRange, field, criteriaRange, this, function (numbers, sum) {
                    return MathUtils.variance(numbers, sum, 'var', false);
                });
            }
        },

        DVARP: {
            category: 'database',
            minParams: 3,
            maxParams: 3,
            type: 'val:num',
            signature: 'ref:single val ref:single',
            resolve: function (dbRange, field, criteriaRange) {
                return getResult(dbRange, field, criteriaRange, this, function (numbers, sum) {
                    return MathUtils.variance(numbers, sum, 'var', true);
                });
            }
        }
    };

});
