/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/formuladescriptor', [
    'io.ox/office/spreadsheet/model/formula/deps/dependencyutils',
    'io.ox/office/spreadsheet/model/formula/deps/referencecollection'
], function (DependencyUtils, ReferenceCollection) {

    'use strict';

    // class FormulaDescriptor ================================================

    /**
     * A descriptor object for a token array in a spreadsheet document, that
     * contains all its dependency settings.
     *
     * @constructor
     *
     * @property {String} sheetUid
     *  The UID of the sheet model containing the formula expression.
     *
     * @property {String} formulaKey
     *  The unique key of this formula descriptor, used e.g. for storing this
     *  descriptor in maps and sets.
     *
     * @property {TokenArray} tokenArray
     *  The token array represented by this descriptor object.
     *
     * @property {Address} refAddress
     *  The reference address of the formula, i.e. the address of the cell the
     *  relative references are associated to. For cell formulas, this is the
     *  address of the cell containing the formula. For shared formulas, this
     *  is the address of its anchor cell. For matrix formulas, for conditional
     *  formatting rules, and for data validation rules, this is the top-left
     *  cell of the bounding range of the parent object. For source links, this
     *  will be the address A1.
     *
     * @property {Address|Null} cellAddress
     *  The actual address of a cell formula, i.e. the address of the cell the
     *  formula is associated with. Will differ to the reference address, if
     *  the cell is part of a shared formula, but is not its anchor cell. Will
     *  be null, if this instance does not represent a cell formula.
     *
     * @property {Range|Null} matrixRange
     *  The bounding range of a matrix formula; otherwise null. The start
     *  address of this range MUST be equal to the properties 'refAddress' and
     *  'cellAddress' of this instance.
     *
     * @property {RuleModel|SourceLinkMixin|Null} tokenModel
     *  For conditions of formatting rules, this will be the parent rule model
     *  containing the formula. For formulas of a source link model, this will
     *  be the parent link model containing the formula (e.g. the data series
     *  model of a chart). Otherwise, this property will be null.
     *
     * @property {ReferenceCollection} references
     *  A collection with source references the token array depends on
     *  directly. If this collection is empty, the token array does not depend
     *  on any other external object in the spreadsheet document.
     *
     * @property {Object<String,Boolean>} missingNames
     *  A flag set with the lower-case labels of all missing defined names (as
     *  object keys) occurring in the formula expression, or in any existing
     *  defined name referred by the formula.
     *
     * @property {Object<String,Boolean>} missingFuncs
     *  A flag set with the resource keys of all unimplemented functions that
     *  occur in the token array.
     *
     * @property {String|Null} recalc
     *  The recalculation mode of the formula expression, as deduced from the
     *  recalculation modes of the functions: Either 'always', 'once', or null.
     *
     * @property {Boolean} circular
     *  If set to true, a circular dependency has been detected for the formula
     *  represented by this descriptor. In this case, resolving the dependency
     *  chain for this formula will be stopped to prevent endless loops.
     */
    function FormulaDescriptor(formulaKey, tokenArray, refAddress, cellAddress, matrixRange, tokenModel) {

        // public properties
        this.sheetUid = tokenArray.getSheetModel().getUid();
        this.formulaKey = formulaKey;
        this.tokenArray = tokenArray;
        this.refAddress = refAddress;
        this.cellAddress = cellAddress || null;
        this.matrixRange = matrixRange || null;
        this.tokenModel = tokenModel || null;
        this.references = null;
        this.missingNames = null;
        this.missingFuncs = null;
        this.recalc = null;
        this.circular = false;

        // initialization -----------------------------------------------------

        // calculate the initial dependencies, and create the reference collection
        this.refreshReferences();

    } // class FormulaDescriptor

    // public methods ---------------------------------------------------------

    /**
     *
     * Returns the model of the sheet containing the formula represented by
     * this instance.
     *
     * @returns {SheetModel}
     *  The model of the sheet containing the formula represented by this
     *  instance.
     */
    FormulaDescriptor.prototype.getSheetModel = function () {
        return this.tokenArray.getSheetModel();
    };

    /**
     * Returns whether the formula represented by this descriptor does not have
     * any external dependencies, and therefore does not need to be handled by
     * the dependency manager.
     *
     * @returns {Boolean}
     *  Whether the formula represented by this descriptor does not have any
     *  external dependencies.
     */
    FormulaDescriptor.prototype.isFixed = function () {
        return !this.recalc && this.references.empty() && _.isEmpty(this.missingNames) && _.isEmpty(this.missingFuncs);
    };

    /**
     * Recalculates the dependency setings for the formula represented by this
     * descriptor.
     */
    FormulaDescriptor.prototype.refreshReferences = function () {

        // get the target position for resolving the dependencies
        var depTarget = this.tokenModel ? this.tokenModel.getDependencyTarget() : (this.cellAddress || this.refAddress);

        // resolve the dependencies of the token array (cell ranges, and defined names)
        var dependencies = this.tokenArray.getDependencies(this.refAddress, depTarget);
        this.references = new ReferenceCollection(this.tokenArray.getDocModel(), dependencies);

        // copy the other settings
        this.missingNames = dependencies.missingNames;
        this.missingFuncs = dependencies.missingFuncs;
        this.recalc = dependencies.recalc;
        this.circular = dependencies.circular;
    };

    /**
     * Returns a string representation for this formula descriptor for debug
     * logging.
     *
     * @returns {String}
     *  A string representation for this formula descriptor for debug logging.
     */
    FormulaDescriptor.prototype.toString = function () {
        if (this.tokenModel) { return DependencyUtils.getTokenLabel(this.tokenModel) + '!' + this.formulaKey.split('!')[1]; }
        if (this.cellAddress) { return this.getSheetModel().getName() + '!' + this.cellAddress; }
        return '<!unknown!>' + this.formulaKey;
    };

    // exports ================================================================

    return FormulaDescriptor;

});
