/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/formula/deps/dependencyutils', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/tk/container/valueset'
], function (Config, Logger, ValueSet) {

    'use strict';

    // small delays in Selenium tests and unit tests
    var FAST_MODE = Config.AUTOTEST || Config.UNITTEST;

    // static class DependencyUtils ===========================================

    var DependencyUtils = {};

    // constants --------------------------------------------------------------

    /**
     * Duration for a single time slice of the background update task of the
     * dependency manager (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.TIMER_SLICE = FAST_MODE ? 300 : 150;

    /**
     * Duration between two time slices of the background update task of the
     * dependency manager (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.TIMER_INTERVAL = FAST_MODE ? 10 : 100;

    /**
     * Duration for a single time slice of the background update task of the
     * dependency manager for high-priority mode (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.PRIORITY_TIMER_SLICE = FAST_MODE ? 300 : 200;

    /**
     * Duration between two time slices of the background update task of the
     * dependency manager for high-priority mode (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.PRIORITY_TIMER_INTERVAL = FAST_MODE ? 10 : 50;

    /**
     * Delay for starting the background update task of the dependency manager
     * after receiving a document event (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.UPDATE_DEBOUNCE_DELAY = FAST_MODE ? 10 : 100;

    /**
     * Additional delay for the first background update task of the dependency
     * manager after import (in milliseconds).
     *
     * @constant
     * @type {Number}
     */
    DependencyUtils.IMPORT_UPDATE_DELAY = FAST_MODE ? 10 : 900;

    // logger interface -------------------------------------------------------

    // debug logger for the dependency manager
    Logger.extend(DependencyUtils, { enable: 'spreadsheet:log-deps', prefix: 'DEP' });

    // classes ----------------------------------------------------------------

    /**
     * A set of instances of the class FormulaDescriptor.
     *
     * @constructor
     *
     * @extends ValueSet
     */
    DependencyUtils.FormulaSet = ValueSet.extend({ constructor: function () {
        ValueSet.call(this, 'formulaKey');
    } });

    // static methods ---------------------------------------------------------

    /**
     * Returns the formula key for the specified sheet UID and cell address.
     *
     * @param {String} sheetUid
     *  The UID of a sheet in a spreadsheet document.
     *
     * @param {Address} address
     *  The cell address to create a cell key for.
     *
     * @returns {String}
     *  The formula key for the specified cell address.
     */
    DependencyUtils.getCellKey = function (sheetUid, address) {
        return sheetUid + '!' + address.key();
    };

    /**
     * Returns the formula key for the specified source link formula.
     *
     * @param {SourceLinkMixin} linkModel
     *  An attributed model instance with the SourceLinkMixin class mixed in.
     *
     * @param {String} linkKey
     *  The fully qualified attribute name of the source link formula.
     *
     * @returns {String}
     *  The formula key for the specified source link formula.
     */
    DependencyUtils.getLinkKey = function (linkModel, linkKey) {
        return linkModel.getUid() + '!' + linkKey;
    };

    // debugging methods ------------------------------------------------------

    /**
     * Returns a readable label for the passed cell range address for debug
     * logging.
     *
     * @param {SheetModel} sheetModel
     *  The model of the sheet containing the cell range.
     *
     * @param {Range} range
     *  The address of a cell range.
     *
     * @returns {String}
     *  A readable label for the passed cell range address for debug logging.
     */
    DependencyUtils.getRangeLabel = function (sheetModel, range) {
        return sheetModel.getName() + '!' + (range.single() ? range.start : range);
    };

    /**
     * Returns a readable label for the passed defined name for debug logging.
     *
     * @param {NameModel} nameModel
     *  The model instance of a defined name.
     *
     * @returns {String}
     *  A readable label for the passed defined name for debug logging.
     */
    DependencyUtils.getNameLabel = function (nameModel) {
        var sheetModel = nameModel.getSheetModel();
        return '{NAME}' + (sheetModel ? sheetModel.getName() : '[0]') + '!' + nameModel.getLabel();
    };

    /**
     * Returns a readable label for the passed table range for debug logging.
     *
     * @param {TableModel} tableModel
     *  The model instance of a table range.
     *
     * @returns {String}
     *  A readable label for the passed table range for debug logging.
     */
    DependencyUtils.getTableLabel = function (tableModel) {
        return '{TABLE}' + tableModel.getSheetModel().getName() + '!' + tableModel.getName();
    };

    /**
     * Returns a readable label for the passed formatting rule for debug
     * logging.
     *
     * @param {RuleModel} ruleModel
     *  The model instance of a formatting rule.
     *
     * @returns {String}
     *  A readable label for the passed formatting rule for debug logging.
     */
    DependencyUtils.getRuleLabel = function (ruleModel) {
        return '{RULE}' + ruleModel.getSheetModel().getName() + '!' + ruleModel.getTargetRanges();
    };

    /**
     * Returns a readable label for the passed source link for debug logging.
     *
     * @param {SourceLinkMixin} linkModel
     *  The model instance of a source link.
     *
     * @returns {String}
     *  A readable label for the passed source link for debug logging.
     */
    DependencyUtils.getLinkLabel = function (linkModel) {
        return '{LINK}' + linkModel.getSheetModel().getName() + '!' + linkModel.getUid();
    };

    DependencyUtils.getTokenLabel = function (tokenModel) {
        return tokenModel.getTargetRanges ? DependencyUtils.getRuleLabel(tokenModel) : DependencyUtils.getLinkLabel(tokenModel);
    };

    /**
     * Prints all formula descriptors in the passed map to the browser console,
     * if dependency logging is active.
     *
     * @param {FormulaSet} formulaSet
     *  All formula descriptors to be logged to the browser console.
     */
    DependencyUtils.logFormulaSet = DependencyUtils.isLoggingActive() ? function (formulaSet) {
        formulaSet.forEach(function (formulaDesc) {
            DependencyUtils.log('location=' + formulaDesc + ' key=' + formulaDesc.formulaKey + ' references=' + formulaDesc.references + ' circular=' + formulaDesc.circular);
        });
    } : _.noop;

    // exports ================================================================

    return DependencyUtils;

});
