/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/sourcelinkmixin', [
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray'
], function (ValueMap, AttributeUtils, SheetUtils, TokenArray) {

    'use strict';

    // convenience shortcuts
    var ErrorCode = SheetUtils.ErrorCode;
    var Address = SheetUtils.Address;

    // mix-in class SourceLinkMixin ===========================================

    /**
     * Mix-in class for attributed object models referring dynamically to cell
     * source data in the spreadsheet document via formula expressions stored
     * in formatting attributes.
     *
     * @constructor
     *
     * @internal
     *  This is a mix-in class supposed to extend an existing instance of the
     *  class AttributedModel or any of its sub classes, used for drawing
     *  objects or any of their sub objects (e.g. chart titles). Expects the
     *  symbol 'this' to be bound to an instance of AttributedModel.
     *
     * @param {SheetModel} sheetModel
     *  The sheet model that contains the model object.
     *
     * @param {String} linkAttrQNames
     *  The fully qualified names of all formatting attributes (with leading
     *  family name, separated by a period, e.g. 'text.link') containing source
     *  link formulas, separated by white-space characters.
     */
    function SourceLinkMixin(sheetModel, linkAttrQNames) {

        // self reference
        var self = this;

        // the document model containing the model object
        var docModel = sheetModel.getDocModel();

        // token arrays for dynamic source ranges, mapped by internal link keys
        var tokenArrayMap = new ValueMap();

        // maps internal link keys to fully qualified attribute names
        var qNameMap = new ValueMap();

        // private methods ----------------------------------------------------

        /**
         * Updates the token arrays, after the formatting attributes of this
         * instance have been changed.
         */
        function updateTokenArrays(oldAttrSet) {

            // the current merged attribute set
            var attrSet = self.getMergedAttributeSet(true);
            // whether any source link attribute has been changed
            var changedTokens = false;

            // process all source link attributes in the current merged attribute set
            qNameMap.forEach(function (qName, linkKey) {

                // the attribute value (string is a formula expression, array represents constant data)
                var oldValue = oldAttrSet ? oldAttrSet[qName.family][qName.attr] : null;
                var newValue = attrSet[qName.family][qName.attr];
                if (_.isEqual(oldValue, newValue)) { return; }

                // create, update, or delete the token array according to type of source data
                if ((typeof newValue === 'string') && (newValue.length > 0)) {
                    // string: parse link formula
                    tokenArrayMap.getOrConstruct(linkKey, TokenArray, sheetModel, 'link').parseFormula('op', newValue);
                    // token array has been created or changed: notify listeners
                    changedTokens = true;
                } else {
                    // else: constant source data, or invalid attribute value: delete token array
                    if (tokenArrayMap.remove(linkKey)) {
                        changedTokens = true;
                    }
                }
            });

            // notify changed source link attributes
            if (changedTokens) {
                self.trigger('change:sourcelinks');
            }
        }

        /**
         * Interprets the passed token array, and returns the resulting cell
         * range addresses referred by the formula expression.
         *
         * @param {TokenArray} tokenArray
         *  The token array to be resolved to the cell range addresses.
         *
         * @returns {Range3DArray}
         *  An array with cell range addresses (with sheet indexes). If the
         *  token array cannot be evaluated successfully, an empty array will
         *  be returned.
         */
        function resolveRangeList(tokenArray) {
            return tokenArray.resolveRangeList({ resolveNames: 'interpret' });
        }

        // public methods -----------------------------------------------------

        /**
         * Returns the sheet model instance that contains this model instance.
         *
         * @returns {SheetModel}
         *  The sheet model instance that contains this model instance.
         */
        this.getSheetModel = function () {
            return sheetModel;
        };

        /**
         * Returns the reference addresses of the link formulas.
         *
         * @returns {Address}
         *  The reference address of the link formulas.
         */
        this.getRefAddress = function () {
            return Address.A1.clone();
        };

        /**
         * Returns the address of the target cell to be used to resolve the
         * dependencies of the link formulas.
         *
         * @returns {Address}
         *  The address of the dependency target cell.
         */
        this.getDependencyTarget = function () {
            return Address.A1.clone();
        };

        /**
         * Returns an iterator that visits all token arrays of this instance.
         * The token arrays will be visited in no specific order.
         *
         * @returns {Object}
         *  An iterator object that implements the standard EcmaScript iterator
         *  protocol, i.e. it provides the method next() that returns a result
         *  object with the following properties:
         *  - {Boolean} done
         *      If set to true, the token arrays have been visited completely.
         *      No more token arrays are available; this result object does not
         *      contain any other properties!
         *  - {TokenArray} value
         *      The token array currently visited.
         *  - {String} key
         *      The fully qualified attribute name (attribute family and
         *      attribute name, separated by a period, e.g. 'text.link') of the
         *      source link formula, used as internal map key.
         */
        this.createTokenArrayIterator = function () {
            return tokenArrayMap.iterator();
        };

        /**
         * Returns the addresses of the source cell ranges referred by the
         * specified formatting attribute, if existing.
         *
         * @param {String} linkKey
         *  The fully qualified attribute name (attribute family and attribute
         *  name, separated by a period, e.g. 'text.link') of the source link
         *  to be resolved.
         *
         * @returns {Range3DArray|Null}
         *  An array of cell range addresses (with sheet indexes), if the
         *  specified link is associated with a cell range in the document; or
         *  null, if the specified link does not exist or contains constant
         *  data.
         */
        this.resolveRanges = function (linkKey) {
            return tokenArrayMap.with(linkKey, resolveRangeList) || null;
        };

        /**
         * Returns the current text representation of the source link referred
         * by the specified formatting attribute.
         *
         * @param {String} linkKey
         *  The fully qualified attribute name (attribute family and attribute
         *  name, separated by a period, e.g. 'text.link') of the source link
         *  to be resolved.
         *
         * @returns {String}
         *  The current text representation of the source link referred by the
         *  specified formatting attribute.
         */
        this.resolveText = function (linkKey) {

            // resolve an existing link into the document
            var tokenArray = tokenArrayMap.get(linkKey, null);
            if (tokenArray) {

                // invalid source links result in the #REF! error code
                var ranges = resolveRangeList(tokenArray);
                if (ranges.empty()) {
                    return docModel.getFormulaGrammar('ui').getErrorName(ErrorCode.REF);
                }

                // resolve cell display strings (also from hidden columns/rows, but skip blank cells)
                var contents = docModel.getRangeContents(ranges, { display: true, maxCount: 100 });

                // skip cells with empty display string
                contents = contents.filter(function (entry) { return entry.display.length > 0; });

                // concatenate all cell display strings with a simple space
                return _.pluck(contents, 'display').join(' ');
            }

            // get the original attribute value for the specified source link
            var attrValue = qNameMap.with(linkKey, function (qName) {
                return this.getMergedAttributeSet(true)[qName.family][qName.attr];
            }, this);

            // filter non-empty strings from an array, concatenate all strings with a simple space
            return _.isArray(attrValue) ? attrValue.filter(function (element) {
                return (typeof element === 'string') && (element.length > 0);
            }).join(' ') : '';
        };

        /**
         * Returns whether any of the source links contained in this object
         * overlaps with any of the passed cell ranges.
         *
         * @param {Range3DArray|Range3D} ranges
         *  The addresses of the cell ranges, or a single cell range address,
         *  to be checked. The cell range addresses MUST be instances of the
         *  class Range3D with sheet indexes.
         *
         * @returns {Boolean}
         *  Whether any of the passed ranges overlaps with the source links of
         *  this object.
         */
        this.rangesOverlap = function (ranges) {
            return tokenArrayMap.some(function (tokenArray) {
                return resolveRangeList(tokenArray).overlaps(ranges);
            });
        };

        /**
         * Invokes the specified callback function for all existing token
         * arrays of this object.
         *
         * @param {Function} callback
         *  The callback function to be invoked for each existing token array.
         *  Receives the following parameters:
         *  (1) {TokenArray} tokenArray
         *      The token array representing the source link formula.
         *  (2) {String} linkKey
         *      The fully qualified attribute name (attribute family and
         *      attribute name, separated by a period, e.g. 'text.link').
         *  (3) {String} attrName
         *      The name of the formatting attribute containing the formula
         *      expression of the source link.
         *  (4) {String} attrFamily
         *      The family name of the formatting attribute containing the
         *      formula expression of the source link.
         *
         * @param {Object} [context]
         *  The calling context for the callback function.
         *
         * @returns {SourceLinkMixin}
         *  A reference to this instance.
         */
        this.iterateTokenArrays = function (callback, context) {
            tokenArrayMap.forEach(function (tokenArray, linkKey) {
                qNameMap.with(linkKey, function (qName) {
                    callback.call(context, tokenArray, linkKey, qName.attr, qName.family);
                });
            });
            return this;
        };

        /**
         * Fires a 'refresh:formulas' event to the listeners of this drawing
         * model, after the source data referred by the source link formulas
         * has been changed. This method will be called automatically by the
         * dependency manager of the spreadsheet document.
         *
         * @returns {SourceLinkMixin}
         *  A reference to this instance.
         */
        this.refreshFormulas = function () {
            return this.trigger('refresh:formulas');
        };

        // operation generators -----------------------------------------------

        /**
         * Generates the operations and undo operations to update or restore
         * the formula expressions of all source links of this object.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {String} operationName
         *  The name of the document operation to be generated for the formula
         *  expressions.
         *
         * @param {Array<Number>} position
         *  The position of the parent drawing object in the sheet, as expected
         *  by the method SheetOperationGenerator.generateDrawingOperation().
         *
         * @param {Object} changeDesc
         *  The properties describing the document change. The properties that
         *  are expected in this descriptor depend on the change type in its
         *  'type' property. See method TokenArray.resolveOperation() for more
         *  details.
         *
         * @param {Object} [properties]
         *  Additional properties to be inserted into the document operations.
         */
        this.implGenerateUpdateFormulaOperations = function (generator, operationName, position, changeDesc, properties) {

            // collect the changed formula expressions
            var attrSet = {};
            var undoAttrSet = {};
            this.iterateTokenArrays(function (tokenArray, linkKey, attrName, attrFamily) {
                tokenArray.resolveOperation('op', changeDesc, function (newFormula, oldFormula) {
                    AttributeUtils.insertAttribute(attrSet, attrFamily, attrName, newFormula);
                    AttributeUtils.insertAttribute(undoAttrSet, attrFamily, attrName, oldFormula);
                });
            });

            // generate the operations, if any formula expression has changed
            if (!_.isEmpty(attrSet)) {
                var operProperties = _.extend({}, properties, { attrs: attrSet });
                generator.generateDrawingOperation(operationName, position, operProperties);
                var undoProperties = _.extend({}, properties, { attrs: undoAttrSet });
                generator.generateDrawingOperation(operationName, position, undoProperties, { undo: true });
            }
        };

        // initialization -----------------------------------------------------

        // split the passed fully-qualified attribute names into family name and attribute name
        linkAttrQNames.split(/\s+/).forEach(function (qName) {
            var tokens = qName.split('.');
            qNameMap.insert(qName, { family: tokens[0], attr: tokens[1] });
        });

        // additional processing and event handling after the document has been imported
        this.waitForImportSuccess(function () {

            // create the token arrays after construction (but wait after the document has been imported,
            // otherwise the sheets referred by the token arrays may be missing)
            updateTokenArrays();

            // update the token arrays after any formatting attributes have changed
            this.on('change:attributes', function (event, newAttrSet, oldAttrSet) {
                updateTokenArrays(oldAttrSet);
            });

            // update the sheet indexes after the sheet collection has been manipulated
            this.listenTo(docModel, 'transform:sheet', function (event, fromSheet, toSheet) {
                tokenArrayMap.forEach('transformSheet', fromSheet, toSheet);
            });

            // register this model in the dependency manager for formula recalculation
            docModel.getDependencyManager().registerLinkModel(this);

        }, this);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel.getDependencyManager().unregisterLinkModel(this);
            self = sheetModel = docModel = null;
            tokenArrayMap = qNameMap = null;
        });

    } // class SourceLinkMixin

    // exports ================================================================

    return SourceLinkMixin;

});
