/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Stefan Eckert <stefan.eckert@open-xchange.com>
 */

define('io.ox/office/spreadsheet/model/drawing/chartformatter', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/object/baseobject',
    'io.ox/office/tk/locale/parser',
    'io.ox/office/spreadsheet/view/labels'
], function (Utils, BaseObject, Parser, Labels) {

    'use strict';

    //'io.ox/office/spreadsheet/model/numberformatter'
    // null date (corresponding to cell value zero) TODO: use null date from document settings

    // class ChartFormatter ===================================================

    function ChartFormatter(docModel, chartModel) {

        var fileFormat = docModel.getApp().getFileFormat();

        // base constructor ---------------------------------------------------

        BaseObject.call(this, docModel);

        // private methods ----------------------------------------------------

        function isAnyDateTime(format, value) {
            if (format && format.isAnyDateTime() && Utils.isFiniteNumber(value)) {
                return true;
            }
        }

        function newInfoHolder(type, label) {
            return {
                max: Number.NEGATIVE_INFINITY,
                date: false,
                type: type,
                label: label,
                unclean: false,
                exp: 0
            };
        }

        function handleData(info, source, dataPoint, format, nullAsZero) {
            var realNr = source.value;
            var display = source.display;

            if (!format) { format = source.format; }

            if (format && !info.format) {
                if (isAnyDateTime(format, realNr)) {
                    info.date = format;
                }
                info.format = format;
            }

            if (!Utils.isFiniteNumber(realNr) && info.type === 'y' && nullAsZero) {
                realNr = 0;
            }

            var label = null;
            var res = null;
            if (info.date) {
                res = realNr;
                info.max = Math.max(info.max, Math.abs(realNr));
                label = display;
            } else if (!Utils.isFiniteNumber(realNr)) {
                info.unclean = true;
                res = nullAsZero ? 0 : undefined;
                label = display;
            } else if (info.format) {
                info.max = Math.max(info.max, Math.abs(realNr));
                res = realNr;
                label = docModel.getNumberFormatter().formatValue(info.format, realNr);
            } else {
                info.max = Math.max(info.max, Math.abs(realNr));
                res = realNr;
                label = docModel.getNumberFormatter().formatStandardNumber(realNr, 12);
            }

            dataPoint[info.label] = label;
            dataPoint[info.type] = res;
        }

        function calcMaxExp(info) {
            if (Utils.isFiniteNumber(info.max)) {
                var norm = Utils.normalizeNumber(info.max);
                if (Math.abs(norm.exp) < 6) {
                    info.exp = 0;
                } else {
                    info.exp = norm.exp;
                }
            } else {
                info.exp = 0;
            }
        }

        function handleMantissa(info, dataPoint) {
            if (info.exp) {
                dataPoint[info.type] = Utils.mant(dataPoint[info.type], info.exp);
            }
        }

        function handleLabelLength(info, dataPoint) {
            var label = dataPoint[info.label];
            if (label && label.length > 30) {
                dataPoint[info.label] = label.substring(0, 27) + '...';
            }
        }

        // public methods -----------------------------------------------------

        this.format = function () {
            var data = chartModel.getModelData();

            var form = null;
            var dateChart = false;
            var xyChart = false;
            var nullAsZero = false;
            var attrs = chartModel.getMergedAttributeSet(true);
            var stacking = attrs.chart.stacking;
            data.linearX = false;

            if (data.series.length) {
                form = data.series[0].model.getExplicitAttributeSet(true).series.format;
                if (form) {
                    form = this.parseFormatCode(form);
                }
                xyChart = chartModel.isXYType();
                nullAsZero = chartModel.isPieOrDonut() || /^(column|bar)/.test(chartModel.getChartType());
            }

            _.each(data.series, function (dataSeries, seriesIndex) {
                dataSeries.seriesIndex = seriesIndex;

                var before = data.series[seriesIndex - 1];

                if (before && before.axisXType === dataSeries.axisXType) {
                    dataSeries.xInfo = before.xInfo;
                } else {
                    dataSeries.xInfo = newInfoHolder('x', 'label');
                }
                if (before && before.axisYType === dataSeries.axisYType) {
                    dataSeries.yInfo = before.yInfo;
                } else {
                    dataSeries.yInfo = newInfoHolder('y', 'name');
                }

                var x = dataSeries.xInfo;
                var y = dataSeries.yInfo;

                if (dataSeries.nameHolder) {
                    dataSeries.name = dataSeries.nameHolder.display;
                } else if (docModel.getApp().isODF()) {
                    var fallback = true;
                    try {
                        var valueRanges = dataSeries.model.resolveRanges('series.bubbles');
                        if (!valueRanges || !valueRanges.length) {
                            valueRanges = dataSeries.model.resolveRanges('series.values');
                        }
                        if (valueRanges && valueRanges.length) {
                            var valueRange = valueRanges[0];
                            if (Math.abs(valueRange.start[0] - valueRange.end[0]) > 0) {
                                dataSeries.name = Labels.getRowLabel(valueRange.start[1]);
                                fallback = false;
                            } else if (Math.abs(valueRange.start[1] - valueRange.end[1]) > 0) {
                                dataSeries.name = Labels.getColLabel(valueRange.start[0]);
                                fallback = false;
                            } else if (dataSeries.dps.length) {
                                dataSeries.name = dataSeries.dps[0].name;
                                fallback = false;
                            }
                        }
                    } catch (e) {
                        Utils.warn('chart error while formatting', e);
                    }

                    if (fallback) {
                        dataSeries.name = Labels.getSeriesLabel(seriesIndex);
                    }
                } else {
                    dataSeries.name = Labels.getSeriesLabel(seriesIndex);
                }

                var dataPoints = dataSeries.dps;
                _.each(dataPoints, function (dataPoint, n) {
                    dataPoint.x = null;
                    dataPoint.y = null;
                    dataPoint.z = null;

                    handleData(y, dataPoint.valueSource, dataPoint, form, nullAsZero);

                    if (!n && !seriesIndex && dataPoint.nameSource) {
                        //first of first
                        if (attrs.chart.stacking === 'clustered' || attrs.chart.stacking === 'standard') {
                            var format = dataPoint.nameSource ? dataPoint.nameSource.format : null;
                            if (isAnyDateTime(format, dataPoint.nameSource.value)) {
                                dateChart = true;
                            }
                        }
                    }

                    if (dateChart) {
                        handleData(x, dataPoint.nameSource, dataPoint, null, nullAsZero);
                    } else if (xyChart && dataPoint.nameSource) {
                        handleData(x, dataPoint.nameSource, dataPoint, null, nullAsZero);
                    } else {
                        dataPoint[x.type] = n + 1;
                        data.linearX = true;
                    }

                    //ugly part against double formatting!!!
                    if (dataPoint.nameSource) {
                        dataPoint[x.label] = dataPoint.nameSource.display;
                    } else {
                        dataPoint[x.label] = String(dataPoint[x.type]);
                    }

                    if (dataPoint.label) { dataPoint.legendLabel = dataPoint.label; }

                    handleLabelLength(x, dataPoint);
                    handleLabelLength(y, dataPoint);

                    if (dataPoint.sizeSource) {
                        dataPoint.z = Math.abs(dataPoint.sizeSource.value);

                        //fix for funny bug in canvasjs
                        dataPoint.markerBorderColor = 'transparent';
                        dataPoint.markerBorderThickness = 1;

                    } else {
                        dataPoint.z = null;
                    }
                    dataPoint.legendText = dataPoint.legendLabel;
                });

                if (xyChart && x.unclean) {
                    _.each(dataPoints, function (dataPoint, n) {
                        dataPoint.x = n + 1;
                    });
                }

                if (xyChart) {
                    calcMaxExp(x);
                }
                calcMaxExp(y);

                if (dataSeries.axisXType === 'secondary') {
                    chartModel.getAxisModel('x2').setFormat(x);
                } else {
                    chartModel.getAxisModel('x').setFormat(x);
                }
                if (dataSeries.axisYType === 'secondary') {
                    chartModel.getAxisModel('y2').setFormat(y);
                } else {
                    chartModel.getAxisModel('y').setFormat(y);
                }
            });

            var chartType = chartModel.getChartType();
            var mustReverseDataPoints = false;
            if (docModel.getApp().isODF()) {
                mustReverseDataPoints = /^(pie|donut)/.test(chartType);
            }

            _.each(data.series, function (dataSeries) {
                var x = dataSeries.xInfo;
                var y = dataSeries.yInfo;
                _.each(dataSeries.dps, function (dataPoint) {
                    if (!xyChart && (stacking === 'clustered' || stacking === 'standard')) {
                        var thresholdY = 0.005;
                        //workaround that too small values stay visible
                        var rel = dataPoint.y / y.max;
                        if (Math.abs(rel) < thresholdY) {
                            if (rel < 0) {
                                dataPoint.y = -thresholdY * y.max;
                            } else {
                                dataPoint.y = thresholdY * y.max;
                            }
                        }
                    }

                    handleMantissa(y, dataPoint);
                    handleMantissa(x, dataPoint);
                });

                dataSeries.dataPoints = dataSeries.dps.slice(0);
                if (mustReverseDataPoints) {
                    dataSeries.dataPoints.reverse();
                }
            });

            data.data = data.series.slice(0);
            var mustReverse = false;
            chartModel.getModelData().legend.reversed = false;

            if (stacking === 'clustered' || stacking === 'standard') {

                chartModel.getModelData().legend.reversed = mustReverseDataPoints || /^(bar)/.test(chartModel.getChartType());

                if (docModel.getApp().isOOXML()) {
                    mustReverse = /^(bar)/.test(chartType);
                } else {
                    mustReverse = /^(bar|area)/.test(chartType);
                }
            }
            if (mustReverse) {
                data.data.reverse();
            }
        };

        this.update = function (seriesIndex, values, titles, names, bubbles) {
            var data = chartModel.getModelData();

            var dataSeries = data.series[seriesIndex];

            if (!dataSeries) {
                //workaround for Bug 46928
                Utils.warn('no dataSeries found for ' + seriesIndex + ' title: ' + titles + ' names: ' + names);
                return;
            }

            // build series title
            if (titles) { dataSeries.nameHolder = titles[0]; }
            if (!seriesIndex) {
                if (dataSeries.nameHolder && data.series.length === 1) {
                    chartModel.setMainName(dataSeries.nameHolder.display);
                } else {
                    chartModel.setMainName(null);
                }
            }
            dataSeries.dps = [];
            _.each(values, function (valueCell, n) {
                var dataPoint = {};
                dataSeries.dps.push(dataPoint);
                dataPoint.nameSource = names[n];    //x
                dataPoint.valueSource = valueCell;  //y
                dataPoint.sizeSource = bubbles[n];  //z
                dataPoint.color = null;
                dataPoint.markerColor = null;
            });
        };

        this.parseFormatCode = function (formatCode) {
            return Parser.parseFormatCode(fileFormat, 'op', formatCode);
        };

        // initialization -----------------------------------------------------

        this.registerDestructor(function () {
            docModel = chartModel = null;
        });

    } // class ChartFormatter

    // exports ================================================================

    return BaseObject.extend({ constructor: ChartFormatter });

});
