/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/namemixin', [
    'io.ox/office/spreadsheet/utils/sheetutils',
    'io.ox/office/spreadsheet/model/formula/tokenarray',
    'io.ox/office/spreadsheet/view/dialogs'
], function (SheetUtils, TokenArray, Dialogs) {

    'use strict';

    // convenience shortcuts
    var Range = SheetUtils.Range;

    // mix-in class NameMixin =================================================

    /**
     * Implementations of all controller items for manipulating defined names
     * in the document, or in the active sheet, intended to be mixed into a
     * document controller instance.
     *
     * @constructor
     *
     * @param {SpreadsheetView} docView
     *  The document view providing view settings such as the current cell
     *  selection.
     */
    function NameMixin(docView) {

        // the document model
        var docModel = docView.getDocModel();

        // index of the active sheet
        var activeSheet = -1;

        // private methods ----------------------------------------------------

        /**
         * Inserts a new defined name into the document.
         *
         * @param {String} label
         *  The label of the new defined name.
         *
         * @param {String} formula
         *  The formula expression to be bound to the new defined name.
         *
         * @param {Number|Null} sheet
         *  The index of the sheet the new defined name shall be inserted into,
         *  or null to create a globally-defined name.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the defined name has been
         *  created successfully, or that will be rejected on any error.
         */
        function insertName(label, formula, sheet) {

            // the parent model for the new defined name (sheet or document)
            var parentModel = (typeof sheet === 'number') ? docModel.getSheetModel(sheet) : docModel;
            // the descriptor for the formula expression
            var formulaDesc = {
                grammarId: 'ui',
                formula: formula,
                refSheet: activeSheet,
                refAddress: docView.getActiveCell()
            };

            // create and apply the operations to create the defined name
            var promise = docModel.createAndApplySheetOperations(function (generator) {
                return parentModel.getNameCollection().generateInsertNameOperations(generator, label, formulaDesc);
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        /**
         * Shows a dialog that allows to create a new defined name for the
         * document. The dialog will be kept open until a valid label and
         * formula expression has been entered for the defined name.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved, after the
         *  defined name has been created successfully; or will be rejected, if
         *  the dialog has been canceled, or if the document has switched to
         *  read-only mode.
         */
        function showInsertNameDialog() {

            // leave text edit mode before creating a name
            return docView.leaveTextEditMode().then(function () {

                // create a formula from the current selection (bug 40477: use start address of single merged range)
                var tokenArray = new TokenArray(docModel, 'name');
                if (docView.isSingleCellSelection()) {
                    tokenArray.appendRange(new Range(docView.getActiveCell()), { sheet: activeSheet, abs: true });
                } else {
                    tokenArray.appendRangeList(docView.getSelectedRanges(), { sheet: activeSheet, abs: true });
                }
                var formula = tokenArray.getFormula('ui');

                // create and show the dialog with empty input fields (inserting the
                // defined name will be done by the callback passed to the dialog)
                var dialog = new Dialogs.DefinedNameDialog(docView, insertName, { formula: formula });
                return dialog.show();
            });
        }

        /**
         * Changes the label or formula definition of an existing defined name.
         *
         * @param {NameModel} nameModel
         *  The model of the defined name to be changed.
         *
         * @param {String|Null} newLabel
         *  The new label for the defined name. If set to null, the label of
         *  the defined name will not be changed.
         *
         * @param {String|Null} newFormula
         *  The new formula expression to be bound to the defined name. If set
         *  to null, the formula expression of the defined name will not be
         *  changed (an invalid formula expression in the defined name will be
         *  retained).
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the defined name has been
         *  changed successfully, or that will be rejected on any error.
         */
        function changeName(nameModel, newLabel, newFormula) {

            // the descriptor for the new formula expression
            var formulaDesc = (typeof newFormula === 'string') ? {
                grammarId: 'ui',
                formula: newFormula,
                refSheet: activeSheet,
                refAddress: docView.getActiveCell()
            } : null;

            // create and apply the operations to change the defined name
            var promise = docModel.createAndApplySheetOperations(function (generator) {
                return nameModel.generateChangeOperations(generator, newLabel, formulaDesc);
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        /**
         * Shows a dialog that allows to change an existing defined name in the
         * document. The dialog will be kept open until a valid label and
         * formula expression has been entered.
         *
         * @param {NameModel} nameModel
         *  The model of the defined name to be changed.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the dialog. Will be resolved, after the
         *  defined name has been changed successfully; or will be rejected, if
         *  the dialog has been canceled, or if the document has switched to
         *  read-only mode.
         */
        function showChangeNameDialog(nameModel) {

            // leave text edit mode before changing a defined name
            return docView.leaveTextEditMode().then(function () {

                // the current label of the name
                var label = nameModel.getLabel();
                // resolve the formula expression relative to the active cell
                var formula = nameModel.getFormula('ui', docView.getActiveCell());
                // the sheet index for sheet-local names
                var sheet = nameModel.getSheetIndex();

                // the callback function invoked from the dialog after pressing the OK button
                function actionHandler(newLabel, newFormula) {
                    return changeName(nameModel, (label !== newLabel) ? newLabel : null, (formula !== newFormula) ? newFormula : null);
                }

                // create and show the dialog with empty input fields (changing the
                // defined name will be done by the callback passed to the dialog)
                var dialog = new Dialogs.DefinedNameDialog(docView, actionHandler, { label: label, formula: formula, sheet: sheet, change: true });
                return dialog.show();
            });
        }

        /**
         * Deletes a defined name from the document.
         *
         * @param {NameModel} nameModel
         *  The model of the defined name to be deleted.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the defined name has been
         *  deleted successfully, or that will be rejected on any error.
         */
        function deleteName(nameModel) {

            // create and apply the operations to delete the defined name
            var promise = docModel.createAndApplySheetOperations(function (generator) {
                return nameModel.generateDeleteOperations(generator);
            });

            // show warning alert if necessary
            return docView.yellOnFailure(promise);
        }

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            // toggles the visibility of the 'Defined Names' floating menu
            'view/namesmenu/toggle': {
                parent: 'document/editable/cell',
                get: function () { return docView.getDefinedNamesMenu().isVisible(); },
                set: function (state) { docView.getDefinedNamesMenu().toggle(state); }
            },

            'name/insert/dialog': {
                parent: 'sheet/operation/unlocked/cell',
                set: showInsertNameDialog
            },

            'name/edit/dialog': {
                parent: 'sheet/operation/unlocked/cell',
                set: showChangeNameDialog
            },

            'name/delete': {
                parent: 'sheet/operation/unlocked/cell',
                set: deleteName
            }
        });

        // initialization -----------------------------------------------------

        // initialize sheet-dependent class members according to the active sheet
        this.listenTo(docView, 'change:activesheet', function (event, sheet) {
            activeSheet = sheet;
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docView = docModel = null;
        });

    } // class NameMixin

    // exports ================================================================

    return NameMixin;

});
