/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/spreadsheet/controller/controller', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/mixedborder',
    'io.ox/office/editframework/app/editcontroller',
    'io.ox/office/spreadsheet/utils/paneutils',
    'io.ox/office/spreadsheet/app/drawingcontroller',
    'io.ox/office/spreadsheet/controller/viewmixin',
    'io.ox/office/spreadsheet/controller/searchmixin',
    'io.ox/office/spreadsheet/controller/sheetmixin',
    'io.ox/office/spreadsheet/controller/sortmixin',
    'io.ox/office/spreadsheet/controller/namemixin',
    'io.ox/office/spreadsheet/controller/tablemixin',
    'io.ox/office/spreadsheet/controller/drawingmixin'
], function (Utils, MixedBorder, EditController, PaneUtils, DrawingControllerMixin, ViewMixin, SearchMixin, SheetMixin, SortMixin, NameMixin, TableMixin, DrawingMixin) {

    'use strict';

    // class SpreadsheetController ============================================

    /**
     * The controller of a spreadsheet application.
     *
     * @constructor
     *
     * @extends EditController
     * @extends DrawingControllerMixin
     */
    var SpreadsheetController = EditController.extend({ constructor: function (app, docModel, docView) {

        // self reference
        var self = this;

        // base constructor ---------------------------------------------------

        EditController.call(this, app, docModel, docView, searchHandler);
        DrawingControllerMixin.call(this, app, docModel, docView);
        ViewMixin.call(this, docView);
        SearchMixin.call(this, docView);
        SheetMixin.call(this, docView);
        SortMixin.call(this, docView);
        NameMixin.call(this, docView);
        TableMixin.call(this, docView);
        DrawingMixin.call(this, docView);

        // private methods ----------------------------------------------------

        /**
         * Executes a search/replace operation, as requested by the base class
         * EditController.
         */
        function searchHandler(command, settings) {
            return self.executeSearchCommand(command, settings);
        }

        /**
         * Returns the merged formatting attributes of the current selection.
         * According to the current selection mode and text edit mode, the
         * following attributes will be returned:
         *  (1) Cell selection mode: The attribute set of the active cell.
         *  (2) Cell edit mode: The attribute set of the edited cell, including
         *      pending edit attributes.
         *  (3) Drawing selection mode: The mixed attribute set of all selected
         *      drawing objects with text contents.
         *  (4) Drawing edit mode: The mixed attribute set of the current text
         *      selection in the edited drawing object.
         *
         * @returns {Object}
         *  The merged formatting attributes of the current selection.
         */
        function getActiveAttributeSet() {

            // return the formatting attributes of the active text editor
            var textEditor = docView.getActiveTextEditor();
            if (textEditor) { return textEditor.getAttributeSet(); }

            // selected drawings: return the mixed attributes of the text framework selection
            if (docView.hasDrawingSelection()) {
                return docView.getTextAttributeSet();
            }

            // otherwise, return attribute set of the active cell
            return docView.getCellCollection().getAttributeSet(docView.getActiveCell());
        }

        /**
         * Changes the formatting attributes of the current selection.
         * According to the current selection mode and text edit mode, the
         * following attributes will be changed:
         *  (1) Cell selection mode: All selected cells.
         *  (2) Cell edit mode: The pending edit attributes of the edited cell.
         *  (3) Drawing selection mode: All selected drawing objects with text
         *      contents.
         *  (4) Drawing edit mode: The current text selection in the edited
         *      drawing object.
         *
         * @param {Object} attributeSet
         *  An incomplete attribute set that will be applied to the current
         *  selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the passed attributes have
         *  been applied successfully.
         */
        function setAttributeSet(attributeSet) {

            // send the formatting attributes to the active text editor
            var textEditor = docView.getActiveTextEditor();
            if (textEditor) { return textEditor.setAttributeSet(attributeSet); }

            // selected drawings: send the attributes to the text framework
            if (docView.hasDrawingSelection()) {
                return docView.setTextAttributeSet(attributeSet);
            }

            // otherwise, fill the entire cell selection with the attributes
            return docView.fillRangeContents({ a: attributeSet });
        }

        /**
         * Changes the character formatting of the current selection. See
         * method SpreadsheetController.setAttributeSet() for details.
         *
         * @param {Object} charAttrs
         *  The character attributes (simple key/value map) that will be
         *  applied to the current selection.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved when the passed attributes have
         *  been applied successfully.
         */
        function setCharacterAttributes(charAttrs) {
            return setAttributeSet({ character: charAttrs });
        }

        /**
         * Returns the list style identifier for the passed mixed paragraph
         * attributes.
         *
         * @param {String} listType
         *  The list type, either 'bullet' or 'numbering'.
         *
         * @param {Object} paraAttrs
         *  The mixed paragraph attributes.
         *
         * @returns {String|Null}
         *  The list style identifier for the passed paragraph attributes.
         */
        function getListStyleId(listType, paraAttrs) {
            return _.isEmpty(paraAttrs) ? '' : docModel.getEffectiveListStyleId(listType, paraAttrs);
        }

        // item registration --------------------------------------------------

        // register all controller items
        this.registerDefinitions({

            // selection mode -------------------------------------------------

            // enabled if the document is editable, and cell selection mode is active (no drawing objects are selected)
            'document/editable/cell': {
                parent: ['document/editable/worksheet', 'view/selection/cell']
            },

            // enabled if the document is in read-only state, and cell selection mode is active (no drawing objects are selected)
            'document/readonly/cell': {
                parent: ['document/readonly/worksheet', 'view/selection/cell']
            },

            // enabled if the document is editable, and at least one drawing object is selected;
            // item value is the drawing selection (array of drawing positions)
            'document/editable/drawing': {
                parent: ['document/editable/worksheet', 'view/selection/drawing'],
                get: function () { return docView.getSelectedDrawings(); }
            },

            // sheet operations -----------------------------------------------

            // Parent item for various sheet-level operations in the active sheet, enabled if:
            // - document is editable,
            // - cell selection (no drawing selection),
            // - text edit mode is not active.
            // Note that the active sheet may be locked.
            'sheet/operation/cell': {
                parent: ['document/editable/cell', 'view/editmode/off']
            },

            // Parent item for various sheet-level operations in the active sheet. Works similar
            // to 'sheet/operation/cell', but the active sheet must be unlocked to enable this item.
            'sheet/operation/unlocked/cell': {
                parent: 'sheet/operation/cell',
                enable: function () { return !docView.isSheetLocked(); }
            },

            // hyperlink operations -------------------------------------------

            'hyperlink/edit/dialog': {
                parent: 'sheet/operation/unlocked/cell',
                set: function () { return docView.editHyperlink(); }
            },

            'hyperlink/delete': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return _.isString(docView.getCellURL()); },
                set: function () { return docView.deleteHyperlink(); }
            },

            // column operations ----------------------------------------------

            // parent item to get the mixed column attributes
            'column/attributes/mixed': {
                get: function () { return docView.getColumnAttributes(); }
            },

            // parent item to get the column settings of the active cell
            'column/active': {
                get: function () { return docView.getActiveColumnDescriptor(); }
            },

            'column/insert': {
                parent: 'sheet/operation/unlocked/cell',
                set: function () { return docView.insertColumns(); }
            },

            'column/delete': {
                parent: 'sheet/operation/unlocked/cell',
                set: function () { return docView.deleteColumns(); }
            },

            'column/hide': {
                parent: ['column/attributes/mixed', 'sheet/operation/unlocked/cell'],
                // enabled when any visible columns are selected
                enable: function (attributes) { return attributes.visible !== false; },
                set: function () { return docView.changeColumns({ visible: false }); }
            },

            'column/show': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.canShowColumns(); },
                set: function () { return docView.showColumns(); }
            },

            'column/width/optimal': {
                parent: 'column/hide', // only enabled when any visible columns (that CAN BE hidden) are selected
                set: function () { return docView.setOptimalColumnWidth(); }
            },

            'column/width/active': {
                parent: ['column/active', 'sheet/operation/unlocked/cell'],
                get: function (colDesc) { return colDesc.sizeHmm; },
                set: function (value) { return docView.setColumnWidth(value); }
            },

            // row operations -------------------------------------------------

            // parent item to get the mixed row attributes
            'row/attributes/mixed': {
                get: function () { return docView.getRowAttributes(); }
            },

            // parent item to get the row settings of the active cell
            'row/active': {
                get: function () { return docView.getActiveRowDescriptor(); }
            },

            'row/insert': {
                parent: 'sheet/operation/unlocked/cell',
                set: function () { return docView.insertRows(); }
            },

            'row/delete': {
                parent: 'sheet/operation/unlocked/cell',
                set: function () { return docView.deleteRows(); }
            },

            'row/hide': {
                parent: ['row/attributes/mixed', 'sheet/operation/unlocked/cell'],
                // enabled when any visible rows are selected
                enable: function (attributes) { return attributes.visible !== false; },
                set: function () { return docView.changeRows({ visible: false }); }
            },

            'row/show': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.canShowRows(); },
                set: function () { return docView.showRows(); }
            },

            'row/height/optimal': {
                parent: 'row/hide', // only enabled when any visible rows (that CAN BE hidden) are selected
                set: function () { return docView.setOptimalRowHeight(); }
            },

            'row/height/active': {
                parent: ['row/active', 'sheet/operation/unlocked/cell'],
                get: function (rowDesc) { return rowDesc.sizeHmm; },
                set: function (value) { return docView.setRowHeight(value); }
            },

            // cell operations on active cell ---------------------------------

            // inserts ready-to-use cell contents into the active cell
            'cell/active/contents': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function (contents) { return docView.setCellContents(contents); }
            },

            // inserts some text (to be parsed) into the active cell
            'cell/active/parse': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function (value) { return docView.parseCellContents(value); }
            },

            'function/insert/dialog': {
                parent: 'document/editable/cell', // can be used in locked sheets
                set: function () { return docView.showInsertFunctionDialog(); }
            },

            'cell/painter': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                get: function () { return docView.isFormatPainterActive(); },
                set: function (state) { return docView.activateFormatPainter(state); }
            },

            // cell operations on selection -----------------------------------

            'cell/fill': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function (contents) { return docView.fillRangeContents(contents); }
            },

            'cell/clear/values': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function () { return docView.fillRangeContents({ v: null, f: null }); }
            },

            'cell/clear/all': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function () { return docView.clearRangeContents(); }
            },

            'cell/merge': {
                parent: 'sheet/operation/unlocked/cell',
                enable: function () { return docView.hasAnyRangeSelected(); },
                get: function () { return docView.hasMergedRangeSelected(); },
                set: function (mergeMode) { return docView.mergeRanges(mergeMode); }
            },

            'cell/autofill': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                set: function (fillData) { return docView.autoFill(fillData.border, fillData.count, { invertType: fillData.invertType }); }
            },

            'cell/autoformula': {
                parent: 'document/editable/cell', // can be used in locked sheets
                enable: function () { return docView.getSelectedRanges().length === 1; },
                set: function (funcName) { return docView.insertAutoFormula(funcName); }
            },

            // formatting attributes ------------------------------------------

            // Parent iten for the attribute set of the current selection:
            // (1) Cell selection mode: The attribute set of the active cell.
            // (2) Cell edit mode: The attribute set of the edited cell (including pending edit attributes).
            // (3) Drawing selection mode: The mixed attribute set of all drawing objects with text contents.
            // (4) Drawing edit mode: The mixed attribute set of the text selection in the edited drawing object.
            'document/attributes': {
                parent: 'document/editable/worksheet',
                enable: function () { return this.getValue() !== null; },
                get: getActiveAttributeSet
            },

            // cell attributes ------------------------------------------------

            'cell/attributes': {
                parent: 'document/editable/cell',
                get: function () { return docView.getCellCollection().getAttributeSet(docView.getActiveCell()); },
                set: function (attributes) { return docView.fillRangeContents({ a: attributes }); }
            },

            'cell/stylesheet': {
                parent: ['cell/attributes', 'view/editmode/off'],
                get: function (attributes) { return attributes.styleId; },
                set: function (styleId) { return docView.fillRangeContents({ a: { styleId: styleId } }); }
            },

            // reset all explicit attributes in the selection (item key predetermined by text framework!)
            'character/reset': {
                parent: ['cell/attributes', 'view/editmode/off'],
                set: function () { return docView.fillRangeContents({ s: '' }); }
            },

            // cell attributes (family 'cell') ----------------------------

            'cell/attributes/cell': {
                parent: ['cell/attributes', 'view/editmode/off'],
                get: function (attributes) { return attributes.cell || {}; }
            },

            'cell/fillcolor': {
                parent: 'cell/attributes/cell',
                get: function (attributes) { return attributes.fillColor; },
                set: function (color) { return docView.setCellAttributes({ fillType: 'solid', fillColor: color }); }
            },

            'cell/linebreak': {
                parent: 'cell/attributes/cell',
                get: function (attributes) { return attributes.wrapText; },
                set: function (wrap) { return docView.setCellAttributes({ wrapText: wrap }); }
            },

            'cell/unlocked': {
                parent: 'cell/attributes/cell',
                enable: function (attributes) { return !attributes.unlocked && !docView.isSheetLocked(); },
                set: function () { return docView.setCellAttributes({ unlocked: true }); }
            },

            'cell/locked': {
                parent: 'cell/attributes/cell',
                enable: function (attributes) { return attributes.unlocked && !docView.isSheetLocked(); },
                set: function () { return docView.setCellAttributes({ unlocked: false }); }
            },

            'cell/unhideformula': {
                parent: 'cell/attributes/cell',
                enable: function (attributes) { return attributes.hidden && !docView.isSheetLocked(); },
                set: function () { return docView.setCellAttributes({ hidden: false }); }
            },

            'cell/hideformula': {
                parent: 'cell/attributes/cell',
                enable: function (attributes) { return !attributes.hidden && !docView.isSheetLocked(); },
                set: function () { return docView.setCellAttributes({ hidden: true }); }
            },

            'cell/alignhor': {
                parent: 'cell/attributes/cell',
                get: function (attributes) { return attributes.alignHor; },
                set: function (alignment) { return docView.setCellAttributes({ alignHor: alignment }); }
            },

            'cell/alignvert': {
                parent: 'cell/attributes/cell',
                get: function (attributes) { return attributes.alignVert; },
                set: function (alignment) { return docView.setCellAttributes({ alignVert: alignment }); }
            },

            // border mode, as Boolean map with position keys (left, top, ...)
            'cell/border/mode': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                get: function () { return docView.getBorderMode(); },
                set: function (borderMode) { return docView.setBorderMode(borderMode); }
            },

            // all mixed border attributes of the selection
            'cell/border/attributes': {
                parent: 'document/editable/cell',
                get: function () { return docView.getBorderAttributes(); }
            },

            // enabled if at least one border in the selection is visible
            'cell/border/visible': {
                parent: 'cell/border/attributes',
                enable: function (borderAttributes) { return _.any(borderAttributes, MixedBorder.isVisibleBorder); }
            },

            // a single mixed border for all borders in the selection
            'cell/border/value': {
                parent: ['cell/border/visible', 'view/editmode/off'],
                get: function (borderAttributes) { return MixedBorder.mixBorders(_.values(borderAttributes)); }
            },

            'cell/border/style': {
                parent: 'cell/border/value',
                get: function (border) { return border.style; },
                set: function (value) { return docView.changeVisibleBorders({ style: value }); }
            },

            // width of the border lines, in points (rounded to 1/2 points)
            'cell/border/width': {
                parent: 'cell/border/value',
                get: function (border) { return !MixedBorder.isVisibleBorder(border) ? 0 : _.isNumber(border.width) ? Utils.convertHmmToLength(border.width, 'pt', 0.5) : null; },
                set: function (value) { return docView.changeVisibleBorders({ width: Utils.convertLengthToHmm(value, 'pt') }); }
            },

            'cell/border/color': {
                parent: 'cell/border/value',
                get: function (border) { return border.color; },
                set: function (value) { return docView.changeVisibleBorders({ color: value }); }
            },

            // combined style and width as enumeration as used by Excel
            'cell/border/style/preset': {
                parent: 'cell/border/value',
                get: function (border) { return PaneUtils.getPresetStyleForBorder(border); },
                set: function (style) { return docView.changeVisibleBorders(PaneUtils.getBorderForPresetStyle(style)); }
            },

            'cell/numberformat/category': {
                parent: 'sheet/operation/cell', // can be used in locked sheets
                get: function () { return docView.getNumberFormatCategory(); },
                set: function (category) { return docView.setNumberFormatCategory(category); },
                shortcut: [
                    { keyCode: '1', shift: true, ctrlOrMeta: true, value: 'number' },
                    { keyCode: '2', shift: true, ctrlOrMeta: true, value: 'scientific' },
                    { keyCode: '3', shift: true, ctrlOrMeta: true, value: 'date' },
                    { keyCode: '4', shift: true, ctrlOrMeta: true, value: 'currency' },
                    { keyCode: '5', shift: true, ctrlOrMeta: true, value: 'percent' },
                    { keyCode: '6', shift: true, ctrlOrMeta: true, value: 'standard' }
                ]
            },

            'cell/numberformat/code': {
                parent: 'cell/numberformat/category',
                enable: function (category) { return !/^(standard|custom)$/.test(category); },
                get: function () { return docView.getNumberFormat(); },
                set: function (format) { return docView.setNumberFormat(format); }
            },

            // cell attributes (family 'character') ---------------------------

            'document/attributes/character': {
                parent: 'document/attributes',
                get: function (attributes) { return attributes.character || {}; }
            },

            'character/bold': {
                parent: 'document/attributes/character',
                get: function (attributes) { return attributes.bold; },
                set: function (state) { return setCharacterAttributes({ bold: state }); },
                shortcut: { keyCode: 'B', ctrlOrMeta: true, value: function (state) { return !state; } }
            },

            'character/italic': {
                parent: 'document/attributes/character',
                get: function (attributes) { return attributes.italic; },
                set: function (state) { return setCharacterAttributes({ italic: state }); },
                shortcut: { keyCode: 'I', ctrlOrMeta: true, value: function (state) { return !state; } }
            },

            'character/underline': {
                parent: 'document/attributes/character',
                get: function (attributes) { return attributes.underline; },
                set: function (state) { return setCharacterAttributes({ underline: state }); },
                shortcut: { keyCode: 'U', ctrlOrMeta: true, value: function (state) { return !state; } }
            },

            'character/strike': {
                parent: 'document/attributes/character',
                get: function (attributes) { return _.isString(attributes.strike) ? (attributes.strike !== 'none') : null; },
                set: function (state) { return setCharacterAttributes({ strike: state ? 'single' : 'none' }); }
            },

            'character/color': {
                parent: 'document/attributes/character',
                get: function (attributes) { return attributes.color; },
                set: function (color) { return setCharacterAttributes({ color: color }); }
            },

            'character/fillcolor': {
                parent: 'document/attributes/character',
                enable: _.constant(false) // TODO
            },

            'character/fontname': {
                parent: 'document/attributes/character',
                get: function (attributes) { return attributes.fontName; },
                set: function (fontName) { return setCharacterAttributes({ fontName: fontName }); }
            },

            'character/fontsize': {
                parent: 'document/attributes/character',
                get: function (attributes) { return attributes.fontSize; },
                set: function (fontSize) { return setCharacterAttributes({ fontSize: fontSize }); }
            },

            'character/vertalign': {
                parent: 'document/attributes/character',
                enable: _.constant(false) // TODO
            },

            // paragraph attributes (family 'paragraph' in shape text) --------

            'document/attributes/paragraphs': {
                parent: ['document/attributes', 'view/selection/drawing'],
                get: function (attributes) { return attributes.paragraph || {}; }
            },

            'paragraph/alignment': {
                parent: 'document/attributes/paragraphs',
                get: function (attributes) { return attributes.alignment; },
                set: function (alignment) { docModel.setAttribute('paragraph', 'alignment', alignment); }
            },

            'paragraph/lineheight': {
                parent: 'document/attributes/paragraphs',
                get: function (attributes) { return attributes.lineHeight; },
                set: function (lineHeight) { docModel.setAttribute('paragraph', 'lineHeight', lineHeight); }
            },

            'paragraph/spacing': {
                parent: 'document/attributes/paragraphs',
                get: function (attributes) { return docModel.getParagraphSpacing(attributes); },
                set: function (multiplier) { return docModel.setParagraphSpacing(multiplier); }
            },

            // toggle default bullet list, or select bullet type
            'paragraph/list/bullet': {
                parent: 'document/attributes/paragraphs',
                enable: function () { return !app.isODF(); },
                get: function (attributes) { return getListStyleId('bullet', attributes); },
                set: function (value) { docModel.setListStyleId('bullet', value, this.getValue()); }
            },

            // toggle default numbered list, or select numbering type
            'paragraph/list/numbered': {
                parent: 'document/attributes/paragraphs',
                enable: function () { return !app.isODF(); },
                get: function (attributes) { return getListStyleId('numbering', attributes); },
                set: function (value) { docModel.setListStyleId('numbering', value, this.getValue()); }
            },

            // increase list level by one
            'paragraph/list/incindent': {
                parent: 'document/attributes/paragraphs',
                enable: function (paraAttrs) { return !app.isODF() && docModel.isListIndentChangeable(paraAttrs, { increase: true }); },
                set: function () { docModel.changeListIndent({ increase: true, validatedLevel: true }); }
            },

            // decrease list level by one
            'paragraph/list/decindent': {
                parent: 'document/attributes/paragraphs',
                enable: function (paraAttrs) { return !app.isODF() && docModel.isListIndentChangeable(paraAttrs, { increase: false }); },
                set: function () { docModel.changeListIndent({ increase: false, validatedLevel: true }); }
            }
        });

        // initialization -----------------------------------------------------

        // update GUI after changed text selection in drawing objects
        this._updateOnEvent(docModel, 'selection');

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = app = docModel = docView = null;
        });

    } }); // class SpreadsheetController

    // exports ================================================================

    return SpreadsheetController;

});
