/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Ingo Schmidt-Rosbiegal <ingo.schmidt-rosbiegal@open-xchange.com>
 */

define('io.ox/office/presentation/format/slidestyles', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/textframework/utils/dom',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/gradient',
    'io.ox/office/editframework/utils/pattern',
    'io.ox/office/editframework/utils/texture',
    'io.ox/office/editframework/model/stylecollection',
    'gettext!io.ox/office/presentation/main'
], function (Utils, Canvas, DOM, AttributeUtils, Gradient, Pattern, Texture, StyleCollection, gt) {

    'use strict';

    // definitions for slide attributes
    var DEFINITIONS = {

        /**
         * Whether the drawings from layout or master slide shall be visible.
         */
        followMasterShapes: { def: true },

        /**
         * Whether the date place holder will be inherited.
         */
        isDate: { def: false },

        /**
         * Whether the header place holder will be inherited.
         */
        isFooter: { def: false },

        /**
         * Whether the footer place holder will be inherited.
         */
        isHeader: { def: false },
        /**
         * Whether the slide number place holder will be inherited.
         */
        isSlideNum: { def: false },

        /**
         * Whether the drawings from layout or master slide shall be visible.
         */
        hidden: { def: false },

        /**
         * The type of a layout slide
         */
        type: { def: '' },

        /**
         * The fixed text in ODF placeholder drawings of type 'ftr'
         */
        footerText: { def: '' },

        /**
         * The fixed text in ODF placeholder drawings of type 'dt'
         */
        dateText: { def: '' },

        /**
         * The field text in ODF placeholder drawings of type 'dt'
         */
        dateField: { def: '' }
    };

    // Info for the properties 'isDate', 'isFooter', 'isHeader' and 'isSildeNum'.
    // In OOXML file there is an 'hf' element. If this does not exist, all values are set
    // to false. This is the default that the client uses. But if there is at least one child
    // (for example 'isFooter') set explicitely to 'false', all other (not defined) children
    // are set automatically to true (the default changes). This needs to be handled by the
    // filter for the loading operations of the document. If the client changes one of these
    // values in the UI, all four values need to be set.

    // class SlideStyles =======================================================

    /**
     * Contains the style sheets for page formatting attributes. The CSS
     * formatting will be read from and written to the slide container elements.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {PresentationModel} docModel
     *  The presentation document model containing this instance.
     */
    var SlideStyles = StyleCollection.extend({ constructor: function (docModel) {

        // SlideIds with background image failures, with a boolean flag which indicates if a notification for this slide was displayed
        var backgroundImageFailureSlideIds = {};

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'slide', { families: 'fill' });

        // private functions --------------------------------------------------

        /**
         * Updating the slide background node with the specified attributes of the family 'fill'.
         *
         * @param {jQuery} $slide
         *  The slide element whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family,
         *  containing the effective attribute values merged from style sheets and
         *  explicit attributes.
         *
         * @param {String} slideId
         *  The ID of the slide whose background will be updated.
         *
         * @param {String} [themeSlideIdStr]
         *  The ID of the slide that determines the theme.
         *
         * @param {Object} [options]
         *  Optional object containing paramters.
         *  @param {Boolean} [options.imageLoad=false]
         *         Optional paramter indicating that updateBackground was triggered by image load in preview dialog.
         *  @param {Boolean} [options.preview=false]
         *         Optional parameter indicating that updateBackground was triggered from preview dialog.
         *  @param {Boolean} [options.onDialogOpen=false]
         *         Optional parameter indicating that updateBackground was triggered from preview dialog on initial dialog openning.
         *
         * @returns {jQuery.Promise}
         */
        function updateBackground($slide, mergedAttributes, slideId, themeSlideIdStr, options) {

            var // the fill attributes of the slide
                fillAttrs = mergedAttributes.fill,
                // the fill attributes of the slide
                inheritedFillAttrs = null,
                // the CSS properties for the slide
                cssProps = {},
                // the slide ID that is used for the theme color
                themeSlideId = themeSlideIdStr || slideId,
                // the target chain needed to resolve the current theme
                targets = docModel.getTargetChain(themeSlideId),
                // if updateBackground was triggered by image load in preview dialog
                isImageLoad = Utils.getBooleanOption(options, 'imageLoad', false),
                // if updateBackground is triggered in preview dialog
                isPreview = Utils.getBooleanOption(options, 'preview', false),
                // if updateBackground is called on dialogOpen
                calledOnDialogOpen = Utils.getBooleanOption(options, 'onDialogOpen', false),
                // slide node for which the background is updated
                livePreviewNode = null,
                // the result promise
                promise = null;

            // Helper function to create a background for a slide, that has its own theme, but no slide background specified.
            // If an underlaying layout or master slide has a specified background that also uses a theme, it it necessary, to
            // update this background, with the theme of the current slide.
            // -> this problem should be solved by reformatting the background of the underlying slide, but such a process
            //    is currently not supported by the slide pane. In this case, there is more than one 'state' for the underlying
            //    slide dependent from the theme of the visible slide.
            // -> in the current solution, the visible slide gets a slide background with correct theming, although no slide
            //    background is specified for this slide.
            function checkInheritedFillAttrs() {
                if (!docModel.isMasterSlideId(slideId) && docModel.hasDefaultTheme(slideId)) {
                    var backgroundSlideId = docModel.getMostTopBackgroundSlide(_.rest(targets));
                    if (backgroundSlideId) {
                        inheritedFillAttrs = docModel.getSlideAttributesByFamily(backgroundSlideId, 'fill');
                        if (AttributeUtils.isFillThemed(inheritedFillAttrs)) {
                            fillAttrs = _.copy(inheritedFillAttrs, true);
                        }
                    }
                }
            }

            // special handling for task 48471 -> setting new fill attrs, if the slide has no background, but a default theme registered.
            if (fillAttrs.type === 'none') { checkInheritedFillAttrs(); }

            // calculate the CSS fill attributes
            switch (fillAttrs.type) {
                case 'none':
                case null:
                    // clear everything: color and bitmaps
                    cssProps.background = '';
                    break;

                case 'solid':
                    cssProps.background = docModel.getCssColor(fillAttrs.color, 'fill', targets);
                    break;

                case 'gradient':
                    var
                        elmSlide  = $slide[0],
                        gradient  = Gradient.create(Gradient.getDescriptorFromAttributes(fillAttrs)),
                        dataUrl   = Gradient.parseImageDataUrl(gradient, docModel, { width: elmSlide.offsetWidth, height: elmSlide.offsetHeight }, targets);

                    cssProps.background = 'url("' + dataUrl + '")';
                    break;

                case 'pattern':
                    cssProps.background = 'url("' + Pattern.createDataURL(fillAttrs.pattern, fillAttrs.color2, fillAttrs.color, docModel.getThemeModel(targets)) + '")';
                    break;

                case 'bitmap':
                    if (!fillAttrs.bitmap) { return $.Deferred().reject(); }

                    if (isPreview && fillAttrs.bitmap.transparency && _.isNumber(fillAttrs.bitmap.transparency)) {
                        cssProps.opacity = 1 - fillAttrs.bitmap.transparency;
                    }
                    if (!isPreview || isImageLoad) {
                        if (calledOnDialogOpen) {
                            livePreviewNode = $slide.parent().children(DOM.LIVE_PREVIEW_SLIDE_BACKGROUND_SELECTOR).removeClass(DOM.LIVE_PREVIEW_SLIDE_BACKGROUND_CLASS);
                        }
                        promise = generateImageURLfromCanvas(fillAttrs, isPreview, slideId).then(function (canvasUrl) {
                            cssProps.background = 'url("' + canvasUrl + '")';
                            $slide.css(cssProps);
                            if (calledOnDialogOpen) {
                                livePreviewNode.addClass(DOM.LIVE_PREVIEW_SLIDE_BACKGROUND_CLASS);
                            }
                        });
                    }
                    break;

                default:
                    Utils.warn('SlideStyles.setSlideBackground(): unknown fill type "' + fillAttrs.type + '"');
            }

            // taking care of helper node for background preview during dialog open
            if (!isPreview) {
                docModel.removeHelpBackgroundNode($slide);
            }
            // apply the fill attributes
            $slide.css(cssProps);

            return promise || $.when();
        }

        /**
         * Helper method for loading image in canvas, make texture tiling, and returning edited image as base64 png image url.
         *
         * @param  {Object} fillAttrs
         *   fill attributes
         * @return {jQuery.Promise|String}
         *   String base64 png image url, if the promise is resolved.
         */
        function generateImageURLfromCanvas(fillAttrs, isPreview, slideId) {
            var slideSize = docModel.getSlideDocumentSize();
            var slideW = Utils.convertHmmToLength(slideSize.width, 'px', 1);
            var slideH = Utils.convertHmmToLength(slideSize.height, 'px', 1);
            var tempCanvas = new Canvas(docModel).initialize({ width: slideW, height: slideH });

            return tempCanvas.render(function (tempCtx) {
                var promise = Texture.getTextureFill(docModel, tempCtx, fillAttrs, slideW, slideH, 0, { isSlide: true, noAlpha: isPreview });
                promise.fail(function () {
                    if (!_.isEmpty(slideId) && !(slideId in backgroundImageFailureSlideIds)) {
                        backgroundImageFailureSlideIds[slideId] = true;
                        showBackgroundImageFailureNotificationDebounced();
                    }
                });
                return promise.then(function (textureFill) {
                    tempCtx.setFillStyle(textureFill).drawRect(0, 0, slideW, slideH, 'fill');
                    return tempCanvas.getDataURL();
                });
            });
        }

        /**
         * Show a notification for all slides with background image failures.
         */
        function showBackgroundImageFailureNotification() {

            if (docModel.getSlideFormatManager().allSlidesFormatted()) {
                var idList = null;
                var firstId = true;
                _.each(backgroundImageFailureSlideIds, function (show, slideId) {
                    // check if a notifaction was displayed for this slide
                    if (show) {
                        var id = '';
                        if (docModel.isLayoutOrMasterId(slideId)) {
                            id = docModel.getSlideFamilyAttributeForSlide(slideId, 'slide', 'name');
                            if (_.isEmpty(id)) {
                                id = null;
                            }
                        } else {
                            id = docModel.getSortedSlideIndexById(slideId);
                            if (id >= 0) {
                                id++;
                            } else {
                                id = null;
                            }
                        }
                        if (id !== null) {
                            if (firstId) {
                                idList = '' + id;
                            } else {
                                idList += ', ' + id;
                            }
                            firstId = false;
                        }
                        backgroundImageFailureSlideIds[slideId] = false;
                    }
                });
                if (idList) {
                    docModel.getApp().getView().yell({ type: 'info', message: gt('This document contains slide background images which can\'t be displayed: %s', idList) });
                }
            } else {
                showBackgroundImageFailureNotificationDebounced();
            }
        }

        // show the Notification if slide background imgage failures exist
        var showBackgroundImageFailureNotificationDebounced = this.createDebouncedMethod('Slidestyles.showBackgroundImageFailureNotificationDebounced', null, showBackgroundImageFailureNotification, { delay: 1000 });

        /**
         * Will be called for every page whose attributes have been changed.
         *
         * @param {jQuery} slide
         *  The slide element whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family,
         *  containing the effective attribute values merged from style sheets and
         *  explicit attributes.
         *
         * @returns {jQuery.Promise}
         *  A promise that is resolved, after the slide background is set.
         */
        function updateSlideFormatting(slide, mergedAttributes) {

            var // the slide id
                slideId = docModel.getSlideId(slide),
                // getting the old 'hidden' state (the model is already updated)
                oldHiddenState = (slide.attr('isHidden') === 'true'),
                // whether the hidden state of a slide was changed
                hiddenStateChanged = (mergedAttributes.slide.hidden !== oldHiddenState),
                // a promise for handling the slide background
                backgroundPromise = null;

            // updating the slide background
            backgroundPromise = updateBackground(slide, mergedAttributes, slideId);

            // setting a marker at the slide, so that changes can be recognized
            if (hiddenStateChanged) { slide.attr('isHidden', mergedAttributes.slide.hidden); }

            // inform the view, that the merged slide attributes (including 'fill') need to be updated
            backgroundPromise.done(function () {
                docModel.trigger('change:slideAttributes', { slideId: slideId, hiddenStateChanged: hiddenStateChanged, isHidden: mergedAttributes.slide.hidden });
            });

            return backgroundPromise;
        }

        // public methods -----------------------------------------------------

        /**
         * Public accesor for updateBackground. See function updateBackground for more details.
         *
         * @param {jQuery} $slide
         *  The slide element whose attributes have been changed, as jQuery object.
         *
         * @param {Object} mergedAttributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family,
         *  containing the effective attribute values merged from style sheets and
         *  explicit attributes.
         *
         * @param {String} slideId
         *  The ID of the slide whose background will be updated.
         *
         * @param {String} themeSlideIdStr
         *  The ID of the slide that determines the theme.
         *
         * @param {Object} [options]
         *  Optional parameters
         *
         * @returns {jQuery.Promise}
         */
        this.updateBackground = function ($slide, mergedAttributes, slideId, themeSlideIdStr, options) {
            return updateBackground($slide, mergedAttributes, slideId, themeSlideIdStr, options);
        };

        /**
         * Forcing an update of slide background with a specified theme (48471).
         *
         * @param {jQuery} slide
         *  The slide element whose attributes have been changed, as jQuery object.
         *
         * @param {Object} attributes
         *  A map of attribute maps (name/value pairs), keyed by attribute family,
         *  containing the effective attribute values merged from style sheets and
         *  explicit attributes.
         *
         * @param {String} slideId
         *  The ID of the slide whose background will be updated.
         *
         * @param {String} themeId
         *  The ID of the slide that determines the theme.
         */
        this.forceSlideBackgroundUpdateWithSpecificId = function (slide, attributes, slideId, themeId) {
            updateBackground(slide, attributes, slideId, themeId);
        };

        // initialization -----------------------------------------------------

        // register the attribute definitions for the style family
        docModel.registerAttributes('slide', DEFINITIONS);

        // register the formatting handler for DOM elements
        this.registerFormatHandler(updateSlideFormatting);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = null;
        });

    } }); // class SlideStyles

    // exports ================================================================

    return SlideStyles;

});
