/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 * @author York Richter <york.richter@open-xchange.com>
 */

define('io.ox/office/editframework/view/control/tablestylepicker', [
    'io.ox/office/tk/config',
    'io.ox/office/tk/utils',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/view/control/stylesheetpicker',
    'gettext!io.ox/office/editframework/main'
], function (Config, Utils, Canvas, Border, StyleSheetPicker, gt) {

    'use strict';

    var CELL_WIDTH = 16;

    var CELL_HEIGHT = 8;

    var COL_COUNT = 5;

    var ROW_COUNT = 6;

    var ICON_WIDTH = CELL_WIDTH * COL_COUNT + 5.5;

    var ICON_HEIGHT = CELL_HEIGHT * ROW_COUNT + 5.5;

    var ICON_CONTAINER_WIDTH = ICON_WIDTH + 5.5;

    var ICON_CONTAINER_HEIGHT = ICON_HEIGHT + 5.5;

    // private global functions ===============================================

    // Comparator for the tablestyle names
    function itemComparator(btnNode1, btnNode2) {

        var priority = Utils.getOption($(btnNode1).data('options'), 'priority', 0) - Utils.getOption($(btnNode2).data('options'), 'priority', 0);
        if (priority) {
            return priority;
        }

        var style1 = $(btnNode1).attr('data-original-title');
        style1 = style1 ? style1.toLowerCase().split(/(\d+)/) : [];

        var style2 = $(btnNode2).attr('data-original-title');
        style2 = style2 ? style2.toLowerCase().split(/(\d+)/) : [];

        while (style1.length && style2.length) {
            var style1Part = style1.shift();
            var style2Part = style2.shift();
            var compareResult = (style1Part - style2Part) || style1Part.localeCompare(style2Part);
            // If it is not equal, return the compare result
            if (compareResult) {
                return compareResult;
            }
        }

        return style1.length - style2.length;
    }

    /**
     * Returns the effective rendering line width for the passed border.
     */
    function getBorderWidth(border) {
        return (border.style === 'double') ? 2 : 1;
    }

    // class TableStylePicker =================================================

    /**
     * A drop-down menu control for table style sheets.
     *
     * @constructor
     *
     * @extends StyleSheetPicker
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all options that are supported by the
     *  method TableStyleCollection.resolveCellAttributeSet() used to resolve
     *  the formatting attributes of the cells depending on their position in
     *  the table, and the following options:
     *  - {Array<Object>} [initOptions.additionalItems]
     *      Additional stylesheet items. The array contains objects with the
     *      style attributes.
     */
    function TableStylePicker(docView, initOptions) {

        // self reference
        var self = this;

        var docModel = docView.getDocModel();

        var tableStyles = docModel.getStyleCollection('table');

        // the canvas element used to render the preview tables
        var canvas = new Canvas(docView);

        // base constructors --------------------------------------------------

        StyleSheetPicker.call(this, docView, 'table', {
            width: 200,
            icon: 'docs-table-style',
            //#. tool tip: predefined styles for text tables
            tooltip: gt('Table style'),
            gridColumns: 7,
            i18nModulePath: 'io.ox/office/editframework/resource/tablestylenames',
            itemComparator: itemComparator,
            sections: ['', 'Light', 'Medium', 'Dark'],
            sectionLabel: true,
            additionalItems: Utils.getArrayOption(initOptions, 'additionalItems', null),
            smallerVersion: { css: { width: 50 }, hideLabel: true }
        });

        // private methods ----------------------------------------------------

        /**
         * Returns the border with stronger visual appearance.
         *
         * @param {Border} border1
         * The first border to compare.
         *
         * @param {Border} border2
         * The second border to compare.
         *
         * @param {String|Array<String>|Null} targets
         *  A name or an array of names to specify the active theme targets.
         *
         * @returns {Border}
         *  One of the passed border with stronger visual appearance. If one of
         *  the passed borders is invisible, the other border will be returned.
         */
        function getStrongerBorder(border1, border2, targets) {

            // handle missing borders (return the other border)
            if (!Border.isVisibleBorder(border1)) { return border2; }
            if (!Border.isVisibleBorder(border2)) { return border1; }

            // compare pixel border width
            var widthDiff = getBorderWidth(border1) - getBorderWidth(border2);
            if (widthDiff > 0) { return border1; }
            if (widthDiff < 0) { return border2; }

            // compare luma value (darker color wins!)
            var y1 = docModel.parseAndResolveColor(border1.color, 'line', targets).y;
            var y2 = docModel.parseAndResolveColor(border2.color, 'line', targets).y;
            return (y1 < y2) ? border1 : border2;
        }

        /**
         * Set the line style for the specified border.
         *
         * @param {CanvasRenderingContext2D} context
         *  The wrapped rendering context of a canvas element.
         *
         * @param {Border} border
         * to get the line style
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {Number} the with of the border
         */
        function setBorderLineStyle(context, border, targets) {
            var cssColor = docModel.getCssColor(border.color, 'line', targets);
            var lineWidth = getBorderWidth(border);
            context.setLineStyle({ style: cssColor, width: lineWidth });
            return lineWidth;
        }

        /**
         * Render a line as a text placeholder to show text color.
         *
         * @param {CanvasRenderingContext2D} context
         *  The wrapped rendering context of a canvas element.
         *
         * @param {Object} styleAttributes
         *  The style attributes of the cell.
         *
         * @param {Array<Object>} fillColors
         *  The fill color of the cell.
         *
         * @param {Number} left
         *  The left position of the cell.
         *
         * @param {Number}
         *  The top top positon of the cell.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         */
        function renderTextPlaceholder(context, styleAttributes, fillColors, left, top, targets) {
            // Paint line for the text color
            var textColor = Utils.getObjectOption(styleAttributes.character, 'color', null);
            var cssColor = docModel.parseAndResolveTextColor(textColor, fillColors, targets).css;
            context.setLineStyle({ style: cssColor, width: 1 });
            context.drawLine(left + 4, top + CELL_HEIGHT - 4.5, left + CELL_WIDTH - 4, top + CELL_HEIGHT - 4.5);
        }

        /**
         * Render Border and the text placeholder see@ renderTextPlaceholder().
         *
         * @param {CanvasRenderingContext2D} context
         *  The wrapped rendering context of a canvas element.
         * @param {Object} styleAttributes
         *  The style attributes of the cell.
         * @param {Number} col the column index of the cell.
         * @param {Number} row the row index of the cell.
         * @param {type} topAttr the style attributes of the cell above
         * @param {type} bottomAttr the style attributes of the cell bellow
         * @param {type} leftAttr the style attributes of the cell on the left side
         * @param {type} rightAttr the style attributes of the cell on the right side
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         */
        function renderBordersAndTextPlaceholder(context, style, row, col, topAttr, bottomAttr, leftAttr, rightAttr, targets) {

            var left = row * CELL_WIDTH + 2;
            var top = col * CELL_HEIGHT + 2;

            var cellStyle = style.cell;
            var fillColors = [];
            if (cellStyle) {

                if (cellStyle.fillColor) {
                    fillColors.push(cellStyle.fillColor);
                }

                var borderWidth, borderTop, borderBottom, borderLeft, border;

                if (cellStyle.borderTop && cellStyle.borderTop.style !== 'none') {
                    border = cellStyle.borderTop;
                    if (row > 0 && topAttr && topAttr.cell && topAttr.cell.borderBottom) {
                        border = getStrongerBorder(border, topAttr.cell.borderBottom, targets);
                    }
                    borderWidth = setBorderLineStyle(context, border, targets);
                    borderTop = top - borderWidth / 2;
                    context.drawLine(left, borderTop, left + CELL_WIDTH, borderTop);
                }
                if (cellStyle.borderBottom && cellStyle.borderBottom.style !== 'none') {
                    border = cellStyle.borderBottom;
                    if (row + 1 < ROW_COUNT && bottomAttr && bottomAttr.cell && bottomAttr.cell.borderTop) {
                        border = getStrongerBorder(border, bottomAttr.cell.borderTop, targets);
                    }
                    borderWidth = setBorderLineStyle(context, border, targets);
                    borderBottom = top + CELL_HEIGHT - borderWidth / 2;
                    context.drawLine(left, borderBottom, left + CELL_WIDTH, borderBottom);
                }

                if (cellStyle.borderLeft && cellStyle.borderLeft.style !== 'none') {
                    border = cellStyle.borderLeft;
                    if (col > 0 && leftAttr && leftAttr.cell && leftAttr.cell.borderRight) {
                        border = getStrongerBorder(border, leftAttr.cell.borderRight, targets);
                    }
                    borderWidth = setBorderLineStyle(context, border, targets);
                    borderTop = borderTop || (top - 0.5);
                    borderBottom = borderBottom || (top + CELL_HEIGHT + 0.5);
                    borderLeft = left - borderWidth / 2;
                    context.drawLine(borderLeft, borderTop, borderLeft, borderBottom);
                }
                if (cellStyle.borderRight && cellStyle.borderRight.style !== 'none') {
                    border = cellStyle.borderRight;
                    if (col + 1 < COL_COUNT && rightAttr && rightAttr.cell && rightAttr.cell.borderLeft) {
                        border = getStrongerBorder(border, rightAttr.cell.borderLeft, targets);
                    }
                    borderWidth = setBorderLineStyle(context, border, targets);
                    borderTop = borderTop || (top - 0.5);
                    borderBottom = borderBottom || (top + CELL_HEIGHT + 0.5);
                    borderLeft = left + CELL_WIDTH - borderWidth / 2;
                    context.drawLine(borderLeft, borderTop, borderLeft, borderBottom);
                }
            }

            renderTextPlaceholder(context, style, fillColors, left, top, targets);
        }

        /**
         * Render the fill color for the specified cell.
         *
         * @param {CanvasRenderingContext2D} context
         *  The wrapped rendering context of a canvas element.
         *
         * @param {Number} col
         *  the column index of the cell.
         *
         * @param {Number} row
         *  the row index of the cell.
         *
         * @param {Object} styleAttributes
         *  The style attributes of the cell.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         */
        function renderCellFillColor(context, col, row, styleAttributes, targets) {

            var cellAttrs = styleAttributes.cell;
            if (cellAttrs && cellAttrs.fillColor) {
                var left = col * CELL_WIDTH + 2;
                var top = row * CELL_HEIGHT + 2;
                context.setFillStyle(docModel.getCssColor(cellAttrs.fillColor, 'fill', targets));
                context.drawRect(left, top, CELL_WIDTH, CELL_HEIGHT, 'fill');
            }
        }

        /**
         * Add the color of the first cell to the Button node for selenium test.
         * @param {jQuery} node button node
         * @param {String} styleId table style id
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         */
        function addFirstCellColorForSelenium(node, styleId, targets) {
            var tableAttrs = tableStyles.getStyleSheetAttributeMap(styleId, true);
            var cellAttrs = tableStyles.resolveCellAttributeSet(tableAttrs, { start: 0, end: 0 }, { start: 0, end: 0 }, COL_COUNT, ROW_COUNT, initOptions);
            if (cellAttrs && cellAttrs.cell && cellAttrs.cell.fillColor) {
                node.css('color', docModel.getCssColor(cellAttrs.cell.fillColor, 'fill', targets));
            }
        }

        /**
         * Renders the specified table into the canvas.
         *
         * @param {Canvas} canvas
         *  the canvas for render the table.
         *
         * @param {String} styleId
         *  the id of the table style.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         */
        function renderTable(canvas, styleId, targets) {
            canvas.clear().render(function (context) {

                var tableAttrs = tableStyles.getStyleSheetAttributeMap(styleId, true);

                // paint background color and cache the cell style attributes
                var attrMatrix = _.times(ROW_COUNT, function (row) {
                    var rowRange = { start: row, end: row };
                    return _.times(COL_COUNT, function (col) {
                        var colRange = { start: col, end: col };
                        var cellAttrs = tableStyles.resolveCellAttributeSet(tableAttrs, colRange, rowRange, COL_COUNT, ROW_COUNT, initOptions);
                        renderCellFillColor(context, col, row, cellAttrs, targets);
                        return cellAttrs;
                    });
                });

                // paint the border by using the cached cell styles
                attrMatrix.forEach(function (attrVector, row) {
                    var prevVector = (row > 0) ? attrMatrix[row - 1] : null;
                    var nextVector = (row + 1 < ROW_COUNT) ? attrMatrix[row + 1] : null;
                    attrVector.forEach(function (cellAttrs, col) {
                        var tAttrs = prevVector ? prevVector[col] : null;
                        var bAttrs = nextVector ? nextVector[col] : null;
                        var lAttrs = (col > 0) ? attrVector[col - 1] : null;
                        var rAttrs = (col + 1 < COL_COUNT) ? attrVector[col + 1] : null;
                        renderBordersAndTextPlaceholder(context, cellAttrs, col, row, tAttrs, bAttrs, lAttrs, rAttrs, targets);
                    });
                });
            });
        }

        /**
         * Registers a new table that will be formatted delayed with a specific
         * table style sheet.
         *
         * @param {jQuery} table
         *  The new table to be formatted.
         *
         * @param {String} styleId
         *  The identifier of the table style sheet.
         */
        var updateTableFormatting = (function () {

            // which tables in list items need formatting
            var pendingTables = [];
            // background loop processing all pending tables
            var formatTimer = null;

            // initialize effective rectangle of the canvas wrapper (size of the DOM canvas node will be initialized too)
            canvas.initialize({ left: -1, top: -1, width: ICON_WIDTH + 2, height: ICON_HEIGHT + 2 });

            // set the CSS location of the canvas element relative to its parent element
            canvas.getNode().css({ left: -1, top: -1 });

            // direct callback: called every time when updateTableFormatting() has been called
            function registerTable(tableNode, styleId) {
                pendingTables.push({ tableNode: tableNode, styleId: styleId });
            }

            // deferred callback: called once after the specified timeout
            function formatTables() {

                // check if the background loop is already running
                if (formatTimer) { return; }

                // create a new background loop that processes all pending tables
                formatTimer = self.repeatSliced(function () {

                    // table info with table and style identifier of the next table to be formatted
                    var tableInfo = null;

                    // find a pending table that is still in the drop-down menu (menu may have been cleared
                    // in the meantime), abort the background loop, if no more tables are available
                    while ((tableInfo = pendingTables.shift()) && !Utils.containsNode(self.getMenuNode(), tableInfo.tableNode)) {}
                    if (!tableInfo) { return Utils.BREAK; }

                    var img = $(tableInfo.tableNode).find('>img');
                    $(tableInfo.tableNode).find('.caption').remove();

                    var targets = docModel.getThemeTargets();
                    renderTable(canvas, tableInfo.styleId, targets);

                    // Only for selenium test
                    if (Config.AUTOTEST) {
                        addFirstCellColorForSelenium($(tableInfo.tableNode), tableInfo.styleId, targets);
                    }

                    img.attr({ src: canvas.getDataURL() });
                }, 'TableStylePicker.updateTableFormatting.formatTables', { slice: 100 });

                // forget reference to the timer, when all tables have been formatted
                formatTimer.always(function () { formatTimer = null; });
            }

            // create and return the debounced updateTableFormatting() method (with a delay of 10ms per table)
            return self.createDebouncedMethod('TableStylePicker.updateTableFormatting', registerTable, formatTables, { delay: 10 });
        }());

        // initialization -----------------------------------------------------

        // register a handler that inserts a table element into each list item
        this.listenTo(this.getMenu(), 'create:item', function (event, buttonNode, styleId) {
            buttonNode.addClass('mini-caption').css({
                minWidth: ICON_CONTAINER_WIDTH,
                maxWidth: ICON_CONTAINER_WIDTH,
                maxHeight: ICON_CONTAINER_HEIGHT,
                padding: 0,
                lineHeight: 'normal'
            }).append('<img style="margin:0px;">');

            updateTableFormatting(buttonNode, styleId);
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            canvas.destroy();
            self = docView = docModel = tableStyles = canvas = null;
        });

    } // class TableStylePicker

    // exports ================================================================

    // derive this class from class StyleSheetPicker
    return StyleSheetPicker.extend({ constructor: TableStylePicker });

});
