/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Oliver Specht <oliver.specht@open-xchange.com>
 */

define('io.ox/office/editframework/model/themecollection', [
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/baseframework/model/modelobject',
    'io.ox/office/editframework/utils/attributeutils'
], function (ValueMap, ModelObject, AttributeUtils) {

    'use strict';

    // the name of the default theme
    var DEFAULT_THEME_NAME = 'Larissa';

    // the color scheme of the built-in default theme for OOXML documents
    var OOXML_BUILTIN_COLOR_SCHEME = {
        text1: '000000',
        text2: '1f497d',
        light1: 'ffffff',
        light2: 'eeece1',
        dark1: '000000',
        dark2: '1f497d',
        background1: 'ffffff',
        background2: 'eeece1',
        accent1: '4f81bd',
        accent2: 'c0504d',
        accent3: '9bbb59',
        accent4: '8064a2',
        accent5: '4bacc6',
        accent6: 'f79646',
        hyperlink: '0000ff',
        followedHyperlink: '800080'
    };

    // the font scheme of the built-in default theme for OOXML documents
    var OOXML_BUILTIN_FONT_SCHEME = {};
    OOXML_BUILTIN_FONT_SCHEME[AttributeUtils.MAJOR_FONT_KEY] = 'Calibri';
    OOXML_BUILTIN_FONT_SCHEME[AttributeUtils.MINOR_FONT_KEY] = 'Calibri';

    // the color scheme of the built-in default theme for ODF documents
    var ODF_BUILTIN_COLOR_SCHEME = _.extend({}, OOXML_BUILTIN_COLOR_SCHEME, {
        accent1: '729fcf', // LibreOffice Color Shape default background
        accent2: 'ff420e', // LibreOffice Color "Chart 2"
        accent3: 'ffd320', // LibreOffice Color "Chart 3"
        accent4: '579d1c', // LibreOffice Color "Chart 4"
        accent5: '7e0021', // LibreOffice Color "Chart 5"
        accent6: 'ff950e'  // LibreOffice Color "Chart 10" (because 6 looks like 1)
    });

    // the font scheme of the built-in default theme for ODF documents
    var ODF_BUILTIN_FONT_SCHEME = {};
    ODF_BUILTIN_FONT_SCHEME[AttributeUtils.MAJOR_FONT_KEY] = 'Liberation Sans';
    ODF_BUILTIN_FONT_SCHEME[AttributeUtils.MINOR_FONT_KEY] = 'Liberation Sans';

    // the keys of all scheme colors, as array
    var COLOR_KEYS = Object.keys(OOXML_BUILTIN_COLOR_SCHEME);

    // the keys of all scheme fonts, as array
    var FONT_KEYS = Object.keys(OOXML_BUILTIN_FONT_SCHEME);

    // private global functions ===============================================

    /**
     * Generates a unique map key from a theme's name and target.
     *
     * @param {String} name
     *  The name of the theme.
     *
     * @param {String} [target]
     *  The target of the theme. If omitted, an empty string is used.
     *
     * @returns {String}
     *  A unique map key from a theme's name and target.
     */
    function generateKey(name, target) {
        return JSON.stringify([name, target || '']);
    }

    // class ThemeModel =======================================================

    /**
     * Contains the definitions of a single theme in a document.
     *
     * @constructor
     *
     * @param {String} name
     *  The name of the theme.
     *
     * @param {Object} colorScheme
     *  The color scheme. MUST be complete according to the specification of
     *  the 'insertTheme' document operation (i.e. MUSt contain RRGGBB strings
     *  for all 12 scheme colors).
     *
     * @param {String} [target]
     *  The optional target of the theme.
     */
    function ThemeModel(name, colorScheme, fontScheme, target) {

        // public methods -----------------------------------------------------

        /**
         * Returns the name of this theme model.
         *
         * @returns {String}
         *  The name of this theme.
         */
        this.getName = function () {
            return name;
        };

        /**
         * Returns the target of this theme model.
         *
         * @returns {String}
         *  The name of the target of this theme; or the empty string, if the
         *  theme is not associated with a specific target.
         */
        this.getTarget = function () {
            return target || '';
        };

        /**
         * Generates a unique map key for this theme's name and target.
         *
         * @returns {String}
         *  A unique map key for this theme's name and target.
         */
        this.getKey = function () {
            return generateKey(this.getName(), this.getTarget());
        };

        /**
         * Returns the RGB color value of the specified scheme color key.
         *
         * @param {String} colorKey
         *  The internal key of the scheme color.
         *
         * @param {String} [defRGB]
         *  The default RGB value to be returned, if the specified scheme color
         *  does not exist. If omitted, null will be returned in that case.
         *
         * @returns {String|Null}
         *  The RGB value of the scheme color as hexadecimal string, if
         *  existing, otherwise the passed default RGB value or null.
         */
        this.getSchemeColor = function (colorKey, defRGB) {
            var color = colorScheme[colorKey];
            return (typeof color === 'string') ? color : (defRGB || null);
        };

        /**
         * Returns the font name of the specified scheme font key.
         *
         * @param {String} fontKey
         *  The internal key of a scheme font.
         *
         * @param {String} [defFont]
         *  The default font name to be returned, if the specified scheme font
         *  does not exist. If omitted, null will be returned in that case.
         *
         * @returns {String|Null}
         *  The name of the specified scheme font, if existing; otherwise the
         *  passed default font name or null.
         */
        this.getSchemeFont = function (fontKey, defFont) {
            var fontName = fontScheme[fontKey];
            return (typeof fontName === 'string') ? fontName : (defFont || null);
        };

    } // class ThemeModel

    // class ThemeCollection ==================================================

    /**
     * Contains the definitions of all themes in a document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     */
    var ThemeCollection = ModelObject.extend({ constructor: function (docModel) {

        // themes, mapped by identifier
        var themeMap = new ValueMap();

        // the default themes, mapped by target names
        var defaultThemeMap = new ValueMap();

        // global default theme (first inserted theme)
        var globalThemeModel = new ThemeModel(DEFAULT_THEME_NAME,
            docModel.getApp().isODF() ? ODF_BUILTIN_COLOR_SCHEME : OOXML_BUILTIN_COLOR_SCHEME,
            docModel.getApp().isODF() ? ODF_BUILTIN_FONT_SCHEME : OOXML_BUILTIN_FONT_SCHEME
        );

        // whether the default theme is still the built-in theme
        var isBuiltInTheme = true;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel);

        // protected methods --------------------------------------------------

        /**
         * Callback handler for the document operation 'insertTheme'. Inserts a
         * new theme into this collection.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'insertTheme' document operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applyInsertThemeOperation = function (context) {

            // the name of the theme
            var name = context.getStr('themeName');
            // the color scheme
            var colorScheme = context.getObj('colorScheme');
            // the font scheme
            var fontScheme = context.getObj('fontScheme');
            // the target name for the theme (empty string for missing target allowed)
            var target = context.getOptStr('target', '', true);

            // validate the color scheme
            COLOR_KEYS.forEach(function (colorKey) {
                var value = colorScheme[colorKey];
                context.ensure((typeof value === 'string') && /^[0-9A-Z]{6}$/i.test(value), 'missing or invalid color ' + colorKey);
            });

            // validate the font scheme
            FONT_KEYS.forEach(function (fontKey) {
                var value = fontScheme[fontKey];
                context.ensure((typeof value === 'string') && (value.length > 0), 'missing or invalid font ' + fontKey);
            });

            // create a theme object with all passed attributes, and insert it into the map
            var themeModel = new ThemeModel(name, colorScheme, fontScheme, target);
            themeMap.insert(themeModel.getKey(), themeModel);

            // set first inserted theme without target as new default theme
            if (!target && isBuiltInTheme) {
                globalThemeModel = themeModel;
                isBuiltInTheme = false;
            }

            // set first theme with target as default theme
            if (target && !defaultThemeMap.has(target)) {
                defaultThemeMap.insert(target, themeModel);
            }

            // notify listeners
            this.trigger('insert:theme', name, target);
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether a default theme exists for the specified target.
         *
         * @param {String} target
         *  A target string to specify the theme's target.
         *
         * @returns {Boolean}
         *  Whether a default theme exists for the specified target.
         */
        this.hasDefaultTheme = function (target) {
            return defaultThemeMap.has(target);
        };

        /**
         * Returns the default theme for the specified target from this
         * collection.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A single target name or an array of target names to specify the
         *  theme's target.
         *
         * @returns {ThemeModel}
         *  The default theme for the specified target, or the global default
         *  theme.
         */
        this.getDefaultModel = function (targets) {

            // single string: resolve theme directly from the map of default themes
            if ((typeof targets === 'string') && (targets.length > 0)) {
                return defaultThemeMap.get(targets, null);
            }

            // array of strings: find the first existing default theme
            if (_.isArray(targets)) {

                var themeModel = null;
                targets.some(function (target) {
                    // empty string addresses the global default theme if inserted explicitly
                    var useGlobal = (target === '') && !isBuiltInTheme;
                    themeModel = useGlobal ? globalThemeModel : defaultThemeMap.get(target, null);
                    return themeModel !== null;
                });

                // no theme found: fall back to global default theme
                return themeModel || globalThemeModel;
            }

            // fall back to global default theme
            return globalThemeModel;
        };

        /**
         * Returns a single theme from this collection.
         *
         * @param {String|Null} [name]
         *  The name of the theme to be returned. If empty or omitted or null
         *  or not existing, the default theme for the target, or the global
         *  default theme will be returned instead.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A single target name or an array of target names to specify the
         *  theme's target.
         *
         * @returns {ThemeModel}
         *  The specified theme if existing, otherwise the default theme for a
         *  target, or the global default theme.
         */
        this.getThemeModel = function (name, targets) {

            // missing theme name: resolve default theme by target
            if (!name) { return this.getDefaultModel(targets); }

            // search theme for the specified name
            var themeModel = null;
            if (!targets || (typeof targets === 'string')) {
                themeModel = themeMap.get(generateKey(name, targets), null);
            } else if (_.isArray(targets)) {
                targets.some(function (target) {
                    themeModel = themeMap.get(generateKey(name, target), null);
                    return themeModel !== null;
                });
            }

            // fall back to global default theme
            return themeModel || globalThemeModel;
        };

        // initialization -----------------------------------------------------

        // register the built-in default theme
        themeMap.insert(globalThemeModel.getKey(), globalThemeModel);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            docModel = themeMap = defaultThemeMap = globalThemeModel = null;
        });

    } }); // class ThemeCollection

    // exports ================================================================

    return ThemeCollection;

});
