/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author York Richter <york.richter@open-xchange.com>
 */

define('io.ox/office/editframework/model/tablestylecollection', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/model/stylecollection'
], function (Utils, AttributeUtils, StyleCollection) {

    'use strict';

    // class TableStyleCollection =============================================

    /**
     * Contains the style sheets for drawing formatting attributes.
     *
     * @constructor
     *
     * @extends StyleCollection
     *
     * @param {EditModel} docModel
     *  The document model containing this instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters passed to the base class StyleCollection. The
     *  following additional options are supported:
     *  - {Boolean} [initOptions.rowBandsOverColBands=false]
     *      If set to true, the inner row band attributes will be merged over
     *      the inner column band attributes. By default, inner column band
     *      attributes will win over inner row band attributes
     *  - {Boolean} [initOptions.expandColBandsToOuterCols=false]
     *      If set to true, the inner column band attributes will be merged
     *      over the active first and last column attributes. By default, inner
     *      column bands will start after the active first column, and will end
     *      before the active last column.
     *  - {Boolean} [initOptions.restrictColBandsToInnerRows=false]
     *      If set to true, the inner column band attributes will not be merged
     *      into the header row nor the footer row.
     */
    var TableStyleCollection = StyleCollection.extend({ constructor: function (docModel, initOptions) {

        // self reference
        var self = this;

        // whether row band attributes will be merged over column band attributes
        var rowBandsOverColBands = Utils.getBooleanOption(initOptions, 'rowBandsOverColBands', false);

        // whether to merge inner column bands over active first and/or last column
        var expandColBandsToOuterCols = Utils.getBooleanOption(initOptions, 'expandColBandsToOuterCols', false);

        // whether to exclude the column bands from the header and footer rows (default: merge into all rows)
        var restrictColBandsToInnerRows = Utils.getBooleanOption(initOptions, 'restrictColBandsToInnerRows', false);

        // base constructor ---------------------------------------------------

        StyleCollection.call(this, docModel, 'table', initOptions);

        // private methods ----------------------------------------------------

        // copies a border attribute (table or cell) to an outer cell border attribute
        function updateOuterCellBorder(targetAttrSet, sourceAttrSet, outerBorderName, innerBorderName, isOuterBorder) {

            // table and cell attributes (either may be missing)
            var tableAttrs = _.isObject(sourceAttrSet.table) ? sourceAttrSet.table : {};
            var cellAttrs = _.isObject(sourceAttrSet.cell) ? sourceAttrSet.cell : {};
            // the source border attribute value
            var border = null;

            if (isOuterBorder) {
                // copy outer table border to cell border if border is missing in cell attributes
                border = _.isObject(cellAttrs[outerBorderName]) ? null : tableAttrs[outerBorderName];
            } else {
                // copy inner table or cell border (cell border wins) to outer cell border
                border = _.isObject(cellAttrs[innerBorderName]) ? cellAttrs[innerBorderName] : tableAttrs[innerBorderName];
            }

            // insert existing border to the specified outer cell border
            if (_.isObject(border)) {
                AttributeUtils.insertAttribute(targetAttrSet, 'cell', outerBorderName, border);
            }
        }

        // copies a padding attribute from a table to a cell, if required
        function updateCellPadding(targetAttrSet, sourceAttrSet, paddingName) {

            // table and cell attributes (either may be missing)
            var tableAttrs = _.isObject(sourceAttrSet.table) ? sourceAttrSet.table : {};
            var cellAttrs = _.isObject(sourceAttrSet.cell) ? sourceAttrSet.cell : {};
            // the padding attribute value
            var padding = (paddingName in cellAttrs) ? cellAttrs[paddingName] : tableAttrs[paddingName];

            // insert existing border to the specified outer cell border
            if (_.isNumber(padding)) {
                AttributeUtils.insertAttribute(targetAttrSet, 'cell', paddingName, padding);
            }
        }

        // merges the specified conditional attributes into the target attribute set
        function mergeConditionalAttributes(targetAttrSet, sourceAttrSet, isRowBand, isColBand, isFirstRow, isLastRow, isFirstCol, isLastCol) {

            if (!_.isObject(sourceAttrSet)) { return; }

            // copy all attributes from the style sheet to the result object
            self.extendAttributeSet(targetAttrSet, sourceAttrSet);

            // INFO: In Presentation app it is possible, that 'borderInsideHor' and 'borderInsideVert' are set
            //       to the 'cell' family in table styles. Therefore it can happen, that these properties are
            //       set via the conditionalAttributes although this might not be wanted (48609). Then this
            //       properties need to be overwritten with hard attributes in operations.

            // copy inner borders to outer cell borders, if cell is located inside the current table area
            updateOuterCellBorder(targetAttrSet, sourceAttrSet, 'borderTop',    'borderInsideHor',  !isColBand || isFirstRow);
            updateOuterCellBorder(targetAttrSet, sourceAttrSet, 'borderBottom', 'borderInsideHor',  !isColBand || isLastRow);
            updateOuterCellBorder(targetAttrSet, sourceAttrSet, 'borderLeft',   'borderInsideVert', !isRowBand || isFirstCol);
            updateOuterCellBorder(targetAttrSet, sourceAttrSet, 'borderRight',  'borderInsideVert', !isRowBand || isLastCol);

            updateCellPadding(targetAttrSet, sourceAttrSet, 'paddingTop');
            updateCellPadding(targetAttrSet, sourceAttrSet, 'paddingBottom');
            updateCellPadding(targetAttrSet, sourceAttrSet, 'paddingLeft');
            updateCellPadding(targetAttrSet, sourceAttrSet, 'paddingRight');
        }

        function addRowBandAttributes(targetAttrSet, tableStyleAttrs, rowRange, isFirstRow, isLastRow, isFirstCol, isLastCol, options) {
            var outerRowActive = (options.firstRow && isFirstRow) || (options.lastRow && isLastRow);
            if (rowRange && !outerRowActive && Utils.getBooleanOption(options, 'bandsHor', false)) {
                // size of horizontal bands, TODO: replace '1' with attribute value
                var bandSize = Math.floor(Math.max(1, 1));
                // ignore first row in calculation of band index
                var bandIndex = Math.floor((rowRange.start - (options.firstRow ? 1 : 0)) / bandSize);
                // resolve odd or even band attributes
                var sourceAttrSet = (bandIndex % 2) ? tableStyleAttrs.band2Hor : tableStyleAttrs.band1Hor;
                mergeConditionalAttributes(targetAttrSet, sourceAttrSet, true, false, isFirstRow, isLastRow, isFirstCol, isLastCol);
            }
        }

        function addColBandAttributes(targetAttrSet, tableStyleAttrs, colRange, isFirstRow, isLastRow, isFirstCol, isLastCol, options) {
            var outerRowActive = (options.firstRow && isFirstRow) || (options.lastRow && isLastRow);
            var outerColActive = (options.firstCol && isFirstCol) || (options.lastCol && isLastCol);
            if (colRange && !(outerRowActive && restrictColBandsToInnerRows) && !(outerColActive && !expandColBandsToOuterCols) && Utils.getBooleanOption(options, 'bandsVert', false)) {
                // size of vertical bands, TODO: replace '1' with attribute value
                var bandSize = Math.floor(Math.max(1, 1));
                // ignore first column in calculation of band index
                var bandIndex = Math.floor((colRange.start - ((options.firstCol && !expandColBandsToOuterCols) ? 1 : 0)) / bandSize);
                // resolve odd or even band attributes
                var sourceAttrSet = (bandIndex % 2) ? tableStyleAttrs.band2Vert : tableStyleAttrs.band1Vert;
                mergeConditionalAttributes(targetAttrSet, sourceAttrSet, false, true, isFirstRow, isLastRow, isFirstCol, isLastCol);
            }
        }

        // public methods -----------------------------------------------------

        /**
         * Return the attributes for a specific table cell, depending on its
         * position in the table.
         *
         * @param {Object} tableStyleAttrs
         *  The raw style attribute set of a table style sheet.
         *
         * @param {Object|Null} colRange
         *  The column range of the cell (column span) with 'start' and 'end'
         *  integer properties.
         *
         * @param {Object|Null} rowRange
         *  The row range of the cell (row span) with 'start' and 'end' integer
         *  properties.
         *
         * @param {Integer} colCount
         *  the number of columns in the table.
         *
         * @param {Integer} rowCount
         *  The number of rows in the table.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.firstRow=false]
         *      Whether the first row (header row) is visible.
         *  @param {Boolean} [options.lastRow=false]
         *      Whether the last row (total row) is visible.
         *  @param {Boolean} [options.firstCol=false]
         *      Whether the first column attributes are used for the cell style
         *      attributes.
         *  @param {Boolean} [options.lastCol=false]
         *      Whether the last column attributes are used for the cell style
         *      attributes.
         *  @param {Boolean} [options.bandsHor=false]
         *      Whether the banded row attributes are used for the cell style
         *      attributes.
         *  @param {Boolean} [options.bandsVert=false]
         *      Whether the banded column attributes are used for the cell
         *      style attributes.
         *
         * @returns {Object}
         *  The resulting resolved attribute set for the specified cell.
         */
        this.resolveCellAttributeSet = function (tableStyleAttrs, colRange, rowRange, colCount, rowCount, options) {

            // table size
            var lastTableRow = rowCount - 1;
            var lastTableCol = colCount - 1;

            var isFirstRow = rowRange && (rowRange.start === 0);
            var isLastRow = rowRange && (rowRange.end === lastTableRow);

            var isFirstCol = colRange && (colRange.start === 0);
            var isLastCol = colRange && (colRange.end === lastTableCol);

            // the resulting style attributes according to the position of the table cell
            var targetAttrSet = {};

            // ensure an options object for easy and fast property access
            options = options || {};

            // wholeTable: always
            mergeConditionalAttributes(targetAttrSet, tableStyleAttrs.wholeTable, true, true, isFirstRow, isLastRow, isFirstCol, isLastCol);

            // horizontal/vertical band attributes
            if (rowBandsOverColBands) {
                addColBandAttributes(targetAttrSet, tableStyleAttrs, colRange, isFirstRow, isLastRow, isFirstCol, isLastCol, options);
                addRowBandAttributes(targetAttrSet, tableStyleAttrs, rowRange, isFirstRow, isLastRow, isFirstCol, isLastCol, options);
            } else {
                addRowBandAttributes(targetAttrSet, tableStyleAttrs, rowRange, isFirstRow, isLastRow, isFirstCol, isLastCol, options);
                addColBandAttributes(targetAttrSet, tableStyleAttrs, colRange, isFirstRow, isLastRow, isFirstCol, isLastCol, options);
            }

            // cell inside first/last row/column
            if (options.firstCol && isFirstCol) {
                mergeConditionalAttributes(targetAttrSet, tableStyleAttrs.firstCol, false, true, isFirstRow, isLastRow, isFirstCol, isLastCol);
            }
            if (options.lastCol && isLastCol) {
                mergeConditionalAttributes(targetAttrSet, tableStyleAttrs.lastCol, false, true, isFirstRow, isLastRow, isFirstCol, isLastCol);
            }
            if (options.firstRow && isFirstRow) {
                mergeConditionalAttributes(targetAttrSet, tableStyleAttrs.firstRow, true, false, isFirstRow, isLastRow, isFirstCol, isLastCol);
            }
            if (options.lastRow && isLastRow) {
                mergeConditionalAttributes(targetAttrSet, tableStyleAttrs.lastRow, true, false, isFirstRow, isLastRow, isFirstCol, isLastCol);
            }

            // single corner cells (only if inside active first/last row AND column areas)
            if (options.firstRow && options.firstCol && isFirstRow && isFirstCol) {
                mergeConditionalAttributes(targetAttrSet, tableStyleAttrs.northWestCell, false, false, isFirstRow, isLastRow, isFirstCol, isLastCol);
            }
            if (options.firstRow && options.last && isFirstRow && isLastCol) {
                mergeConditionalAttributes(targetAttrSet, tableStyleAttrs.northEastCell, false, false, isFirstRow, isLastRow, isFirstCol, isLastCol);
            }
            if (options.lastRow && options.firstCol && isLastRow && isFirstCol) {
                mergeConditionalAttributes(targetAttrSet, tableStyleAttrs.southWestCell, false, false, isFirstRow, isLastRow, isFirstCol, isLastCol);
            }
            if (options.lastRow && options.lastCol && isLastRow && isLastCol) {
                mergeConditionalAttributes(targetAttrSet, tableStyleAttrs.southEastCell, false, false, isFirstRow, isLastRow, isFirstCol, isLastCol);
            }

            return targetAttrSet;
        };

        // initialization -----------------------------------------------------

        // destroy class members on destruction
        this.registerDestructor(function () {
            self = docModel = initOptions = null;
        });

    } }); // class TableStyleCollection

    // exports ================================================================

    return TableStyleCollection;

});
