/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/modelattributesmixin', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/tk/render/font',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/gradient',
    'io.ox/office/editframework/model/fontcollection',
    'io.ox/office/editframework/model/themecollection',
    'io.ox/office/editframework/model/attributepool'
], function (Utils, ValueMap, Font, AttributeUtils, Color, Border, Gradient, FontCollection, ThemeCollection, AttributePool) {

    'use strict';

    // identifier of the default attribute pool
    var DEFAULT_POOL_ID = '__default';

    // the empty theme target chain (used as fall-back for missing resolver callback)
    var DEFAULT_THEME_TARGETS = [];

    // definitions for global document attributes -----------------------------

    var DOCUMENT_ATTRIBUTE_DEFINITIONS = {
    };

    // private global functions ===============================================

    /**
     * Returns the effective identifier of the attribute pool targeted by the
     * passed options.
     *
     * @param {Object} [options]
     *  Optional parameters that have been passed to a public method of the
     *  class AttributePool. Returns the string value of the option 'poolId' if
     *  existing, otherwise the built-in identifier of the default pool of a
     *  document.
     *
     * @returns {String}
     *  The effective identifier of the attribute pool targeted by the passed
     *  options.
     */
    function getPoolId(options) {
        return Utils.getStringOption(options, 'poolId', DEFAULT_POOL_ID);
    }

    // mix-in class ModelAttributesMixin ======================================

    /**
     * A mix-in class for the document model class EditModel that provides the
     * global document attributes, the default values for all formatting
     * attributes of all registered attribute families, the style collections
     * for all attribute families used in a document, a font collection, and a
     * collection of themes.
     *
     * @constructor
     *
     * @param {Object} [initOptions]
     *  Optional parameters passed to the constructor of the class EditModel.
     */
    function ModelAttributesMixin(initOptions) {

        // self reference (the document model)
        var self = this;

        // the collection of document themes
        var themeCollection = new ThemeCollection(this);

        // the collection of document fonts
        var fontCollection = new FontCollection(this);

        // the attribute pool for global document attributes
        var documentPool = new AttributePool(this);

        // all attribute definition pools, mapped by pool identifier
        var attributePoolMap = new ValueMap();

        // style sheet collections mapped by attribute family
        var styleCollectionMap = new ValueMap();

        // auto-style collections mapped by attribute family
        var autoStyleCollectionMap = new ValueMap();

        // callback function that returns the theme target chain, e.g. according to current selection
        var themeTargetsResolver = Utils.getFunctionOption(initOptions, 'themeTargetsResolver', null);

        // protected methods --------------------------------------------------

        /**
         * Callback handler for the document operation 'setDocumentAttributes'.
         * Changes the global document attributes, and/or the default values
         * for various other formatting attributes. These values override the
         * defaults of the attribute definitions passed in the method
         * AttributePool.registerAttributes(), and will be used before the
         * values of any style sheet attributes and explicit element attributes
         * will be resolved.
         *
         * @param {OperationContext} context
         *  A wrapper representing the 'setDocumentAttributes' document
         *  operation.
         *
         * @throws {OperationError}
         *  If applying the operation fails, e.g. if a required property is
         *  missing in the operation.
         */
        this.applySetDocumentAttributesOperation = function (context) {

            // the attributes to be changed
            var attributeSet = context.getObj('attrs');

            // change the global document attributes
            if ('document' in attributeSet) {
                documentPool.changeDefaultValues(attributeSet);
            }

            // change the default values of formatting attributes
            attributePoolMap.with(DEFAULT_POOL_ID, function (attributePool) {
                attributePool.changeDefaultValues(attributeSet);
            });
        };

        // public methods -----------------------------------------------------

        /**
         * Registers global document attributes, or updates the settings of
         * global document attributes that have been registered already.
         *
         * @param {Object} definitions
         *  The definitions of all global document attributes to be registered.
         *  See description of the method AttributePool.registerAttributes()
         *  for details.
         *
         * @returns {ModelAttributesMixin}
         *  A reference to this instance.
         */
        this.registerDocumentAttributes = function (definitions) {
            documentPool.registerAttributes('document', definitions);
            return this;
        };

        /**
         * Returns the value of a specific global document attribute.
         *
         * @param {String} attrName
         *  The name of the global document attribute.
         *
         * @returns {Any|Null}
         *  A deep copy of the specified global document attribute.
         */
        this.getDocumentAttribute = function (attrName) {
            return documentPool.getDefaultValue('document', attrName);
        };

        /**
         * Returns the global document attributes as complete map.
         *
         * @returns {Object}
         *  A deep copy of the global document attributes.
         */
        this.getDocumentAttributes = function () {
            return documentPool.getDefaultValues('document');
        };

        /**
         * Registers new formatting attributes for a specific attribute family,
         * or updates the settings of formatting attributes that have been
         * registered already.
         *
         * @param {String} attrFamily
         *  The name of the attribute family the new formatting attributes are
         *  associated with. MUST NOT be the string 'styleId' which is reserved
         *  to be used as style sheet identifier in attribute sets. MUST NOT be
         *  the string 'document' which is reserved for the definitions of
         *  global document attributes.
         *
         * @param {Object} definitions
         *  The definitions of all formatting attributes to be registered. See
         *  description of the method AttributePool.registerAttributes() for
         *  details.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {String} [options.poolId]
         *      The identifier of an attribute pool to be used (can be any
         *      non-empty string). If omitted, the default attribute pool of
         *      this document will be used. If specified the first time, a new
         *      attribute pool will be created automatically. MUST NOT start
         *      with an underscore character!
         *
         * @returns {ModelAttributesMixin}
         *  A reference to this instance.
         */
        this.registerAttributes = function (attrFamily, definitions, options) {

            // get an existing attribute pool, or create a new instance that forwards the change events
            var attributePool = attributePoolMap.getOrCreate(getPoolId(options), function () {
                var attributePool = new AttributePool(self);
                self.forwardEvents(attributePool);
                return attributePool;
            });

            // register the passed attributes
            attributePool.registerAttributes(attrFamily, definitions);
            return this;
        };

        /**
         * Returns the specified attribute pool of this document.
         *
         * @param {String|Null} [attrPoolId]
         *  The identifier of the attribute pool to be returned. If omitted or
         *  set to null, the default attribute pool of this document will be
         *  returned.
         *
         * @returns {AttributePool|Null}
         *  The specified attribute pool of this document; or null, if there is
         *  no attribute pool with the specified identifier.
         */
        this.getAttributePool = function (attrPoolId) {
            return attributePoolMap.get(attrPoolId || DEFAULT_POOL_ID, null);
        };

        /**
         * Adds a style sheet collection to the document.
         *
         * @param {StyleCollection} styleCollection
         *  The new style sheet collection to be inserted into the document.
         *
         * @returns {ModelAttributesMixin}
         *  A reference to this instance.
         */
        this.addStyleCollection = function (collection) {

            // prevent double registration
            var styleFamily = collection.getStyleFamily();
            if (styleCollectionMap.has(styleFamily)) {
                throw new Error('style family "' + styleFamily + '" already registered');
            }

            styleCollectionMap.insert(styleFamily, collection);
            return this;
        };

        /**
         * Returns the style sheet collection for the specified attribute
         * family.
         *
         * @param {String} styleFamily
         *  The name of the style attribute family.
         *
         * @returns {StyleCollection|Null}
         *  The specified style sheet collection if existing, otherwise null.
         */
        this.getStyleCollection = function (styleFamily) {
            return styleCollectionMap.get(styleFamily, null);
        };

        /**
         * Adds an auto-style collection to the document.
         *
         * @param {AutoStyleCollection} styleCollection
         *  The new auto-style collection to be inserted into the document.
         *
         * @returns {ModelAttributesMixin}
         *  A reference to this instance.
         */
        this.addAutoStyleCollection = function (collection) {

            // prevent double registration
            var styleFamily = collection.getStyleFamily();
            if (autoStyleCollectionMap.has(styleFamily)) {
                throw new Error('auto-style family "' + styleFamily + '" already registered');
            }

            autoStyleCollectionMap.insert(styleFamily, collection);
            return this;
        };

        /**
         * Returns the auto-style collection for the specified attribute
         * family.
         *
         * @param {String} family
         *  The name of the auto-style attribute family.
         *
         * @returns {AutoStyleCollection|Null}
         *  The specified auto-style collection if existing, otherwise null.
         */
        this.getAutoStyleCollection = function (styleFamily) {
            return autoStyleCollectionMap.get(styleFamily, null);
        };

        /**
         * Returns the collection with all registered themes.
         *
         * @returns {ThemeCollection}
         *  The collection with all registered themes.
         */
        this.getThemeCollection = function () {
            return themeCollection;
        };

        /**
         * Returns whether a default theme exists for the specified target.
         *
         * @param {String} target
         *  A target string to specify the theme's target.
         *
         * @returns {Boolean}
         *  Whether a default theme exists for the specified target.
         */
        this.hasDefaultTheme = function (target) {
            return themeCollection.hasDefaultTheme(target);
        };

        /**
         * Returns the theme target chain provided by the resolver callback
         * function passed to the constructor of this instance (see constructor
         * option 'themeTargetsResolver').
         *
         * @param {HTMLElement|jQuery} [elementNode]
         *  The DOM element node to resolve the theme targets for. Will be
         *  passed to the resolver callback function.
         *
         * @returns {Array<String>|Null}
         *  The theme target chain according to the target resolver callback.
         */
        this.getThemeTargets = function (elementNode) {
            return themeTargetsResolver ? themeTargetsResolver.call(this, elementNode) : DEFAULT_THEME_TARGETS;
        };

        /**
         * Returns the default theme from the document's theme collection for
         * an optional theme target.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be returned.
         *
         * @returns {ThemeModel}
         *  The model of the specified theme. If no target has been specified,
         *  the active target chain of the document will be used. If no theme
         *  has been inserted for the effective targets, the global default
         *  theme will be returned.
         */
        this.getThemeModel = function (targets) {
            return themeCollection.getDefaultModel(targets || this.getThemeTargets());
        };

        /**
         * Returns the default theme from the document's theme collection for
         * the passed DOM element.
         *
         * @param {HTMLElement|jQuery} elementNode
         *  The DOM element node to resolve the theme targets for. Will be
         *  passed to the resolver callback function passed to the constructor
         *  of this instance (see constructor option 'themeTargetsResolver').
         *
         * @returns {ThemeModel}
         *  The theme model associated with the passed DOM element.
         */
        this.getThemeModelForNode = function (elementNode) {
            return this.getThemeModel(this.getThemeTargets(elementNode));
        };

        /**
         * Returns the collection with all registered fonts.
         *
         * @returns {FontCollection}
         *  The collection with all registered fonts.
         */
        this.getFontCollection = function () {
            return fontCollection;
        };

        /**
         * Returns the effective font name for the passed attribute value.
         *
         * @param {String} fontName
         *  The name of a font (case-insensitive), or the internal key of a
         *  scheme font.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {String}
         *  The effective font name of the specified scheme font if existing;
         *  otherwise the passed font name as is.
         */
        this.resolveFontName = function (fontName, targets) {
            return this.getThemeModel(targets).getSchemeFont(fontName, fontName);
        };

        /**
         * Returns the value of the CSS 'font-family' attribute containing the
         * specified font name and all alternative font names.
         *
         * @param {String} fontName
         *  The name of of the font (case-insensitive).
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {String}
         *  The value of the CSS 'font-family' attribute containing the
         *  specified font name and all alternative font names.
         */
        this.getCssFontFamily = function (fontName, targets) {
            return fontCollection.getCssFontFamily(this.resolveFontName(fontName, targets));
        };

        /**
         * Creates and returns a font descriptor for rendering from the passed
         * character formatting attributes.
         *
         * @param {Object} charAttributes
         *  The character formatting attributes, as used in operations, with
         *  the following properties:
         *  - {String} charAttributes.fontName
         *      The original font name (case-insensitive).
         *  - {Number} charAttributes.fontSize
         *      The original font size, in points.
         *  - {Boolean} [charAttributes.bold=false]
         *      Whether the text will be rendered in bold characters.
         *  - {Boolean} [charAttributes.italic=false]
         *      Whether the text will be rendered in italic characters.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @param {Number} [zoom=1]
         *  The zoom factor used to scale the font size. The value 1 represents
         *  the normal zoom (of 100%).
         *
         * @returns {Font}
         *  The resulting font. The property 'family' includes all available
         *  fall-back fonts for the font. The property 'size' contains the font
         *  size as number (scaled according to the zoom factor if specified).
         */
        this.getRenderFont = function (charAttributes, targets, zoom) {
            var fontFamily = this.getCssFontFamily(charAttributes.fontName, targets);
            var fontSize = Utils.round(charAttributes.fontSize * (zoom || 1), 0.1);
            return new Font(fontFamily, fontSize, charAttributes.bold, charAttributes.italic);
        };

        /**
         * Resolves the passed color. Scheme colors will be resolved using the
         * current document theme.
         *
         * @param {Color} color
         *  The color to be resolved.
         *
         * @param {String|Color} auto
         *  Additional information needed to resolve the automatic color. See
         *  method Color.resolve() for details about this parameter.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {ColorDescriptor}
         *  A descriptor for the resulting color.
         */
        this.resolveColor = function (color, auto, targets) {
            return color.resolve(auto, this.getThemeModel(targets));
        };

        /**
         * Resolves the passed color. Scheme colors will be resolved using the
         * current document theme.
         *
         * @param {Object} jsonColor
         *  The JSON representation of the color to be resolved, as used in
         *  document operations.
         *
         * @param {String|Color} auto
         *  Additional information needed to resolve the automatic color. See
         *  method Color.resolve() for details about this parameter.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {ColorDescriptor}
         *  A descriptor for the resulting color.
         */
        this.parseAndResolveColor = function (jsonColor, jsonAuto, targets) {
            var color = Color.parseJSON(jsonColor);
            var auto = _.isString(jsonAuto) ? jsonAuto : Color.parseJSON(jsonAuto);
            return this.resolveColor(color, auto, targets);
        };

        /**
         * Converts the passed color attribute value to a CSS color value.
         * Scheme colors will be resolved by using the current theme.
         *
         * @param {Object} jsonColor
         *  The JSON representation of the source color, as used in document
         *  operations.
         *
         * @param {String|Object} jsonAuto
         *  Additional information needed to resolve the automatic color. See
         *  method Color.resolve() for details about this parameter.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {String}
         *  The CSS color value converted from the passed color object.
         */
        this.getCssColor = function (jsonColor, jsonAuto, targets) {
            return this.parseAndResolveColor(jsonColor, jsonAuto, targets).css;
        };

        /**
         * Returns a color descriptor with details about the passed color,
         * assuming it to be a text color on a specific background style.
         * Scheme colors will be resolved using the current document theme. See
         * method Color.resolveText() for more details.
         *
         * @param {Color} textColor
         *  The color to be resolved.
         *
         * @param {Array<Color>} fillColors
         *  The source fill colors, from outermost to innermost level.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {ColorDescriptor}
         *  A descriptor for the resulting color.
         */
        this.resolveTextColor = function (textColor, fillColors, targets) {
            return textColor.resolveText(fillColors, this.getThemeModel(targets));
        };

        /**
         * Returns a color descriptor with details about the passed color,
         * assuming it to be a text color on a specific background style.
         * Scheme colors will be resolved using the current document theme. See
         * method Color.resolveText() for more details.
         *
         * @param {Object} jsonTextColor
         *  The JSON representation of the text color to be resolved, as used
         *  in document operations.
         *
         * @param {Array<Object>} jsonFillColors
         *  The source fill colors, from outermost to innermost level, as array
         *  of JSON objects used in document operations.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {ColorDescriptor}
         *  A descriptor for the resulting color.
         */
        this.parseAndResolveTextColor = function (jsonTextColor, jsonFillColors, targets) {
            var textColor = Color.parseJSON(jsonTextColor);
            var fillColors = jsonFillColors.map(Color.parseJSON);
            return this.resolveTextColor(textColor, fillColors, targets);
        };

        /**
         * Resolves the passed text color to an explicit color if it is set to
         * 'auto', according to the specified fill colors.
         *
         * @param {Object} jsonTextColor
         *  The JSON representation of the source text color, as used in
         *  document operations.
         *
         * @param {Array} jsonFillColors
         *  The JSON representation of the source fill colors, from outermost
         *  to innermost level, as used in document operations.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {String}
         *  The CSS text color value converted from the passed color objects.
         */
        this.getCssTextColor = function (jsonTextColor, jsonFillColors, targets) {
            return this.parseAndResolveTextColor(jsonTextColor, jsonFillColors, targets).css;
        };

        /**
         * Extends a JSON color object with the passed optional transformation,
         * and the effective RGB value for ODF files.
         *
         * @param {Object} jsonColor
         *  The original JSON color descriptor. MUST NOT contain a color
         *  transformation.
         *
         * @param {String|Color} auto
         *  Additional information needed to resolve the automatic color. See
         *  method Color.resolve() for details about this parameter.
         *
         * @param {Object} [transform]
         *  An optional color transformation to be inserted into the resulting
         *  JSON color.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {Object}
         *  The resulting JSON color with the color transformation, and for ODF
         *  files with an additional property 'fallbackValue' containing the
         *  resulting RGB color value.
         */
        this.createJSONColor = function (jsonColor, auto, transform, targets) {

            // create a Color instance from the passed JSON color, and color transformation
            var color = new Color(jsonColor.type, jsonColor.value, transform ? [transform] : null);
            jsonColor = color.toJSON();

            // add the fall-back RGB value for ODF
            if (this.getApp().isODF()) {
                jsonColor.fallbackValue = this.resolveColor(color, auto, targets).hex;
            }

            return jsonColor;
        };

        /**
         *
         * @param gradient
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {*}
         */
        this.getGradientValue = function (gradient, targets) {
            return Gradient.parseCssColors(gradient, this, targets);
        };

        /**
         *
         * @param gradient
         * @param contextWrapper
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @returns {*}
         */
        this.getGradientFill = function (gradient, pathCoords, contextWrapper, targets) {
            return Gradient.parseFillStyle(this, gradient, contextWrapper.valueOf(), targets, pathCoords);
        };

        /**
         * Returns the effective CSS attributes for the passed border value.
         * Scheme colors will be resolved by using the current theme of an
         * optionally specified target.
         *
         * @param {Object} border
         *  The border object as used in operations.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.preview=false]
         *      If set to true, the border will be rendered for a preview
         *      element in the GUI. The border width will be restricted to two
         *      pixels in that case.
         *  - {Object} [options.autoColor]
         *      If specified, a replacement color to be used in case the border
         *      object contains the automatic color.
         *
         * @returns {Object}
         *  A map with CSS border attributes. Contains the property 'style'
         *  with the effective CSS border style as string; the property 'width'
         *  with the effective border width in pixels (as number, rounded to
         *  entire pixels); and the property 'color' with the effective CSS
         *  color (as string with leading hash sign).
         */
        this.getCssBorderAttributes = function (border, targets, options) {
            // use the static helper function from module Border, pass current theme
            return Border.getCssBorderAttributes(border, this.getThemeModel(targets), options);
        };

        /**
         * Converts the passed border attribute object to a CSS border value.
         * Scheme colors will be resolved by using the current theme of an
         * optionally specified target.
         *
         * @param {Object} border
         *  The border object as used in operations.
         *
         * @param {String|Array<String>|Null} [targets]
         *  A name or an array of names to specify the target theme. If omitted
         *  or set to null, the active theme of the document will be used.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  - {Boolean} [options.clearNone=false]
         *      If set to true, the return value for invisible borders will be
         *      the empty string instead of the keyword 'none'.
         *  - {Boolean} [options.preview=false]
         *      If set to true, the border will be rendered for a preview
         *      element in the GUI. The border width will be restricted to two
         *      pixels in that case.
         *  - {Object} [options.autoColor]
         *      If specified, a replacement color to be used in case the border
         *      object contains the automatic color.
         *
         * @returns {String}
         *  The CSS border value converted from the passed border object.
         */
        this.getCssBorder = function (border, targets, options) {
            // use the static helper function from module Border, pass current theme
            return Border.getCssBorder(border, this.getThemeModel(targets), options);
        };

        /**
         * Adding the fallbackValue property to all themed colors in a specified object.
         * The themed colors are searched recursively inside the given object.
         * Info: The specified object is modified within this function.
         *
         * @param {Object} obj
         *  An object, whose themed colors get the property 'fallbackValue'. This is
         *  especially required for ODF file format, because themed colors are not
         *  supported in this format.
         */
        this.addFallbackValueToThemeColors = function (obj) {
            if (_.isObject(obj)) {
                _.each(obj, function (value) {
                    if (_.isObject(value)) {
                        if (AttributeUtils.isColorThemed(value)) {
                            if (!value.fallbackValue) { value.fallbackValue = self.parseAndResolveColor(value, '').hex; }
                        } else {
                            self.addFallbackValueToThemeColors(value);
                        }
                    }
                });
            }
        };

        // initialization -----------------------------------------------------

        // forward changes of document attributes as 'change:attributes' event
        this.listenTo(documentPool, 'change:defaults', function (event, newValueSet, oldValueSet, changedValueSet) {
            self.trigger('change:attributes', newValueSet.document, oldValueSet.document, changedValueSet.document);
        });

        // global document attributes registration
        this.registerDocumentAttributes(DOCUMENT_ATTRIBUTE_DEFINITIONS);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            autoStyleCollectionMap.forEach(function (autoStyleCollection) { autoStyleCollection.destroy(); });
            styleCollectionMap.forEach(function (styleCollection) { styleCollection.destroy(); });
            attributePoolMap.forEach(function (attributePool) { attributePool.destroy(); });
            documentPool.destroy();
            fontCollection.destroy();
            themeCollection.destroy();
            self = themeCollection = fontCollection = null;
            attributePoolMap = documentPool = null;
            styleCollectionMap = autoStyleCollectionMap = null;
        });

    } // class ModelAttributesMixin

    // exports ================================================================

    return ModelAttributesMixin;

});
