/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/editmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/utils/logger',
    'io.ox/office/tk/utils/scheduler',
    'io.ox/office/tk/container/valuemap',
    'io.ox/office/baseframework/model/basemodel',
    'io.ox/office/editframework/utils/operations',
    'io.ox/office/editframework/utils/operationutils',
    'io.ox/office/editframework/model/operationcontext',
    'io.ox/office/editframework/model/operationgenerator',
    'io.ox/office/editframework/model/modelattributesmixin',
    'io.ox/office/editframework/model/undomanager'
], function (Utils, Logger, Scheduler, ValueMap, BaseModel, Operations, OperationUtils, OperationContext, OperationGenerator, ModelAttributesMixin, UndoManager) {

    'use strict';

    // operations logger
    var logger = new Logger({ enable: 'office:log-ops', prefix: 'OP' });

    // operation filter predicate that accepts all operations
    var ACCEPT_FILTER = _.constant(true);

    // operation filter predicate that rejects all operations
    var REJECT_FILTER = _.constant(false);

    // private global functions ===============================================

    /**
     * Extracts and returns the operation filter callback function from the
     * passed options.
     */
    function getFilterCallback(options) {
        var filterFunc = Utils.getFunctionOption(options, 'filter');
        return _.isFunction(filterFunc) ? filterFunc : Utils.getBooleanOption(options, 'filter', true) ? ACCEPT_FILTER : REJECT_FILTER;
    }

    // class EditModel ========================================================

    /**
     * The base class for editable document models. Adds generic support for
     * operations and undo action management.
     *
     * Triggers the following events:
     * - 'operations:before'
     *      Before an array of operations will be applied via the method
     *      EditModel.applyOperations(). The event handler receives the
     *      operations array, and the external state passed to the method. If a
     *      single operation has been passed to that method, it will be
     *      converted to an array with one element before calling the event
     *      handler.
     * - 'operations:after'
     *      After an array of operations have been applied via the method
     *      EditModel.applyOperations(), regardless if successfully or not. The
     *      event handler receives the external state passed to the method
     *      EditModel.applyOperations().
     * - 'operations:success'
     *      Directly after the event 'operations:after', if the operations have
     *      been applied successfully. The event handler receives the
     *      operations array, and the external state passed to the method
     *      EditModel.applyOperations(). If a single operation has been passed
     *      to that method, it will be converted to an array with one element
     *      before calling the event handler.
     * - 'operations:error'
     *      Directly after the event 'operations:after', if applying the
     *      operations has failed. The event handler receives the external
     *      state passed to the method EditModel.applyOperations().
     * - 'change:attributes':
     *      After the document attributes have been changed by a
     *      'setDocumentAttributes' document operation. Event handlers receive
     *      the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} newAttributes
     *          The new (complete) map of document attributes.
     *      (3) {Object} oldAttributes
     *          The old (complete) map of document attributes.
     *      (4) {Object} changedAttributes
     *          A map with the document attributes that have really changed.
     * - 'change:defaults':
     *      After the default values of formatting attributes have been changed
     *      by a 'setDocumentAttributes' document operation. Event handlers
     *      receive the following parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} newAttributeSet
     *          A complete attributes set with all current default values.
     *      (3) {Object} oldAttributeSet
     *          A complete attributes set with all previous default values.
     *      (4) {Object} changedAttributeSet
     *          An incomplete attributes set with all default values that have
     *          really changed.
     *
     * @constructor
     *
     * @extends BaseModel
     * @extends ModelAttributesMixin
     *
     * @param {EditApplication} app
     *  The application containing this document model instance.
     *
     * @param {Object} [initOptions]
     *  Optional parameters that will be passed to the constructor of the undo
     *  manager (see class UndoManager for details). Additionally, the
     *  following options are supported:
     *  @param {Function} [initOptions.generatorClass=OperationGenerator]
     *      The constructor function of the operations generator used by this
     *      document model. MUST be a sub class of class OperationGenerator.
     *  @param {Function} [initOptions.contextClass=OperationContext]
     *      The constructor function of the operation context class used to
     *      wrap JSON operation objects, passed to operation handler callback
     *      functions. MUST be a sub class of class OperationContext.
     *  @param {Function} [initOptions.selectionStateHandler]
     *      A function that implements to return the current selection state of
     *      the document, and to change the selection state passed to it. The
     *      selection state will be tracked for example by the undo manager to
     *      automatically restore the document selection after applying undo or
     *      redo operations. If the callback function is invoked without a
     *      parameter, it MUST return an object representing the current
     *      selection state of the document. Otherwise, if the callback
     *      function is invoked with one (object) parameter, it MUST restore
     *      the selection state of the document according to that value.
     *  @param {Function} [initOptions.operationsFinalizer]
     *      A callback function that will be invoked before applying operations
     *      and sending them to the server. Receives an array of JSON operation
     *      objects, and MUST return an array of processed operations (the
     *      return value may be the same array modified in-place). Will be
     *      called in the context of this document model.
     *  @param {Function} [initOptions.themeTargetsResolver]
     *      A callback function that returns the theme target chain for a
     *      specific DOM element or the active theme target chain, e.g.
     *      according to the current selection of this document model. Receives
     *      the following parameters:
     *      (1) {HTMLElement|jQuery} [elementNode]
     *          The DOM element node to resolve the theme target for. If
     *          omitted, the active theme target must be returned.
     *      Will be called in the context of this document model instance. May
     *      return null, a single string, or an array of strings.
     *  @param {Boolean} [initOptions.slideMode=false]
     *      Whether the model supports the slide mode (used in the Presentation
     *      application).
     */
    function EditModel(app, initOptions) {

        // the constructor of the operation generator class used by this document
        var GeneratorClass = Utils.getFunctionOption(initOptions, 'generatorClass', OperationGenerator);

        // the constructor of the operation context class used by this document
        var ContextClass = Utils.getFunctionOption(initOptions, 'contextClass', OperationContext);

        // self reference for local functions
        var self = this;

        // the undo manager of this document
        var undoManager = null;

        // callback handler to receive or restore the document selection state
        var selectionStateHandler = Utils.getFunctionOption(initOptions, 'selectionStateHandler', null);

        // callback to preprocess operations before applying them
        var operationsFinalizer = Utils.getFunctionOption(initOptions, 'operationsFinalizer', null);

        // Deferred object for current asynchronous actions, will remain in
        // resolved/rejected state as long as no new actions will be applied
        var actionsDef = $.when();

        // maps all operation names to operation handler functions
        var operationHandlerMap = new ValueMap();

        // whether operations are currently applied (prevent recursive calls)
        var processingOperations = false;

        // the total number of operations applied successfully
        var totalOperations = 0;

        // a counter for the operations, that adds all single operations, taking care of merged
        // operations. So this number must be equal on every client, independent from merge
        // processes on client or server side.
        var operationStateNumber = -1;

        // collecting the last two operations in cache
        var operationsCache = { previous: null, current: null };

        // flag to enable slide specific behavior
        var slideMode = Utils.getBooleanOption(initOptions, 'slideMode', false);

        // flag to enable slide specific behavior in iOS and Android
        var slideTouchMode = slideMode && (Utils.IOS || Utils.CHROME_ON_ANDROID);

        // base constructors --------------------------------------------------

        BaseModel.call(this, app);
        ModelAttributesMixin.call(this, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Switches this document model into 'action processing mode', and
         * invokes the passed callback function. As long as processing mode is
         * active, the method EditModel.getActionsPromise() returns a pending
         * promise. The callback function may return a promise to control the
         * duration of the action processing mode. This method MUST NOT be
         * called recursively! The methods EditModel.applyOperations() and
         * EditModel.applyActions() must not be used (these methods will enter
         * the action processing mode by themselves).
         *
         * @param {Function} callback
         *  The callback function that will be invoked with activated action
         *  processing mode. Must return either a boolean value that represents
         *  the result of the callback function, or a promise to extend the
         *  processing mode until the promise has been resolved or rejected.
         *  Will be called in the context of this instance.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.async=false]
         *      If set to true, the return value of this method will always be
         *      a promise. Internal errors will be represented by a rejected
         *      promise with an object with property 'cause'.
         *
         * @returns {Boolean|jQuery.Promise}
         *  The return value of the callback function; or false on an internal
         *  error.
         */
        function enterProcessActionsMode(callback, options) {

            // whether to process the callback asynchronously
            var async = Utils.getBooleanOption(options, 'async', false);

            // prevent recursive invocations
            if (actionsDef.state() === 'pending') {
                Utils.error('EditModel.enterProcessActionsMode(): recursive call');
                return async ? $.Deferred().reject({ cause: 'recursive' }) : false;
            }

            // create a new deferred object which remains in pending state as long
            // as the callback function runs (and the promise it returns is pending)
            actionsDef = Scheduler.createDeferred(app, 'EditModel.enterProcessActionsMode').always(function () {
                var state = actionsDef ? actionsDef.state().replace(/ed$/, 'ing') : 'destructed';
                logger.log('EditModel.enterProcessActionsMode(): ' + state + ' promise...');
            });

            // invoke the callback function
            var result = callback.call(self);

            // resolve the actions promise
            if (Utils.isPromise(result)) {
                result.progress(actionsDef.notify.bind(actionsDef));
                result.done(actionsDef.resolve.bind(actionsDef));
                result.fail(actionsDef.reject.bind(actionsDef));
                return result;
            }

            // reject on any value but true
            if (result === true) { actionsDef.resolve(); } else { actionsDef.reject(); }
            return async ? actionsDef.promise() : result;
        }

        /**
         * Called before an operations array will be applied at the document
         * model. Triggers an 'operations:before' event.
         *
         * @param {Array} operations
         *  The JSON operations to be applied. Will be passed to the listeners
         *  of the 'operations:before' event.
         *
         * @param {Boolean} external
         *  Whether the operations have been received from the server. Will be
         *  passed to the listeners of the 'operations:before' event.
         *
         * @returns {Boolean}
         *  False if this method has been called recursively, i.e. the model
         *  already processes operations. This is considered a fatal error.
         */
        function beginProcessOperations(operations, external) {

            if (processingOperations) {
                Utils.error('EditModel.beginProcessOperations(): recursive call - currently processing operations.');
                return false;
            }

            if (app.isInternalError()) {
                Utils.error('EditModel.beginProcessOperations(): illegal call - application is in error state.');
                return false;
            }

            self.trigger('operations:before', operations, external);
            processingOperations = true;
            return true;
        }

        /**
         * Called after an operations array has been applied at the document
         * model. Triggers an 'operations:after' event, and one of the
         * 'operations:success' or 'operations:error' events.
         *
         * @param {Boolean} success
         *  Whether applying the operations has succeeded.
         *
         * @param {Array} operations
         *  The operations that have been applied. Will be passed to the
         *  listeners of the operations events.
         *
         * @param {Boolean} external
         *  Whether the operations have been received from the server. Will be
         *  passed to the listeners of the operations events.
         */
        function endProcessOperations(success, operations, external) {
            processingOperations = false;
            self.trigger('operations:after', operations, external);
            self.trigger(success ? 'operations:success' : 'operations:error', operations, external);
        }

        /**
         * Executes the handler function for the passed operation.
         *
         * @param {Object} operation
         *  The JSON operation object to be applied at this document model.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Function} filter
         *  Whether to actually apply the operation locally (i.e. whether to
         *  invoke the registered callback handler).
         *
         * @param {Boolean} importing
         *  Whether the operation is part of the initial document import.
         *
         * @returns {Boolean}
         *  Whether the operation has been applied successfully.
         */
        function applyOperation(operation, external, filter, importing) {

            function failAndReturn() {
                var message = 'EditModel.applyOperation(): ' + _.printf.apply(_, arguments);
                Utils.error(message);
                app.sendLogMessage(message);
                return false;
            }

            // log operation
            logger.log('i=' + totalOperations + ', op=', operation);

            // check operation and operation name
            if (!_.isObject(operation)) {
                return failAndReturn('Expecting operation object.');
            }

            // operation handler may throw an exception
            try {

                // the context passed to the operation handler
                var context = new ContextClass(self, operation, external, importing);
                // the callback handler for the operation
                var handler = operationHandlerMap.get(context.getStr('name'), null);

                // check existence of operation handler (always, also if operation will be filtered below)
                context.ensure(handler, 'unknown operation');

                // Check the correct operation state number of an external operation. The OSN must be
                // identical on every client. The OSN is sent from server, when the document is loaded.
                if (external && context.has(OperationUtils.OSN)) {

                    // the OSN carried in the operation
                    var opOSN = context.getInt(OperationUtils.OSN);
                    context.ensure(opOSN >= 0, 'invalid OSN: %d', opOSN);

                    // During loading the document, the initial OSN is sent by the server.
                    // If we don't have a valid value use that OSN.
                    if (operationStateNumber === -1) {
                        self.setOperationStateNumber(opOSN);
                    }

                    // the current OSN must be equal to the OSN contained in the external operation
                    if (operationStateNumber !== opOSN) {
                        return failAndReturn('Wrong operation state number. current=%d operation=%d', operationStateNumber, operation.osn);
                    }
                }

                // insert current OSN and OPL into internal operations
                if (!external) {
                    operation.osn = operationStateNumber;
                    operation.opl = 1;
                }

                // update the current OSN of the document
                if (operationStateNumber >= 0) {
                    self.setOperationStateNumber(operationStateNumber + context.getOptInt(OperationUtils.OPL, 1));
                }

                // cache the last two operations
                operationsCache.previous = operationsCache.current;
                operationsCache.current = operation;

                // do not apply operations scoped to the export filter
                var scope = context.getOptStr('scope');

                // invoke the operation handler (unless the operation scope is 'filter' for operations intended
                // to be executed in the filter component only, or the filter callback function returns false)
                if ((scope !== 'filter') && (filter.call(self, operation) === true)) {
                    handler.call(self, context);
                }

            } catch (ex) {
                return failAndReturn('Operation failed. name="%s" msg="%s" osn=%d source=%s op=%s', operation.name, Utils.getStringOption(ex, 'message', ''), totalOperations + 1, external ? 'ext' : 'int', JSON.stringify(operation));
            }

            totalOperations += 1;
            return true;
        }

        /**
         * Executes the handler functions for all passed operations. If the
         * undo manager is active, all applied operations will be embedded into
         * a single undo group action.
         *
         * @param {Object|Array|OperationGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations to be applied at this
         *  document model. Note that an operations generator passed to this
         *  method will NOT be cleared after its operations have been sent.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Function|Boolean} [options.filter=true]
         *      A predicate function that will be invoked for each of the
         *      passed operations. Receives the operation as first parameter,
         *      and must return a boolean value specifying whether to invoke
         *      the operation handler. Can be set to false to prevent applying
         *      any operation locally (only send them to the server). The
         *      parameter value can also be a boolean value to enable or
         *      disable applying all operations.
         *  @param {Boolean} [options.undo=false]
         *      If set to true, and the parameter 'operations' is an instance
         *      of the class OperationGenerator, its undo operations will be
         *      applied.
         *
         * @returns {Boolean}
         *  Whether applying all operations was successful.
         */
        function applyOperationsSync(operations, external, options) {

            // operation filter callback function
            var filter = getFilterCallback(options);
            // whether document import is still running
            var importing = !self.isImportFinished();
            // the boolean result for synchronous mode
            var success = false;

            // convert input parameter to an array of JSON operations
            operations = OperationGenerator.getArray(operations, options);

            // if the operation's origin is not an external client and
            // we don't have edit rights then we skip the operations silently.
            if ((!external && !app.isEditable()) || (operations.length === 0)) {
                return true;
            }

            // allow subclasses to modify locally generated operations
            if (!external && _.isFunction(operationsFinalizer)) {
                operations = operationsFinalizer.call(self, operations);
            }

            // enter operations mode (prevent recursive calls)
            if (beginProcessOperations(operations, external)) {
                // apply all operations at once, group into an undo action
                undoManager.enterUndoGroup(function () {
                    success = operations.every(function (operation) {
                        return applyOperation(operation, external, filter, importing);
                    });
                });
            }

            // post-processing (trigger end events, leave operations mode),
            // also if beginProcessOperations() has failed (notify the error)
            endProcessOperations(success, operations, external);
            return success;
        }

        /**
         * Executes the handler functions for all passed operations in an
         * asynchronous background loop. If the undo manager is active, all
         * applied operations will be embedded into a single undo group action.
         *
         * @param {Object|Array|OperationGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations to be applied at this
         *  document model. Note that an operations generator passed to this
         *  method will NOT be cleared after its operations have been sent.
         *
         * @param {Boolean} external
         *  Whether the operation has been received from the server.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Function|Boolean} [options.filter=true]
         *      A predicate function that will be invoked for each of the
         *      passed operations. Receives the operation as first parameter,
         *      and must return a boolean value specifying whether to invoke
         *      the operation handler. Can be set to false to prevent applying
         *      any operation locally (only send them to the server). The
         *      parameter value can also be a boolean value to enable or
         *      disable applying all operations.
         *  @param {Boolean} [options.undo=false]
         *      If set to true, and the parameter 'operations' is an instance
         *      of the class OperationGenerator, its undo operations will be
         *      applied.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all operations have been
         *  applied successfully, or rejected immediately after applying an
         *  operation has failed. The promise regularly sends progress updates
         *  while applying the operations.
         */
        function applyOperationsAsync(operations, external, options) {

            // operation filter callback function
            var filter = getFilterCallback(options);
            // whether document import is still running
            var importing = !self.isImportFinished();
            // the resulting Promise
            var promise = null;
            // the global index of the first applied operation
            var startIndex = totalOperations;

            // convert input parameter to an array of JSON operations
            operations = OperationGenerator.getArray(operations, options);

            // immediate success on empty operations array
            if (operations.length === 0) { return self.createResolvedPromise(null); }

            // enter operations mode (prevent recursive calls)
            if (beginProcessOperations(operations, external)) {

                // apply all operations in a background task, group into an undo action
                promise = undoManager.enterUndoGroup(function () {

                    // return the promise to enterUndoGroup() to defer the open undo group
                    return self.iterateArraySliced(operations, function (operation) {

                        // if the operation's origin is not an external client and
                        // we don't have edit rights then we skip the operations silently
                        if (!external && !app.isEditable()) {
                            return Utils.BREAK;
                        }

                        // break the loop and reject the promise if an operation has failed
                        if (!applyOperation(operation, external, filter, importing)) {
                            return self.createRejectedPromise({ cause: 'operation' });
                        }
                    }, 'EditModel.applyOperationsAsync');
                });

            } else {
                // currently applying operations (called recursively)
                promise = self.createRejectedPromise({ cause: 'recursive' });
            }

            // all passed operations applied successfully: trigger end events, leave operations mode
            self.waitForSuccess(promise, function () {
                endProcessOperations(true, operations, external);
            });

            // operations failed: trigger end events, leave operations mode
            self.waitForFailure(promise, function (result) {
                if (result === 'abort') {
                    // manually aborted: notify applied operations only
                    endProcessOperations(true, operations.slice(0, totalOperations - startIndex), external);
                } else {
                    // operations failed: notify all operations
                    endProcessOperations(false, operations, external);
                }
            });

            return promise;
        }

        // protected methods --------------------------------------------------

        /**
         * Registers a callback function that will be executed when a document
         * operation with the specified name will be applied.
         *
         * @param {String} name
         *  The name of the document operation, as contained in the property
         *  'name' of an operation JSON object, to be handled by the passed
         *  callback function.
         *
         * @param {Function} handler
         *  The callback function that will be invoked for every operation with
         *  the specified name. Receives the following parameters:
         *  (1) {OperationContext} context
         *      A wrapper for the JSON operation object, representing the
         *      operation to be applied for this document. The exact type of
         *      this parameter depends on the class passed to the constructor
         *      parameter 'contextClass' of this document model.
         *  Will be called in the context of this document model instance. If
         *  the operation handler throws an exception, further processing of
         *  other operations will be stopped immediately, and the event
         *  'operations:error' will be triggered. The return value of the
         *  callback function will be ignored.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.registerContextOperationHandler = function (name, handler) {
            operationHandlerMap.insert(name, handler);
            return this;
        };

        /**
         * Registers a callback function that will be executed when a document
         * operation with the specified name will be applied.
         *
         * @deprecated
         *  Use method EditModel.registerContextOperationHandler() instead.
         *
         * @param {String} name
         *  The name of the document operation, as contained in the property
         *  'name' of an operation JSON object, to be handled by the passed
         *  callback function.
         *
         * @param {Function} handler
         *  The callback function that will be invoked for every operation with
         *  the specified name. Receives the following parameters:
         *  (1) {Object} operation
         *      The operation to be applied for this document, as JSON object.
         *  (2) {Boolean} external
         *      The external state flag that has been passed e.g. to the method
         *      EditModel.applyActions().
         *  Will be called in the context of this document model instance. If
         *  the operation handler returns the Boolean value false, or throws an
         *  exception, further processing of other operations will be stopped
         *  immediately, and the event 'operations:error' will be triggered.
         *  Every other return value (also undefined) indicates that the
         *  operation has been applied successfully.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.registerOperationHandler = function (name, handler) {
            return this.registerContextOperationHandler(name, function (context) {
                // old operation handlers may return false for failed operation
                var result = handler.call(this, context.operation, context.external);
                context.ensure(result !== false, 'operation handler failed');
            });
        };

        // public methods -----------------------------------------------------

        /**
         * Returns the current selection state of the document, according to
         * the callback function passed to the constructor.
         *
         * @returns {Object|Null}
         *  The current selection state of this document (the return value of
         *  the callback function passed as option 'selectionStateHandler' to
         *  the constructor ); or null, if no selection state handler has been
         *  registered.
         */
        this.getSelectionState = function () {
            return selectionStateHandler ? selectionStateHandler.call(this) : null;
        };

        /**
         * Changes the selection state of the document by invoking the callback
         * function passed to the constructor.
         *
         * @param {Object} selectionState
         *  The new selection state to be set at this document. Will be passed
         *  to the callback function passed as option 'selectionStateHandler'
         *  to the constructor.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.setSelectionState = function (selectionState) {
            if (selectionState && selectionStateHandler) {
                selectionStateHandler.call(this, selectionState);
            }
            return this;
        };

        /**
         * Returns whether the 'action processing mode' of this document model
         * is currently active (whether this model currently generates,
         * applies, or sends operation actions, either synchronously or
         * asynchronously).
         *
         * @returns {Boolean}
         *  Whether the 'action processing mode' is currently active.
         */
        this.isProcessingActions = function () {
            return actionsDef.state() === 'pending';
        };

        /**
         * Returns a promise representing the state of the 'action processing
         * mode'. If in pending state, this model currently generates, applies,
         * or sends operation actions (either synchronously or asynchronously).
         * If in resolved state, the model is ready to process new actions.
         *
         * @returns {jQuery.Promise}
         *  A promise representing the state of the action processing mode.
         */
        this.waitForActionsProcessed = function () {
            // do not expose the failed state of the deferred object
            return actionsDef.then(null, function () { return $.when(); });
        };

        /**
         * Returns whether operations are currently processed.
         *
         * @returns {Boolean}
         *  Whether the model processes operations.
         */
        this.isProcessingOperations = function () {
            return processingOperations;
        };

        /**
         * Sends the passed operations to the server without applying them
         * locally. In special situations, a client may choose to apply the
         * operations manually before they will be sent to the server. In this
         * case, the registered operation handlers must not be called.
         *
         * @param {Object|Array|OperationGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations to be sent to the
         *  server. Note that an operations generator passed to this method
         *  will NOT be cleared after its operations have been sent.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.undo=false]
         *      If set to true, and the parameter 'operations' is an instance
         *      of the class OperationGenerator, its undo operations will be
         *      sent to the server.
         *
         * @returns {Boolean}
         *  Whether registering all operations for sending was successful.
         */
        this.sendOperations = logger.profileMethod('EditModel.sendOperations()', function (operations, options) {
            return applyOperationsSync(operations, false, _.extend({}, options, { filter: false }));
        });

        /**
         * Executes the handler functions for all passed operations.
         *
         * @param {Object|Array|OperationGenerator} operations
         *  A single JSON operation, or an array of JSON operations, or an
         *  operations generator instance with operations to be applied at this
         *  document model. Note that an operations generator passed to this
         *  method will NOT be cleared after its operations have been applied.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Function|Boolean} [options.filter=true]
         *      A predicate function that will be invoked for each of the
         *      passed operations. Receives the operation as first parameter,
         *      and must return a boolean value specifying whether to invoke
         *      the operation handler. Can be set to false to prevent applying
         *      any operation locally (only send them to the server). The
         *      parameter value can also be a boolean value to enable or
         *      disable applying all operations.
         *  @param {Boolean} [options.async=false]
         *      If set to true, the operations will be applied asynchronously
         *      in a browser timeout loop. The promise returned by this method
         *      will be resolved or rejected after applying all operations, and
         *      will be notified about the progress of the operation.
         *  @param {Boolean} [options.undo=false]
         *      If set to true, and the parameter 'operations' is an instance
         *      of the class OperationGenerator, its undo operations will be
         *      applied.
         *
         * @returns {Boolean|jQuery.Promise}
         *  A boolean value specifying whether applying all operations was
         *  successful in synchronous mode (option 'async' has not been set),
         *  or a promise for asynchronous mode. The promise will be resolved
         *  after all operations have been applied successfully, or rejected
         *  immediately after applying an operation has failed. The caller has
         *  to wait for the promise before new operations can be applied. The
         *  promise regularly sends progress updates while applying the
         *  operations. When rejected, the promise will pass a result object
         *  with the following properties:
         *  - {String} result.cause
         *      The error code. Will be set to 'operation', if an operation
         *      callback handler has failed; or to 'recursive', if this method
         *      has been invoked recursively.
         */
        this.applyOperations = logger.profileMethod('EditModel.applyOperations()', function (operations, options) {

            // apply the operations synchronously, but enter action processing mode anyway,
            // operation handler may register themselves at the actions promise
            return enterProcessActionsMode(function () {

                // whether to execute asynchronously
                var async = Utils.getBooleanOption(options, 'async', false);
                // the private method to be called
                var applyOperationsMethod = async ? applyOperationsAsync : applyOperationsSync;

                // apply the passed operations
                return applyOperationsMethod(operations, false, options);
            });
        });

        /**
         * Executes the handler functions for all operations contained in the
         * passed actions. The operations will always be applied in an
         * asynchronous background loop. The operations of each action will be
         * enclosed into distinct undo groups automatically.
         *
         * @param {Object|Array} actions
         *  A single document action, or an array with document actions, to be
         *  applied at this document model. A document action is a JSON object
         *  with an optional array property 'operations' containing JSON
         *  operations as elements.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Function|Boolean} [options.filter=true]
         *      A predicate function that will be invoked for each operation
         *      contained in the passed actions. Receives the operation as
         *      first parameter, and must return a boolean value specifying
         *      whether to invoke the operation handler. Can be set to false to
         *      prevent applying any operation locally (only send them to the
         *      server). The parameter value can also be a boolean value to
         *      enable or disable applying all operations.
         *  @param {Boolean} [options.external=false]
         *      If set to true, the actions have been received from an external
         *      sender. Otherwise, the actions have been generated by the
         *      document model itself.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after all actions have been applied
         *  successfully, or rejected immediately after applying an action has
         *  failed. The caller has to wait for the promise before new actions
         *  can be applied. The promise regularly sends progress updates while
         *  applying the actions. It will contain an additional method abort()
         *  that can be used to stop applying the operations at any time. When
         *  rejected, the promise will pass a result object with the following
         *  properties:
         *  - {String} result.cause
         *      The error code. Will be set to 'operation', if an operation
         *      callback handler has failed; or to 'recursive', if this method
         *      has been invoked recursively.
         */
        this.applyActions = logger.profileAsyncMethod('EditModel.applyActions()', function (actions, options) {

            // total number of operations
            var operationCount = 0;

            // convert parameter to an array, filter for actions containing
            // at least one operation, get total number of operations
            actions = _.getArray(actions).filter(function (action) {
                var count = _.isArray(action.operations) ? action.operations.length : 0;
                operationCount += count;
                return count > 0;
            });

            // no operations: early exit
            if (operationCount === 0) { return $.when(); }

            // apply in action processing mode
            return enterProcessActionsMode(function () {

                // whether the actions have been received from the server
                var external = Utils.getBooleanOption(options, 'external', false);
                // resulting deferred object returned by this method
                var resultDef = Scheduler.createDeferred(app, 'EditModel.applyActions');
                // start index of first operations in the current action (for progress)
                var startIndex = 0;
                // the promise of the inner loop processing operations of an action
                var innerPromise = null;

                // apply the actions in a browser timeout loop
                var timer = self.iterateArraySliced(actions, function (action) {

                    // apply the operations of the current action
                    innerPromise = applyOperationsAsync(action.operations, external, options);

                    // calculate and notify the total progress of all actions
                    innerPromise.progress(function (progress) {
                        var currIndex = startIndex + action.operations.length * progress;
                        resultDef.notify(currIndex / operationCount);
                    });

                    // update the start index for the next action
                    innerPromise.done(function () { startIndex += action.operations.length; });

                    // clean up after finishing the loop
                    return innerPromise.always(function () { innerPromise = null; });

                }, 'EditModel.applyActions');

                // forward the result of the timer to the own deferred object
                timer.done(function () { resultDef.notify(1).resolve(); });
                timer.fail(resultDef.reject.bind(resultDef));

                // create an abortable promise for the deferred object
                return self.createAbortablePromise(resultDef, function () {
                    // abort the outer loop processing the actions
                    timer.abort();
                    // abort the inner loop processing the operations of an action
                    if (innerPromise) { innerPromise.abort(); }
                });
            }, { async: true });
        });

        /**
         * Executes the operation handler callback function for the passed JSON
         * operation locally and silently. No events will be generated at all
         * (the operation will not be sent to the server, the internal OSN will
         * not be changed, and it will not be processed in any other way).
         *
         * @param {Object} operation
         *  The JSON operation object of the document operation to be applied
         *  locally and silently.
         *
         * @returns {Boolean}
         *  Whether the passed operation has been applied successfully.
         */
        this.invokeOperationHandler = function (operation) {

            // callback handler throws exceptions to indicate operation errors
            try {

                // the operation context passed to the callback
                var context = new ContextClass(this, operation, false);

                // the callback handler for the operation
                var handler = operationHandlerMap.get(context.getStr('name'), null);
                context.ensure(handler, 'unknown operation');

                // suppress implicit undo operations generated e.g. in text framework
                undoManager.disableUndo(function () {
                    handler.call(this, context);
                }, this);

                return true;

            } catch (ex) {
                Utils.exception(ex);
                return false;
            }
        };

        /**
         * Creates and returns an initialized operations generator instance (an
         * instance of the class specified in the constructor parameter
         * 'generatorClass').
         *
         * @param {Object} [options]
         *  Optional parameters that will be passed to the constructor of the
         *  operations generator.
         *
         * @returns {OperationGenerator}
         *  A new operations generator instance.
         */
        this.createOperationGenerator = function (options) {
            return new GeneratorClass(this, options);
        };

        /**
         * Creates a new operations generator for document operations and undo
         * operations, invokes the callback function, applies all operations
         * contained in the generator, sends them to the server, and creates an
         * undo action with the undo operations that have been generated by the
         * callback function.
         *
         * @param {Function} callback
         *  The callback function. Receives the following parameters:
         *  (1) {OperationGenerator} generator
         *      The operations generator to be filled with the document
         *      operations, and undo operations.
         *  May return a promise to defer applying and sending the operations
         *  until the promise has been resolved. Will be called in the context
         *  of this model instance.
         *
         * @param {Object} [options]
         *  Optional parameters. Supports all options that are supported by the
         *  constructor of the operation generator class registered at this
         *  model instance (see constructor option 'generatorClass'), and the
         *  following options:
         *  @param {Boolean|Function} [options.storeSelection=false]
         *      If set to true, the original selection state will be queried
         *      from the document before invoking the callback function, and
         *      will be stored with the undo operations in the undo manager;
         *      AND the selection state will be queried again after invoking
         *      the callback function (the callback function may change the
         *      selection) will be stored with the redo operations. If set to a
         *      function, it will be invoked after applying the generated
         *      operations, and after that the selection state will be queried
         *      and stored with the redo actions (can be used to select new
         *      contents in the document that have been created with the new
         *      operations).
         *  @param {String} [options.undoMode='generate']
         *      Specifies how to treat the undo operations contained in the
         *      operations generator. MUST be one of the following values:
         *      - 'generate' (default): An undo action will be created in the
         *          undo manager.
         *      - 'skip': No undo action will be generated, regardless of the
         *          contents of the operations generator.
         *      - 'clear': No undo action will be generated, regardless of the
         *          contents of the operations generator, AND the undo manager
         *          will be cleared completely.
         *  @param {Function} [options.generatorClass]
         *      The constructor function of the operations generator used by
         *      this invocation. Can be used to override the default operations
         *      generator passed to the constructor of this docuemnt model.
         *      MUST be a sub class of class OperationGenerator.
         *  @param {Object} [options.undoOptions]
         *      Additional optional parameters that will be passed to the
         *      method UndoManager.enterUndoGroup(). These options will be
         *      stored in the undo/redo action group, and will be passed to all
         *      listeners of undo/redo events triggered by the undo manager.
         *
         * @returns {jQuery.Promise}
         *  A promise that will be resolved after the operations have been
         *  applied and sent successfully (with the result of the callback
         *  function); or rejected, if the callback has returned a rejected
         *  promise (with the result of that promise), or if applying the
         *  operations has failed (with the object {cause:'operation'}).
         */
        this.createAndApplyOperations = logger.profileAsyncMethod('EditModel.createAndApplyOperations()', function (callback, options) {

            // create a custom operations generator type if requested
            var CurrentGeneratorClass = Utils.getFunctionOption(options, 'generatorClass', GeneratorClass);
            // create the operation generator instance to be used during this method call
            var generator = new CurrentGeneratorClass(this, options);

            // whether to store the old and new selection state in the undo action
            var storeSelection = Utils.getBooleanOption(options, 'storeSelection', false);
            var storeSelectionHandler = storeSelection ? _.noop : Utils.getFunctionOption(options, 'storeSelection', null);

            // whether the generator applies all generated operations immediately
            var immediateMode = generator.isImmediateApplyMode();

            // simulate actions processing mode for deferred callbacks
            return enterProcessActionsMode(function () {

                // get the current selection state of the document
                var oldSelectionState = storeSelectionHandler ? self.getSelectionState() : null;
                // invoke the callback function
                var result = callback.call(self, generator);
                // start with a promise
                var promise = self.convertToPromise(result);

                // clean-up if the callback function fails
                promise = promise.then(null, function (result) {

                    // no thing to do, if application is already in internal error state, or if the
                    // operations have not been applied directly while generating them
                    if (app.isInternalError() || !immediateMode) { return result; }

                    // revert all operations that have been applied directly in the callback function
                    var promise2 = self.iterateArraySliced(generator.getOperations({ undo: true }), function (operation) {
                        if (!self.invokeOperationHandler(operation)) {
                            return $.Deferred().reject({ cause: 'operation' });
                        }
                    }, 'EditModel.createAndApplyOperations');

                    // restore the rejected state of the outer promise (unless reverting the operations fails too)
                    return promise2.then(_.constant($.Deferred().reject(result)));
                });

                // apply and send the operations after the callback has finished, and return the original result of the generator callback function
                promise = promise.then(function (result) {

                    // only send the generated operations in immediate mode
                    if (immediateMode) {
                        return self.sendOperations(generator) ? result : $.Deferred().reject({ cause: 'operation' });
                    }

                    // apply and send the operations, and return the original result of the generator callback function
                    return applyOperationsAsync(generator, false).then(_.constant(result));
                });

                // invoke the post-processor callback function, and create the undo action with the new selection
                promise = promise.then(function (result) {

                    // invoke the callback function to allow to change the document selection after applying the operations
                    var newSelectionState = null;
                    if (storeSelectionHandler) {
                        storeSelectionHandler.call(self);
                        newSelectionState = self.getSelectionState();
                    }

                    // create the undo action with the collected operations and selection states
                    switch (Utils.getStringOption(options, 'undoMode', 'generate')) {
                        case 'generate':
                            undoManager.enterUndoGroup(function () {
                                undoManager.addUndo(generator.getOperations({ undo: true }), generator.getOperations(), oldSelectionState, newSelectionState);
                            }, self, Utils.getObjectOption(options, 'undoOptions', null));
                            break;
                        case 'clear':
                            undoManager.clearUndoActions();
                            break;
                        case 'skip':
                            break;
                        default:
                            Utils.error('EditModel.createAndApplyOperations(): unknown undo mode');
                    }

                    // return the original result of the generator callback function
                    return result;
                });

                return promise;
            }, { async: true });
        });

        /**
         * Returns the total number of operations applied successfully.
         */
        this.getOperationsCount = function () {
            return totalOperations;
        };

        /**
         * Returns the operation state number of the document.
         */
        this.getOperationStateNumber = function () {
            return operationStateNumber;
        };

        /**
         * Sets the operation state number of the document. Used when empty doc
         * is fast displayed.
         *
         * @param {Number} osn
         *  new osn number
         */
        this.setOperationStateNumber = function (osn) {
            if (operationStateNumber !== osn) {
                operationStateNumber = osn;
                this.trigger('change:osn', osn);
            }
            return this;
        };

        /**
         * Returns the operation cache with the last two operations.
         *
         * @returns {Object}
         *  An object containing the last two operations in the properties
         *  last and current.
         */
        this.getOperationsCache = function () {
            return operationsCache;
        };

        /**
         * Returns the previous operation saved in the operation cache.
         * This is required for performance reasons. So no copy is returned
         * but the original operation object.
         *
         * @returns {Object}
         *  An object containing the previous operation.
         */
        this.getPreviousOperation = function () {
            return operationsCache.previous;
        };

        /**
         * Returns the undo manager of this document containing the undo action
         * stack and the redo action stack.
         *
         * @returns {UndoManager}
         *  The undo manager of this document.
         */
        this.getUndoManager = function () {
            return undoManager;
        };

        /**
         * Whether the slide mode needs to be used. This is typically the case
         * in the presentation application.
         *
         * @returns {Boolean}
         *  Whether the slide mode is used for this application.
         */
        this.useSlideMode = function () {
            return slideMode;
        };

        /**
         * Whether the touch mode should be used in Presentation.
         *
         * @returns {Boolean}
         *  Whether the touch mode is used in Presentation.
         */
        this.getSlideTouchMode = function () {
            return slideTouchMode;
        };

        // operation generators -----------------------------------------------

        /**
         * Generates an 'insertStyleSheet' operation, if the specified style
         * sheet exists in the style sheet collection, and is marked dirty.
         *
         * @param {SheetOperationGenerator} generator
         *  The operations generator to be filled with the operations.
         *
         * @param {String} styleFamily
         *  The style family that specifies which style sheet collection will
         *  be checked for a dirty style sheet.
         *
         * @param {String} styleId
         *  The identifier of the style sheet.
         *
         * @returns {EditModel}
         *  A reference to this instance.
         */
        this.generateMissingStyleSheetOperations = function (generator, styleFamily, styleId) {
            var styleSheets = this.getStyleCollection(styleFamily);
            if (styleSheets) { styleSheets.generateMissingStyleSheetOperations(generator, styleId); }
            return this;
        };

        // operation handlers -------------------------------------------------

        this.registerContextOperationHandler(Operations.SET_DOCUMENT_ATTRIBUTES, function (context) {
            this.applySetDocumentAttributesOperation(context);
        });

        this.registerContextOperationHandler(Operations.INSERT_THEME, function (context) {
            this.getThemeCollection().applyInsertThemeOperation(context);
        });

        this.registerContextOperationHandler(Operations.INSERT_FONT_DESCRIPTION, function (context) {
            this.getFontCollection().applyInsertFontOperation(context);
        });

        this.registerContextOperationHandler(Operations.INSERT_STYLESHEET, function (context) {

            // the style family name, and target style sheet collection
            var styleFamily = context.getStr('type');
            var styleCollection = this.getStyleCollection(styleFamily);
            context.ensure(styleCollection, 'invalid style family \'%s\'', styleFamily);

            // create undo operation (bug 40878: but only for custom style sheets, not for latent style sheets)
            var styleId = context.getStr('styleId');
            if (context.getOptBool('custom')) {
                var undoOp = { name: Operations.DELETE_STYLESHEET, type: styleFamily, styleId: styleId };
                undoManager.addUndo(undoOp, context.operation);
            }

            // apply the operation (throws on error)
            styleCollection.applyInsertStyleSheetOperation(context);
        });

        this.registerContextOperationHandler(Operations.DELETE_STYLESHEET, function (context) {

            // the style family name, target style sheet collection
            var styleFamily = context.getStr('type');
            var styleCollection = this.getStyleCollection(styleFamily);
            context.ensure(styleCollection, 'invalid style family \'%s\'', styleFamily);

            // create undo operation
            var styleId = context.getStr('styleId');
            var undoOp = { name: Operations.INSERT_STYLESHEET, type: styleFamily, styleId: styleId };
            undoOp.styleName = styleCollection.getName(styleId);
            undoOp.parentId = styleCollection.getParentId(styleId);
            undoOp.attrs = styleCollection.getStyleSheetAttributeMap(styleId, true);
            undoOp.hidden = styleCollection.isHidden(styleId);
            undoOp.uiPriority = styleCollection.getUIPriority(styleId);
            undoOp.custom = styleCollection.isCustom(styleId);
            undoManager.addUndo(undoOp, context.operation);

            // apply the operation (throws on error)
            styleCollection.applyDeleteStyleSheetOperation(context);
        });

        this.registerContextOperationHandler(Operations.CHANGE_STYLESHEET, function (context) {

            // the style family name, and target style sheet collection
            var styleFamily = context.getStr('type');
            var styleCollection = this.getStyleCollection(styleFamily);
            context.ensure(styleCollection, 'invalid style family \'%s\'', styleFamily);

            // create undo operation
            var styleId = context.getStr('styleId');
            var undoOp = { name: Operations.CHANGE_STYLESHEET, type: styleFamily, styleId: styleId };
            if (context.has('styleName')) { undoOp.styleName = styleCollection.getName(styleId); }
            if (context.has('attrs')) { undoOp.attrs = styleCollection.getStyleSheetAttributeMap(styleId, true); }
            undoManager.addUndo(undoOp, context.operation);

            // apply the operation (throws on error)
            styleCollection.applyChangeStyleSheetOperation(context);
        });

        this.registerContextOperationHandler(Operations.INSERT_AUTOSTYLE, function (context) {

            // the style family name, and target auto-style collection
            var styleFamily = context.getStr('type');
            var styleCollection = this.getAutoStyleCollection(styleFamily);
            context.ensure(styleCollection, 'invalid style family \'%s\'', styleFamily);

            // apply the operation (throws on error)
            styleCollection.applyInsertAutoStyleOperation(context);
        });

        this.registerContextOperationHandler(Operations.CHANGE_AUTOSTYLE, function (context) {

            // the style family name, and target auto-style collection
            var styleFamily = context.getStr('type');
            var styleCollection = this.getAutoStyleCollection(styleFamily);
            context.ensure(styleCollection, 'invalid style family \'%s\'', styleFamily);

            // apply the operation (throws on error)
            styleCollection.applyChangeAutoStyleOperation(context);
        });

        this.registerContextOperationHandler(Operations.DELETE_AUTOSTYLE, function (context) {

            // the style family name, and target auto-style collection
            var styleFamily = context.getStr('type');
            var styleCollection = this.getAutoStyleCollection(styleFamily);
            context.ensure(styleCollection, 'invalid style family \'%s\'', styleFamily);

            // apply the operation (throws on error)
            styleCollection.applyDeleteAutoStyleOperation(context);
        });

        // register handler for the operation 'noOp': does nothing, used for updating OSN etc.
        this.registerContextOperationHandler(Operations.NOOP, _.noop);

        // initialization -----------------------------------------------------

        // create the undo manager
        undoManager = new UndoManager(this, initOptions);

        // destroy all class members on destruction
        this.registerDestructor(function () {
            undoManager.destroy();
            self = initOptions = actionsDef = null;
            operationsFinalizer = operationHandlerMap = null;
            undoManager = GeneratorClass = ContextClass = null;
        });

    } // class EditModel

    // exports ================================================================

    // derive this class from class BaseModel
    return BaseModel.extend({ constructor: EditModel });

});
