/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define('io.ox/office/editframework/model/attributedmodel', [
    'io.ox/office/tk/utils',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/baseframework/model/modelobject'
], function (Utils, AttributeUtils, ModelObject) {

    'use strict';

    // class AttributedModel ==================================================

    /**
     * An abstract model object containing formatting attributes and the
     * reference to a style sheet of a specific attribute family.
     *
     * Triggers the following events:
     * - 'change:attributes'
     *      After the explicit attributes or the style sheet identifier have
     *      been changed, either directly by using the method
     *      AttributedModel.setAttributes(), or indirectly after the style
     *      sheet has been changed. The event handlers receive the following
     *      parameters:
     *      (1) {jQuery.Event} event
     *          The jQuery event object.
     *      (2) {Object} newAttributes
     *          The current (new) merged attribute set.
     *      (3) {Object} oldAttributes
     *          The previous (old) merged attribute set.
     * No events will be generated while the application imports the document.
     *
     * @constructor
     *
     * @extends ModelObject
     *
     * @param {EditModel} docModel
     *  The document model containing this attributed model object.
     *
     * @param {Object} [initAttributes]
     *  An attribute set with initial formatting attributes for the attributed
     *  model object.
     *
     * @param {Object} [initOptions]
     *  Optional parameters. Supports all attributes also supported by the base
     *  class ModelObject, and the following additional options:
     *  @param {String} [initOptions.styleFamily]
     *      The attribute family of style sheets that can be referred by this
     *      attributed model object. If omitted, this object supports explicit
     *      attributes only. All attribute families that are supported by these
     *      style sheets will also be supported by this model object.
     *  @param {String} [initOptions.families]
     *      The attribute families of explicit attributes supported by this
     *      attributed model object (space-separated).
     *  @param {Boolean} [initOptions.addStyleFamilies=true]
     *      Whether to add the attribute families supported by the style sheet
     *      container (as specified in the constructor of the respective
     *      container) to the own supported attribute families automatically.
     *  @param {Boolean} [initOptions.listenToStyles=true]
     *      If set to false, this instance will not listen to change events of
     *      the style sheet collection specified with the option 'styleFamily'.
     *      This may be desired for performance reasons to reduce the number of
     *      event listeners in collections with many instances of this class.
     *  @param {String} [initOptions.poolId]
     *      The identifier of an attribute pool to be used. This option will be
     *      ignored, if this model instance uses style sheets (see option
     *      'styleFamily'. In this case, the attribute pool of the style sheet
     *      collection will be used instead. If omitted and no style family has
     *      been specified, the default attribute pool of the document will be
     *      used.
     *  @param {AttributedModel} [initOptions.parentModel]
     *      A parent model object (instance of AttributedModel) that supports
     *      all or a few of the attribute families supported by this instance.
     *      The merged attributes of that parent model will be added to the own
     *      merged attribute set before adding the own style attributes and
     *      explicit attributes. This works also recursively (the parent model
     *      may define its own parent model). Attribute changes in the parent
     *      model will be forwarded to this instance (unless the option
     *      'listenToParent' has been set to false, see below).
     *  @param {Boolean} [initOptions.listenToParent=true]
     *      If set to false, this instance will not listen to change events of
     *      the parent model specified with the option 'parentModel'. This may
     *      be desired for performance reasons to reduce the number of event
     *      listeners in collections with many instances of this class.
     *  @param {Boolean} [initOptions.autoClear=false]
     *      If set to true, explicit attributes that are equal to the default
     *      attribute values (as specified in the attribute definitions of the
     *      respective attribute families), will not be inserted into the
     *      explicit attribute set, but the old explicit attribute will be
     *      removed from the explicit attribute set, when using the method
     *      AttributedModel.setAttributes(). This option has no effect, if this
     *      model instance uses style sheets (see option 'styleFamily'), or
     *      refers to a parent model object (see option 'parentModel').
     */
    function AttributedModel(docModel, initAttributes, initOptions) {

        // self reference
        var self = this;

        // the attribute family of the style sheets
        var styleFamily = Utils.getStringOption(initOptions, 'styleFamily', null, true);

        // the parent model used to build the merged attribute set
        var parentModel = Utils.getObjectOption(initOptions, 'parentModel', null);

        // whether to remove explicit attributes equal to the attribute defaults
        var autoClear = !styleFamily && !parentModel && Utils.getBooleanOption(initOptions, 'autoClear', false);

        // the collection of style sheets for the style family
        var styleCollection = styleFamily ? docModel.getStyleCollection(styleFamily) : null;

        // the attribute pool to be used for the formatting attributes
        var attributePool = styleCollection ? styleCollection.getAttributePool() :
            docModel.getAttributePool(Utils.getStringOption(initOptions, 'poolId', null));

        // the names of all supported attribute families, as flag set
        var supportedFamilySet = null;

        // the (incomplete) explicit attributes of this model object
        var explicitAttributeSet = {};

        // the (complete) cached merged attributes of this model object
        var mergedAttributeSet = null;

        // base constructor ---------------------------------------------------

        ModelObject.call(this, docModel, initOptions);

        // private methods ----------------------------------------------------

        /**
         * Recalculates the merged attribute set.
         */
        function calculateMergedAttributeSet(notify) {

            // remember the old merged attributes
            var oldMergedAttributeSet = mergedAttributeSet;

            // start with default attribute values of all supported families
            mergedAttributeSet = attributePool.getDefaultValueSet(supportedFamilySet);

            // add merged attributes of parent model object
            if (parentModel) {
                attributePool.extendAttributeSet(mergedAttributeSet, parentModel.getMergedAttributeSet(true));
            }

            // add the style sheet attributes to resulting merged attribute set
            var styleId = null;
            if (styleCollection) {
                styleId = explicitAttributeSet.styleId || '';
                attributePool.extendAttributeSet(mergedAttributeSet, styleCollection.getStyleAttributeSet(styleId));
                styleId = mergedAttributeSet.styleId;
            }

            // add the explicit attributes of this instance
            attributePool.extendAttributeSet(mergedAttributeSet, explicitAttributeSet);

            // delete all unsupported attribute families
            _.each(mergedAttributeSet, function (attributes, family) {
                if (!self.supportsFamily(family)) {
                    delete mergedAttributeSet[family];
                }
            });

            // restore the effective style sheet identifier
            if (styleCollection) {
                mergedAttributeSet.styleId = styleId;
            }

            // notify all listeners
            if (notify) {
                self.trigger('change:attributes', mergedAttributeSet, oldMergedAttributeSet);
            }
        }

        // protected methods --------------------------------------------------

        /**
         * Clones all contents from the passed attributed model into this model
         * instance.
         *
         * @internal
         *  Used during clone construction. DO NOT CALL from external code!
         *
         * @param {AttributedModel} sourceModel
         *  The source model instance whose attributes will be cloned into this
         *  model instance.
         *
         * @returns {AttributedModel}
         *  A reference to this instance.
         */
        this.cloneFrom = function (sourceModel) {

            // deep copy of all explicit attributes
            explicitAttributeSet = sourceModel.getExplicitAttributeSet();

            // recalculate the merged attributes (without notifications)
            calculateMergedAttributeSet(false);

            return this;
        };

        // public methods -----------------------------------------------------

        /**
         * Returns whether this instances supports attributes of the specified
         * attribute family.
         *
         * @param {String} family
         *  The name of an attribute family.
         *
         * @returns {Boolean}
         *  Whether this instances supports the specified attribute family.
         */
        this.supportsFamily = function (family) {
            return family in supportedFamilySet;
        };

        /**
         * Returns the identifier of the style sheet used to format this model.
         *
         * @returns {String|Null}
         *  The identifier of the style sheet used to format this model; or the
         *  value null, if this model does not support style sheets.
         */
        this.getStyleId = function () {
            return styleCollection ? mergedAttributeSet.styleId : null;
        };

        /**
         * Returns whether this model object contains any explicit attributes,
         * or a reference to a style sheet.
         *
         * @returns {Boolean}
         *  Whether this model object contains any explicit attributes, or a
         *  reference to a style sheet.
         */
        this.hasExplicitAttributes = function () {
            return !_.isEmpty(explicitAttributeSet);
        };

        /**
         * Returns the explicit attribute set of this model object.
         *
         * @param {Boolean} [direct=false]
         *  If set to true, the returned attribute set will be a reference to
         *  the original map stored in this instance, which MUST NOT be
         *  modified! By default, a deep clone of the attribute set will be
         *  returned that can be freely modified.
         *
         * @returns {Object}
         *  An (incomplete) attribute set (a map of attribute maps with
         *  name/value pairs, keyed by the attribute families, optionally
         *  containing the style sheet reference in the 'styleId' property),
         *  containing values for all attributes that have been explicitly set.
         */
        this.getExplicitAttributeSet = this.getExplicitAttributes = function (direct) {
            return (direct === true) ? explicitAttributeSet : _.copy(explicitAttributeSet, true);
        };

        /**
         * Returns the merged attribute set of this model object.
         *
         * @param {Boolean} [direct=false]
         *  If set to true, the returned attribute set will be a reference to
         *  the original map stored in this instance, which MUST NOT be
         *  modified! By default, a deep clone of the attribute set will be
         *  returned that can be freely modified.
         *
         * @returns {Object}
         *  A complete attribute set (a map of attribute maps with name/value
         *  pairs, keyed by the attribute families, additionally containing the
         *  style sheet reference in the 'styleId' property), containing values
         *  for all known attributes, collected from the document default
         *  values, the style sheet currently referenced by this model object,
         *  and the explicit attributes of this model object.
         */
        this.getMergedAttributeSet = function (direct) {
            return (direct === true) ? mergedAttributeSet : _.copy(mergedAttributeSet, true);
        };

        /**
         * Returns a reduced copy of the passed attribute set that contains
         * only the attributes that are not equal to the attributes of this
         * model instance (i.e. all attributes that will change this model if
         * applied as a document operation).
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set to be reduced according to the
         *  explicit attributes of this instance.
         *
         * @returns {Object}
         *  A dep copy of the passed attribute set that will only contain the
         *  attribute values that are not equal to the explicit attibute values
         *  of this model instance (incluging missing explicit attributes); or
         *  null values that will delete an existing explicit attribute.
         */
        this.getReducedAttributeSet = function (attributeSet) {

            var reducedAttributeSet = {};

            // copy all new attributes that change a value, or clear an existing value
            _.each(attributeSet, function (newAttrs, family) {

                // skip unsupported attribute families
                if (!self.supportsFamily(family)) { return; }

                // the explicit attributes of the current family
                var oldExplicitAttrs = explicitAttributeSet[family];
                // the attribute values for comparison, according to 'autoClear' mode
                var oldAttrValues = autoClear ? mergedAttributeSet[family] : oldExplicitAttrs;

                // new attribute is null: clear existing attributes; otherwise copy attributes that will change the current values
                _.each(newAttrs, function (value, name) {
                    if (value === null) {
                        if (oldExplicitAttrs && (name in oldExplicitAttrs)) {
                            AttributeUtils.insertAttribute(reducedAttributeSet, family, name, null);
                        }
                    } else {
                        if (!oldExplicitAttrs || !(name in oldExplicitAttrs) || !_.isEqual(value, oldAttrValues[name])) {
                            AttributeUtils.insertAttribute(reducedAttributeSet, family, name, value);
                        }
                    }
                });
            });

            return reducedAttributeSet;
        };

        /**
         * Returns an attribute set that will restore the current state of the
         * explicit attributes, after they have been changed according to the
         * passed attribute set. Intended to be used to create undo operations.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set to be merged over the current
         *  explicit attributes of this instance.
         *
         * @returns {Object}
         *  An attribute set that will restore the current explicit attributes
         *  after they have been changed according to the passed attributes.
         */
        this.getUndoAttributeSet = function (attributeSet) {

            var undoAttributeSet = {};

            // handle the style sheet identifier
            if (styleCollection) {
                var oldStyleId = explicitAttributeSet.styleId;
                var newStyleId = attributeSet.styleId;
                if ((typeof newStyleId === 'string') && (newStyleId !== oldStyleId)) {
                    undoAttributeSet.styleId = (typeof oldStyleId === 'string') ? oldStyleId : null;
                } else if ((newStyleId === null) && (typeof oldStyleId === 'string')) {
                    undoAttributeSet.styleId = oldStyleId;
                }
            }

            // restore all old attributes that will be changed or cleared
            _.each(explicitAttributeSet, function (oldAttrs, family) {

                // the new attributes of the current family (nothing to undo, if the family will not be changed)
                var newAttrs = attributeSet[family];
                if (!_.isObject(newAttrs)) { return; }

                // copy the old attribute values to the undo attribute set, if it will change
                _.each(oldAttrs, function (value, name) {
                    if ((name in newAttrs) && !_.isEqual(value, newAttrs[name])) {
                        AttributeUtils.insertAttribute(undoAttributeSet, family, name, value);
                    }
                });
            });

            // clear all new attributes with the undo operation, that currently do not exist
            _.each(attributeSet, function (newAttrs, family) {

                // skip unsupported attribute families
                if (!self.supportsFamily(family)) { return; }

                // the explicit attributes of the current family
                var oldAttrs = explicitAttributeSet[family];
                // the definitions of the current family
                var definitionMap = attributePool.getRegisteredAttributes(family);
                // all attributes to be restored in undo even if they will not be changed
                var restoreAttrs = null;

                // process all new attributes
                definitionMap.forEachSupported(newAttrs, function (value, name, definition) {

                    // if the attribute does not exist, reset it in undo (with a null value)
                    if (!oldAttrs || !(name in oldAttrs)) {
                        AttributeUtils.insertAttribute(undoAttributeSet, family, name, null);
                    }

                    // if the attribute has been changed, and the definition contains additional
                    // attributes to always be restored, add them to the undo attribute set
                    var undoAttrs = undoAttributeSet[family];
                    if (oldAttrs && undoAttrs && (name in undoAttrs)) {
                        if (definition.undo === true) {
                            restoreAttrs = oldAttrs;
                        } else if (_.isArray(definition.undo) && (restoreAttrs !== oldAttrs)) {
                            restoreAttrs = restoreAttrs || {};
                            definition.undo.forEach(function (attrName) {
                                if (attrName in oldAttrs) {
                                    restoreAttrs[attrName] = oldAttrs[attrName];
                                }
                            });
                        }
                    }
                });

                // add all attributes to the undo attribute set that need to be updated
                // according to the attribte definition
                if (restoreAttrs) {
                    undoAttributeSet[family] = _.extend({}, restoreAttrs, undoAttributeSet[family]);
                }
            });

            return undoAttributeSet;
        };

        /**
         * Returns whether this attributed model contains the same explicit
         * attributes as the passed attributed model.
         *
         * @param {AttributedModel} attribModel
         *  The other attributed model to be compared to this model.
         *
         * @returns {Boolean}
         *  Whether both attributed models contain the same explicit formatting
         *  attributes.
         */
        this.hasEqualAttributes = function (attribModel) {
            return (this === attribModel) || _.isEqual(explicitAttributeSet, attribModel.getExplicitAttributeSet(true));
        };

        /**
         * Changes and/or removes specific explicit formatting attributes, or
         * the style sheet reference of this model object.
         *
         * @param {Object} attributeSet
         *  An (incomplete) attribute set (a map of attribute maps with
         *  name/value pairs, keyed by the attribute families, optionally
         *  containing the style sheet reference in the 'styleId' property)
         *  with all formatting attributes to be changed. To clear an explicit
         *  attribute value (thus defaulting to the current style sheet), the
         *  respective value in this attribute set has to be set explicitly to
         *  the null value. Attributes missing in this attribute set will not
         *  be modified.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {String} [options.notify='auto']
         *      Specifies whether to trigger the event 'change:attributes'. If
         *      set to 'auto' or omitted, the event will be triggered if at
         *      least one formatting attribute has changed its value. If set to
         *      'always', the event will always be triggered without comparing
         *      the old and new attribute values. If set to 'never', no event
         *      will be triggered.
         *  @param {Boolean} [options.special=false]
         *      If set to true, 'special' attributes (marked with the 'special'
         *      flag in the attribute definitions denoting an attribute for
         *      internal use only, not for usage in document operations) can be
         *      changed too. Otherwise, special attributes will not be
         *      recognized by this method.
         *
         * @returns {Boolean}
         *  Whether the attributes of this model object have actually changed.
         */
        this.setAttributes = function (attributeSet, options) {

            // whether to trigger the event
            var notify = Utils.getStringOption(options, 'notify', 'auto');
            // whether any explicit attribute or the style identifier has been changed
            var changed = false;

            // set new style sheet or auto style
            if ('styleId' in attributeSet) {

                // style collection must exist (this model must support style sheets at all)
                if (!styleCollection) {
                    Utils.error('AttributedModel.setAttributes(): setting style sheets not supported');
                    return false;
                }

                // the style sheet identifier from the passed attributes
                var styleId = attributeSet.styleId;

                // remove explicit reference to default style sheet (missing style
                // reference will automatically fall-back to the default style sheet)
                if (!_.isString(styleId) || (styleId === '') || (styleId === styleCollection.getDefaultStyleId())) {
                    styleId = null;
                }

                // change the style sheet identifier in the internal map, update the resulting changed flag
                if (AttributeUtils.updateAttribute(explicitAttributeSet, 'styleId', styleId)) {
                    changed = true;
                }
            }

            // update the remaining explicit attributes of all supported attribute families
            _.each(supportedFamilySet, function (flag, family) {

                // attribute definitions of the current family
                var definitionMap = attributePool.getRegisteredAttributes(family);

                // check if new attributes exist in the passed attribute set
                if (!definitionMap || !_.isObject(attributeSet[family])) { return; }

                // target attribute map for the current family
                var explicitAttributes = explicitAttributeSet[family] || (explicitAttributeSet[family] = {});

                // update the attribute map with the new attribute values
                definitionMap.forEachSupported(attributeSet[family], function (value, name, definition) {
                    if (AttributeUtils.updateAttribute(explicitAttributes, name, value, autoClear ? definition : null)) {
                        changed = true;
                    }
                }, options);

                // remove empty attribute value maps completely
                if (_.isEmpty(explicitAttributes)) {
                    delete explicitAttributeSet[family];
                }
            });

            // recalculate merged attributes and notify all listeners
            if (changed || !mergedAttributeSet) {
                calculateMergedAttributeSet((changed && (notify === 'auto')) || (notify === 'always'));
            }

            return changed;
        };

        /**
         * Updates the merged attribute set, according to the current defaults,
         * the style sheet and parent model referenced by this instance, and
         * its explicit attributes.
         *
         * @returns {AttributedModel}
         *  A reference to this instance.
         */
        this.refreshMergedAttributeSet = function () {
            calculateMergedAttributeSet(false);
            return this;
        };

        // initialization -----------------------------------------------------

        // get all supported attribute families
        supportedFamilySet = Utils.makeSet(Utils.getTokenListOption(initOptions, 'families', []));
        if (styleCollection && Utils.getBooleanOption(initOptions, 'addStyleFamilies', true)) {
            _.extend(supportedFamilySet, styleCollection.getSupportedFamilies());
        }

        // set the attributes passed to the constructor (filters by supported attributes)
        this.setAttributes(_.isObject(initAttributes) ? initAttributes : {}, { notify: 'never' });
        initAttributes = null;

        // listen to changes in the parent object, update the merged attributes and notify own listeners
        if (parentModel && Utils.getBooleanOption(initOptions, 'listenToParent', true)) {
            this.listenTo(parentModel, 'change:attributes', function () {
                calculateMergedAttributeSet(true);
            });
        }

        // listen to style sheet changes, update the merged attributes and notify own listeners
        if (styleCollection && Utils.getBooleanOption(initOptions, 'listenToStyles', true)) {
            this.listenTo(styleCollection, 'triggered', function () {
                calculateMergedAttributeSet(true);
            });
        }

        // update merged attributes silently once after import (object may be destroyed before import finishes)
        this.waitForImportSuccess(function (alreadyFinished) {
            if (!alreadyFinished) { calculateMergedAttributeSet(false); }
        });

        // destroy all class members on destruction
        this.registerDestructor(function () {
            self = docModel = parentModel = styleCollection = attributePool = null;
            explicitAttributeSet = mergedAttributeSet = null;
        });

    } // class AttributedModel

    // exports ================================================================

    // derive this class from class ModelObject
    return ModelObject.extend({ constructor: AttributedModel });

});
