/**
 * This work is provided under the terms of the CREATIVE COMMONS PUBLIC
 * LICENSE. This work is protected by copyright and/or other applicable
 * law. Any use of the work other than as authorized under this license
 * or copyright law is prohibited.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Daniel Rentz <daniel.rentz@open-xchange.com>
 */

define.async('io.ox/office/drawinglayer/view/drawingframe', [
    'io.ox/office/tk/utils',
    'io.ox/office/tk/forms',
    'io.ox/office/tk/io',
    'io.ox/office/tk/locale/localedata',
    'io.ox/office/tk/render/path',
    'io.ox/office/tk/render/rectangle',
    'io.ox/office/tk/render/canvas',
    'io.ox/office/editframework/utils/attributeutils',
    'io.ox/office/editframework/utils/border',
    'io.ox/office/editframework/utils/color',
    'io.ox/office/editframework/utils/gradient',
    'io.ox/office/editframework/utils/pattern',
    'io.ox/office/editframework/utils/texture',
    'io.ox/office/drawinglayer/utils/drawingutils',
    'io.ox/office/drawinglayer/model/drawingmodel',
    'io.ox/office/drawinglayer/view/drawinglabels',
    'io.ox/office/drawinglayer/lib/canvasjs.min',
    'less!io.ox/office/drawinglayer/view/drawingstyle'
], function (Utils, Forms, IO, LocaleData, Path, Rectangle, Canvas, AttributeUtils, Border, Color, Gradient, Pattern, Texture, DrawingUtils, DrawingModel, Labels, CanvasJS) {

    'use strict';

    // mathematical constants
    var PI_180 = Math.PI / 180;
    var PI_RAD = Math.PI / 10800000;

    // convenience shortcuts
    var abs = Math.abs;
    var floor = Math.floor;
    var ceil = Math.ceil;
    var round = Math.round;
    var min = Math.min;
    var max = Math.max;
    var sqrt = Math.sqrt;
    var sin = Math.sin;
    var cos = Math.cos;
    var tan = Math.tan;
    var atan2 = Math.atan2;

    // name of the jQuery data attribute that stores the unique identifier of a drawing frame node
    var DATA_UID = 'uid';

    // name of the jQuery data attribute that stores the drawing model instance for a drawing frame node
    var DATA_MODEL = 'model';

    // the CSS class name of the drawing content node
    var CONTENT_CLASS = 'content';

    // the CSS class used to crop images
    var CROPPING_CLASS = 'cropping-frame';

    // the CSS class name of the selection root node
    var SELECTION_CLASS = 'selection';

    // the CSS class name for the tracker node inside the selection
    var TRACKER_CLASS = 'tracker';

    // the CSS class name for active tracking mode
    var TRACKING_CLASS = 'tracking-active';

    var CULTURE_INFO = null;

    var PRESET_GEOMETRIES = null;

    var MAX_CANVAS_SIZE = (_.browser.iOS || _.browser.Android || _.browser.Safari || (_.browser.IE <= 10)) ? 2156 : 4096;

    // cache during resize for texture bitmap used as shape backgound
    var textureCanvasPatternCache = [];

    // private global functions ===============================================

    /**
     * Returns the selection root node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The selection root node of the specified drawing frame. Will be an
     *  empty collection, if the drawing frame is not selected currently.
     */
    function getSelectionNode(drawingFrame) {
        return $(drawingFrame).first().children('.' + SELECTION_CLASS);
    }

    /**
     * Calculates the offset and size of the bitmap in an image frame for one
     * dimension (either horizontally or vertically), according to the passed
     * cropping settings.
     *
     * @param {Number} frameSize
     *  Visible size (with or height) of the drawing frame containing the image
     *  bitmap, in 1/100 of millimeters.
     *
     * @param {Number} leadingCrop
     *  The leading cropping value (left/top).
     *
     * @param {Number} trailingCrop
     *  The trailing cropping value (right/bottom).
     *
     * @returns {Object}
     *  An object containing 'offset' and 'size' CSS attributes specifying how
     *  to format the bitmap node (in pixels with 'px' unit name).
     */
    function calculateBitmapSettings(app, frameSize, leadingCrop, trailingCrop) {

        // ODF: absolute cropping length (in 1/100mm), OOXML: percentage
        if (app.isODF()) {
            leadingCrop /= frameSize;
            trailingCrop /= frameSize;
        } else {
            leadingCrop /= 100;
            trailingCrop /= 100;
        }

        var // sum of leading and trailing cropping (must not exceed a specific amount)
            totalCrop = leadingCrop + trailingCrop,
            // resulting settings for the bitmap
            size = 0, offset = 0;

        // do not crop away more than 90% of the bitmap
        if (totalCrop > 0.9) {
            leadingCrop *= (0.9 / totalCrop);
            trailingCrop *= (0.9 / totalCrop);
        }

        // bitmap size and offset, according to object size and cropping
        size = frameSize / (1 - leadingCrop - trailingCrop);
        offset = size * leadingCrop;

        // convert to CSS pixels
        return {
            offset: Utils.convertHmmToCssLength(-offset, 'px', 1),
            size: Utils.convertHmmToCssLength(size, 'px', 1)
        };
    }

    /**
     * Returns the path rendering mode for the specified flags, as expected by
     * the rendering methods of a canvas context.
     *
     * @param {Boolean} hasFill
     *  Whether the shape area will be filled.
     *
     * @param {Boolean} hasLine
     *  Whether the shape outline will be rendered.
     *
     * @returns {String|Null}
     *  The path rendering mode as string, if either of the passed flags is
     *  true, otherwise null to indicate that nothing needs to be drawn.
     */
    function getRenderMode(hasFill, hasLine) {
        return hasFill ? (hasLine ? 'all' : 'fill') : (hasLine ? 'stroke' : null);
    }

    /**
     * Calculating an optional increase of a canvas node inside a drawing node caused
     * by specified line endings.
     *
     * @param {Object} lineAttrs
     *  The line attributes of the drawing.
     *
     * @param {Object} geometry
     *  The geometry attribute of a drawing.
     *
     * @param {Number} expansion
     *  The precalculated expansion.
     *
     * @returns {Number[]|Number}
     *  The expansion (in pixel) required for a canvas node, so that the path together
     *  with the line ends can be displayed completely. This can be a single number, so
     *  that every side gets the same expansion, or an object of numbers, with the
     *  properties 'left', 'top', 'right' and 'bottom' representing the specific canvas
     *  expansions for left, top, right and bottom.
     */
    function getCanvasExpansionForLineEnds(lineAttrs, geometry, expansion) {

        var // the minimum expansion for arrows in pixel
            arrowExpansion = 10;

        // additional expansion for arrows at line end
        if (lineAttrs && geometry) {
            if ((lineAttrs.headEndType && lineAttrs.headEndType !== 'none') ||
                (lineAttrs.tailEndType && lineAttrs.tailEndType !== 'none')) {

                if (('pathList' in geometry) && (geometry.pathList.length > 0)) {

                    arrowExpansion =  max(Utils.round(2.5 * Utils.convertHmmToLength((lineAttrs.width  || 100), 'px'), 1), arrowExpansion);

                    if (_.isNumber(expansion)) {
                        expansion += arrowExpansion;
                    } else if (_.isObject(expansion)) {
                        for (var key in expansion) { expansion[key] += arrowExpansion; }
                    }
                }
            }
        }

        return expansion;
    }

    /**
     * Calculating an optional increase of a canvas node inside a drawing node.
     * Info: If the canvas needs to be expanded to the left or top, the value for the
     *       expansion needs to be positive.
     *
     * @param {Object} geometry
     *  The geometry attribute of a drawing.
     *
     * @param {Object} snapRect
     *  The dimension of the drawing without any expansion.
     *
     * @param {Object} guideCache
     *  Caching the already calculated guide variables.
     *
     * @param {Number} expansion
     *  The precalculated expansion.
     *
     * @returns {Number[]|Number}
     *  The expansion (in pixel) required for a canvas node, so that the path together
     *  with the line ends can be displayed completely. This can be a single number, so
     *  that every side gets the same expansion, or an object of numbers, with the
     *  properties 'left', 'top', 'right' and 'bottom' representing the specific canvas
     *  expansions for left, top, right and bottom.
     */
    function getCanvasExpansion(geometry, snapRect, guideCache, expansion) {

        var // the required expansion to the left
            dLeft = 0,
            // the required expansion to the top
            dTop = 0,
            // the required expansion to the right
            dRight = 0,
            // the required expansion to the bottom
            dBottom = 0,
            // the collector for the calculated extreme values inside a shape
            collector = { cxMin: 0, cxMax: 0, cyMin: 0, cyMax: 0 };

        // helper function for calculating the geometry values
        function getGeoValue(value) {
            return getGeometryValue(value, snapRect, geometry.avList, geometry.gdList, guideCache);
        }

        if (!geometry || !geometry.pathList) { return expansion; }  // do nothing, if no path specified

        // iterating over the path list
        geometry.pathList.forEach(function (onePath) {

            var path        = _.copy(onePath, true),

                widthPx     = snapRect.width,
                heightPx    = snapRect.height,

                logWidth    = ('width' in path) ? path.width : widthPx,
                logHeight   = ('height' in path) ? path.height : heightPx,

                fXScale     = (widthPx / logWidth),
                fYScale     = (heightPx / logHeight),

                onePathCollector = null;

            if (onePath.commands) {

                onePathCollector = onePath.commands.reduce(executePathCommands, {
                    canvasPath: new Path(),
                    cx: 0,
                    cy: 0,
                    fXScale: fXScale,
                    fYScale: fYScale,

                    getGeometryValue: getGeoValue,

                    cxMin: 0,
                    cxMax: 0,
                    cyMin: 0,
                    cyMax: 0
                });

            }

            // collecting the extreme values for every path
            if (onePathCollector.cxMin < collector.cxMin) { collector.cxMin = onePathCollector.cxMin; }
            if (onePathCollector.cxMax > collector.cxMax) { collector.cxMax = onePathCollector.cxMax; }
            if (onePathCollector.cyMin < collector.cyMin) { collector.cyMin = onePathCollector.cyMin; }
            if (onePathCollector.cyMax > collector.cyMax) { collector.cyMax = onePathCollector.cyMax; }

        });

        // comparing the calculated values with the specified snapRect
        if (collector.cxMin < snapRect.left) { dLeft = -(collector.cxMin - snapRect.left); } // using positive value
        if (collector.cxMax > (snapRect.width - snapRect.left)) { dRight = collector.cxMax - (snapRect.width - snapRect.left); }
        if (collector.cyMin < snapRect.top) { dTop = -(collector.cyMin - snapRect.top); } // using positive value
        if (collector.cyMax > (snapRect.height - snapRect.top)) { dBottom = collector.cyMax - (snapRect.height - snapRect.top); }

        if (dLeft || dRight || dTop || dBottom) {
            expansion = { left: Utils.round(dLeft, 1), top: Utils.round(dTop, 1), right: Utils.round(dRight, 1), bottom: Utils.round(dBottom, 1) };
        }

        return expansion;
    }

    /**
     * Checking whether a line specified by two points p1 (x1, y1) and p2 (x2, y2)
     * goes from p2 to p1 to the right direction. If it does not go from left to
     * right, it is checked, if it goes from top to down.
     * This function can be used to check the direction of an arrowhead. This can
     * typically only be used for 'simple' arrows, that are drawn in a canvas from
     * top left to bottom right.
     *
     * @param {Number} x1
     *  The x-position of point 1.
     *
     * @param {Number} y1
     *  The y-position of point 1.
     *
     * @param {Number} x2
     *  The x-position of point 2.
     *
     * @param {Number} x2
     *  The y-position of point 2.
     *
     * @returns {Boolean}
     *  Whether the specified two points describe a line from left-to-right or if
     *  the line is exact vertical from top-to-bottom.
     */
    function isRightOrDownDirection(x1, y1, x2, y2) {
        return (x2 < x1) || ((x2 === x1) && (y2 < y1));
    }

    /**
     * Analyzing a path to get a start or an end point together with a reference
     * point. The reference point can be used to calculate an angle of an arrowhead
     * at the head or the tail of the path.
     *
     * @param {Object[]} commands
     *  The command to draw the path. There must be at least two commands in the path.
     *  For the head the first two commands are evaluated, for the tail the final two
     *  paths.
     *
     * @param {Object} [options]
     *  Optional parameter:
     *  @param {Boolean} [options.first=false]
     *      If set to true, the head of the path is evaluated, otherwise the tail.
     *  @param {Object} [snapRect]
     *     The rectangle with the properties left, top, width and height
     *     for setting the size of the canvas.
     *  @param {Object} [avList]
     *     The used adjustment values.
     *  @param {Object[]} [gdList]
     *     An array with formulas that can be used.
     *  @param {Object} [guideCache]
     *     The cache with already calculated positioning values.
     *
     * @returns {Object}
     *  An object containing the properties 'start' and 'end'. The values of these
     *  properties are again objects with the properties 'x' and 'y' that are
     *  calculated from the commands.
     */
    function getReferencePointsFromPath(commands, options) {

        // whether the first visible line of the path is searched
        var first = Utils.getBooleanOption(options, 'first', false);
        // the first visible line of the path
        var length = 0;
        // the point of the first or the second last command
        var start = {};
        // the point of the second or the last command
        var end = {};
        // the first command of the command list
        var firstCommand = null;
        // the last command of the command list
        var secondCommand = null;
        // the end point of a curved line
        var endPoint = null;
        // the reference point of a curved line
        var refPoint = null;
        // the number of point of a specific path command
        var pointCount = 0;

        // helper function, that calculates, if a specified reference point is left
        // of the start point or right from the end point at bezier curves.
        // pointA is the start or end point, pointB the corresponding reference point.
        function isOutside(pointA, pointB, isStart) {

            var pointAX = 0;
            var pointBX = 0;
            var isOutside = false;

            if (options.snapRect && options.avList && options.gdList && options.guideCache) {
                pointAX = getGeometryValue(pointA.x, options.snapRect, options.avList, options.gdList, options.guideCache);
                pointBX = getGeometryValue(pointB.x, options.snapRect, options.avList, options.gdList, options.guideCache);
                isOutside = isStart ? (pointBX < pointAX) : (pointBX > pointAX);
            }

            return isOutside;
        }

        if (first) {

            firstCommand = commands[0];
            secondCommand = commands[1];

            start.x = firstCommand.x;
            start.y = firstCommand.y;

            end.x = secondCommand.x;
            end.y = secondCommand.y;

            if (!end.x && start.x &&  secondCommand.pts) { // handling of non-linear curves (cubicBezierTo)
                end.x = start.x;
                end.y = start.y;
                // comparing start point with first reference point to check direction of curve
                end.xOffset = isOutside(secondCommand.pts[0], secondCommand.pts[1], true) ? -5 : +5; // must be negative, if the bezier point is outside the shape -> enabling valid direction of arrow at line start
            }

        } else {

            length = commands.length;
            firstCommand = commands[length - 2];
            secondCommand = commands[length - 1];

            start.x = firstCommand.x;
            start.y = firstCommand.y;

            end.x = secondCommand.x;
            end.y = secondCommand.y;

            if (!start.x &&  secondCommand.pts) { // handling of non-linear curves (cubicBezierTo)

                pointCount = secondCommand.pts.length;

                endPoint = secondCommand.pts[pointCount - 1];
                refPoint = secondCommand.pts[pointCount - 2];

                start.x = endPoint.x;
                start.y = endPoint.y;

                end.x = endPoint.x;
                end.y = endPoint.y;

                // comparing end point with last reference point to check direction of curve
                start.xOffset = isOutside(endPoint, refPoint, false) ? 5 : -5; // must be positive, if the bezier point is outside the shape -> enabling valid direction of arrow at line end
            }
        }

        return { start: start, end: end };
    }

    /**
     * Handling the line ends for a specified path. The line end is handled by the line attributes
     * "headEndType", "headEndLength", "headEndWidth", "tailEndType", "tailEndLength" and "tailEndWidth".
     *
     * @param {ContextWrapper} context
     *  Canvas context wrapper.
     *
     * @param {Object} lineAttrs
     *  Object containing line attributes for the canvas
     *
     * @param {Number} lineWidth
     *  The line width in pixel.
     *
     * @param {Object} color
     *  The line color object.
     *
     * @param {Object} path
     *  The line path element.
     *
     * @param {String} arrowType
     *  The type of the line ending. Supported values are:
     *  'oval', 'diamond', 'triangle', 'stealth' and 'arrow'.
     *
     * @param {String} endType
     *  The end type for the specified path. Supported values are:
     *  'head' and 'tail'.
     *
     * @param {Object} [options]
     *  Optional parameter:
     *  @param {Object} [snapRect]
     *     The rectangle with the properties left, top, width and height
     *     for setting the size of the canvas.
     *  @param {Object} [avList]
     *     The used adjustment values.
     *  @param {Object[]} [gdList]
     *     An array with formulas that can be used.
     *  @param {Object} [guideCache]
     *     The cache with already calculated positioning values.
     *
     * @returns {Array<Object>|Null}
     *  The collector for the paths that are necessary to draw the line endings. This can be an
     *  array, that already contains paths, or an empty array, or null.
     */
    function handleLineEnd(context, lineAttrs, lineWidth, color, path, arrowType, endType, options) {

        // an additional line end path
        var lineEndPath = null;
        // the length of the arrow (in pixel)
        var length = 0;
        // the width of the arrow (in pixel)
        var width = 0;
        // whether the arrow head required a 'fill' additionally to the 'stroke' for the path
        var fillRequired = false;
        // a collector for all commands of the path
        var allCommands = null;
        // an object containing a start and an end point received from the commands of the path
        var refPoints = null;
        // the start point object received from the specified path
        var start = null;
        // the end point object received from the specified path
        var end = null;

        // helper function to get display factors for different line widths
        // values 2, 4, 6 for lineWidth < 2
        // values 1, 2, 3 for lineWidth 2 < 8
        // values 0.5, 1, 1.5 for lineWidth > 8
        function getFactor(type, wid, aType) {

            var factor = 1;
            var arrowFactor = 1;
            var ovalFactor = 1.3;

            if (wid <= 2) {  // larger values for thickness below 3 px
                factor = 3;
                if (type === 'small') {
                    factor = 2;
                } else if (type === 'large') {
                    factor = 5;
                }
            } else {
                factor = 0.75; //  1.5;
                if (type === 'small') {
                    factor = 0.5; // 1;
                } else if (type === 'large') {
                    factor = 1.25; // 2.5;
                }
            }

            // additionally increasing length and width for arrows
            if (aType === 'arrow') {
                arrowFactor = 1.5;
                if (type === 'small') {
                    arrowFactor = 1.8;
                } else if (type === 'large') {
                    arrowFactor = 1.2;
                }

                factor *= arrowFactor;
            } else if (aType === 'oval') {
                factor *= ovalFactor; // small increasement for circles and ellipses
            }

            return factor;
        }

        // helper function to get line property values for width and height in pixel
        function getLineEndAttribute(attrs, property, type) {

            var value = getFactor('medium', lineWidth, type) * lineWidth;
            if (attrs && attrs[property] === 'small') {
                value = getFactor('small', lineWidth, type) * lineWidth;
            } else if (attrs && attrs[property] === 'large') {
                value = getFactor('large', lineWidth, type) * lineWidth;
            }

            return value;
        }

        if (endType === 'head') {

            if (path.commands.length < 2) { return null; }

            lineEndPath = _.copy(path, true);
            lineEndPath.commands = [];
            allCommands = path.commands; // the array with the commands

            options = options || {};
            options.first = true;
            refPoints = getReferencePointsFromPath(allCommands, options);

            start = refPoints.start;
            end = refPoints.end;

            length = getLineEndAttribute(lineAttrs, 'headEndLength', arrowType);
            width = getLineEndAttribute(lineAttrs, 'headEndWidth', arrowType);

            if (arrowType === 'oval') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'oval', x: start.x, y: start.y, r1: length, r2: width, rpx: end.x, rpy: end.y, rpxOffset: (end.xOffset || 0) });
            } else if (arrowType === 'diamond') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'diamond', x: start.x, y: start.y, r1: length, r2: width, rpx: end.x, rpy: end.y, rpxOffset: (end.xOffset || 0) });
            } else if (arrowType === 'triangle') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'triangle', x: start.x, y: start.y, r1: length, r2: width, rpx: end.x, rpy: end.y, rpxOffset: (end.xOffset || 0) });
            } else if (arrowType === 'stealth') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'stealth', x: start.x, y: start.y, r1: length, r2: width, rpx: end.x, rpy: end.y, rpxOffset: (end.xOffset || 0) });
            } else if (arrowType === 'arrow') {
                lineEndPath.commands.push({ c: 'arrow', x: start.x, y: start.y, r1: length, r2: width, rpx: end.x, rpy: end.y, rpxOffset: (end.xOffset || 0) });
            }

        } else if (endType === 'tail') {

            if (path.commands.length < 2) { return null; }

            lineEndPath = _.copy(path, true);
            lineEndPath.commands = [];
            allCommands = path.commands; // the array with the commands

            options = options || {};
            options.first = false;
            refPoints = getReferencePointsFromPath(allCommands, options);

            start = refPoints.start;
            end = refPoints.end;

            length = getLineEndAttribute(lineAttrs, 'tailEndLength', arrowType);
            width = getLineEndAttribute(lineAttrs, 'tailEndWidth', arrowType);

            if (arrowType === 'oval') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'oval', x: end.x, y: end.y, r1: length, r2: width, rpx: start.x, rpy: start.y, rpxOffset: (start.xOffset || 0) });
            } else if (arrowType === 'diamond') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'diamond', x: end.x, y: end.y, r1: length, r2: width, rpx: start.x, rpy: start.y, rpxOffset: (start.xOffset || 0) });
            } else if (arrowType === 'triangle') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'triangle', x: end.x, y: end.y, r1: length, r2: width, rpx: start.x, rpy: start.y, rpxOffset: (start.xOffset || 0) });
            } else if (arrowType === 'stealth') {
                context.setFillStyle(color);
                fillRequired = true;
                lineEndPath.commands.push({ c: 'stealth', x: end.x, y: end.y, r1: length, r2: width, rpx: start.x, rpy: start.y, rpxOffset: (start.xOffset || 0) });
            } else if (arrowType === 'arrow') {
                lineEndPath.commands.push({ c: 'arrow', x: end.x, y: end.y, r1: length, r2: width, rpx: start.x, rpy: start.y, rpxOffset: (start.xOffset || 0) });
            }

        }

        if (lineEndPath && lineEndPath.commands && lineEndPath.commands.length > 0) {
            lineEndPath.drawMode = fillRequired ? 'all' : 'stroke';
            return lineEndPath;
        }

        return null;
    }

    function getLineWidthPx(lineAttrs) {
        return lineAttrs.width ? max(1, Utils.convertHmmToLength(lineAttrs.width, 'px', 1)) : 1;
    }

    /**
     * Sets the line attributes to the given canvas context.
     *
     * @param {Object} docModel
     *  The application model object
     *
     * @param {ContextWrapper} context
     *  Canvas context wrapper.
     *
     * @param {Object} lineAttrs
     *  Object containing line attributes for the canvas
     *
     * @param {Number} [lineWidth]
     *  Optional parameter for line width in px.
     *
     * @param {Object} [path]
     *  Optional parameter for the line path.
     *
     * @param {Object} [options]
     *  Optional parameter for the line handling:
     *  @param {Boolean} [options.isFirstPath=false]
     *      If set to true, this is the first path element of a specified path.
     *  @param {Boolean} [options.isLastPath=false]
     *      If set to true, this is the last path element of a specified path.
     *  @param {Object} [snapRect]
     *     The rectangle with the properties left, top, width and height
     *     for setting the size of the canvas.
     *  @param {Object} [avList]
     *     The used adjustment values.
     *  @param {Object[]} [gdList]
     *     An array with formulas that can be used.
     *  @param {Object} [guideCache]
     *     The cache with already calculated positioning values.
     *
     * @returns {Array<Objects>}
     *  An array containing additional paths that are necessary to draw the line
     *  endings.
     */
    function setLineAttributes(docModel, context, lineAttrs, lineWidth, path, options) {

        var lineColor = docModel.getCssColor(lineAttrs.color, 'line', lineAttrs[DrawingFrame.TARGET_CHAIN_PROPERTY]);
        var locLineWidth = lineWidth || getLineWidthPx(lineAttrs);
        var pattern = Border.getBorderPattern(lineAttrs.style, locLineWidth);

        context.setLineStyle({ style: lineColor, width: locLineWidth, pattern: pattern });

        var newPaths = [];
        if (path && lineAttrs) {
            if (Utils.getBooleanOption(options, 'isFirstPath', false) && lineAttrs.headEndType && lineAttrs.headEndType !== 'none') {
                var firstPath = handleLineEnd(context, lineAttrs, locLineWidth, lineColor, path, lineAttrs.headEndType, 'head', options);
                if (firstPath) { newPaths.push(firstPath); }
            }
            if (Utils.getBooleanOption(options, 'isLastPath', false) && lineAttrs.tailEndType && lineAttrs.tailEndType !== 'none') {
                var lastPath = handleLineEnd(context, lineAttrs, locLineWidth, lineColor, path, lineAttrs.tailEndType, 'tail', options);
                if (lastPath) { newPaths.push(lastPath); }
            }
        }
        return newPaths;
    }

    /**
     * Sets the bitmap texture fill style to the given canvas context, according to different properties.
     *
     * @param {DocumentModel}  docModel
     *  Current document model.
     * @param {ContextWrapper}  context
     *  Canvas context wrapper.
     * @param {function}  drawMethod
     *   Method which should be called for drawing canvas
     * @param {CanvasPattern}  fillStyle
     * @param {Number}  widthPx
     *  Width of shape in pixels.
     * @param {Number}  heightPx
     *  Height of shape in pixels.
     * @param {Boolean} hasLine
     *  If shape has line or not.
     * @param {Object}  lineAttrs
     *  Attribute set for the line.
     * @param {Boolean} isTiling
     *  If bitmap is tiling.
     * @param {Boolean}  notRotatedWithShape
     *  If bitmap should contain its own rotation angle and not rotate with shape.
     * @param {Number|null}  rotationAngle
     *  If exists and shape shouldn't rotate with shape, use it as counter angle for texture.
     */
    function setBitmapTextureFillStyle(docModel, context, drawMethod, fillStyle, widthPx, heightPx, hasLine, lineAttrs, isTiling, notRotatedWithShape, rotationAngle) {
        var nWidth, nHeight;

        context.clearRect(0, 0, widthPx, heightPx);
        if (notRotatedWithShape && rotationAngle) {
            var rectPos = max(widthPx, heightPx);
            // initialize context with fill to mask path with pattern
            context.setFillStyle('#000000');
            drawMethod({ mode: 'fill' });

            // local transformations will be reset after the callback has been executed
            context.render(function () {

                context.translate(widthPx / 2, heightPx / 2);
                context.rotate(rotationAngle);
                context.setFillStyle(fillStyle);
                context.setGlobalComposite('source-in'); // The new shape is drawn only where both the new shape and the destination canvas overlap. Everything else is made transparent.

                if (notRotatedWithShape && !isTiling) {
                    nWidth = widthPx * abs(cos(rotationAngle)) + heightPx * abs(sin(rotationAngle));
                    nHeight = heightPx * abs(cos(rotationAngle)) + widthPx * abs(sin(rotationAngle));
                    context.translate(-nWidth / 2, -nHeight / 2);
                    context.drawRect(0, 0, nWidth, nHeight, 'fill'); // cover whole canvas area with pattern fill
                } else {
                    context.drawRect(2 * -rectPos, 2 * -rectPos, 4 * rectPos, 4 * rectPos, 'fill'); // cover whole canvas area with pattern fill
                }
            });

            if (hasLine) {
                setLineAttributes(docModel, context, lineAttrs);
            }
            drawMethod({ mode: 'stroke' }); // stroke needs to be painted again after overlap
        } else {
            context.setFillStyle(fillStyle);
            // initialize context with proper line attributes for the next path
            if (hasLine) {
                setLineAttributes(docModel, context, lineAttrs);
            }
            drawMethod();
        }
    }

    /**
     * fix for Bug 48477 explicit style-entries contentNode got lost,
     * so we changed to css-classes
     * html-attributes (interpredet by css would be also a nice solution)
     */
    function handleFlipping(docModel, drawingFrame, drawingAttributes) {

        var angle = drawingAttributes.rotation || 0;
        var textFrameNode = DrawingFrame.getOriginalTextFrameNode(drawingFrame);
        var isFlipH = false;
        var isFlipV = false;

        if (drawingAttributes) {
            // convert to single jQuery object
            drawingFrame = $(drawingFrame).first();

            isFlipH = drawingAttributes.flipH;
            isFlipV = drawingAttributes.flipV;
        }
        DrawingFrame.setCssTransform(drawingFrame, angle, isFlipH, isFlipV);

        drawingFrame.toggleClass(DrawingFrame.FLIPPED_HORIZONTALLY_CLASSNAME, isFlipH);
        drawingFrame.toggleClass(DrawingFrame.FLIPPED_VERTICALLY_CLASSNAME, isFlipV);

        textFrameNode.toggleClass(DrawingFrame.FLIPPED_HORIZONTALLY_CLASSNAME, DrawingFrame.isOddFlipHVCount(textFrameNode));
    }

    // updating the size attributes at an image inside a drawing (also supported inside a drawing group)
    function setImageCssSizeAttributes(app, imageParent, drawingWidth, drawingHeight, drawingAttrs, imageAttrs) {

        var docModel = app.getModel(),
            // IE requires to swap cropping settings if image is flipped
            swapHorCrop = _.browser.IE && drawingAttrs.flipH,
            swapVertCrop = _.browser.IE && drawingAttrs.flipV,
            // effective cropping values (checking for 0, because explicit attributes might be used)
            cropLeft = swapHorCrop ? (imageAttrs.cropRight || 0) : (imageAttrs.cropLeft || 0),
            cropRight = swapHorCrop ? (imageAttrs.cropLeft || 0) : (imageAttrs.cropRight || 0),
            cropTop = swapVertCrop ? (imageAttrs.cropBottom || 0) : (imageAttrs.cropTop || 0),
            cropBottom = swapVertCrop ? (imageAttrs.cropTop || 0) : (imageAttrs.cropBottom || 0),
            // horizontal offset/size of cropped bitmaps, as CSS attributes
            horizontalSettings = calculateBitmapSettings(app, drawingWidth, cropLeft, cropRight),
            // vertical offset/size of cropped bitmaps, as CSS attributes
            verticalSettings = calculateBitmapSettings(app, drawingHeight, cropTop, cropBottom);

        // workaround for Bug 48435
        if (cropLeft < 0 && docModel.useSlideMode()) { horizontalSettings.offset = Utils.convertHmmToCssLength(cropLeft, 'px', 1); }

        imageParent.find('img').css({
            left: horizontalSettings.offset,
            top: verticalSettings.offset,
            width: horizontalSettings.size,
            height: verticalSettings.size
        });
    }

    /**
     * Updates the fill formatting of the drawing frame.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     * @param {jQuery} [node]
     *  An optional jQuery node, to which the css properties are
     *  assigned. If it is not set, the content node inside the
     *  drawing frame is used. Setting this node, is only required
     *  for drawing groups, where all children need to be modified.
     * @param {Object} mergedAttributes
     *  Merged attributes of the given drawing node.
     */
    function updateFillFormatting(app, contentNode, mergedAttributes) {

        var // the document model
            docModel = app.getModel(),
            // the CSS properties for the drawing frame
            cssProps = {},
            // the node, whose properties will be set
            cssNode = contentNode,
            // the fill attributes
            fillAttrs = mergedAttributes.fill;

        if (fillAttrs) {
            // calculate the CSS fill attributes
            switch (fillAttrs.type) {
                case 'none':
                    // clear everything: color and bitmaps
                    cssProps.background = '';
                    break;
                case 'solid':
                    // use 'background' compound attribute to clear bitmaps
                    cssProps.background = docModel.getCssColor(fillAttrs.color, 'fill', fillAttrs[DrawingFrame.TARGET_CHAIN_PROPERTY]);
                    break;
                default:
                    Utils.warn('DrawingFrame.updateFillFormatting(): unknown fill type "' + fillAttrs.type + '"');
            }

            // apply the fill attributes
            cssNode.css(cssProps);
        }
    }

    // helper function to get all node attributes (TODO: This needs to be valid for all applications)
    function getAllDrawingAttributes(app, node, family) {

        var // the drawing style collection
            styleCollection = app.getModel().getStyleCollection('drawing'),
            // the element specific attributes
            elementAttributes = styleCollection ? styleCollection.getElementAttributes(node) : null,
            // the family specific attributes of the element
            familyAttributes = elementAttributes ? elementAttributes[family] : null;

        return familyAttributes;
    }

    /**
     * Updates the border formatting of the drawing frame.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {jQuery} node
     *  jQuery node, to which the css properties are
     *  assigned. If it is not used, the content node inside the
     *  drawing frame is passed. Setting this node, is only required
     *  for drawing groups, where all children need to be modified.
     *
     * @param {Object} mergedAttributes
     *  Merged attributes of the given drawing node.
     *
     * @param {Object} [options]
     *  @param {Boolean} [options.useExplicitAttrs=false]
     *      If explicit drawing attributes should be used
     */
    function updateLineFormatting(app, node, mergedAttributes, options) {

        var // gets the correct node to work on (parent() used at grouped drawings)
            currentNode = DrawingFrame.getDrawingNode(node),
            // if explicit drawing attributes should be used
            useExplicit = Utils.getBooleanOption(options, 'useExplicitAttrs', false),
            // the line attributes
            lineAttrs = useExplicit ? AttributeUtils.getExplicitAttributes(currentNode).line : mergedAttributes.line;

        // Explicit attributes are not sufficient, if default line attributes are used (36892)
        if (currentNode && useExplicit && (!lineAttrs || !lineAttrs.type)) { lineAttrs = getAllDrawingAttributes(app, node, 'line'); }

        if (lineAttrs && lineAttrs.type) {
            // calculate the border properties
            switch (lineAttrs.type) {
                case 'none':
                    // clear everything
                    clearCanvas(app, currentNode);
                    break;
                case 'solid':
                    DrawingFrame.drawBorder(app, currentNode, lineAttrs);
                    break;
                default:
                    Utils.warn('DrawingFrame.updateLineFormatting(): unknown line type "' + lineAttrs.type + '"');
            }
        }
    }

    /**
     * Handling an optionally required div-element in those drawings, that contain
     * a canvas node larger than the drawing itself. This helper drawing allows moving
     * and resizing and acts like a filter to catch events for the drawing.
     *
     * @param {jQuery} node
     *  The drawing node.
     *
     * @param {Number|Number[]} expansion
     *  The object that contains the information about the size of the canvas.
     */
    function handleDrawingExpansionNode(node, expansion) {

        var // an optionally already existing expansion node
            oldExpansionNode = null,
            // an optionally required new expansion node
            expansionNode = null,
            // the minimum width and height for a drawing without expansion node in px
            minValue = 8,
            // the width and height of the drawing node
            width = 0, height = 0;

        // helper function to calculate the required size for the
        function calculateDrawingExpansion(drawingNode, exp) {

            var width = drawingNode.width();
            var height = drawingNode.height();
            var left = 0;
            var top = 0;

            if (_.isNumber(exp)) {
                left = -exp;
                top = -exp;
                width += (2 * exp);
                height += (2 * exp);
            } else if (_.isObject(exp)) {
                left = -exp.left;
                top = -exp.top;
                width = width + exp.left + exp.right;
                height = height + exp.top + exp.bottom;
            }

            return { left: left, top: top, width: width, height: height };
        }

        oldExpansionNode = node.children(DrawingFrame.CANVAS_EXPANSION_SELECTOR); // searching an already existing child node

        if (!expansion) {

            width = node.width();
            height = node.height();

            if (width < minValue || height < minValue) {
                expansion = minValue - min(width, height); // setting expansion also for thin lines
            }
        }

        if (expansion) {
            expansionNode = (oldExpansionNode.length > 0) ? oldExpansionNode : $('<div class="' + DrawingFrame.CANVAS_EXPANSION_CLASS + '">');
            expansionNode.css(calculateDrawingExpansion(node, expansion));
            if (oldExpansionNode.length === 0) { node.append(expansionNode); }
            // node.addClass(DrawingFrame.CONTAINS_CANVAS_EXPANSION); // Setting marker at the drawing node itself
        } else {
            if (oldExpansionNode.length > 0) { oldExpansionNode.remove(); }
            // node.removeClass(DrawingFrame.CONTAINS_CANVAS_EXPANSION); // Setting marker at the drawing node itself
        }

    }

    /**
     * Updating the 'shape' attributes at the drawing.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {jQuery} drawingFrame
     *  The DOM drawing frame.
     *
     * @param {Object} mergedAttributes
     *  Merged attributes of the drawing node.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.isResizeActive=false]
     *      If this method is called during resizing of the shape.
     *  @param {Boolean} [options.textFrameOnly=false]
     *      If set to true, only the position and size of the text frame will
     *      be updated.
     *  @param {Boolean} [options.useExplicitAttrs=false]
     *      If explicit drawing attributes should be used
     */
    function updateShapeFormatting(app, drawingFrame, mergedAttributes, options) {
        var // the document model
            docModel = app.getModel(),
            // if formatting is called during resize
            isResizeActive = Utils.getBooleanOption(options, 'isResizeActive', false),
            // if explicit drawing attributes should be used
            useExplicit = Utils.getBooleanOption(options, 'useExplicitAttrs', false),
            // whether the update is triggered from updateTextFramePosition function (reduced rendering)
            textFrameOnlyParam = Utils.getBooleanOption(options, 'textFrameOnly', false),
            // whether the drawing frame is used as internal model container (reduced rendering)
            internalModel = textFrameOnlyParam || drawingFrame.hasClass(DrawingFrame.INTERNAL_MODEL_CLASS),
            // gets the correct node to work on (parent() used at grouped drawings)
            currentNode = DrawingFrame.getDrawingNode(drawingFrame),
            // the content node inside the drawing frame
            contentNode = DrawingFrame.getContentNode(currentNode, options), // needs to be fetched each time, because of grouped drawings, see #47962
            // the explicit attributes of the handled node
            explicitAttributes = AttributeUtils.getExplicitAttributes(currentNode),
            // the shape attributes
            shapeAttrs = useExplicit ? explicitAttributes.shape : mergedAttributes.shape,
            // the line attributes
            lineAttrs = useExplicit ? explicitAttributes.line : mergedAttributes.line,
            // the fill attributes
            fillAttrs = useExplicit ? explicitAttributes.fill : mergedAttributes.fill,
            // shape geometry
            geometryAttrs = explicitAttributes.geometry ? explicitAttributes.geometry : mergedAttributes.geometry,
            // the canvas wrapper
            canvas = null,
            // copied node of textframecontent
            textFrameContentCopy = currentNode.children('.copy'),
            // the text frame node of the drawing
            textFrame = isResizeActive ? textFrameContentCopy.children('.textframe') : DrawingFrame.getTextFrameNode(currentNode),
            // if no word wrap property is set
            noWordWrap = docModel.useSlideMode() ? (shapeAttrs && shapeAttrs.wordWrap === false) : false, // not supported in OX Text
            // variables for storing current width and height
            currentHeight = null,
            currentWidth = null,
            // whether autoResizeHeight is enabled for the drawing
            autoResizeHeight = DrawingFrame.isAutoResizeHeightAttributes(shapeAttrs);

        // depending on angle, orientate text (horz/vert/vert reversed) inside shape
        function orientateTextInShape(angle) {
            // width and height of shape node
            var nodeWidth, nodeHeight, maxHeight;
            // delta values to normalize top and left positions of rotated textframe inside shape
            var topDelta, leftDelta;
            // deviation from default rotation
            var notDefaultAngle = angle !== 0;

            currentNode.data({ widthHeightRev: notDefaultAngle });
            if (notDefaultAngle) {
                nodeWidth = isResizeActive ? textFrameContentCopy.width() : currentNode.width();
                nodeHeight = isResizeActive ? textFrameContentCopy.height() : currentNode.height();
                maxHeight = textFrame[0].style.maxHeight;
                leftDelta = (nodeWidth - nodeHeight) / 2;
                topDelta = leftDelta * -1;
                textFrame
                    .css({ width: nodeHeight,
                        height: nodeWidth,
                        maxHeight: '',
                        maxWidth: maxHeight,
                        top: topDelta,
                        left: leftDelta,
                        transform: 'rotate(' + angle + 'deg)' })
                    .addClass(DrawingFrame.ROTATED_TEXT_IN_DRAWING_CLASSNAME);
            } else {
                textFrame.css({ transform: 'rotate(0deg)' }).removeClass(DrawingFrame.ROTATED_TEXT_IN_DRAWING_CLASSNAME);
            }
        }

        /**
         * Getting width and height of the drawing.
         *
         * @param {Object} [options]
         *  Optional parameters:
         *  @param {Boolean} [options.getminimumheight=false]
         *      If the drawing grows or shrinks with its content ('autoResizeHeight' is 'true'), it is necessary to receive the minimum height
         *      of the drawing content (52953). This is the height of the text frame (without padding), not the height of the drawing itself.
         *      This value is only evaluated, if 'autoResizeHeight' is 'true'.
         *      The default is, that the height of the drawing node is returned.
         */
        function getSnapRect(options) {
            var snapRect = new Rectangle(0, 0, 0, 0);
            if (isResizeActive) {
                snapRect.width = textFrameContentCopy.outerWidth(true);
                snapRect.height = textFrameContentCopy.outerHeight(true);
            } else {
                snapRect.width = noWordWrap ? textFrame.outerWidth(true) : currentNode.first().width();
                snapRect.height = (autoResizeHeight && Utils.getBooleanOption(options, 'getminimumheight', false)) ? textFrame.first().height() : currentNode.first().height();
            }
            return snapRect;
        }

        function drawGeometry(geometry, snapRect, boundRect) {
            if (!docModel || docModel.destroyed) { return; } // the model might have been destroyed in the meantime

            function getTextRectValue(value, defaultValue) {
                return value ? Utils.round(getGeometryValue(value, snapRect, geometry.avList, geometry.gdList, guideCache), 1) : defaultValue;
            }

            var visibilityChanged = false;
            var lineWidth = 1;
            var expansion = 0; // additional canvas size outside the snap rect
            var guideCache = {}; // caching the already calculated guide variables.
            var presetShape = (geometry && geometry.presetShape) ? geometry.presetShape : ''; // the value of the preset shape property
            var maxSizeScale = 0;

            if (lineAttrs && lineAttrs.width) { lineWidth = getLineWidthPx(lineAttrs); }

            // taking care of special text rects
            if (textFrame.length > 0) {
                var left = getTextRectValue(geometry.textRect ? geometry.textRect.l : null, snapRect.left) + Utils.convertHmmToLength(mergedAttributes.shape.paddingLeft, 'px');
                var top = getTextRectValue(geometry.textRect ? geometry.textRect.t : null, snapRect.top) + Utils.convertHmmToLength(mergedAttributes.shape.paddingTop, 'px');
                var right = getTextRectValue(geometry.textRect ? geometry.textRect.r : null, snapRect.left + snapRect.width) - Utils.convertHmmToLength(mergedAttributes.shape.paddingRight, 'px');

                var paddingBottom = (snapRect.height - getTextRectValue(geometry.textRect ? geometry.textRect.b : null, (snapRect.top + snapRect.height) - 1))
                                    + top
                                    // + Utils.convertHmmToLength(mergedAttributes.shape.paddingTop, 'px') /* commented out for Bug 51824 - Shape autofit: to much space at the bottom of shape */
                                    + Utils.convertHmmToLength(mergedAttributes.shape.paddingBottom, 'px');

                if (docModel.useSlideMode()) {
                    // workaround for Bug 48476, drawings in powerpoint are 2px bigger than they save in document, reduce padding is easier than changing all sizes
                    left -= 1; right += 2;
                }

                left = round(left);
                top = round(top);
                right = round(right);
                paddingBottom = round(paddingBottom);

                // workarounds for vertical aligment @see drawingstyle.less div.textframe
                // Bug 48665, Bug 46803, Bug 42893, Bug 44965, Bug 52808, Bug 52226 & Bug 51824

                textFrame.css({ left: left + 'px', top: top + 'px', width: (right - left) + 'px', height: 'calc(100% - ' + paddingBottom + 'px)', marginBottom: paddingBottom + 'px' });
                if (noWordWrap) {
                    textFrame.css({ position: 'relative', width: 'auto', left: '', marginLeft: left, marginRight: left, whiteSpace: 'nowrap' });
                }

                // IE 11 has focus problems with combination of contenteditable & flexbox
                textFrame.toggleClass('no-flex', _.browser.IE <= 11);

                // restored this code from history for Bug 51185
                // the textframe been changed so also the snaprect, we have to reset the guideCache and also
                // reset the new snaprect
                guideCache = {};

                if (docModel.handleContainerVisibility) {
                    visibilityChanged = docModel.handleContainerVisibility(currentNode.first(), { makeVisible: true });
                }
                snapRect = getSnapRect();  // getting snap rect again, this time using current node also if autoResizeHeight is true
                if (visibilityChanged && docModel.handleContainerVisibility) {
                    docModel.handleContainerVisibility(currentNode.first(), { makeVisible: false });
                }
            }

            // nothing more to do when updating an internal model frame
            if (internalModel) { return; }

            if (!PRESET_GEOMETRIES[presetShape] || Labels.CANVAS_EXPANSION_PRESET_SHAPES[presetShape]) {
                expansion = getCanvasExpansion(geometry, snapRect, guideCache, expansion);
            }

            if (lineAttrs && geometry) { // adding the canvas expansion required by line ends
                expansion = getCanvasExpansionForLineEnds(lineAttrs, geometry, expansion);
            }

            if (expansion) { currentNode.data('canvasexpansion', expansion); }

            if (!isResizeActive) { handleDrawingExpansionNode(currentNode, expansion); } // handle an additional child node, if canvas exceeds the drawing size to simplify selection

            // bug 48101: reduce canvas size to supported dimensions
            maxSizeScale = max(snapRect.width, snapRect.height) / MAX_CANVAS_SIZE;
            if (maxSizeScale > 1) {
                snapRect.width = round(snapRect.width / maxSizeScale);
                snapRect.height = round(snapRect.height / maxSizeScale);
            }

            if (_.isNumber(expansion)) {
                boundRect = snapRect.clone().expandSelf(expansion);
            } else if (_.isObject(expansion)) {
                boundRect = snapRect.clone().expandSelf(expansion.left, expansion.top, expansion.right, expansion.bottom);
            } else {
                boundRect = snapRect.clone();
            }

            canvas = initializeCanvas(app, currentNode, boundRect, lineWidth, isResizeActive ? { resizeIsActive: true } : null);
            if (maxSizeScale > 1) { canvas.getNode().css({ width: boundRect.width * maxSizeScale, height: boundRect.height * maxSizeScale, left: boundRect.left * maxSizeScale, top: boundRect.top * maxSizeScale }); } // #52379

            canvas.render(function (context) {
                if (('pathList' in geometry) && (geometry.pathList.length > 0)) {
                    drawShape(app, context, snapRect, geometry, { fill: fillAttrs, line: lineAttrs }, guideCache, { isResizeActive: isResizeActive, currentNode: currentNode });
                    return;
                }

                var hasFill = fillAttrs && (fillAttrs.type !== 'none');
                var hasLine = lineAttrs && (lineAttrs.type !== 'none');
                var renderMode = getRenderMode(hasFill, hasLine);
                var rotationAngle = null;
                var notRotatedWithShape = null;

                function drawRect(options) {
                    context.drawRect(boundRect, Utils.getStringOption(options, 'mode', renderMode));
                }

                if (hasFill) {
                    var targets = fillAttrs[DrawingFrame.TARGET_CHAIN_PROPERTY];

                    if (fillAttrs.type === 'gradient') {
                        var gradient = Gradient.create(Gradient.getDescriptorFromAttributes(fillAttrs));
                        context.setFillStyle(docModel.getGradientFill(gradient, boundRect, context, targets));
                    } else if (fillAttrs.type === 'pattern') {
                        context.setFillStyle(Pattern.createCanvasPattern(fillAttrs.pattern, fillAttrs.color2, fillAttrs.color, docModel.getThemeModel(targets)));
                    } else if (fillAttrs.type === 'bitmap') {
                        notRotatedWithShape = fillAttrs.bitmap.rotateWithShape === false;
                        if (notRotatedWithShape && currentNode) {
                            rotationAngle = DrawingFrame.getDrawingRotationAngle(docModel, currentNode) || 0;
                            rotationAngle = ((360 - rotationAngle) % 360) * PI_180; // normalize and convert to radians
                        }
                        Texture.getTextureFill(docModel, context, fillAttrs, snapRect.width, snapRect.height, rotationAngle).done(function (textureFillStyle) {
                            setBitmapTextureFillStyle(docModel, context, drawRect, textureFillStyle, snapRect.width, snapRect.height, hasLine, lineAttrs, fillAttrs.bitmap.tiling, notRotatedWithShape, rotationAngle);
                        });
                        return;
                    } else {
                        context.setFillStyle(docModel.getCssColor(fillAttrs.color, 'fill', targets));
                    }
                }

                if (hasLine) {
                    setLineAttributes(docModel, context, lineAttrs, lineWidth);
                }

                if (renderMode) {
                    context.drawRect(boundRect, renderMode);
                }
            });
        }

        // Handling the target chain inheritance also for grouped drawings (48096)
        // Or during active resizing of shape to fetch correct color from themes
        if (DrawingFrame.isGroupedDrawingFrame(drawingFrame) || isResizeActive) {
            DrawingFrame.handleTargetChainInheritance(docModel, drawingFrame, explicitAttributes);

            // Fixing problem with grouped shapes, that have no explicit fill color set. In ODF the color of the
            // default drawing style is used then. Info: The mergedAttributes are the merged attributes of the
            // group in that case!
            if (app.isODF() && useExplicit) {
                if (!lineAttrs) { lineAttrs = mergedAttributes.line; }
                if (!fillAttrs) { fillAttrs = mergedAttributes.fill; }
            }
        }

        // applying text padding & vertical alignment
        if (shapeAttrs && (textFrame.length > 0)) {
            if (shapeAttrs.anchor) {
                textFrame.attr(DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE, shapeAttrs.anchor);
            }
            if (!geometryAttrs) {
                textFrame.css({
                    paddingTop: Utils.convertHmmToCssLength(mergedAttributes.shape.paddingTop, 'px', 1),
                    paddingLeft: Utils.convertHmmToCssLength(mergedAttributes.shape.paddingLeft, 'px', 1),
                    paddingBottom: Utils.convertHmmToCssLength(mergedAttributes.shape.paddingBottom, 'px', 1),
                    paddingRight: Utils.convertHmmToCssLength(mergedAttributes.shape.paddingRight, 'px', 1)
                });
            }
        }

        // setting the value for automatic vertical resizing of the shape.
        if (DrawingFrame.isAutoResizeHeightAttributes(shapeAttrs)) {
            if (currentNode.data('widthHeightRev')) {
                textFrame.css('height', 'auto');
                currentNode.width(textFrame.outerHeight(true));
                currentNode.height(textFrame.outerWidth(true));
            }
            if (!currentNode.data('drawingHeight') && !DrawingFrame.isGroupContentNode(currentNode.parent())) {
                currentHeight = (explicitAttributes && explicitAttributes.drawing && Utils.convertHmmToLength(explicitAttributes.drawing.height, 'px', 1)) || currentNode.height();
                currentNode.data('drawingHeight', currentHeight);
            }
            contentNode.addClass(DrawingFrame.AUTORESIZEHEIGHT_CLASS);
            // updating the height of a surrounding group node
            if (DrawingFrame.isGroupedDrawingFrame(contentNode.parent())) { DrawingFrame.updateDrawingGroupHeight(app, contentNode.parent()); }
        } else {
            contentNode.removeClass(DrawingFrame.AUTORESIZEHEIGHT_CLASS);
            currentNode.data('drawingHeight', null);
            //if (text app) {}
            if (!docModel.useSlideMode() && (_.isEmpty(geometryAttrs) || _.isEmpty(geometryAttrs.presetShape) || geometryAttrs.presetShape === 'rect')) {
                DrawingFrame.addHintArrowInTextframe(docModel, currentNode);
            }
        }

        if (docModel.useSlideMode()) {
            // setting the value for automatic text height in the shape.
            if (DrawingFrame.isAutoTextHeightAttributes(shapeAttrs)) {
                contentNode.css('font-size', (shapeAttrs.fontScale * 16) + 'px');
                contentNode.attr('font-scale', shapeAttrs.fontScale);
                contentNode.attr('lineReduction', round(min(0.2, shapeAttrs.lineReduction) * 100));
                contentNode.addClass(DrawingFrame.AUTORESIZETEXT_CLASS);
            } else {
                contentNode.css('font-scale', null);
                contentNode.css('font-size', null);
                contentNode.attr('lineReduction', null);
                contentNode.removeClass(DrawingFrame.AUTORESIZETEXT_CLASS);
            }
            if (noWordWrap) {
                textFrame.css('width', 'auto');
                if (DrawingFrame.isAutoResizeHeightAttributes(shapeAttrs)) { // #47690, #48153
                    currentNode.css('width', 'auto');
                }
                contentNode.addClass(DrawingFrame.NO_WORDWRAP_CLASS);
                if (!currentNode.data('drawingWidth') && !DrawingFrame.isGroupContentNode(currentNode.parent())) {
                    currentWidth = (explicitAttributes && explicitAttributes.drawing && Utils.convertHmmToLength(explicitAttributes.drawing.width, 'px', 1)) || currentNode.width();
                    currentNode.data('drawingWidth', currentWidth);
                }
            } else {
                contentNode.removeClass(DrawingFrame.NO_WORDWRAP_CLASS);
                currentNode.data('drawingWidth', null);
                textFrame.css('white-space', '');
            }
        }

        if (geometryAttrs) {
            // bugfix :: BUG#45141 :: https://bugs.open-xchange.com/show_bug.cgi?id=45141 :: Vertical drawing object is shown diagonally
            if (geometryAttrs.presetShape === 'line') {
                if (currentNode.width() === 0) {
                    currentNode.width(1);
                }
                if (currentNode.height() === 0) {
                    currentNode.height(1);
                }
            }
            // the canvas wrapper of the border node
            var snapRect = getSnapRect({ getminimumheight: true });
            var boundRect = snapRect.clone();

            if (('presetShape' in geometryAttrs) && PRESET_GEOMETRIES[geometryAttrs.presetShape]) {
                var presetGeo = PRESET_GEOMETRIES[geometryAttrs.presetShape];
                var geo = _.extend({}, geometryAttrs, presetGeo);

                if (!_.isEmpty(geometryAttrs.avList)) { // explicit attributes in avList must overwrite again
                    geo.avList = _.extend({}, geo.avList, geometryAttrs.avList);
                }

                drawGeometry(geo, snapRect, boundRect);
            } else {
                drawGeometry(geometryAttrs, snapRect, boundRect);
            }
        } else if (!internalModel) {
            // do not format internal model frames
            updateFillFormatting(app, contentNode, mergedAttributes);
            updateLineFormatting(app, drawingFrame, mergedAttributes, { useExplicitAttrs: true });
        }

        // applying text direction and stacking
        if (shapeAttrs && shapeAttrs.vert && (textFrame.length > 0)) {
            switch (shapeAttrs.vert) {
                case 'vert':
                case 'eaVert':
                    orientateTextInShape(90); // vertical (90deg)
                    break;
                case 'vert270':
                    orientateTextInShape(270); // vertical reversed (270deg)
                    break;
                case 'horz':
                    if (DrawingFrame.isRotatedTextInDrawingNode(textFrame)) {
                        orientateTextInShape(0); // revert from vertical to horizontal
                    }
                    break;
            }
        }

        return true;
    }

    /**
     * Updates the size and position of all children of a drawing group.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {jQuery} drawing
     *  Drawing node in jQuery format
     *
     * @param {Object} [options]
     *  @param {Boolean} [options.resizeActive=false]
     *      If function is called during resizing of drawing
     */
    function resizeDrawingsInGroup(app, drawing, mergedAttributes, options) {

        var // the document model
            docModel = app.getModel(),
            // if resize is active, calculate live dimensions, not from attributes
            isResizeActive = Utils.getBooleanOption(options, 'resizeActive', false),
            // copied content of the drawing
            $content = DrawingFrame.isGroupedDrawingFrame(drawing) ? drawing.children('.content') : drawing.children('.copy'),
            // the direct drawing children of the drawing group
            allChildren = isResizeActive ? $content.children(DrawingFrame.NODE_SELECTOR) : DrawingFrame.getAllGroupDrawingChildren(drawing, { onlyDirectChildren: true }),
            // the maximum width of the children
            maxWidth = 0,
            // the maximum height of the children
            maxHeight = 0,
            // width of the group
            groupWidth = 0,
            // height of the group
            groupHeight = 0,
            // all horizontal offsets
            allHorzOffsets = [],
            // all vertical offsets
            allVertOffsets = [],
            // the horizontal offset of one drawing
            offsetHorz = 0,
            // the vertical offset of one drawing
            offsetVert = 0,
            // the horizontal stretch factor for the children
            horzFactor = 1,
            // the vertical stretch factor for the children
            vertFactor = 1,
            // zoom factor of the application
            zoomFactor = app.getView().getZoomFactor(),
            // rotation angle of the drawing group element
            rotationAngle = DrawingFrame.getDrawingRotationAngle(docModel, drawing),
            // collection of drawing attributes for each drawing in group
            drawingAttrsCollection = [],
            // if rotation angle is set
            isRotationAngle = _.isNumber(rotationAngle) && rotationAngle !== 0,
            // if drawing is flipped horizontally
            isFlipH = DrawingFrame.isFlippedHorz(drawing),
            // if drawing is flipped vertically
            isFlipV = DrawingFrame.isFlippedVert(drawing),
            // if some of transformations is applied to the drawing
            isTransformed = isRotationAngle || isFlipH || isFlipV;

        function getGroupWidthHeight() {
            var groupWidth, groupHeight;

            if (isResizeActive) {
                var contentBoundingRect = $content[0].getBoundingClientRect();
                groupWidth = Utils.convertLengthToHmm(contentBoundingRect.width, 'px');
                groupHeight = Utils.convertLengthToHmm(contentBoundingRect.height, 'px');
            } else {
                // whether the explicit drawing attributes can be used (this is not the case for groups in groups, where
                // the explicit attributes do not contain values in hmm, but in an arbitrary unit).
                var isToplevelGroup = !DrawingFrame.isDrawingContentNode(drawing.parent());  // Info: 'grouped' not set for groups in groups in first run (46784)
                // the explicit attributes at the group node (should be used for precision)
                var explicitGroupAttributes = isToplevelGroup ? AttributeUtils.getExplicitAttributes(drawing) : null;
                // the explicit drawing attributes
                var explicitDrawingAttributes = explicitGroupAttributes && explicitGroupAttributes.drawing;
                // the width of the drawing group element in 1/100 mm
                groupWidth = (explicitDrawingAttributes && _.isNumber(explicitDrawingAttributes.width)) ? explicitDrawingAttributes.width : Utils.convertLengthToHmm(drawing.children().first().width(), 'px');
                // the height of the drawing group element in  1/100 mm
                groupHeight = (explicitDrawingAttributes && _.isNumber(explicitDrawingAttributes.height)) ? explicitDrawingAttributes.height : Utils.convertLengthToHmm(drawing.children().first().height(), 'px');
            }
            return { width: groupWidth, height: groupHeight };
        }

        function getUnrotatedDimensions(oneDrawing) {
            var boundRect;
            var $oneDrawing = $(oneDrawing);
            var degrees = DrawingFrame.getDrawingRotationAngle(docModel, oneDrawing) || 0;
            var isFlipH = DrawingFrame.isFlippedHorz($oneDrawing);
            var isFlipV = DrawingFrame.isFlippedVert($oneDrawing);
            var isTransformed = (degrees !== 0) || isFlipH || isFlipV;
            var moveBox, moveBoxFlipH, moveBoxFlipV, transformProp = null;

            // if drawing is part of a group, and that group is beeing flipped while resizing,
            // it needs to be reset, for fetching proper left/top pos of one drawing
            if (DrawingFrame.isGroupedDrawingFrame(oneDrawing)) {
                moveBox = $oneDrawing.parentsUntil('.slide', '.copy');
                moveBoxFlipH = DrawingFrame.isFlippedHorz(moveBox);
                moveBoxFlipV = DrawingFrame.isFlippedVert(moveBox);
                transformProp = 'scaleX(' + (moveBoxFlipH ? -1 : 1) + ') scaleY(' + (moveBoxFlipV ? -1 : 1) + ')';
                moveBox.css('transform', '');
                if (moveBoxFlipH) { moveBox.removeClass('flipH'); }
                if (moveBoxFlipV) { moveBox.removeClass('flipV'); }
            }
            if (isTransformed) {
                $oneDrawing.css('transform', ''); // reset to def to get values
            }
            boundRect = oneDrawing.getBoundingClientRect();
            if (isTransformed) {
                DrawingFrame.setCssTransform($oneDrawing, degrees, isFlipH, isFlipV);
            }
            if (DrawingFrame.isGroupedDrawingFrame(oneDrawing)) { // and return back to previous values
                moveBox.css('transform', transformProp);
                if (moveBoxFlipH) { moveBox.addClass('flipH'); }
                if (moveBoxFlipV) { moveBox.addClass('flipV'); }
            }

            return { width: Utils.convertLengthToHmm(boundRect.width, 'px'), height: Utils.convertLengthToHmm(boundRect.height, 'px'), top: Utils.convertLengthToHmm(boundRect.top, 'px'), left: Utils.convertLengthToHmm(boundRect.left, 'px') };
        }

        // updates the size and position of one drawing child inside a drawing group
        function handleDrawingInsideDrawingGroup(drawing, attrs, horzFactor, vertFactor, offsetX, offsetY, index) {

            var // the attributes at the drawing property
                drawingAttrs = isResizeActive ? drawingAttrsCollection[index] : attrs.drawing,
                // the required css attributes
                cssAttrs = {},
                // an optional horizontal resize factor
                horzResizeFactor = horzFactor || 1,
                // an optional vertical resize factor
                vertResizeFactor = vertFactor || 1,
                // type of the drawing object inside the group: 'image', ...
                type = DrawingFrame.getDrawingType(drawing);

            // updates attributes of the image node of a drawing frame inside a drawing group container
            function updateImageAttributesInGroup() {

                var // current width of the image, in 1/100 mm
                    drawingWidth = max(Utils.round(drawingAttrs.width * horzResizeFactor, 1), 100),
                    // current height of the image, in 1/100 mm
                    drawingHeight = max(Utils.round(drawingAttrs.height * vertResizeFactor, 1), 100);

                setImageCssSizeAttributes(app, drawing.children().first(), drawingWidth, drawingHeight, attrs.drawing, attrs.image);
            }

            if (!attrs.drawing) { return; } // drawing attributes must be defined (47633)

            // set the CSS attributes for the drawing
            if (_.isNumber(drawingAttrs.left))  { cssAttrs.left     = Utils.convertHmmToCssLength((drawingAttrs.left - offsetX) * horzResizeFactor, 'px', 1); }
            if (_.isNumber(drawingAttrs.top))   { cssAttrs.top      = Utils.convertHmmToCssLength((drawingAttrs.top - offsetY) * vertResizeFactor, 'px', 1); }
            if (_.isNumber(drawingAttrs.width)) { cssAttrs.width    = max(Utils.convertHmmToLength(drawingAttrs.width * horzResizeFactor, 'px', 1), 1) + 'px'; }
            if (_.isNumber(drawingAttrs.height) && !DrawingFrame.isAutoResizeHeightDrawingFrame(drawing)) { // not setting height for auto resize drawings
                cssAttrs.height = max(Utils.convertHmmToLength(drawingAttrs.height * vertResizeFactor, 'px', 1), 1) + 'px';
            }

            drawing.css(cssAttrs);

            // updating the size of the images inside the drawing
            if (type === 'image') { updateImageAttributesInGroup(drawing); }

            // updating the canvas (border and background)
            if (type !== 'group') { updateShapeFormatting(app, drawing, mergedAttributes, { useExplicitAttrs: true }); }

            // handling groups in groups
            if (DrawingFrame.isGroupDrawingFrame(drawing)) {
                resizeDrawingsInGroup(app, drawing, mergedAttributes, options);
            }
        }

        if (drawing.hasClass('activedragging') && !isResizeActive && !drawing.hasClass('activeresizing')) {
            return; // if updateShape was triggered by move operation, resizing inside groups is unnecessary
        }

        if (isResizeActive && isTransformed) {
            drawing.css({ transform: '' }); // reset to def to get values
        }
        var groupWidthHeight = getGroupWidthHeight();
        groupWidth = groupWidthHeight.width;
        groupHeight = groupWidthHeight.height;

        // iterating over all drawing children of the group
        if (allChildren.length > 0) {

            // comparing width and height of the group with all values of the children
            // -> the aim is, that all children fit optimal into the group
            // -> this needs to be set after loading and after resizing

            // getting the maximum value of the children for 'left + width' and 'top + height')
            _.each(allChildren, function (drawing) {

                var // the explicit drawing attributes
                    drawingAttrs = isResizeActive ? getUnrotatedDimensions(drawing) : AttributeUtils.getExplicitAttributes(drawing),
                    // the horizontal expansion of one drawing
                    currentHorz = 0,
                    // the vertical expansion of one drawing
                    currentVert = 0;

                if (drawingAttrs) {
                    if (!isResizeActive) {
                        if (!drawingAttrs.drawing) { return; }

                        drawingAttrs = drawingAttrs.drawing;
                        if ('rotation' in drawingAttrs) {
                            drawingAttrs = _.extend(drawingAttrs, DrawingUtils.getRotatedDrawingPoints(drawingAttrs, drawingAttrs.rotation));
                        }
                    } else {
                        drawingAttrsCollection.push(drawingAttrs); // cache drawing attrs in collection for later reading
                    }

                    if (_.isNumber(drawingAttrs.left)) { currentHorz = drawingAttrs.left; }
                    if (_.isNumber(drawingAttrs.width)) { currentHorz += drawingAttrs.width; }
                    if (_.isNumber(drawingAttrs.top)) { currentVert = drawingAttrs.top; }
                    if (_.isNumber(drawingAttrs.height)) { currentVert += drawingAttrs.height; }

                    if (currentHorz > maxWidth) { maxWidth = currentHorz; }
                    if (currentVert > maxHeight) { maxHeight = currentVert; }

                    // collecting all horizontal and vertical offset positions
                    allHorzOffsets.push(_.isNumber(drawingAttrs.left) ? drawingAttrs.left : 0);
                    allVertOffsets.push(_.isNumber(drawingAttrs.top) ? drawingAttrs.top : 0);
                }
            });

            if (maxWidth > 0 || maxHeight > 0) {

                offsetHorz = min.apply(null, allHorzOffsets);
                offsetVert = min.apply(null, allVertOffsets);

                // now checking values for maxWidth and maxHeight
                // -> no rounding, because values for maxWidth can be very large (no hmm)
                horzFactor = groupWidth / (maxWidth - offsetHorz + 1);
                vertFactor = groupHeight / (maxHeight - offsetVert + 1);

                if (isResizeActive) {
                    horzFactor /= zoomFactor;
                    vertFactor /= zoomFactor;
                }

                // getting the maximum value of the children for 'left + width' and 'top + height')
                _.each(allChildren, function (drawing, index) {
                    handleDrawingInsideDrawingGroup($(drawing), AttributeUtils.getExplicitAttributes(drawing), horzFactor, vertFactor, offsetHorz, offsetVert, index);
                });
            }
        }
        if (isResizeActive && isTransformed) {
            DrawingFrame.setCssTransform(drawing, rotationAngle, isFlipH, isFlipV);
        }
    }

    // static class DrawingFrame ==============================================

    /**
     * Contains common static helper functions to display and handle drawing
     * frames in any document.
     */
    var DrawingFrame = {};

    // constants --------------------------------------------------------------

    /**
     * The CSS class used to mark drawing frame nodes.
     *
     * @constant
     */
    DrawingFrame.NODE_CLASS = 'drawing';

    /**
     * A jQuery selector that matches nodes representing a drawing frame.
     *
     * @constant
     */
    DrawingFrame.NODE_SELECTOR = '.' + DrawingFrame.NODE_CLASS;

    /**
     * A jQuery selector that matches nodes representing the content nodes in a drawing frame.
     *
     * @constant
     */
    DrawingFrame.CONTENT_NODE_SELECTOR = '.' + CONTENT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes as place holder.
     *
     * @constant
     */
    DrawingFrame.PLACEHOLDER_CLASS = 'placeholder';

    /**
     * A jQuery selector that matches nodes representing a place holder element
     * of type 'div'.
     *
     * @constant
     */
    DrawingFrame.PLACEHOLDER_SELECTOR = 'div.' + DrawingFrame.PLACEHOLDER_CLASS;

    /**
     * A jQuery selector that matches content nodes of group drawing objects.
     *
     * @constant
     */
    DrawingFrame.GROUPCONTENT_SELECTOR = DrawingFrame.NODE_SELECTOR + '[data-type="group"]>.' + CONTENT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.AUTORESIZEHEIGHT_CLASS = 'autoresizeheight';

    /**
     * The CSS class used to mark drawing content node for no wrapping of words.
     *
     * @constant
     */
    DrawingFrame.NO_WORDWRAP_CLASS = 'no-wordwrap';

    /**
     * A jQuery selector that matches nodes representing a content node inside a
     * test frame that automatically resizes its height.
     *
     * @constant
     */
    DrawingFrame.AUTORESIZEHEIGHT_SELECTOR = 'div.' + DrawingFrame.AUTORESIZEHEIGHT_CLASS;

    DrawingFrame.AUTORESIZETEXT_CLASS = 'autoresizetext';

    DrawingFrame.AUTORESIZETEXT_SELECTOR = 'div.' + DrawingFrame.AUTORESIZETEXT_CLASS;

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.VERTICAL_ALIGNMENT_ATTRIBUTE = 'verticalalign';

    /**
     * The CSS class used to mark drawing content nodes inside text frames
     * for automatic resizing of height.
     *
     * @constant
     */
    DrawingFrame.ODFTEXTFRAME_CLASS = 'odftextframe';

    /**
     * A jQuery selector that matches nodes representing a text frame node that
     * are 'classical' text frames in odf format.
     *
     * @constant
     */
    DrawingFrame.ODFTEXTFRAME_SELECTOR = 'div.' + DrawingFrame.ODFTEXTFRAME_CLASS;

    /**
     * The CSS class used to mark empty text frames.
     *
     * @constant
     */
    DrawingFrame.EMPTYTEXTFRAME_CLASS = 'emptytextframe';

    /**
     * A jQuery selector that matches elements representing a table text frame element.
     *
     * @constant
     */
    DrawingFrame.TABLE_TEXTFRAME_NODE_SELECTOR = '.drawing[data-type="table"]';

    /**
     * The CSS class used to mark table nodes inside drawings of type 'table'.
     *
     * @constant
     */
    DrawingFrame.TABLE_NODE_IN_TABLE_DRAWING_CLASS = 'isdrawingtablenode';

    /**
     * A jQuery selector that matches elements representing a text frame element
     * inside a drawing frame of type shape.
     */
    DrawingFrame.TEXTFRAME_NODE_SELECTOR = 'div.textframe';

    /**
     * The CSS class used to mark text frame content nodes inside a
     * drawing node.
     *
     * @constant
     */
    DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS = 'textframecontent';

    /**
     * A jQuery selector that matches nodes representing a text frame content
     * element inside a drawing node.
     *
     * @constant
     */
    DrawingFrame.TEXTFRAMECONTENT_NODE_SELECTOR = 'div.' + DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS;

    /**
     * The CSS class used to mark drawing nodes inside a group container.
     *
     * @constant
     */
    DrawingFrame.GROUPED_NODE_CLASS = 'grouped';

    /**
     * A jQuery selector that matches nodes representing a grouped element.
     *
     * @constant
     */
    DrawingFrame.GROUPED_NODE_SELECTOR = '.' + DrawingFrame.GROUPED_NODE_CLASS;

    /**
     * The CSS class used to mark drawing nodes that cannot be grouped.
     *
     * @constant
     */
    DrawingFrame.NO_GROUP_CLASS = 'nogroup';

    /**
     * The CSS class used to mark canvas nodes of a drawing.
     *
     * @constant
     */
    DrawingFrame.CANVAS_CLASS = 'canvasgeometry';

    /**
     * A jQuery selector that matches nodes representing a canvas element.
     *
     * @constant
     */
    DrawingFrame.CANVAS_NODE_SELECTOR = 'canvas.' + DrawingFrame.CANVAS_CLASS;

    /**
     * The CSS class used to mark nodes inside drawing nodes, that are used to
     * handle canvas nodes exceeding the drawing.
     *
     * @constant
     */
    DrawingFrame.CANVAS_EXPANSION_CLASS = 'canvasexpansion';

    /**
     * A jQuery selector that matches nodes inside drawing nodes, that are used to
     * handle canvas nodes exceeding the drawing.
     *
     * @constant
     */
    DrawingFrame.CANVAS_EXPANSION_SELECTOR = '.' + DrawingFrame.CANVAS_EXPANSION_CLASS;

    /**
     * The CSS class used to mark drawing nodes, that contain a canvas expansion node
     * because of a canvas with exceeded size.
     *
     * @constant
     */
    DrawingFrame.CONTAINS_CANVAS_EXPANSION = 'containsexpansionnode';

    /**
     * The CSS class used to mark unsupported drawing nodes.
     *
     * @constant
     */
    DrawingFrame.UNSUPPORTED_CLASS = 'unsupported';

    /**
     * A jQuery selector that matches nodes representing unsupported drawings.
     *
     * @constant
     */
    DrawingFrame.UNSUPPORTED_SELECTOR = '.' + DrawingFrame.UNSUPPORTED_CLASS;

    /**
     * A class name that matches elements representing a rotated drawing.
     */
    DrawingFrame.ROTATED_DRAWING_CLASSNAME = 'rotated-drawing';

    /**
     * A class name that matches elements representing a rotated text in drawing.
     */
    DrawingFrame.ROTATED_TEXT_IN_DRAWING_CLASSNAME = 'rotated-text-drawing';

    /**
     * A class name that matches elements representing a drawing mirrored horizontally.
     */
    DrawingFrame.FLIPPED_HORIZONTALLY_CLASSNAME = 'flipH';

    /**
     * A class name that matches elements representing a drawing mirrored vertically.
     */
    DrawingFrame.FLIPPED_VERTICALLY_CLASSNAME = 'flipV';

    /**
     * The name of the property for the target chain that needs to be added to
     * the 'fill' and the 'line' family.
     */
    DrawingFrame.TARGET_CHAIN_PROPERTY = 'targetchain';

    /**
     * Class name that maches elements that need temporary border drawn around them,
     * usually on resize or move.
     */
    DrawingFrame.FRAME_WITH_TEMP_BORDER = 'temp-border-frame';

    /**
     * A CSS class that can be added to a DOM drawing frame to specify its only
     * usage as internal model container. Existence of this class will cause
     * significantly reduced rendering (e.g. no canvas elements in shape
     * objects).
     *
     * @constant
     */
    DrawingFrame.INTERNAL_MODEL_CLASS = 'internal-model-frame';

    // methods ----------------------------------------------------------------

    /**
     * Returns a new drawing frame node without specific contents.
     *
     * @param {DrawingModel|String} modelOrType
     *  The drawing model the new drawing frame will be based on. Alternatively
     *  (for applications not supporting drawing models), this value can be a
     *  string with the type of the drawing object.
     *
     * @returns {jQuery}
     *  A new drawing frame with empty content node, as jQuery object.
     */
    DrawingFrame.createDrawingFrame = function (modelOrType) {

        // the drawing model
        var model = (modelOrType instanceof DrawingModel) ? modelOrType : null;
        // the drawing type identifier
        var type = model ? model.getType() : modelOrType;
        // the resulting drawing frame node
        var drawingFrame = $('<div class="' + DrawingFrame.NODE_CLASS + '" data-type="' + type + '" contenteditable="false"><div class="' + CONTENT_CLASS + '"></div></div>');

        // add data attributes, event handlers, and the content node
        drawingFrame
            .data(DATA_UID, 'frame' + _.uniqueId())
            .data(DATA_MODEL, model)
            .on('dragstart', false);

        return drawingFrame;
    };

    /**
     * Returns whether the passed node is the root node of a drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is the root node of a drawing frame.
     */
    DrawingFrame.isDrawingFrame = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR);
    };

    /**
     * Returns the type of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The type of the specified drawing frame, as specified while the frame
     *  has been created.
     */
    DrawingFrame.getDrawingType = function (drawingFrame) {
        return $(drawingFrame).first().attr('data-type');
    };

    /**
     * Returns whether the specified drawing frame is a line/connector shape
     * (i.e. it will be selected in two-point mode instead of the standard
     * rectangle mode).
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the passed drawing frame is a line/connector shape.
     */
    DrawingFrame.isTwoPointShape = function (drawingFrame) {
        var drawingType = DrawingFrame.getDrawingType(drawingFrame);
        var geoAttrs = AttributeUtils.getExplicitAttributes(drawingFrame).geometry;
        return DrawingUtils.isTwoPointShape(drawingType, geoAttrs);
    };

    /**
     * Returns whether the passed node is a content node inside the drawing frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a content node inside a drawing frame.
     */
    DrawingFrame.isDrawingContentNode = function (node) {
        return $(node).is('.' + CONTENT_CLASS);
    };

    /**
     * Returns whether the passed node is a content node inside the drawing frame
     * that has no content.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a content node inside a drawing frame that has
     *  no content.
     */
    DrawingFrame.isEmptyDrawingContentNode = function (node) {
        return DrawingFrame.isDrawingContentNode(node) && $(node).children().not(DrawingFrame.CANVAS_NODE_SELECTOR).length === 0;
    };

    /**
     * Returns whether the passed node is a placeholder node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a place holder node (of a drawing frame).
     */
    DrawingFrame.isPlaceHolderNode = function (node) {
        return $(node).is(DrawingFrame.PLACEHOLDER_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'shape'.
     */
    DrawingFrame.isShapeDrawingFrame = function (node) { // TODO: type 'table' should be handled differently
        var nodeType = DrawingFrame.getDrawingType(node);
        return DrawingFrame.isDrawingFrame(node) && (nodeType === 'shape' || nodeType === 'table' || nodeType === 'connector');
        // TODO This should be activated asap: return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'shape';
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'shape'.
     */
    DrawingFrame.isTableDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'table';
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape' AND
     * that is used as text frame.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'shape' and it is used
     *  as text frame.
     */
    DrawingFrame.isTextFrameShapeDrawingFrame = function (node) {
        return DrawingFrame.isShapeDrawingFrame(node) && DrawingFrame.getTextFrameNode(node).length > 0;
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'connector'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'connector'.
     */
    DrawingFrame.isConnectorDrawingFrame = function (node) {
        var nodeType = DrawingFrame.getDrawingType(node);
        return DrawingFrame.isDrawingFrame(node) && nodeType === 'connector';
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'group'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'group'.
     */
    DrawingFrame.isGroupDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'group';
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'group' and that contains
     * at least one child drawing of type 'shape'.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame of type 'group', that contains at least one child
     *  drawing of type 'shape'.
     */
    DrawingFrame.isGroupDrawingFrameWithShape = function (node) {
        return DrawingFrame.isDrawingFrame(node) && DrawingFrame.getDrawingType(node) === 'group' && _.isObject(_.find(DrawingFrame.getAllGroupDrawingChildren(node), function (child) { return DrawingFrame.isTextFrameShapeDrawingFrame(child); }));
    };

    /**
     * Returns whether the passed node is a drawing frame that is grouped inside a
     * group.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a grouped drawing frame.
     */
    DrawingFrame.isGroupedDrawingFrame = function (node) {
        return DrawingFrame.isDrawingFrame(node) && $(node).is(DrawingFrame.GROUPED_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that can be moved only
     * in the region near the border. In its center, it can contain text frames.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that cannot be moved by clicking
     *  in its center.
     */
    DrawingFrame.isOnlyBorderMoveableDrawing = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) || DrawingFrame.isGroupDrawingFrame(node);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'
     * and that contains text content. Furthermore the drawing is NOT marked as
     * 'classical' odf text frame, so that there is only reduced text functionality
     * available.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that is of type 'shape'
     *  and that contains text content. Furthermore the drawing is NOT marked as
     *  'classical' odf text frame, so that there is only reduced text functionality
     *  available.
     */
    DrawingFrame.isReducedOdfTextframeNode = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && !$(node).is(DrawingFrame.ODFTEXTFRAME_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing frame, that is of type 'shape'
     * and that contains text content. Furthermore the drawing is marked as
     * 'classical' odf text frame, so that there is no text functionality
     * available.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing frame, that is of type 'shape'
     *  and that contains text content. Furthermore the drawing is marked as
     *  'classical' odf text frame, so that there is no reduced text functionality
     *  available.
     */
    DrawingFrame.isFullOdfTextframeNode = function (node) {
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && $(node).is(DrawingFrame.ODFTEXTFRAME_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node that's rotated by some angle.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a drawing node that has rotation applied.
     */
    DrawingFrame.isRotatedDrawingNode = function (node) {
        return $(node).hasClass(DrawingFrame.ROTATED_DRAWING_CLASSNAME);
    };

    /**
     * Returns whether the passed node is text node in drawing, that is rotated by some angle.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text node in drawing that has rotation applied.
     */
    DrawingFrame.isRotatedTextInDrawingNode = function (node) {
        return $(node).hasClass(DrawingFrame.ROTATED_TEXT_IN_DRAWING_CLASSNAME);
    };

    /**
     * Returns whether the passed node is drawing flipped horizontally.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is drawing flipped horizontally.
     */
    DrawingFrame.isFlippedHorz = function (node) {
        return $(node).hasClass(DrawingFrame.FLIPPED_HORIZONTALLY_CLASSNAME);
    };

    /**
     * Returns whether the passed node is drawing flipped vertically.
     *
     * @param {Node|jQuery} node
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is drawing flipped vertically.
     */
    DrawingFrame.isFlippedVert = function (node) {
        return $(node).hasClass(DrawingFrame.FLIPPED_VERTICALLY_CLASSNAME);
    };

    /**
     * Returns the number of drawing frames inside a specified drawing group. Only
     * the direct children are counted, not the children in sub groups.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Number|Null}
     *  The number of drawing frames inside a specified drawing group
     */
    DrawingFrame.getGroupDrawingCount = function (node) {
        if (!DrawingFrame.isGroupDrawingFrame(node)) { return null; }
        return $(node).children().first().children().length;
    };

    /**
     * Returns the child drawing frame of a drawing group at a specified
     * position. Or null, if it cannot be determined.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @param {Number} [number]
     *  The zero based number of the child to be returned.
     *
     * @returns {Node|Null}
     *  The child drawing frame of a drawing group at a specified position.
     *  Or null, if it cannot be determined.
     */
    DrawingFrame.getGroupDrawingChildren = function (node, number) {
        if (!DrawingFrame.isGroupDrawingFrame(node) || !_.isNumber(number)) { return null; }
        return $(node).children().first()[0].childNodes[number];
    };

    /**
     * Returns all drawing frames inside a specified drawing group. This does
     * not only contain direct children, but also the children of further groups
     * inside the group. If the option 'onlyDirectChildren' is set to true,
     * only the direct children are returned.
     * Returns null, if the specified node is not a group drawing node.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @param {Object} [options]
     *  A map with options to control the selection. The following options are
     *  supported:
     *  @param {Boolean} [options.onlyDirectChildren=false]
     *      If set to true, only those drawing nodes are returned, that are
     *      direct children of the group.
     *
     * @returns {Node|Null}
     *  The child drawing frames of a drawing group at a specified position.
     *  Or null, if it cannot be determined.
     */
    DrawingFrame.getAllGroupDrawingChildren = function (node, options) {

        if (!DrawingFrame.isGroupDrawingFrame(node)) { return null; }

        if (Utils.getBooleanOption(options, 'onlyDirectChildren', false)) {
            return $(node).children().children(DrawingFrame.NODE_SELECTOR);
        } else {
            return $(node).find(DrawingFrame.NODE_SELECTOR);
        }
    };

    /**
     * Returns the closest ancestor drawing frame that contains the passed DOM
     * node.
     *
     * @param {HTMLDivElement|jQuery} node
     *  The descendant DOM node of a drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery|Null}
     *  The closest ancestor drawing frame that contains the passed DOM node;
     *  or null, if the passed node is not a descendant of a drawing frame.
     */
    DrawingFrame.getDrawingNode = function (node) {
        var drawingFrame = $(node).closest(DrawingFrame.NODE_SELECTOR);
        return (drawingFrame.length === 1) ? drawingFrame : null;
    };

    /**
     * Returns the closest drawing node of a given node that is a text frame shape
     * drawing node. If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} node
     *  If this object is a jQuery collection, uses the first DOM node it contains.
     *
     * @returns {jQuery|Null}
     *  The text frame drawing node, or null, if it cannot be determined.
     */
    DrawingFrame.getClosestTextFrameDrawingNode = function (node) {

        // the closest drawing node that is ancestor of the specified node
        var drawingNode = DrawingFrame.getDrawingNode(node);
        if (!drawingNode) { return null; }

        return DrawingFrame.isTextFrameShapeDrawingFrame(drawingNode) ? drawingNode : null;
    };

    /**
     * Returns the closest drawing group node of a drawing node that is grouped.
     * If this cannot be found, null is returned.
     *
     * @param {HTMLDivElement|jQuery} drawingNode
     *  The grouped drawing node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to determine the group node. The following options are
     *  supported:
     *  @param {Boolean} [options.farthest=false]
     *      If set to true, that group node is returned, that is the highest in
     *      the dom. This is necessary, because groups can be included into groups.
     *      Otherwise the closest group node is returned.
     *  @param {Node} [options.rootNode]
     *      If the option 'farthest' is set to true, the root node must be determined,
     *      until which the ancestors will be searched.
     *
     * @returns {jQuery|Null}
     *  The closest or farthest drawing group node, or null, if it cannot be determined.
     */
    DrawingFrame.getGroupNode = function (node, options) {

        var // the group content node is the direct child of the group node itself
            groupContentNode = null,
            // the root node, until which the ancestors are searched. This is only
            // used, if the option 'farthest' is set to true
            rootNode = null;

        if (Utils.getBooleanOption(options, 'farthest', false)) {
            rootNode = Utils.getOption(options, 'rootNode');
            if (rootNode) { groupContentNode = $(Utils.findFarthest(rootNode, node, DrawingFrame.GROUPCONTENT_SELECTOR)); }
        } else {
            groupContentNode = $(node).closest(DrawingFrame.GROUPCONTENT_SELECTOR);
        }

        return (groupContentNode && (groupContentNode.length > 0)) ? groupContentNode.parent() : null;
    };

    /**
     * Returns all container nodes of a specified drawing frame. The container nodes
     * contain the paragraphs and tables and have the class 'textframe' set. Typically
     * a drawing has only one of this container nodes. But if the specified drawing
     * is a group drawing, there can also be more than one container node.
     *
     * @param {HTMLDivElement|jQuery} drawingNode
     *  The drawing frame DOM node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with supported options:
     *  @param {Boolean} [options.deep=false]
     *      If set to true, all text frames inside the specified drawing are searched. This
     *      is especially relevant for group drawings, that can contain several text frames
     *      inside the child drawings.
     *      If set to false, only the text frame that belongs to the specified drawing, is
     *      searched.
     *
     * @returns {jQuery}
     *  The container DOM node from the passed drawing frame that contains all
     *  top-level content nodes (paragraphs and tables). This can be a collection
     *  of several container nodes, if the specified drawing node is a group
     *  drawing.
     */
    DrawingFrame.getTextFrameNode = function (drawingNode, options) {
        return Utils.getBooleanOption(options, 'deep', false) ? $(drawingNode).find(DrawingFrame.TEXTFRAME_NODE_SELECTOR) : $(drawingNode).find('> * > ' + DrawingFrame.TEXTFRAME_NODE_SELECTOR);
    };

    /**
     * Returns the container node of a drawing frame, that contains all
     * top-level content nodes (paragraphs and tables).
     * Info: In addition to getTextFrameNode function, this will not return any cloned node,
     * which can be the case when calling it during move or resize.
     *
     * @param {HTMLDivElement|jQuery} drawingNode
     *  The drawing frame DOM node. If this object is a jQuery collection, uses
     *  the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The container DOM node from the passed drawing frame that contains all
     *  top-level content nodes (paragraphs and tables).
     */
    DrawingFrame.getOriginalTextFrameNode = function (drawingNode) {
        return $(drawingNode).children(':not(.copy)').find('>' + DrawingFrame.TEXTFRAME_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame <div> element, that is
     * used inside drawing frames
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div textframe element.
     */
    DrawingFrame.isTextFrameNode = function (node) {
        return $(node).is(DrawingFrame.TEXTFRAME_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame content <div> element, that is
     * used inside drawing frames
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div textframecontent element.
     */
    DrawingFrame.isTextFrameContentNode = function (node) {
        return $(node).is(DrawingFrame.TEXTFRAMECONTENT_NODE_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * automatically resizes its height. In this case the content node contains a
     * specific class.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an automatically resizing text frame drawing node.
     */
    DrawingFrame.isAutoResizeHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has class 'autoresizeheight'
        return DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * automatically resizes its font size. In this case the content node contains a
     * specific class.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is an text frame drawing node which automatically resizes its text.
     */
    DrawingFrame.isAutoTextHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has class 'autotextheight'
        return DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZETEXT_SELECTOR);
    };

    /**
     * Returns whether the passed shape attributes "autoResizeHeight" is true
     * but only if "noAutoResize" is false
     *
     * @param {Object} [shapeAttrs]
     *  shape attributes
     *
     * @returns {Boolean}
     *  Whether the attributes automatically resizing height.
     */
    DrawingFrame.isAutoResizeHeightAttributes = function (shapeAttrs) {
        return shapeAttrs && (shapeAttrs.autoResizeHeight && !shapeAttrs.noAutoResize);
    };

    /**
     * Returns whether the passed shape attributes "autoResizeText" is true
     * but only if "noAutoResize" is false
     *
     * @param {Object} [shapeAttrs]
     *  shape attributes
     *
     * @returns {Boolean}
     *  Whether the attributes automatically resizing font size.
     */
    DrawingFrame.isAutoTextHeightAttributes = function (shapeAttrs) {
        return shapeAttrs && (shapeAttrs.autoResizeText && !shapeAttrs.noAutoResize);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * has word wrap property set to false
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node has no word wrap property.
     */
    DrawingFrame.isNoWordWrapDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has class 'no-wordwrap'
        return DrawingFrame.getContentNode(node).hasClass(DrawingFrame.NO_WORDWRAP_CLASS);
    };

    /**
     * Returns whether the passed node is a drawing node, that is a text frame, that
     * has a fixed height.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a text frame drawing node with fixed height.
     */
    DrawingFrame.isFixedHeightDrawingFrame = function (node) {
        // checking if the content node inside the text frame drawing has NOT the class 'autoresizeheight'
        return DrawingFrame.isTextFrameShapeDrawingFrame(node) && !DrawingFrame.getContentNode(node).is(DrawingFrame.AUTORESIZEHEIGHT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a text frame <div> element, that is
     * used inside drawing frames as container for additional drawings.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a div groupcontent element.
     */
    DrawingFrame.isGroupContentNode = function (node) {
        return $(node).is(DrawingFrame.GROUPCONTENT_SELECTOR);
    };

    /**
     * Returns whether the passed node is a unsupported drawing.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed node is a unsupported drawing element.
     */
    DrawingFrame.isUnsupportedDrawing = function (drawingFrame) {
        return $(drawingFrame).hasClass(DrawingFrame.UNSUPPORTED_CLASS);
    };

    /**
     * Returns the model of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Object|Null}
     *  The model object of the specified drawing frame, as specified while the
     *  frame has been created.
     */
    DrawingFrame.getModel = function (drawingFrame) {
        return $(drawingFrame).first().data(DATA_MODEL) || null;
    };

    /**
     * Returns the unique identifier of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {String}
     *  The unique identifier of the specified drawing frame, as specified
     *  while the frame has been created.
     */
    DrawingFrame.getUid = function (drawingFrame) {
        return $(drawingFrame).first().data(DATA_UID);
    };

    /**
     * Returns the root content node of the specified drawing frame containing
     * all type specific contents of the drawing.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with additional options. The following options are supported:
     *  @param {Boolean} [options.isResizeActive=false]
     *     If this method is called during resizing of the shape.
     *
     * @returns {jQuery}
     *  The root content node of the specified drawing frame.
     */
    DrawingFrame.getContentNode = function (drawingFrame, options) {
        var selector = '.' + CONTENT_CLASS + (Utils.getBooleanOption(options, 'isResizeActive', false) ? '.copy' : ':not(.copy)');
        return $(drawingFrame).first().children(selector);
    };

    /**
     * Returns the Canvas node (canvas) of the specified drawing frame
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The canvas node of the specified drawing frame.
     */
    DrawingFrame.getCanvasNode = function (drawingFrame) {
        // images inside shapes (OX Text) can have border which is also implemented as canvas,
        // therefore search query needs to be more specific than just "find", see #50908, #50909
        return $(drawingFrame).children(DrawingFrame.CONTENT_NODE_SELECTOR).children(DrawingFrame.CANVAS_NODE_SELECTOR);
    };

    /**
     * Returns whether the specified node is a canvas node (canvas)
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  Whether the specified node is a canvas node.
     */
    DrawingFrame.isCanvasNode = function (node) {
        return $(node).is(DrawingFrame.CANVAS_NODE_SELECTOR);
    };

    /**
     * Returns whether the specified node is currently modified by move, resize or rotate operations.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node element.
     *
     * @returns {Boolean}
     */
    DrawingFrame.isModifyingDrawingActive = function (drawingNode) {
        var $node = $(drawingNode);

        return $node.hasClass('activedragging') || $node.hasClass('rotating-state');
    };

    /**
     * Clears and returns the root content node of the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The empty root content node of the specified drawing frame.
     */
    DrawingFrame.getAndClearContentNode = function (drawingFrame) {
        return DrawingFrame.getContentNode(drawingFrame).empty().removeClass(DrawingFrame.PLACEHOLDER_CLASS).attr('style', '');
    };

    /**
     * For text shapes without paragraph it is necessary to insert an implicit paragraph, so that
     * the user can set the selection into the shape and insert text content (50644, 52378).
     *
     * @param {Object} docModel
     *  The application model.
     *
     * @param {HTMLElement|jQuery} drawing
     *  The drawing node.
     */
    DrawingFrame.checkEmptyTextShape = function (docModel, drawing) {

        var // a required text frame node in the shape
            textFrameNode = null,
            // the attributes of families 'character' and 'paragraph' for the implicit paragraph
            attributes = null;

        if (DrawingFrame.getDrawingType(drawing) === 'shape' && !DrawingFrame.isTextFrameContentNode(DrawingFrame.getContentNode(drawing))) {
            if (!DrawingFrame.isTwoPointShape(drawing)) { // not all shapes get a paragraph
                textFrameNode = DrawingFrame.prepareDrawingFrameForTextInsertion(drawing);  // inserting an implicit paragraph into the drawing
                if (textFrameNode) {
                    attributes = DrawingUtils.getParagraphAttrsForDefaultShape(docModel);
                    textFrameNode.append(docModel.getValidImplicitParagraphNode(attributes));
                }
            }
        }
    };

    /**
     * Generating the target inheritance chain for a specified drawing node. This inheritance
     * chain need to be defined, if a color scheme is used in the merged attributes. Because
     * the drawing is painted in 'DrawingFrame.drawShape', where the drawing node is no longer
     * available, it is necessary to add the target chain into the merged attributes. For
     * performance reasons this is only done, is the attributes contain a scheme color for
     * fill color or line color.
     *
     * @param {Object} docModel
     *  The application model.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The drawing node.
     *
     * @param {Object} mergedAttrs
     *  The complete set of drawing attributes. It can happen, that this object is modified
     *  within this function!
     *  This can also be an incomplete set of attributes, if it is called for grouped
     *  drawings (48096).
     */
    DrawingFrame.handleTargetChainInheritance = function (docModel, drawingFrame, mergedAttrs) {

        if (!mergedAttrs) { return; }
        if (!AttributeUtils.isFillThemed(mergedAttrs.fill) && !AttributeUtils.isLineThemed(mergedAttrs.line)) { return; }

        // setting theme information for colors at 'line' and 'fill' family
        // -> setting the target chain always for gradients, because the theme color might be hidden in
        //    color stops (47924)
        var themeTargets = docModel.getThemeTargets(drawingFrame);
        if (!themeTargets) { return; }

        mergedAttrs.fill = mergedAttrs.fill || {};
        mergedAttrs.line = mergedAttrs.line || {};
        mergedAttrs.fill[DrawingFrame.TARGET_CHAIN_PROPERTY] = themeTargets;
        mergedAttrs.line[DrawingFrame.TARGET_CHAIN_PROPERTY] = themeTargets;
    };

    /**
     * Increasing the height of a group drawing frame, that contains text frames
     * with auto resize height functionality. Those text frames expand automatically
     * the height of the drawing, if required.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The text frame drawing node with auto resize height functionality.
     */
    DrawingFrame.updateDrawingGroupHeight = function (app, drawingNode) {

        // comparing lower border of auto resize text frame node with the group

        var // the offset of the bottom border in pixel
            drawingBottom = 0,
            // an optional group node surrounding the specified text frame drawing node
            groupNode = null,
            // the offset of the group border in pixel
            groupBottom = 0,
            // the new height of the group drawing node
            newHeight = 0,
            // whether the group iteration need to be continued
            doContinue = true,
            // zoom factor of the view
            zoomFactor = app.getView().getZoomFactor(),
            // collection of all parent group drawing nodes of current drawing node
            rotatedGroupParentNodes = [];

        // helper function to find the pixel positions of all bottom borders inside a group
        function getLowestChildDrawingBottom(node) {

            var // the lowest bottom pixel position of all child drawings inside the specified node
                lowestBottom = null;

            // iterating over all children of the specified (group) node
            _.each(node.children(), function (oneDrawing) {

                var // the bottom pixel position of one child node
                    localBottom = Utils.round(($(oneDrawing).offset().top / zoomFactor) + $(oneDrawing).height(), 1);

                // the larger the value, the deeper the drawing
                if (lowestBottom === null || localBottom > lowestBottom) { lowestBottom = localBottom; }
            });

            return lowestBottom;
        }

        // helper function to unrotate all rotated group parent nodes of current node,
        // so that calling offset on that node will return correct value.
        function unrotateAllGroupParentNodes(drawingNode) {
            var rotatedGroupParentNodes = [];
            var groupParentNodes = $(drawingNode).parents(DrawingFrame.NODE_SELECTOR);

            _.each(groupParentNodes, function (parentNode) {
                var rotationAngle = DrawingFrame.getDrawingRotationAngle(app.getModel(), parentNode);
                if (_.isNumber(rotationAngle) && rotationAngle !== 0) {
                    $(parentNode).css({ transform: '' }); // reset to def to get values
                    rotatedGroupParentNodes.push({ node: parentNode, angle: rotationAngle });
                }
            });
            return rotatedGroupParentNodes;
        }

        var rotationAngle = null;

        // iterating over all groups
        while (doContinue && DrawingFrame.isGroupedDrawingFrame(drawingNode)) {
            rotatedGroupParentNodes = unrotateAllGroupParentNodes(drawingNode);
            rotationAngle = DrawingFrame.getDrawingRotationAngle(app.getModel(), groupNode);
            if (_.isNumber(rotationAngle) && rotationAngle !== 0) {
                $(groupNode).css({ transform: '' }); // reset to def to get values
                rotatedGroupParentNodes.push({ node: groupNode, angle: rotationAngle });
            }

            groupNode = DrawingFrame.getGroupNode(drawingNode);
            groupBottom = groupNode ? Utils.round((groupNode.offset().top / zoomFactor) + groupNode.height(), 1) : null;
            drawingBottom = getLowestChildDrawingBottom($(drawingNode).parent());

            // comparing the bottom borders of group and specified text frame
            if (_.isNumber(groupBottom) && _.isNumber(drawingBottom) && (groupBottom !== drawingBottom)) {
                // increasing or decreasing the height of the group without operation
                newHeight = Utils.round(groupNode.height() + (drawingBottom - groupBottom), 1);
                groupNode.height(newHeight);
                drawingNode = groupNode;
            } else {
                doContinue = false;
            }
        }
        _.each(rotatedGroupParentNodes, function (obj) { // all collected rotated group root nodes need to be returned to original rotation
            $(obj.node).css({ transform: 'rotate(' + obj.angle + 'deg)' });
        });
    };

    /**
     * Prepares a specified drawing frame for text insertion. This requires
     * that the content node inside drawing frame is cleared and then a new
     * text frame child is appended. This new text frame element is returned
     * and can be used as parent for paragraphs, tables, ... .
     * Only drawingFrames of type 'shape' allow this conversion.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The new text frame 'div' element, that can be used as container for
     *  the text content.
     */
    DrawingFrame.prepareDrawingFrameForTextInsertion = function (drawingFrame) {

        // only drawing frames of type 'shape' can be converted
        if (!DrawingFrame.isShapeDrawingFrame(drawingFrame)) { return $(); }

        // the content node of the drawing
        var contentNode = DrawingFrame.getAndClearContentNode(drawingFrame);
        // the text frame node inside the content node, parent of paragraphs and tables
        // -> required for setting cursor type, distance to content node and click event
        var textFrameNode = $('<div class="textframe" contenteditable="true">');

        // making content node visible by assigning a border via class 'textframecontent'
        contentNode.addClass(DrawingFrame.TEXTFRAMECONTENT_NODE_CLASS).append(textFrameNode);
        // in IE the drawing node itself must have contenteditable 'false', otherwise the
        // grabbers will be visible. Other browsers have to use the value 'true'.
        $(drawingFrame).attr('contenteditable', false);

        // returning the new created text frame node
        return textFrameNode;
    };

    /**
     * Inserts replacement layout nodes for unsupported drawing types. Inserts
     * the passed name and description of the drawing as text.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {String} [name]
     *  The name of the drawing frame.
     *
     * @param {String} [description]
     *  The description text for the drawing frame.
     */
    DrawingFrame.insertReplacementNodes = function (drawingFrame, name, description) {

        var // the type of the drawing frame
            type = DrawingFrame.getDrawingType(drawingFrame),
            // the empty content node
            contentNode = DrawingFrame.getAndClearContentNode(drawingFrame),
            // the inner width and height available in the content node
            innerWidth = max(0, drawingFrame.outerWidth() - 2),
            innerHeight = max(0, drawingFrame.outerHeight() - 2),
            // the vertical padding in the content node
            verticalPadding = Utils.minMax(min(innerWidth, innerHeight) / 24, 1, 6),
            // the font size of the picture icon
            pictureIconSize = Utils.minMax(min(innerWidth - 16, innerHeight - 28), 8, 72),
            // the base font size of the text
            fontSize = Utils.minMax(min(innerWidth, innerHeight) / 4, 9, 13);

        // set border width at the content node, insert the picture icon
        contentNode
            .addClass(DrawingFrame.PLACEHOLDER_CLASS)
            .css({
                padding: floor(verticalPadding) + 'px ' + floor(verticalPadding * 2) + 'px',
                fontSize: fontSize + 'px'
            })
            .append(
                $('<div class="abs background-icon" style="line-height:' + innerHeight + 'px;">')
                    .append(Forms.createIconMarkup(Labels.getDrawingTypeIcon(type), { style: 'font-size:' + pictureIconSize + 'px' })),
                $('<p>').text(_.noI18n(name) || Labels.getDrawingTypeLabel(type))
            );

        // insert description if there is a reasonable amount of space available
        if ((innerWidth >= 20) && (innerHeight >= 20)) {
            contentNode.append($('<p>').text(_.noI18n(description)));
        }
    };

    // formatting -------------------------------------------------------------

    /**
     * Updates the CSS formatting of the passed drawing frame, according to the
     * passed generic formatting attributes.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame to be updated. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {Object} mergedAttributes
     *  The drawing attribute set (a map of attribute maps as name/value pairs,
     *  keyed by the 'drawing' attribute family), containing the effective
     *  attribute values merged from style sheets and explicit attributes.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Boolean} [options.skipFlipping=false]
     *      If set to true, all flipping attributes will be ignored, and the
     *      drawing frame will be rendered unflipped.
     */
    DrawingFrame.updateFormatting = function (app, drawingFrame, mergedAttributes, options) {

        // ensure to have a jQuery object
        drawingFrame = $(drawingFrame);

        // the document model
        var docModel = app.getModel();
        // the drawing attributes of the passed attribute set
        var drawingAttributes = mergedAttributes.drawing;
        // type of the drawing object: 'image', ...
        var type = DrawingFrame.getDrawingType(drawingFrame);
        // the content node inside the drawing frame
        var contentNode = DrawingFrame.getContentNode(drawingFrame);
        // whether to skip flipping attributes
        var skipFlipping = Utils.getBooleanOption(options, 'skipFlipping', false);
        // whether the drawing frame is used as internal model container (reduced rendering)
        var internalModel = drawingFrame.hasClass(DrawingFrame.INTERNAL_MODEL_CLASS);
        // the group node belonging to a grouped drawing node
        var drawingGroupNode = null;
        // the replacement data for unsupported drawings
        var replacementData = Utils.getStringOption(drawingAttributes, 'replacementData', '');
        // rendering result (false to fall-back to placeholder graphic)
        var rendered = false;
        var drawingAngle = (drawingAttributes.rotation) || 0;
        var isInlineDrawing = (drawingAttributes.inline) || false;

        // first remove the class which marks this drawing as unsupported
        drawingFrame.removeClass(DrawingFrame.UNSUPPORTED_CLASS);

        // creates an image node in the content node with the specified source URL
        function createImageNodeFromUrl(srcUrl) {
            return app.createImageNode(srcUrl, { timeout: 15000 }).done(function (imgNode) {
                var cropNode = $('<div class="' + CROPPING_CLASS + '">').append(imgNode);
                DrawingFrame.getAndClearContentNode(drawingFrame).append(cropNode);
            });
        }

        // creates an image node or an SVG image according to the passed image data
        function createImageNodeFromData(imageData) {
            if (/^data:/.test(imageData)) {
                return createImageNodeFromUrl(imageData);
            }
            if (/^<svg/.test(imageData)) {
                DrawingFrame.getAndClearContentNode(drawingFrame)[0].innerHTML = imageData;
                return $.when();
            }
            Utils.warn('DrawingFrame.createImageNodeFromData() - unsupported source data for the image object');
            return $.Deferred().reject();
        }

        // updates all formatting for unsupported objects
        function updateUnsupportedFormatting() {
            if (replacementData.length === 0) {
                // insert default replacement nodes, if no replacement data is available
                DrawingFrame.insertReplacementNodes(drawingFrame, drawingAttributes.name, drawingAttributes.description);
            } else if (!drawingFrame.data('has-replacement')) {
                // replacement data MUST NOT change at runtime
                createImageNodeFromData(replacementData);
                drawingFrame.data('has-replacement', true);
            }
            // add class to mark this drawing as a unsupported drawing.
            drawingFrame.addClass(DrawingFrame.UNSUPPORTED_CLASS);
        }

        // updates all formatting for image objects
        function updateImageFormatting() {

            var // special attributes for images
                imageAttributes = mergedAttributes.image,
                // the promise that handles the loading of the image
                imageLoadPromise = null;

            // updates attributes of the image node after loading
            // -> but not for grouped drawings. Those drawings are handled by the group container
            function updateImageAttributes(options) {

                var // current width of the drawing frame, in 1/100 mm
                    drawingWidth = Utils.convertLengthToHmm(drawingFrame.width(), 'px'),
                    // current height of the drawing frame, in 1/100 mm
                    drawingHeight = Utils.convertLengthToHmm(drawingFrame.height(), 'px'),
                    // whether the call of this function is asynchronous
                    isAsync = Utils.getBooleanOption(options, 'isAsync', true),
                    // whether the visibility was changed (sometimes required in async mode)
                    visibilityChanged = false;

                if ((drawingWidth > 0) && (drawingHeight > 0)) {

                    if (DrawingFrame.isGroupedDrawingFrame(drawingFrame)) {
                        // For grouped drawings, it is useless to set the image attributes directly,
                        // because these attributes are dependent from the group itself.
                        // -> after the image is loaded (deferred!), the image attributes need to be set again
                        // to be correct for the complete group.
                        // The call of resizeDrawingsInGroup() is only required after loading the document for
                        // grouped drawings when loading happens without fast load and without
                        // local storage.
                        if (isAsync) {
                            if (docModel.handleContainerVisibility) {
                                visibilityChanged = docModel.handleContainerVisibility(drawingFrame, { makeVisible: true });
                            }
                            resizeDrawingsInGroup(app, DrawingFrame.getGroupNode(drawingFrame), mergedAttributes);
                            if (visibilityChanged && docModel.handleContainerVisibility) {
                                docModel.handleContainerVisibility(drawingFrame, { makeVisible: false });
                            }
                        }
                    } else {
                        setImageCssSizeAttributes(app, contentNode, drawingWidth, drawingHeight, drawingAttributes, imageAttributes);
                    }
                    // remove class which marks this drawing as unsupported
                    drawingFrame.removeClass(DrawingFrame.UNSUPPORTED_CLASS);
                    // update the border
                    updateLineFormatting(app, drawingFrame, mergedAttributes);
                }
            }

            updateLineFormatting(app, drawingFrame, mergedAttributes);

            // create the image or SVG node (considered to be constant at runtime)
            if (contentNode.find('>.' + CROPPING_CLASS).children().length === 0) {
                // start with replacement graphics (image data will be loaded asynchronously)
                updateUnsupportedFormatting();
                if (imageAttributes.imageData.length > 0) {
                    imageLoadPromise = createImageNodeFromData(imageAttributes.imageData).done(updateImageAttributes);
                } else if (imageAttributes.imageUrl.length > 0) {
                    // convert relative URL to an application-specific absolute URL
                    imageLoadPromise = createImageNodeFromUrl(/:\/\//.test(imageAttributes.imageUrl) ? imageAttributes.imageUrl : app.getServerModuleUrl(IO.FILTER_MODULE_NAME, { action: 'getfile', get_filename: imageAttributes.imageUrl })).done(updateImageAttributes);
                } else {
                    Utils.warn('DrawingFrame.updateFormatting() - missing source data for the image object');
                    imageLoadPromise = $.when();
                }
            } else {
                // update attributes of an existing image element
                updateImageAttributes({ isAsync: false });
                imageLoadPromise = $.when();
            }

            imageLoadPromise.then(function () {
                if (docModel.useSlideMode()) { docModel.trigger('image:loaded', drawingFrame); } // this might be interesting for other apps, too
            });

            // image successfully rendered
            return true;
        }

        // updates chart formatting
        function updateChartFormatting() {

            // the drawing object model
            var model = DrawingFrame.getModel(drawingFrame);
            // drawing object model must exist for chart objects
            if (!model) { return false; }

            if (!model.isValidChartType()) { return false; }

            // the chart rendering engine (lazy initialization, see below)
            var renderer = drawingFrame.data('chart-renderer');

            // lazy creation of the renderer
            if (!renderer) {
                // unique DOM element identifier for the renderer
                var chartId = 'io-ox-documents-chart-' + DrawingFrame.getUid(drawingFrame);
                contentNode.addClass('chartholder');

                // localization for CanvasJS
                if (!CULTURE_INFO) {
                    CULTURE_INFO = {
                        decimalSeparator: LocaleData.DEC,
                        digitGroupSeparator: LocaleData.GROUP,
                        shortDays: LocaleData.SHORT_WEEKDAYS.slice(),
                        days: LocaleData.LONG_WEEKDAYS.slice(),
                        shortMonths: LocaleData.SHORT_MONTHS.slice(),
                        months: LocaleData.LONG_MONTHS.slice()
                    };
                    CanvasJS.addCultureInfo('en', CULTURE_INFO);
                }

                contentNode.append('<div id="' + chartId + '" class="chartnode"></div>');

                // really important call, because of a bug in canvasJS
                model.resetData();

                renderer = new CanvasJS.Chart(chartId, model.getModelData());

                // workaround for broken tooltip when changing chart type
                var updateToolTip = renderer.toolTip._updateToolTip;
                renderer.toolTip._updateToolTip = function () {
                    try {
                        updateToolTip.apply(this, arguments);
                    } catch (e) {
                        Utils.warn('chart tooltip error', e);
                    }
                };

                drawingFrame.data('chart-renderer', renderer);
                model.firstInit();
            }

            // render the chart object
            model.updateRenderInfo();
            try {
                renderer.resetOverlayedCanvas();
                renderer.render();
            } catch (ex) {
                Utils.exception(ex, 'chart rendering error');
            }

            // background color of model data is already prepared for css
            contentNode.css('backgroundColor', model.getModelData().cssBackgroundColor);

            // chart successfully rendered
            return true;
        }

        // updates drawings of type 'table'
        function updateTableFormatting() {

            // adding auto resize height to the table
            contentNode.addClass(DrawingFrame.AUTORESIZEHEIGHT_CLASS);

            // table successfully rendered
            return true;
        }

        // updates the class at the drawing, that is used to find standard odt text frames (and not shapes with text)
        function updateTextFrameState() {
            if (app.isODF()) {
                // the explicit drawing attributes (not using merge attributes, because they always contain a style id)
                var attrs = AttributeUtils.getExplicitAttributes(drawingFrame);
                drawingFrame.toggleClass(DrawingFrame.ODFTEXTFRAME_CLASS, _.isString(attrs.styleId));
            }
        }

        /**
         * Rotated inline shape should have correction margins, so that text flows around surrounding bound box.
         *
         * @param {Number} drawingWidth
         * @param {Number} drawingHeight
         * @param {Number} rad
         *
         * @returns {Object} horizontal and vertical correction values.
         */
        function getCorrectionMarginsForRotatedShape(drawingWidth, drawingHeight, rad) {
            var boundingHeight = drawingWidth * Math.abs(Math.sin(rad)) + drawingHeight * Math.abs(Math.cos(rad));
            var boundingWidth = drawingWidth * Math.abs(Math.cos(rad)) + drawingHeight * Math.abs(Math.sin(rad));
            var marginDiffH = (boundingWidth - drawingWidth) / 2;
            var marginDiffV = (boundingHeight - drawingHeight) / 2;

            return { tb: marginDiffV, lr: marginDiffH };
        }

        // handle theme color inheritance chain (the merged attributes object might be modified)
        DrawingFrame.handleTargetChainInheritance(docModel, drawingFrame, mergedAttributes);

        if (!skipFlipping) {
            handleFlipping(docModel, drawingFrame, drawingAttributes);
        }

        // update drawing frame specific to the drawing type
        switch (type) {

            case 'image':
                rendered = internalModel || updateImageFormatting();
                break;

            case 'ole':
                rendered = internalModel || updateImageFormatting();
                break;

            case 'chart':
                rendered = internalModel || updateChartFormatting();
                break;

            case 'table':
                rendered = internalModel || updateTableFormatting();
                updateTextFrameState();
                break;

            case 'connector':
            case 'shape':
                rendered = updateShapeFormatting(app, drawingFrame, mergedAttributes);
                updateTextFrameState();
                break;

            case 'group':
                // the group needs to take care of the size of all children
                resizeDrawingsInGroup(app, drawingFrame, mergedAttributes);
                rendered = true;
                break;
        }

        // an additional resize of drawings in groups is required after all children are added to the group.
        // -> during working with document, operations are only generated for the group itself, so that
        //    'DrawingFrame.isGroupedDrawingFrame()' should always be false.
        // -> but during working with the document, this step is also necessary after undo (47204)
        // -> Performance: Setting the data attribute 'grouphandling' to the parent drawing makes it
        //                 possible that 'resizeDrawingsInGroup' is not called after every child drawing is
        //                 added to the group. In this case it is necessary, that the drawing of type
        //                 'group' is formatted, after all children are inserted.
        if (DrawingFrame.isGroupedDrawingFrame(drawingFrame)) {
            drawingGroupNode = DrawingFrame.getGroupNode(drawingFrame);
            if (drawingGroupNode && drawingGroupNode.length > 0 && !drawingGroupNode.data('grouphandling')) {
                resizeDrawingsInGroup(app, drawingGroupNode, mergedAttributes);
            }
        }

        // add replacement in case the drawing type is not supported, or if rendering was not successful
        if (!rendered && !internalModel) {
            updateUnsupportedFormatting();
        }

        if (isInlineDrawing && drawingAngle) { // rotated inline drawings need corrected margins not to colide with surrounding text
            var corMargins = getCorrectionMarginsForRotatedShape(drawingFrame.width(), drawingFrame.height(), drawingAngle * PI_180);
            drawingFrame.css({ marginTop: corMargins.tb, marginBottom: corMargins.tb, marginLeft: corMargins.lr, marginRight: corMargins.lr });
        }
    };

    /**
     * returns the wrapped canvas for the current node.
     *
     * @param {jQuery} [node]
     *
     * @param {Object} [options]
     *  @param {Boolean} [options.resizeIsActive=false]
     *      If this method is called during resizing of the shape.
     */
    function createWrappedCanvas(app, currentNode, options) {

        // if process of resizing of drawing is currently active
        var resizeIsActive  = Utils.getBooleanOption(options, 'resizeIsActive', false);
        // the canvasNode inside the drawing frame
        var canvasNode = DrawingFrame.getCanvasNode(currentNode);
        // the canvas wrapper
        var canvas = null;

        // while resizing of the drawing is active, there are cloned and original canvas,
        // only first has to be fetched
        if (resizeIsActive) {
            canvasNode = canvasNode.first();
        } else if (canvasNode.length > 1) {
            canvasNode = canvasNode.last();
        }

        if (canvasNode.length > 0) {
            canvas = new Canvas(app, { classes: DrawingFrame.CANVAS_CLASS, node: canvasNode });
        } else {
            canvas = new Canvas(app, { classes: DrawingFrame.CANVAS_CLASS });
            DrawingFrame.getContentNode(currentNode).prepend(canvas.getNode());
        }
        return canvas;
    }

    /**
     * Initializes the position and size of the passed canvas wrapper, relative
     * to its parent container element.
     *
     * @param {Canvas} canvas
     *  The canvas wrapper to be initialized.
     *
     * @param {String} drawingType
     *  The type of the drawing frame used to decide how much to pull out the
     *  canvas element from the parent element (full border for images, half
     *  border otherwise).
     *
     * @param {Rectangle} snapRect
     *  The location of the shape.
     *
     * @param {Number} lineWidth
     *  The width of the border lines of the shape, in pixels.
     */
    function initializeCanvasSize(canvas, drawingType, snapRect, lineWidth) {

        // how much to pull out the canvas element from the parent element (full border for images)
        var offsetSize = (drawingType === 'image') ? lineWidth : ceil(lineWidth / 2);
        // total size of the canvas element
        var totalWidth = snapRect.width + 2 * offsetSize;
        var totalHeight = snapRect.height + 2 * offsetSize;
        // translation for the canvas rectangle (middle of leading borders on zero coordinates)
        var translateLeft = snapRect.left - ceil(lineWidth / 2);
        var translateTop = snapRect.top - ceil(lineWidth / 2);
        // CSS offset for the canvas element
        var nodeLeft = snapRect.left - offsetSize;
        var nodeTop = snapRect.top - offsetSize;

        // initialize effective rectangle of the canvas wrapper (size of the DOM canvas node will be initialized too)
        canvas.initialize({ left: translateLeft, top: translateTop, width: totalWidth, height: totalHeight });

        // set the CSS location of the canvas element relative to its parent element
        canvas.getNode().css({ left: nodeLeft, top: nodeTop });
    }

    /**
     * a initialized canvas is returned for the current node
     *
     * @param {jQuery} [node]
     */
    function initializeCanvas(app, currentNode, snapRect, lineWidth, options) {

        // the canvas wrapper of the border node
        var canvas = createWrappedCanvas(app, currentNode, options);

        // initialize location of the canvas element
        initializeCanvasSize(canvas, DrawingFrame.getDrawingType(currentNode), snapRect, lineWidth);

        return canvas;
    }

    /**
     *
     * @param collector
     * @param command
     * @returns {*}
     */
    function collectPathCoordinates(collector, command) {
        var
            pathCoords  = collector.pathCoords,

            cx          = collector.cx || 0,
            cy          = collector.cy || 0,
            fXScale     = collector.fXScale || 1,
            fYScale     = collector.fYScale || 1,

            getValue    = collector.getGeometryValue,

            i           = 0,
            pts         = null,
            length      = null,

            commandType = command.c;

        // calculates the point on an arc that corresponds to the given angle (ms like)
        function calcArcPoint(rx, ry, angle) {
            var at2 = atan2(rx * sin(angle), ry * cos(angle));
            return { x: rx * cos(at2) * fXScale, y: ry * sin(at2) * fYScale };
        }

        if (commandType === 'moveTo') {

            pathCoords.xValues.push(round(getValue(command.x) * fXScale));
            pathCoords.yValues.push(round(getValue(command.y) * fYScale));

        } else if (commandType === 'lineTo') {

            pathCoords.xValues.push(round(getValue(command.x) * fXScale));
            pathCoords.yValues.push(round(getValue(command.y) * fYScale));

        } else if (commandType === 'arcTo') {
            var
                rx    = getValue(command.wr),
                ry    = getValue(command.hr),
                stAng = getValue(command.stAng) * PI_RAD,
                swAng = getValue(command.swAng),

                // calculating new currentPoint
                startPoint  = calcArcPoint(rx, ry, stAng),
                endPoint    = calcArcPoint(rx, ry, stAng + swAng * PI_RAD);

            pathCoords.xValues.push(cx);
            pathCoords.yValues.push(cy);
            pathCoords.xValues.push(cx = round(cx + endPoint.x - startPoint.x));
            pathCoords.yValues.push(cy = round(cy + endPoint.y - startPoint.y));

        } else if (commandType === 'quadBezierTo') {

            if (command.pts) {
                pts     = command.pts;
                length  = pts.length;

                if (length % 2 === 0) {
                    for (i = 0; i < length; i += 2) {

                        // pathCoords.push({
                        //     type: 'quad',
                        //
                        //     x1: getValue(pts[i].x) * fXScale,
                        //     y1: getValue(pts[i].y) * fYScale,
                        //     x2: (cx = getValue(pts[i + 1].x) * fXScale),
                        //     y2: (cy = getValue(pts[i + 1].y) * fYScale)
                        // });
                        pathCoords.xValues.push(round(getValue(pts[i].x) * fXScale));
                        pathCoords.yValues.push(round(getValue(pts[i].y) * fYScale));
                        pathCoords.xValues.push(cx = round(getValue(pts[i + 1].x) * fXScale));
                        pathCoords.yValues.push(cy = round(getValue(pts[i + 1].y) * fYScale));
                    }
                }

            }
        } else if (commandType === 'cubicBezierTo') {

            if (command.pts) {
                pts     = command.pts;
                length  = pts.length;

                if (length % 3 === 0) {
                    for (i = 0; i < length; i += 3) {

                        // pathCoords.push({
                        //     type: 'cubic',
                        //
                        //     x1: getValue(pts[i].x) * fXScale,
                        //     y1: getValue(pts[i].y) * fYScale,
                        //     x2: getValue(pts[i + 1].x) * fXScale,
                        //     y2: getValue(pts[i + 1].y) * fYScale,
                        //     x3: (cx = getValue(pts[i + 2].x) * fXScale),
                        //     y3: (cy = getValue(pts[i + 2].y) * fYScale)
                        // });
                        pathCoords.xValues.push(round(getValue(pts[i].x) * fXScale));
                        pathCoords.yValues.push(round(getValue(pts[i].y) * fYScale));
                        pathCoords.xValues.push(round(getValue(pts[i + 1].x) * fXScale));
                        pathCoords.yValues.push(round(getValue(pts[i + 1].y) * fYScale));
                        pathCoords.xValues.push(cx = round(getValue(pts[i + 2].x) * fXScale));
                        pathCoords.yValues.push(cy = round(getValue(pts[i + 2].y) * fYScale));
                    }
                }
            }
        }
        collector.cx = cx;
        collector.cy = cy;

        return collector;
    }

    /**
     *
     * @param collector
     * @param command
     * @returns {*}
     */
    function executePathCommands(collector, command) {

        var canvasPath  = collector.canvasPath,

            cx          = collector.cx || 0,
            cy          = collector.cy || 0,
            fXScale     = collector.fXScale || 1,
            fYScale     = collector.fYScale || 1,

            getValue    = collector.getGeometryValue,

            i           = 0,
            pts         = null,
            length      = null,

            r           = 0,
            r1          = 0,
            r2          = 0,
            rpx         = 0,
            rpy         = 0,

            rxArc       = 0,
            ryArc       = 0,

            angle       = 0,
            rightDirection = false; // the direction of triangle, arrow, ...

        // calculates the point on an arc that corresponds to the given angle (ms like)
        function calcArcPoint(rx, ry, angle) {
            var at2 = atan2(rx * sin(angle), ry * cos(angle));
            return { x: rx * cos(at2) * fXScale, y: ry * sin(at2) * fYScale };
        }

        switch (command.c) {
            case 'moveTo':
                cx = getValue(command.x) * fXScale;
                cy = getValue(command.y) * fYScale;
                canvasPath.moveTo(cx, cy);
                break;

            case 'lineTo':
                cx = getValue(command.x) * fXScale;
                cy = getValue(command.y) * fYScale;
                canvasPath.lineTo(cx, cy);
                break;

            case 'circle':
                r = getValue(command.r) * fXScale;
                cx = getValue(command.x) * fXScale;
                cy = getValue(command.y) * fYScale;
                canvasPath.pushCircle(cx, cy, r);
                break;

            case 'oval':
            case 'diamond':
            case 'triangle':
            case 'stealth':
            case 'arrow':
                r1 = getValue(command.r1) * fXScale;
                r2 = getValue(command.r2) * fYScale;

                cx = getValue(command.x) * fXScale;
                cy = getValue(command.y) * fYScale;

                // reference point for calculating the angle (of the line)
                rpx = getValue(command.rpx) * fXScale;
                rpy = getValue(command.rpy) * fYScale;

                // an optional offset, for calculation arrow direction at bezier curves
                if (command.rpxOffset) { rpx += command.rpxOffset; }

                if (rpy !== cy) { angle = atan2(abs(rpy - cy), abs(rpx - cx)); }

                switch (command.c) {
                    case 'oval':
                        canvasPath.pushEllipse(cx, cy, r1, r2, angle);
                        break;
                    case 'diamond':
                        canvasPath.pushDiamond(cx, cy, r1, r2, angle);
                        break;
                    case 'triangle':
                        if (isRightOrDownDirection(cx, cy, rpx, rpy)) {
                            rightDirection = true; // setting the direction of the triangle
                        }
                        canvasPath.pushTriangle(cx, cy, r1, r2, angle, rightDirection);
                        break;
                    case 'stealth':
                        if (isRightOrDownDirection(cx, cy, rpx, rpy)) {
                            rightDirection = true;  // setting the direction of the stealth arrow head
                        }
                        canvasPath.pushStealth(cx, cy, r1, r2, angle, rightDirection);
                        break;
                    case 'arrow':
                        if (isRightOrDownDirection(cx, cy, rpx, rpy)) {
                            rightDirection = true; // setting the direction of the arrow head
                        }
                        canvasPath.pushArrow(cx, cy, r1, r2, angle, rightDirection);
                        break;
                }
                break;

            case 'arcTo':
                var rx    = getValue(command.wr),
                    ry    = getValue(command.hr),
                    stAng = getValue(command.stAng) * PI_RAD,
                    swAng = getValue(command.swAng),

                    // calculating new currentPoint
                    startPoint  = calcArcPoint(rx, ry, stAng),
                    endPoint    = calcArcPoint(rx, ry, stAng + swAng * PI_RAD),

                    calcAngle = function (x, y) {
                        var x1 = rx * fXScale;
                        var y1 = ry * fYScale;
                        return (x1 > y1) ? atan2(y * x1 / y1, x) : atan2(y, x * y1 / x1);
                    },

                    eStAng  = calcAngle(startPoint.x, startPoint.y),
                    eEndAng = calcAngle(endPoint.x, endPoint.y);

                // check if a full circle has to be drawn
                if (abs(swAng) >= 21600000) {
                    swAng   = -1;
                    eEndAng = eStAng + 0.0001;
                }
                canvasPath.ellipseTo(cx, cy, rx * fXScale, ry * fYScale, eStAng, eEndAng, swAng < 0);

                cx = cx + endPoint.x - startPoint.x;
                cy = cy + endPoint.y - startPoint.y;

                rxArc = 2 * rx * fXScale;
                ryArc = 2 * ry * fYScale;

                break;

            case 'quadBezierTo':
                if (command.pts) {
                    pts     = command.pts;
                    length  = pts.length;

                    if (length % 2 === 0) {
                        for (i = 0; i < length; i += 2) {

                            canvasPath.quadraticCurveTo(
                                getValue(pts[i].x) * fXScale,
                                getValue(pts[i].y) * fYScale,
                                cx = getValue(pts[i + 1].x) * fXScale,
                                cy = getValue(pts[i + 1].y) * fYScale
                            );
                        }
                    }
                }
                break;

            case 'cubicBezierTo':
                if (command.pts) {
                    pts     = command.pts;
                    length  = pts.length;

                    if (length % 3 === 0) {
                        for (i = 0; i < length; i += 3) {

                            canvasPath.bezierCurveTo(
                                getValue(pts[i].x) * fXScale,
                                getValue(pts[i].y) * fYScale,
                                getValue(pts[i + 1].x) * fXScale,
                                getValue(pts[i + 1].y) * fYScale,
                                cx = getValue(pts[i + 2].x) * fXScale,
                                cy = getValue(pts[i + 2].y) * fYScale
                            );
                        }
                    }
                }
                break;

            case 'close':
                canvasPath.close();
                break;
        }

        collector.cx = cx;
        collector.cy = cy;

        if (cx < collector.cxMin) { collector.cxMin = cx; }
        if (cx > collector.cxMax) { collector.cxMax = cx; }
        if (cy < collector.cyMin) { collector.cyMin = cy; }
        if (cy > collector.cyMax) { collector.cyMax = cy; }

        if (rxArc) {
            if ((cx - rxArc) < collector.cxMin) { collector.cxMin = cx - rxArc; }
            if ((cx + rxArc) > collector.cxMax) { collector.cxMax = cx + rxArc; }
        }

        if (ryArc) {
            if ((cy - ryArc) < collector.cyMin) { collector.cyMin = cy - ryArc; }
            if ((cy + ryArc) > collector.cyMax) { collector.cyMax = cy + ryArc; }
        }

        return collector;
    }

    // overlay colors for a shaded fill color, mapped by fill mode attribute values
    var SHADED_OVERLAY_MAP = {
        lighten:     { color: Color.parseJSON(Color.WHITE), alpha: 0.4 },
        lightenLess: { color: Color.parseJSON(Color.WHITE), alpha: 0.2 },
        darken:      { color: Color.parseJSON(Color.BLACK), alpha: 0.4 },
        darkenLess:  { color: Color.parseJSON(Color.BLACK), alpha: 0.2 }
    };

    function drawShape(app, context, snapRect, geometryAttrs, attrSet, guideCache, options) {

        if (!geometryAttrs.pathList) {
            return;
        }

        var docModel = app.getModel(),

            widthPx  = snapRect.width,
            heightPx = snapRect.height,

            avList = geometryAttrs.avList,
            gdList = geometryAttrs.gdList,

            fillAttrs = attrSet.fill,
            lineAttrs = attrSet.line,

            targets = fillAttrs ? fillAttrs[DrawingFrame.TARGET_CHAIN_PROPERTY] : null,
            themeModel = docModel.getThemeModel(targets),

            isResizeActive          = Utils.getBooleanOption(options, 'isResizeActive', false),
            currentNode             = Utils.getObjectOption(options, 'currentNode', null),
            drawingUID              = DrawingFrame.getUid(currentNode),
            rotationAngle           = null,
            isBitmapFill            = fillAttrs && fillAttrs.type === 'bitmap' && !_.isEmpty(fillAttrs.bitmap),
            notRotatedWithShape     = fillAttrs && fillAttrs.bitmap && fillAttrs.bitmap.rotateWithShape === false,
            cachedTextureObj        = null;

        function getGeoValue(value) {
            return getGeometryValue(value, snapRect, avList, gdList, guideCache);
        }

        function calcFinalPath(origPath, scaleX, scaleY) {
            return origPath.commands.reduce(executePathCommands, {
                canvasPath: context.createPath(),
                cx: 0,
                cy: 0,
                fXScale: scaleX,
                fYScale: scaleY,
                getGeometryValue: getGeoValue
            }).canvasPath;
        }

        function getCachedTextureFill(textureCanvasPatternCache, drawingUID) {
            return _.find(textureCanvasPatternCache, function (obj) { return obj.id === drawingUID; });
        }

        function setBitmapFillForPathShape(textureFillStyle, drawingUID) {
            // store texture for performance
            if (isResizeActive) {
                if (!getCachedTextureFill(textureCanvasPatternCache, drawingUID)) { // save texture canvas pattern fill to cache with drawing id, if not there already
                    textureCanvasPatternCache.push({ id: drawingUID, texture: textureFillStyle });
                }
            } else {
                textureCanvasPatternCache = [];
            }

            context.render(function () {
                context.setGlobalComposite('source-in'); // The new shape is drawn only where both the new shape and the destination canvas overlap. Everything else is made transparent.
                if (notRotatedWithShape && rotationAngle) {
                    var rectPos = max(widthPx, heightPx);
                    context.translate(widthPx / 2, heightPx / 2);
                    context.rotate(rotationAngle);
                    context.setFillStyle(textureFillStyle);

                    if (notRotatedWithShape && !fillAttrs.bitmap.tiling) {
                        var nWidth = widthPx * abs(cos(rotationAngle)) + heightPx * abs(sin(rotationAngle));
                        var nHeight = heightPx * abs(cos(rotationAngle)) + widthPx * abs(sin(rotationAngle));
                        context.translate(-nWidth / 2, -nHeight / 2);
                        context.drawRect(0, 0, nWidth, nHeight, 'fill'); // cover whole canvas area with pattern fill
                    } else {
                        context.drawRect(2 * -rectPos, 2 * -rectPos, 4 * rectPos, 4 * rectPos, 'fill'); // cover whole canvas area with pattern fill
                    }
                } else {
                    context.setFillStyle(textureFillStyle);
                    context.drawRect(0, 0, widthPx, heightPx, 'fill');
                }
            });
            // second pass for rendering path on top, after bitmap fill is applied
            geometryAttrs.pathList.forEach(function (onePath, index) {
                processOnePath(onePath, index, { secondPass: true });
            });
        }

        function processOnePath(onePath, index, options) {
            // initialize context with proper fill attributes for the next path
            var path            = _.copy(onePath, true); // creating a deep copy of the path, so that it can be expanded
            var logWidth        = ('width' in path) ? path.width : widthPx;
            var logHeight       = ('height' in path) ? path.height : heightPx;
            var fXScale         = widthPx / logWidth;
            var fYScale         = heightPx / logHeight;
            var hasFill         = fillAttrs && (fillAttrs.type !== 'none') && (path.fillMode !== 'none');
            var hasLine         = lineAttrs && (lineAttrs.type !== 'none') && (path.isStroke !== false);
            var renderMode      = getRenderMode(hasFill, hasLine);
            var shadeMode       = path.fillMode;
            var isSecondPass    = (options && options.secondPass) || false;
            var fillStyle       = null;

            // nothing to do without valid path size
            if ((logWidth <= 0) || (logHeight <= 0) || !renderMode) { return; }

            function shadeColor(jsonColor) {
                var overlayData = shadeMode ? SHADED_OVERLAY_MAP[shadeMode] : null;
                if (!overlayData) { return jsonColor; }

                var color = Color.parseJSON(jsonColor);
                var colorDesc = color.resolve('fill', themeModel);
                var mixedDesc = color.resolveMixed(overlayData.color, overlayData.alpha, 'fill', themeModel);
                return new Color('rgb', mixedDesc.hex).transform('alpha', colorDesc.a * 100000).toJSON();
            }

            function getShadedBitmapFill(bitmapFill) {
                var overlayColor, sumAlpha, bitmapColor, shadedFill;
                var overlayDataBitmap = SHADED_OVERLAY_MAP[shadeMode];
                if (overlayDataBitmap) {
                    overlayColor = overlayDataBitmap.color.toJSON();
                    sumAlpha = _.isFinite(bitmapFill.transparency) ? overlayDataBitmap.alpha * bitmapFill.transparency : overlayDataBitmap.alpha;
                    bitmapColor = new Color('rgb', overlayColor.value).transform('alpha', sumAlpha * 100000).toJSON();
                    shadedFill = docModel.getCssColor(bitmapColor, 'fill', targets);
                } else {
                    shadedFill = 'transparent';
                }
                return shadedFill;
            }

            /**
             * Local function to draw final path of canvas.
             *
             * @param {Object} [options]
             *  Optional parameters:
             *  @param {String} [options.mode]
             *      A specific rendering mode used to override the original
             *      rendering mode of the path to be drawn (one of 'stroke',
             *      'fill', or 'all').
             */
            function drawFinalPath(options) {

                // the new converted path with resolved path commands
                var newPath = calcFinalPath(path, fXScale, fYScale);
                // the effective rendering mode (may be overridden by passed options)
                var localMode = Utils.getStringOption(options, 'mode', renderMode);

                context.drawPath(newPath, localMode);
            }

            if (hasFill) {
                if (fillAttrs.type === 'gradient') {
                    var angle = !fillAttrs.gradient.isRotateWithShape ? DrawingFrame.getDrawingRotationAngle(docModel, currentNode) || 0 : 0;
                    if (angle) {
                        // rotate the triangle (rotate the gradient not with shape) to get the new width and height
                        var drawingAttrs = docModel.getDrawingStyles().getElementAttributes(currentNode).drawing; // #53436
                        var rotatedAttrs = DrawingUtils.getRotatedDrawingPoints(drawingAttrs, angle);
                        logWidth = Utils.convertHmmToLength(rotatedAttrs.width, 'px', 1);
                        logHeight = Utils.convertHmmToLength(rotatedAttrs.height, 'px', 1);
                    }

                    // extract the JSON representation of the gradient from the fill attributes
                    var jsonGradient = Gradient.getDescriptorFromAttributes(fillAttrs);

                    // apply shading effect to all color stops of the gradient
                    if (shadeMode) {
                        jsonGradient.color = shadeColor(jsonGradient.color);
                        jsonGradient.color2 = shadeColor(jsonGradient.color2);
                        if (_.isArray(jsonGradient.colorStops)) {
                            jsonGradient.colorStops = jsonGradient.colorStops.map(function (colorStop) {
                                colorStop = _.clone(colorStop);
                                colorStop.color = shadeColor(colorStop.color);
                                return colorStop;
                            });
                        }
                    }

                    var gradient    = Gradient.create(jsonGradient),

                        halfWidth   = round(logWidth / 2),
                        halfHeight  = round(logHeight / 2),

                        pathCoords  = path.commands.reduce(collectPathCoordinates, {
                            pathCoords: {
                                xValues: [halfWidth, logWidth, halfWidth, 0],
                                yValues: [0, halfHeight, logHeight, halfHeight],
                                mx: halfWidth,
                                my: halfHeight
                            },
                            cx: 0,
                            cy: 0,
                            fXScale: fXScale,
                            fYScale: fYScale,

                            getGeometryValue: getGeoValue

                        }).pathCoords;

                    if (angle) {
                        // rotate the triangle (rotate the gradient not with shape) coords
                        if (pathCoords.xValues.length === pathCoords.yValues.length) {
                            for (var i = 0; i < pathCoords.xValues.length; i++) {
                                var x = pathCoords.xValues[i];
                                var y = pathCoords.yValues[i];
                                var rotPoint = DrawingUtils.rotatePointWithAngle(pathCoords.mx, pathCoords.my, x, y, angle);
                                pathCoords.xValues[i] = rotPoint.left;
                                pathCoords.yValues[i] = rotPoint.top;
                            }
                        }
                    }

                    fillStyle = docModel.getGradientFill(gradient, pathCoords, context, targets);
                } else if (fillAttrs.type === 'pattern') {
                    fillStyle = Pattern.createCanvasPattern(fillAttrs.pattern, shadeColor(fillAttrs.color2), shadeColor(fillAttrs.color), themeModel);
                } else if (isBitmapFill) {
                    if (isSecondPass) {
                        if (shadeMode) { fillStyle = getShadedBitmapFill(fillAttrs.bitmap); }
                    } else {
                        fillStyle = '#f8f8f8'; // temp light value, used for masking in applying bitmap fill
                    }
                } else {
                    fillStyle = docModel.getCssColor(shadeColor(fillAttrs.color), 'fill', targets);
                }
                context.setFillStyle(fillStyle);
            }

            // initialize context with proper line attributes for the next path
            var additionalPaths = hasLine ? setLineAttributes(docModel, context, lineAttrs, null, path, {
                isFirstPath: index === 0,
                isLastPath: index + 1 === geometryAttrs.pathList.length,
                snapRect: snapRect,
                avList: avList,
                gdList: gdList,
                guideCache: guideCache
            }) : null;

            drawFinalPath();

            if (additionalPaths && (additionalPaths.length > 0)) {
                // bug 50284: no line patterns for arrows at line ends
                context.setLineStyle({ pattern: null });
                _.each(additionalPaths, function (onePath) {
                    var newPath = calcFinalPath(onePath, fXScale, fYScale);
                    context.drawPath(newPath, onePath.drawMode);
                });
            }
        } // end of processOnePath function

        if (isBitmapFill) {
            context.clearRect(0, 0, widthPx, heightPx);
        }

        geometryAttrs.pathList.forEach(function (onePath, index) {
            processOnePath(onePath, index, { secondPass: false });
        });

        if (isBitmapFill) {
            if (notRotatedWithShape && currentNode) {
                rotationAngle = DrawingFrame.getDrawingRotationAngle(docModel, currentNode) || 0;
                rotationAngle = ((360 - rotationAngle) % 360) * PI_180; // normalize and convert to radians
            }
            cachedTextureObj = getCachedTextureFill(textureCanvasPatternCache, drawingUID);
            if (isResizeActive && cachedTextureObj && cachedTextureObj.texture) { // Performance: during resize, try to fetch texture pattern from cache
                setBitmapFillForPathShape(cachedTextureObj.texture, cachedTextureObj.id);
            } else {
                Texture.getTextureFill(docModel, context, fillAttrs, widthPx, heightPx, rotationAngle).done(function (textureFillStyle) {
                    setBitmapFillForPathShape(textureFillStyle, drawingUID);
                });
            }
        }
    } // end of drawShape function

    function getGeometryValue(value, snapRect, avList, gdList, guideCache, maxGdIndex) {
        if (typeof value === 'number') {
            return value;
        }
        if (value in guideCache) {
            return guideCache[value];
        }
        switch (value) {
            // 3 x 360deg / 4 = 270deg
            case '3cd4':
                return 270 * 60000;
            // 3 x 360deg / 8 = 135deg
            case '3cd8':
                return 135 * 60000;
            // 5 x 360deg / 8 = 225deg
            case '5cd8':
                return 225 * 60000;
            // 7 x 360deg / 8 = 315deg
            case '7cd8':
                return 315 * 60000;
            // bottom
            case 'b':
                return snapRect.top + snapRect.height - 1;
            // 360deg / 2 = 180deg
            case 'cd2':
                return 180 * 60000;
            // 360deg / 4 = 90deg
            case 'cd4':
                return 90 * 60000;
            // 360deg / 8 = 45deg
            case 'cd8':
                return 45 * 60000;
            // horizontal center
            case 'hc':
                return snapRect.left + snapRect.width / 2;
            // height
            case 'h':
                return snapRect.height;
            // height / 2
            case 'hd2':
                return snapRect.height / 2;
            // height / 3
            case 'hd3':
                return snapRect.height / 3;
            // height / 4
            case 'hd4':
                return snapRect.height / 4;
            // height / 5
            case 'hd5':
                return snapRect.height / 5;
            // height / 6
            case 'hd6':
                return snapRect.height / 6;
            // height / 8
            case 'hd8':
                return snapRect.height / 8;
            // left
            case 'l':
                return snapRect.left;
            // long side
            case 'ls':
                return max(snapRect.width, snapRect.height);
            // right
            case 'r':
                return snapRect.left + snapRect.width - 1;
            // short side
            case 'ss':
                return min(snapRect.width, snapRect.height);
            // short side / 2
            case 'ssd2':
                return min(snapRect.width, snapRect.height) / 2;
            // short side / 4
            case 'ssd4':
                return min(snapRect.width, snapRect.height) / 4;
            // short side / 6
            case 'ssd6':
                return min(snapRect.width, snapRect.height) / 6;
            // short side / 8
            case 'ssd8':
                return min(snapRect.width, snapRect.height) / 8;
            // short side / 16
            case 'ssd16':
                return min(snapRect.width, snapRect.height) / 16;
            // short side / 32
            case 'ssd32':
                return min(snapRect.width, snapRect.height) / 32;
            // top
            case 't':
                return snapRect.top;
            // vertical center
            case 'vc':
                return snapRect.top + snapRect.height / 2;
            // width
            case 'w':
                return snapRect.width;
            // width / 2
            case 'wd2':
                return snapRect.width / 2;
            case 'wd3':
                return snapRect.width / 3;
            // width / 4
            case 'wd4':
                return snapRect.width / 4;
            // width / 5
            case 'wd5':
                return snapRect.width / 5;
            // width / 6
            case 'wd6':
                return snapRect.width / 6;
            // width / 8
            case 'wd8':
                return snapRect.width / 8;
            // width / 10
            case 'wd10':
                return snapRect.width / 10;
            // width / 32
            case 'wd32':
                return snapRect.width / 32;
        }
        // the value is not a constant value, next check if this is a guide name
        if (avList && value in avList) {
            return avList[value];
        }
        if (gdList) {
            if (typeof maxGdIndex !== 'number') { maxGdIndex = 1000; }
            for (var i = 0; i < gdList.length; i += 1) {
                if (gdList[i].name === value) {
                    if (i > maxGdIndex) {
                        return 0.0;
                    }
                    var result = calcGeometryValue(gdList[i], snapRect, avList, gdList, guideCache, i - 1);
                    guideCache[value] = result;
                    return result;
                }
            }
        }

        return parseInt(value, 10);
    }

    function calcGeometryValue(gdEntry, snapRect, avList, gdList, guideCache, maxGdIndex) {

        // formulas with one parameter
        var p0 = getGeometryValue(gdEntry.p0, snapRect, avList, gdList, guideCache, maxGdIndex);
        switch (gdEntry.op) {
            // Absolute value: "abs x" := |x|
            case 'abs': return abs(p0);
            // Square root: "sqrt x" := sqrt(x)
            case 'sqrt': return sqrt(p0);
            // Literal value: "val x" := x
            case 'val': return p0;
        }

        // formulas with two parameters
        var p1 = getGeometryValue(gdEntry.p1, snapRect, avList, gdList, guideCache, maxGdIndex);
        switch (gdEntry.op) {
            // Arcus tangent: "at2 x y" := arctan2(y,x)
            case 'at2': return atan2(p1, p0) / PI_RAD;
            // Cosine: "cos x y" := x * cos(y)
            case 'cos': return p0 * cos(p1 * PI_RAD);
            // Maximum value: "max x y" := max(x,y)
            case 'max': return max(p0, p1);
            // Minimum value: "min x y" := min(x,y)
            case 'min': return min(p0, p1);
            // Sine: "sin x y" := x * sin(y)
            case 'sin': return p0 * sin(p1 * PI_RAD);
            // Tangent: "tan x y" = x * tan(y)
            case 'tan': return p0 * tan(p1 * PI_RAD);
        }

        // formulas with three parameters
        var p2 = getGeometryValue(gdEntry.p2, snapRect, avList, gdList, guideCache, maxGdIndex);
        switch (gdEntry.op) {
            // Multiply and divide: "*/ x y z" := x * y / z
            case '*/': return p0 * p1 / p2;
            // Add and subtract: "+- x y z" := x + y - z
            case '+-': return p0 + p1 - p2;
            // Add and divide: "+/ x y z" := (x + y) / z
            case '+/': return (p0 + p1) / p2;
            // If-then-else: "?: x y z" := if (x > 0) then y else z
            case '?:': return (p0 > 0) ? p1 : p2;
            // Cosine of arcus tangent: "cat2 x y z" := x * cos(arctan(z,y))
            case 'cat2': return p0 * cos(atan2(p2, p1));
            // Modulo: "mod x y z" := sqrt(x^2 + y^2 + z^2)
            case 'mod': return sqrt(p0 * p0 + p1 * p1 + p2 * p2);
            // Pin to: "pin x y z" := if (y < x) then x else if (y > z) then z else y
            case 'pin': return Utils.minMax(p1, p0, p2);
            // Sine of arcus tangent: "sat2 x y z" := x * sin(arctan(z,y))
            case 'sat2': return p0 * sin(atan2(p2, p1));
        }

        Utils.warn('DrawingFrame.calcGeometryValue(): invalid operation: ', gdEntry.op);
        return 0;
    }

    /**
     * Renders a preview of the specified predefined shape object into a canvas
     * that already exists in or will be embedded into the passed DOM element.
     *
     * @param {EditApplication} app
     *  The application instance containing the passed frame node.
     *
     * @param {jQuery|HTMLElement} frameNode
     *  The DOM container element for the canvas element to be rendered into.
     *  If this container element is empty, a new canvas element will be
     *  created and inserted automatically.
     *
     * @param {Object} shapeSize
     *  The total size of the shape, in pixels, in the properties 'width' and
     *  'height'.
     *
     * @param {String} presetShapeId
     *  The identifier for a predefined shape type.
     *
     * @param {Object} [options]
     *  Optional parameters:
     *  @param {Rectangle} [options.snapRect]
     *      The location of the shape in the bounding box defined by the size
     *      passed with the parameter 'shapeSize'. If omitted, the shape will
     *      be located in the entire bounding box.
     *  @param {Object} [options.attrs]
     *      Additional formatting attributes to be merged over the attributes
     *      returned by DrawingUtils.createDefaultShapeAttributeSet() according
     *      to the specified shape type.
     *  @param {Boolean} [options.flipH=false]
     *      If set to true, the shape will be flipped horizontally.
     *  @param {Boolean} [options.flipV=false]
     *      If set to true, the shape will be flipped vertically.
     *  @param {Boolean} [options.transparent=false]
     *      If set to true, the shape will be rendered semi-trnasparently.
     *  @param {Boolean} [options.drawIconLabel=false]
     *      Whether to render the predefined icon label into the shape center.
     *  @param {String} [options.iconLabelFont]
     *      The font style of the icon label. This option will have any effect
     *      only if the option 'drawIconLabel' has been set.
     *
     * @returns {jQuery}
     *  The DOM canvas node, as jQuery element.
     */
    DrawingFrame.drawPreviewShape = function (app, frameNode, shapeSize, presetShapeId, options) {

        // geometry data for the specified shape type
        var presetData = Labels.getPresetShape(presetShapeId);
        if (!presetData) { return; }

        // location of the canvas (always at zero offset)
        var boundRect = new Rectangle(0, 0, shapeSize.width, shapeSize.height);
        // location of the shape in the canvas
        var snapRect = Utils.getObjectOption(options, 'snapRect', boundRect);

        // the canvas element (create a new element on demand)
        var canvasNode = $(frameNode).find('>canvas').first();
        if (canvasNode.length === 0) {
            canvasNode = $('<canvas class="' + DrawingFrame.CANVAS_CLASS + '" style="position:absolute;">').appendTo(frameNode);
        }

        // set flipping styles to the canvas element
        var scaleX = Utils.getBooleanOption(options, 'flipH', false) ? -1 : 1;
        var scaleY = Utils.getBooleanOption(options, 'flipV', false) ? -1 : 1;
        canvasNode.css('transform', 'scaleX(' + scaleX + ') scaleY(' + scaleY + ')');

        // set the canvas element to semi-transparency if specified
        var transparent = Utils.getBooleanOption(options, 'transparent', false);
        canvasNode.css('opacity', transparent ? '0.6' : '');

        // create the formatting attributes for the shape
        var shapeAttrSet = DrawingUtils.createDefaultShapeAttributeSet(app.getModel(), presetShapeId);
        var addAttrSet = Utils.getObjectOption(options, 'attrs', null);
        if (addAttrSet) { app.getModel().getStyleCollection('drawing').extendAttributeSet(shapeAttrSet, addAttrSet); }

        // the rendering helper for the canvas element in the passed container node
        var canvas = new Canvas(app, { node: canvasNode });
        initializeCanvasSize(canvas, 'shape', boundRect, getLineWidthPx(shapeAttrSet.line));

        // render the shape into the canvas element
        canvas.render(function (context) {
            drawShape(app, context, snapRect, presetData.shape, shapeAttrSet, {});
            if (presetData.iconLabel && Utils.getBooleanOption(options, 'drawIconLabel', false)) {
                var fontStyle = { align: 'center', baseline: 'middle' };
                var font = Utils.getStringOption(options, 'iconLabelFont', null);
                if (font) { fontStyle.font = font; }
                context.setFontStyle(fontStyle);
                context.drawText(presetData.iconLabel, snapRect.centerX(), snapRect.centerY(), 'fill');
            }
        });

        // destroy the wrapper for the canvas node
        canvas.destroy();
        return canvasNode;
    };

    // draws the border to the canvas
    DrawingFrame.drawBorder = function (app, currentNode, lineAttrs) {

        var // the document model
            docModel = app.getModel(),
            // border width
            lineWidth = getLineWidthPx(lineAttrs),
            halfBorderSize = ceil(lineWidth / 2),
            // dimensions of the canvas element, special behavior for text frames. Their border covers their boundary
            cWidth = currentNode.first().width(),
            cHeight = currentNode.first().height(),
            // the canvas wrapper of the border node
            canvas = initializeCanvas(app, currentNode, new Rectangle(0, 0, cWidth, cHeight), lineWidth);

        canvas.render(function (context, width, height) {
            var color = docModel.getCssColor(lineAttrs.color, 'line', lineAttrs[DrawingFrame.TARGET_CHAIN_PROPERTY]),
                pattern = Border.getBorderPattern(lineAttrs.style, lineWidth);

            context.setLineStyle({ style: color, width: lineWidth, pattern: pattern });

            var drawX = 0,
                drawY = 0,
                drawW = (width - 2 * halfBorderSize),
                drawH = (height - 2 * halfBorderSize);

            context.drawRect(drawX, drawY, drawW, drawH, 'stroke');
        });

        if (app && app.attributes && app.attributes.name === 'io.ox/office/spreadsheet') {
            //TODO: should be checked from outside!
            DrawingFrame.setCanvasDynHeight(currentNode);
        }
    };

    // clears the canvas
    function clearCanvas(app, currentNode) {

        var // the border canvas node inside the drawing frame
            canvasNode = DrawingFrame.getCanvasNode(currentNode),
            // the canvas wrapper of the border node
            canvas = null;

        if (canvasNode.length > 0) {
            canvas = new Canvas(app, { classes: DrawingFrame.CANVAS_CLASS, node: canvasNode });
            canvas.initialize({ width: canvasNode.width(), height: canvasNode.height() });
        }
    }

    /**
     * Returns rotation angle of given drawing node.
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Number|null}
     *  Angle of rotation, or null if no rotation is set.
     */
    DrawingFrame.getDrawingRotationAngle = function (docModel, drawingNode) {
        var attrs = null;
        var rotation = null;
        if (DrawingFrame.isRotatedDrawingNode(drawingNode)) {
            // first trying to fetch direct attributes of the element
            attrs = AttributeUtils.getExplicitAttributes(drawingNode);
            rotation = (attrs && attrs.drawing && attrs.drawing.rotation) || null;
            if (!rotation) {
                // if not fetched, get from inherited attributes
                attrs = docModel.getDrawingStyles().getElementAttributes(drawingNode);
                rotation = (attrs && attrs.drawing && attrs.drawing.rotation) || null;
            }
        }
        return rotation;
    };

    /**
     * Sets the value of the CSS attribute 'transform' for the passed rotation
     * angle and flipping flags.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame to be updated. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {Number} degrees
     *  The rotation angle, in degrees.
     *
     * @param {Boolean} flipH
     *  Whether the drawing frame will be flipped horizontally.
     *
     * @param {Boolean} flipV
     *  Whether the drawing frame will be flipped vertically.
     */
    DrawingFrame.setCssTransform = function (drawingFrame, degrees, flipH, flipV) {
        $(drawingFrame).css('transform', DrawingUtils.getCssTransform(degrees, flipH, flipV));
    };

    /**
     * Updates the value of the CSS attribute 'transform' for the passed
     * rotation angle, and the current flipping flags of the drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame to be updated. If this object is a
     *  jQuery collection, uses the first DOM node it contains.
     *
     * @param {Number} degrees
     *  The rotation angle, in degrees.
     */
    DrawingFrame.updateCssTransform = function (drawingFrame, degrees) {
        var flipH = DrawingFrame.isFlippedHorz(drawingFrame);
        var flipV = DrawingFrame.isFlippedVert(drawingFrame);
        DrawingFrame.setCssTransform(drawingFrame, degrees, flipH, flipV);
    };

    /**
     * Normalizes pixel coordinates for move of rotated drawings.
     *
     * @param {Number} shiftX
     *  Original (from event) move shift in x axis.
     *
     * @param {Number} shiftY
     *  Original (from event) move shift in y axis.
     *
     * @param {Number} angle
     *  Rotation angle of the drawing object in degrees.
     *
     * @param {Boolean} flipH
     *  Whether or not drawing object is flipped (mirrored) horizontally.
     *
     * @param {Boolean} flipV
     *  Whether or not drawing object is flipped (mirrored) vertically.
     *
     * @returns {Object} normalized x and y values.
     */
    DrawingFrame.getNormalizedMoveCoordinates = function (shiftX, shiftY, angle, flipH, flipV) {

        var rad = _.isNumber(angle) ? (angle * PI_180) : 0;
        if (rad) {
            var tempX = shiftX * Math.cos(rad) + shiftY * Math.sin(rad);
            var tempY = shiftY * Math.cos(rad) - shiftX * Math.sin(rad);
            shiftX = tempX;
            shiftY = tempY;
        }

        return {
            x: flipH ? -shiftX : shiftX,
            y: flipV ? -shiftY : shiftY
        };
    };

    /**
     * Normalizes pixel coordinates for resize of rotated drawings.
     *
     * @param {Number} deltaXo
     *  Original (from event) delta value of resized shape in x axis.
     *
     * @param {Number} deltaYo
     *  Original (from event) delta value of resized shape in y axis.
     *
     * @param {Boolean} useX
     *  Whether or not x resize is used.
     *
     * @param {Boolean} useY
     *  Whether or not y resize is used.
     *
     * @param {Number} scaleX
     *  If x value is scaled (inverted). Can be 1 or -1.
     *
     * @param {Number} scaleY
     *  If y value is scaled (inverted). Can be 1 or -1.
     *
     * @param {Number} angle
     *  Rotation angle of the drawing object
     *
     *
     * @returns {Object} normalized x and y values.
     */
    DrawingFrame.getNormalizedResizeDeltas = function (deltaXo, deltaYo, useX, useY, scaleX, scaleY, angle) {
        var localX = null;
        var localY = null;
        var rad = _.isNumber(angle) ? (angle * PI_180) : null;

        if (rad) {
            localX = useX ? (deltaXo * Math.cos(rad) + deltaYo * Math.sin(rad)) * scaleX : 0;
            localY = useY ? (deltaYo * Math.cos(rad) - deltaXo * Math.sin(rad)) * scaleY : 0;
        } else {
            localX = useX ? deltaXo * scaleX : 0;
            localY = useY ? deltaYo * scaleY : 0;
        }

        return { x: localX, y: localY };
    };

    /**
     * Normalizes offset of rotated drawings. Returns object of top and left offset properties.
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {HTMLElement|jQuery} moveBox
     *  Helper move box frame node used on move. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Object} left and top calculated values.
     */
    DrawingFrame.normalizeMoveOffset = function (docModel, drawingNode, moveBox) {
        var $drawingNode = $(drawingNode);
        var moveBoxOffset = moveBox.offset();
        var moveBoxOffsetLeft = moveBoxOffset.left;
        var moveBoxOffsetTop = moveBoxOffset.top;
        var angle = DrawingFrame.getDrawingRotationAngle(docModel, drawingNode);
        var tLeft, tTop, dLeft, dTop;
        var flipH = DrawingFrame.isFlippedHorz(drawingNode);
        var flipV = DrawingFrame.isFlippedVert(drawingNode);
        var drawingNodeOffset = null;

        if (_.isNumber(angle)) {
            $drawingNode.css('transform', ''); // reset to def to get values
            drawingNodeOffset = $drawingNode.offset();
            tLeft = drawingNodeOffset.left;
            tTop = drawingNodeOffset.top;
            DrawingFrame.setCssTransform($drawingNode, angle, flipH, flipV);

            drawingNodeOffset = $drawingNode.offset();
            dLeft = tLeft - drawingNodeOffset.left;
            dTop = tTop - drawingNodeOffset.top;

            moveBoxOffsetLeft += dLeft;
            moveBoxOffsetTop += dTop;
        }
        return { left: moveBoxOffsetLeft, top: moveBoxOffsetTop };
    };

    /**
     * Normalize resize offset of rotated drawings. Returns object of top and left offset properties,
     * and sendMoveOperation flag, to send appropriate move operations if drawing is rotated and needs to be translated to the position.
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {HTMLElement|jQuery} moveBox
     *  Helper move box frame node used on resize. If this object is a jQuery
     *  collection, uses the first DOM node it contains.,
     *
     * @param {Boolean} useTop
     *  If top edge is used during resize.
     *
     * @param {Boolean} useLeft
     *  If left edge is used during resize.
     *
     * @returns {Object} left and top calculated values, plus flag for sending move operation or not.
     *
     */
    DrawingFrame.normalizeResizeOffset = function (docModel, drawingNode, moveBox, useTop, useLeft) {
        var moveBoxOffset, moveBoxOffsetTop, moveBoxOffsetLeft;
        var sendMoveOperation = false;
        var angle = DrawingFrame.getDrawingRotationAngle(docModel, drawingNode);
        var isFlipH = DrawingFrame.isFlippedHorz(drawingNode);
        var isFlipV = DrawingFrame.isFlippedVert(drawingNode);

        if (_.isNumber(angle)) {
            if (isFlipH !== isFlipV) { angle = 360 - angle; }
            moveBox.css({ transform: 'rotate(' + (-angle) + 'deg)' });
            moveBoxOffset = moveBox.offset();
            moveBoxOffsetLeft = moveBoxOffset.left;
            moveBoxOffsetTop = moveBoxOffset.top;
            moveBox.css({ transform: '' });

            sendMoveOperation = true;
        } else {
            moveBoxOffset = moveBox.offset();
            moveBoxOffsetLeft = moveBoxOffset.left;
            moveBoxOffsetTop = moveBoxOffset.top;

            if ((!useTop && isFlipV) || (!useLeft && isFlipH)) { sendMoveOperation = true; }
        }

        return { left: moveBoxOffsetLeft, top: moveBoxOffsetTop, sendMoveOperation: sendMoveOperation };
    };

    /**
     * Returns difference in top, left or bottom positions of original drawing node, and move box, on resize finalize.
     * It also normalizes position values against rotation angle or flips.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {HTMLElement|jQuery} moveBox
     *  Helper move box frame node used on resize. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Number|Null} angle
     *  Rotation angle of drawing, in degrees.
     *
     * @param {Boolean} [isFlipH]
     *  If passed, determines if shape is flipped horizontally.
     *
     * @param {Boolean} [isFlipV]
     *  If passed, determines if shape is flipped vertically.
     *
     * @returns {Object} left, top and bottom difference values.
     */
    DrawingFrame.getPositionDiffAfterResize = function (drawingNode, moveBox, angle, isFlipH, isFlipV) {
        var $drawingNode = $(drawingNode);
        var drawingNodeRect = null;
        var moveBoxRect = null;
        if (typeof isFlipH !== 'boolean') { isFlipH = DrawingFrame.isFlippedHorz(drawingNode); }
        if (typeof isFlipV !== 'boolean') { isFlipV = DrawingFrame.isFlippedVert(drawingNode); }

        if (_.isFinite(angle) && angle !== 0) {
            $drawingNode.css('transform', ''); // reset to def to get values
            drawingNodeRect = $drawingNode[0].getBoundingClientRect();
            DrawingFrame.setCssTransform($drawingNode, angle, isFlipH, isFlipV);
            if (isFlipH !== isFlipV) { angle = 360 - angle; }
            moveBox.css('transform', 'rotate(' + (-angle) + 'deg)');
            moveBoxRect = moveBox[0].getBoundingClientRect();
            moveBox.css('transform', '');
        } else {
            drawingNodeRect = $drawingNode[0].getBoundingClientRect();
            moveBoxRect = moveBox[0].getBoundingClientRect();
        }
        return { leftDiff: drawingNodeRect.left - moveBoxRect.left, topDiff: drawingNodeRect.top - moveBoxRect.top, bottomDiff: drawingNodeRect.bottom - moveBoxRect.bottom };
    };

    DrawingFrame.setCanvasDynHeight = function (drawingFrame) {
        var canvasNode = DrawingFrame.getCanvasNode(drawingFrame);
        if (!canvasNode.length) { return; }

        var position = canvasNode.position();
        // css calc is used to keep canvas in its frame while zooming
        canvasNode.css({ width: 'calc(100% - ' + (2 * position.left) + 'px)',  height: 'calc(100% - ' + (2 * position.top) + 'px)' });
    };

    // selection --------------------------------------------------------------

    /**
     * Returns whether the specified drawing frame is currently selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {Boolean}
     *  Whether the specified drawing frame is selected.
     */
    DrawingFrame.isSelected = function (drawingFrame) {
        return Forms.isSelectedNode(drawingFrame);
    };

    /**
     * Creates or updates additional nodes displayed while a drawing frame is
     * selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Boolean} [options.movable=false]
     *      If set to true, mouse pointer will be changed to a movable pointer
     *      while the mouse hovers the drawing.
     *  @param {Boolean} [options.resizable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      resize pointer while the mouse hovers a resize handle.
     *  @param {Boolean} [options.rotatable=false]
     *      If set to true, mouse pointer will be changed to an appropriate
     *      rotate pointer while the mouse hovers a rotate handle.
     *  @param {Number} [options.scaleHandles=1]
     *      the usable handles are scaled or themselves
     *      (without changing the size of the whole selection)
     *  @param {Number} [options.rotation=0]
     *      The rotation angle of the drawing frame.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.drawSelection = function (drawingFrame, options) {

        // HTML mark-up for the selection
        var markup = null;
        // whether the two point selection for specified drawings can be used
        var twoPointShape = DrawingFrame.isTwoPointShape(drawingFrame);
        // the drawing's content node
        var contentNode = null;
        // whether the drawing's content is horizontally flipped
        var isHorzFlip = false;
        // whether the drawing's content is vertically flipped
        var isVertFlip = false;
        // rotation angle in degrees
        var angle = Utils.getNumberOption(options, 'rotation', 0);

        // restrict to a single node
        drawingFrame = $(drawingFrame).first();

        // create new selection nodes if missing
        if (!DrawingFrame.isSelected(drawingFrame)) {

            // create a new selection node
            markup = '<div class="' + SELECTION_CLASS + '">';

            // create the border nodes
            markup += '<div class="borders">';
            _.each('tblr', function (pos) {
                markup += '<div data-pos="' + pos + '"></div>';
            });
            markup += '</div>';

            // create resizer handles
            var drawingResizers = null;

            if (twoPointShape) { // finding the required resize handler nodes for connector drawings
                contentNode = DrawingFrame.getContentNode(drawingFrame);
                isHorzFlip = DrawingFrame.isFlippedHorz(contentNode);
                isVertFlip = DrawingFrame.isFlippedVert(contentNode);
                drawingResizers = (isHorzFlip !== isVertFlip) ? ['tr', 'bl'] : ['br', 'tl'];
            } else {
                drawingResizers = ['t', 'tr', 'r', 'br', 'b', 'bl', 'l', 'tl'];
            }

            markup += '<div class="resizers">';
            drawingResizers.forEach(function (pos) {
                markup += '<div data-pos="' + pos + '"></div>';
            });
            markup += '</div>';

            // create rotate handle
            markup += '<div class="rotate-handle"></div>';

            // create the tracker node
            markup += '<div class="' + TRACKER_CLASS + '"></div>';

            // close the selection node
            markup += '</div>';

            drawingFrame.append(markup);
        }

        // update state classes at the drawing frame
        drawingFrame
            .addClass(Forms.SELECTED_CLASS)
            .toggleClass('movable', Utils.getBooleanOption(options, 'movable', false))
            .toggleClass('resizable', Utils.getBooleanOption(options, 'resizable', false))
            .toggleClass('rotatable', Utils.getBooleanOption(options, 'rotatable', false))
            .toggleClass('two-point-shape', twoPointShape);

        DrawingFrame.updateScaleHandles(drawingFrame, options);
        if (angle) { DrawingFrame.updateResizersMousePointers(drawingFrame, angle); }

        return getSelectionNode(drawingFrame);
    };

    /**
     * updates the additional nodes displayed while a drawing frame is selected.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Object} [options]
     *  A map with options to control the appearance of the selection. The
     *  following options are supported:
     *  @param {Number} [options.scaleHandles=1]
     *      the usable handles are scaled or themselves
     *      (without changing the size of the whole selection)
     *      range is 1 - 3
     */
    DrawingFrame.updateScaleHandles = function (drawingFrame, options) {
        var
            zoomValue     = Utils.getNumberOption(options, 'zoomValue', 100),
            scaleHandles  = Utils.getNumberOption(options, 'scaleHandles', 1, 1, 3), // scaleHandles for setting fontsize to scale inner elements

            selectionNode = getSelectionNode(drawingFrame);

        if (zoomValue >= 100) {
            selectionNode.attr('data-zoom-value', zoomValue); // - new css rule based approach for reasonable selection rendering for "zoom in".
        }
        selectionNode.css('font-size', scaleHandles + 'rem'); // - the already working "root element width" based approach for "zoo out".
    };

    /**
     * Updates mouse pointers for the resizers of the drawing frame, according to the rotation angle of the drawing.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @param {Number} angle
     *  Rotation angle of the drawing, in degrees.
     */
    DrawingFrame.updateResizersMousePointers = function (drawingFrame, angle) {
        var resizeHandles, shift;
        var isFlipH = DrawingFrame.isFlippedHorz(drawingFrame);
        var isFlipV = DrawingFrame.isFlippedVert(drawingFrame);
        var xorFlip = isFlipH !== isFlipV;
        var pointersClockwise = ['n-resize', 'ne-resize', 'e-resize', 'se-resize', 's-resize', 'sw-resize', 'w-resize', 'nw-resize'];
        var pointersCounterCw = ['n-resize', 'nw-resize', 'w-resize', 'sw-resize', 's-resize', 'se-resize', 'e-resize', 'ne-resize'];
        var pointers = xorFlip ? pointersCounterCw : pointersClockwise;

        angle = _.isNumber(angle) ? angle : 0;

        if (angle || xorFlip) {
            if (xorFlip) { angle = 360 - angle; }
            angle += 22.5; // start offset
            angle = Utils.mod(angle, 360);
            shift = Math.floor(angle / 45); // 8 handles makes 8 areas with 45 degrees each
            resizeHandles = drawingFrame.find('.resizers').children();
            _.each(resizeHandles, function (handle, index) {
                var pointerInd = (index + shift) % pointers.length;
                $(handle).css('cursor', pointers[pointerInd]);
            });
        }
    };

    /**
     * Removes the selection node from the specified drawing frame.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The selection root node contained in the drawing frame, as jQuery
     *  object.
     */
    DrawingFrame.clearSelection = function (drawingFrame) {
        $(drawingFrame).first().removeClass(Forms.SELECTED_CLASS + ' movable resizable');
        getSelectionNode(drawingFrame).remove();
        DrawingFrame.removeDrawingSelection(drawingFrame);
    };

    /**
     * Helper function to set the size, position and rotation of a selection of a drawing node.
     * This handles selection objects that are located in the selection overlay node.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node, whose selection size, position and rotation shall be updated.
     */
    DrawingFrame.updateSelectionDrawingSize = function (drawingNode) {
        var drawing = $(drawingNode);
        if (drawing.data('selection')) {
            drawing.data('selection').css({ transform: drawing.css('transform'), left: drawing.css('left'), top: drawing.css('top'), width: drawing.width(), height: drawing.height() });
        }
    };

    /**
     * Helper function to remove the selection of a drawing node. This handles selection
     * objects that are located in the selection overlay node.
     *
     * @param {HTMLElement|jQuery} drawingNode
     *  The drawing node, whose selection shall be removed.
     */
    DrawingFrame.removeDrawingSelection = function (drawingNode) {
        var drawing = $(drawingNode);
        if (drawing.data('selection')) {
            drawing.data('selection').remove();
            drawing.removeData('selection');
        }
    };

    /**
     * Returns the tracker node that will be used to visualize the current
     * position and size of a drawing frame during move/resize tracking.
     *
     * @param {HTMLElement|jQuery} drawingFrame
     *  The root node of the drawing frame. If this object is a jQuery
     *  collection, uses the first DOM node it contains.
     *
     * @returns {jQuery}
     *  The tracker node if available (especially, teh drawing frame must be
     *  selected to contain a tracker node); otherwise an empty jQuery
     *  collection.
     */
    DrawingFrame.getTrackerNode = function (drawingFrame) {
        return getSelectionNode(drawingFrame).find('>.' + TRACKER_CLASS);
    };

    /**
     * Sets or resets the CSS class representing the active tracking mode for
     * the specified drawing frames.
     *
     * @param {HTMLElement|jQuery} drawingFrames
     *  The root node of a single drawing frame, or multiple drawing frames in
     *  a jQuery collection. This method changes the tracking mode of all
     *  drawing frames at once.
     *
     * @param {Boolean} active
     *  Whether tracking mode is active.
     */
    DrawingFrame.toggleTracking = function (drawingFrames, active) {
        $(drawingFrames).find('>.' + SELECTION_CLASS).toggleClass(TRACKING_CLASS, active === true);
    };

    /**
     * Returns the tracking direction stored in a resizer handle node of a
     * drawing frame selection box.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {String|Null}
     *  If the passed node is a resizer handle of a drawing frame selection
     *  box, returns its direction identifier (one of 'l', 'r', 't', 'b', 'tl',
     *  'tr', 'bl', or 'br'). Otherwise, returns null.
     */
    DrawingFrame.getResizerHandleType = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR + '>.' + SELECTION_CLASS + '>.resizers>[data-pos]') ? $(node).attr('data-pos') : null;
    };

    /**
     * Returns whether the passed DOM node is the handle for rotation.
     *
     * @param {Node|jQuery|Null} [node]
     *  The DOM node to be checked. If this object is a jQuery collection, uses
     *  the first DOM node it contains. If missing or null, returns false.
     *
     * @returns {Boolean}
     *  Whether the passed DOM node is the handle for rotation.
     */
    DrawingFrame.isRotateHandle = function (node) {
        return $(node).is(DrawingFrame.NODE_SELECTOR + '>.' + SELECTION_CLASS + '>.rotate-handle');
    };

    /**
     * Updates the size and position of all children of a drawing group,
     * but only temporary, during resize of the group.
     *
     * INFO: Use private method for final resize!
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     * @param {jQuery} drawing
     *  Shape node to be resized.
     * @param {Object} mergedAttributes
     *  Merged attributes of drawing node.
     */
    DrawingFrame.previewOnResizeDrawingsInGroup = function (app, drawing, mergedAttributes) {
        resizeDrawingsInGroup(app, drawing, mergedAttributes, { resizeActive: true });
    };

    /**
     * Public method for updating shape formatting.
     * Intended for rendering preview of shape during active resizing.
     * Therefore, explicit shape attributes are used.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {jQuery} drawingFrame
     *  The DOM drawing frame of a shape object that will be formatted.
     *
     * @param {Object} mergedAttributes
     *  The merged attribute set of the drawing object.
     *
     * @param {Object} [options]
     *  @param {Boolean} [options.isResizeActive=false]
     *      If this method is called during resizing of the shape.
     */
    DrawingFrame.updateShapeFormatting = function (app, drawingFrame, mergedAttributes, options) {
        updateShapeFormatting(app, drawingFrame, mergedAttributes, _.extend({}, options, { useExplicitAttrs: true }));
    };

    /**
     * Updates the position and size of the text frame located in the passed
     * DOM drawing frame, but does not update any other formatting.
     *
     * @param {EditApplication} app
     *  The application instance containing the drawing frame.
     *
     * @param {jQuery} drawingFrame
     *  The DOM drawing frame whose text frame will be updated.
     *
     * @param {Object} mergedAttributes
     *  The merged attribute set of the drawing object.
     */
    DrawingFrame.updateTextFramePosition = function (app, drawingFrame, mergedAttributes) {
        updateShapeFormatting(app, drawingFrame, mergedAttributes, { textFrameOnly: true });
    };

    /**
     * Adds hint bottom arrow class inside textframes with autoResizeHeight=false,
     * when text is hidden in overflow.
     *
     * @param {TextModel} docModel
     *  The text document model containing instance.
     *
     * @param {jQuery} drawingNode
     *  Text frame drawing node that has fixed height.
     */
    DrawingFrame.addHintArrowInTextframe = function (docModel, drawingNode) {
        var angle = DrawingFrame.getDrawingRotationAngle(docModel, drawingNode);
        var isFlipH = DrawingFrame.isFlippedHorz(drawingNode);
        var isFlipV = DrawingFrame.isFlippedVert(drawingNode);
        var boundRectDrawingNode;
        var contentNode = drawingNode.children('.textframecontent');
        var textFrameNode = contentNode.children('.textframe');
        var textFramePosTop, isOverflow;
        var firstChildNode, firstChildTop, lastChildNode, lastChildBottom;

        if (textFrameNode.length) {
            drawingNode.css('transform', ''); // reset to def to get values
            boundRectDrawingNode = drawingNode[0].getBoundingClientRect();
            textFramePosTop = textFrameNode.position().top;
            textFramePosTop = _.isFinite(textFramePosTop) ? textFramePosTop : 0;
            firstChildNode = textFrameNode[0].firstChild;
            lastChildNode = textFrameNode[0].lastChild;
            firstChildTop = firstChildNode ? firstChildNode.getBoundingClientRect().top : 0;
            lastChildBottom = lastChildNode ? lastChildNode.getBoundingClientRect().bottom : 0;

            isOverflow = boundRectDrawingNode.bottom < lastChildBottom + textFramePosTop || boundRectDrawingNode.top > firstChildTop + textFramePosTop;

            contentNode.toggleClass('b-arrow', isOverflow);
            DrawingFrame.setCssTransform(drawingNode, angle, isFlipH, isFlipV);
        }
    };

    /**
     * Used as fix for Bugs 45323, 48187
     * Special behavior of presentation files
     * Shapes do ignore their horizontal flip state,
     * if parent group has standard transform or completely inverted transform.
     *
     * @param {jQuery|Node} textFrameNode
     *  Text frame drawing node for which flip count is checked.
     * @param {String} [boundary]
     *  If specified, css selector as a string, which serves as end point until where the node parents are traversed.
     *
     * @returns {Boolean}
     *  If count of all flips is odd number.
     */
    DrawingFrame.isOddFlipHVCount = function (textFrameNode, boundary) {
        var queryEnd = boundary || '.app-content';
        var flipHCount = $(textFrameNode).parentsUntil(queryEnd, '.flipH').length; // must be separate queries, because if element has flipH and flipV,
        var flipVCount = $(textFrameNode).parentsUntil(queryEnd, '.flipV').length; // both need to be counted
        return (flipHCount + flipVCount) % 2 !== 0;
    };

    // exports ================================================================

    var promise = IO.loadJSON('io.ox/office/drawinglayer/view/presetGeometries');
    promise.done(function (preset) { PRESET_GEOMETRIES = preset; });

    return promise.then(_.constant(DrawingFrame));

});
