/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks of the OX Software GmbH group of companies.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016-2020 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.file.storage.dropbox.access;

import java.io.IOException;
import java.net.HttpURLConnection;
import com.dropbox.core.DbxException;
import com.dropbox.core.DbxRequestConfig;
import com.dropbox.core.http.HttpRequestor;
import com.dropbox.core.http.StandardHttpRequestor;
import com.dropbox.core.v2.DbxClientV2;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.FileStorageAccount;
import com.openexchange.file.storage.FileStorageExceptionCodes;
import com.openexchange.file.storage.dropbox.DropboxConfiguration;
import com.openexchange.file.storage.dropbox.DropboxConstants;
import com.openexchange.file.storage.dropbox.DropboxServices;
import com.openexchange.oauth.OAuthAccount;
import com.openexchange.oauth.OAuthService;
import com.openexchange.oauth.access.AbstractOAuthAccess;
import com.openexchange.oauth.access.OAuthAccess;
import com.openexchange.oauth.access.OAuthClient;
import com.openexchange.oauth.scope.OXScope;
import com.openexchange.session.Session;

/**
 * {@link DropboxOAuth2Access}
 *
 * @author <a href="mailto:ioannis.chouklis@open-xchange.com">Ioannis Chouklis</a>
 */
public class DropboxOAuth2Access extends AbstractOAuthAccess {

    private final FileStorageAccount fsAccount;

    /**
     * Initialises a new {@link DropboxOAuth2Access}.
     */
    public DropboxOAuth2Access(FileStorageAccount fsAccount, Session session) throws OXException {
        super(session);
        this.fsAccount = fsAccount;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.oauth.access.OAuthAccess#initialise()
     */
    @Override
    public void initialize() throws OXException {
        final OAuthService oAuthService = DropboxServices.getService(OAuthService.class);
        try {
            final OAuthAccount oauthAccount = oAuthService.getAccount(getAccountId(), getSession(), getSession().getUserId(), getSession().getContextId());
            verifyAccount(oauthAccount, OXScope.drive);
            HttpRequestor httpRequestor = new DropboxStandardHttpRequestor();
            DbxRequestConfig config = new DbxRequestConfig(DropboxConfiguration.getInstance().getProductName(), null, httpRequestor);
            String accessToken = oauthAccount.getToken();
            DbxClientV2 dbxClient = new DbxClientV2(config, accessToken);
            OAuthClient<DbxClientV2> oAuthClient = new OAuthClient<DbxClientV2>(dbxClient, accessToken);
            setOAuthClient(oAuthClient);
            setOAuthAccount(oauthAccount);
        } catch (RuntimeException e) {
            throw FileStorageExceptionCodes.UNEXPECTED_ERROR.create(e, e.getMessage());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.oauth.access.OAuthAccess#ensureNotExpired()
     */
    @Override
    public OAuthAccess ensureNotExpired() throws OXException {
        return this;
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.oauth.access.OAuthAccess#ping()
     */
    @Override
    public boolean ping() throws OXException {
        try {
            DbxClientV2 client = (DbxClientV2) getClient().client;
            client.users().getCurrentAccount();
            return true;
        } catch (DbxException e) {
            throw DropboxExceptionHandler.handle(e, getSession(), getOAuthAccount());
        }
    }

    /*
     * (non-Javadoc)
     *
     * @see com.openexchange.oauth.access.OAuthAccess#getAccountId()
     */
    @Override
    public int getAccountId() throws OXException {
        try {
            return getAccountId(fsAccount.getConfiguration());
        } catch (IllegalArgumentException e) {
            throw FileStorageExceptionCodes.MISSING_CONFIG.create(DropboxConstants.ID, fsAccount.getId());
        }
    }

    // ----------------------------------------------------------------------------------------------------------

    private static class DropboxStandardHttpRequestor extends StandardHttpRequestor {

        /**
         * Initializes a new {@link DropboxOAuth2Access.DropboxStandardHttpRequestor}.
         */
        DropboxStandardHttpRequestor() {
            super(Config.DEFAULT_INSTANCE);
        }

        @Override
        protected void configure(HttpURLConnection conn) throws IOException {
            super.configure(conn);
            conn.setChunkedStreamingMode(65536);
        }
    }

}
