/**
 * 
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 * 
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 * 
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com 
 * 
 * @author Matthias Biggeleben <matthias.biggeleben@open-xchange.com>
 * 
 */

/*jslint white: false, browser: true, devel: true, evil: true, forin: true, undef: true, eqeqeq: true, immed: true */

/*global ox, window, jQuery, _, register, triggerEvent, addTranslated
*/

// namespace
ox.upsell = {};

(function () {

    // shortcut to jQuery
    var $ = jQuery,
        // namespace
        up = ox.upsell,
        // shortcut to config
        config = ox.api.config,
        // shortcut
        isFunction = ox.util.isFunction;
    
    /**
     * Permission matrix
     */
    up.matrix = {};
    
    (function () {
        
        // shortcut to matrix namespace
        var matrix = up.matrix,
            // permission levels (set of features)
            levels = {},
            // features
            features = {};
        
        /**
         * Add level
         */
        matrix.addLevel = function (id, fn) {
            if (id !== undefined && isFunction(fn)) {
                levels[String(id).toLowerCase()] = { id: id, check: fn };
            }
            return matrix; // chaining
        };
        
        /**
         * Check if user has a permission level
         */
        var l = matrix.hasLevel = function (id) {
            id = String(id).toLowerCase();
            return levels[id] !== undefined ? levels[id].check() : false;
        };
        
        /**
         * List levels
         */
        matrix.listLevels = function () {
            return levels;
        };
        
        /**
         * Add feature
         */
        matrix.addFeature = function (id, fn) {
            if (id !== undefined) {
                fn = isFunction(fn) ? fn : function () { return true; };
                features[String(id).toLowerCase()] = { id: id, check: fn };
            }
            return matrix; // chaining
        };
        
        /**
         * Remove feature
         */
        matrix.removeFeature = function (id) {
            delete features[String(id).toLowerCase()];
            return matrix; // chaining
        };
        
        /**
         * Check for feature
         */
        var f = matrix.hasFeature = function (str) {
            str = jQuery.trim(String(str)).toLowerCase().split(" ");
            var has = true, i = 0, $i = str.length, id,
                disabled = ox.api.config.get(["ui", "global", "disableFeature"], {});
            for (; has && i < $i; i++) {
                if ((id = str[i]) !== "") {
                    has = has && !disabled[id] && (features[id] !== undefined ? features[id].check() : false);
                }
            }
            return has;
        };
        
        /**
         * Get features
         */
        matrix.listFeatures = function () {
            return features;
        };
        
        // add features
        matrix
            .addFeature("configuration", function () {
                return true; // always - but in case someone checks that...
            })
            .addFeature("portal", function () {
                return config.get("modules.portal.module") === true;
            })
            .addFeature("mail", function () {
                return config.get("modules.mail.module") === true;
            })
            .addFeature("contacts", function () {
                return config.get("modules.contacts.module") === true;
            })
            .addFeature("calendar", function () {
                return config.get("modules.calendar.module") === true;
            })
            .addFeature("tasks", function () {
                return config.get("modules.tasks.module") === true;
            })
            .addFeature("usm", function () {
                return config.get(["modules", "com.openexchange.usm", "active"]) === true;
            })
            .addFeature("eas", function () {
                return config.get(["modules", "com.openexchange.usm.eas", "active"]) === true;
            })
            .addFeature("syncML", function () {
                return config.get(["modules", "com.openexchange.usm.syncml", "active"]) === true;
            })
            .addFeature("mobileApp", function () {
                return false;
            })
            .addFeature("infostore", function () {
                return config.get("modules.infostore.module") === true;
            })
            .addFeature("shared", function () {
                return config.get("modules.folder.read_create_shared_folders") === true;
            })
            .addFeature("public", function () {
                return config.get("modules.folder.public_folders") === true;
            })
            .addFeature("delegate", function () {
                return config.get("modules.tasks.delegate_tasks") === true;
            })
            .addFeature("teamview", function () {
                return f("shared") && f("public") && f("delegate");
            })
            .addFeature("olox", function () {
                return config.get("modules.olox20.active") === true;
            })
            .addFeature("subscribe", function () {
                return config.get(["modules", "com.openexchange.subscribe"]) === true;
            })
            .addFeature("freebusy", function () {
                return config.get("modules.calendar.calendar_freebusy") === true;
            })
            .addFeature("participants", function () {
                return config.get("participants.showDialog") === true;
            })
            // new folder
            .addFeature("mail-new-folder")
            .addFeature("contacts-new-folder")
            .addFeature("calendar-new-folder")
            .addFeature("tasks-new-folder")
            .addFeature("infostore-new-folder")
            
            // atomic features
            .addFeature("calendar-attachments")
            .addFeature("calendar-flags")
            .addFeature("calendar-copy-move")
            
            .addFeature("contacts-attachments")
            .addFeature("contacts-flags")
            .addFeature("contacts-copy-move")
            
            .addFeature("tasks-attachments")
            .addFeature("tasks-flags")
            .addFeature("tasks-copy-move")
            
            .addFeature("infostore-flags")
            .addFeature("infostore-copy-move")
        
            .addFeature("mail-flags")
            .addFeature("mail-copy-move");
        
        // levels
        matrix
            .addLevel("Webmail", function () {
                return f("contacts");
            })
            .addLevel("PIM", function () {
                return f("contacts") && f("calendar") && f("tasks");
            })
            .addLevel("PIM+Mobility", function () {
                return l("PIM") && f("usm") && (f("eas") || f("syncML") || f("mobileApp"));
            })
            .addLevel("PIM+Infostore", function () {
                return l("PIM") && f("infostore");
            })
            .addLevel("Standard", function () {
                return l("PIM+Infostore") && f("shared") && f("public") && f("delegate");
            })
            .addLevel("Premium", function () {
                return l("Standard") && f("usm") && (f("eas") || f("syncML") || f("olox") || f("mobileApp"));
            });
        
    }());
    
    // helper for visible checks
    var disabled = function (feature) {
        return ox.api.config.get(["ui", "global", "disableUpsell", feature], false) || ox.api.config.get("ui.global.disableUpsell.all", false);
    };

    /**
     * Visible for upsell?
     */
    up.visible = function (feature) {
        var hide;
        return function () {
            hide = hide !== undefined ? hide : disabled(feature);
            return hide === false || (up.matrix.hasFeature(feature) && String(url.disable).indexOf(feature) === -1);
        };
    };
    
    up.isVisible = function (feature) {
        return up.visible(feature)();
    };
    
    up.hasTrigger = function (feature) {
        return (!up.matrix.hasFeature(feature) || String(url.disable).indexOf(feature) > -1) && !disabled(feature);
    };
    
    /**
     * Default upsell action
     */
    up.defaultUpsellAction = function (id, win, trigger, e) {
        triggerEvent("Feature_Not_Available", id, win, trigger, e);
    };
    
    // helper
    var eq = function (type, id, bool) {
        if (type === "level") {
            return function () {
                return up.matrix.hasLevel(id) === bool;
            };
        } else if (type === "feature") {
            return function () {
                return up.matrix.hasFeature(id) === bool;
            };
        }
    };
    
    // default features
    var features = {
        
        "premium": {
            node: $.button({
                //#. Upsell: Premium upgrade. Must contain the ellipsis!
                title: _("Free trial..."),
                click: function (e) {
                    // trigger
                    triggerEvent("Feature_Not_Available", "premium", window, "sidepanel", e);
//                    // direct upgrade, no prompt
//                    up.external({
//                        id: "premium",
//                        trigger: "sidepanel",
//                        //#. Upsell
//                        title: _("Upgrade to premium"),
//                        src: "http://en.wikipedia.org/wiki/Loyalty_marketing"
//                    });
                }
            }).css({ "float": "right "}),
            visible: eq("level", "premium", false),
            forsale: true,
            module: "*",
            prio: 10000,
            upgrade: "premium"
        },
        "calendar.share": {
            //#. Upsell
            title: _("Share calendar"),
            visible: eq("level", "premium", false),
            forsale: eq("level", "premium", false),
            module: "calendar",
            tour: "Tour1",
            prio: 1000
        },
        "calendar.sync.mobile": {
            //#. Upsell
            title: _("Sync with mobile phone"),
            visible: true,
            forsale: eq("feature", "eas", false),
            module: "calendar",
            tour: "Tour1",
            prio: 100,
            action: function () {
                triggerEvent("Feature_SyncWithMobile");
            }
        },
        "sync.outlook": {
            //#. Upsell
            title: _("Synchronize with Outlook"),
            visible: true,
            forsale: eq("level", "premium", false),
            module: "mail calendar contacts",
            tour: "Tour1",
            prio: 50,
            action: function () {
                triggerEvent("Feature_MicrosoftOutlook");
            }
        },
        "contacts.share": {
            //#. Upsell
            title: _("Share your contacts"),
            visible: eq("level", "premium", false),
            forsale: eq("level", "premium", false),
            module: "contacts",
            tour: "Tour1",
            prio: 1000
        },
        "contacts.sync.mobile": {
            //#. Upsell
            title: _("Sync with mobile phone"),
            visible: true,
            forsale: eq("feature", "eas", false),
            module: "contacts",
            tour: "Tour1",
            prio: 100,
            action: function () {
                triggerEvent("Feature_SyncWithMobile");
            }
        },
        "contacts.subscribe": {
            //#. Upsell
            title: _("Subscribe contacts from LinkedIn, Facebook, Google..."),
            visible: true,
            forsale: eq("feature", "subscribe", false),
            module: "contacts",
            tour: "Tour1",
            prio: 101,
            action: function () {
                triggerEvent("Feature_Subscriptions");
            }
        },
        "tasks.share": {
            //#. Upsell
            title: _("Share tasks"),
            visible: eq("feature", "eas", false),
            forsale: eq("feature", "eas", false),
            module: "tasks",
            tour: "Tour1",
            prio: 1000
        }
    };
    
    // active features
    up.features = {};
    
    /**
     * Get feature
     */
    up.getFeature = function (id) {
        return up.features[id] || features[id];
    };
    
    /**
     * Enable default features
     */
    up.enableDefaultFeatures = function () {
        var id;
        for (id in features) {
            // copy
            features[id].id = id;
            up.features[id] = features[id];
        }
    };
    
    /**
     * Disable default features
     */
    up.disableDefaultFeatures = function () {
        var id;
        for (id in features) {
            // delete
            delete up.features[id];
        }
    };
    
    /**
     * Add custom feature
     */
    up.addFeature = function (id, options) {
        if (id !== undefined && options !== undefined) {
            options.id = id;
            up.features[id] = options;
        }
    };
    
    // internal helpers
    var sorterPrio = function (a, b) {
        return (a.prio || 100) - (b.prio || 100);
    };
    
    var evalState = function (flag) {
        // var
        var retVal;
        // is function?
        if (isFunction(flag)) {
            // get promise or boolean
            retVal = flag();
        } else {
            // get boolean
            retVal = !!flag;
        }
        // boolean?
        if (typeof retVal === "boolean") {
            // wrap boolean by deferred
            return $.Deferred().resolve(retVal).promise();
        } else {
            // return deferred
            return retVal;
        }
    };
    
    /**
     * Get features for module
     */
    up.getFeaturesByModule = function (module) {
        // loop over upgrades to get array
        var tmp = [], when = [];
        $.each(up.features, function (id) {
            // get
            var feature = up.features[id],
                m = String(feature.module);
            // proper module and for sale?
            if ((module === undefined || m === "*" || m.indexOf(module) > -1)) {
                // eval visible and wait for it
                when.push(
                    evalState(feature.visible)
                    .done(function (visible) {
                        if (visible) {
                            tmp.push(feature);
                        }
                    })
                );
            }
        });
        // new deferred
        var df = $.Deferred();
        $.when.apply(window, when)
        .then(function () {
            // sort by priority (asc)
            tmp.sort(sorterPrio);
            df.resolve(tmp);
        });
        return df;
    };
    
    /**
     * Call feature
     */
    up.callFeature = function (id, win, e) {
        // get feature
        var f = up.getFeature(id) || {};
        // is visible?
        evalState(f.visible).done(function (visible) {
            if (visible) {
                // is for sale?
                evalState(f.forsale).done(function (forsale) {
                    if (forsale) {
                        // custom/default upsell action?
                        var fn = isFunction(f.upsellAction) ? f.upsellAction : up.defaultUpsellAction;
                        // trigger upsell
                        fn(f.id, win, "sidepanel", e);
                    } else {
                        // call feature
                        if (isFunction(f.action)) {
                            f.action();
                        }
                    }
                });
            }
        });
    };
    
    // default paint methods
    var p = up.paint = {};
    
    /**
     * Common paint method
     */
    p.smartFeatures = function (node, opt, cont) {
        // vars
        var self = this,
            container,
            module = ox.UIController.getModule(),
            fnClick = function (e) {
                up.callFeature(e.data.id, window, e);
                e.preventDefault();
            };
        // get features
        up.getFeaturesByModule(module)
        .done(function (features) {
            // remember number of features
            self.numFeatures = features.length;
            // draw title?
            if (opt.title !== undefined) {
                node.append(
                    $("<div/>")
                        .addClass("sidepanelHeader upsell")
                        .append(addTranslated(opt.title))
                );
            }
            // add inner node
            container = $("<div/>").css("padding", "10px");
            // loop over features
            $.each(features, function (i, feature) {
                // create new node
                var div = $("<div/>").css({
                    lineHeight: "1em",
                    margin: "0 0 0.5em 0",
                    clear: "both"
                });
                // add custom node?
                if (feature.node !== undefined) {
                    div.append(feature.node.clone(true, true));
                } else {
                    div.append(
                        $("<a/>", { href: "#upgrade" })
                            .bind("click", feature, fnClick)
                            .append(addTranslated(feature.title))
                    );
                }
                // support customization
                p.smartFeaturesCustomize(div, feature);
                // add
                container.append(div);
            });
            // add tailing element
            container.append($("<div/>").css("clear", "both"));
            node.append(container);
            // continuation
            ox.util.call(cont);
        });
    };
    
    /**
     * Support for customized features
     */
    p.smartFeaturesCustomize = $.noop;
    
    /**
     * Extend sidepanel with custom widget
     */
    up.extendSidepanel = (function () {
    
        var defaultHeight = 50,
            guid = 0;
        
        var getHeight = function (opt, node) {
            return opt.height === "auto" ?
                $(node).children().first().outerHeight(true) :
                parseInt(opt.height, 10) || defaultHeight;
        };
        
        var applyModule = function (opt) {
            
            var module = ox.UIController.getModule(),
                self = this,
                validate = function () {
                    self.setLayoutParam("height", self.numFeatures > 0 ? getHeight(opt, self.dom.node) : 0);
                    ox.widgets.sidepanel.validate();
                };
            
            if (module !== "configuration" && (opt.module.indexOf(module) > -1 || opt.module === "*")) {
                // show
                // refresh first? (not for iframes)
                if (opt.refresh === true && opt.src === undefined) {
                    self.paint(function () {
                        validate();
                    });
                } else {
                    validate();
                }
            } else {
                // hide
                this.setLayoutParam("height", 0);
                ox.widgets.sidepanel.validate();
            }
        };
        
        return function (options) {
            
            // options
            var opt = $.extend({
                module: "*", // show for these modules (* = always)
                paint: $.noop, // paint method
                height: 50, // height in pixel
                css: {}, // custom css
                position: "below", // below/above mini calendar
                src: undefined, // external iframe
                refresh: false // refresh on module change
            }, options || {});
            
            // create widget
            var widget = new ox.gui.Custom(
                function (cont) {
                    // clear
                    var node = $(this.dom.node).empty();
                    // external page?
                    if (opt.src !== undefined) {
                        // add iframe
                        node.css({ padding: "0px" }).append(
                            $("<iframe/>", {
                                src: opt.src,
                                frameborder: "no",
                                scrolling: "no",
                                id: this.id + "-iframe"
                            }).css({
                                width: "100%",
                                height: opt.height
                            })
                        );
                    } else {
                        var div = $("<div/>")
                            .addClass("sidepanelContent upsell")
                            .css($.extend({}, opt.css))
                            .appendTo(node);
                        // call custom paint handler
                        $.proxy(opt.paint, this)(div, opt, cont);
                    }
                }, 
                "upsell-sidepanel-" + (guid++)
            );
            widget.addCSSClass("oxStretch")
                .css({
                    backgroundColor: "white"
                })
                .setLayoutParam({
                    position: "bottom",
                    height: 0
                });
                
            if (opt.position !== "below") {
                widget.css({ borderBottom: "1px solid #ccc" });
            }
            
            // add to sidepanel
            ox.widgets.sidepanel.insert(widget, opt.position === "below" ? 2 : 3);
            
            // extend widget
            widget.applyModule = applyModule;
            
            // pre-validate for auto height?
            if (opt.height === "auto") {
                ox.widgets.sidepanel.validate();
            }
            
            // apply current module
            widget.applyModule(opt);
            
            // bind to UIController
            ox.UIController.dispatcher.bind("modulechange", function (e) {
                // re-apply on module change
                widget.applyModule(opt);
            });
            
            return widget;
        };
    }());
    
    /**
     * Default source for external upsell processes
     */
    up.defaultSrc = "http://en.wikipedia.org/wiki/Upselling#feature=[id]&trigger=[trigger]&upgrade=[upgrade]";
    
    /**
     * Use default behavior
     */
    up.useDefaults = function (registerEvent) {
        // use default features
        up.enableDefaultFeatures();
        // extend side panel
        up.extendSidepanel({
            module: "*", // <-- all modules,
            paint: up.paint.smartFeatures,
            refresh: true, // refresh on module change
            height: "auto",
            css: {},
            //#. Upsell
            title: _("Premium features")
        });
        // register event?
        if (registerEvent !== false) {
            register("Feature_Not_Available", function (id, win, trigger, e) {
                up.prompt(id, trigger, function () {
                    // get feature
                    var f = up.getFeature(id) || {};
                    up.external({
                        id: id,
                        trigger: "sidepanel/" + trigger,
                        upgrade: f.upgrade || ""
                    });
                });
            });
        }
    };
    
    /**
     * Open external upsell process
     */
    up.external = function (options) {
        
        var opt, params, src, node;
        
        opt = $.extend({
            id: "",
            trigger: "",
            upgrade: "",
            src: up.defaultSrc,
            //#. Upsell
            title: _("Account upgrade")
        }, options);
        
        // format src
        params = {
            id: encodeURIComponent(opt.id),
            trigger: encodeURIComponent(opt.trigger || ""),
            upgrade: encodeURIComponent(opt.upgrade || "")
        };
        src = String(opt.src).format(params);
        
        node = $("<div/>")
            .addClass("busy-spinner-white")
            .append(
                $("<div/>")
                .css({ fontSize: "16pt", color: "#aaa", margin: "0 0 1em 0" })
                .text(String(opt.title))
            )
            .append(
                $("<iframe/>", {
                    src: src,
                    scrolling: "auto",
                    frameborder: "no"
                })
                .css({
                    height: "350px",
                    width: "100%"
                })
                .one("load", function () {
                    $(this).parent().removeClass("busy-spinner-white");
                })
            ).append(
                $("<div/>").css({
                    height: "30px",
                    margin: "10px 0 0 0"
                }).append(
                    $.button({
                        title: _("Close"),
                        click: function () {
                            node.popup("disable");
                            node = null;
                        }
                    }).css("float", "right")
                )
            )
            .popup($.extend({
                width: "850px",
                modal: true,
                left: Math.max(10, ($("body").width() - 850) / 2 >> 0),
                top: Math.max(10, ($("body").height() - 450) / 2 >> 0)
            }, options));
    };
    
    /**
     * Prompt for upsell
     */
    up.prompt = function (id, trigger, cont) {
        
        var controls, opt = up.getFeature(id) || {};
        
        var node = $("<div/>")
            .addClass("upsell-prompt")
            .append(
                $("<h2/>")
                    .css({ margin: "1em 0 2em 0"})
                    .text(
                        //#. Full sentence:
                        //#. This feature is not available but you can upgrade your account right now
                        _("This feature is not available") + " ..."
                    )
             )
            .append(
                $("<h1/>")
                    .css({
                        color: "#8BAD2D", width: "15em",
                        margin: "0 auto 2em auto", lineHeight: "1.2em",
                        textShadow: "5px 5px 10px #ddd", fontWeight: "normal"
                    })
                    .text(
                        //#. Full sentence:
                        //#. This feature is not available but you can upgrade your account right now.
                        "... " + _("but you can upgrade your account right now" + "!")
                    )
            )
            .append(
                controls = $("<div/>").css("textAlign", "right")
            );
            
            controls.append(
                $.button({
                    //#. Upsell
                    title: _("Yes, upgrade my account") + "...",
                    click: function (e) {
                        node.popup("disable");
                        node = null;
                        if (cont) {
                            cont(id, trigger);
                        }
                    }
                })
                .css({ fontWeight: "bold", margin: "0 1em 0 0" })
            );
            
            // add tour?
            if (opt.tour !== undefined && ox.api.help.Tour.exists(opt.tour)) {
                // add tour button
                controls.append(
                    $.button({
                        //#. Upsell
                        title: _("Tell me more..."),
                        click: function (e) {
                            node = null;
                            ox.api.help.Tour.load(opt.tour, function () {
                                up.prompt(id, trigger, cont);
                            });
                        }
                    })
                    .css({ margin: "0 1em 0 0" })
                );
            }
            
            controls.append(
                $.button({
                    //#. Upsell
                    title: _("No, not now"),
                    click: function (e) {
                        node.popup("disable");
                        node = null;
                    }
                })
            );
            
            node.popup({
                width: "500px",
                height: "200px",
                modal: true,
                left: Math.max(10, ($("body").width() - 500) / 2 >> 0),
                top: Math.max(10, ($("body").height() - 300) / 2 >> 0)
            });
    };
    
}());

/* ----------- Examples ------------ 

// "Hello World" upsell
ox.upsell.extendSidepanel({
    module: "mail", // <-- mail only
    paint: function (node, options) {
        node.html("Hello World!");
        // do whatever you want here (see http://api.jquery.com/)
        // "node" is a jQuery object referring to the widget's DOM node
    }
});

// "Hello World" upsell with different height & CSS
ox.upsell.extendSidepanel({
    module: "mail tasks", // <-- mail and tasks
    paint: function (node, options) {
        node.html("Hello World!");
    },
    height: 30,
    css: {
        backgroundColor: "lightyellow",
        border: "3px solid #fc0"
    }
});

// Upsell via external page -- show on ALL modules
ox.upsell.extendSidepanel({
    module: "*", // <-- all modules
    src: "http://www.open-xchange.com/en/node/1262/",
    height: 100
});

// Use common paint method
ox.upsell.enableDefaultFeatures();
ox.upsell.extendSidepanel({
    module: "*", // <-- all modules,
    paint: ox.upsell.paint.smartFeatures,
    refresh: true, // refresh on module change
    height: "auto",
    title: "Upgrade your account"
});

// Fast and easy upsell:
ox.upsell.useDefaults();

*/

