/**
 *
 * All content on this website (including text, images, source
 * code and any other original works), unless otherwise noted,
 * is licensed under a Creative Commons License.
 *
 * http://creativecommons.org/licenses/by-nc-sa/2.5/
 *
 * Copyright (C) 2016 OX Software GmbH
 * Mail: info@open-xchange.com
 *
 * @author Matthias Biggeleben <matthias.biggeleben@open-xchange.com>
 *
 */

/*jslint bitwise: false, nomen: false, onevar: false, plusplus: false, regexp: false, white: true, browser: true, devel: true, evil: true, forin: true, undef: true, eqeqeq: true, immed: true, continue: true */

/*global ox, jQuery, registerModule, _, PortalItem, registerView, showNode, hideNode,
cPortal, OXCache, register, triggerEvent, configuration_changed_fields,
getFullImgSrc, PortalExternalItem, url, UWA, internalCache, Join, clone,
addTranslated, getGreetingPhrase, newtext, IE, isEmpty, trimStr */

registerModule("portal", _("Start Page"), 1);

var bPortalNeedsRepaint = false;
var oPortalObject = null;

// -------

var oPortal = (function ($) {
    
    // dom nodes
    var dom = {};

    // widgets
    var widgets = [];
    var grid = [];

    // layout nodes
    var layoutNodes = $();

    // placeholder for DnD
    var placeholder = $("<div/>").addClass("ox-portal-placeholder");

    // config shortcut/keys
    var config = $.noop;
    var keyInternal = "gui.portal.internalcontents";

    // current layout
    var currentLayout = 2;

    // DnD
    var dnd = {};
    (function () {

        var gridX = [], gridY = {};

        var px, py;
        var ppx, ppy;
        var outer, width, height;
        var iframes;

        var numSort = function (a, b) {
            return a - b;
        };

        var getGridPosition = function (x, y) {
            // consider dimensions
            x = x + (width >> 1);
            // transform mouse position to grid position
            px = 0;
            py = -1;
            var i, $i;
            for (i = 0, $i = gridX.length; i < $i; i++) {
                if (x > gridX[i]) {
                    px = i;
                }
            }
            if (gridY[px] !== undefined) {
                for (i = 0, $i = gridY[px].length; i < $i; i++) {
                    if (y > gridY[px][i]) {
                        py = i;
                    }
                }
            }
        };

        dnd.getWidget = function (x, y) {
            // loop widgets
            var i = 0, $i = grid.length, obj;
            for (; i < $i; i++) {
                obj = grid[i];
                if (obj.x === x && obj.y === y) {
                    return obj;
                }
            }
            return null;
        };

        dnd.start = function (n) {

            // remember
            outer = n;
            width = n.outerWidth();
            height = n.outerHeight();

            // resize placeholder
            placeholder.height((height - 4) + "px");

            // vars
            var i, $i, node, offset, x, y;

            // get vertical borders
            gridX = [];
            for (i = 0, $i = layoutNodes.length; i < $i; i++) {
                // get node
                node = layoutNodes.eq(i);
                // get offset
                offset = node.offset();
                // floor
                x = offset.left >> 0;
                // add
                gridX.push(x);
            }

            // get horizontal borders
            gridY = {};
            // loop widgets
            for (i = 0, $i = grid.length; i < $i; i++) {
                // get node
                node = grid[i].node;
                x = grid[i].x;
                // get offset
                offset = node.offset();
                // floor
                y = (offset.top >> 0);
                if (gridY[x] === undefined) {
                    gridY[x] = [];
                }
                gridY[x].push(y);
            }
            // sort x
            gridX.sort(numSort);
            // sort y
            for (x in gridY) {
                gridY[x].sort(numSort);
            }

            // apply iframe fix
            iframes = dom.container.find("iframe").overlay();
        };

        dnd.update = function (x, y) {
            // add scroll position
            y += dom.container.scrollTop();
            // get grid position (set globalish px/py)
            getGridPosition(x, y);
            // change position?
            if (px !== ppx || py !== ppy) {
                if (py === -1) {
                    // add as first
                    placeholder.prependTo(layoutNodes.eq(px).children());
                } else {
                    var widget = dnd.getWidget(px, py);
                    if (widget) {
                        // insert after
                        placeholder.insertAfter(widget.node);
                    } else {
                        // append
                        placeholder.appendTo(layoutNodes.eq(px).children());
                    }
                }
                ppx = px;
                ppy = py;
            }
        };

        dnd.stop = function () {

            // remove iframe fix
            iframes.overlay(false);

            // insert after
            outer.insertAfter(placeholder);
            // remove placeholder
            placeholder.remove();
            // show
            outer.show();
            // clear grid
            grid = [];
            // reset dnd
            ppx = ppy = undefined;

            // loop over layoutNodes
            layoutNodes.each(function (x, node) {
                $(node).children().first().children().each(function (y, node) {
                    // get widget
                    var widget = $(node).data("widget");
                    // update grid
                    grid.push({
                        widget: widget,
                        node: $(node),
                        x: x,
                        y: y
                    });
                });
            });

            // update configuration
            dnd.store();
        };

        dnd.store = function () {

            // clone grid
            var tmp = grid.slice();

            var getPosition = function (key, value) {
                var i = 0, $i = tmp.length, obj, pos = null;
                for (; i < $i; i++) {
                    obj = tmp[i];
                    if (obj.widget[key] === value && value !== undefined) {
                        pos = { x: obj.x, y: obj.y };
                        break;
                    }
                }
                return pos;
            };

            var updatePositions = function (internal, list) {
                var i = 0, $i = list.length, obj, pos;
                for (; i < $i; i++) {
                    obj = list[i];
                    if (internal) {
                        pos = getPosition("module", obj.module);
                    } else {
                        pos = getPosition("id", obj.id || obj.title);
                    }
                    if (pos) {
                        obj.adj = pos;
                        // updating external
                        if (!internal && obj.visible === true) {
                            ox.uwa.update(obj);
                        }
                    }
                }
                return list;
            };

            // update internal
            config.set(keyInternal, updatePositions(true, config.get(keyInternal)));

            // updating external positions
            if (ox.api.config.get("modules.uwaWidgets.enabled") === true) {
                ox.uwa.list(function (data) {
                    updatePositions(false, data);
                });
            };

            configuration_changed_fields.gui = true;
        };

        dnd.close = function (data) {

            var key, value, list, obj;
            var isInternal = data.module !== undefined;
            var i = 0, $i;

            // internal?
            if (isInternal) {
                key = "module";
                value = data.module;
                list = config.get(keyInternal);

                // loop
                for ($i = list.length; i < $i; i++) {
                    obj = list[i];
                    if (obj[key] === value) {
                        // set invisible
                        obj.visible = false;
                        if (isInternal) {
                            // update limit
                            obj.params.limit = 0;
                        }
                        break;
                    }
                }

                // update config
                config.set(keyInternal, list);
            } else {
                obj = ox.uwa.get(data.id || data.title);
                if (obj) {
                    obj.visible = false;
                    ox.uwa.update(obj);
                }
            }
            configuration_changed_fields.gui = true;
        };

    }());

    /*
     * Widget registry. Can be used by plugins to add pseudo-internal widgets
     */
    var registry = (function () {

        var getHash = function (list) {
            var i = 0, $i = list.length, obj, tmp = {};
            for (; i < $i; i++) {
                obj = list[i];
                if (obj.id !== undefined) {
                    tmp[obj.id] = i;
                }
            }
            return tmp;
        };

        var widgets = {};

        return {

            merge: function (list) {
                try {
                    // get hash
                    var tmp = getHash(list);
                    // loop over widgets
                    var id, widget;
                    for (id in widgets) {
                        // get widget
                        widget = widgets[id];
                        // not in list?
                        if (tmp[id] === undefined) {
                            list.push(widget.data);
                        }
                    }
                    // remove undefined custom widgets
                    var i = 0, obj;
                    for (; i < list.length;) {
                        obj = list[i];
                        if (obj.module === "custom" && widgets[obj.id] === undefined) {
                            list.splice(i, 1);
                        } else {
                            i++;
                        }
                    }
                } catch (e) {
                    if (url.dev) {
                        console.error(e);
                    }
                }
                return list;
            },

            add: function (options) {

                var opt = $.extend({
                    x: 0,
                    y: 0,
                    paint: $.noop,
                    title: _("Custom Widget")
                }, options);

                if (opt.id !== undefined) {
                    // add widget
                    widgets[opt.id] = {
                        data: {
                            id: opt.id,
                            adj: {
                                x: opt.x,
                                y: opt.y
                            },
                            module: "custom",
                            external: false,
                            visible: true,
                            params: { limit: 0 },
                            "protected": true // no close allowed
                        },
                        painter: opt.paint,
                        title: opt.title
                    };
                }
            },

            getPainter: function (id) {
                return widgets[id] !== undefined ? widgets[id].painter : $.noop;
            },

            getTitle: function (id) {
                return widgets[id] !== undefined ? widgets[id].title : _("Custom Widget");
            }
        };

    }());

    // local functions
    var paint, drawCheckboxes, drawWidget, enable;

    drawWidget = function (node, data, isInternal) {

        var outer, header, title, inner;
        var close = $();

        // protected - no close allowed
        if (data["protected"] !== true) {
            close = $("<img/>")
                .attr({
                    src: getFullImgSrc("img/dummy.gif"),
                    alt: ""
                })
                .addClass("ox-portal-close")
                .bind("click", function () {
                    dnd.close(data);
                    drawCheckboxes();
                    oPortal.repaint();
                });
        }

        outer = $("<div/>")
            .addClass("ox-portal-widget").css({
                position: "relative"
            })
            .append(
                header = $("<div/>")
                    .addClass("ox-portal-header")
                    .attr('data-ox-module', data.module || '')
                    .append(
                        close
                    )
                    .append(
                        title = $("<div/>").text("\u00a0").addClass("ox-portal-header-handle")
                    )
            )
            .append(
                inner = $("<div/>")
                    .css({
                        position: "relative",
                        minHeight: "50px",
                        padding: "8px"
                    })
            )
            .appendTo(node);

        // set title
        switch (data.module) {
        case "calendar":
            title.text(String(_("Calendar")));
            break;
        case "mail":
            title.text(String(_("E-Mail")));
            break;
        case "contacts":
            title.text(String(_("Contacts")));
            break;
        case "tasks":
            title.text(String(_("Tasks")));
            break;
        case "infostore":
            title.text(String(_("InfoStore")));
            break;
        case "custom":
            title.text(String(registry.getTitle(data.id)));
            break;
        default: // external
            title.text(String(data.title));
            break;
        }

        // prevent selection
        if (ox.browser.Gecko) {
            // Firefox
            outer.css("MozUserSelect", "-moz-none");
        } else {
            // IE, WebKit
            outer.bind("selectstart", function () {
                return false;
            });
        }

        // create new portal object
        var widget;
        if (isInternal) {
            widget = new PortalItem(
                data.module,
                inner.get(0),
                header.get(0),
                data.params.limit,
                data.adj,
                $("<div/>").get(0),
                dom.container.get(0)
            );
        } else {
            widget = new PortalExternalItem(
                data.id || data.title, // id
                data.module,
                inner.get(0),
                header.get(0),
                (data.parameter || data.parameters),
                data.adj,
                $("<div/>").get(0),
                dom.container.get(0),
                data.url
            );
            widget.autorefresh = data.autorefresh;
            widget.standalone = data.standalone;
        }

        outer.data("widget", widget);

        // make draggable
        outer.draggable({
            handle: ".ox-portal-header, .ox-portal-header-handle",
            helper: function () {
                // "clone" fix for jQuery (>1.5), since clone's default is now "true"
                return $(this).clone(false);
            },
            opacity: 0.75,
            revert: false,
            start: function (e, ui) {
                ui.helper.css({
                    width: outer.width() + "px",
                    height: outer.height() + "px",
                    zIndex: 65000
                }).find("iframe").remove();
                // get grid
                dnd.start(outer);
                // hide
                outer.hide();
            },
            drag: function (e, ui) {
                dnd.update(ui.position.left, ui.position.top);
            },
            stop: function (e, ui) {
                ui.helper.remove();
                dnd.stop();
                outer.show();
                placeholder.remove();
            }
        });

        return {
            x: data.adj.x,
            y: data.adj.y,
            node: outer,
            widget: widget
        };
    };

    var getAllData = function () {

        var meta = [];

        var success = function (response) {
            // loop over response
            var i = 0, $i = response.length, data, widget;
            for (; i < $i; i++) {
                // get widget
                widget = meta[i].widget;
                // error?
                if (response[i].error) {
                    widget.setError(response[i]);
                } else {
                    // set data
                    widget.setData(
                        response[i].data,
                        meta[i].index,
                        response[i].timestamp,
                        meta[i].limit
                    );
                }
            }

            enable();

            meta = response = data = widget = null;
        };

        // loop over internal widget instances
        var i = 0, $i = widgets.length, widget, req, all = [], j, $j;
        for (; i < $i; i++) {
            // get
            widget = widgets[i];
            // internal?
            if (!widget.extern) {
                // get requests
                req = widget.getRequests();
                // add
                for (j = 0, $j = req.length; j < $j; j++) {
                    all.push(req[j]);
                    meta.push({ widget: widget, index: j, limit: req[j].limit });
                }
            }
        }

        if (all.length !== 0) {
            // send "multiple" server request
            ox.api.http.PUT({
                module: "multiple",
                data: all,
                success: success
            });
        }
    };

    /*
     * Layout
     */

    var applyLayout = function () {

        // event
        triggerEvent('OX_Portal_BeforeLayoutChanged');

        var tmpl = $("<div/>").css({
            "float": "left",
            width: "100%"
        }).append(
            $("<div/>").css({
                marginRight: "15px",
                minHeight: "1px"
            })
        );

        // park widgets
        dom.container.find(".ox-portal-widget").detach();

        // clear
        layoutNodes.remove();
        layoutNodes = $();

        var add = function (node, last) {
            if (last) {
                node.children().first().css("marginRight", "0px");
            }
            layoutNodes = layoutNodes.add(
                node.appendTo(dom.columnContainer)
            );
        };

        // layout #1? (for IE7 one percent less than 100!)
        switch (currentLayout) {
        case 1:
            add(tmpl.clone().css({ width: "100%" }), true);
            break;
        case 2:
            add(tmpl.clone().css({ width: "50%" }));
            add(tmpl.clone().css({ width: "50%" }), true);
            break;
        case 3:
            add(tmpl.clone().css({ width: "33%" }));
            add(tmpl.clone().css({ width: "33%" }));
            add(tmpl.clone().css({ width: "34%" }), true);
            break;
        case 4:
            add(tmpl.clone().css({ width: "40%" }));
            add(tmpl.clone().css({ width: "60%" }), true);
            break;
        case 5:
            add(tmpl.clone().css({ width: "60%" }));
            add(tmpl.clone().css({ width: "40%" }), true);
            break;
        }

        // event
        triggerEvent('OX_Portal_LayoutChanged');
    };

    var getLayoutNode = function (adj) {
        var $l = layoutNodes.length, index = Math.max(0, Math.min(adj.x || 0, $l - 1));
        return layoutNodes.eq(index).children().first();
    };

    var refreshLayout = function () {
        // apply layout
        applyLayout();
        // loop over grid
        var i = 0, $i = grid.length, obj, node;
        for (; i < $i; i++) {
            obj = grid[i];
            // get layout node
            node = getLayoutNode({ x: obj.x, y: obj.y });
            // append
            node.append(obj.node);
        }
    };

    var drawPortal = function () {

        // apply layout
        applyLayout();

        // get internal objects
        var internal = config.get(keyInternal);
        var inbox = config.get("mail.folder.inbox");

        // concat array holding all widgets
        var concat = [];

        var sorter = function (a, b) {
            if (a.adj !== undefined && b.adj !== undefined) {
                return a.adj.y - b.adj.y;
            } else {
                return -1;
            }
        };

        // clear list
        widgets = [];
        grid = [];

        // add custom widgets?
        internal = registry.merge(internal);
        config.set(keyInternal, internal);

        // loop over internal widgets
        var i = 0;
        for (; i < internal.length; i++) {
            // get
            var item = internal[i];
            // defined?
            if (item.adj !== undefined) {
                // inbox missing?
                if (item.module === "mail" && inbox === null) {
                    item.visible = false;
                }
                // visible?
                if (item.visible) {
                    item.internal = true;
                    concat.push(item);
                }
            }
        }

        // finally build the grid
        var join = new Join(function () {

            // sort positions
            concat.sort(sorter);

            // building final grid
            var i = 0, $i = concat.length, item, node, obj;
            for (; i < $i; i++) {

                item = concat[i];

                // just in case no positions given!
                if (item.adj === undefined) {
                    item.adj = { x: 1, y: 1 };
                }

                // get layout node
                node = getLayoutNode(item.adj);
                // draw widget (returns instance of PortalItem)
                obj = drawWidget(node, item, item.internal);
                widgets.push(obj.widget);
                grid.push(obj);

                // paint frame for external widgets
                if (item.internal === false) {
                    obj.widget.appendFrame();
                } else if (item.module === "custom") {
                    // paint custom widget
                    $.proxy(registry.getPainter(item.id), $(obj.widget.container))();
                }
            }
            // get all data
            getAllData();
        });

        var last = join.add();

        // get external uwas
        ox.uwa.list(join.add(function (external) {
            external = clone(external);
            if (url.external !== 0) {
                // loop over external widgets
                var i = 0, $i = external.length, item;
                for (; i < $i; i++) {
                    // get object
                    item = external[i];
                    // visible?
                    if (item.visible) {
                        item.internal = false;
                        item["protected"] = ! ox.api.config.get("modules.uwaWidgets.enabled", true);
                        concat.push(item);
                    };
                };
            };
        }));

        last();
    };

    drawCheckboxes = function () {

        var limits = { mail: 10, calendar: 10, contacts: 0, tasks: 10, infostore: 5 };

        var fnChange = function (e) {
            // get module
            var module = e.data.module;
            var checked = $(this).is(":checked");
            // get internal widgets
            var i = 0, list = config.get(keyInternal), $i = list.length, obj;
            for (; i < $i; i++) {
                obj = list[i];
                if (obj.module === module) {
                    obj.visible = checked;
                    obj.params.limit = checked ? limits[module] : 0;
                    break;
                }
            }
            // update config
            config.set(keyInternal, list);
            // repaint
            paint(true);
        };

        var isChecked = function (module) {
            // get internal widgets
            var i = 0, list = config.get(keyInternal), $i = list.length, obj, checked = false;
            for (; i < $i; i++) {
                obj = list[i];
                if (obj.module === module) {
                    checked = obj.visible;
                    break;
                }
            }
            return checked;
        };

        var checkbox = function (module, title) {

            // consider features
            var hasFeature = ox.upsell.matrix.hasFeature(module),
                // get checkbox
                box = $.checkbox(
                    "ox-portal-show-" + module, // id
                    "1", // value
                    hasFeature && isChecked(module), // checked
                    title, // label
                    fnChange, // callback
                    { module: module }, // data
                    hasFeature // enabled
                );
            // upsell?
            if (!hasFeature) {
                box
                .css("opacity", "0.60")
                .bind("click", module, function (e) {
                    // trigger
                    triggerEvent("Feature_Not_Available", "modules/" + module, window, "portal", e);
                    return false;
                });
            }
            // return
            return $().add(box).add($("<br/>"));
        };

        // clear
        dom.checkboxes1.empty();
        dom.checkboxes2.empty();

        // add
        if (ox.upsell.isVisible("mail")) {
            dom.checkboxes1.append(checkbox("mail", addTranslated("Show mails"))); /*i18n*/
        }
        if (ox.upsell.isVisible("calendar")) {
            dom.checkboxes1.append(checkbox("calendar", addTranslated("Show appointments"))); /*i18n*/
        }
        if (ox.upsell.isVisible("contacts")) {
            dom.checkboxes1.append(checkbox("contacts", addTranslated("Show contacts"))); /*i18n*/
        }
        if (ox.upsell.isVisible("tasks")) {
            dom.checkboxes2.append(checkbox("tasks", addTranslated("Show tasks"))); /*i18n*/
        }
        if (ox.upsell.isVisible("infostore")) {
            dom.checkboxes2.append(checkbox("infostore", addTranslated("Show files"))); /*i18n*/
        }
    };

    paint = function (force) {

        var isFirst = false, className = "", link;

        if (dom.container === undefined) {

            // set shortcut
            config = ox.api.config;

            // get layout
            currentLayout = config.get("gui.portal.layout", 2);

            // topbar
            dom.topbar = $("<div/>")
                .addClass("ox-portal-topbar")
                .css({
                    position: "absolute",
                    top: "0px",
                    right: "0px",
                    height: "30px",
                    lineHeight: "30px",
                    left: "0px",
                    padding: "0px 28px 0px 10px",
                    textAlign: "left",
                    zIndex: 2
                })
                .append(
                    $("<i/>")
                        .append(getGreetingPhrase())
                        .append(newtext(", "))
                )
                .append($("<b/>").text(""))
                .append($("<span/>").text(". \u00a0 "))
                .append(
                    link = $("<a/>")
                )
                .appendTo("#portalContainer");

            var id = config.get("identifier");
            var setName = function(data) {
                dom.topbar.children().eq(1).text(String(data[id].display_name));
            };
            internalCache.getUsers([id], setName);
            internalCache.registerUpdate(setName);

            var fnChangeLayout = function (e) {
                // get nodes
                var nodes = dom.controls.find(".ox-portal-layout");
                // remove class
                nodes.eq(currentLayout - 1).removeClass("active");
                // set new layout
                currentLayout = e.data.layout || 2;
                // update config
                config.set("gui.portal.layout", currentLayout);
                configuration_changed_fields.gui = true;
                // add class
                nodes.eq(currentLayout - 1).addClass("active");
                // refresh layout
                refreshLayout();
            };

            // controls
            dom.controls = $("<div/>")
                .addClass("ox-portal-controls")
                .css({
                    position: "absolute",
                    top: "30px",
                    right: "0px",
                    height: "60px",
                    left: "0px",
                    padding: "10px",
                    zIndex: 1
                })
                .append(
                    $("<table/>", { border: "0", cellPadding: "0", cellSpacing: "0" })
                        .append(
                            $("<tbody/>")
                                .append(
                                    $("<tr/>")
                                        .append(
                                            $("<td>")
                                                .append(
                                                    $("<img/>", { src: getFullImgSrc("img/dummy.gif"), alt: "" })
                                                        .addClass("ox-portal-layout one")
                                                        .bind("click", { layout: 1 }, fnChangeLayout)
                                                )
                                                .css("padding", "0")
                                        )
                                        .append(
                                            $("<td>")
                                                .append(
                                                    $("<img/>", { src: getFullImgSrc("img/dummy.gif"), alt: "" })
                                                        .addClass("ox-portal-layout two")
                                                        .bind("click", { layout: 2 }, fnChangeLayout)
                                                )
                                                .css("padding", "0")
                                        )
                                        .append(
                                            $("<td>")
                                                .append(
                                                    $("<img/>", { src: getFullImgSrc("img/dummy.gif"), alt: "" })
                                                        .addClass("ox-portal-layout three")
                                                        .bind("click", { layout: 3 }, fnChangeLayout)
                                                )
                                                .css("padding", "0")
                                        )
                                        .append(
                                            $("<td>")
                                                .append(
                                                    $("<img/>", { src: getFullImgSrc("img/dummy.gif"), alt: "" })
                                                        .addClass("ox-portal-layout four")
                                                        .bind("click", { layout: 4 }, fnChangeLayout)
                                                )
                                                .css("padding", "0")
                                        )
                                        .append(
                                            $("<td>")
                                                .append(
                                                    $("<img/>", { src: getFullImgSrc("img/dummy.gif"), alt: "" })
                                                        .addClass("ox-portal-layout five")
                                                        .bind("click", { layout: 5 }, fnChangeLayout)
                                                )
                                                .css("padding", "0")
                                        )
                                        .append(
                                            dom.checkboxes1 = $("<td/>")
                                                .css({
                                                    verticalAlign: "top",
                                                    height: "60px",
                                                    paddingRight: "20px"
                                                })
                                        )
                                        .append(
                                            dom.checkboxes2 = $("<td/>")
                                                .css({
                                                    verticalAlign: "top",
                                                    height: "60px",
                                                    paddingRight: "20px"
                                                })
                                        )
                                )
                        )
                )
                .hide()
                .appendTo("#portalContainer");

            // add jump into UWA config
            if (ox.api.config.get("modules.uwaWidgets.enabled") === true) {
                dom.controls.find("tr").append(
                    $("<td/>")
                        .css({
                            verticalAlign: "top",
                            height: "60px",
                            padding: "10px 20px 0px 0px"
                        }).append(
                            $.button({
                                title: "UWA Widgets...", /*i18n*/
                                theme: "dark",
                                click: function () {
                                    // jump
                                    ox.UIController.setModule({
                                        module: "configuration",
                                        view: "configuration/portal/external",
                                        success: function () {
                                            ox.widgets.configTree.get("configuration/portal").open();
                                            ox.widgets.configTree.selection.click("configuration/portal/external", false, false, true);
                                        }
                                    });
                                }
                            })
                        )
                );
            }

            drawCheckboxes();

            // which browser?
            if (ox.browser.Gecko) {
                className = "firefox";
            } else if (ox.browser.WebKit) {
                className = "webkit";
            } else if (ox.browser.IE) {
                className = "IE";
            }

            // main container
            var parent = $("#portalContainer").addClass(className);
            dom.container = $("<div>")
                .addClass("ox-portal")
                .css({
                    position: "absolute",
                    top: "30px",
                    right: "0px",
                    bottom: "0px",
                    left: "0px",
                    padding: "10px",
                    paddingRight: IE <= 7 ? "28px" : "10px", // IE7 hack
                    overflow: "hidden",
                    overflowX: "hidden",
                    overflowY: "scroll",
                    zIndex: 3
                })
                .append(
                    dom.columnContainer = $('<div>').addClass('ox-portal-column-container')
                )
                .appendTo(parent);

            // turn into quick config
            $.quickConfig(
                dom.container,
                dom.controls,
                link,
                "Change layout", /*i18n*/
                "Close configuration" /*i18n*/
            );

            // mark active layout
            dom.controls.find(".ox-portal-layout").eq(currentLayout - 1).addClass("active");

            isFirst = true;
        }
        if (isFirst || force) {
            drawPortal();
        } else {
            enable();
        }
        return isFirst;
    };

    /**
     * Enables all portal items.
     * Called automatically when the user enters the portal view.
     */
    enable = function () {
        var i = 0, $i = widgets.length, widget;
        for (; i < $i; i++) {
            widget = widgets[i];
            if (ox.util.isFunction(widget.update)) {
                widget.update();
            }
        }
    };

    // public function/variables
    return {

        // initialize
        paint: paint,

        repaint: function () {
            paint(true);
            drawCheckboxes();
        },

        refresh: function () {
            if (ox.UIController.getModule() === "portal") {
                getAllData();
            } else {
                bPortalNeedsRepaint = true;
            }
        },

        getGrid: function () {
            return grid;
        },

        refreshLayout: function () {
            refreshLayout();
        },

        applyConfig: function () {
            if (ox.UIController.getModule() === "portal") {
                drawCheckboxes();
                getAllData();
            } else {
                bPortalNeedsRepaint = true;
            }
        },

        /**
         * Disables all portal items.
         * Should be called when the user leaves the portal view.
         */
        disable: function () {
            var i = 0, $i = widgets.length, widget;
            for (; i < $i; i++) {
                widget = widgets[i];
                // uwa widgets cannot be disabled, so...
                if (ox.util.isFunction(widget.disable)) {
                    widget.disable();
                }
            }
        },

        /* widget registry */
        registry: registry
    };

}(jQuery));

// -------

registerView(
    "portal",
    // show
    function () {
        showNode("portal");
    },
    // enter
    function () {
        // use auto refresh?
        var autoRefresh = ox.api.config.get("ui.portal.autoRefresh", false) === true;
        
        if (bPortalNeedsRepaint) {
            // repaint
            oPortal.repaint(); // includes drawCheckboxes, getAllData
            bPortalNeedsRepaint = autoRefresh;
        } else {
            if (oPortal.paint()) {
                // set default folder for current module
                ox.api.ui.setFolder();
                // register
                register("OX_Refresh", oPortal.refresh);
                register("OX_Configuration_Changed", oPortal.applyConfig);
                bPortalNeedsRepaint = autoRefresh;
                register("TimeUpdated", function() {
                    if (ox.UIController.getModule() === "portal") {
                        oPortal.paint(true);
                        oPortal.drawCheckboxes();
                    } else {
                        bPortalNeedsRepaint = true;
                    }
                });
            }
        }

        // update toolbar
        ox.ToolBarController.processSelection("default", false, [], true);
    },
    // leave
    function () {
        //unregister("OX_Refresh", oPortal.refresh);
        oPortal.disable();
    },
    // hide
    function () {
        hideNode("portal");
    },
    // change
    function () {
    }
);

function fn_clickedItem(sModule, oObj) {
    OXCache.newRequest(null, sModule, { objects: [oObj] }, null, function () {
        triggerEvent("OX_Direct_Linking", sModule, oObj);
    });
}

register("OX_Portal_Click_Item", fn_clickedItem);

register("OX_Save_Configuration", function () {
    if (ox.api.ui.isModule("portal")) {
        // make sure changes got saved
        ox.uwa.save(jQuery.noop);
    }
});

/*
 * UWA Messaging
 */

(function () {

    var msgHandler = function (message) {
        var id = message.id;
        switch (message.action) {
        case "resizeHeight":
            var frame = document.getElementById('frame_' + id);
            if (frame) {
                // set height
                frame.setAttribute("height", message.value);
                // fix flicker (assumption: widgets that tell about their height don't need scroll bars)
                frame.setAttribute("scrolling", "no");
            }
            break;
        }
    };

    UWA.MessageHandler = new UWA.iFrameMessaging();
    UWA.MessageHandler.init({
        'eventHandler': msgHandler
    });

}());

ox.uwa = { util: {}, toStore: {} };

ox.uwa.list = function (cont) {
    function load() {
        ox.api.http.GET({
            module: "uwaWidgets",
            params: { action: "all" },
            appendColumns: false,
            success: function (response) {
                ox.uwa.cache = response;
                cont(ox.uwa.cache);
            }
        });
    }

    if (ox.api.config.contains("gui.portal.externalcontents") &&
            ox.api.config.get("gui.portal.externalcontents_migration") !== true) {
        ox.uwa.util.migrateUWAs("externalcontents", load);
    } else if (ox.uwa.cache !== undefined) {
        cont(ox.uwa.cache);
    } else {
        load();
    }
};

ox.uwa.exists = function (id) {
    return ox.uwa.get(id) ? true : false;
};

ox.uwa.get = function (id) {
    if (ox.uwa.cache) {
        var i;
        for (i in ox.uwa.cache) {
            if (ox.uwa.cache[i].id === id) {
                return ox.uwa.cache[i];
            }
        }
    }
    return null;
};

ox.uwa.add = function (uwa) {
    // do already exists or is name empty?
    if (isEmpty(uwa.title) || isEmpty(uwa)) {
        return null;
    }
    uwa.id = "tmpid" + ox.util.now();
    // required
    if (!uwa.adj) {
        uwa.adj = { x: 1, y: 1 };
    }

    ox.uwa.cache.push(uwa);
    ox.uwa.toStore[uwa.id] = "new";
    return uwa;
};

ox.uwa.update = function (uwa) {
    // mandatory fields set
    if ((isEmpty(uwa.title) || isEmpty(uwa.url)) && !ox.uwa.exists(uwa.id)) {
        return null;
    }
    var tmpUwa = ox.uwa.get(uwa.id);
    if (tmpUwa) {
        var i;
        for (i in uwa) {
            tmpUwa[i] = uwa[i];
        }
        // it's a new one, so just delete it
        if (!ox.uwa.toStore[tmpUwa.id] && ox.uwa.toStore[tmpUwa.id] !== "new") {
            ox.uwa.toStore[tmpUwa.id] = "update"; // store id for saving
        }
    }
    return tmpUwa;
};

ox.uwa.remove = function (ids) {

    if (typeof ids !== "object") {
        ids = [ ids ];
    }

    var i, ia, $l1 = ids.length;
    for (i = 0; i < ox.uwa.cache.length; i++) {
        for (ia = 0; ia < $l1; ia++) {
            if (ox.uwa.cache[i] && ox.uwa.cache[i].id === ids[ia]) {
                ox.uwa.cache.splice(i, 1);
                // it's a new one, so just delete it
                if (ox.uwa.toStore[ids[ia]] && ox.uwa.toStore[ids[ia]] === "new") {
                    delete ox.uwa.toStore[ids[ia]];
                } else {
                    ox.uwa.toStore[ids[ia]] = "delete"; // store id for saving
                }
            }
        }
    }
};

ox.uwa.save = function (cont) {
    var toStore = [];

    // prepare final object
    var i;
    for (i in ox.uwa.toStore) {
        var status = ox.uwa.toStore[i];
        var object = { action: status, module: "uwaWidgets" };
        if (status === "delete") {
            // delete looks different
            object.id = i;
        } else {
            var tmp = clone(ox.uwa.get(i));
            if (!tmp) {
                // ups, object does not exists
                continue;
            }
            // re-convert from ugly params format
            if (tmp.parameters && typeof tmp.patameters === "string") {
                tmp.parameters =  ox.uwa.util.convertParameters(tmp.parameters);
            }
            // only param adj is allowed for protected widgets
            if (tmp["protected"] === true && tmp.adj) {
                tmp = { id: tmp.id, adj: tmp.adj };
            }
            // set data object
            object.data = tmp;
        }
        // mix it back to final object
        toStore.push(object);
    }

    // store it to the server
    if (toStore.length !== 0) {
        ox.api.http.PUT({
            module: "multiple",
            appendColumns: false,
            processData: false,
            data: toStore,
            success: function (data) {
                // some evil id mapping
                for (var i=0; i < toStore.length; i++) {
                    var tmpObj = toStore[i];
                    // only if new and no error
                    if (tmpObj.action === "new" && !data[i].error) {
                        var oldWidget = ox.uwa.get(tmpObj.data.id);
                        // store old reference, required to update the grid later on
                        data[i].data.tmp_id = oldWidget.id;
                        // update cache
                        oldWidget.id = data[i].data.id;
                    }
                    tmpObj = null;
                }
                // reset temp cache
                ox.uwa.toStore = {}, toStore = null;
                if (cont) {
                    cont(data);
                }
            }
        });
    } else {
        // nothing to do
        if (cont) {
            cont(null);
        }
    }
};

ox.uwa.reset = function () {
    ox.uwa.cache = undefined;
    ox.uwa.toStore = {};
};

ox.uwa.util.convertParameters = function (parameters) {

    if (!parameters) {
        return parameters;
    }

    var formatedParams = "", a, i;

    // re-convert from ugly params format
    if (typeof parameters === "string") {
        formatedParams = {};
        var params1 = parameters.split(",");
        for (a = 0; a < params1.length; a++) {
            var params2 = params1[a].split(":", 1);
            if (params2.length === 1) {
                var key = params2[0];
                formatedParams[trimStr(key)] = trimStr(params1[a].substr(key.length + 1));
            }
        }

    } else if (typeof parameters === "object") {
        for (i in (parameters || {})) {
            var param = parameters[i];
            formatedParams += i + ": " + param + ", ";
        }
        formatedParams = formatedParams.replace(/,(\s+)?$/, "");
    }

    return formatedParams;
};

ox.uwa.util.migrateUWAs = function (type, cont) {
    var uwas = ox.api.config.get("gui.portal." + type, null);
    // nothing to migrate, continue
    if (uwas === null || uwas.length === 0 
            || ox.api.config.get("gui.portal." + type + "_migration") === true) {
        cont();
        return;
    }

    var toStore = [], i = 0, uwa;
    for (; i < uwas.length; i++) {
        // get
        uwa = uwas[i];
        // adj is required, otherwise won't be able to save it
        if (uwa.adj === undefined) {
            uwa.adj = "";
        }
        // new servlet accepts parameters, not parameter
        if (uwa.parameter !== undefined) {
            uwa.parameters = uwa.parameter;
        }
        // prepare multiple
        toStore.push({ action: "new", module: "uwaWidgets", data: uwa });
    }

    // store it to the server
    ox.api.http.PUT({
        module: "multiple",
        appendColumns: false,
        processData: false,
        data: toStore,
        success: function (data) {
            ox.api.config.set("gui.portal." + type + "_migration", true);
            configuration_changed_fields.gui = true;
            triggerEvent("OX_Save_Configuration", false, true);
            cont(data);
        }
    });
};
