/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */
package test.com.openexchange.office.tools.reflection;

import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

//==============================================================================
/** provide several functions where you can manipulate objects at runtime.
 */
public class ObjectManipulation
{
    //--------------------------------------------------------------------------
	/** set a new value on the specified object field ... even it's private
	 *  or final .-)
	 *
	 * 	@param	aTargetObject [IN]
	 * 			the target object where the field value must be set.
	 *
	 * 	@param	sFieldName [IN]
	 * 			the name of the field.
	 *
	 * 	@param	aValue [IN]
	 * 			the new value.
	 */
	public static < TType > void setFieldValue (Object aTargetObject,
					 		     	            String sFieldName   ,
					 		    	            TType  aValue       )
		throws Exception
	{
		Class< ? > aClass = aTargetObject.getClass ();
		Field      aField = aClass.getDeclaredField(sFieldName);
		impl_makeFieldAccessible (aField);
		aField.set(aTargetObject, aValue);
	}

    //--------------------------------------------------------------------------
	/** get the current value of the specified static field ... even it's private
	 *  or final .-)
	 *
	 * 	@param	aClass [IN]
	 * 			the class where the static field value must be retrieved from.
	 *
	 * 	@param	sFieldName [IN]
	 * 			the name of the field.
	 *
	 * 	@return	the current value of that field.
	 */
	@SuppressWarnings("unchecked")
	public static < TType > TType getStaticFieldValue (Class< ? > aClass    ,
					 		     	                   String     sFieldName)
		throws Exception
	{
		Field  aField = aClass.getDeclaredField(sFieldName);
		impl_makeFieldAccessible (aField);
		Object aValue = aField.get (null);
		return (TType) aValue;
	}

	//--------------------------------------------------------------------------
	/** set a new value on the specified static field ... even it's private
	 *  or final .-)
	 *
	 * 	@param	aClass [IN]
	 * 			the class where the static field value must be set.
	 *
	 * 	@param	sFieldName [IN]
	 * 			the name of the field.
	 *
	 * 	@param	aValue [IN]
	 * 			the new value.
	 */
	public static < TType > void setStaticFieldValue (Class< ? > aClass    ,
					 		     	                  String     sFieldName,
					 		    	                  TType      aValue    )
		throws Exception
	{
		Field aField = aClass.getDeclaredField(sFieldName);
		impl_makeFieldAccessible (aField);
		aField.set(null, aValue);
	}

	//--------------------------------------------------------------------------
	/** get the current value from specified field.
	 *
	 * 	@param	aSourceObject [IN]
	 * 			the source object where the field value must be read.
	 *
	 * 	@param	sFieldName [IN]
	 * 			the name of the field.
	 *
	 * 	@return	the current value of these field.
	 * 			Can be null .-)
	 */
	@SuppressWarnings("unchecked")
	public static < TType > TType getFieldValue (Object aSourceObject,
										         String sFieldName   )
		throws Exception
	{
		Class< ? > aClass = aSourceObject.getClass ();
		Field      aField = aClass.getDeclaredField(sFieldName);
		aField.setAccessible(true);
		return (TType) aField.get(aSourceObject);
	}

    //--------------------------------------------------------------------------
	@SuppressWarnings("unchecked")
	public static < TReturnType > TReturnType callPrivateMethod (Object    aTargetObject,
						  				  		     			 String    sMethodName  ,
						  				  		     			 Object... lParameter   )
		throws Exception
	{
		Class< ? >[] lParameterTypes = ObjectManipulation.impl_getParameterTypes(lParameter);
		Class< ? >   aClass          = aTargetObject.getClass ();
		Method[]     lMethods        = aClass.getDeclaredMethods();
		Method       aTargetMethod   = null;

		for (Method aMethod : lMethods)
		{
		    String sNameCheck = aMethod.getName();
		    if ( ! StringUtils.equals(sNameCheck, sMethodName))
		        continue;

		    aTargetMethod = aMethod;

		    Class< ? >[] lParamCheck = aMethod.getParameterTypes();
		    if (ArrayUtils.isEquals(lParamCheck, lParameterTypes))
		        break;
		}

/* ifdef DEBUG .-)
		System.out.println ("ObjectManipulation.callPrivateMethod () invoke '"+aTargetObject+"'.'"+aTargetMethod+"' () ...");
		for (Object aParameter : lParameter)
		    System.out.println (aParameter + " type["+aParameter.getClass ()+"]");
*/

		aTargetMethod.setAccessible(true);
		return (TReturnType) aTargetMethod.invoke(aTargetObject, lParameter);
	}

    //--------------------------------------------------------------------------
	private static Class< ? >[] impl_getParameterTypes (Object... lParameter)
		throws Exception
	{
		int          i               = 0;
		int          c               = lParameter.length;
		Class< ? >[] lParameterTypes = new Class[c];

		for (Object aParameter : lParameter)
			lParameterTypes[i++] = aParameter.getClass ();

		return lParameterTypes;
	}

	//--------------------------------------------------------------------------
	private static void impl_makeFieldAccessible (final Field aField)
		throws Exception
	{
		// remove 'private'
		aField.setAccessible(true);

		// remove 'final'
		final Field aModifiersField = Field.class.getDeclaredField("modifiers");
	    aModifiersField.setAccessible(true);
	    aModifiersField.setInt(aField, aField.getModifiers() & ~Modifier.FINAL);
	}
}