/*
 *
 *    OPEN-XCHANGE legal information
 *
 *    All intellectual property rights in the Software are protected by
 *    international copyright laws.
 *
 *
 *    In some countries OX, OX Open-Xchange, open xchange and OXtender
 *    as well as the corresponding Logos OX Open-Xchange and OX are registered
 *    trademarks.
 *    The use of the Logos is not covered by the GNU General Public License.
 *    Instead, you are allowed to use these Logos according to the terms and
 *    conditions of the Creative Commons License, Version 2.5, Attribution,
 *    Non-commercial, ShareAlike, and the interpretation of the term
 *    Non-commercial applicable to the aforementioned license is published
 *    on the web site http://www.open-xchange.com/EN/legal/index.html.
 *
 *    Please make sure that third-party modules and libraries are used
 *    according to their respective licenses.
 *
 *    Any modifications to this package must retain all copyright notices
 *    of the original copyright holder(s) for the original code used.
 *
 *    After any such modifications, the original and derivative code shall remain
 *    under the copyright of the copyright holder(s) and/or original author(s)per
 *    the Attribution and Assignment Agreement that can be located at
 *    http://www.open-xchange.com/EN/developer/. The contributing author shall be
 *    given Attribution for the derivative code and a license granting use.
 *
 *     Copyright (C) 2016 OX Software GmbH
 *     Mail: info@open-xchange.com
 *
 *
 *     This program is free software; you can redistribute it and/or modify it
 *     under the terms of the GNU General Public License, Version 2 as published
 *     by the Free Software Foundation.
 *
 *     This program is distributed in the hope that it will be useful, but
 *     WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 *     or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *     for more details.
 *
 *     You should have received a copy of the GNU General Public License along
 *     with this program; if not, write to the Free Software Foundation, Inc., 59
 *     Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */

package com.openexchange.office.tools.files;

import java.io.BufferedInputStream;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashMap;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.json.JSONObject;
import com.openexchange.ajax.requesthandler.AJAXRequestData;
import com.openexchange.ajax.requesthandler.crypto.CryptographicServiceAuthenticationFactory;
import com.openexchange.exception.OXException;
import com.openexchange.file.storage.DefaultFile;
import com.openexchange.file.storage.Document;
import com.openexchange.file.storage.File;
import com.openexchange.file.storage.FileStorageFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccess;
import com.openexchange.file.storage.composition.IDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographicAwareIDBasedFileAccessFactory;
import com.openexchange.file.storage.composition.crypto.CryptographyMode;
import com.openexchange.folderstorage.ContentType;
import com.openexchange.folderstorage.FolderService;
import com.openexchange.folderstorage.UserizedFolder;
import com.openexchange.folderstorage.database.contentType.InfostoreContentType;
import com.openexchange.groupware.infostore.InfostoreExceptionCodes;
import com.openexchange.groupware.ldap.User;
import com.openexchange.java.Strings;
import com.openexchange.office.tools.ConfigurationHelper;
import com.openexchange.office.tools.StorageHelper;
import com.openexchange.office.tools.StreamInfo;
import com.openexchange.office.tools.doc.DocumentFormat;
import com.openexchange.office.tools.doc.DocumentType;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.ExceptionToErrorCode;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.session.ServerSession;
import com.openexchange.tx.TransactionAwares;
import com.openexchange.user.UserService;

/**
 * {@link DocFileHelper}
 *
 * The DocFileHelper class contains many utility methods to retrieve a document
 * stream from a file or write a stream to a file. To retrieve meta data from
 * a file, e.g. permissions, locking and determine mine type, extension, document
 * type/format.
 *
 * @author <a href="mailto:kai.ahrens@open-xchange.com">Kai Ahrens</a>
 * @author <a href="mailto:carsten.driesner@open-xchange.com">Carsten Driesner</a>
 */
public class DocFileHelper {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(DocFileHelper.class);

    // stream names used within the document stream
    static public final String OX_RESCUEDOCUMENT_DOCUMENTSTREAM_NAME = "oxoffice/documentStream";
    static public final String OX_RESCUEDOCUMENT_OPERATIONSSTREAM_NAME = "oxoffice/operationUpdates";
    static public final String OX_RESCUEDOCUMENT_RESOURCESTREAM_NAME_PREFIX = "oxoffice/resource/";
    static public final String OX_DOCUMENT_EXTENDEDDATA = "oxoffice/extendedData.json";

    // Mime types and extensions
    // ------------------------------------------------------
    static public final String IMAGE_JPEG = "image/jpeg";
    static public final String EXTENSION_JPG_1 = "jpg";
    static public final String EXTENSION_JPG_2 = "jpeg";
    static public final String IMAGE_EMF = "image/x-emf";
    static public final String IMAGE_EMF2 = "image/emf";
    static public final String EXTENSION_EMF = "emf";
    static public final String IMAGE_WMF = "image/x-wmf";
    static public final String EXTENSION_WMF = "wmf";
    static public final String IMAGE_GIF = "image/gif";
    static public final String EXTENSION_GIF = "gif";
    static public final String IMAGE_PICT = "image/pict";
    static public final String EXTENSION_PICT = "pct";
    static public final String IMAGE_PNG = "image/png";
    static public final String EXTENSION_PNG = "png";
    public static final String IMAGE_TIFF = "image/tiff";
    public static final String EXTENSION_TIFF = "tiff";
    public static final String EXTENSION_TIFF2 = "tif";
    public static final String IMAGE_EPS = "application/postscript"; // as reported by XmlGraphics
    public static final String EXTENSION_EPS = "eps";
    public static final String IMAGE_BMP = "image/bmp";
    public static final String EXTENSION_BMP = "bmp";
    public static final String EXTENSION_ODFTEXT = "odt";
    public static final String EXTENSION_ODT = "odt";
    public static final String DOC_ODF_TEXT = "application/vnd.oasis.opendocument.text";
    public static final String EXTENSION_ODS = "ods";
    public static final String DOC_ODF_SPREADSHEET = "application/vnd.oasis.opendocument.spreadsheet";
    public static final String EXTENSION_ODP = "odp";
    public static final String DOC_ODF_PRESENTATION = "application/vnd.oasis.opendocument.presentation";
    public static final String EXTENSION_ODG = "odg";
    public static final String DOC_ODF_GRAPHIC = "application/vnd.oasis.opendocument.graphics";
    public static final String EXTENSION_DOCX = "docx";
    public static final String DOC_DOCX_TEXT = "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
    public static final String EXTENSION_XLSX = "xlsx";
    public static final String DOC_DOCX_SPREADSHEET = "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet";
    public static final String EXTENSION_PPTX = "pptx";
    public static final String DOC_DOCX_PRESENTATION = "application/vnd.openxmlformats-officedocument.presentationml.presentation";
    public static final String EXTENSION_XML = "xml";
    public static final String TEXT_XML = "text/xml";
    public static final String OCTET_STREAM = "application/octet-stream";

    // - Static members --------------------------------------------------------

    static private HashMap<String, DocumentFormat> m_extensionMap = null;
    static private volatile Object m_Synchronizer = new Object();

    // - Members --------------------------------------------------------

    private ServiceLookup m_services = null;
    private DocumentFormat m_documentFormat = DocumentFormat.NONE;
    private DocumentType m_documentType = DocumentType.NONE;

    /**
     * Property class to transport result data for a write stream request.
     *
     * @author Carsten Driesner
     *
     */
    public static class WriteInfo {

        public String version;
        public ErrorCode errorCode;

        /**
         * Creates a write info object with the provided version and error
         * code information resulting from the write stream operation.
         *
         * @param version the version of the written stream or null.
         * @param errorCode the error code of the write stream operation.
         */
        public WriteInfo(String version, ErrorCode errorCode) {
            this.version = version;
            this.errorCode = errorCode;
        }
    }

    /**
     * Property class to transport result data for a rename request.
     *
     * @author Carsten Driesner
     *
     */
    public static class RenameResult {

        public ErrorCode errorCode;
        public File metaData;
        public JSONObject jsonResult;
        public String newFileId;
        public boolean reload;

        /**
         * Creates a success rename result object.
         */
        public RenameResult() {
            this.errorCode = ErrorCode.NO_ERROR;
            this.metaData = null;
            this.jsonResult = new JSONObject();
            this.newFileId = null;
            this.reload = false;
        }

        /**
         * Creates a rename result object with the provided
         * error code and meta data.
         *
         * @param errorCode the error code from the rename processing.
         * @param metaData the new mata data or null if not available.
         */
        public RenameResult(final ErrorCode errorCode, final File metaData, final String newFileId) {
            this.errorCode = errorCode;
            this.metaData = metaData;
            this.jsonResult = new JSONObject();
            this.newFileId = newFileId;
        }
    }

    /**
     * Initializes a new {@link DocFileHelper}.
     */
    public DocFileHelper(ServiceLookup services) {
        super();

        m_services = services;
    }

    /**
     * Retrieves the service lookup to create new services.
     *
     * @return
     *         The service lookup used by the file helper instance.
     */
    public ServiceLookup getServiceLookup() {
        return m_services;
    }

    /**
     * Retrieves the document format.
     *
     * @return
     *         The document format of the file used by the file helper
     *         instance.
     */
    public DocumentFormat getDocumentFormat() {
        return m_documentFormat;
    }

    /**
     * Retrieves the document type.
     *
     * @return
     *         The document format of the file used by the file helper
     *         instance.
     */
    public DocumentType getDocumentType() {
        return m_documentType;
    }

    /**
     * Access the responsible storage adapter to retrieve the meta
     * data directly from the storage. This method doesn't cache
     * the data and is therefore expensive!
     *
     * @param session
     *            The session to be used for a possible access to the real
     *            meta data.
     *
     * @return
     */
    public File getMetaDataFromStorage(Session session, final String fileId) throws OXException {
        return getMetaDataFromFile(session, fileId, FileStorageFileAccess.CURRENT_VERSION);
    }

    /**
     * Retrieves the document stream from the file.
     *
     * @param session
     *            The session of the user who requested to retrieve the document
     *            stream.
     *
     * @return
     *         The document stream or null if the document stream could not
     *         be retrieved.
     */
    public StreamInfo getDocumentStream(Session session, StorageHelper storageHelper, final String folderId, final String fileId, String encryptionInfo) {
        final StreamInfo streamInfo = DocFileHelper.getDocumentStream(m_services, session, folderId, fileId, FileStorageFileAccess.CURRENT_VERSION, storageHelper, encryptionInfo);

        if (null != streamInfo) {
            m_documentFormat = streamInfo.getDocumentFormat();
            m_documentType = getDocumentTypeFromDocumentFormat(m_documentFormat);
        }

        return streamInfo;
    }

    /**
     * Retrieves the document stream from the provided path.
     *
     * @param path
     *            A full-qualified path to a file stored in local file system.
     *
     * @return
     *         The document data as an input stream or null if there was a problem
     *         accessing the file.
     */
    static public InputStream getDocumentStreamFromLocalFile(String path) {
        final java.io.File localFile = new java.io.File(path);
        InputStream inputStream = null;

        if (localFile.exists() && localFile.isFile()) {
            try {
                return new BufferedInputStream(new FileInputStream(localFile));
            } catch (FileNotFoundException e) {
                // nothing to do - we return null instead of the stream
            }
        }

        return inputStream;
    }

    /**
     * Retrieves the document stream requested by the AJAX request data.
     *
     * @param services
     *            The service lookup to be used by the method to create necessary
     *            services.
     *
     * @param session
     *            The session of the user who requested to retrieve the document
     *            stream.
     *
     * @param requestData
     *            The AJAX request data which must contain the id of the file, folder_id and
     *            the version to be used to retrieve the document stream. If the version
     *            is omitted the method uses the current version of the file.
     *
     * @return
     *         The document stream or null if the document stream could not
     *         be retrieved.
     */
    static public StreamInfo getDocumentStream(ServiceLookup services, Session session, AJAXRequestData requestData, final StorageHelper storageHelper, final String encryptionInfo ) {
        final String fileId = requestData.getParameter("id");
        final String folderId = requestData.getParameter("folder_id");
        String version = requestData.getParameter("version");

        if ((null == version) || (0 == version.length()) || (version.equals("0"))) {
            version = FileStorageFileAccess.CURRENT_VERSION;
        }

        // create temporary DocFileHelper instance and cal
        return getDocumentStream(services, session, folderId, fileId, version, storageHelper, encryptionInfo);
    }

    /**
     * Retrieves the document stream and the meta data specified by the fileId,
     * folderId and version.
     *
     * @param services
     *  The service lookup to be used by the method to create necessary
     *  services.
     *
     * @param session
     *  The session of the user who requested to retrieve the document
     *  stream.
     *
     * @param folderId
     *  The folder id where the file resides.
     *
     * @param fileId
     *  The file id of the file to read the document stream from.
     *
     * @param version
     *  The version of the file to be used to read the document stream from.
     *  If the version is null the method uses the current version.
     *
     * @param writeProtected
     *  Provides the write protected state of the file from the perspective
     *  of the user.
     *
     * @param lockedUntil
     *  Provides how long the file has exclusively been locked.
     *
     * @param lockedByUser
     *  Provides who has locked the file exclusively.
     *
     * @param documentFormat
     *  Provides the document format of the file.
     *
     * @param documentSize
     *  Provides the size of the file.
     *
     * @return
     *  A StreamInfo object which contains the stream and meta data of the requested
     *  file. ATTENTION: The stream and/or meta data member of the StreamInfo object can be null
     *  and must be checked by the caller, including the errorCode member which provides
     *  more information about the cause of any problem.
     */
    protected static StreamInfo getDocumentStream(ServiceLookup services, Session session, String folderId, String fileId, String version, final StorageHelper storageHelper, final String encryptionInfo) {
        final StreamInfo streamInfo = new StreamInfo();
        final IDBasedFileAccessFactory fileFactory = services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFolderAccessFactory folderFactory = services.getService(IDBasedFolderAccessFactory.class);
        final UserService userService = services.getService(UserService.class);
        final IDBasedFolderAccess folderAccess = ((null != folderFactory) && (null != session)) ? folderFactory.createAccess(session) : null;
        final long time1 = System.currentTimeMillis();

        IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session) && !Strings.isEmpty(fileId) && !Strings.isEmpty(folderId)) ? fileFactory.createAccess(session) : null;
        IDBasedFileAccess fileAccess4MetaData = fileAccess;

        if ((null != fileAccess) && (null != encryptionInfo) && (encryptionInfo.length() > 0)) {
            CryptographicAwareIDBasedFileAccessFactory encryptionAwareFileAccessFactory = services.getOptionalService(CryptographicAwareIDBasedFileAccessFactory.class);
            CryptographicServiceAuthenticationFactory encryptionAuthenticationFactory = services.getOptionalService(CryptographicServiceAuthenticationFactory.class);

            try {

                if ((encryptionAwareFileAccessFactory != null) && (encryptionAuthenticationFactory != null)) {
                    EnumSet<CryptographyMode> cryptMode = EnumSet.of(CryptographyMode.DECRYPT);
                    // String authentication = encryptionAuthenticationFactory.createAuthenticationFrom(request.optHttpServletRequest());
                    String authentication = encryptionInfo;
                    fileAccess = encryptionAwareFileAccessFactory.createAccess(fileAccess, cryptMode, session, authentication );
                }
                else {
                    // file is encrypted, but we can't decrypt...
                    fileAccess = null;
                }
            } catch (final OXException e) {
                fileAccess = null;
            }

        }

        if (null != fileAccess) {
            try {
                final String accessId = fileId;
                Document docData = null;
                InputStream inputStream = null;
                File metaData = null;

                // first check the support for the efficient data/metadata retrieval
                if (storageHelper.supportsEfficientRetrieval()) {
                    docData = fileAccess.getDocumentAndMetadata(accessId, version);
                    if (null != docData) {
                        inputStream = docData.getData();
                        if (fileAccess4MetaData == fileAccess) { // otherwise get it via fileAccess4MetaData later, so we get the correct name and size for encrypted files
                            metaData = docData.getFile();
                        }
                    }
                }

                // Fallback, if that doesn't work
                if (null == inputStream) {
                    inputStream = fileAccess.getDocument(accessId, version);
                }

                // set the initial properties
                streamInfo.setFileAccess(fileAccess);
                streamInfo.setDocumentStream(inputStream);

                if (null != streamInfo.getDocumentStream()) {

                    // possible fallback, if we didn't receive the meta data
                    if (null == metaData) {
                        try {
                            metaData = fileAccess4MetaData.getFileMetadata(accessId, version);
                        } catch (OXException e) {
                            LOG.warn("Couldn't retrieve meta data of the file", e);
                        }
                    }

                    if (null != metaData) {
                        final Date now = new Date();
                        final Date lockedUntil = metaData.getLockedUntil();

                        // determine write protected state using the write permissions & file locking
                        streamInfo.setWriteProtected(!FileHelper.canWriteToFile(services, session, metaData, session.getUserId()));

                        streamInfo.setMetaData(metaData);
                        streamInfo.setDocumentFormat(getDocumentFormat(metaData.getFileName()));
                        if ((null != lockedUntil) && (lockedUntil.after(now))) {
                            // determine special lock properties only, if lock is still active
                            streamInfo.setLockedByUserId(metaData.getModifiedBy());

                            if (null != userService) {
                                try {
                                    final User user = userService.getUser(metaData.getModifiedBy(), userService.getContext(session.getContextId()));
                                    streamInfo.setLockedByUser(user.getDisplayName());
                                } catch (OXException e) {
                                    // set user name to empty for unknown user name
                                    streamInfo.setLockedByUser("");
                                    LOG.warn("Couldn't retrieve user name who locked the file", e);
                                }
                            }
                        }
                    } else {
                        LOG.warn("Couldn't retrieve meta data of the file");
                    }
                }
            } catch (OXException e) {
                LOG.error("Couldn't read document stream", e);
                streamInfo.setDocumentStream(null);
                streamInfo.setFileAccess(null);
                streamInfo.setErrorCode(ExceptionToErrorCode.map(e, ErrorCode.LOADDOCUMENT_FAILED_ERROR, true));
            } catch (Exception e) {
                LOG.error("Couldn't read document stream", e);
                streamInfo.setDocumentStream(null);
                streamInfo.setFileAccess(null);
                // set general error as we cannot extract the exact cause
                streamInfo.setErrorCode(ErrorCode.LOADDOCUMENT_FAILED_ERROR);
            } finally {
                closeFileAccess(fileAccess);
            }
        }
        LOG.trace("RT connection: TIME to get document stream = " + (System.currentTimeMillis() - time1) + "ms");

        return streamInfo;
    }

    /**
     * Finishes the access to the IDBasedFileAccess instance.
     *
     * @param fileAccess a IDBasedFileAccess instance or null
     */
    private static void closeFileAccess(IDBasedFileAccess fileAccess) {
        try {
            if (null != fileAccess) {
                fileAccess.finish();
            }
        } catch (OXException e) {
            LOG.warn("Couldn't finish the file access", e);
        }
    }

    /**
     * Writes the provided document stream to the file associated with this
     * FileHelper instance.
     *
     * @param services
     *            The service lookup to be used by the method to create necessary
     *            services.
     *
     * @param session
     *            The session of the user who wants to write the document stream to the
     *            file.
     *
     * @param documentStream
     *            The document stream that should be stored into the file.
     *
     * @param fileId
     *            The file id of the file to read the document stream from.
     *
     * @param folderId
     *            The folder id where the file resides.
     *
     * @param metaData
     *            Optional meta data of the file to be written. Will be used to optimize the
     *            access to the storage, if present.
     *
     * @param storageHelper
     *            An instance which provides storage specific properties in the context of the
     *            session and folder used by this call.
     *
     * @param userExtension
     *            Specifies the extension to be used for the file.
     *
     * @param setUserExtension
     *            Specifies if the extension of the file should be set or
     *            removed. If TRUE the user extension will be set otherwise the user
     *            extension will be removed.
     *
     * @param revisionless
     *            Specifies if the stream should be written to the file using a new
     *            version or overwrite the latest version. TRUE means overwrite
     *            the latest version with the provided document stream. FALSE to
     *            create a new version.
     *
     * @return
     *         A WriteInfo instance which contains the information about the write document
     *         stream process. Information like error code, version and other properties
     *         can be retrieved.
     */
    public static WriteInfo writeDocumentStream(final ServiceLookup services, final Session session, final InputStream documentStream, String fileId, String folderId, final File metaData, final StorageHelper storageHelper, String userExtension, boolean setUserExtension, boolean revisionless, boolean encrypt) {

        final WriteInfo writeInfo = new WriteInfo(null, ErrorCode.NO_ERROR);
        final IDBasedFileAccessFactory fileFactory = services.getService(IDBasedFileAccessFactory.class);
        IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
        final long time1 = System.currentTimeMillis();

        if ((null != fileAccess) && encrypt) {
            CryptographicAwareIDBasedFileAccessFactory encryptionAwareFileAccessFactory = services.getOptionalService(CryptographicAwareIDBasedFileAccessFactory.class);
            CryptographicServiceAuthenticationFactory encryptionAuthenticationFactory = services.getOptionalService(CryptographicServiceAuthenticationFactory.class);

            try {

                if ((encryptionAwareFileAccessFactory != null) && (encryptionAuthenticationFactory != null)) {
                    EnumSet<CryptographyMode> cryptMode = EnumSet.of(CryptographyMode.ENCRYPT);
                    fileAccess = encryptionAwareFileAccessFactory.createAccess(fileAccess, cryptMode, session );
                }
                else {
                    // file is encrypted, but we can't decrypt...
                    fileAccess = null;
                }
            } catch (final OXException e) {
                fileAccess = null;
            }

        }

        if ((null != documentStream) && (null != fileAccess) && !Strings.isEmpty(fileId)) {
            final String accessId = fileId;
            final String version = FileStorageFileAccess.CURRENT_VERSION;
            boolean rollback = false;

            try {
                final File docMetaData = (null != metaData) ? metaData : fileAccess.getFileMetadata(accessId, version);
                final DefaultFile newMetadata = new DefaultFile();
                final String fileName = docMetaData.getFileName();
                final String title = docMetaData.getTitle();
                final ArrayList<com.openexchange.file.storage.File.Field> modifiedColumns = new ArrayList<com.openexchange.file.storage.File.Field>();

                // make sure that the important parts of the meta data are filled
                newMetadata.setId(fileId);
                newMetadata.setFolderId(folderId);
                newMetadata.setFileMIMEType(docMetaData.getFileMIMEType());
                newMetadata.setFileName(fileName);
                newMetadata.setTitle((null == title) ? fileName : title);

                modifiedColumns.add(com.openexchange.file.storage.File.Field.FILENAME);
                modifiedColumns.add(com.openexchange.file.storage.File.Field.TITLE);

                // ensure that the filename has the given user extension, if set as call parameter
                if (null != userExtension) {
                    final boolean hasUserExtension = fileName.endsWith(userExtension);

                    if (!hasUserExtension && setUserExtension) {
                        // add user extension
                        newMetadata.setFileName(fileName + userExtension);
                    } else if (hasUserExtension && !setUserExtension) {
                        // remove user extension
                        newMetadata.setFileName(fileName.substring(0, fileName.length() - userExtension.length()));
                    }
                }
                fileAccess.startTransaction();
                rollback = true;
                LOG.trace("RT connection: [writeDocumentStream] saveDocument using meta data: " + ((null != newMetadata) ? newMetadata.toString() : "null"));
                if (revisionless && storageHelper.supportsIgnorableVersion()) {
                    // use revision-less save, if the underlying storage supports it
                    fileAccess.saveDocument(newMetadata, documentStream, FileStorageFileAccess.DISTANT_FUTURE, modifiedColumns, true);
                } else {
                    // fall-back for storages where non-revision save is not supported
                    fileAccess.saveDocument(newMetadata, documentStream, FileStorageFileAccess.DISTANT_FUTURE, modifiedColumns);
                }
                fileAccess.commit();
                rollback = false;

                writeInfo.version = newMetadata.getVersion();
            } catch (final OXException e) {
                LOG.error("Couldn't write document stream, Exception caught: ", e);
                writeInfo.errorCode = ExceptionToErrorCode.map(e, ErrorCode.SAVEDOCUMENT_FAILED_ERROR, true);
            } catch (final Exception e) {
                LOG.error("Couldn't write document stream, Exception caught: ", e);
                writeInfo.errorCode = ErrorCode.SAVEDOCUMENT_FAILED_ERROR;
            } finally {
                // Roll-back (if needed) and finish
                if (rollback) {
                    TransactionAwares.rollbackSafe(fileAccess);
                }
                TransactionAwares.finishSafe(fileAccess);
            }
        }
        LOG.trace("RT connection: TIME to write document stream = " + (System.currentTimeMillis() - time1) + "ms");

        return writeInfo;
    }

    /**
     * Renames the document file in the info store using the provided
     * new file name. ATTENTION: This method doesn't check for the
     * storage capabilities. It just
     *
     * @param session
     *            The session of the user who requested to rename the document file.
     *
     * @param newFileName
     *            The new file name of the document file.
     *
     * @return
     *         The new file name of the document file. Can be null if the rename
     *         operation failed.
     */
    public RenameResult renameDocument(Session session, String newFileName, final String folderId, final String fileId) {
        RenameResult renameResult = new RenameResult();
        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
        final String currentVersion = FileStorageFileAccess.CURRENT_VERSION;
        final long time1 = System.currentTimeMillis();

        ErrorCode result = ErrorCode.NO_ERROR;

        if ((null != fileAccess) && (!Strings.isEmpty(newFileName))) {
            try {
                final ArrayList<com.openexchange.file.storage.File.Field> modifiedColumns = new ArrayList<com.openexchange.file.storage.File.Field>();
                final DefaultFile metadata = new DefaultFile();

                metadata.setId(fileId);
                metadata.setFolderId(folderId);
                metadata.setFileName(newFileName);
                metadata.setVersion(currentVersion);

                modifiedColumns.add(com.openexchange.file.storage.File.Field.FILENAME);
                renameResult.newFileId = fileAccess.saveFileMetadata(metadata, FileStorageFileAccess.DISTANT_FUTURE, modifiedColumns);
                LOG.debug("RT connection: Renamed document file-id: " + renameResult.newFileId);
                fileAccess.commit();

                // retrieve the real filename from the metadata, it may have
                // been changed from the requested one by the file service;
                // add a short sleep in order to avoid asynchronous errors while
                // getting the updated metadata, which may currently happen with
                // cascaded DB backends
                // (see also https://bugs.open-xchange.com/show_bug.cgi?id=26080)
                try {
                    renameResult.metaData = fileAccess.getFileMetadata(renameResult.newFileId, currentVersion);
                } catch (OXException e) {
                    LOG.debug("RT connection: Access meta data of renamed file failed!", e);
                    result = ErrorCode.RENAMEDOCUMENT_FAILED_ERROR;
                }
            } catch (Exception e) {
                result = ErrorCode.RENAMEDOCUMENT_FAILED_ERROR;
                if (e instanceof OXException) {
                    // map possible exceptions to our error codes
                    result = ExceptionToErrorCode.map((OXException) e, ErrorCode.RENAMEDOCUMENT_FAILED_ERROR, false);
                }
                LOG.error(result.getDescription(), e);
            }
        } else {
            // map this problem to validation error
            result = ErrorCode.RENAMEDOCUMENT_VALIDATION_FAILED_CHARACTERS_ERROR;
        }

        renameResult.errorCode = result;
        LOG.trace("RT connection: TIME to rename document file = " + (System.currentTimeMillis() - time1) + "ms");

        return renameResult;
    }

    /**
     * Provides the latest version string of the file
     *
     * @param session
     *            The session to be used to access the file store.
     *
     * @return
     *         The version string or null.
     *
     * @throws OXEception
     *             Throws an exception if the meta data of the document file
     *             couldn't be retrieved.
     */
    public String getVersionFromFile(final Session session, final String fileId) throws OXException {
        return (String) getFieldValueFromFile(session, fileId, com.openexchange.file.storage.File.Field.VERSION);
    }

    /**
     * Provides the file name of the file
     *
     * @param session
     *            The session to be used to access the file store.
     *
     * @param fileId
     *            The id of the file which name should be retrieved.
     *
     * @return
     *         The file name or null.
     *
     * @throws OXEception
     *             Throws an exception if the meta data of the document file
     *             couldn't be retrieved.
     */
    public String getFilenameFromFile(final Session session, final String fileId) throws OXException {
        return (String) getFieldValueFromFile(session, fileId, com.openexchange.file.storage.File.Field.FILENAME);
    }

    /**
     * Retrieves the file meta data using the provided session.
     *
     * @param session
     *            The session to be used as the context to retrieve the file meta data.
     *
     * @param fileId
     *            The id of the file which name should be retrieved.
     *
     * @param version
     *            A version of the file where the meta data should be retrieved from.
     *            Can be null where the method uses the latest version.
     *
     * @return
     *         The file meta data or null if necessary services are not available.
     *
     * @throws OXException
     *             Throws an exception if the meta data of the document file couldn't
     *             be retrieved.
     */
    public File getMetaDataFromFile(Session session, String fileId, String version) throws OXException {
        final IDBasedFileAccessFactory fileFactory = m_services.getService(IDBasedFileAccessFactory.class);
        final IDBasedFileAccess fileAccess = ((null != fileFactory) && (null != session)) ? fileFactory.createAccess(session) : null;
        final String fileVersion = (null == version) ? FileStorageFileAccess.CURRENT_VERSION : version;
        final long time1 = System.currentTimeMillis();
        com.openexchange.file.storage.File metaData = null;

        if (null != fileAccess) {
            final String accessId = fileId;
            metaData = fileAccess.getFileMetadata(accessId, fileVersion);
        }
        LOG.trace("RT connection: TIME to get meta data from file = " + (System.currentTimeMillis() - time1) + "ms");

        return metaData;
    }

    /**
     * Provides the specified field value of the file
     *
     * @param session
     *            The session to be used to access the file store.
     *
     * @return
     *         The field value or null.
     *
     * @throws OXEception
     *             Throws an exception if the meta data of the document file
     *             couldn't be retrieved.
     */
    protected Object getFieldValueFromFile(Session session, final String fileId, com.openexchange.file.storage.File.Field field) throws OXException {

        Object result = null;

        try {
            final String currentVersion = FileStorageFileAccess.CURRENT_VERSION;
            final File metaData = getMetaDataFromFile(session, fileId, currentVersion);

            if (null != metaData) {
                switch (field) {
                    case FILENAME: {
                        result = metaData.getFileName();
                        break;
                    }

                    case FILE_SIZE: {
                        result = Long.valueOf(metaData.getFileSize());
                        break;
                    }

                    case FILE_MIMETYPE: {
                        result = metaData.getFileMIMEType();
                        break;
                    }

                    case VERSION: {
                        result = metaData.getVersion();
                        break;
                    }

                    default: {
                        break;
                    }
                }
            }
        } catch (OXException e) {
            if (LOG.isWarnEnabled()) {
                LOG.warn("RT connection: Couldn't retrieve field value from document meta data", e);
            }
            throw e;
        }

        return result;
    }

    // - Statics ---------------------------------------------------------------

    /**
     * Retrieve a file access.
     *
     * @param session
     *            The session associated with the file access.
     *
     * @return
     *         The folder access or null if no folder access is available.
     *
     * @throws OXException
     */
    public IDBasedFileAccess getFileAccess(Session session) throws OXException {
        IDBasedFileAccessFactory factory = m_services.getService(IDBasedFileAccessFactory.class);
        return factory.createAccess(session);
    }

    /**
     * Maps the user language provided by a User object to a ISO compliant
     * language code. E.g. en_US => en-US
     *
     * @param userLanguage
     *            The user language as provided by the User methods getPreferredLanguage
     *
     * @return
     *         An ISO compliant language code or null if language code couldn't be
     *         parsed correctly.
     */
    static public String mapUserLanguageToLangCode(String userLanguage) {
        return (null != userLanguage) ? (userLanguage.indexOf('_') >= 0 ? userLanguage.replace('_', '-') : userLanguage) : null;
    }

    /**
     * Tries to determine the mime type from a given file name.
     *
     * @param fileName
     *            The file name that should be used to retrieve the mime type from.
     *
     * @return
     *         The mime type of the fileName or the empty string if the
     *         mime type could not be determined.
     */
    static public String getMimeType(String fileName) {
        final String extension = FileHelper.getExtension(fileName);
        String mimeType = "";

        if (extension.length() > 0) {
            switch (getDocumentFormat(fileName)) {
                case DOCX: {
                    mimeType = DOC_DOCX_TEXT;
                    break;
                }

                case PPTX: {
                    mimeType = DOC_DOCX_PRESENTATION;
                    break;
                }

                case XLSX: {
                    mimeType = DOC_DOCX_SPREADSHEET;
                    break;
                }

                case ODT: {
                    mimeType = DOC_ODF_TEXT;
                    break;
                }

                case ODS: {
                    mimeType = DOC_ODF_SPREADSHEET;
                    break;
                }

                case NONE:
                case ODG:
                case ODP:
                default: {
                    if (extension.equals(EXTENSION_DOCX)) {
                        mimeType = DOC_DOCX_TEXT;
                    } else if (extension.equals(EXTENSION_XLSX)) {
                        mimeType = DOC_DOCX_SPREADSHEET;
                    } else if (extension.equals(EXTENSION_PPTX)) {
                        mimeType = DOC_DOCX_PRESENTATION;
                    } else if (extension.equals(EXTENSION_ODT)) {
                        mimeType = DOC_ODF_TEXT;
                    } else if (extension.equals(EXTENSION_ODS)) {
                        mimeType = DOC_ODF_SPREADSHEET;
                    } else if (extension.equals(EXTENSION_ODP)) {
                        mimeType = DOC_ODF_PRESENTATION;
                    } else if (extension.equals(EXTENSION_ODG)) {
                        mimeType = DOC_ODF_GRAPHIC;
                    } else if (extension.equals(EXTENSION_BMP)) {
                        mimeType = IMAGE_BMP;
                    } else if (extension.equals(EXTENSION_EMF)) {
                        mimeType = IMAGE_EMF;
                    } else if (extension.equals(EXTENSION_GIF)) {
                        mimeType = IMAGE_GIF;
                    } else if (extension.equals(EXTENSION_JPG_1)) {
                        mimeType = IMAGE_JPEG;
                    } else if (extension.equals(EXTENSION_JPG_2)) {
                        mimeType = IMAGE_JPEG;
                    } else if (extension.equals(EXTENSION_PICT)) {
                        mimeType = IMAGE_PICT;
                    } else if (extension.equals(EXTENSION_PNG)) {
                        mimeType = IMAGE_PNG;
                    } else if (extension.equals(EXTENSION_TIFF)) {
                        mimeType = IMAGE_TIFF;
                    } else if (extension.equals(EXTENSION_TIFF2)) {
                        mimeType = IMAGE_TIFF;
                    } else if (extension.equals(EXTENSION_WMF)) {
                        mimeType = IMAGE_WMF;
                    } else if (extension.equals(EXTENSION_XML)) {
                        mimeType = TEXT_XML;
                    }

                    break;
                }
            }
        }

        return ((mimeType.length() < 1) ? OCTET_STREAM : mimeType);
    }

    /**
     * Provides a global map containing associations between extension
     * and general document format.
     *
     * @return
     *         The global map containing associations between extension
     *         and general document format.
     */
    static public HashMap<String, DocumentFormat> getExtensionMap() {
        // Java DCL pattern with volatile member
        HashMap<String, DocumentFormat> serviceMap = m_extensionMap;

        if (null == serviceMap) {
            synchronized (m_Synchronizer) {
                if (null == (serviceMap = m_extensionMap)) {
                    serviceMap = m_extensionMap = new HashMap<String, DocumentFormat>();

                    m_extensionMap.put("docx", DocumentFormat.DOCX);
                    m_extensionMap.put("docx_ox", DocumentFormat.DOCX);
                    m_extensionMap.put("docm", DocumentFormat.DOCX);
                    m_extensionMap.put("docm_ox", DocumentFormat.DOCX);
                    m_extensionMap.put("dotx", DocumentFormat.DOCX);
                    m_extensionMap.put("dotx_ox", DocumentFormat.DOCX);
                    m_extensionMap.put("dotm", DocumentFormat.DOCX);
                    m_extensionMap.put("dotm_ox", DocumentFormat.DOCX);
                    m_extensionMap.put("pptx", DocumentFormat.PPTX);
                    m_extensionMap.put("pptx_ox", DocumentFormat.PPTX);
                    m_extensionMap.put("potx", DocumentFormat.PPTX);
                    m_extensionMap.put("potx_ox", DocumentFormat.PPTX);
                    m_extensionMap.put("potm", DocumentFormat.PPTX);
                    m_extensionMap.put("potm_ox", DocumentFormat.PPTX);
                    m_extensionMap.put("xlsx", DocumentFormat.XLSX);
                    m_extensionMap.put("xlsx_ox", DocumentFormat.XLSX);
                    m_extensionMap.put("xltx", DocumentFormat.XLSX);
                    m_extensionMap.put("xltx_ox", DocumentFormat.XLSX);
                    m_extensionMap.put("xlsm", DocumentFormat.XLSX);
                    m_extensionMap.put("xlsm_ox", DocumentFormat.XLSX);
                    m_extensionMap.put("xltm", DocumentFormat.XLSX);
                    m_extensionMap.put("xltm_ox", DocumentFormat.XLSX);
                    m_extensionMap.put("odt", DocumentFormat.ODT);
                    m_extensionMap.put("odt_ox", DocumentFormat.ODT);
                    m_extensionMap.put("ott", DocumentFormat.ODT);
                    m_extensionMap.put("ott_ox", DocumentFormat.ODT);
                    m_extensionMap.put("ods", DocumentFormat.ODS);
                    m_extensionMap.put("ods_ox", DocumentFormat.ODS);
                    m_extensionMap.put("ots", DocumentFormat.ODS);
                    m_extensionMap.put("ots_ox", DocumentFormat.ODS);

                }
            }
        }

        return serviceMap;
    }

    /**
     * Tries to determine the general document format from a given file name.
     *
     * @param fileName
     *            The file name to be used to determine the document format.
     *
     * @return
     *         The document format or DocumentFormat.NONE if no known format could
     *         be found.
     */
    static public DocumentFormat getDocumentFormat(String fileName) {
        final String fileExtension = FileHelper.getExtension(fileName, true);
        final HashMap<String, DocumentFormat> extensionMap = getExtensionMap();
        DocumentFormat documentFormat = DocumentFormat.NONE;

        if ((null != fileExtension) && extensionMap.containsKey(fileExtension)) {
            documentFormat = extensionMap.get(fileExtension);
        }

        return documentFormat;
    }

    /**
     * Tries to determine the general document type from a given file name.
     *
     * @param fileName
     *            The file name to be used to determine the document format.
     *
     * @return
     *         The document type or DocumentType.NONE if no known type could
     *         be found.
     */
    static public DocumentType getDocumentType(String fileName) {
        return getDocumentTypeFromDocumentFormat(getDocumentFormat(fileName));
    }

    /**
     * Maps a document format to a document type.
     *
     * @param documentFormat
     *            The document format where the type should be determined.
     *
     * @return
     *         The document type or DocumentType.NONE if no known format has been
     *         provided.
     */
    static public DocumentType getDocumentTypeFromDocumentFormat(DocumentFormat documentFormat) {
        DocumentType documentType = DocumentType.NONE;

        if (null != documentFormat) {
            switch (documentFormat) {
                case DOCX:
                case ODT: {
                    documentType = DocumentType.TEXT;
                    break;
                }

                case XLSX:
                case ODS: {
                    documentType = DocumentType.SPREADSHEET;
                    break;
                }

                case PPTX:
                case ODP:
                case ODG: {
                    documentType = DocumentType.PRESENTATION;
                    break;
                }

                case NONE:
                default: {
                    break;
                }
            }
        }

        return documentType;
    }

    /**
     * Provides the user's default folder id from the session.
     *
     * @param folderService
     *            The folder service to be used to retrieve the user folder id.
     *
     * @param session
     *            The session of the user.
     *
     * @return
     *         The folder id of the user or null if the folder id couldn't be
     *         retrieved.
     */
    static public String getUserFolderId(FolderService folderService, final Session session) {
        String folderId = null;

        if ((null != folderService) && (null != session) && (session instanceof ServerSession)) {
            final String treeId = com.openexchange.folderstorage.FolderStorage.REAL_TREE_ID;
            final ContentType contentType = InfostoreContentType.getInstance();

            try {
                UserizedFolder defaultFolder = folderService.getDefaultFolder(((ServerSession) session).getUser(), treeId, contentType, session, null);

                if (null != defaultFolder) {
                    folderId = defaultFolder.getID();
                }
            } catch (OXException e) {
                LOG.warn("GetUserFolderId - retrieving default folder of user failed with exception", e);
            }
        }

        return folderId;
    }

    /**
     * Provides the user's documents folder id from the session.
     *
     * @param session
     *            The session of the user.
     *
     * @param services
     *            The service lookup instance to be used by this method.
     *
     * @return
     *         The folder id of the users documents.
     */
    static public String getUserDocumentsFolderId(final Session session, ServiceLookup services) {
        String folderId = ConfigurationHelper.getCascadeConfigurationValue(services, session, "io.ox/files", "folder/documents");
        return folderId;
    }

    /**
     * Provides the user's templates folder id from the session.
     *
     * @param session
     *            The session of the user.
     *
     * @param services
     *            The service lookup instance to be used by this method.
     *
     * @return
     *         The folder id of the users templates.
     */
    static public String getUserTemplatesFolderId(final Session session, final ServiceLookup services) {
        String folderId = ConfigurationHelper.getCascadeConfigurationValue(services, session, "io.ox/files", "folder/templates");
        if (StringUtils.isEmpty(folderId)) {
            // fall-back if templates folder is not set - can be configured by administrator
            // instead use the user's default folder
            folderId = getUserFolderId(session, services);
        }
        return folderId;
    }

    /**
     * Provides the user's default folder id from the session.
     *
     * @param session
     *            The session of the user.
     *
     * @param services
     *            The service lookup instance to be used by this method.
     *
     * @return
     *         The folder id of the user or null if the folder id couldn't be
     *         retrieved.
     */
    static public String getUserFolderId(final Session session, ServiceLookup services) {
        final long time1 = System.currentTimeMillis();
        String folderId = null;

        if (null != services) {
            FolderService folderService = services.getService(FolderService.class);
            folderId = DocFileHelper.getUserFolderId(folderService, session);
        }
        LOG.trace("RT connection: TIME to get user folder = " + (System.currentTimeMillis() - time1) + "ms");

        return folderId;
    }

}
