package com.openexchange.office.tools;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Set;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.logging.Log;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import com.openexchange.exception.OXException;
import com.openexchange.file.storage.FileStorageFolder;
import com.openexchange.file.storage.composition.IDBasedFolderAccess;
import com.openexchange.file.storage.composition.IDBasedFolderAccessFactory;
import com.openexchange.office.tools.UserConfigurationHelper.Mode;
import com.openexchange.office.tools.error.ErrorCode;
import com.openexchange.office.tools.error.ExceptionToErrorCode;
import com.openexchange.office.tools.files.FileHelper;
import com.openexchange.office.tools.files.FolderHelper;
import com.openexchange.server.ServiceLookup;
import com.openexchange.session.Session;
import com.openexchange.tools.session.ServerSession;

/**
 * RecentFileListManager provides methods to read/write the user recent file
 * list from the user configuration. Additional support for methods to
 * add/update the list to reflect the latest changes.
 *
 * @author Carsten Driesner
 * @since 7.6.0
 */
public class RecentFileListManager {

    private static final Log LOG = com.openexchange.log.Log.loggerFor(RecentFileListManager.class);
    private static final String[] m_neededKeys = {"folder_id", "id", "last_opened", "last_modified", "creation_date", "title", "file_mimetype", "filename" };
    private static final HashSet<String> m_filterKeys = new HashSet<String>(Arrays.asList(m_neededKeys));
    private static final int MAX_RECENT_ENTRIES = 30;

	private ServiceLookup m_services = null;
	private Session m_session = null;
	private UserConfigurationHelper m_userConfHelper = null;

    /**
     * Creates a new instance of a RecentFileListManager.
     *
     * @param services
     *  The service lookup instance to be used by the new instance.
     *
     * @param session
     *  The session of the user where the recent file list should be read or
     *  modified.
     */
	public RecentFileListManager(ServiceLookup services, Session session) {
        m_services = services;
        m_session = session;
        m_userConfHelper = new UserConfigurationHelper(m_services, m_session, "io.ox/office", Mode.WRITE_BACK);
    }

	/**
	 * Reads the recent file list from the user that is referenced by the
	 * session provided to this instance.
	 *
	 * @return
	 *  A list of recent files as JSONObjects.
	 */
	public List<JSONObject> readRecentFileList(ApplicationType app) {
		ArrayList<JSONObject> result = null;

    	if ((null != m_services) && (null != m_session)) {
    		String userConfEntryKey = "portal/" + ApplicationType.enumToString(app) + "/recents";
        	String jsonRecentFileListString = m_userConfHelper.getString(userConfEntryKey);

        	if (null != jsonRecentFileListString) {
        		try {
        		    final JSONArray recentFileArray = new JSONArray(jsonRecentFileListString);
        		    final ArrayList<JSONObject> list = new ArrayList<JSONObject>();

        		    for (int i = 0; i < Math.min(recentFileArray.length(), MAX_RECENT_ENTRIES); i++) {
            		    	Object obj = recentFileArray.get(i);
            		    	if (obj instanceof JSONObject) {
            		    		final JSONObject fileDescriptor = (JSONObject)obj;
            				    // filter out non-important attributes
            				    filterFileDescriptorAttributes(fileDescriptor);
            				    fileDescriptor.put("path", getFilePath(fileDescriptor.getString("folder_id")));
            		    		list.add(fileDescriptor);
            		    	}
        		    }
        		    result = list;
                } catch (final JSONException e) {
        			LOG.error("RT connection: Exception while reading recent file list entry " + userConfEntryKey , e);
        		}
        	}
    	}

    	return result;
    }

	private JSONArray getFilePath(String folderId) throws JSONException
    {
	    final IDBasedFolderAccess folders = m_services.getService(IDBasedFolderAccessFactory.class).createAccess(m_session);

	    final List<JSONObject> result = new ArrayList<JSONObject>();
	    Locale locale = null;
	    if (m_session instanceof ServerSession) {
	        locale = ((ServerSession) m_session).getUser().getLocale();
	    }

	    FileStorageFolder folder = null;
	    while(null != folderId) {
	        try {
                folder = folders.getFolder(folderId);

                if (FolderHelper.isRootFolder(folder)) {
                    break;
                }

            } catch (final OXException f) {
                final ErrorCode errorCode = ExceptionToErrorCode.map(f, ErrorCode.GENERAL_UNKNOWN_ERROR, false);
                if (isImportantException(errorCode)) {
                    LOG.warn("OXException while trying to fetch file path in folder " + folderId, f);
                } else {
                    LOG.debug("OXException while trying to fetch file path in folder " + folderId);
                }
                break;
            } catch (final Exception e){
                LOG.warn("Error while fetching file path of " + folderId, e);
                break;
            }

	        final JSONObject folderInfo = new JSONObject();
	        folderInfo.put("id", folder.getId());

	        String fileName = null;
	        if (null != locale) {
                fileName = folder.getLocalizedName(locale);
	        } else {
                fileName =folder.getName();
	        }
	        if(StringUtils.isEmpty(fileName)) {
	            fileName = "...";
	        }
            folderInfo.put("filename", fileName);

	        result.add(0, folderInfo);
	        folderId = folder.getParentId();
	    }

	    if (result.size()>1) {
	        //remove "IPM-Root"
	        result.remove(0);

	        if (result.size()>1) {
	            //remove "Drive"
	            result.remove(0);
	        }
	    }

        return new JSONArray(result);
    }

    /**
	 * Adds a recentFile JSONObject to the recent file list. If the entry
	 * already exists in the list it will be removed and added at the front.
	 *
	 * @param recentFiles
	 *  A list of JSONObjects describing recent file list entries.
	 *
	 * @param recentFile
	 *  A JSONObject describing a recent file list entry.
	 *
	 * @param lastModified
	 *  The date of the last modification or null if there was no modification.
	 */
    public void addCurrentFile(List<JSONObject> recentFiles, JSONObject recentFile, java.util.Date lastModified) {
    	if ((null != recentFiles) && (null != recentFile)) {
    		String id1 = recentFile.optString("id", "");
    		String folder1 = recentFile.optString("folder_id", "");

    		if ((id1.length() > 0) && (folder1.length() > 0)) {
    			int foundIndex = -1;

			    for (int i = 0; i < recentFiles.size(); i++) {
			    	Object obj = recentFiles.get(i);
			    	if (obj instanceof JSONObject) {
			    	    final JSONObject json = (JSONObject)obj;
			    		if (id1.equals(json.optString("id", "")) &&
	    				    folder1.equals(json.optString("folder_id", ""))) {
			    			foundIndex = i;
			    			break;
			    		}
			    	}
			    }

			    // an entry which already exists must be moved to the first pos
			    if (foundIndex >= 0) {
			    	recentFiles.remove(foundIndex);
			    }

		    	// filter out non-important attributes
			    filterFileDescriptorAttributes(recentFile);
			    recentFiles.add(0, recentFile);

                // remove entries that exceeds the max size of the recent list
                if (recentFiles.size() > MAX_RECENT_ENTRIES) {
                    recentFiles.subList(MAX_RECENT_ENTRIES, recentFiles.size()).clear();
                }
    		}
    	}
    }

    /**
     * Updates a recent file entry without changing the order of entries. If the
     * entry cannot be found it will be added at the front.
     *
     * @param recentFiles
     * @param fileId
     * @param updatedFileDescriptorProps
     * @param lastModified
     */
    public void updateFile(final List<JSONObject> recentFiles, final String fileId, final JSONObject updatedFileDescriptorProps, java.util.Date lastModified) {
        if ((null != recentFiles) && (StringUtils.isNotEmpty(fileId))) {

            int foundIndex = -1;

            for (int i = 0; i < recentFiles.size(); i++) {
                Object obj = recentFiles.get(i);
                if (obj instanceof JSONObject) {
                    if (fileId.equals(((JSONObject) obj).optString("id", ""))) {
                        foundIndex = i;
                        break;
                    }
                }
            }

            if (foundIndex >= 0) {
                // update existing entry
                final JSONObject oldRecentFile = recentFiles.get(foundIndex);
                final JSONObject newRecentFile = RecentFileListManager.makeUpdatedCopy(oldRecentFile, updatedFileDescriptorProps);
                // filter out non-important attributes
                filterFileDescriptorAttributes(newRecentFile);
                recentFiles.set(foundIndex, newRecentFile);
            }
        }
    }

    /**
     * Removes a file from the recent file list if it part of the list.
     * It does nothing if the file is not part of the list.
     *
     * @param recentFiles
     *  The current recent file list.
     *
     * @param fileToBeRemoved
     *  A file descriptor which references the file to be removed. Must at
     *  least contain a valid folder and file id.
     *
     * @return
     *  TRUE if the file has been removed or FALSE if nothing has been
     *  changed.
     */
    public boolean removeFileFromList(final List<JSONObject> recentFiles, final JSONObject fileToBeRemoved) {
    	if ((null != recentFiles) && (null != fileToBeRemoved)) {
    		String id1 = fileToBeRemoved.optString("id", "");
    		String folder1 = fileToBeRemoved.optString("folder_id", "");

    		if ((id1.length() > 0) && (folder1.length() > 0)) {
    			int foundIndex = -1;

			    for (int i = 0; i < recentFiles.size(); i++) {
			    	Object obj = recentFiles.get(i);
			    	if (obj instanceof JSONObject) {
			    		if (id1.equals(((JSONObject) obj).optString("id", "")) &&
	    				    folder1.equals(((JSONObject)obj).optString("folder_id", ""))) {
			    			foundIndex = i;
			    			break;
			    		}
			    	}
			    }

			    if (foundIndex >= 0) {
			    	recentFiles.remove(foundIndex);
			    	return true;
			    }
    		}
    	}

    	return false;
    }

    /**
     * Writes the recent file list from a user to the recent file list. The
     * old list entries will be overwritten.
     *
     * @param app
     *  Specifies for which application type the list should be written.
     *
     * @param recentFiles
     *  The JSONObject list of recent files.
     *
     * @return
     *  TRUE if writing the list was successful, otherwise FALSE.
     */
    public boolean writeRecentFileList(ApplicationType app, final List<JSONObject> recentFiles) {
    	boolean result = false;

    	if ((null != m_userConfHelper) && (app != ApplicationType.APP_NONE) && (null != recentFiles)) {
    		String userConfEntryKey = "portal/" + ApplicationType.enumToString(app) +"/recents";

		    JSONArray recentFileArray = new JSONArray(recentFiles);
		    String jsonString = recentFileArray.toString();
		    m_userConfHelper.setValue(userConfEntryKey, jsonString);
		    result = true;
    	}

    	return result;
    }

    /**
     * Flush writes the recent file list data persistently to the user
     * configuration.
     *
     * @return
     *  TRUE if flushing the data was successful, otherwise FALSE.
     */
    public boolean flush() {
        boolean result = false;
        final UserConfigurationHelper userConfHelper = m_userConfHelper;

    	if (null != userConfHelper) {
    		result = userConfHelper.flushCache();
    	}

    	return result;
    }

    /**
     * Migrate an old recent file user configuration structure to the new
     * data structure. The old recent file user configuration stays intact
     * to enable users to use an old version. Be careful calling this more
     * than once for any application will overwrite current user data.
     * The method doesn't check if there are existing recent files in the
     * new data structure.
     *
     * @param app
     *  The application type which should be migrated to the new data
     *  structure.
     *
     * @return
     *  TRUE if the migration was successful, otherwise FALSE.
     */
    public boolean migrateOldRecentFileList(ApplicationType app) {
    	boolean result = false;

    	String userConfEntryKey = "portal/recents";
    	String oldRecentFiles = m_userConfHelper.getString(userConfEntryKey);

    	if ((null != oldRecentFiles) && (app != ApplicationType.APP_NONE)) {
    		HashSet<String> appExtensionSet = null;

    		if (app == ApplicationType.APP_TEXT)
    			appExtensionSet = new HashSet<String>(Arrays.asList(MimeTypeHelper.TEXT_EXTENSIONS));
    		else if (app == ApplicationType.APP_SPREADSHEET)
    			appExtensionSet = new HashSet<String>(Arrays.asList(MimeTypeHelper.SPREADSHEET_EXTENSIONS));
    		else if (app == ApplicationType.APP_PRESENTATION)
    			appExtensionSet = new HashSet<String>(Arrays.asList(MimeTypeHelper.PRESENTATION_EXTENSIONS));

    		ArrayList<JSONObject> recentAppFiles = new ArrayList<JSONObject>();
    		try {
    		    JSONArray oldRecentFilesArray = new JSONArray(oldRecentFiles);
    		    for (int i = 0; i < oldRecentFilesArray.length(); i++) {
    		        Object obj = oldRecentFilesArray.get(i);
    		        if (obj instanceof JSONObject) {
    		        	JSONObject fileDescriptor = (JSONObject)obj;

    		        	filterFileDescriptorAttributes(fileDescriptor);
    		        	String fileName = fileDescriptor.optString("filename", null);
    		        	String ext = FileHelper.getExtension(fileName);
    		        	if (appExtensionSet.contains(ext)) {
    		        		recentAppFiles.add(fileDescriptor);
    		        	}
    		        }
    		    }

    		    // prepare the new data structure to write to the new recent
    		    // data structure
    		    prepareNewRecentDataStructure(app);
    		    if (!recentAppFiles.isEmpty()) {
    		        writeRecentFileList(app, recentAppFiles);
    		    }
    		} catch (JSONException e) {
    			LOG.error("Exception caught while migrating old user recent files to new data structure", e);
    		}
    	} else {
    		// prepare the new data structure to enable writing to the new
    		// recent data structure.
    		prepareNewRecentDataStructure(app);
    	}

    	return result;
    }

    /**
     * Creates a simple file descriptor to check if a file is part of a
     * recent file list. It's not compliant to be added to the recent file
     * list.
     *
     * @param folderId
     *  The folder id of the new file descriptor.
     *
     * @param fileId
     *  The file identifier of the new file descriptor.
     *
     * @return
     *  A file descriptor as a JSONObject or null if one of the arguments is
     *  not set.
     */
    public static JSONObject createFileDescriptor(String folderId, String fileId) {
    	JSONObject result = null;

    	if ((null != folderId) && (null != fileId)) {
	    	try {
	    		JSONObject jsonObject = new JSONObject();
	    	    jsonObject.put("id", fileId);
	    	    jsonObject.put("folder_id", folderId);
	    	    result = jsonObject;
	    	} catch (JSONException e) {
	    		// no exception -> error case is mapped to result == null
	    	}
    	}
    	return result;
    }

    /**
     * Prepares the new recent file data structure. For a specific application
     * type.
     *
     * @param app
     *  The application type
     * @return
     *  TRUE if the new data structure has been written, otherwise FALSE.
     */
    private boolean prepareNewRecentDataStructure(ApplicationType app) {
    	boolean result = false;

    	if (app != ApplicationType.APP_NONE) {
    		final String appString = ApplicationType.enumToString(app);
	    	final String userConfAppEntryKey = "portal/" + appString + "/recents";
			final String strEmptyJSONArray = (new JSONArray()).toString();

	    	if (null != m_userConfHelper) {
	    		try {
		    		// check "portal" config tree and create if missing
		    		String data = null;
		    		JSONObject portalConfTree = m_userConfHelper.getJSONObject("portal");
		    		if (null == portalConfTree) {
		    			// create portal config tree
		    			portalConfTree = new JSONObject();
		    			m_userConfHelper.setValue("portal", portalConfTree);
		    		}
		    		// check text application JSONObject - create if missing
		    		data = m_userConfHelper.getString(userConfAppEntryKey);
		    		if (null == data) {
		    			// create application JSONObject which will contain the
		    			// "recents" JSONString attribute
		    			JSONObject appContainer = new JSONObject();
		    			appContainer.put("recents", strEmptyJSONArray);
		    			// put the application JSONObject to the portal config
		    			// tree
		    			portalConfTree.put(appString, appContainer);
		    		}
		    		m_userConfHelper.flushCache();
		    		result = true;
	    		}
	            catch (JSONException e) {
	    		    LOG.error("Exception caught while creating new structure for user recent files", e);
	    		}
	        }
    	}

    	return result;
    }

    /**
     * Filters non-important attributes from a JSONObject that contains
     * file descriptor attributes. This saves memory and time.
     *
     * @param fileDescriptor
     *  A JSONObject that contains the data of a file descriptor.
     */
    private void filterFileDescriptorAttributes(JSONObject fileDescriptor) {
    	if (null != fileDescriptor) {
    		Set<String> keys = fileDescriptor.asMap().keySet();
    		for (String key : keys) {
    			if (!m_filterKeys.contains(key)) {
    			    fileDescriptor.remove(key);
    			}
    		}
    	}
    }

    /**
     * Creates an updated copy of the provided JSONObject.
     *
     * @param source the source JSONObject to be copied
     * @param updateFileProps the updated properties merged into the copy
     * @return the copied JSONObject containing the updated properties
     */
    private static final JSONObject makeUpdatedCopy(final JSONObject source, final JSONObject updateFileProps) {
        JSONObject result = source;

        try {
            result = new JSONObject(source, source.keySet().toArray(new String[0]));

            final Iterator<String> props = updateFileProps.keys();
            while (props.hasNext()) {
                final String key = props.next();
                result.put(key, updateFileProps.get(key));
            }
        } catch (JSONException e) {

        }

        return result;
    }

    /**
     * Determines the importance of an exception dependent on the
     * mapped error code.
     *
     * @param errorCode
     * @return TRUE if the exception is important and should be logged on
     *         WARN/ERROR. FALSE if the exception must not be logged on
     *         ERROR/WARN or INFO.
     */
    public static boolean isImportantException(final ErrorCode errorCode) {
        boolean result = true;

        if (errorCode.isNoError())
            result = false;
        else {
            final int code = errorCode.getCode();
            switch (code) {
                // Currently we define folder exceptions where the folder is not found
                // or invisible (due to missing permissions) as "normal" and NOT important
                // The owner of a folder can set the permissions or delete it at runtime
                // therefore it's "normal" that we encounter these exceptions/error checking
                // the recent file list.
                case ErrorCode.CODE_GENERAL_FOLDER_NOT_FOUND_ERROR:
                case ErrorCode.CODE_GENERAL_FOLDER_NOT_VISIBLE_ERROR: result = false; break;
                default: result = true; break;
            }
        }

        return result;
    }
}
